/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */


#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <netdb.h>
#include <net/if.h>             /* ifreq, IFNAMSIZ */
#include <sys/socket.h>         /* inet_network    */
#include <netinet/in.h>         /* inet_network    */
#include <arpa/inet.h>          /* inet_network    */
#include <sys/time.h>
#include <sys/resource.h>       /* getrlimit       */
#include <sys/types.h>          /* FD_ZERO, FD_SET */
#include <unistd.h>             /* close           */
#include <syslog.h>             /* LOG_DAEMON      */
#include <errno.h>              /* errno           */

#include "config.h"
#include "pidfile.h"
#include "interface.h"
#include "enbd_conf.h"
#include "select.h"

#define PIDDIR "/var/run"

static int debug;
#define PDEBUG(x...) { if (debug) fprintf(stderr,"enbd-cstatd: " x); }
#define PERROR(x...) { syslog((debug?LOG_PERROR:0)|LOG_DAEMON|LOG_ERR,"enbd-cstatd: " x); }

/*
 * Get a line beginning ACK command .. from the socket and report the
 * error code (OK or ERROR) as 0 or -1.
 * */
static int
get_ack(int sock, char *command) {

    int err;
    int offset = 0;
    const int buflen = 1024;
    char buffer[buflen];
    const unsigned long utimeout = 5000000;

    buffer[offset] = 0;

    // listen for an ACK reply
    while (offset < buflen) {

        char *p;
        static fd_set rfds, efds;

        FD_ZERO(&rfds);
        FD_ZERO(&efds);
        FD_SET(sock, &rfds);
        FD_SET(sock, &efds);

        if (microselect(sock+1,&rfds,NULL,&efds, utimeout) < 0) {
            PDEBUG("get_ack: socket timed out after %d chars\n", offset);
            return -1;
        }

        // PTB byte or err waiting
        
        err = recv(sock, &buffer[offset], 1, MSG_NOSIGNAL|MSG_WAITALL);
        if (err < 1) {
            PDEBUG("get_ack: socket closed after %d chars\n", offset);
            return -1;
        }

        // successful read
        offset += err;
        buffer[offset] = 0;

        if (buffer[offset - 1] != '\n') {
            // PTB get more tokens as we have no newline yet
            continue;
        }

        PDEBUG("get_ack: socket had newline at posn %d after %s\n",
                offset, buffer);
        // PTB now we have a newline (first time)
        p = strtok(buffer, " \t\n\r");
        if (strcmp(buffer, "ACK")==0) {
            p = strtok(NULL, " \t\n\r");
            if (p && strcmp(p, command) == 0) {
                p = strtok(NULL, " \t\n\r");
                if (p && strcmp(p, "OK") == 0) {
                        return 0;
                }
            }
            PDEBUG("get_ack: failed after %d chars\n", offset);
            return -1;
        }

        // PTB discard buffer and continue to next line
        offset = 0;
        buffer[offset] = 0;
        
    } // end while
    
    PDEBUG("get_ack: failed after %d chars\n", offset);
    return -1;
}

/*
 * We talk to the server helper daemon and tell it to stop
 * */
static int 
notify_server_client_stop(short server_port_nr, char * server_ipaddrs[]) {

  struct interface ife;
  char * server_ipaddr;
  short sstatd_port_nr;
  struct servent * sstatd_servent;

  if ((sstatd_servent = getservbyname("enbd-sstatd", "tcp")) == NULL 
  &&  (sstatd_servent = getservbyname("nbd-sstatd",  "tcp")) == NULL ) {
    static short count;
    if (count++ <= 0)
      PERROR("there is no endb- or nbd-sstatd entry in /etc/services\n");
    return -1;
  }
  sstatd_port_nr = sstatd_servent->s_port;

  init_interface(&ife);
  if (ife.open(&ife) < 0)
    return -1;

  for (server_ipaddr = *server_ipaddrs;
       server_ipaddr; 
       server_ipaddr = *++server_ipaddrs) {

    long server_ipaddr_nr = inet_addr(server_ipaddr);
    struct sockaddr_in addrin;
    int sock;
    const int buflen = 1024;
    char buffer[buflen];
    int offset = 0;
    int err;

    if (server_ipaddr_nr == -1)
      continue;
    if ((htonl(0xff000000) & server_ipaddr_nr) == htonl(0x7f000000))
      continue;

    // open socket to other side
    sock = socket(AF_INET,SOCK_STREAM,IPPROTO_TCP);
    if (sock < 0) {
      PERROR("cannot make tcp socket\n");
      return -1;
    }

    addrin.sin_family = AF_INET;
    addrin.sin_port = htons(sstatd_port_nr);
    addrin.sin_addr.s_addr = server_ipaddr_nr;

    err = connect (sock, (struct sockaddr *) &addrin, sizeof(addrin));
    if (err < 0) {
      close(sock);
      continue;
    }

    offset += sprintf(buffer+offset, "notice client-stop %hd ", server_port_nr);

    while(ife.next(&ife) >= 0) {
      char * ipaddr = ife.sprintaddr(&ife);
      unsigned long ipaddr_nr = ife.getaddr(&ife);
      if ((htonl(0xff000000) & ipaddr_nr) == htonl(0x7f000000))
        continue;
      if (offset > buflen - 40)
        break;
      offset += sprintf(buffer+offset,"%s ", ipaddr);
    }

    offset += sprintf(buffer+offset, "\n");
    send(sock, buffer, offset, 0);
    err = get_ack(sock, "notice");

    offset = 0;
    buffer[offset] = 0;

    offset += sprintf(buffer+offset,"quit\n");
    send(sock, buffer, offset, 0);
    err = get_ack(sock, "quit");

    offset = 0;
    PDEBUG("debug: send message to %s: <<%s>>\n", server_ipaddr, buffer);
    close(sock);
  }

  // ...
  ife.close(&ife);
  return 0;
}

static int
restart_client(char * id) {

     char format[] = "/etc/init.d/enbd start client %s >/dev/null 2>&1\n";
     char cmdbuf[256];
     int err;

     sprintf(cmdbuf, format, id);
     PDEBUG("restart client with %s\n", cmdbuf);
     err = system(cmdbuf);
     return err;
}

static int
check_and_restart_or_signal_client(char * id) {

     char format[] = PIDDIR "/" "enbd-client-%s.pid";
     char *pidfile;
     struct nbd_pidfile client_pidfile;
     int cpid;

     if (!id) {
       PERROR("null id!\n");
       return -1;
     }

     pidfile = malloc(strlen(id)+strlen(format));

     if (pidfile) {
       sprintf(pidfile, format, id);
     } else {
          PERROR("could not allocate %lu bytes of memory\n",
              (unsigned long)(strlen(id)+strlen(format)));
          return -4;
     }
     initpidfile(&client_pidfile,pidfile);

     cpid = client_pidfile.read(&client_pidfile);
     free(pidfile);

     if (cpid < 0) {
       int err;
       PERROR("no client known for %s - restarting\n", id);
       err = restart_client(id);
       // return err so that we launch a process to check success later
       return -2;
     }
     if (kill(cpid,0) < 0 && errno == ESRCH) {
       int err;
       PERROR("client %s (%d) not running - restarting\n", id, cpid);
       err = restart_client(id);
       // return err so that we launch a process to check success later
       return -3;
     }
     PERROR("sending SIGPWR to live client %s (process %d)\n", id, cpid);
     if (kill(cpid,SIGPWR) < 0) {
       PERROR("signal to client %s (%d) failed - giving up\n", id, cpid);
       return -1;
     }
     return 0;
}

static void usage() {

    PERROR("commands:\n");
    PERROR("\tnotice server-start PORT IPADDR ...\n");
    PERROR("\thelp TOPIC\n");
    PERROR("\tquit\n");
}

static char *get_client_id(short port, char * ipaddrs[]) {

    char * ipaddr;
    struct nbd_confent line;

    init_nbd_conf(&line);

    for (ipaddr = *ipaddrs; ipaddr; ipaddr = *++ipaddrs) {

      struct hostent * hostent;
      
      line.set(&line);

      while (line.get(&line) >= 0) {

        int i = 0;

        PDEBUG("get_client_id: target: %s\n", line.target);
        if (0 != strcmp("client", line.target))
          continue;

        PDEBUG("get_client_id: id: %s\n", line.id);
        if (!line.id || 0 == strcmp("", line.id))
          continue;

        PDEBUG("get_client_id: server: %s\n", line.server);
        if  (!line.server)
          continue;

        PDEBUG("get_client_id: port: %d\n", line.port);
        if (line.port != port)
          continue;

        goto by_name;

     by_name:
        hostent = gethostbyname(line.server);
        if  (!hostent)
          goto by_addr;

        PDEBUG("get_client_id: %s name: %s\n", line.server, hostent->h_name);
        if (0 == strcmp(hostent->h_name,ipaddr)) {
          return line.id;
        }
        PDEBUG("get_client_id: %s addrlen: %d\n", line.server, hostent->h_length);

        i = 0;
        while (1) {
          char * server_ipaddr = hostent->h_addr_list[i++];
          if (!server_ipaddr)
            break;
          PDEBUG("get_client_id: %s addr: %x\n", line.server, *(unsigned*)server_ipaddr);
          if (*(unsigned*)server_ipaddr == inet_addr(ipaddr)) {
            return line.id;
          }
        } // ewhile

        i = 0;
        while (1) {
          char * server_alias = hostent->h_aliases[i++];
          if (!server_alias)
            break;
          PDEBUG("%s alias: %s\n", line.server, server_alias);
          if (0 == strcmp(server_alias,ipaddr)) {
            return line.id;
          }
        } // ewhile

     by_addr:
        // maybe the conf file contained an addr
        if (0 == strcmp(line.server, ipaddr)) {
          return line.id;
        }

        hostent = gethostbyaddr(line.server, strlen(line.server), AF_INET);
        if  (!hostent)
          goto more_bright_ideas;

        PDEBUG("get_client_id: %s name: %s\n", line.server, hostent->h_name);
        if (0 == strcmp(hostent->h_name,ipaddr)) {
          return line.id;
        }

        i = 0;
        while (1) {
          char * server_ipaddr = hostent->h_addr_list[i++];
          if (!server_ipaddr)
            break;
          PDEBUG("get_client_id: %s addr: %x\n", line.server, *(unsigned*)server_ipaddr);
          if (*(unsigned*)server_ipaddr == inet_addr(ipaddr)) {
            return line.id;
          }
        } // ewhile

        i = 0;
        while (1) {
          char * server_alias = hostent->h_aliases[i++];
          if (!server_alias)
            break;
          PDEBUG("get_client_id: %s alias: %s\n", line.server, server_alias);
          if (0 == strcmp(server_alias,ipaddr)) {
            return line.id;
          }
        } // ewhile

     more_bright_ideas:
        // somehow try harder to identify what we've been given
        i = i; // yes - to stop compiler moaning about label at end
        
      } // ewhile
    } // efor
    line.end(&line);
    return NULL;
}

#ifndef HAVE_DAEMON
static int
daemon(int nochdir, int noclose) {
    int pid = fork();
    if (pid > 0) {
        // we are the parent
        exit (0);
    }
    if (pid < 0) {
        // failed
        return -1;
    }
    if (!nochdir) {
        chdir("/");
    }
    if (!noclose) {
        int i, j = 0, m, n;
        struct rlimit rlim;
        if (getrlimit(RLIMIT_NOFILE, &rlim) < 0) {
            n = m = 8;
        } else  {
            n = rlim.rlim_cur;
            m = rlim.rlim_max;
        }
        for (i = 0; i < m && j < n ; i++) {
            if (close(i) >= 0)
                j++;
        }
    }
    if (pid > 0) {
        // we are the parent
        exit (0);
    }
    if (pid < 0) {
        // failed
        return -1;
    }
    return 0;
}
#endif

/*
 * Send signal SIGPWR to the appropriate client
 * */
static int notice(int argc, char * argv[]) {

  char * news = argv[0];

  if (argc < 1) {
    PERROR("too few arguments to notice\n");
    return -1;
  }
  if (!news || 0 == strcmp("",news)) {
    PERROR("empty qualifier to notice\n");
    return -1;
  }

  if (0 == strcmp("help", news)) {
      printf("MSG notice server-start server_port server_ipaddr ipaddr ..\n");
      printf("MSG notice server-stop  server_port server_ipaddr ipaddr ..\n");
      return 0;
  }
  if (0 == strcmp("server-stop", news)) {
      // PTB nothing yet
      return 0;
  }
  if (0 == strcmp("server-start", news)) {

      char *server_port = argv[1];
      char **server_ipaddrs;
      short server_port_nr = -1;
      char *id;

      if (argc < 2) {
        PERROR("too few arguments to notice\n");
        return -1;
      }
      if (!server_port || 0 == strcmp("",server_port)) {
        PERROR("empty port argument to notice\n");
        return -1;
      }
      if (sscanf(server_port, " %hd ", &server_port_nr) < 1) {
        PERROR("malformed port argument to notice\n");
        return -1;
      }
      if (server_port_nr == -1) {
        PERROR("illegal port argument to notice\n");
        return -1;
     }

      server_ipaddrs = argv + 2;
      id = get_client_id(server_port_nr, server_ipaddrs);
      if (!id || 0 == strcmp("",id)) {
        PERROR("no client configured to servers %s:%d ...\n",
             server_ipaddrs[0], server_port_nr);
        notify_server_client_stop(server_port_nr,server_ipaddrs);
        return -1;
      }
      PDEBUG("notice: check/restart client id %s\n", id);
      if (check_and_restart_or_signal_client(id) < 0) {

          pid_t pid;

          char format[] = PIDDIR "/" "enbd-cstatd-%s.pid";
          char *pidfile;
          struct nbd_pidfile cstatd_pidfile;

          PERROR("client %s check or restart failed\n", id);
          // PTB FIXME - don't launch if there is already one

          pidfile = malloc(strlen(id)+strlen(format));

          if (pidfile) {
              sprintf(pidfile, format, id);
          } else {
              PERROR("could not allocate %lu bytes of memory\n",
                  (unsigned long)(strlen(id)+strlen(format)));
              return -ENOMEM;
          }
          initpidfile(&cstatd_pidfile,pidfile);

          if ((pid = fork()) == 0) {

              // child
              void removepidfile(void) {
                  cstatd_pidfile.unlock(&cstatd_pidfile);
              };

              // PTB try and detach. Don't worry if we can't.
              daemon(0,0);

              if (cstatd_pidfile.lock(&cstatd_pidfile) < 0)
                  exit (1);

              atexit(removepidfile);

              // 30s wait to start checker
              do {
                  unsigned tosleep = 30;
                  while ((tosleep = sleep(tosleep)) > 0);
              } while (0);

              // 60s intervals between checks
              while (check_and_restart_or_signal_client(id) < 0) {
                  unsigned tosleep = 60;
                  while ((tosleep = sleep(tosleep)) > 0);
              }
              cstatd_pidfile.unlock(&cstatd_pidfile);
              exit (0);
          }

          // parent
          if (pid < 0)
              return -1;

          return -1;
      }
      return 0;
  }
  // ...
  return -1;
}

static int quit(int argc, char * argv[]) {

  char * what = argv[0];

  if (argc < 1) {
    exit(0);
    return 0;
  }

  if (!what || 0 == strcmp("",what))
    return -1;

  if (0 == strcmp("help", what)) {
      printf("MSG quit\n");
      return 0;
  }

  return 0;
}

static int help(int argc, char * argv[]) {

  char * subject = argv[0];

  if (argc < 1) {
    printf("MSG help notice quit\n");
    return 0;
  }
  if (!subject || 0 == strcmp("",subject)) {
    printf("MSG help notice quit\n");
    return 0;
  }

  if (0 == strcmp("notice", subject)) {
      notice(1, (char*[]){ "help", NULL});
      return 0;
  }
  if (0 == strcmp("quit", subject)) {
      quit(1, (char*[]){ "help", NULL});
      return 0;
  }
  if (0 == strcmp("help", subject)) {
      printf("MSG help subject\n");
      return 0;
  }
  printf("MSG help notice quit\n");
  return 0;
}

static int status(int argc, char * argv[]) {

  char * dowhat, * port;
  short port_nr;

  if (argc < 2)
    return -1;

  dowhat  = argv[0];
  port    = argv[1];

  if (0 == strcmp("help", dowhat)) {
     printf("MSG status client-show server_port server_ipaddr\n");
     return 0;
  }

  if (sscanf(port, " %hd ", &port_nr) < 1) {
    PERROR("illegal port number %s\n", port);
    return -1;
  }


  if (0 == strcmp("client-show", dowhat)) {

     char  ** ipaddrs = argv + 2;

     char * id = get_client_id(port_nr,ipaddrs);

     char format[] = PIDDIR "/" "enbd-client-%s.pid";
     char *pidfile;
     struct nbd_pidfile client_pidfile;
     int cpid;

     if (!id) {
       printf("not configured\n");
       return 0;
     }

     pidfile = malloc(strlen(id)+strlen(format));
     if (pidfile) {
         sprintf(pidfile, format, id);
     } else {
         PERROR("could not allocate %lu bytes of memory\n",
              (unsigned long)(strlen(id)+strlen(format)));
          return -1;
     }

     initpidfile(&client_pidfile,pidfile);

     cpid = client_pidfile.read(&client_pidfile);
     free(pidfile);

     if (cpid <= 0) {
          printf("not running\n");
          return 0;
     }
     if (kill(cpid,0) >= 0) {
          printf("running\n");
          return 0;
     }

     printf("not running\n");
     return 0;

  } // end if client-show

  return -1;
  
}

static int proxy(int argc, char * argv[]) {

  char * dowhat, * port;
  short port_nr;

  if (argc < 2)
    return -1;

  dowhat  = argv[0];
  port    = argv[1];

  if (sscanf(port," %hd ",&port_nr) < 1) 
     return -1;

  if (0 == strcmp("help", dowhat)) {
     printf("MSG proxy client-stop  server_port server_ipaddr\n");
     printf("MSG proxy client-start server_port server_ipaddr\n");
     return 0;
  }
  if (0 == strcmp("client-stop", dowhat)) {
     char * id = get_client_id(port_nr, argv + 2);
     char cmdbuf[256];
     if (!id || strlen(id) > 220)
       return -1;
     sprintf(cmdbuf,"/etc/init.d/enbd stop  client %s\n", id);
     system(cmdbuf);
     return 0;
  }
  if (0 == strcmp("client-start", dowhat)) {
     char * id = get_client_id(port_nr, argv + 2);
     char cmdbuf[256];
     if (!id || strlen(id) > 220)
       return -1;
     sprintf(cmdbuf,"/etc/init.d/enbd start client %s\n", id);
     system(cmdbuf);
     return 0;
  }
  return -1;
  
}

void sighandler(int signo) {
    syslog(LOG_DAEMON|LOG_ERR,"enbd-cstatd: received signal %d\n", signo);
    exit (signo);
}

int main(int argc, char *argv[]) {

  # define BUFLEN 1024
  const int buflen = BUFLEN;
  char buffer[BUFLEN];
  int err = -1;

  if (argc >= 2 && !strcmp("-d",argv[1]))
    debug = 1;

  // To stop us dying with SIGPIPE when we write to stderr and the
  // other side has closed the socket to which stderr is redirected
  signal(SIGPIPE, sighandler);

  buffer[0] = 0;

  while (fgets(buffer, buflen, stdin)) {
  
    char * command;
    int wordc = 0;
    char *wordv[buflen/2];

    command = strtok(buffer," \t\r\n");

    if (!command || 0 == strcmp("",command))
      continue;
    
    while (1) {
       char * word = strtok(NULL," \t\r\n");
       if (!word)
         break;
       wordv[wordc++] = word;
    }
    // null terminate the sequence
    wordv[wordc] = NULL;

    if (0 == strcmp("help",command)) {
        err = help(wordc,wordv);
    } else if (0 == strcmp("quit",command)) {
        err = quit(wordc,wordv);
    } else if (0 == strcmp("notice",command)) {
        err = notice(wordc,wordv);
    } else if (0 == strcmp("proxy",command)) {
        err = proxy(wordc,wordv);
    } else if (0 == strcmp("status",command)) {
        err = status(wordc,wordv);
    } else {
        usage();
        exit(1);
    }

    if (err >= 0) {
        printf("ACK %s OK\n", command);
    } else {
        printf("ACK %s ERROR\n", command);
    }

    fflush(NULL);
  }
  return 0;
}
