#! /bin/sh

CONFIG=${NBD_CONFIG-/etc/enbd.conf}
[ -r $CONFIG ] || exit 1

set -x

#################################################################
# get IP address from a hostname
#
gethostbyname() {

  local host="$1"

  case "$host" in
    [0-9]*.[0-9]*.[0-9]*.[0-9]*) 
       echo -n $1 ; return ;;
  esac

  local results

  # lookup /etc/hosts first
  set -- `grep "[ \t]$host" /etc/hosts | grep -v '#'`
  local ipaddr=""
  for i; do
    case $i in
    [0-9]*.[0-9]*.[0-9]*.[0-9]*) ipaddr=$i ;;
    $host) [ -n "$ipaddr" ] && echo -n " $ipaddr" && return ;;
    esac
  done

  # now try named, maybe ypcat first?
  results="`nslookup "$host" | grep ^Address: | tail +2 | cut -d: -f 2`"
  for result in $results; do

    case $result in
    [0-9]*.[0-9]*.[0-9]*.[0-9]*) echo -n " $result" ;;
    *) continue ;;
    esac

  done
}

#################################################################
# list our interface IP addresses
#
my_interface_ips() {

     /sbin/ifconfig \
   | grep 'inet addr:' \
   | sed -e 's/.*inet addr:\([0-9.][0-9.]*\).*/\1/'
}

#################################################################
# send our hostnames ip address to a given sstatd
# (this is so the other side can erase us from its client list)
# I guess we need our other interfaces too ..
#
notify_server_client_stop() {

  local server_port="$1"
  local server_ipaddr="$2"
  local my_ips="`my_interface_ips`"
  local ips=""

  # don't emit local ips unless we need to
  case $server_ipaddr in
    127.*.*.*) ;;
    *) for i in $my_ips ; do
         case $i in
           127.*.*.*) ;;
           *) ips="$ips $i" ;;
         esac
       done
    ;;
  esac

  ( echo notice client-stop $server_port $ips; echo quit ) \
  | socket $server_ipaddr enbd-sstatd
}

#################################################################
# do a (re)start of a given client id (because it's not up)
#
restart_client() {

  id="$1"
  #rm -f /var/run/enbd-client-$id.pid
  #echo restarting client $id
  /etc/init.d/enbd start client $id >/dev/null 2>&1
}

#################################################################
# look at the pids listed for a given client id and restart the
# missing processes, and signal the live processes with SIGPWR
# (this is to tell the client that the server has restarted)
#
check_and_restart_or_signal_client() {

  id="$1"
  #echo checking client $id
  if [ ! -s /var/run/enbd-client-$id.pid ] ; then
    restart_client $id
    return
  fi
  local cpid="`head -1 /var/run/enbd-client-$id.pid`"

  for i in $cpid dummy; do
    cpid=$i ; break
  done
  cpid=${cpid%%[^0-9]*}

  [ $cpid -gt 0 ] || return

  if ! kill -0 $cpid >/dev/null 2>&1 ; then
    restart_client $id
    return
  fi
  #echo sending SIGPWR to process $cpid
  kill -PWR $cpid >/dev/null 2>&1
}

#################################################################
# usage message
#
usage() {
 echo COMMANDS
 echo "notice server-start PORT IPADDR [IPADDR ..]"
}

#################################################################
# list the config file, sans commentary. This is a lexer, really.
#
config() {
   cat $CONFIG | grep -v '^[ ]*$' | while read line; do

     case "$line" in
       "#"*) continue ;;
     esac

     set -- $line

     [ -z "$1" ] && continue

     echo -n "$1" ; shift

     for word; do
       case "$word" in
         "#"*) break ;;
          *)   echo -n " $word" ;;
       esac
     done
     echo
   done
}

#################################################################
# get client id given server port and server ipaddr
#
get_client_id() {
  echo $* | while read port ipaddr; do
    config | while read target id rest; do

      [ "$target" = client ] || continue
      [ -z "$id" ] && continue

      echo "$rest" | while read device server server_port options; do

        server_ipaddrs="`gethostbyname $server`"
        found=n
        for server_ipaddr in $server_ipaddrs dummy; do
          server_addr=${server_addr%%[^0-9.]*}
          if [ $server_ipaddr = $ipaddr ] ; then
            found=y
            break
          fi
        done

        [ $found = y ] || continue
                  
        [ $server_port -eq $port ] || continue

        # OK, this instance should have been talking to the server
        echo $id
        break;
      done 
    done 
  done
}

#################################################################
# breakout client line from conf file, given id
#
get_client_line() {

  local cid="$1"
  config | while read target id rest; do
      [ "$target" = client ] || continue
      [ "$id" = "$cid" ] || continue
      echo $rest
  done
  
}

#################################################################
# handle notice from server telling us it starts up
#
notice() {

      local news=$1
      shift
      case $news in
      help)
        echo notice server-start server_port server_ipaddr ipaddr ..
        echo notice server-stop  server_port server_ipaddr ipaddr ..
        return
        ;;
      server-stop)
        : # nothing yet
        ;;
      server-start)
  
        echo $* | while read server_port server_ipaddrs ; do

          server_port=${server_port%%[^0-9]*}
          [ -n "$server_port" -a "$server_port" -gt 0 ] || continue

          for server_ipaddr in $server_ipaddrs dummy; do

            case $server_ipaddr in
              [0-9]*.[0-9]*.[0-9]*.[0-9]*) ;;
              *) continue ;;
            esac

            server_ipaddr=${server_ipaddr%%[^0-9.]*}
       
            local id=`get_client_id $server_port $server_ipaddr`
            if [ -n "$id" ] ; then
              check_and_restart_or_signal_client $id
              echo $id
            fi
          done | wc -l | while read n rest; do
             if [ -n "$n" -a "$n" -le 0 ]; then
               # no record .. tell the server to stop pestering us
               notify_server_client_stop $server_port $server_ipaddr
             fi
          done
        done # end treatment of one input line
        ;;
      esac
}

#################################################################
# quit
#
quit() {
  local what=$1
  case $what in
  help) echo quit ;;
  *)    exit 0 ;;
  esac
}

#################################################################
# help message
#
help() {
  local subject=$1
  case $subject in
  notice|quit)
      $subject help
      ;;
  help)
      echo help subject
      ;;
  *)  echo help notice quit
      ;;
  esac
}

#################################################################
# get some info on tthe client daemon state
#
status() {
  echo $* | while read dowhat port ipaddr ; do
    case $dowhat in
      help)  echo status client-show server_port server_ipaddr
             continue
        ;;
      client-show) 
             local id=`get_client_id $port $ipaddr`
             if [ -z "$id" ]; then
               echo not configured
             else
               PIDFILE=/var/run/enbd-client-$id.pid
               if [ ! -s $PIDFILE ]; then
                 echo not running
               else
                 local cpid="`head -1 $PIDFILE`"

                 for i in $cpid dummy; do
                   cpid=$i ; break
                 done

                 cpid=${cpid%%[^0-9]*}

                 if ! [ $cpid -gt 0 ] ; then
                   echo not running
                 elif kill -0 $cpid >/dev/null 2>&1 ; then
                     echo running
                 else
                     echo not running
                 fi

               fi
             fi
        ;;
      *) continue
        ;;
    esac
  done
}

#################################################################
# do init commands
#
proxy() {

  echo $* | while read dowhat port ipaddrs ; do

    case $dowhat in
      help)  echo proxy client-stop  server_port server_ipaddr
             echo proxy client-start server_port server_ipaddr
        ;;
      client-stop) 
             local id=`get_client_id $port $ipaddrs`
              /etc/init.d/enbd stop client $id
        ;;
      client-start) 
             local id=`get_client_id $port $ipaddrs`
             /etc/init.d/enbd start client $id
        ;;
      *) continue
        ;;
    esac
  done 
 
}

#################################################################
# read the servers ipaddr on stdin and check our conf file for the
# client ids that should be connected to it, then signal them that the
# server has restarted
#
main() {
  while read command rest; do

    [ -n "$command" ] || continue

    case ${command} in
    help|quit|notice|proxy|status)
        $command $rest
        ;;
    *)  usage
        exit 1
        ;;
    esac
  done
}

main $*
exit 0

