/*
 * Network Block Device - server
 *
 * Copyright 1996-1998 Pavel Machek, distribute under GPL
 *  <pavel@atrey.karlin.mff.cuni.cz>
 *
 * Version 1.0 - hopefully 64-bit-clean
 * Version 1.1 - merging enhancements from Josh Parsons, <josh@coombs.anu.edu.au>
 * Version 2.1 - enhancements from Peter Breuer, <ptb@it.uc3m.es>
 * Version 2.2 - SSL additions by Andres Marin, <amarin@it.uc3m.es>
 *             - more improvements from ptb, also negotiate cleanups
 *          and improvements from James MacLean <macleajb@EDnet.NS.CA>
 * Version 2.4 - negotiations moved into master daemons <ptb@it.uc3m.es>
 *             - 64 bit fixes <ptb@it.uc3m.es>
 *             - accept new connections on new top-level daemon
 */

#include <sys/socket.h>
#include <netinet/tcp.h>
#include <netinet/in.h>         /* sockaddr_in, htons, in_addr */
#include <netdb.h>              /* hostent, gethostby*, getservby* */
#include <syslog.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <fcntl.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <setjmp.h>
#include <errno.h>

#include <sys/types.h>
//#include <sys/resource.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/mman.h>           /* MAP_FAILED */

#include <net/if.h>             /* IFNAMSIZ */

#include <asm/atomic.h>         /* atomic_t, atomic_read, atomic_set, ... */

#define ISSERVER
#define MY_NAME "enbd-server"
#include <linux/enbd.h>
#include "cliserv.h"
#include "db.h"
#include "file.h"

#ifdef USING_SSL
#include <openssl/lhash.h>
#include <openssl/bn.h>
#include "sslincl.h"
#endif

#include "stream.h"
#include "server_stub.h"
#include "socket.h"
#include "select.h"
#include "logging.h"
#include "alarm.h"
#include "ipaddrfile.h"
#include "pidfile.h"
#include "interface.h"
#include "md5.h"
#include "lock.h"
#include "list.h"
#include "hash.h"
#include "shmem.h"
#include "time.h"


#ifdef USING_SSL
#define TEST_CERT       "server.pem"
#define BUFSIZE 16*1024
#endif

#ifdef USING_SSL
extern int verify_callback(int ok, X509_STORE_CTX *ctx);
extern int set_cert_stuff(SSL_CTX *ctx, char *cert_file, char *key_file);
static int init_ssl_connection(SSL *con);
#endif
  
# define MAXFILE 32
# define LISTEN_QUEUE_LENGTH 10
# define MAXSESSION 128
#define SYNC_INTVL 1



   static char signature[ENBD_SIGLEN];
                                   /* session id for exported file */
   static int has_sig = 0;         /* chosen a sig? */
                                   /* Files I'm exporting */
   int debug_level = DEBUG_DFLT;   /* debugging */
#ifdef USING_SSL
   static int using_ssl       = 0; /* amended later */
   static int verify_depth    = 0;
   static char *cipher        = NULL;
   static int s_server_verify = SSL_VERIFY_NONE;
   static char *s_cert_file   = TEST_CERT,
               *s_key_file    = NULL;
   static char *s_dcert_file  = NULL,
               *s_dkey_file   = NULL;
   static SSL_CTX *ctx        = NULL;
   static int s_server_session_id_context[ENBD_MAXCONN];
   static int nocert          = 0;
   static char *CApath        = NULL,
               *CAfile        = NULL;
           BIO *bio_err       = NULL;
   SSL_METHOD  *meth          = NULL;
#endif

   struct nbd_server {
       int       i;                 /* PTB identifier 0 to MAXCONN-1 */
       int       port;              /* PTB port we listen to */
       int       freeportmin;       /* SE  where to look for next available */
       int       freeportmax;
       int       pid;               /* PTB server process id */
       int       sock;              /* PTB descriptor of net connection */
       int       socket;            /* PTB principal socket for bind */

//#define F_READONLY      0x0001      /* same as ENBD_CLIENT_READ_ONLY */
//#define F_INITIALISED   0x0004      /* same as ENBD_CLIENT_INITIALISED */
//#define F_SIGNED        0x0008      /* same as ENBD_CLIENT_SIGNED */
//#define F_DEBUG         0x0010      /* same as ENBD_CLIENT_DEBUG */
//#define F_NEGOTIATED    0x0020      /* same as ENBD_CLIENT_NEGOTIATED */
//#define F_SERVER_RO     0x0040      /* same as ENBD_CLIENT_SERVER_RO */

//#define F_INTRODUCED    0x0400      /* same as ENBD_CLIENT_INTRODUCED */


       unsigned long flags;
       u64       size;              /* PTB exportsize */
       struct nbd_file file;
       int       nfile;             /* PTB number of files */
       char *    names[MAXFILE];
       int blksize;                 /* PTB host device blocksize (default 1k) */

# define ENBD_LINEAR_MODE 'l'
# define ENBD_STRIPE_MODE 0
# define ENBD_MIRROR_MODE 1

       int mode;                  
#ifdef USING_SSL
       SSL*      con;
#endif
       struct nbd_stub_server server;
       int errs;
       unsigned long caddr;         /* PTB client IP address */
       int nconn;                   /* PTB number of ports */
       int child[ENBD_MAXCONN];      /* PTB child slave pid list */
       struct nbd_server *          /* PTB session managers */
           session[MAXSESSION];
       struct nbd_db spids;         /* PTB session manager pid/ipaddr db */
       struct nbd_stream stream;
       int negotiate_timeout;       /* PTB die and restart if not done */
       long pulse_intvl;            /* PTB heartbeat */
       long cache_lim;              /* PTB max cached results */
       int data_timeout;            /* PTB reply from client */
       struct nbd_pidfile
             pidfile;               /* PTB pid file */
       struct nbd_statfile
             statfile;              /* PTB state file */
       void (*sighandler)(int);     /* PTB signal handler */

# define ENBD_SERVER_MASTER  0x1000
# define ENBD_SERVER_SESSION 0x2000
# define ENBD_SERVER_SLAVE   0x4000

       unsigned long type;          /* PTB kind of server */
       char * buffer;               /* PTB for transfers to/from kernel */
       size_t buflen;               /* PTB for transfers to/from kernel */
       void * shmem;                /* PTB for intercommunication */
       int ordered_write_timeout;   /* PTB number of millis - 1000  off*/
       s64  sizes[MAXFILE];         /* PTB cmdline sizes of compinents */
       char *lockdir;               /* PTB where to place locks */
       char *statdir;               /* PTB where to place state */
       long sync_intvl;             /* PTB  us between file sync pusles */
   };
   static struct nbd_server manager = {
       i:            -2,    // PTB id
       port:         -1,    // PTB port
       freeportmin:  -1,    // SE Start of range to look for free port
       freeportmax:  64*1024 - 1,   // SE End of range to look for free port
       pid:          -1,    // PTB pid
       sock:         -1,    // PTB sock
       socket:       -1,    // PTB socket
       flags:        0,
       size:         0L,    // PTB exportsize
       nfile:        0,     // PTB number of files comprising 
       names:        {0, },
       blksize:      1024,  // PTB host size blksize (default 1024)
       mode:         ENBD_LINEAR_MODE,
                            // PTB default linear raid
#ifdef USING_SSL
       con:          0,     // PTB SSL *con
#endif
       errs:         0,
       caddr:        0,
       nconn:        0,
       child:        {0, 0, },
       session:      {NULL, NULL, },
       spids:        {0, },
       negotiate_timeout:120, // PTB die and restart if not done 
       pulse_intvl:  10,    // PTB heartbeat 
       cache_lim:    10000, // PTB cached results limit
       data_timeout: 30,    // PTB reply from client 
       pidfile:      { NULL,NULL,},
       statfile:     { NULL,NULL,},
       sighandler:   SIG_DFL,
       type:         ENBD_SERVER_MASTER,
       buffer:       NULL,
       buflen:       0,
       shmem:        NULL,
       ordered_write_timeout:
                     0,      // max number of millis to delay for
       sizes:        { 0L, 0L, 0L, },
       lockdir:      PIDDIR,
       statdir:      STATDIR,
       sync_intvl:   SYNC_INTVL * 1000000,
   };
   static struct nbd_server * self = &manager;


   static inline int writenet(struct nbd_server *self, char *buf, int len)
   {
      struct nbd_stream * stream = & self->stream;
      int res;
      // PTB safety measure
      if (!stream->magic) {
         PERR("stream has magic %lx and port %d\n", stream->magic,stream->port);
         PERR("stream socket %d, our socket %d\n", stream->sock, self->sock);
      }
      DEBUG("write %d bytes on socket %d\n", len, self->sock);
      res = stream->write(stream,buf,len);
      DEBUG("write %d on socket %d reurned %d\n", len, self->sock, res);
      return res;
   }

   static inline int readnet(struct nbd_server *self, char *buf, int len)
   {
      struct nbd_stream * stream = & self->stream;
      int res;
      if (!stream->magic) {
         PERR("stream has magic %lx and port %d\n", stream->magic,stream->port);
         PERR("stream socket %d, our socket %d\n", stream->sock, self->sock);
      }
      DEBUG("read %d on socket %d\n", len, self->sock);
      res = stream->read(stream,buf,len);
      DEBUG("read %d on socket %d reurned %d\n", len, self->sock, res);
      return res;
   }

   /*
    * Bug (#2) identified by Christian  Schmid (webmaster@rapidforum.com).
    * Thanks! Expand path so as not to lose relative path when we go
    * daemon and do a chdir /.
    * */
   static char *expandpath(char *name) {

       unsigned max = PATH_MAX;
       char *path;
       int n, m;
       if (name == NULL)
           return NULL;
       if (*name == '/')
           return name;
       n = strlen(name);
       path = malloc(max + n + 2);
       if (path == NULL)
           return NULL;
       while (getcwd(path, PATH_MAX) == NULL && errno == ERANGE) {
           free(path);
           max <<= 1;
           path = malloc(max + n + 2);
           if (path == NULL)
               return NULL;
       }
       m = strlen(path);
       path[m] = '/';
       strncpy(&path[m+1], name, n+1);
       return path;
   }

   static char usage_str[] =
                "Usage: ctlport file_to_export .. [size | -s nblks] "
                "[-l|-0] [-r] [-i sig] [-b blksiz] [-t timeout] "
                "[-p pulse_intvl] [-K] [-n] [-e] [-h cachelim] [-L lockdir]"
                "[-S statdir] [-F minport[-maxport]]"
#ifdef USING_SSL
                "[-cert certfile] [-key keyfile] "
                "[-CApath CApath] [-CAfile CAfile] [-verify depth]\n"
#endif
               "        if port is set to 0, stdin is used (for running from inetd)\n"
               "                address of machine trying to connect\n";

   static void usage() {
         printf(usage_str);
    }

    static int
    cmdline (struct nbd_server *self, int argc, char *argv[]) {
    
        int i = 0;
    
        if (argc < 3)
            return 1;
    
        if (DEBUG_DFLT > 0) {
            self->flags |= F_DEBUG;
        }
        // now start searching for opts 
        for (i = 1; i < argc; i++) {
    
            if (*argv[i] != '-') {
    
                char *strerr, *name;
    
                // look for port first
                if (self->port < 0) {
                    int port = strtol (argv[i], &strerr, 10);
                    if (*argv[i] && strerr && !*strerr) {
                        self->port = port;
                        continue;
                    }
                }
                // now must be a file or the final size
    
                if (self->size <= 0) {
                    int es = strtol (argv[i], &strerr, 10);
                    if (*argv[i] && strerr && !*strerr) {
                        self->size = es;
                        switch (*strerr) {
                          case 'K':
                            self->size <<= 10;
                            break;
                          case 'M':
                            self->size <<= 20;
                            break;
                          case 'G':
                            self->size <<= 30;
                            break;
                        }
                        continue;
                    }
                }
                // seems to be not a pure number, maybe a file
                name = argv[i];
                if (*name != '/') {
                    name = expandpath (name);
                }
                self->names[self->nfile++] = name;
                continue;
    
            }

            // now we're an opt
            switch (argv[i][1]) {
              case '0':
                self->mode = ENBD_STRIPE_MODE;
                break;
              case '1':
                self->mode = ENBD_MIRROR_MODE;
                break;
#ifdef USING_SSL
              case 'C':
                if (i + 1 >= argc)
                    return 1;
                if (!strcmp (argv[i], "-CApath")) {
                    CApath = argv[++i];
                    using_ssl = 1;
                    DEBUG ("client CApath is %s\n", CApath);
                    break;
                }
                if (!strcmp (argv[i], "-CAfile")) {
                    CAfile = argv[++i];
                    using_ssl = 1;
                    DEBUG ("client CAfile is %s\n", CAfile);
                    break;
                }
                PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                return 1;
#endif
              case 'K':
                self->flags |= F_DOLOCK;
                break;
              case 'L':
                if (i + 1 >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                    return 1;
                }
                self->lockdir = argv[++i];
                DEBUG ("server lock directory is %s\n", self->lockdir);
                break;
              case 'S':
                if (i + 1 >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                    return 1;
                }
                self->lockdir = argv[++i];
                DEBUG ("server state directory is %s\n", self->statdir);
                break;
              case 'a':
                self->flags |= F_ASYNC;
                self->flags &= ~F_SYNC;
                break;
              case 'F':
                if (++i >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n",
                          argv[i - 1]);
                    return 1;
                } else {
                    int freeport;
                    char *strerr, *strptr;
                    strptr = strchr (argv[i], '-');
                    if (strptr != NULL) {
                        *strptr++ = 0;
                        freeport = strtol (strptr, &strerr, 10);
                        if (!*strptr || *strerr) {
                            PERR ("server gets illegal cmdline arg [%s]\n",
                                  strptr);
                            return 1;
                        }
                        if (self->port > 0
                            && freeport >= self->freeportmin) {
                            self->freeportmax = freeport;
                        } else {
                            PERR
                             ("freeport option must be after port option\n");
                            return 1;
                        }
                    }
                    freeport = strtol (argv[i], &strerr, 10);
                    if (!*argv[i] || *strerr) {
                        PERR ("server gets illegal cmdline arg [%s]\n",
                              argv[i]);
                        return 1;
                    }
                    if (self->port > 0 && freeport <= self->freeportmax) {
                        self->freeportmin = freeport;
                    } else {
                        PERR ("freeport option must be after port option\n");
                        return 1;
                    }
                }
                break;
              case 'b':
                if (++i >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n",
                          argv[i - 1]);
                    return 1;
                } else {
                    char *strerr;
                    int oldblksize = self->blksize;
                    self->blksize = strtol (argv[i], &strerr, 10);
                    if (!*argv[i]) {
                        PERR ("server gets illegal cmdline arg [%s]\n",
                              argv[i]);
                        return 1;
                    }
                    switch (*strerr) {
                      case 'K':
                        self->blksize <<= 10;
                        break;
                      case 'M':
                        self->blksize <<= 20;
                        break;
                      case 'G':
                        self->blksize <<= 30;
                        break;
                    }
                    if (self->blksize > oldblksize && oldblksize != 0)
                        self->size *= self->blksize / oldblksize;
                    else if (self->blksize < oldblksize && self->blksize != 0)
                        self->size /= oldblksize / self->blksize;
                }
                break;
#ifdef USING_SSL
              case 'c':
                if (i + 1 >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                    return 1;
                }
                if (!strcmp (argv[i], "-cert")) {
                    s_cert_file = argv[++i];
                    using_ssl = 1;
                    DEBUG ("server cert file is %s\n", s_cert_file);
                    break;
                }
                if (!strcmp (argv[i], "-cipher")) {
                    cipher = argv[++i];
                    using_ssl = 1;
                    DEBUG ("client cipher: %s\n", cipher);
                    break;
                }
                PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                return 1;
#endif
              case 'd':
                self->flags |= F_DEBUG;
                debug_level = 1;
                break;
              case 'h':
                if (i + 1 >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                    return 1;
                } else {
                    char *strerr;
                    int cache_lim;
                    i++;
                    cache_lim = strtol (argv[i], &strerr, 10);
                    if (!*argv[i] || *strerr) {
                        PERR ("server gets illegal cmdline arg [%s]\n",
                              argv[i]);
                        return 1;
                    }
                    self->cache_lim = cache_lim;
                    DEBUG ("client says cache reqs limit is %ld\n",
                            (long)cache_lim);
                    continue;
                }
                break;
              case 'i':
                if (++i >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n",
                          argv[i - 1]);
                    return 1;
                }
                strncpy (signature, argv[i], sizeof (signature));
                has_sig = 1;
                break;
#ifdef USING_SSL
              case 'k':
                if (i + 1 >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                    return 1;
                }
                if (strcmp (argv[i], "-keyfile")) {
                    PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                    return 1;
                }
                s_key_file = argv[++i];
                using_ssl = 1;
                DEBUG ("server key cert file is %s\n", s_key_file);
                break;
#endif
              case 'l':
                self->mode = ENBD_LINEAR_MODE;
                break;
              case 'n':
                self->flags |= F_DIRECT;
                break;
              case 'p':             /* WG */
                if (i + 1 >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                    return 1;
                } else {
                    char *strerr;
                    int pulse_intvl;
                    i++;
                    pulse_intvl = strtol (argv[i], &strerr, 10);
                    if (!*argv[i] || *strerr) {
                        PERR ("server gets illegal cmdline arg [%s]\n",
                              argv[i]);
                        return 1;
                    }
                    self->pulse_intvl = pulse_intvl;
                    DEBUG ("client says pulse interval is %ld\n",
                            (long)pulse_intvl);
                    self->data_timeout = self->pulse_intvl * 3;
                    continue;
                }
                break;
              case 'r':
                self->flags |= F_READONLY;
                break;
              case 's':
                if (i + 1 >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                    return 1;
                } else {
                    int j;
                    char *p;
                    i++;
                    self->size = 0;
                    for (j = 0, p = strtok (argv[i], ",");
                         j < MAXFILE && p; j++, p = strtok (NULL, ",")) {
                        unsigned long size = 0;
                        char *strerr;
                        size = strtol (p, &strerr, 10);
                        if (!*p) {
                            self->sizes[j] = size = 0;      // PTB - restore default
                            continue;
                        }
                        DEBUG ("client says resource size is %ld blocks\n",
                               size);
                        self->sizes[j] = size;
                        switch (*strerr) {
                          case 'K':
                            self->sizes[j] <<= 10;
                            break;
                          case 'M':
                            self->sizes[j] <<= 20;
                            break;
                          case 'G':
                            self->sizes[j] <<= 30;
                            break;
                        }
                        self->size += self->sizes[j] *= self->blksize;
                    }
                    continue;
                }
                break;
              case 't':
                if (i + 1 >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                    return 1;
                } else {
                    char *strerr;
                    i++;
                    self->negotiate_timeout = strtol (argv[i], &strerr, 10);
                    if (!*argv[i] || *strerr) {
                        self->negotiate_timeout = 60l;      // PTB - restore default
                        PERR ("server gets illegal cmdline arg [%s]\n",
                              argv[i]);
                        return 1;
                    }
                    DEBUG ("client says negotiation timeout is %d\n",
                           self->negotiate_timeout);
                    continue;
                }
                break;
#ifdef USING_SSL
              case 'v':
                if (i + 1 >= argc)
                    return 1;
                if (!strcmp (argv[i], "-verify")) {
                    char *err;
                    i++;
                    verify_depth = strtol (argv[i], &err, 10);
                    s_server_verify = SSL_VERIFY_PEER;
                    if (!*argv[i] || *err) {
                        PERR ("server gets illegal cmdline arg [%s]\n",
                              argv[i]);
                        return 1;
                    }
                    using_ssl = 1;
                    DEBUG ("server says verify_depth is %d\n", verify_depth);
                    break;
                }
                return 1;
#endif
              case 'w':
                if (i + 1 >= argc) {
                    PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                    return 1;
                } else {
                    char *strerr;
                    i++;
                    self->ordered_write_timeout =
                     strtol (argv[i], &strerr, 10);
                    if (!*argv[i] || *strerr) {
                        self->ordered_write_timeout = 1000; /* PTB - default */
                        PERR ("server gets illegal cmdline arg [%s]\n",
                              argv[i]);
                        return 1;
                    }
                    DEBUG ("client max wait for ordered write is %dms\n",
                           self->ordered_write_timeout);
                    continue;
                }
                break;
              case 'y':
                self->flags |= F_SYNC;
                self->flags &= ~F_ASYNC;
                break;
    
              default:
                PERR ("server gets illegal cmdline arg [%s]\n", argv[i]);
                return 1;
            }                       // eswitch
        }                           // efor
    
        return 0;
    
    }

/*
 * [make and] listen on a socket for a connection and split off and
 * return a new socket when someone tries to talk to us on it.
 */
static int
connectme (struct nbd_server *self)
{
	static struct sockaddr_in addrin;
	int addrinlen = sizeof (addrin);
	int err = 0;
	int sock;

	if (self->socket >= 0) {
		DEBUG ("Server (%d) reuses primary socket %d\n", self->i,
		       self->socket);
		goto accept;
	}

	self->socket = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (self->socket < 0) {
		MSG ("socket: %m\n");
		return -1;
	}

	DEBUG ("Server (%d) opens primary socket %d\n", self->i,
	       self->socket);

	addrin.sin_family = AF_INET;
	addrin.sin_port = htons (self->port);
	addrin.sin_addr.s_addr = 0;

	setmysockopt (self->socket);

#ifdef SO_RCVTIMEO
	if (1) {
		struct timeval timeout = { tv_sec:60, tv_usec:0 };
		if (setsockopt
		    (self->socket, SOL_SOCKET, SO_RCVTIMEO, &timeout,
		     sizeof (timeout)) < 0) {
			PERR
			 ("notice: setsockopt RCVTIMEO failed with %m\n");
		}
	}
#endif
#ifdef F_SETFL
#ifdef USING_SSL
	if (!using_ssl) {
#endif
		if (fcntl (self->socket, F_SETFL, O_NONBLOCK) < 0) {
			PERR
			 ("(%d) notice: fcntl NONBLOCK on socket %d failed with %m\n",
			  self->i, self->socket);
		}
		else {
			DEBUG ("(%d) server set socket %d nonblocking\n",
			       self->i, self->socket);
		}
#ifdef USING_SSL
	}
#endif
#endif

	DEBUG ("bind, ");
	if (bind (self->socket, (struct sockaddr *) &addrin, addrinlen) <
	    0) {
		PERR ("bind: %m\n");
		err = -2;
		goto fail;
	}
	goto listen;

      listen:
	DEBUG
	 ("Server (%d) waiting for connections from port %d on socket %d\n",
	  self->i, self->port, self->socket);

	DEBUG ("listen, ");
	if (listen (self->socket, LISTEN_QUEUE_LENGTH) < 0) {
		PERR ("listen: %m\n");
		err = -3;
		goto fail;
	}
	goto accept;

      accept:
	DEBUG ("accept, ");
	do {
		fd_set rfds;
		// PTB arbitary timeout of 5s per select on accept (will repeat)

		FD_ZERO (&rfds);
		FD_SET (self->socket, &rfds);

		err =
		 microselect (self->socket + 1, &rfds, NULL, NULL,
			      5 * 1000000);
		switch (err) {

		  case 1:	// PTB read data ready
			break;	// PTB ok

		  case -ETIME:	// PTB timeo
			goto accept;	// PTB try again

		  default:	// PTB interrupt
			close (self->socket);
			self->socket = -1;
			goto fail;
		}
	} while (0);

	sock = accept (self->socket, (struct sockaddr *) &addrin, &addrinlen);
	if (sock < 0) {
		PERR ("accept: %m\n");
		err = -4;
		goto fail;
	}

	do {
		unsigned long ipaddr = ntohl (addrin.sin_addr.s_addr);
		unsigned short q[4];
		int i;

		for (i = 3; i >= 0; i--) {
			q[i] = ipaddr & 0xff;
			ipaddr >>= 8;
		}

		MSG("server (%d) opened port %d (socket %d)"
                  " for client %d.%d.%d.%d\n",
		  self->i, self->port, sock, q[0], q[1], q[2], q[3]);

		// PTB now we hav to go on and check a list of our connects and
		// make sure we don't have two of these. If so, we kill the
		// others

		self->caddr = ipaddr;	// PTB this is purely temporary.

		// PTB apparently we now should now set stream->sock too.

	} while (0);

#ifdef USING_SSL
	if (!using_ssl) {
#endif
		if (fcntl (sock, F_SETFL, O_NONBLOCK) < 0) {
			MSG
			 ("warning: fcntl NONBLOCK on socket %d failed with %m\n",
			  sock);
		}
		else {
			DEBUG ("server set socket %d nonblocking\n", sock);
		}
#ifdef USING_SSL
	}
#endif

	self->sock = sock;
	return sock;

      fail:
	return err;

}

   static void disconnectme(struct nbd_server *self){
     DEBUG ("server (%d) closing socket on port #%d\n",self->i,self->port);
     if (self->sock > 0) {
       shutdown(self->sock,2);
       close(self->sock);
     }
     self->sock = -1;
#ifdef USING_SSL
     if (using_ssl)  {
       SSL_set_shutdown(self->con,SSL_SENT_SHUTDOWN|SSL_RECEIVED_SHUTDOWN);
#ifndef TRY_SSL_RECONNECT
       if (self->con)
         SSL_free(self->con);
       self->con = 0;
#endif
     }
#endif

   }

   static void closenet(int sig){
      // PTB this is a disconnectme() without benefit of pointer to self.
      disconnectme(self);
   }


  static void slavesighandler(int n) {
        static void nullsigchldhandler();

        if (self->type != ENBD_SERVER_SLAVE) {
            PERR( "server (%d) activates slave sighandler when not slave\n",
               self->i);
            return;
        }

        PERR( "server (%d) activates slave sighandler for signal %d\n",
             self->i, n);
 
        switch(n) {
           case SIGUSR1:
             break;
           case SIGUSR2:
             break;
           case SIGCHLD:
            nullsigchldhandler();
            break;
           case SIGTERM:
           case SIGINT:
           case SIGILL:
           case SIGSEGV:
           case SIGBUS:
            MSG( "server (%d) sighandler terminates slave %d safely\n",
                    self->i, self->pid);
            exit(0);
           default:     // PTB ignore 
            MSG( "server (%d) sighandler ignores signal %d\n", self->i, n);
            break;
         }
  }


#undef SEND
#define SEND(reply) writenet(self, (char*)(reply), sizeof(struct enbd_reply))
#undef ERROR
#define ERROR(reply) { \
   static int local_nerrs; \
   const int local_retries = 3; \
   if (local_nerrs > 1000) \
     local_nerrs = 0; \
   if ((reply)->error == 0) { \
      (reply)->error = htonl(1); \
   } else { \
      (reply)->error = htonl((reply)->error); \
   } \
   (reply)->magic = htonl(ENBD_REPLY_MAGIC); \
   SEND(reply); \
   if (local_nerrs++ < local_retries) \
     PERR("errored request!\n"); \
}
#undef SUCCESS
#define SUCCESS(reply) { \
   (reply)->error = 0; \
   (reply)->magic = htonl(ENBD_REPLY_MAGIC); \
   SEND(reply); \
}
                 

/*
 * An attempt to help the mainloop code look tidier. This is the part
 * that reads data from the disk and stuffs it into a reply. It writes
 * data into a buffer (buf). 
 */
static int
read_data (struct nbd_stub_server *server, char * buf, int reqlen, u64 reqfrom) {
      int len = 0;    // PTB number of reply buffer bytes filled

      DEBUG("Read request len %d from sector %Ld\n", reqlen, reqfrom >> 9 );

      // PTB read the data stream off the disk
      len = server->read(server,buf,reqlen,reqfrom);

      if (len != reqlen) {
           DEBUG("Read failed in mainloop: %m");
      }
  
      return len;
}

static int
read_md5_data (struct nbd_stub_server *server, char * buf, int reqlen,
u64 reqfrom, enbd_digest_t digest) {

      int len = 0;    // PTB number of reply buffer bytes filled

      DEBUG("Read request len %d from sector %Ld\n", reqlen, reqfrom >> 9 );

      // PTB read the data stream off the disk
      len = server->read(server,buf,reqlen,reqfrom);

      if (len != reqlen) {
           DEBUG("Read failed in mainloop: %m");
           return -1;
      }
      md5_buffer(buf, reqlen, (char *)digest);
  
      return len;
}

/*
 * Make the casts necessary to call the server ioctl 
 */
static int
srv_ioctl (struct nbd_stub_server *server, struct enbd_request *request, struct enbd_reply *reply, char * buf)
{
    int res;
    char *arg = (char*)(unsigned long)(((u64)request->from) & 0xffffffffL);
    int ioctl = (int)((((u64)request->from) >> 32) & 0xffffffffL);	

    DEBUG ("srv_ioctl received IOCTL req id %#x arg %p len %d\n",
            (unsigned)ioctl, arg, request->len);

#ifndef _IOC_DIR
    return -EINVAL;
#else 

    if (_IOC_DIR(ioctl) & _IOC_READ) {
        // we need to pass a local copy as a dummy.
            if (request->len > 0) {
                arg = buf;
                DEBUG ("srv_ioctl uses buffer %p (%#x..) for IOCTL %#x len %d\n",
                 arg, *(unsigned*)arg, (unsigned)ioctl, request->len);
            } else {
                arg = NULL;
                DEBUG ("srv_ioctl uses NULL buffer for IOCTL %#x\n",
                        (unsigned)ioctl);
            }
    }

    res = server->ioctl (server, ioctl, arg);

    if (_IOC_DIR(ioctl) & _IOC_READ) {
        if (request->len > 0) {
            DEBUG ("srv_ioctl returned %d for indirect IOCTL %#x arg %p (%#x..) len %d\n",
                res, (unsigned)ioctl, arg,
                *(unsigned*)arg, request->len);
        } else {
            DEBUG ("srv_ioctl returned %d for indirect IOCTL %#x arg %p len 0\n",
                res, (unsigned)ioctl, arg);
        }
    } else {
        DEBUG ("srv_ioctl returned %d for direct IOCTL %#x arg %p\n",
            res, (unsigned)ioctl, arg);
    }

    return res;

#endif /* !defined _IOC_DIR */

}

/*
 * minor setup for special call
 * */
static int
srv_special (struct nbd_stub_server *server, struct enbd_request *request, struct enbd_reply *reply)
{
    int res;
    int rw = (request->flags & ENBD_REQUEST_SPECIALRW)? WRITE : READ;
    int len = request->len;
    unsigned long long from = request->from;
    res = server->special (server, rw, len, from);
    return res;
}

/*
 * construct the write reply and send it
 * */
static int
do_srv_write (struct nbd_server *self, struct enbd_request *request,
	      struct enbd_reply *reply)
{
    struct nbd_stub_server *server = &self->server;
    struct nbd_shmem         *data = self->shmem;
    int res, err;

    /* PTB test the offset is in range */
    if (request->from + request->len > self->size) {
	PERR ("[RANGE! (+%Lu)]\n",
	     (unsigned long long)((request->from + request->len) - self->size));
	ERROR (reply);		// PTB give error reply
	return 0;		// try again
    }

    /* PTB test the type isn't WRITE while we're RO */
    if (self->flags & F_READONLY) {
	PERR ("[TYPE! (%d)\n]", request->type);
	ERROR (reply);		// PTB give error reply
	return 0;		// try again
    }

    DEBUG ("wr: net->buf, ");


    // PTB write. First wait for our turn.
    if (data && self->ordered_write_timeout > 0) {
        err = data->wait_seqno_timeout(data, request->seqno,
	        self->ordered_write_timeout);
        if (err < 0) {
	    switch (err) {
	      case -ETIME:		// PTB timed out - discard
                PERR ("server (%d) loses patience waiting with req %d, "
	              "sector %Ld-%Ld\n", self->i, request->seqno,
		      (unsigned long long)(request->from >> 9),
		 (unsigned long long)((request->from + request->len - 1) >> 9));
	        // PTB don't discard it or error it or we'll never get
	        // past this far future object!
	        break;
	    }
        }

        // PTB we arrived at our turn or we gave up waiting for it
        if (data && self->cache_lim > 0) {
            // PTB timeout is in millis
            unsigned long timeout = self->ordered_write_timeout;

            DEBUG("lock req %d by process %d\n", request->seqno, self->pid);
            // PTB the method is a dummy if cache_lim is zero
retry:
            err = data->lock_req(data, request->handle, request->seqno, timeout);
            if (err < 0) {
                switch (err) {
                  case -EINVAL:      // has already been done with error
                  default:
                    ERROR (reply);
                    return 0;
                    break;
                  case -EALREADY:    // already done successfully
	            SUCCESS (reply); // PTB give success reply
                    return 0;
                    break;
                  case -EINPROGRESS:
                    if (timeout <= 0) {
                            // PTB we give up waiting and drop it
                            err = -ETIME;
                            ERROR(reply);
                            return 0;
                            break;
                    } 
                    // PTB somebody else valid is still handling it
                    microsleep(1000);
                    // PTB waited one milli
                    PERR("waiting for lock on req %d\n", request->seqno);
                    timeout--;
                    goto retry;
                    break;
                }
            }
        }

    }

    while (request->len > 0) {

        unsigned long flags;
        struct nbd_stream * stream = & self->stream;

        /* PTB fill data buffer from net - nonblockingly, and repeat as reqd */
        flags = stream->flags;
        stream->flags |= ENBD_STREAM_NONBLOCK;
        res = readnet (self, self->buffer, request->len);
        stream->flags = flags;

        if (res < 0) {
            DEBUG ("failed to read %d bytes from net: %m\n", request->len);
           ERROR (reply);              // PTB give error reply
           return -EINVAL;
        }

        DEBUG ("upcoming write sector %Ld-%Ld seqno %d, ondisk seqno %d\n",
           request->from >> 9, (request->from + request->len - 1) >> 9,
           request->seqno, self->shmem ? *(int *) self->shmem : 0);

        if (0) {
                  // PTB testing
                  PERR("skipped req type %d for sectors %ld-%ld\n",
                          request->type,
                          (long) (request->from >> 9),
                          (long)((request->from + res - 1) >> 9));
                  request->from += res;
                  request->len  -= res;
                  continue;
        }
        // PTB write_to_disk: 
        if (server->write (server, self->buffer, res, request->from) < res) {
           PERR ("write to sector %Ld-%Ld seqno %d failed (%d): %m\n",
	      (unsigned long long)(request->from >> 9),
              (unsigned long long)((request->from + request->len - 1) >> 9),
	      request->seqno, res);
	  ERROR (reply);		// PTB give error reply
	  return 0;		//  try again
        }
        request->from += res;
        request->len  -= res;
    }

    // PTB tell world we handled it
    if (data && self->ordered_write_timeout > 0)
        data->update_seqno(data, request->seqno);

    if (data && self->cache_lim > 0) {
        // PTB the method is a dummy if cache_lim is zero
        err = data->unlock_req(data, request->handle, request->seqno);
        DEBUG("unlock req %d (err %d) by process %d\n",
                request->seqno, err, self->pid);
        if (err < 0) {
            // PTB somebody else handled it and we just boobed
            ERROR (reply);
            return -EINVAL;
        }
    }

    DEBUG ("buf->exp, ");

    DEBUG ("server (%d) prepares reply fields:\n"
	   "\t magic %#x\n"
	   "\terror %#x\n"
	   "\thandle %#lx\n"
	   "\tflags %d\n"
	   "\tdigest %.8x%.8x%.8x%.8x\n"
	   "on cmd %d len %d sector %lu\n",
	   self->i,
	   reply->magic,
	   reply->error,
	   (unsigned long) reply->handle,
	   reply->flags,
	   reply->data.digest[0], reply->data.digest[1],
	   reply->data.digest[2], reply->data.digest[3],
	   request->type, request->len,
	   (unsigned long) (request->from >> 9));

    /* PTB send a reply back through net  */
    return 0;
}

static int
do_srv_ioctl (struct nbd_server *self, struct enbd_request *request,
	      struct enbd_reply *reply)
{

    int res;
    int ioctl = request->from >> 32;
    struct nbd_stub_server *server = &self->server;

    DEBUG ("do_srv_ioctl IOCTL %#x received\n", (unsigned) ioctl);

#ifndef _IOC_DIR
    return -EINVAL;
#else

    if ((_IOC_DIR (ioctl) & _IOC_READ)
	&& (_IOC_DIR (ioctl) & _IOC_WRITE)) {
	// PTB we may have to read some more data
	if (request->len > 0) {
	    res = readnet (self, self->buffer, request->len);
	    if (res < request->len) {
		ERROR (reply);	// PTB give error reply
		return 0;
	    }
	    DEBUG ("do_srv_ioctl indirect IOCTL uses %dB buffer %p (%#x..)\n",
		   request->len, self->buffer,
		   *(unsigned *) self->buffer);
	} else {
	    // PTB we don't read any more data. Debugging.
	    DEBUG ("do_srv_ioctl indirect IOCTL uses 0B buffer %p\n",
		   self->buffer);
	}
    }

    res = srv_ioctl (server, request, reply, self->buffer);

    if (res < 0) {
	// PTB invalid ioctl - we can't handle this or the
	// resource can't, but we don't know which
	DEBUG ("server returns error reply (%d) for ioctl\n", res);
	ERROR (reply);		// PTB give error reply
	return 0;
    }

    // PTB FIXME need to put return value in the reply
    DEBUG ("do_srv_ioctl prepares OK reply (%d) for ioctl\n", res);

    reply->error = 0;

    if (_IOC_DIR (ioctl) & _IOC_READ) {
	if (request->len > 0) {
	    DEBUG("do_srv_ioctl prepares %dB data (%#x..) for indirect IOCTL\n",
		   request->len, *(unsigned *) self->buffer);
            res = request->len;
            // request send res bytes
            return res;
	}
    }
    return 0;

#endif /* !defined _IOC_DIR */

}

static int
do_srv_cksum (struct nbd_server *self, struct enbd_request *request,
	      struct enbd_reply *reply)
{
    int res;
    struct nbd_stub_server *server = &self->server;
    struct nbd_shmem * data = self->shmem;

    DEBUG ("exp->buf, ");

    // PTB - fill the reply digest with a 16 byte md5 checksum
    DEBUG ("received req to cksum %d bytes locally\n", request->len);

    DEBUG ("received checksum for %dB as %.8x%.8x%.8x%.8x for req %#lx\n",
	   request->len,
	   request->data.digest[0], request->data.digest[1],
	   request->data.digest[2], request->data.digest[3],
	   (unsigned long) request->handle);


    if (data && self->ordered_write_timeout > 0) {
        data->wait_seqno_timeout(data, request->seqno,
	        self->ordered_write_timeout);
    }
    res = read_md5_data (server, self->buffer,
			 request->len, request->from,
			 &reply->data.digest[0]);

    if (res < 0) {
	PERR ("failed to read %d bytes locally: %m\n", request->len);
	ERROR (reply);		// PTB give error reply
	return 0;		// try again
    }

    DEBUG ("checksummed %d local bytes as %.8x%.8x%.8x%.8x for req %#lx\n",
	   request->len,
	   reply->data.digest[0], reply->data.digest[1],
	   reply->data.digest[2], reply->data.digest[3],
	   (unsigned long) request->handle);

    // PTB - just send back the reply with no following data

    DEBUG ("server (%d) prepares reply fields:\n\t magic %#x\n"
	   "\terror %#x\n\thandle %#lx\n\tflags %d\n"
           "\tdigest %.8x%.8x%.8x%.8x\n"
	   "on cmd %d len %d sector %lu\n",
	   self->i,
	   reply->magic,
	   reply->error,
	   (unsigned long) reply->handle,
	   reply->flags,
	   reply->data.digest[0], reply->data.digest[1],
	   reply->data.digest[2], reply->data.digest[3],
	   request->type, request->len,
	   (unsigned long) (request->from >> 9));

    // PTB tell world we handled it
    if (data && self->ordered_write_timeout != 0) {
        if (0 == memcmp (&reply->data.digest[0], &request->data.digest[0], ENBD_DIGEST_LENGTH)) {
            data->update_seqno(data, request->seqno);
            DEBUG("md5sum seqno %d\n", request->seqno);
        } else {
        // PTB requests differ so we wait for the real write
        }
    }

    return 0;
}

static int
do_srv_special (struct nbd_server *self, struct enbd_request *request,
		struct enbd_reply *reply)
{
    int res;
    struct nbd_stub_server *server = &self->server;

    MSG ("got special %#lx type %d from=%Ld len=%d\n",
	 (unsigned long) request->handle,
	 request->type, (unsigned long long)request->from, request->len);
    // PTB we want to hold off here until the disk counter is clear
    res = srv_special (server, request, reply);
    if (res < 0) {
	// PTB invalid ioctl - we can't handle this or the
	// resource can't, but we don't know which
	DEBUG ("server returns error reply (%d) for special\n", res);
	ERROR (reply);		// PTB give error reply
	return 0;
    }
    reply->error = 0;
    return 0;
}

static int
do_srv_read (struct nbd_server *self, struct enbd_request *request,
	     struct enbd_reply *reply)
{
    int res, err;
    struct nbd_stub_server *server = &self->server;

    DEBUG ("exp->buf, ");

    // PTB len 0 reads are CHKNETs from the client
    if (request->len <= 0) {

	DEBUG ("read sector %Ld len %d\n", request->from >> 9, request->len);
	res = server->check (server);

    }
    else {
	// PTB default case is a straight read
	res = read_data (server, self->buffer, request->len, request->from);
    }

    if (res < 0) {
	ERROR (reply);		// PTB give error reply
	return 0;		// try again
    }

    // PTB what was this for?
    //if (res == 0 && request.len > 0)
    //  goto success;  // try again

    if (res < request->len) {
	ERROR (reply);		// PTB give error reply
	// PTB now tell the server that we had no problem
	return 0;		// try again new
    }

    err = 0;


    DEBUG ("buf(%d bytes)->net, ", request->len);

    DEBUG ("server (%d) prepares reply fields:\n\t magic %#x\n"
	   "\terror %#x\n\thandle %#lx\n\tflags %d\n"
           "\tdigest %.8x%.8x%.8x%.8x\n"
	   "on cmd %d len %d sector %lu\n",
	   self->i,
	   reply->magic,
	   reply->error,
	   (unsigned long) reply->handle,
	   reply->flags,
	   reply->data.digest[0], reply->data.digest[1],
	   reply->data.digest[2], reply->data.digest[3],
	   request->type, request->len, (unsigned long) (request->from >> 9));

    // PTB request write res = request->len bytes from self->buffer
    return res;
}

   /*
    * Error return causes client death. Anything else and client goes
    * round its main loop again.
    */
   static int newproto(struct nbd_server *self) {

      struct enbd_request request;
      struct enbd_reply reply;
      int err = 0;
      int res;
      struct timezone zone;
      struct timeval  time;

           DEBUG("newproto enters and waits for readnet\n"); 

           /*
            * PTB step 1. get client request from network
            */

           err = readnet(self, (char*)&request, sizeof(struct enbd_request));
  
           if (err < sizeof(request)) {
             PERR("net errored on packet. Breaking off.\n"); 
             disconnectme(self);
             err = -1;
             goto fail;
           }
  
           /* PTB un-bigendian and test the request magic */
           if (request.magic != htonl(ENBD_REQUEST_MAGIC)) {
             PERR("Not enough magic in packet. Breaking off.\n"); 
             disconnectme(self);
             err = -1;       /*  PTB can drop instead of erroring out */
             goto fail;
             // PTB it was corrupt, but if it was a write the next bytes
             // on the net will also be bad. We should try and resync,
             // perhaps scanning forwards till we find REQUEST_MAGIC
           }
  
           /* PTB un-bigendian the (64bit) offset of the request */
           switch (sizeof(request.from)) {
           case 4: request.from = ntohl(request.from); 
              break;
           case 8: request.from = ntohll(request.from); 
              break;
           default: 
              PERR("Impossible from field size: %m\n");
              disconnectme(self);
              err = -1;
              goto fail;
           }
  
           /* PTB un-bigendian the request type (R/W/T) */
           request.type = ntohl(request.type);
           /* PTB un-bigendian the requested number of bytes */
           request.len = ntohl(request.len);
           /* PTB un-bigendian the sequence number */
           request.seqno = ntohl(request.seqno);
           DEBUG("newproto treats cmd type %d from sector %d len %d\n",
                 request.type, (int) (request.from >> 9), request.len);
           /* PTB un-bigendian the unique id */
           request.handle = ntohl(request.handle);

           // PTB get the unique id of the request and transfer to reply 
           //     before we inadvertently stamp on it when reply and
           //     request are in the same bufferspace

           reply.handle = request.handle;
           reply.handle = htonl(reply.handle);

           DEBUG("newproto transfered handle %#lx for cmd type %d on sector %d\n",
                 (unsigned long)request.handle, request.type,
                 (int) (request.from >> 9));

           // PTB start constructing the rest of reply. Magic, error and tod

           reply.magic = htonl(ENBD_REPLY_MAGIC);
           reply.error = 0;
           mygettimeofday(&time, &zone);
           // minuteswest is the low bits
           reply.zone = zone.tz_minuteswest;
           reply.zone = htonll(reply.zone);
           // count time in us
           reply.time = time.tv_sec;
           reply.time *= 1000000;
           reply.time += time.tv_usec;
           reply.time = htonll(reply.time);

           DEBUG("newproto got reply TOD for cmd type %d on sector %d\n",
                 request.type, (int) (request.from >> 9));

           memset(&reply.data.digest[0], 0, ENBD_DIGEST_LENGTH);

           DEBUG("newproto zeroed reply digest for cmd type %d on sector %d\n",
                 request.type, (int) (request.from >> 9));

           reply.flags = 0;
  
           // PTB from now on we can reply with an ERROR code
  
           DEBUG("newproto switches for cmd type %d from sector %d len %d\n",
                 request.type, (int) (request.from >> 9), request.len);

           /*
            * PTB Step 2. Treat the incoming request. On read we will
            *     get extra stuff into the buffer too.
            */

           switch (request.type) {
  
/* when write is async one has to do the ack (see later) before writing */
           case WRITE:  /* 1 */

              res = (self->flags & F_ASYNC) ? 0 : do_srv_write(self,&request,&reply);
              break;
  
           case IOCTL: /* 2 IOCTL */
              res = do_srv_ioctl(self,&request,&reply);
              break;

           case CKSUM: /* 3 MD5 */
              res = do_srv_cksum(self,&request,&reply);
              break;

           case SPECIAL: /* 4 */
              res = do_srv_special(self,&request,&reply);
              break;

           default:
           case READ: /* 0 */
              res = do_srv_read(self,&request,&reply);
              break;

           } // PTB appears to be the end of the case switch

           if (res < 0) {
                DEBUG("req type %d FAILED!\n", request.type);
                goto fail;
           }

           /*
            * PTB Step 3. here's the ack, possibly followed by data. If
            * res is positive above, then we have to write data later.
            */

           if (writenet (self, (char *)&reply, sizeof (struct enbd_reply)) < 0) {
	        DEBUG ("failed to write %d bytes to net: %m\n",
                        sizeof (struct enbd_reply));
	        goto fail;
           }

           // PTB do an async write now
           switch (request.type) {
               case  WRITE:
                   if (self->flags & F_ASYNC) {
                        res = do_srv_write(self,&request,&reply);
                        DEBUG("async write from sector %u len %d result %d\n",
                             (unsigned) (request.from >> 9), request.len, res);
                        if (res < 0)
                                goto fail;
                   }
                   break;
           }

           /*
            * PTB Step 4. Any following data.
            */

           if (res > 0) {
               int len = res;

               if (request.type == IOCTL) {
                   DEBUG("writing %dB data to net from %p (%#x..) for indirect IOCTL\n",
                           len,
                           self->buffer,
                           *(unsigned*)self->buffer);
               }

               // PTB there was extra data to be written);
               res = writenet (self, self->buffer, len);

               if (res < len) {
	           DEBUG ("failed to write %dB of %d to net: %m\n",
                           len - res, len);
	           goto fail;
               }

               if (request.type == IOCTL) {
                   DEBUG ("wrote %dB data (%#x..) to net for indirect IOCTL\n",
                           res, *(unsigned *)self->buffer);
               }

               DEBUG ("wrote %d+%d bytes from sector %d to net "
	        "magic %lx handle %lx\n",
	        sizeof (struct enbd_reply), res,
	        (int) (request.from >> 9),
	        (unsigned long) ntohl (reply.magic),
	        (unsigned long) reply.handle);

               return res;
           }
        
           return err = 0;

         fail:
           if (err >= 0)
             err = -1;
           return err;
   }

   static int mainloop(struct nbd_server *self)
   {
      int err = 0;
      const long timeout = self->data_timeout;
   
      DEBUG("Entering request loop!\n");

      /* PTB main loop cycles continuously */

      while (1) {

         struct alarm my_jmp;

      #ifdef DODBG
         static int i;
         fprintf(stderr,"%d.%d: ", self->port, ++i);
      #endif

   	 DEBUG("Server sets 30s timer for mainloop\n");
        
         if (catch_alarm(&my_jmp,timeout)) {

   	   PERR("Server time out waiting 30s in mainloop. Breaking off\n");
             disconnectme(self);
             err = -1;
             goto exit;

         } else {
  
           err = newproto(self);

           DEBUG("Server unsets timer for mainloop\n");
           uncatch_alarm(&my_jmp);

           // PTB error from neproto is fatal
           if (err < 0)
             goto exit;
         }

      } /* PTB endwh */

    exit:
      return err;
   }

  /*
   * generate an ascii signature of length @n in @buf.
   *
   * Seed the sig consistently so that we always get the same sig on
   * the same host with the same resource files.
   *
   * Here we use the canonical IP address and the source file names to
   * generate a hash which serves as the seed.
   *
   * The sig chosen is ascii printable.
   */
  static int getrandsig(struct nbd_server *self, char * buf, int n) {

        int i;
        int seeded = 0;
        int j;
        unsigned char c = 0;
        char hostname[256];

        do {

            int s = 0;
            struct hostent *he;
            struct in_addr ia;

            if (gethostname(hostname, sizeof(hostname)-1) < 0) {
		PERR ("cannot find my own hostname!\n");
                break;
            }

            he = gethostbyname(hostname);
            if (!he) {
		PERR ("cannot resolve hostname <%s> to address\n", hostname);
                break;
            }

            ia.s_addr = *(int*)he->h_addr;
            //PERR("received address <%s> len %d for <%s>\n",
            //        inet_ntoa(ia), he->h_length, hostname); 

            s = ia.s_addr;
            for (i = 0; i < self->nfile; i++) {

                char *name = self->names[i];
                int k;
                int namlen = strlen(name);

                for (j = 0; j < namlen/sizeof(int); j++) {
                    memcpy(&k, &name[j * sizeof(int)], sizeof(int));
                    s ^= k;
                }
                k = 0;
                memcpy(&k, &name[j * sizeof(int)], namlen % sizeof(int));
                s ^= k;
            }

            srandom(s);
            seeded = 1;

        } while (0);

        if (!seeded) {
            // PTB fall back to time of day
            struct timeval tv;
            int s;
            if (mygettimeofday(&tv, NULL) < 0)
                return -ENODEV;
            s = tv.tv_usec + tv.tv_sec;
            srandom(s);
            seeded = 1;
        }
        for (i = 0; i < n; i++) {
          int paranoia = 1000;
          do {
            c = random() & 0x7fff;
            c %= 63;
            if (c < 26) {
              c += 'a';
            } else if (c < 52) {
              c -= 26;
              c += 'A';
            } else if (c < 62) {
              c -= 52;
              c += '0';
            } else {
              c = '_';
            }
          } while (!isdigit(c) && !isalpha(c) && c != '_' && paranoia-- > 0);
          if (paranoia < 0)
            c = 'X';
          buf[i] = c;
        }
        // PTB the 0 terminator is taken care of by the caller.
        //buf[i] = 0;
        return 0;
  }

#ifdef USING_SSL
  static int init_ssl_connection(SSL *con)
  {
        int i;
        const char *str;
        long verify_error;
        static char buf[BUFSIZE];
  	DEBUG("Negotiation of SSL connection...\n");

        if ((i=SSL_accept(con)) <= 0) {
              MSG("ERROR in init_ssl_connection\n");
              DEBUG("ERROR in init_ssl_connection\n");
              verify_error = SSL_get_verify_result(con);
              if (verify_error != X509_V_OK) {
                      MSG("verify error:%s\n",
                              X509_verify_cert_error_string(verify_error));
              } else
                      ERR_print_errors(bio_err);
  	      DEBUG("SSL negotiation failed!\n");
              return 0;
        }

        if (SSL_get_shared_ciphers(con,buf,BUFSIZE))
                 DEBUG("Shared ciphers: %s\n",buf);
        str = SSL_CIPHER_get_name(SSL_get_current_cipher(con));
        DEBUG("CIPHER is %s\n",(str)?str:"(NONE)");
        if (con->hit){
          DEBUG("Reconnection...");
#ifdef TRY_SSL_RECONNECT
          // PTB - we should try this anyway. Looks harmless.
#endif
          SSL_renegotiate(con);
          i = SSL_do_handshake(con);
          DEBUG("%s... Reused session-id\n",
              (i>=0)?"Good handshake!":"Bad handshake"); 
          if (i < 0) {                          // PTB add
  	      DEBUG("SSL negotiation failed!\n");
              return 0;
          }
        }
        if (SSL_ctrl(con,SSL_CTRL_GET_FLAGS,0,NULL) &
                TLS1_FLAGS_TLS_PADDING_BUG) {
                MSG("Peer has incorrect TLSv1 block padding\n");
        }

 	DEBUG("SSL connection ready!\n");
        return 1;
  }
  
  static int init_server_ssl_stuff(struct nbd_server *self) {
    if (nocert)
      {
	s_cert_file  = NULL;
	s_key_file   = NULL;
	s_dcert_file = NULL;
	s_dkey_file  = NULL;
      }

    DEBUG("Initializing SSL context\n");

    meth = SSLv23_server_method();

    SSL_load_error_strings();
    SSLeay_add_ssl_algorithms();
    
    if (! bio_err)
      bio_err = BIO_new_fp(stderr,BIO_NOCLOSE);
    
    ctx = SSL_CTX_new(meth);
    if (! ctx)
      {
        ERR_print_errors(bio_err);
        goto end;
      }
    
    SSL_CTX_set_quiet_shutdown(ctx,1);
    SSL_CTX_set_options(ctx,0);
    SSL_CTX_sess_set_cache_size(ctx,128);
    
    if (!SSL_CTX_load_verify_locations(ctx,CAfile,CApath) ||
        !SSL_CTX_set_default_verify_paths(ctx))
      {
        ERR_print_errors(bio_err);
        /* goto end; */
      }
        
    DEBUG("  loading cert from file %s\n", s_cert_file);

    if (!set_cert_stuff(ctx,s_cert_file,s_key_file))
      goto end;
    if (s_dcert_file)
      {
        if (!set_cert_stuff(ctx,s_dcert_file,s_dkey_file))
          goto end;
      }

    DEBUG("  preferred cipher is (%s)\n",cipher);

    if (cipher)
      SSL_CTX_set_cipher_list(ctx,cipher);
    SSL_CTX_set_verify(ctx,s_server_verify,verify_callback);
    SSL_CTX_set_session_id_context(ctx,(void*)&s_server_session_id_context,
                                   sizeof s_server_session_id_context);

    SSL_CTX_set_client_CA_list(ctx,SSL_load_client_CA_file(CAfile));

    DEBUG("... SSL initialized\n");
    
    return 0;

  end:
    DEBUG("Error in SSL initialization\n");
    if (ctx) SSL_CTX_free(ctx);
    return -1;

  }

#endif 

  static void setsighandler(void (*sighandler)(int)) {
       // PTB handle all signals in one sighandler
       int k = 0;
       for (k = 1; k < 30; k++) {
#if defined(__GLIBC__) && __GLIBC__ >= 2
    
         struct sigaction sa = { 
	#ifndef sa_handler
		sa_handler: {sighandler},
	#endif
		sa_mask:    {{0}},
		sa_flags: SA_RESTART,
		//sa_restorer: NULL,
	 };
	#ifdef sa_handler
	 sa.sa_handler = sighandler;
	#endif
#else
         struct sigaction sa = { {sighandler}, 0, SA_RESTART, NULL };
#endif
         sigfillset(& sa.sa_mask);
         sigaction(k, & sa, NULL);
       }

       MSG("server (%d) set %ssignal handlers for %s server %d\n", self->i,
          sighandler == SIG_DFL ? "default " :
          sighandler == SIG_IGN ? "ignore " :
          "new ", 
          self->type == ENBD_SERVER_MASTER ? "master" :
          self->type == ENBD_SERVER_SESSION? "session":
          self->type == ENBD_SERVER_SLAVE? "slave":
          "unknown",
          self->pid);                  
  }

  static int talk_hello (struct nbd_server *self)
  {
        char * id = NULL;

     	DEBUG("hello...\n");

        switch (self->i) {
            case -2:
            case -1: id = "nbd-client";
                     break;
            default: id = "old-friend";
                     break;
        }
	if (writenet (self, "helo ", 5) < 0) {
		PERR ("Introduction failed: %m\n");
		return -2;
	}
	if (writenet (self, id, strlen(id)) < 0) {
		PERR ("Introduction failed: %m\n");
		return -2;
	}
	if (writenet (self, "\n", 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -2;
	}

	MSG ("server (%d) sent hello ok\n", self->i);
	return 0;
  }              

  static int talk_passwd (struct nbd_server *self)
  {
        int err;

	DEBUG ("OK!, passwd...\n");

	err = writenet (self, "pass " INIT_PASSWD "\n", 5 + ENBD_PASSWD_LEN + 1);

	if (err < 0) {
		PERR ("Introduction failed/1: %m\n");
		return -3;
	}

	MSG ("server (%d) sent passwd ok\n", self->i);
        return 0;
  }

  static int talk_magic (struct nbd_server *self)
  {
	char magic[ENBD_MAGIC_LEN + 6];

	DEBUG ("OK!, cliserv_magic...\n");

	if (writenet (self, "mgck ", 5) < 0) {
		PERR ("Introduction failed/2: %m\n");
		return -4;
	}
	sprintf (magic, "0x%.*Lx\n", ENBD_MAGIC_LEN,
		 (unsigned long long) cliserv_magic);
	if (writenet (self, magic, ENBD_MAGIC_LEN + 3) < 0) {
		PERR ("Introduction failed/2: %m\n");
		return -4;
	}

	MSG ("server (%d) got cliserv magic ok\n", self->i);
	return 0;
  }

  static int talk_device (struct nbd_server *self)
  {
	char rdev_host[ENBD_RDEV_LEN + 6];
	unsigned long rdev;

	DEBUG ("OK!, device ...\n");

	if (writenet (self, "rdev ", 5) < 0) {
		PERR ("Introduction failed: %m\n");
		return -17;
	}

	if (readnet (self, rdev_host, ENBD_RDEV_LEN + 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -17;
	}

	sscanf (rdev_host, "%lx\n", &rdev);

	MSG ("server (%d) received id device %lx ok\n", self->i, rdev);

	sprintf (rdev_host, "%*.*lx\n", ENBD_RDEV_LEN, ENBD_RDEV_LEN, rdev);
	if (writenet (self, rdev_host, ENBD_RDEV_LEN + 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -17;
	}
        return 0;
  }

  static int talk_size (struct nbd_server *self)
  {
	char size_host[36];

	DEBUG ("OK!, size (%Lu)...\n", self->size);

	if (writenet (self, "size ", 5) < 0) {
		PERR ("Introduction failed: %m\n");
		return -5;
	}
	sprintf (size_host, "%.*Lu\n", ENBD_SIZE_LEN,
		 (unsigned long long) self->size);
	if (writenet (self, size_host, 32) < 0) {
		PERR ("Introduction failed: %m\n");
		return -5;
	}

	MSG ("server (%d) sent size %Lu ok\n", self->i,
	     (unsigned long long) self->size);
	return 0;
  }

  static int talk_sign (struct nbd_server *self)
  {
	DEBUG ("OK!, signature...\n");

	if (writenet (self, "sign ", 5) < 0) {
		PERR ("Introduction failed: %m\n");
		return -6;
	}
	if (writenet (self, signature, ENBD_SIGLEN) < 0) {
		PERR ("Introduction failed: %m\n");
		return -6;
	}
	if (writenet (self, "\n", 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -6;
	}

	MSG ("server (%d) sent sig ok\n", self->i);
	return 0;
  }

  static int talk_ro (struct nbd_server *self)
  {

	int ro = (self->flags & F_READONLY) ? 1 : 0;

	DEBUG ("OK!, ro/rw... ");
	if (writenet (self, "ro ", 3) < 0) {
		PERR ("Introduction failed: %m\n");
		return -10;
	}
	if (writenet (self, ro ? "yes\n" : "no \n", ENBD_RO_LEN + 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -10;
	}

	MSG ("server (%d) suggested ro flags %d ok\n", self->i, ro);
	return 0;
  }

  static int talk_blksize (struct nbd_server *self)
  {
	char blksize_host[ENBD_BLKSIZE_LEN + 6];
	unsigned long blksize;

	DEBUG ("OK!, blksize...\n");

	if (writenet (self, "bksz ", 5) < 0) {
		PERR ("Introduction failed: %m\n");
		return -7;
	}
	if (readnet (self, blksize_host, ENBD_BLKSIZE_LEN + 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -7;
	}
	sscanf (blksize_host, "%lu", &blksize);

	MSG ("server (%d) received blksize %ld ok\n", self->i, blksize);

	if (blksize > self->blksize)
		self->blksize = blksize;

	sprintf (blksize_host, "%*lu\n", ENBD_BLKSIZE_LEN,
		 (unsigned long) self->blksize);

	if (writenet (self, blksize_host, ENBD_BLKSIZE_LEN + 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -8;
	}

	MSG ("server (%d) sent/negotiated blksize %d ok\n", self->i,
	     self->blksize);
	return 0;
  }

  static int talk_pulse_intvl (struct nbd_server *self)
  {
	char pulse_intvl_host[ENBD_PULSE_INTVL_LEN + 6];
	unsigned long pulse_intvl;
	DEBUG ("(%d) OK!, pulse_intvl...\n", self->blksize);

	if (writenet (self, "rqto ", 5) < 0) {
		PERR ("Introduction failed: %m\n");
		return -7;
	}
	if (readnet (self, pulse_intvl_host, ENBD_PULSE_INTVL_LEN + 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -7;
	}
	sscanf (pulse_intvl_host, "%lu", &pulse_intvl);

	MSG ("server (%d) received pulse_intvl %lu ok\n", self->i, pulse_intvl);

	if (pulse_intvl > self->pulse_intvl)
		self->pulse_intvl = pulse_intvl;

	self->data_timeout = 3 * self->pulse_intvl;

	sprintf (pulse_intvl_host, "%*lu\n", ENBD_PULSE_INTVL_LEN,
		 self->pulse_intvl);
	if (writenet (self, pulse_intvl_host, ENBD_PULSE_INTVL_LEN + 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -8;
	}

	MSG ("server (%d) sent/negotiated pulse interval %ld ok\n",
	     self->i, self->pulse_intvl);
	return 0;
  }

  static int talk_nport (struct nbd_server *self)
  {
	char nport_host[ENBD_NPORT_LEN + 6];

	DEBUG ("(%lu) OK!, nconn...\n", self->pulse_intvl);

	if (writenet (self, "nprt ", 5) < 0) {
		PERR ("Introduction failed: %m\n");
		return -10;
	}

	if (readnet (self, nport_host, ENBD_NPORT_LEN + 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -11;
	}
	sscanf (nport_host, "%d", &self->nconn);

	if (writenet (self, nport_host, ENBD_NPORT_LEN + 1) < 0) {
		PERR ("Introduction failed: %m\n");
		return -10;
	}

	MSG ("server (%d) agreed %d channels ok\n", self->i, self->nconn);
        return 0;
  }

  static int talk_port (struct nbd_server *self)
  {
	static struct sockaddr_in addrin;
	int addrinlen = sizeof (addrin);
	char port_host[ENBD_PORT_LEN + 6];
        unsigned short port = 0;
        short sock = -1;

	if (self->port <= 0)
		return 0;

	// can I find a free port number?

	if (self->freeportmin > 0)
		port = self->freeportmin;
	else
		port = self->port + 1;

	sock = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP);

	addrin.sin_family = AF_INET;
	addrin.sin_addr.s_addr = 0;
	addrin.sin_port = htons (port);

	if (sock < 0) {
		PERR ("server (%d) cannot open new socket\n", self->i);
		return -12;
	}

	while (port && port < self->freeportmax) {
		int err = bind (sock, (struct sockaddr *) &addrin,
				addrinlen);
		if (err >= 0) {
			DEBUG
			 ("server (%d) found free port at %d\n",
			  self->i, port);
			if (listen (sock, 1) >= 0)
				break;
		}
		DEBUG
		 ("server (%d) skipped occupied port at %d: %m\n",
		  self->i, port);
		addrin.sin_port = htons (++port);
	}			// endwh

	if (!port || port >= self->freeportmax) {
		PERR ("server (%d) finds no ports\n", self->i);
		return -13;
	}
	MSG ("server (%d) selected free port at %d\n", self->i, port);

	if (writenet (self, "port ", 5) < 0) {
		PERR ("Introduction failed: %m\n");
		return -14;
	}
	sprintf (port_host, "%*hu\n", ENBD_PORT_LEN, port);
	if (writenet (self, port_host, 6) < 0) {
		PERR ("Introduction failed: %m\n");
		return -15;
	}
	MSG ("server (%d) posted port %d ok\n", self->i, port);

        // replace the primary socket with our new one
        DEBUG("server (%d) replaces primary port and socket\n", self->i);
        // don't ask me why, but close mucks up local seeks
        dup2(sock,self->socket);
        //self->socket = sock;
        close(sock);
        // this seems to be unnecessary! FIXME
        self->port   = port;

	return 0;
  }

static int introduction(struct nbd_server *self) {

         int err = 0, i;
         struct alarm my_jmp;
         unsigned long timeout = self->negotiate_timeout;
         int (*f)();
         /* PTB here's the intro sequence */
         static int (*talk_table[])() = {
           talk_hello,
           talk_passwd,
	   talk_magic,
	   talk_device,
	   talk_size,
	   talk_sign,
	   talk_ro,
	   talk_blksize,
	   talk_pulse_intvl,
	   talk_nport,
	   talk_port,
           NULL,
         };

 	 DEBUG("Server begins introduction on port %d\n", self->port);
         setsighandler(SIG_DFL);


 	 DEBUG("Server sets timer for introduction\n");
         if (catch_alarm(&my_jmp, timeout)) {
 	    PERR("Server timed out in introduction\n");
            err = -15;
            goto exit_fail;

         } else { // PTB begin giant block
  
           if (self->sock > 0)
             setmysockopt(self->sock);

   #ifdef USING_SSL
           if (using_ssl) {
  
   	   DEBUG("(%d) Socket connected, now initialize the SSL connection\n",
               self->i);
  
               if (!self->con)  // make new ssl
                 self->con = (SSL *) SSL_new(ctx);
               DEBUG("(%d) SSL connect %p made\n",
                 self->i, self->con);
  
               if (!SSL_clear(self->con))
                 DEBUG("SSL_clear returned 0, are we using session id?\n");
    
               SSL_set_accept_state(self->con);
    
               if  (!SSL_set_fd(self->con,self->sock)
                 || !init_ssl_connection(self->con)) {
                  PERR("(%d) SSL session uninitialized\n", self->i);
                  err = -1;
                  goto fail;
               }
           
               DEBUG("(%d) SSL connection initialized (con = %p)\n",
                  self->i, self->con);

              
               // PTB tell the stream object about the con
               self->stream.con = self->con;
           } // PTB fi using ssl
  #endif
  
           for (i = 0; f = talk_table[i], f; i++) {
                err = (*f)(self);
                if (err  < 0)
                    goto fail;
           }
           goto success;

         success:
           uncatch_alarm(&my_jmp);
           goto exit_success;

         fail:
           uncatch_alarm(&my_jmp);
           goto exit_fail;
         } // PTB end of giant block! 
 
exit_success:

 	 DEBUG("(%d) OK!\n Introduction done, disabling timer and return\n",
                      self->nconn);

         return err = 0;

exit_fail:
 	 DEBUG("Introduction failed, sock descriptor is %d\n", self->sock);
         disconnectme(self);
 	 DEBUG("Introduction failed, disabling timer and return\n");
         return err;
  } 
  
  static int negotiate(struct nbd_server *self) {

         int err = 0;
         struct alarm my_jmp;
         unsigned long timeout = self->negotiate_timeout;
         int i, (*f)(struct nbd_server *);
         static int (*talk_table[])() = {
           talk_hello,
           talk_passwd,
	   talk_magic,
	   talk_sign,
           NULL,
         };

 	 DEBUG("Server begins negotiation on port %d with primary socket %d\n",
                (unsigned short)self->port, self->socket);
         setsighandler(SIG_DFL);

         if (catch_alarm(&my_jmp, timeout)) {
 	   PERR("Server timed out in negotiate\n");
           err = -15;
           goto exit_fail;
         } else {
  
           if (self->sock > 0) {
   	   DEBUG("Server closes pending connections\n");
             disconnectme(self);
           }
      
   	 DEBUG("Server sets timer for negotiation\n");
  
         // PTB this where we make the session socket

           // if port==0 we use stdin as socket, i.e. sock=0
           if (self->port != 0) {
             // apparently we reuse the primary socket
             err = connectme(self);
           } else {
             self->sock = 0;
             err = 0;
           }
      
           if (err < 0) {
             PERR("Server failed connectme\n");
             // PTB go round again, after pause
             err = -1;
             goto fail;
           }
  
           if (self->sock > 0) {

             setmysockopt(self->sock);

           }

           // PTB reset the stream abstraction to match socket we got
           //     during the negotiation
#ifdef USING_SSL
           if (using_ssl) {
               initstream(&self->stream,self->data_timeout,&ctx);
               self->con = self->stream.con;
           } else
#endif
               initstream(&self->stream,self->data_timeout,NULL);
           self->stream.sock = self->sock;
 
  #ifdef USING_SSL
           if (using_ssl) {
  
   	   DEBUG("(%d) Socket connected, now initialize the SSL connection\n",
               self->i);
  
               if (!self->con)  // make new ssl
                 self->con = (SSL *) SSL_new(ctx);
               DEBUG("(%d) SSL connect %p made\n", self->i, self->con);
  
               if (!SSL_clear(self->con))
                 DEBUG("SSL_clear returned 0, are we using session id?\n");
    
               SSL_set_accept_state(self->con);
    
               if  (!SSL_set_fd(self->con,self->sock)
                  || !init_ssl_connection(self->con)) {
                  PERR("(%d) SSL session uninitialized\n", self->i);
                  err = -1;
                  goto fail;
               }
           
               self->stream.con = self->con;
               DEBUG("(%d) SSL connection initialized (con = %p)\n",
                     self->i, self->con);
           }
  #endif
   
   	 DEBUG("Setting TERM and INT handlers\n");
  
           signal(SIGTERM,closenet);
           signal(SIGINT, closenet);
  
    	 DEBUG("Go on with the enbd negotiation: ");
    
         for (i = 0; f = talk_table[i], f; i++) {
              err = (*f)(self);
              if (err  < 0)
                  goto fail;
         }


/*
 	 MSG("OK!, server pid (%x)...\n", self->pid);

           if (writenet(self, "svpd", 4) < 0) { // PTB who knows why 4 not 5?
                PERR("Negotiation failed/7: %m\n");
                err = -7;                       // PTB it didn't work with 5!
                goto fail;
           }
           if (1) {
             int svpd = self->pid;
             svpd = htonl(svpd);
             if (writenet(self, (char *)&svpd, sizeof(svpd)) < 0) {
                PERR("Negotiation failed/8: %m\n");
                err = -7;
                goto fail;
             }
           }
*/

           goto success;
 

         success:
           uncatch_alarm(&my_jmp);
           goto exit_success;

         fail:
           uncatch_alarm(&my_jmp);
           goto exit_fail;
         }

exit_success:
 	 DEBUG("OK!\n Negotiation done, disabling timer and return\n");
         return err = 0;

exit_fail:
 	 DEBUG("Negotiation failed, sock descriptor is %d\n", self->sock);
         disconnectme(self);
 	 DEBUG("Negotiation failed, disabling timer and return\n");
         return err;
  }

  static void setself(struct nbd_server * me) {
        self = me;
  }

  static void invalidate_slave_list(struct nbd_server *self) {
        int i;
        for (i = 0; i < self->nconn; i++) {
             self->child[self->i] = -1; // PTB for completeness
        }
        self->nconn = 0;
  }


  /*
   * We, the session master, launch a slave
   */
  static int launch(struct nbd_server * self) {

         int pid = -1;
         int err = 0;
         struct nbd_file * file = & self->file;

         if (!self) {
           PERR("I tried to launch a nonexistant server struct!\n");
           return -1;
         }

         // detach the child
         fflush(NULL);
         pid = fork();

         switch(pid) {
         case 0:  // PTB child continues
            break;
         case -1: // PTB fork error
            return self->pid = -1;
         default: // PTB parent returns with pid
            return self->pid = pid;
         }

         // child is a slave
         setself(self);
         self->pid = getpid();
         self->sock = -1;                  // PTB invalidate current descriptor
         self->type = ENBD_SERVER_SLAVE;

         // PTB invalidate slave list inherited from parent
         invalidate_slave_list(self);

         // PTB copy instead of share the resource FDs to avoid file
         // pointer mixups between processes
         //file->reopen(file);
         file->close(file);
         file->flags |= F_OPENWANTED;

         // PTB direct connection without cache
         init_fileserver(&self->server, self->i,
             self->size, file, self->flags & F_READONLY, NULL);
         
         self->flags &= ~F_NEGOTIATED;

         // PTB try to set up initial connection
             
         while (! (self->flags & F_NEGOTIATED)) {
           static int retry_negotiation_count;
           if (retry_negotiation_count++ >= ENBD_MAX_LIVES) {
             retry_negotiation_count = 0;
             goto die_horribly;
           }
           err = negotiate(self);
           if (err >= 0)  {
                 // PTB success
                 self->flags |= F_NEGOTIATED;
#ifdef HAVE_MLOCKALL
                 mlockall(MCL_CURRENT|MCL_FUTURE);
#endif
                 goto mainloop;
           }
           // up to 10s wait
           microsleep(rand() / (RAND_MAX/10000000));
         }

       mainloop:

         // handle usr1 and usr2 signals
         self->sighandler = slavesighandler;
         setsighandler(self->sighandler);

         // do protocol
         err = mainloop(self);

         if (err >= 0) 
               // PTB success = planned termination
               goto die_peacefully;
           
         goto die_horribly;

       die_peacefully:
         /* child death - ptb */

         // PTB attempt delay to avoid fast SIGCHLDs in sequence
 	 DEBUG("Server tries to sleep 5 after planned protocol exit\n");
         fflush(NULL);
         microsleep(5000000);
         return err;

       die_horribly:
 	 DEBUG("Server tries to sleep 10 after protocol error\n");
         fflush(NULL);
         microsleep(9000000);
         kill(getpid(),SIGTERM);
         microsleep(1000000);
         kill(getpid(),SIGKILL);
         exit(2);
         return 0;
  }

       
  static int restart(struct nbd_server * self, int died) {
 
    // check and restart a known slave pid. (or all if pid = -1)
 
    int i = 0;
    int count = 0;

    if (self->type != ENBD_SERVER_SESSION) {
 	 PERR("Server %d tried to relaunch slave when not session master\n",
            self->i);
         return -1;
    }
 
    for ( i = 0; i < self->nconn; i++)  {
 
         int pid  = self->child[i];
 
         DEBUG("server (%d) main childminder checking pid %d\n", self->i, pid);
 
         if ( pid < 0 || kill(pid,0) < 0 || died == pid )  {
 
             self->child[i] = -1;
 
             MSG("server (%d) slave pid %d is down, launching new\n",
                    self->i, pid);
             pid = launch(self->session[i]);
             if (pid > 0) {
               MSG("server (%d) launched slave pid %d\n", self->i, pid);
               self->child[i] = pid;
               count ++;
             }
         }
    }
    return count;
  }

  static void nullsigchldhandler() {
 
    // reap dead children.
 
    while (1) {
      int pid = wait4(-1,0,WNOHANG,0);
      if (pid <= 0)
        break;
      PERR("child with pid %d died, remove from DB?\n", pid);
    }
 
  }                                

  static void sigchldhandler() {
 
    // reap and restart dead children.
 
    int count = 0;
 
    while (1) {
      int pid = wait4(-1,0,WNOHANG,0);
      if (pid <= 0 )
        break;
      count += restart(self, pid) ;
    }
 
    if (count <= 0)
      // an unknown child died so check all
      count += restart(self, -1) ;
  }                                

 /*
  * use the pids list (slave pids) db to get a list of child pids, not
  * the spids db (session pids).
  */
  static void sessionpropagate(struct nbd_server * self, int n) {
            int i;
            for (i = 0; i < self->nconn; i++) {
              int pid = self->child[i];
              if (pid <= 0)
                continue;
              if (kill(pid,0) < 0)
                 continue;
              kill(pid,n);
              MSG( "server (%d) propagated signal %d to live process %d\n",
                    self->i, n, pid);
            }
  }

 /*
  * use the spids (session pids) db to get a list of child pids, not
  * the pids list (slave pids).
  */
 static void masterpropagate(struct nbd_server * self, int n) {
          int i;
          int pid;
          struct nbd_db * spids_db = &self->spids; 

          for (
            i = spids_db->first(spids_db,(void**)&pid,NULL);
            i >= 0;
            i = spids_db->next(spids_db,i,(void**)&pid,NULL)
          ) {
              if (pid <= 0)
                continue;
              if (kill(pid,0) < 0)
                 continue;
              kill(pid,n);
              MSG( "server (%d) propagated signal %d to live process %d\n",
                    self->i, n, pid);
          }
  }

  static void propagate(struct nbd_server * self, int n) {

            switch (self->type) {
              case ENBD_SERVER_MASTER:
                 masterpropagate(self, n);
                 break;
              case ENBD_SERVER_SESSION:
                 sessionpropagate(self, n);
                 break;
            }
  }

  static void mastersighandler(int n) {
 
        if (self->type != ENBD_SERVER_MASTER) {
            PERR( "server (%d) activates master sighandler when not master\n",
               self->i);
            return;
        }

        switch (n) {
        case SIGTTIN:
        case SIGTTOU:
        case SIGPIPE:
        case SIGPWR:
        case SIGHUP:
            MSG( "server (%d) sighandler ignores signal %d\n", self->i, n);
            break;
        case SIGCHLD:
            MSG( "server (%d) sighandler acks SIGCHLD\n", self->i);
            nullsigchldhandler();
            break;
        case SIGTERM:
        case SIGINT:
        case SIGILL:
            MSG( "server (%d) sighandler propagates signal %d\n", self->i, n);
            propagate(self,n);
            MSG( "server (%d) sighandler terminates safely on signal %d\n",
                   self->i, n);
            exit(0);
        case SIGSEGV:
        case SIGBUS:
            MSG( "server (%d) sighandler terminates safely on signal %d\n",
                   self->i, n);
            exit(0);
        default:     // PTB ignore 
            MSG( "server (%d) sighandler ignores signal %d \n", self->i, n);
            break;
        }                
  }            

  static void sessionsighandler(int n) {

         if (self->type != ENBD_SERVER_SESSION) {
            PERR( "server (%d) activates session sighandler "
               " when not session manager\n", self->i);
            return;
        }

        switch (n) {
        case SIGHUP:  // PTB ignore
            propagate(self,n);
            MSG( "server (-1) sighandler propagates signal %d\n", n);
            break;
        case SIGTTIN:
        case SIGTTOU:
        case SIGPIPE:
            MSG( "server (-1) sighandler ignores signal %d\n", n);
            break;
        case SIGCHLD:
            MSG( "server (-1) relaunches child after SIGCHLD \n");
            sigchldhandler();
            break;
        case SIGTERM:
        case SIGINT:
            propagate(self,n);
            MSG( "server (-1) sighandler propagates signal %d\n", n);
            microsleep(1000000);
            propagate(self,SIGKILL);
            // PTB drop thru to SEGV and exit(0)
        case SIGSEGV:
        case SIGBUS:
            MSG( "server (-1) sighandler terminates manager safely\n");
            exit(0);
        default:     // PTB ignore 
            MSG( "server (-1) sighandler ignores signal %d \n", n);
            break;
        }                
  }            


  void init_slave( struct nbd_server * self,
       int i,
       short socket,
       short port,
       u64 size,
       unsigned long flags,
       int nfile,
       char * names[],
       int blksize,
       int mode,
       struct nbd_file *file,
       int nconn,
       int child[],
       struct nbd_server *session[],
       int negotiate_timeout,       /* PTB die and restart if not done */
       long pulse_intvl,            /* PTB heartbeat */
       int data_timeout,            /* PTB reply from client */
       char *buffer,
       void * shmem,                /* PTB shared region */
       int ordered_write_timeout,
       s64 sizes[],
       char *lockdir,
       char *statdir,
       long sync_intvl,
       long cache_lim
       ) {

       int j;

       self->i = i;
       self->socket = socket;    // PTB primary socket
       self->port = port;
       self->size = size;
       self->flags = flags;
       self->nfile = nfile;
       if (names)
         for (j = 0; j < nfile; j++) {
           self->names[j] = names[j];
         }
       if (sizes)
         for (j = 0; j < nfile; j++) {
           self->sizes[j] = sizes[j];
         }
       self->blksize
                   = blksize;
       self->mode  = mode;

       self->file  = *file;
       self->nconn = nconn;
       if (child)
         for (j = 0; j < nconn; j++) 
           self->child[j]    = child[j];
       if (session)
         for (j = 0; j < nconn; j++) 
           self->session[j] = session[j];

       self->negotiate_timeout = negotiate_timeout;
       self->pulse_intvl = pulse_intvl;
       self->data_timeout = data_timeout;
       self->sighandler  = SIG_DFL;
       self->type  = ENBD_SERVER_SLAVE;
       self->buffer= buffer;
       self->lockdir = lockdir;
       self->statdir = statdir;
       self->sync_intvl = sync_intvl;
       self->cache_lim = cache_lim;

       // PTB shared mem open
       self->shmem = shmem;

       self->ordered_write_timeout
                   = ordered_write_timeout;
  }

/*
  static int send(int sock, char * buffer, int size) {
       int offset = 0;
       while (size > 0) {
      
         int res = write(sock, buffer+offset, size);
         if (res <= 0) {
            PERR("server (%d) failed to write %d bytes to socket %d\n",
                  self->i, size, sock);
            break;
         }
         size   -= res;
         offset += res;
       } // ewhil
       return offset;
  }
*/

static int
do_ife (struct interface *ife,unsigned long ipaddr)
{
    unsigned long ifaddr, ifmask, ifneta;

    //DEBUG ("server (%d) if %s\n", self->i, ife.getname (&ife));

    if (!ife->has_ip)
	return 0;		// not up, certainly

    ifaddr = ife->getaddr (ife);
    ifmask = ife->getmask (ife);
    ifneta = ifaddr & ifmask;

    DEBUG ("translated if %s addr %08lx\n", ife->getname (ife), ifaddr);
    DEBUG ("translated if %s mask %08lx\n", ife->getname (ife), ifmask);
    DEBUG ("translated if %s neta %08lx\n", ife->getname (ife), ifneta);
    DEBUG ("translated remote addr %08lx\n", (unsigned long) htonl (ipaddr));
    DEBUG ("translated masked remote addr %08lx\n", ifmask & htonl (ipaddr));

    // PTB need to check addr and netmask against target ip
    if ((ifmask & htonl (ipaddr)) != ifneta) {
	//DEBUG("server (%d) skips if %s\n", self->i, ife.getname(&ife));
	//return 0;
    }
    if ((htonl (0xff000000) & htonl (ipaddr)) == htonl (0x7f000000)) {
	// it's a loopback target address which won't route
	if ((htonl (0xff000000) & ifaddr) != htonl (0x7f000000)) {
	    // but the if isn't a loopback interface, so skip it
	    //DEBUG("server (%d) skips if %s\n", self->i, ife.getname(&ife));
	    // PTB next interface
	    return 0;
	}
    }
    if ((htonl (0xff000000) & ifaddr) == htonl (0x7f000000)) {
	// it's a loopback interface
	if ((htonl (0xff000000) & htonl (ipaddr)) != htonl (0x7f000000)) {
	    // but the target isn't a loopback address, so skip it
	    //DEBUG("server (%d) skips if %s\n", self->i, ife.getname(&ife));
	    // PTB next interface
	    return 0;
	}
    }
    return 0;
}

static void
notify_clients (struct nbd_server *self)
{
    struct nbd_statfile *statfile = &self->statfile;
    struct servent *servent;
    int port;

    if ((servent = getservbyname ("enbd-cstatd", "tcp")) == NULL
    &&  (servent = getservbyname ("nbd-cstatd", "tcp")) == NULL) {
        static short count;
        if (count++ <= 0)
            PERR (
            "server (%d) can't find enbd- or nbd-cstatd in /etc/services\n",
            self->i);
        return;
    }

    port = ntohs (servent->s_port);

    if (statfile->lock (statfile) < 0)
	return;

    MSG ("server (%d) locked %s\n", self->i, statfile->statfile);

    if (statfile->open (statfile) < 0) {
	PERR ("server (%d) could not open %s\n", self->i, statfile->statfile);
	statfile->unlock (statfile);
	MSG ("server (%d) unlocked %s\n", self->i, statfile->statfile);
	return;
    }

    statfile->reset (statfile);

    // PTB for each session in statfile
    while (1) {

	unsigned long ipaddr;	// host order
	int err;
	int sock;
	struct sockaddr_in saddr;
	int size;
	const int buflen = 256;
	char buffer[buflen];
	int offset = 0;
	struct interface ife;

	// get next client to notify
	err = statfile->next (statfile, &ipaddr);

	if (err < 0)
	    break;

	sock = socket (PF_INET, SOCK_STREAM, IPPROTO_TCP);

	if (sock < 0) {
	    PERR ("server (%d) failed to make new socket\n", self->i);
	    // PTB try next conffile entry
	    continue;
	}

	// PTB connect to the IP client_ips file at the enbd-cstatd port
	saddr = ((typeof (saddr)) {
	  sin_family: AF_INET,
          sin_port: htons (port),
          sin_addr: { s_addr: htonl (ipaddr), } ,
          });

	err = connect (sock, (struct sockaddr *) &saddr, sizeof (saddr));

	if (err < 0) {
	    PERR ("server (%d) failed connect to %s on port %d\n",
		  self->i, inet_ntoa (saddr.sin_addr), port);
	    // PTB try next conffile entry
	    continue;
	}

	init_interface (&ife);
	if (ife.open (&ife) < 0) {
	    PERR ("server (%d) can't read kernel if list\n", self->i);
	    close (sock);
	    // PTB hopeless so give up altogether
	    return;
	}

	offset += sprintf (buffer + offset, "notice server-start %hd",
			   self->port);
	if (offset > buflen - 40)
	    // PTB the buffer is just way to short. Give up now
	    return;

	while ((err = ife.next (&ife)) >= 0) {
            if (do_ife(&ife, ipaddr) < 0)
                break;
            offset += sprintf (buffer + offset, " %s", ife.sprintaddr (&ife));
            if (offset > buflen - 40)
	        // PTB stop writing interfaces in buffer
                break;
	}
	offset += sprintf (buffer + offset, " \n");

	ife.close (&ife);

	offset += sprintf (buffer + offset, "quit \n");

	size = strlen (buffer);
	if (send (sock, buffer, size, MSG_NOSIGNAL) < size) {
	    struct in_addr in = { s_addr:saddr.sin_addr.s_addr };
	    PERR ("server (%d) could not send %d bytes to %s\n",
		  self->i, size, inet_ntoa (in));
	    close (sock);
	    continue;
	}

	close (sock);
	{
	    struct in_addr in = { s_addr:saddr.sin_addr.s_addr };
	    MSG ("server (%d) pinged service enbd-cstatd at %s:%d\n",
		 self->i, inet_ntoa (in), port);
	}
	MSG ("with news \"%s\"\n", buffer);

    }				// ewhile 1
    statfile->close (statfile);
    statfile->unlock (statfile);
    MSG ("server (%d) unlocked %s\n", self->i, statfile->statfile);

}

  static void 
  init_master(struct nbd_server * self, unsigned long ipaddr) {
        
          int shmemfd;
          int shmemsz;

          int open_tmpfile(unsigned long sz) {
              FILE *f = tmpfile();
              int fd;
              if (!f) {
                  PERR("could not open a tmpfile\n");
                  return -1;
              }
              fd = fileno(f);
              if (fd < 0) {
                  PERR("got bad shmemfd descriptor from tmpfile\n");
                  return -1;
              }
              if (lseek(fd, sz - 1, SEEK_SET) != sz - 1) {
                  close(fd);
                  return -1;
              }
              if (write(fd,"",1) < 1) {
                  close(fd);
                  return -1;
              }
              if (lseek(fd, 0, SEEK_SET) != 0) {
                  close(fd);
                  return -1;
              }
              return fd;
          };

          self->caddr = ipaddr;
          self->type  = ENBD_SERVER_SESSION;
          self->pid   = getpid();
          self->i     = -1;
          self->shmem = NULL;

           // PTB we share this with daughter processes
          if (self->shmem != NULL && self->shmem != (void *)MAP_FAILED) {
              DEBUG("asked to init while shmem (0x%lx) already exists\n",
                    (unsigned long)self->shmem);
              goto end;
          }

          // PTB round up to page size as its aligned to pages
          shmemsz = ((sizeof (struct nbd_shmem) >> 12) + 1) << 12;

          // PTB make a bit of shared memory to communicate over.
          shmemfd = open("/dev/zero", O_RDWR);
          if (shmemfd < 0) {
              PERR("bad file descriptor from open of /dev/zero %d!\n", shmemfd);
              goto end;
          }

          self->shmem = mmap(0, shmemsz, PROT_READ|PROT_WRITE,
              MAP_SHARED, shmemfd, 0);
   
          if (self->shmem == (void *)MAP_FAILED || self->shmem == NULL)  {
              DEBUG("mmap failed on /dev/zero fd %d: %m\n", shmemfd);
              self->shmem = NULL;
              close(shmemfd);
              shmemfd = open_tmpfile(shmemsz);
              if (shmemfd < 0) {
                  PERR("mmap failed on /dev/zero and make tmpfile failed!\n");
                  goto end;
              }
              self->shmem = mmap(0, shmemsz, PROT_READ|PROT_WRITE,
                  MAP_SHARED, shmemfd, 0);
          }
          if (self->shmem == (void *)MAP_FAILED || self->shmem == NULL)  {
              PERR("mmap failed on both /dev/zero and tmpfile! %m\n");
              self->shmem = NULL;
              if (shmemfd >= 0)
                  close(shmemfd);
              goto end;
          }
          
          init_shmem(self->shmem, shmemsz, shmemfd, 0, self->cache_lim);

          //close(shmemfd); // keep it open for fcntl locking in the object
end:
          // PTB set a session identifier for the file locking
          self->file.pid = self->pid;
#ifdef HAVE_MLOCKALL
          mlockall(MCL_CURRENT|MCL_FUTURE);
#endif
  }

  static void
  removepidfile() {
      struct nbd_server * myself = self ?: & manager;
      struct nbd_pidfile * pidfile = &myself->pidfile;
      // PTB ignore integer result
      pidfile->unlock(pidfile);
  }

  static void
  register_session_pid (struct nbd_server * self, int pid, unsigned long ipaddr)
  {
    // PTB register our (child) sessions pid and addr in our dbase
    struct nbd_db *spids_db = &self->spids;
    int otherpid;
    unsigned long otheripaddr;

    int recheck_count = 0;
    const int recheck_max = 5;
    int i;

    for (i = spids_db->first (spids_db, (void **) &otherpid, (void **) &otheripaddr);
	 i >= 0;
	 i = spids_db->next (spids_db, i, (void **) &otherpid, (void **) &otheripaddr)) {
      checkpid:
	if (kill (otherpid, 0) < 0) {
	    // PTB already deceased
	    spids_db->del (spids_db, (void *) (unsigned long) otherpid);
	    continue;
	}
	// PTB the otherpid is healthy
	if (otheripaddr == ipaddr) {
	    // PTB other is candidate for later slaughter
	    if (recheck_count++ < recheck_max) {
		microsleep (1000000);
		goto checkpid;
	    }
	    PERR ("session master detects live previous "
		  "pid %d for addr %lx\n", otherpid, ipaddr);
	    continue;
	}
	// PTB the otherpid is a valid pid

	/*
         * PTB but don't keep it in the list of pids as we may
	 * handle a lot more sessions than ENBD_MAXCONN!
	 * self->child[count++] = otherpid;
         */
    }

    // PTB add session master to the db for this addr
    spids_db->add (spids_db, (void *) (unsigned long) pid, (void *) ipaddr);

  }

  static void
  kill_previous_sessions (struct nbd_server *self)
  {
    struct nbd_db *spids_db = &self->spids;
    int otherpid;
    unsigned long otheripaddr;
    typeof (self->statfile) * statfile = &self->statfile;
    int i;
    for (i = spids_db->first (spids_db, (void **) &otherpid, (void **) &otheripaddr);
	 i >= 0;
	 i = spids_db->next (spids_db, i, (void **) &otherpid, (void **) &otheripaddr)) {

	if (kill (otherpid, 0) < 0) {
	    // PTB already deceased
	    spids_db->del (spids_db, (void *) (unsigned long) otherpid);
	    continue;
	}

	if (otheripaddr == self->caddr) {
	    PERR ("killing live session manager pid %d on addr %lx\n",
		  otherpid, otheripaddr);
	    for (i = 0; i < 4; i++) {
		if (kill (otherpid, SIGTERM) < 0)
		    break;
		microsleep (1000000);
	    }
	    if (kill (otherpid, 0) >= 0)
		kill (otherpid, SIGKILL);
	}

    }

    // PTB add ourselves as session master to the db for this addr
    spids_db->add (spids_db, (void *) (unsigned long) self->pid,
            (void *) self->caddr);
    // PTB possibly add ourselves to the corresponding file list
    DEBUG ("Server (%d) adds %8lx to %s\n", self->i,
	   self->caddr, statfile->statfile);

    statfile->lock (statfile);
    statfile->open (statfile);
    statfile->add (statfile, self->caddr);
    statfile->close (statfile);
    statfile->unlock (statfile);

  }

  static int
  make_buffer (struct nbd_server *self)
  {
    size_t buflen = ENBD_MAX_SECTORS * 512 + 1024 + ENBD_MAXCONN + 8192;
    char * buffer;
    // PTB forks duplicate this buffer in the slaves
    buffer = calloc (1, buflen);
    if (!buffer) {
	PERR ("fatal! Could not allocate %lu bytes for data buffer\n",
                (unsigned long)buflen);
        self->buffer = NULL;
        self->buflen = 0;
	return -ENOMEM;
    }
    // PTB align the buffer at 4096 in case it's a raw device
    self->buffer = (char *) ((((unsigned long) buffer + 4095) >> 12) << 12);
    self->buflen = buflen -= self->buffer - buffer;
    return buflen;
  }

  int main(int argc, char *argv[]) {

      int i, npids=0;
      struct nbd_server * self = & manager;
      struct nbd_file * file = & self->file;
      int err;
   
      logging();
      if (cmdline(self, argc, argv)) {
        usage();
        exit(0);
      }

      if (!has_sig) 
        has_sig = !getrandsig(self, signature, 6);

      init_file(file,
        self->nfile,
        ({ int file_mode;
           switch(self->mode) {
              case ENBD_LINEAR_MODE: file_mode = NBD_LINEAR_MODE; break;
              case ENBD_STRIPE_MODE: file_mode = NBD_STRIPE_MODE; break;
              case ENBD_MIRROR_MODE: file_mode = NBD_MIRROR_MODE; break;
              default: exit(1);
           }
           file_mode;
        }),
        self->sizes,
        self->blksize,
        self->names,
        self->flags & F_READONLY,
        self->flags & F_SYNC,
        self->flags & F_ASYNC,
        self->flags & F_DOLOCK,
        self->flags & F_DIRECT,
        &self->caddr
      );

      // PTB we do open on demand internally in the file object
      if (file->open(file) < 0) {
          MSG("server: Could not open all resources immediately: %m\n");
      }

      // PTB for removeable media .. open on demand will happen in slaves
      file->close(file);
      file->flags |= F_OPENWANTED;

      if ((file->flags & F_SYNC) && ! (self->flags & F_SYNC)) {
          MSG("server: flagged resource synchronous for writes.\n");
          self->flags |= F_SYNC;
          self->flags &= ~F_ASYNC;
      }
      if ((file->flags & F_ASYNC) && ! (self->flags & F_ASYNC)) {
          MSG("server: flagged resource asynchronous for writes.\n");
          self->flags |= F_ASYNC;
          self->flags &= ~F_SYNC;
      }
      if ((file->flags & F_READONLY) && ! (self->flags & F_READONLY)) {
          MSG("server: flagged resource readonly.\n");
          self->flags |= F_READONLY;
      }

      // PTB see about fixing up our notion of size

      if (self->size > 0) {
          // PTB might have to disregard the cmdline size if component unsized
          int i; 
          for (i = 0; i < self->nfile; i++) {
              if (self->sizes[i] <= 0)
                  break;
          }
          if (i < self->nfile) 
             self->size = file->size;
      }
      if (self->size <= 0) {
          // PTB in any case, ask the file if we really don't know our size
          self->size = file->size;
      } 

      DEBUG ("Completed open resources size %Ld\n", self->size);

      if (self->size <= 0) {
         PERR("server: Could not find size of exported block device: %m\n");
         MSG("server: assuming very large number for size and hoping.\n");
         self->size = (((typeof(self->size))1) << (sizeof(self->size)*8 - 1))
                    - 512; 
      }
      syslog(LOG_INFO, "size of exported file/device is %Ld",
		      (unsigned long long)self->size);

      self->pid = getpid();

      //PTB set up our pidfile before fork
      if (signature) {
         // PTB directory can be given on commandline. Id certainly is.
         char format[] = "%s/enbd-server-%s.pid";
         size_t pidfilenamelen = strlen(self->lockdir)
                    + strlen(signature) + strlen(format);
         char *pidfile = malloc(pidfilenamelen);

         if (!pidfile) {
            PERR("could not allocate %lu bytes of memory\n",
                     (unsigned long)pidfilenamelen);
            exit(1);
         } 
         sprintf(pidfile, format, self->lockdir, signature);
         initpidfile(&self->pidfile, pidfile);
      }

      //PTB check our pidfile before fork
      self->pid = getpid();

      // PTB set up our statfile before forks
      if (signature) {
        char format[] = "%s/%s/server-%s.client_ips";
        size_t filenamelen = strlen(self->statdir) + 4
                           + strlen(signature) + strlen(format);
        char * filename = malloc(filenamelen);
        struct stat statbuf;

        // PTB first do some hunting for the statdir. Use the old
        // statdir if it is available, otherwise use the new, making
        // it if necessary

        if (!filename) {
            PERR("could not allocate %lu bytes of memory\n",
                    (unsigned long)filenamelen);
            exit(1);
        }

        sprintf(filename, format, STATDIR, "nbd", signature);
        if (stat (self->statdir, &statbuf) < 0
            || ! S_ISDIR(statbuf.st_mode)) {
            sprintf(filename, format, STATDIR, "enbd", signature);
            if (stat (self->statdir, &statbuf) < 0
                || ! S_ISDIR(statbuf.st_mode)) {
                // PTB don't care if doesn't succeed as we can't do any better
            }
        }

        initipaddrfile(&self->statfile, filename);
      }

      // detach from terminal
      close(0);
      close(1);

      // pass zombies on to init
      signal(SIGCHLD, SIG_IGN);

      // go daemon ... first fork
      fflush(NULL);
      if (fork()) {
        return 0;
      }

      // PTB do the pidfile thing
      self->pid = getpid();

      err = self->pidfile.lock(&self->pidfile);
      if (err < 0) {
          // PTB this probably better not be deadly, so users can launch
          switch (err) {
              case -EBUSY:
                PERR ("There is already a enbd-server with id %s running."
                  " Check in %s!\n", signature, self->pidfile.pidfile);
                exit(5);
                break;
              default:
                PERR ("Warning - couldn't make a new pidfile %s!\n",
                         self->pidfile.pidfile);
                break;
          }
      }

      // PTB kill pidfile at exit
      atexit(removepidfile);

      // We're a detached child of the original proc with stdio open, and we
      // set about listening for connections like a good lil daemon.

      chdir("/");

      init_db(&self->spids,MAXSESSION);

      // PTB read the file list of sessions to notify people we're alive
      // PTB and notify them via their cstatd

      notify_clients(self);

      // PTB need to set signal handler to propagate signals
      self->sighandler = mastersighandler;
      setsighandler(self->sighandler);

      while (1) {
        // this is the listen, accept, fork loop
        int pid;
        unsigned long ipaddr;
    
        // if port==0 we use stdin as socket, i.e. sock=0
        switch (self->port) {

            case 0:
              self->sock = 0;
              break;

            default:
              if (connectme(self) < 0) {
                PERR("Server failed connectme!\n");
                self->sock = -1;
                microsleep(1000000);
                // try again
                continue;
              }
              break;
        }

#ifdef USING_SSL
        if (using_ssl)
          initstream(&self->stream,self->data_timeout,&ctx);
        else
#endif
          initstream(&self->stream,self->data_timeout,NULL);

        // PTB fill in the stream's details for it
        self->stream.sock = self->sock;
        self->stream.port = self->port;

        /* PTB parent determines who's calling */
        ipaddr = 0;
        if (self->sock > 0) {
           static struct sockaddr_in addr;
#ifdef HAVE_SOCKLEN_T
           socklen_t addrlen = sizeof(addr);
#else
           unsigned addrlen = sizeof(addr);
#endif
           if (getpeername(self->sock,(struct sockaddr*)&addr,&addrlen) >= 0) {
             ipaddr = ntohl(addr.sin_addr.s_addr);
             DEBUG("server (%d) got client addr 0x%lx\n", self->i,ipaddr);
           }
        }

        // deal with the client in a subprocess while we listen again
        fflush(NULL);
        pid = fork();
        if (pid < 0) {
          PERR("Master daemon could not fork: %m\n");
        }

        if (!pid) {
          // child
          init_master(self, ipaddr);
          // PTB child goes on to introduce itself to the connectee
          break;
        }

        // parent

        // PTB register our child sessions pid and addr in our dbase
        register_session_pid(self, pid, ipaddr);

        microsleep(5000000);
        // we have to wait for others to register too
        disconnectme(self);
        // parent goes round again to accept new connections
        continue;

      } // PTB end of main listen loop

      /*
       * PTB we now have branched off a new session server. It now
       * wants to invent its own port. And then accept (re)connections
       * on that port. At the moment its got a secure connection to do
       * the intro with.

       * We really should start new pgrp now instead of after intro,
       * but then I can't see the server messages on console

       * setsid();   // PTB start new session
       */

#ifdef USING_SSL
      if (using_ssl && init_server_ssl_stuff(self) < 0) {
        PERR("Server: ssl setup failed\n");
        return -1;
      }
#endif

      // PTB We, the child of the daemon, talk to new connection. Can die.

      self->flags &= ~F_INTRODUCED;
      err =  introduction(self);
      if (err < 0) {
        PERR("Server: initial server/client communication failed %m\n");
        // this is a death. Let's hope the parent ignores or accepts.
        disconnectme(self);
        exit(-err);
        return -1;
      }
      DEBUG("server (%d) nconn is %d after intro\n", self->i, self->nconn);

      // PTB OK. Introductions were successful.
      self->flags |= F_INTRODUCED;

      // PTB now kill other pretenders to our session management
      kill_previous_sessions(self);

      /*
       * PTB at this point we have a new port available too, and a
       * socket on that port. We could put it in listen state
       * and let the launches split off a connection on it?
       */

      setsid();   // PTB start new session
      setpgrp();
      MSG("server (%d) manager started new process group %d\n",
             self->i, getpgrp());
   
      make_buffer(self);
      if (!self->buffer) {
        exit(-1);
        return -ENOMEM;
      }

      // PTB nconn should have been set in intro. Init and launch servers
      for (i = 0; i < self->nconn; i++) {

         int pid = -1;
         if (!self->session[i]) {
           struct nbd_server * server = calloc(1,sizeof(*server));
           // PTB exit on fail
           if (!server) {
             PERR("could not allocate %lu bytes for session server struct\n",
                            (unsigned long)sizeof(*server));
             return -ENOMEM;
           }
           self->session[i] = server;
         }


         init_slave( self->session[i],
            i,
            self->socket,
            self->port,
            self->size,
            self->flags,
            self->nfile,
            self->names,
            self->blksize,
            self->mode,
            file,
            self->nconn,
            self->child,
            NULL,
            self->negotiate_timeout,
            self->pulse_intvl,
            self->data_timeout,
            self->buffer,
            self->shmem,
            self->ordered_write_timeout,
            self->sizes,
            self->lockdir,
            self->statdir,
            self->sync_intvl,
            self->cache_lim
          );

         /*
          * PTB here we fork the new daemons, and this time we want to watch
          * for their deaths so we can resuscitate them
          */

         pid = launch(self->session[i]);

         // PTB the session master really does care about the slave pids 
         if (pid > 0)  {
           self->child[i] = pid;
           npids ++;
         }
      }

     self->sock = -1;

     // PTB it's a fine call as to when we set the sighandlers

     self->sighandler = sessionsighandler; 
     setsighandler(self->sighandler);    // PTB handle CHLD USR1 HUP TERM

     // PTB keep file closed most of the time, and reopen for sync in
     // loop, so as to allow removable media to be removed!
     file->close(file);
     file->flags |= F_ASYNC;
     file->flags &= ~F_SYNC;

     // PTB maybe waste time syncing the resource in order to empty VMS
      while (npids > 0) {

          struct timeval tv0, tv1;
          long usleep;
          static short tv0_known;

          tv0_known = (mygettimeofday(&tv0, NULL) >= 0);

          if (file->open(file) >= 0) {
              if (!(self->flags & F_ASYNC))
                  file->sync(file);
              file->close(file);
          }

          if (mygettimeofday(&tv1, NULL) < 0 || !tv0_known) {
              microsleep(self->sync_intvl);
              continue;
          }

          // PTB how long since last time
          usleep = tv1.tv_sec - tv0.tv_sec;
          usleep *= 1000000;
          usleep += tv1.tv_usec - tv0.tv_usec;

          // PTB how many us still to wait
          usleep = self->sync_intvl - usleep;

          if (usleep <= 0) {
              // PTB this is arithmetic error. Let the cpu cool down
              usleep = self->sync_intvl;
          }
          microsleep(usleep);
      }
   
      return 0;
  }



