/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */

#include <stdio.h>      // for fprintf
#include <sys/stat.h>   // for struct stat
#include <unistd.h>     // for struct stat
#include <stdlib.h>     // for malloc
#include <signal.h>     // for signal and SIGPIPE
#include <arpa/inet.h>  // for inet_addr
#include <string.h>     // for strtok
#include <syslog.h>     // for LOG_DAEMON and syslog


#include "config.h"
#include "enbd_conf.h"
#include "ipaddrfile.h"

#ifndef STATDIR
#  define STATDIR "/var/state"
#endif

static int debug;
int debug_level;

#define PDEBUG(x...) { if (debug) fprintf(stderr,"enbd-sstatd: " x); }
#define PERROR(x...) { syslog((debug?LOG_PERROR:0)|LOG_DAEMON|LOG_ERR,"enbd-sstatd: " x); }


static int notice(int argc, char * argv[]) {

  char * news = argv[0];

  if (argc < 1) {
    PERROR("too few arguments to notice\n");
    return -1;
  }
  if (!news || 0 == strcmp("",news)) {
    PERROR("empty qualifier to notice\n");
    return -1;
  }

  if (0 == strcmp("help", news)) {
      printf("MSG notice client-stop client_port client_ipaddr ipaddr ..\n");
      return 0;
  }

  if (0 == strcmp("client-stop", news)) {

      char *client_port = argv[1];
      char **client_ipaddrs;
      short client_port_nr = -1;

      struct nbd_confent conf;

      if (argc < 2) {
        PERROR("too few arguments to notice\n");
        return -1;
      }
      if (!client_port || 0 == strcmp("",client_port)) {
        PERROR("empty port argument to notice\n");
        return -1;
      }
      if (sscanf(client_port, " %hd ", &client_port_nr) < 1) {
        PERROR("malformed port argument to notice\n");
        return -1;
      }
      if (client_port_nr == -1) {
        PERROR("illegal port argument to notice\n");
        return -1;
      }

      client_ipaddrs = argv + 2;

        // go through all the possible server entries and find out
        // which one could have been talkiing on the given port.
        // Then, when we find it, go through all the clients
        // registered with it, and remove our ipaddr from it.

      init_nbd_conf(&conf);
      conf.set(&conf);

      while (conf.get(&conf) >= 0) {

        char *client_ipaddr;

        char * statfile_pattern = STATDIR "/%s/server-%s.client_ips";
        char * statfile;
        int    statfilelen;
        struct nbd_statfile sf;
        struct stat statbuf;
  
        PDEBUG("target: %s\n", conf.target);
        if (0 != strcmp("server", conf.target))
            continue;

        PDEBUG("port: %d\n", conf.port);
        if (conf.port != client_port_nr)
            continue;

        // PTB ok .. have a possible conf file entry. Check its client ips.

        statfilelen = strlen(statfile_pattern) + 4 + strlen(conf.id);
        statfile = malloc(statfilelen);
        if (!statfile) {
            PERROR("cannot obtain %d bytes of memory\n", statfilelen);
            break;
        }
        sprintf(statfile, statfile_pattern, "nbd", conf.id);

        if (stat(statfile, &statbuf) < 0
        || statbuf.st_size <= 0) {
            sprintf(statfile, statfile_pattern, "enbd", conf.id);
            if (stat(statfile, &statbuf) < 0
            || statbuf.st_size <= 0) {
                PERROR("no statfile %s\n", statfile);
                free(statfile);
                continue;
            }
        }

        // PTB looks like there is an IPs file. Read it.

        initipaddrfile(&sf, statfile);

        if (sf.lock(&sf) < 0) {
            PERROR("cannot lock statfile %s\n", statfile);
            free(statfile);
            continue;
        }
        if (sf.open(&sf) < 0) {
            PERROR("cannot open statfile %s\n", statfile);
            sf.unlock(&sf);
            free(statfile);
            continue;
        }

        for (client_ipaddr = *client_ipaddrs;
           client_ipaddr;
           client_ipaddr = *++client_ipaddrs
        ){

          unsigned long client_ipaddr_nr;

          client_ipaddr_nr = inet_network(client_ipaddr);

          if (sf.get(&sf,client_ipaddr_nr) < 0) {
            PERROR("address %s not in statfile %s\n", client_ipaddr, statfile);
            continue;
          }

          PDEBUG("deleting address %s from statfile %s\n", client_ipaddr,
                    statfile);
          sf.del(&sf,client_ipaddr_nr);

        } // efor
        sf.close(&sf);
        sf.unlock(&sf);

      } // ewhil
      conf.end(&conf);

      return 0;
  }
  // ...
  return -1;
}

static int quit(int argc, char * argv[]) {

  char * what = argv[0];

  if (argc < 1) {
    exit(0);
    return 0;
  }

  if (!what || 0 == strcmp("",what))
    return -1;

  if (0 == strcmp("help", what)) {
      printf("MSG quit\n");
      return 0;
  }

  return 0;
}


static int help(int argc, char * argv[]) {

  char * subject = argv[0];

  if (argc < 1) {
    printf("MSG help notice quit\n");
    return 0;
  }
  if (!subject || 0 == strcmp("",subject)) {
    printf("MSG help notice quit\n");
    return 0;
  }

  if (0 == strcmp("notice", subject)) {
      notice(1, (char*[]){ "help", NULL});
      return 0;
  }
  if (0 == strcmp("quit", subject)) {
      quit(1, (char*[]){ "help", NULL});
      return 0;
  }
  if (0 == strcmp("help", subject)) {
      printf("MSG help subject\n");
      return 0;
  }
  printf("MSG help notice quit\n");
  return 0;
}

static void usage() {

    PERROR("commands:\n");
    PERROR("\tnotice client-stop PORT IPADDR ...\n");
    PERROR("\thelp TOPIC\n");
    PERROR("\tquit\n");
}

void sighandler(int signo) {
    syslog(LOG_DAEMON|LOG_ERR,"enbd-sstatd: received signal %d\n", signo);
    exit (signo);
}

int main(int argc, char *argv[]) {

  # define BUFLEN 1024
  const int buflen = BUFLEN;
  char buffer[BUFLEN];

  if (argc >= 2 && !strcmp("-d",argv[1]))
    debug = 1;

  // To stop us dying with SIGPIPE when we write to stderr and the
  // other side has closed the socket to which stderr is redirected
  signal(SIGPIPE, sighandler);

  buffer[0] = 0;

  while (fgets(buffer, buflen, stdin)) {
  
    char * command;
    int wordc = 0;
    char *wordv[buflen/2];
    int err = -1;

    command = strtok(buffer," \t\r\n");

    if (!command || 0 == strcmp("",command))
      continue;
    
    while (1) {
       char * word = strtok(NULL," \t\r\n");
       if (!word)
         break;
       wordv[wordc++] = word;
    }
    // null terminate the sequence
    wordv[wordc] = NULL;

    if (0 == strcmp("help",command)) {
        err = help(wordc,wordv);
    } else if (0 == strcmp("quit",command)) {
        err = quit(wordc,wordv);
    } else if (0 == strcmp("notice",command)) {
        err = notice(wordc,wordv);
//    } else if (0 == strcmp("proxy",command)) {
//        err = proxy(wordc,wordv);
//    } else if (0 == strcmp("status",command)) {
//        err = status(wordc,wordv);
    } else {
        usage();
        exit(1);
    }

    if (err >= 0) {
        printf("ACK %s OK\n", command);
    } else {
        printf("ACK %s ERROR\n", command);
    }

    fflush(NULL);
  }
  return 0;
}
