#! /bin/sh

CONFIG=${NBD_CONFIG-/etc/enbd.conf}
[ -r $CONFIG ] || exit 1

set -x

# ho hum .. we should obey a remove command


####################################################################
# given foo as arg, make foo.lock atomically, with our pid inside.
# try 5 times at 2s intervals. Succeed only when there isn't already a
# valid lock.
#
lock() {
  local file=$1
  local lockfile=$file.lock
  local tmpfile=`dirname $lockfile`/.enbd-sstatd-lock.$$
  echo $$ > $tmpfile || return 1
  local count=0

  while ! ln $tmpfile $lockfile ; do

    if [ $count -gt 5 ] ; then
      rm -f $tmpfile 
      return 1
    fi
    count=$[ $count + 1 ]

    local pid="`head -1 $lockfile`"
    for p in $pid; do
      pid=$p ; break
    done

    pid=${pid%%[^0-9]*}

    if ! [ -n "$pid" -a "$pid" -gt 0 ] ; then
      rm -f $lockfile
      continue
    fi

    if kill -0 $pid >/dev/null 2>&1; then
      rm -f $tmpfile 
      return 1
    fi
    sleep 2
  done
  return 0

}

####################################################################
# given foo as arg, unlink foo.lock atomically
#
unlock() {
  local file=$1
  local lockfile=$file.lock
  rm -f $lockfile
}

####################################################################
# handle quit from client
#
quit() {
  local what=$1
  case $what in
  help) echo quit ;;
  *)    exit 0;;
  esac
}

#################################################################
# list the config file, sans commentary. This is a lexer, really.
#
config() {
   cat $CONFIG | grep -v '^[ ]*$' | while read line; do

     case "$line" in
       "#"*) continue ;;
     esac

     set -- $line

     [ -z "$1" ] && continue

     echo -n "$1" ; shift

     for word; do
       case "$word" in
         "#"*) break ;;
          *)   echo -n " $word" ;;
       esac
     done
     echo
   done
}

#################################################################
# get server id given server/client port 
#
get_server_id() {

  echo $* | while read port etc; do
    config | while read target id rest; do

      [ "$target" = server ] || continue
      [ -z "$id" ] && continue

      echo "$rest" | while read server_port etc; do

        [ $server_port -eq $port ] || continue
        echo $id
      done 
    done 
  done
}


#################################################################
# get some info on the server daemon state
#
status() {
  echo $* | while read dowhat port etc ; do
    case $dowhat in
      help)  echo status server-show server_port 
             continue
        ;;
      server-show) 
             local id=`get_server_id $port`
             if [ -z "$id" ]; then
               echo not configured
             else
               PIDFILE=/var/run/enbd-server-$id.pid
               if [ ! -s $PIDFILE ]; then
                 echo not running
               else
                 local spid="`head -1 $PIDFILE`"

                 for i in $spid dummy; do
                   spid=$i ; break
                 done

                 spid=${spid%%[^0-9]*}

                 if ! [ $spid -gt 0 ] ; then
                   echo not running
                 elif kill -0 $spid >/dev/null 2>&1 ; then
                     echo running
                 else
                     echo not running
                 fi

               fi
             fi
        ;;
      *) continue
        ;;
    esac
  done
}


####################################################################
# handle notice from client telling us it's now dead
#
notice() {

   local news=$1
   shift
   [ -n "$news" ] || return

   case $news in
    help)
        echo notice client-stop port ipaddr ipaddr ..
        ;;
    client-stop)
        echo $* | while read port rest ; do

          # the client passes us its IPs.

          for ipaddr in $rest; do

            ipaddr=${ipaddr%%[^0-9.]*}

            [ -n "$ipaddr" ] || continue

            # we search the conf file to get each servers ID in turn
       
            config | while read target id rest; do

                [ "$target" = server ] || continue
                [ -z "$id" ] && continue

                echo "$rest" | while read server_port resource options; do

                  server_port=${server_port%%[^0-9]*}

                  [ -n "$server_port" -a "$server_port" -gt 0 ] || continue
                  [ $server_port = $port ] || continue

                  CLIENT_IPS_FILE=/var/state/enbd/server-$id.client_ips
                  [ -s $CLIENT_IPS_FILE ] || continue

                  PIDFILE=/var/run/enbd-server-$id.pid
                  [ -s $PIDFILE ] || continue

                  pid="`head -1 $PIDFILE`"
                  for p in $pid; do
                    pid=$p ; break
                  done
                  pid=${pid%%[^0-9.]*}

                  [ -n "$pid" -a "$pid" -gt 0 ] || continue
                  kill -0 $pid >/dev/null 2>&1 || continue

                  # ok .. live server. check the clients listed in its
                  # state file. We need to get the cstatd to send all
                  # its if IPs so that we can hit on an alias too. With
                  # luck, however, the principal IP will be the one
                  # that is registered (don't solve a problem before it
                  # appears).

                  lock $CLIENT_IPS_FILE || continue
                  if grep -qs $ipaddr $CLIENT_IPS_FILE ; then
                    grep -v $ipaddr < $CLIENT_IPS_FILE > $CLIENT_IPS_FILE.new
                    mv $CLIENT_IPS_FILE.new $CLIENT_IPS_FILE
                  fi
                  unlock $CLIENT_IPS_FILE


                done 
            done
          done
        done
        ;;
    esac
}

####################################################################
# help message
#
help() {

  local subject=$1

  case $subject in
    quit|notice)
      $subject help ;;
    help)
      echo help subject ;;
    *)
      echo help notice quit ;;
  esac
}

####################################################################
# do our stuff .. no args. Entry from stdin for inetd
#
main() {
  while read command rest; do

    [ -n "$command" ] || continue
  
    case ${command} in
    help|notice|quit|status)
       $command $rest
       ;;
    *) usage
       exit 1
       ;;
    esac
  done
}

main $*

