/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */

#define _GNU_SOURCE 1
#define _XOPEN_SOURCE 1
#define _LARGEFILE_SOURCE 1
#define _LARGEFILE64_SOURCE 1
#define _FILE_OFFSET_BITS 64

#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
//#include <linux/fs.h>n// for BLKFLSBUF but pulls in string grunge
#include <syslog.h>
#include <string.h>
#include <sys/time.h>
#include <time.h>
#include <sys/sysinfo.h>
#include <unistd.h>
#include <netinet/in.h>
#include <linux/enbd.h>
#include <sys/mount.h> // for BLKGETSIZE
#include "config.h"
#ifndef MY_NAME
#define MY_NAME "nbd-test"
#endif
#include "cliserv.h"

#define BITESIZE sizeof(int)

#ifndef BLKFLSBUF  
#define BLKFLSBUF  _IO(0x12,97) /* flush buffer cache */
#endif


static void
usage (char **args, int line)
{
    fprintf (stderr, "%s %d usage: %s ", args[0], line, args[0]);
    fprintf (stderr, "device "
    "[-b blksize] "
    "[-s size] "
    "[-o offset] "
    "[-r seeks] "
    "[-y] "
#ifdef O_DIRECT
    "[-n] "
#endif
    "-t [:]{1:|2:|3:|4:}* "
    "-i [:]{1:|2:|3:|4:}* "
    "\n");
}

static char *filename;
static void setfilename(char *fn) {
    filename = strdup(fn);
}
static void unlinkfilename(void) {
    if (filename)
        unlink(filename);
}

static void
fill0 (int *buf, unsigned hoffset, unsigned loffset, size_t len)
{
    int i;
    int count = len / sizeof (int);
    u64 offset = (((u64)hoffset) << 32) + loffset;

    for (i = 0; count >= 0; count--, i++, offset += sizeof (int)) {
	int sector  = (int) (offset >> 9);
	int pattern = (sector << 9) + i;
        *buf++      = pattern;
    }
}

static void
fill1 (int *buf, unsigned hoffset, unsigned loffset, size_t len)
{
    int i;
    int count = len / sizeof (int);
    u64 offset = (((u64)hoffset) << 32) + loffset;

    for (i = 0; count >= 0; count--, i++, offset += sizeof (int)) {
	int sector  = (int) (offset >> 9);
	int pattern = -((sector << 9) + i);
        *buf++      = pattern;
    }
}

typedef void
(*FILL_T) (int *buf, unsigned hoffset, unsigned loffset, size_t len);

static FILL_T
fills[] = { &fill0, &fill1, 0 };

static int
mywrite (int fd, char *buf, size_t len)
{

    int n = 0;

    while (n < len) {
	int w = write (fd, buf + n, len - n);
	if (w <= 0) {
	    break;
	}
	n += w;
    }
    return n;
}

static int
myread (int fd, char *buf, size_t len)
{

    int n = 0;

    while (n < len) {
	int r = read (fd, buf + n, len - n);
	if (r <= 0) {
	    break;
	}
	n += r;
    }
    return n;
}

static int
skip (int fd, int k) {
            // PTB skip the remainder of the block
	    s64 err;
#ifdef HAVE_LSEEK64
	    err = lseek64 (fd, k, SEEK_CUR);
#elif HAVE_LLSEEK
	    err = llseek (fd, k, SEEK_CUR);
#else
	    err = lseek (fd, k, SEEK_CUR);
#endif
	    if (err == (off_t) -1) 
                return -EINVAL;
            return k;
}

static int
dowrite (FILL_T fill, unsigned long initial_block, unsigned long blocks, int blksize, int fd, int sync)
{
    long i;
    int total_errs = 0;
    long percent = 0;
    int * realbuf = malloc(blksize + 4096);
    int * buf;

    if (!realbuf)  {
	PERR ("Cannot allocate %dB of memory!\n", 2 * blksize);
        return -ENOMEM;
    }
    buf = (int *)(((((unsigned long)realbuf) + 4095) >> 12) << 12);

    fprintf (stderr, "writing");
    for (i = 0; i < blocks; i++) {

	int n;
        u64 offset = (initial_block + (u64)i) * blksize;
        long hoffset = offset >> 32;
        long loffset = offset & (u64)~((unsigned)0);

	(*fill) (buf, hoffset, loffset, blksize);

	n = mywrite (fd, (char *)buf, blksize);

        if (sync)
	    fdatasync (fd);

	if (n < blksize) {
	    PERR ("\nerror (%d) writing block %ld on fd %d: %m\n", n, i, fd);
            if (skip(fd, blksize - n) < 0) {
	        PERR ("Cannot seek to block %ld on fd %d!\n", i, fd);
                free (realbuf);
		return -EINVAL;
            }
            if (n < 0) 
                n = 0;
	    total_errs++;
	}

        if (i % (blocks / 100) == 0 && (100 * i) / blocks != percent) {
            percent++;
            if (percent % 5 == 0) 
	        fprintf (stderr, "%ld%%", percent);
            else
	        fprintf (stderr, ".");
        }
	fflush (stderr);
    }

    free (realbuf);
    return total_errs;
}

static int
doread (FILL_T fill, unsigned long initial_block, unsigned long blocks,
         int blksize, int fd)
{
    long i;
    int total_errs = 0;
    long percent = 0;
    int * realwbuf = malloc(blksize + 4096);
    int * realrbuf = malloc(blksize + 4096);
    int * wbuf, * rbuf;

    if (!realwbuf || !realrbuf)  {
        if (realwbuf)
            free(realwbuf);
        if (realrbuf)
            free(realrbuf);
	PERR ("Cannot allocate %dB of memory!\n", 2 * blksize);
        return -ENOMEM;
    }
    rbuf = (int *)(((((unsigned long)realrbuf) + 4095) >> 12) << 12);
    wbuf = (int *)(((((unsigned long)realwbuf) + 4095) >> 12) << 12);

    fprintf (stderr, "reading");
    for (i = 0; i < blocks; i++) {

	int n;
        u64 offset = ((u64)(initial_block + i)) * blksize;
        long hoffset = offset >> 32;
        long loffset = offset & (u64)~((unsigned)0);

	(*fill) (wbuf, hoffset, loffset, blksize);

	n = myread (fd, (char *)rbuf, blksize);

	if (n < blksize) {
            // PTB skip the remainder of the block
	    PERR ("\nerror reading block %ld: %m", i);
            if (skip(fd, blksize - n) < 0) {
		PERR("Cannot seek to more than 2^31-1 (0x%Lx) in one file!\n",
		  (unsigned long long) offset);
                free(realwbuf);
                free(realrbuf);
		return -EINVAL;
	    }
            if (n < 0) 
                n = 0;
	    total_errs++;
	}

	if (memcmp (wbuf, rbuf, blksize) != 0) {
            static int errs = 0;
            if (errs++ < 10) {
                int k;
                for (k = 0; k < blksize / sizeof(int) && k < 10; k++) {
                    if ( ((int*)wbuf)[k] != ((int*)rbuf)[k] )
                        fprintf(stderr,
                                "read %#.8x.. != %#.8x.. expected at block %ld+%lu\n",
                                ((unsigned*)rbuf)[k],
                                ((unsigned*)wbuf)[k],
                                i,
                                (unsigned long)k*sizeof(int));
                }
                fprintf(stderr, "...\n");
            }
	    total_errs++;
        }

        if (i % (blocks / 100) == 0 && (100 * i) / blocks != percent) {
            percent++;
            if (percent % 5 == 0) 
	        fprintf (stderr, "%ld%%", percent);
            else
	        fprintf (stderr, ".");
        }
        fflush (stderr);
    }

    free(realwbuf);
    free(realrbuf);
    return total_errs;
}

static int
readarg(char* buf, int *arg) {

        char c;
        int scale = 0;

	if (sscanf (buf, "%d%c", arg, &c) < 2) {
	    if (sscanf (buf, "%d", arg) < 1) {
	        return 0;
            }
        }
        switch (c) {
            case 0: scale = 0;
                    break;
            case 'k': case 'K': scale = 10;
                    break;
            case 'm': case 'M': scale = 20;
                    break;
            case 'g': case 'G': scale = 30;
                    break;
            case 't': case 'T': scale = 40;
                    break;
            case 'p': case 'P': scale = 50;
                    break;
        }
        *arg <<= scale;
        return 1;
}
static int
readlonglongarg(char* buf, long long *arg) {

        char c;
        int scale = 0;

	if (sscanf (buf, "%Ld%c", arg, &c) < 2) {
	    if (sscanf (buf, "%Ld", arg) < 1) {
	        return 0;
            }
        }
        switch (c) {
            case 0: scale = 0;
                    break;
            case 'k': case 'K': scale = 10;
                    break;
            case 'm': case 'M': scale = 20;
                    break;
            case 'g': case 'G': scale = 30;
                    break;
            case 't': case 'T': scale = 40;
                    break;
            case 'p': case 'P': scale = 50;
                    break;
        }
        *arg <<= scale;
        return 1;
}

static s64
seek(int fd, loff_t offset, int whence) {
      s64 size;
#ifdef HAVE_LSEEK64
      size = lseek64 (fd, (loff_t) offset, whence);
#elif HAVE_LLSEEK
      size = llseek (fd, (loff_t) offset, whence);
#else
      size = lseek (fd, (off_t) offset, whence);
      if (size < 0) {
	PERR ("Cannot seek to more than 2^31-1 (0x%Lx) in one file!\n",
	      offset);
	return -EINVAL;
      }
#endif
      return size;
}

static int
doseeks (long long seeks, unsigned long blocks, int blksize, int fd,
	 int sync)
{
    unsigned long i, j;
    unsigned long *posArray =
        (unsigned long *) malloc ((unsigned long)seeks * sizeof (*posArray));
    unsigned long r;
    unsigned long errors = 0;
    int ok = 0;
    int n;
    long percent = 0;

    if (!posArray) {
        fprintf(stderr, "Not enough memory for %ld bytes\n",
                (unsigned long)seeks * sizeof(*posArray));
        return -ENOMEM;
    }

    srandom ((unsigned int) time (NULL));

    if (seeks > blocks) {
	fprintf (stderr,
		 "The number of seeks must not be greater than the number of blocks.\n");
        free (posArray);
	return 1;
    }

    // write
    fprintf (stderr, "seeking and writing");
    for (i = 0; i < seeks; i++) {
	do {
	    ok = 1;
	    posArray[i] = random () % blocks;
            // make the seek positions all different. Why? Surely 
            // consecutive ones different is only what one wants.
	    for (j = 0; j < i; j++) {
		if (posArray[i] == posArray[j]) {
		    ok = 0;
		    break;
		}
	    }
	} while (!ok);

	lseek (fd, posArray[i] * blksize, SEEK_SET);
	//write( fd, &i, sizeof(unsigned long) );
	n = mywrite (fd, (char *) &i, sizeof (i));
	if (n != sizeof (i)) {
	    fprintf (stderr, "doseeks: Could not write to block %ld!\n", i);
	    errors++;
	}

	if (sync)
	    fdatasync (fd);

	if (seeks >= 100) {
	    if (i % (seeks / 100) == 0 && (100 * i) / seeks != percent) {
		percent++;
		if (percent % 5 == 0)
		    fprintf (stderr, "%ld%%", percent);
		else
		    fprintf (stderr, ".");
	    }
	}
	fflush (stderr);
    }

    fprintf (stderr, "done");
    fprintf (stderr, "\nflushing buffers..");
    fflush (stderr);
    fdatasync (fd);
    fprintf (stderr, "done");
    fflush (stderr);

    // read and check
    fprintf (stderr, "\nreading and checking");
    percent = 0;

    for (i = 0; i < seeks; i++) {
	seek (fd, posArray[i] * blksize, 0);
	myread (fd, (char *) &r, sizeof (unsigned long));
	if (r != i) {
	    fprintf (stderr,
		     "doseeks: Read error at pos %lu: is %lu, should be %lu!\n",
		     i, r, i);
	    errors++;
	}
	if (seeks >= 100) {
	    if (i % (seeks / 100) == 0 && (100 * i) / seeks != percent) {
		percent++;
		if (percent % 5 == 0)
		    fprintf (stderr, "%ld%%", percent);
		else
		    fprintf (stderr, ".");
	    }
	}
	fflush (stderr);
    }

    free (posArray);
    return errors;
}

int
main (int nargs, char *argv[])
{

    char *devname = NULL;
    char *filename = NULL;
    int args_seen = 0;

    int fd;
    unsigned long blocks, i, initial_block;
    int total_errs;
    int g;
    short sync = 0;
#ifdef O_LARGEFILE
    short nobuffer = 0;
#endif
    int openflags;

    int loop[] = { 1, 2, 3, 4, 5, -1, 0 };
    int aioctl[16] = { 1, 2, 3, 4, 5, 6, 7, };
    int nloop = -1;    // telltale
    int nioctl = -1;   // telltale
    int blksize = 1024; // default
    long long size = -1;    // telltale
    long long seeks = -1;   // telltale
    long long offset = -1;  // telltale
    struct stat stat_buf;

    if (nargs - 1 < 1) {
	usage (argv, __LINE__);
	return 1;
    }

    // first arg is devicename
    devname = argv[1];
    args_seen = 1;

    // begin cmdline analysis

    while (nargs - 1 >= args_seen + 1 && *argv[args_seen+1] == '-') {

      // -b blksize
      if (strcmp("-b",argv[args_seen+1]) == 0) {

        if (nargs - 1 < args_seen + 2) {
	    usage (argv, __LINE__);
	    return 1;
        }
        if (readarg(argv[args_seen+2], &blksize) < 1 || blksize < 0) {
	        usage (argv, __LINE__);
	        return 1;
        }
        if (blksize != 512 
        &&  blksize != 1024
        &&  blksize != 2048
        &&  blksize != 4096) {
	    usage (argv, __LINE__);
	    return 1;
        }
        args_seen += 2;
        continue;
      }

      // -s size
      if (strcmp("-s",argv[args_seen+1]) == 0) {

        if (nargs - 1 < args_seen + 2) {
	    usage (argv, __LINE__);
	    return 1;
        }
        if (readlonglongarg(argv[args_seen+2], &size) < 1 || size < 0) {
	        usage (argv, __LINE__);
	        return 1;
        }
        args_seen += 2;
        continue;
      }

      // -o offset
      if (strcmp("-o",argv[args_seen+1]) == 0) {

        if (nargs - 1 < args_seen + 2) {
	    usage (argv, __LINE__);
	    return 1;
        }
        if (readlonglongarg(argv[args_seen+2], &offset) < 1 || offset < 0) {
	        usage (argv, __LINE__);
	        return 1;
        }
        args_seen += 2;
        continue;
      }

      // -y sync
      if (strcmp("-y",argv[args_seen+1]) == 0) {
        sync = 1;
        args_seen += 1;
        continue;
      }

      // -r number of random seeks
      if (strcmp("-r",argv[args_seen+1]) == 0) {
        
        if (nargs - 1 < args_seen + 2) {
           usage (argv, __LINE__);
           return 1;
        }
        if (readlonglongarg(argv[args_seen+2], &seeks) < 1 || seeks < 0) {
           usage (argv, __LINE__);
           return 1;
        }
        args_seen += 2;
        continue;
      }
                    


#ifdef O_DIRECT
      // nobuffer
      if (strcmp("-n",argv[args_seen+1]) == 0) {
        nobuffer = 1;
        args_seen += 1;
        continue;
      }
#endif

      // -t testseq (':' or ',' separators)
      if (strcmp("-t",argv[args_seen+1]) == 0
      ||  strcmp("--test",argv[args_seen+1]) == 0) {
            char *loopstr, *ptr;
        if (nargs - 1 < args_seen + 2) {
	    usage (argv, __LINE__);
	    return 1;
        }
        loopstr = strdup(argv[args_seen+2]);
        ptr = strtok(loopstr,":,;");
        for (i = 0; ptr != NULL; ptr = strtok(NULL,":,;"), i++) {
          sscanf (ptr, "%d", &loop[i]);
          // sanity check for termination
	  if (loop[i] < 1 || i >= sizeof(loop)/sizeof(*loop)) {
	      usage (argv, __LINE__);
	      return 1;
	  }       
	  //fprintf (stderr, "arg %ld %d\n", i, loop[i]);
        }
        nloop = i;
        loop[i] = 0;
        args_seen += 2;
        continue;
      }

      // -i testseq (':' or ',' separators)
      if (strcmp("-i",argv[args_seen+1]) == 0
      ||  strcmp("--ioctl",argv[args_seen+1]) == 0) {
        char *ioctlstr, *ptr;
        if (nargs - 1 < args_seen + 2) {
	    usage (argv, __LINE__);
	    return 1;
        }
        ioctlstr = strdup(argv[args_seen+2]);
        ptr = strtok(ioctlstr,":,;");
        for (i = 0; ptr != NULL; ptr = strtok(NULL,":,;"), i++) {
          sscanf (ptr, "%d", &aioctl[i]);
	  if (aioctl[i] < 1 || aioctl[i] > 8) {
	      usage (argv, __LINE__);
	      return 1;
	  }       
	  //fprintf (stderr, "arg %ld %d\n", i, aioctl[i]);
        }
        nioctl = i;
        aioctl[i] = 0;
        args_seen += 2;
        continue;
      }
      break;

    }

    // if no test count given, assume all wanted
    if (nloop < 0) {
        nloop = sizeof(loop)/sizeof(*loop) - 1;
        loop[nloop] = 0;
    }

    // if no ioctl count givem assume none wanted
    if (nioctl < 0) {
        nioctl = 0;
        aioctl[nioctl] = 0;
    }

    // end of cmdline analysis

    openflags = O_RDWR|O_LARGEFILE;
#ifdef O_DIRECT
    if (nobuffer)
             openflags |= O_DIRECT;
#endif
    fd = open (devname, openflags, 0600);
    if (fd < 0) {
        if (stat(devname, &stat_buf) < 0) {
	    PERR ("cannot stat %s: %m\n", devname);
	    return -EINVAL;
        }
        // PTB exists but cannot open rw
        if (!S_ISDIR(stat_buf.st_mode)) {
	    PERR ("cannot open %s rw: %m\n", devname);
	    return -EINVAL;
        }
        // PTB is a dir, so make a dir in it
        putenv("TMPDIR");
        filename = tempnam(devname, "foo");
        if (!filename) {
	    PERR ("cannot mktemp in %s: %m\n", devname);
	    return -EINVAL;
        }
        fd = open (filename, O_CREAT|O_EXCL|openflags, 0600);
        if (fd < 0) {
	    PERR ("cannot create %s: %m\n", filename);
	    return -EINVAL;
        }

        setfilename(filename);
        atexit(unlinkfilename);

        if (seek(fd, size-1, SEEK_SET) != size-1) {
	    PERR ("cannot stretch %s to size %Ld: %m\n", filename, size);
            close(fd);
	    return -EINVAL;
        }
        if (write(fd, "", 1) < 1){
	    PERR ("cannot write %s at offset %Ld: %m\n", filename, size);
            close(fd);
	    return -EINVAL;
        }
        lseek(fd, 0, SEEK_SET);
    }
 
    if ( seeks < 0 ) {
        // PTB do 1000 seeks if nobody says any different
        seeks = 1000;
     }
    
    if (size < 0) {
        size = seek(fd, 0, SEEK_END);
    }

    if (size <= 0) {
        if (fstat(fd, &stat_buf) >= 0 && S_ISBLK(stat_buf.st_rdev)) {

#ifdef BLKGETSIZE64
            if (size <= 0) {
                if (ioctl(fd,  BLKGETSIZE64, &size) < 0)
                size = -1;
            }
#endif

#ifdef BLKGETSIZE
            if (size <= 0) {
                int sectors;
                if (ioctl(fd,  BLKGETSIZE, &sectors) < 0)
                    size = -1;
                else
                    size = sectors << 9;
            }
#endif
        }
        if (size <= 0) {
            struct sysinfo sysinfo_buf;
            if (sysinfo(&sysinfo_buf) >= 0) {
                size = seek(fd, sysinfo_buf.totalram * 2, SEEK_SET);
            }
        }
    }

    if (size < 0) 
        return -EINVAL;

    blocks = size / blksize;
    fprintf (stderr, "%s has %Ld bytes in %ld blocks of %d bytes each\n",
	     devname, size, blocks, blksize);

    if (offset < 0) {
        offset = 0;
    }
    initial_block = offset / blksize;

    seek(fd, 0, SEEK_SET);

    if (blocks <= 100)
        goto test_ioctls;

    for (g = 0; g < nloop && loop[g] > 0; sleep(3), g++) {

	int f = loop[g];

	FILL_T fill = fills[(f - 1) / 2];
	//fprintf (stderr, "test %d:\n", f);

        seek(fd, 0, SEEK_SET);

	fprintf (stderr, "flushing buffers..");
	fflush (stderr);
	fdatasync (fd);
	//ioctl (fd, BLKFLSBUF, 0);
	fprintf (stderr, "done\n");
	fflush (stderr);

	switch (f) {
	  case 2:
	  case 4:
	    total_errs = doread (fill, initial_block, blocks, blksize, fd);
	    break;
	  case 1:
	  case 3:
	    total_errs = dowrite (fill, initial_block, blocks, blksize, fd, sync);
	    break;
	  case 5:
            total_errs = doseeks ( seeks, blocks, blksize, fd, sync );
	    break;
	  default:
	    usage (argv, __LINE__);
	    return 3;
	}

	fprintf (stderr, "done\n");
	fflush (stderr);
	fprintf (stderr, "test %d %s:  %ld incorrect blocks\n",
                f, total_errs!=0 ? "FAILED" : "succeeded",
                total_errs >= 0 ? total_errs : blocks);
	fflush (stderr);

	fprintf (stderr, "flushing buffers..");
	fflush (stderr);
	fdatasync (fd);
	//ioctl (fd, BLKFLSBUF, 0);
	fprintf (stderr, "done\n");
	fflush (stderr);

    } // end for

test_ioctls:
    // test ioctls

    srandom(time(NULL));

    for (i = 0; i < nioctl; i++) {

        int err, tintval;
        int j;
        static int intval;
        char cbuf[256], tbuf[256];
        int cmd = aioctl[i];

        switch (cmd) {
            case 1:
                if (intval == 0)
                    intval = random();
	        err = ioctl (fd, ENBD_TEST_IOCTL1, intval);
	        if (err < 0) {
		        fprintf (stderr, "ioctl %d (write direct) failed %m\n",
                                cmd);
                        break;
	        }
		fprintf (stderr, "ioctl %d succeeded\n", cmd);
                break;
            case 2:
                err = -1;
                for (j =0; j < 100; j++) {
                        // PTB try several times as may not work first time
	                err = ioctl (fd, ENBD_TEST_IOCTL2, &tintval);
	                if (err < 0) 
                            continue;
                        if (tintval != intval && intval != 0)
                            continue;
                        break;
                }
	        if (err < 0) {
		        fprintf (stderr,
                                "ioctl %d (read indirect) failed %m\n",
                                cmd);
                        break;
                }
                if (tintval != intval && intval != 0) {
		        fprintf (stderr,
                                "ioctl %d (read indirect) failed, read %#x\n",
                                cmd, (unsigned) tintval);
                        break;
                }
		fprintf (stderr, "ioctl %d succeeded\n", cmd);
                break;
            case 3:
	        intval = random();
                tintval = intval;
	        err = ioctl (fd, ENBD_TEST_IOCTL3, &tintval);
	        if (err < 0) {
		        fprintf (stderr,
                                "ioctl %d (write + read indirect) failed %m\n",
                                cmd);
                        break;
	        }
                if (tintval != ~intval && intval != 0) {
		        fprintf (stderr,
                                "ioctl %d (write + read indirect) failed, wrote %#x read %#x\n",
                                cmd, (unsigned)intval, (unsigned)tintval);
                        break;
                }
		fprintf (stderr, "ioctl %d succeeded\n", cmd);
                break;
            case 4:
	        err = ioctl (fd, ENBD_TEST_IOCTL4, &cbuf[0]);
	        if (err < 0) {
		        fprintf (stderr,
                                "ioctl %d (large read indirect) failed %m\n",
                                cmd);
                        break;
	        }
		fprintf (stderr, "ioctl %d succeeded\n", cmd);
                break;
            case 5:
                for (j = 0; j < 256; j += sizeof(int)) {
                    *(int*)&tbuf[j] = random();
                }
	        memcpy (&cbuf[0], &tbuf[0], 256);
	        err = ioctl (fd, ENBD_TEST_IOCTL5, &cbuf[0]);
	        if (err < 0) {
		        fprintf (stderr,
                                "ioctl %d (large write + read indirect) failed %m\n",
                                cmd);
                        break;
	        }
                for (j = 0; j < 256; j += sizeof(int)) {
                    if (*(int*)&tbuf[j] != ~*(int*)&cbuf[j]) {
		        fprintf (stderr,
                                "ioctl %d (large write + read indirect) failed\n",
                                cmd);
                        break;
                    }
                }
                if (j < 256)
                    break;

		fprintf (stderr, "ioctl %d succeeded\n", cmd);
                break;
        }

    };
    close(fd);

    return 0;

}
