#include <asm/atomic.h>         /* atomic_t, atomic_read, atomic_set, ... */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include "select.h"
// PTB two implementations, one using fcntl, one using atomic.h, see lock.h
#include "lock.h"

static int
lock_try (struct nbd_lock *m)
{
#ifdef USE_FCNTL_LOCK
    struct flock lock = {
      l_type: F_WRLCK,
      l_whence: SEEK_SET,
      l_start: m->off,
      l_len: sizeof(*m),
    };
    if (fcntl(m->mfd, F_SETLK, &lock) < 0) {
#else
    if (!atomic_dec_and_test (&m->count)) {
	atomic_inc (&m->count);
#endif
	return 1;
    }
    return 0; /* success */
}

#ifdef USE_FCNTL_LOCK
 // nada
#else
static inline int
wait_on_timeout(int fd, unsigned long udelay) {
    fd_set rfds;
    FD_ZERO(&rfds);
    FD_SET(fd, &rfds);
    if (microselect (fd, &rfds, NULL, NULL, udelay) == 0) {
        // timeout
        return 0;
    } 
    // event
    return 1;
}

static inline unsigned long
adjust_udelay(int type, unsigned long udelay) {
        switch (type) {
            case 0: // timeout
                     return (udelay <= NBD_LOCK_UDMAX/2) ?
                         udelay << 1 : NBD_LOCK_UDMAX/2 + (udelay >> 1);
                     break;
            case 1: // event
            default:
                     return NBD_LOCK_UDELAY;
                     break;
        }
}
#endif

static void
lock_down (struct nbd_lock *m)
{
#ifdef USE_FCNTL_LOCK
    struct flock lock = {
      l_type: F_WRLCK,
      l_whence: SEEK_SET,
      l_start: m->off,
      l_len: sizeof(*m),
    };
    while (fcntl(m->mfd, F_SETLKW, &lock) < 0);
    // only signals break fcntl
#else
    char c;
    unsigned long udelay = NBD_LOCK_UDELAY;

    while (lock_try (m) != 0) {
        if (m->err) {
            // if we don't have socketpair, just poll without backoff
            microsleep (udelay);
            continue;
        }
        udelay = adjust_udelay(wait_on_timeout(m->fd[1], udelay), udelay);
    }
    if (!m->err) {
        read(m->fd[1],&c,1);
    }
#endif
}
static void
lock_up (struct nbd_lock *m)
{
#ifdef USE_FCNTL_LOCK
    struct flock lock = {
      l_type: F_UNLCK,
      l_whence: SEEK_SET,
      l_start: m->off,
      l_len: sizeof(*m),
    };
    fcntl(m->mfd, F_SETLK, &lock);
#else
    static char c;
    if (!m->err) {             // must write before unlock so that read 
        write(m->fd[0],&c,1);  // always removes our char. If we write 
    }                          // after unlocking the read might miss it
    atomic_inc (&m->count);    // by running too early
#endif
}


static int
rwlock_try_write (struct nbd_rwlock *m)
{
#ifdef USE_FCNTL_LOCK
    struct flock lock = {
      l_type: F_WRLCK,
      l_whence: SEEK_SET,
      l_start: m->off,
      l_len: sizeof(*m),
    };
    if (fcntl(m->mfd, F_SETLK, &lock) < 0) {
	return 1;
    }
#else
    m->lock.down(&m->lock);
    if (m->writers > 0 || m->readers > 0) {
	m->lock.up(&m->lock);
	return 1;
    }
    m->writers++;
    m->lock.up(&m->lock);
#endif
    return 0; // success
}
static void
rwlock_up_write (struct nbd_rwlock *m)
{
#ifdef USE_FCNTL_LOCK
    struct flock lock = {
      l_type: F_UNLCK,
      l_whence: SEEK_SET,
      l_start: m->off,
      l_len: sizeof(*m),
    };
    fcntl(m->mfd, F_SETLK, &lock);
#else
    m->lock.down (&m->lock);
    m->writers--;
    m->lock.up (&m->lock);
#endif
}

static void
rwlock_down_write (struct nbd_rwlock *m)
{
#ifdef USE_FCNTL_LOCK
    struct flock lock = {
      l_type: F_WRLCK,
      l_whence: SEEK_SET,
      l_start: m->off,
      l_len: sizeof(*m),
    };
    while (fcntl(m->mfd, F_SETLKW, &lock) < 0);
    // only signals break fcntl
#else
    char c;
    unsigned long udelay = NBD_LOCK_UDELAY;
    while (m->try_write (m) != 0) {
        if (m->err) {
            // if we don't have socketpair, just poll without backoff
            microsleep (udelay);
            continue;
        }
        udelay = adjust_udelay(wait_on_timeout(m->fd[1], udelay), udelay);
    }
    if (!m->err) {
        read(m->fd[1],&c,1);
    }
#endif
}


static int
rwlock_try_read (struct nbd_rwlock *m)
{
#ifdef USE_FCNTL_LOCK
    struct flock lock = {
      l_type: F_RDLCK,
      l_whence: SEEK_SET,
      l_start: m->off,
      l_len: sizeof(*m),
    };
    if (fcntl(m->mfd, F_SETLK, &lock) < 0) {
	return 1;
    }
#else
    m->lock.down (&m->lock);
    if (m->writers > 0) {
	m->lock.up(&m->lock);
	return 1;
    }
    m->readers++;
    m->lock.up(&m->lock);
#endif
    return 0; // success
}

static void
rwlock_up_read (struct nbd_rwlock *m)
{
#ifdef USE_FCNTL_LOCK
    struct flock lock = {
      l_type: F_UNLCK,
      l_whence: SEEK_SET,
      l_start: m->off,
      l_len: sizeof(*m),
    };
    fcntl(m->mfd, F_SETLK, &lock);
#else
    m->lock.down (&m->lock);
    m->readers--;
    m->lock.up (&m->lock);
#endif
}
static void
rwlock_down_read (struct nbd_rwlock *m)
{
#ifdef USE_FCNTL_LOCK
    struct flock lock = {
      l_type: F_RDLCK,
      l_whence: SEEK_SET,
      l_start: m->off,
      l_len: sizeof(*m),
    };
    while (fcntl(m->mfd, F_SETLKW, &lock) < 0);
    // only signals break fcntl
#else
    char c;
    unsigned long udelay = NBD_LOCK_UDELAY;
    while (m->try_read (m) != 0) {
        if (m->err) {
            // if we don't have socketpair, just poll without backoff
            microsleep (udelay);
            continue;
        }
        udelay = adjust_udelay(wait_on_timeout(m->fd[1], udelay), udelay);
    }
    if (!m->err) {
        read(m->fd[1],&c,1);
    }
#endif
}


void
init_lock (struct nbd_lock *m, int mfd, int off)
{
    m->down = lock_down;
    m->up   = lock_up;
    m->try  = lock_try;
#ifdef USE_FCNTL_LOCK
    m->mfd = mfd;
    m->off = off;
#else
    atomic_set (&m->count, 1);
    m->err  = socketpair(PF_UNIX,SOCK_STREAM,0,m->fd);
    if (m->err == 0) {
        fcntl(m->fd[0], F_SETFL, O_NONBLOCK);
        fcntl(m->fd[1], F_SETFL, O_NONBLOCK);
    }
#endif
}


void
init_rwlock (struct nbd_rwlock *m, int mfd, int off)
{
    init_lock(&m->lock, mfd, off + ((char *)&m->lock - (char *)m));
    m->writers    = m->readers = 0;
    m->down_write = rwlock_down_write;
    m->up_write   = rwlock_up_write;
    m->try_write  = rwlock_try_write;
    m->down_read  = rwlock_down_read;
    m->up_read    = rwlock_up_read;
    m->try_read   = rwlock_try_read;
#ifdef USE_FCNTL_LOCK
    m->mfd = mfd;
    m->off = off;
#else
    m->err  = socketpair(PF_UNIX,SOCK_STREAM,0,m->fd);
    if (m->err == 0) {
        fcntl(m->fd[0], F_SETFL, O_NONBLOCK);
        fcntl(m->fd[1], F_SETFL, O_NONBLOCK);
    }
#endif
}


