/* 
 * Copyright (C) 1999-2001 Peter T. Breuer <ptb@it.uc3m.es>
 */

#ifndef MY_NAME
  #define MY_NAME "nbd-client-server"
#endif
#define ISCLIENT

/*
 * This is the read/write interface to the server. Underneath it calls
 * either the cacheserver or the netserver. This interface is called
 * directly by the nbd-client top-level code whenever it needs to do a
 * read or write (or presumably, a sync).
 *
 * int (*write)();
 * int (*read)();
 * int (*check)();
 * int (*sync)();
 */

#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <syslog.h>
#include <stdlib.h>
#include <setjmp.h>
#include <sys/time.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <ctype.h>
#include <sys/mman.h>

#include <pthread.h>

#ifdef USING_SSL
#include "sslincl.h"
#endif

#include <linux/ioctl.h>
#include <linux/enbd.h>
#include "cliserv.h"


#ifndef NBD_NO_CACHING
  #include "cache/cache.h"
#endif
#include "stream.h"
#include "netserver.h"
#include "cacheserver.h"
#include "server.h"
#include "nbd-client.h"
#include "socket.h"
#include "select.h"

static int
readcache(struct nbd_stub_server * self, char *buf, int len, s64 from) {
     int err = 0;
     int result = 0;
     NBD_REQUEST_BITMAP_T bitmap;
     struct nbd_stub_server * cacheserver
            = (struct nbd_stub_server *)self->altmedia;

     if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return err = -EINVAL;
     }

     memset(bitmap,0,/*NBD_BITMAP_LENGTH*/(((len+511)>>9)+7)>>3);
     //PERR("try cache read %d bytes at sector %d\n",len,(unsigned)(from>>9));
     err = cacheserver->read(cacheserver, buf, len, from, bitmap);
     //PERR("cache read returned %d\n", err);
     if (err >= 0)
       result += err;
     if (err < 0) {
       PERR("readcache exits FAIL\n");
       return err;
     }
     DEBUG("readcache exits OK\n");
     return result;
}

static int
readnet(struct nbd_stub_server * self, char *buf, int len, s64 from) {
     int err = 0;
     int result = 0;
     NBD_REQUEST_BITMAP_T bitmap;
     struct nbd_stub_server * netserver 
            = (struct nbd_stub_server *)self->media;

     if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return err = -EINVAL;
     }

     memset(bitmap,255,/*NBD_BITMAP_LENGTH*/(((len+511)>>9)+7)>>3);
     //PERR("try net read %d bytes at sector %d\n",len,(unsigned)(from>>9));
     err = netserver->read(netserver, buf, len, from, bitmap);
     //PERR("net read returned %d\n", err);
     if (err >= 0)
       result += err;
     if (err < 0) {
       PERR("readnet exits FAIL\n");
       return err;
     }
     DEBUG("readnet exits OK\n");
     return result;
}

static int
writecache(struct nbd_stub_server * self, char *buf, int len, s64 from) {

     int err = 0;
     int result = 0;
     struct nbd_stub_server * cacheserver
            = (struct nbd_stub_server *)self->altmedia;

     DEBUG("writecache enters\n");

     if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return err = -EINVAL;
     }

     //PERR("try cache write %d bytes at sector %d\n",len,(unsigned)(from>>9));
     err = cacheserver->write(cacheserver, buf, len, from);
     //PERR("cache write returned %d for %d bytes at sector %d\n",
     //  err, len, (unsigned)(from>>9));
     if (err >= 0)
       result = err;
     if (err < 0) {
       PERR("writecache exits FAIL\n");
       return err;
     }
     DEBUG("writecache exits OK\n");
     return result;
}

static int
writenet(struct nbd_stub_server * self, char *buf, int len, s64 from) {

     int err = 0;
     int result = 0;
     struct nbd_stub_server * netserver 
            = (struct nbd_stub_server *)self->media;

     DEBUG("writenet enters\n");

     if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return err = -EINVAL;
     }

     //PERR("try net write %d bytes at sector %d\n",len,(unsigned)(from>>9));
     err = netserver->write(netserver, buf, len, from);
     //PERR("net write returned %d for %d bytes at sector %d\n",
     //  err, len, (unsigned)(from>>9));
     if (err >= 0)
       result = err;
     if (err < 0) {
       PERR("writenet exits FAIL\n");
       return err;
     }
     DEBUG("writenet exits OK\n");
     return result;
}



static int
checknet(struct nbd_stub_server * self) {
    struct nbd_stub_server * netserver 
            = (struct nbd_stub_server *)self->media;
    // PTB make dummy request of type READ len 0 to send out
    if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return -EINVAL;
    }
    return netserver->check(netserver);
}

static int
sync_cache(struct nbd_stub_server * self, unsigned how) {
    struct nbd_stub_server * cacheserver 
            = (struct nbd_stub_server *)self->altmedia;
    MSG("cacheserver syncs cache\n");
    if (self->magic != NBD_CLIENT_CACHE_GENMAGIC) {
         PERR("server %x with bad magic\n", (unsigned)self);
         return -EINVAL;
    }
    return cacheserver->sync(cacheserver,how);
}

static void setcache(struct nbd_stub_server *self, unsigned long on) {

  if (on) {
    self->flags |= NBD_CLIENT_CACHE_ENABLED;
    self->read = readcache;
    self->write= writecache;
    self->sync = sync_cache;
  } else {
    self->flags &= ~NBD_CLIENT_CACHE_ENABLED;
    self->read = readnet;
    self->write= writenet;
    self->sync = 0;
  }
}

void init_stub_server(struct nbd_stub_server *self,
    unsigned long flags, int i, u64 size, int journal,
    //struct nbd_stream * stream, struct nbd_cache * cache
    struct nbd_stub_server * netserver, struct nbd_stub_server * cacheserver
    ) {

  //struct nbd_stub_server *netserver = NULL, *cacheserver = NULL;
  
  self->flags  = 0;
  if (!(flags & NBD_CLIENT_SERVER_RO))
     self->flags |= NBD_CLIENT_CACHE_WRITETHROUGH;
  self->i      = i;
  self->size   = size;
  if (journal >= -1) {
     self->flags |= NBD_CLIENT_CACHE_WRITEBACK | NBD_CLIENT_CACHE_READBACK;
     self->flags |= NBD_CLIENT_CACHE_ENABLED;
  }
  self->flags |= NBD_CLIENT_CACHE_READTHROUGH;

  //self->media = calloc(1,sizeof(struct nbd_stub_server));
  //if (!self->media) {
  //  PERR("cannot allocate %dB of memory\n", sizeof(struct nbd_stub_server));
  //  exit(1);
  //}
  //self->altmedia = calloc(1,sizeof(struct nbd_stub_server));
  //if (!self->altmedia) {
  //  PERR("cannot allocate %dB of memory\n", sizeof(struct nbd_stub_server));
  //  exit(1);
  //}

  if (self->flags & NBD_CLIENT_CACHE_ENABLED) {
    setcache(self, 1);
  } else {
    setcache(self, 0);
  }
  self->check  = checknet;
  self->set    = setcache;

  //netserver   = (struct nbd_stub_server *)self->media;
  //cacheserver = (struct nbd_stub_server *)self->altmedia;

  self->media = netserver;
  self->altmedia = cacheserver;

  //init_netserver(netserver,     i, size, stream, flags&NBD_CLIENT_SERVER_RO, cacheserver); 
  //init_cacheserver(cacheserver, i, size, cache, flags&NBD_CLIENT_SERVER_RO, netserver); 

  self->magic = NBD_CLIENT_CACHE_GENMAGIC;

}

