#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "guirgbimg.h"

#include "edv_types.h"
#include "edv_mime_type.h"
#include "edv_utils_gtk.h"


/* MIME Type Commands */
const gchar *EDVMimeTypeGetCommandValueByName(
	edv_mime_type_struct *m, const gchar *name
);
edv_mime_type_command_struct *EDVMimeTypeGetCommandByName(
	edv_mime_type_struct *m, const gchar *name
);

edv_mime_type_command_struct *EDVMimeTypeCommandNew(void);
edv_mime_type_command_struct *EDVMimeTypeCommandCopy(
	edv_mime_type_command_struct *cmd
);
void EDVMimeTypeCommandDelete(edv_mime_type_command_struct *cmd);


/* MIME Types */
edv_mime_type_struct *EDVMimeTypeNew(
	const edv_mime_type_class mt_class,
	const gchar *value,
	const gchar *type,              /* MIME Type name */
	const gchar *description	/* Short verbose description */
);
edv_mime_type_struct *EDVMimeTypeCopy(
	edv_mime_type_struct *m
);
static void EDVMimeTypeLoadIconsNexus(
	GdkPixmap **pixmap, GdkBitmap **mask,
	guint8 ***data, const gchar **file,
	const gint req_width, const gint req_height,
	const gboolean allow_resize
);
void EDVMimeTypeLoadSmallIconsData(
	edv_mime_type_struct *m, guint8 ***data
);
void EDVMimeTypeLoadMediumIconsData(
	edv_mime_type_struct *m, guint8 ***data
);
void EDVMimeTypeLoadLargeIconsData(
	edv_mime_type_struct *m, guint8 ***data
);

void EDVMimeTypeRealize(
	edv_mime_type_struct *m, const gboolean force_rerealize
);

void EDVMimeTypeDelete(edv_mime_type_struct *m);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the command value of the MIME Type command that matches
 *	the specified command name.
 */
const gchar *EDVMimeTypeGetCommandValueByName(
	edv_mime_type_struct *m, const gchar *name
)
{
	GList *glist;
	edv_mime_type_command_struct *cmd;

	if((m == NULL) || STRISEMPTY(name))
	    return(NULL);

	for(glist = m->commands_list; glist != NULL; glist = g_list_next(glist))
	{
	    cmd = EDV_MIME_TYPE_COMMAND(glist->data);
	    if(cmd == NULL)
		continue;

	    if(cmd->name == NULL)
		continue;

	    if(!g_strcasecmp(cmd->name, name))
		return(cmd->command);
	}

	return(NULL);
}

/*
 *	Gets the MIME Type command that matches the specified command
 *	name.
 */
edv_mime_type_command_struct *EDVMimeTypeGetCommandByName(
	edv_mime_type_struct *m, const gchar *name
)
{
	GList *glist;
	edv_mime_type_command_struct *cmd;

	if((m == NULL) || STRISEMPTY(name))
	    return(NULL);

	for(glist = m->commands_list; glist != NULL; glist = g_list_next(glist))
	{
	    cmd = EDV_MIME_TYPE_COMMAND(glist->data);
	    if(cmd == NULL)
		continue;

	    if(cmd->name == NULL)
		continue;

	    if(!g_strcasecmp(cmd->name, name))
		return(cmd);
	}

	return(NULL);
}


/*
 *	Creates a new MIME Type command.
 */
edv_mime_type_command_struct *EDVMimeTypeCommandNew(void)
{
	return(EDV_MIME_TYPE_COMMAND(g_malloc0(
	    sizeof(edv_mime_type_command_struct)
	)));
}

/*
 *	Coppies the MIME Type command.
 */
edv_mime_type_command_struct *EDVMimeTypeCommandCopy(
	edv_mime_type_command_struct *cmd
)
{
	edv_mime_type_command_struct *cmd2;

	if(cmd == NULL)
	    return(NULL);

	cmd2 = EDVMimeTypeCommandNew();
	if(cmd2 == NULL)
	    return(NULL);

	cmd2->name = STRDUP(cmd->name);
	cmd2->command = STRDUP(cmd->command);

	return(cmd2);
}

/*
 *	Deletes the MIME Type command.
 */
void EDVMimeTypeCommandDelete(edv_mime_type_command_struct *cmd)
{
	if(cmd == NULL)
	    return;

	g_free(cmd->name);
	g_free(cmd->command);
	g_free(cmd);
}


/*
 *	Creates a new MIME Type.
 */
edv_mime_type_struct *EDVMimeTypeNew(
	const edv_mime_type_class mt_class,
	const gchar *value,
	const gchar *type,
	const gchar *description
)
{
	edv_mime_type_struct *m = EDV_MIME_TYPE(
	    g_malloc0(sizeof(edv_mime_type_struct))
	);
	if(m == NULL)
	    return(m);

	m->mt_class = mt_class;

	m->value = STRDUP(value);
	m->type = STRDUP(type);
	m->description = STRDUP(description);

	m->realized = FALSE;

	m->read_only = FALSE;

	memset(
	    m->small_pixmap, 0x00,
	    EDV_MIME_TYPE_TOTAL_ICON_STATES * sizeof(GdkPixmap *)
	);
	memset(
	    m->small_mask, 0x00,
	    EDV_MIME_TYPE_TOTAL_ICON_STATES * sizeof(GdkBitmap *)
	);

	memset(
	    m->medium_pixmap, 0x00,
	    EDV_MIME_TYPE_TOTAL_ICON_STATES * sizeof(GdkPixmap *)
	);
	memset(
	    m->medium_mask, 0x00,
	    EDV_MIME_TYPE_TOTAL_ICON_STATES * sizeof(GdkBitmap *)
	);

	memset(
	    m->large_pixmap, 0x00,
	    EDV_MIME_TYPE_TOTAL_ICON_STATES * sizeof(GdkPixmap *)
	);
	memset(
	    m->large_mask, 0x00,
	    EDV_MIME_TYPE_TOTAL_ICON_STATES * sizeof(GdkBitmap *)
	);

	m->handler = EDV_MIME_TYPE_HANDLER_COMMAND;

	m->commands_list = NULL;

	m->access_time = 0l;
	m->modify_time = 0l;
	m->change_time = 0l;

	return(m);
}

/*
 *	Coppies the MIME Type.
 */
edv_mime_type_struct *EDVMimeTypeCopy(
	edv_mime_type_struct *m
)
{
	gint i;
	GList *glist;
	edv_mime_type_struct *m2;

	if(m == NULL)
	    return(NULL);

	m2 = EDVMimeTypeNew(
	    m->mt_class,
	    m->value,
	    m->type,
	    m->description
	);
	if(m2 == NULL)
	    return(NULL);

	m2->realized = m->realized;
	m2->read_only = m->read_only;

	for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	{
	    GDK_PIXMAP_REF(m->small_pixmap[i]);
	    m2->small_pixmap[i] = m->small_pixmap[i];

	    GDK_BITMAP_REF(m->small_mask[i]);
	    m2->small_mask[i] = m->small_mask[i];

	    m2->small_icon_file[i] = STRDUP(m->small_icon_file[i]);
	}

	for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	{
	    GDK_PIXMAP_REF(m->medium_pixmap[i]);
	    m2->medium_pixmap[i] = m->medium_pixmap[i];

	    GDK_BITMAP_REF(m->medium_mask[i]);
	    m2->medium_mask[i] = m->medium_mask[i];

	    m2->medium_icon_file[i] = STRDUP(m->medium_icon_file[i]);
	}

	for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	{
	    GDK_PIXMAP_REF(m->large_pixmap[i]);
	    m2->large_pixmap[i] = m->large_pixmap[i];

	    GDK_BITMAP_REF(m->large_mask[i]);
	    m2->large_mask[i] = m->large_mask[i];

	    m2->large_icon_file[i] = STRDUP(m->large_icon_file[i]);
	}

	m2->handler = m->handler;

	m2->commands_list = NULL;
	for(glist = m->commands_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	    m2->commands_list = g_list_append(
		m2->commands_list,
		EDVMimeTypeCommandCopy(EDV_MIME_TYPE_COMMAND(
		    glist->data
		))
	    );

	m2->access_time = m->access_time;
	m2->modify_time = m->modify_time;
	m2->change_time = m->change_time;

	return(m2);
}

/*
 *	Loads the pixmap and mask pairs list.
 *
 *	The pixmap and mask specifies the pixmap and mask pairs list,
 *	any existing (non NULL) pixmap and mask pairs in this list will
 *	be unref'ed.
 *
 *	The data specifies the XPM data. If data is not NULL then the
 *	pixmap and mask pair will be loaded from the XPM data.
 *
 *	The file specifies the path to the XPM file. If file is not NULL
 *	then the pixmap and mask pair will be loaded from the XPM file.
 *
 *	If both data and file are NULL then the existing pixmap and
 *	mask pair will be unref'ed and no new pixmap and mask pair will
 *	be loaded.
 *
 *	The req_width and req_height specifies the requested size in
 *	pixels.
 *
 *	If allow_resize is TRHE then the pixmap and mask pair will be
 *	resized to the requested size as needed.
 */
static void EDVMimeTypeLoadIconsNexus(
	GdkPixmap **pixmap, GdkBitmap **mask,
	guint8 ***data, const gchar **file,
	const gint req_width, const gint req_height,
	const gboolean allow_resize
)
{
	gint i;

	/* Unload the icon pixmap and masks */
	for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	{
	    GDK_PIXMAP_UNREF(pixmap[i]);
	    pixmap[i] = NULL;
	    GDK_BITMAP_UNREF(mask[i]);
	    mask[i] = NULL;
	}

	/* Load icons if the XPM datas are specified */
	if(data != NULL)
	{
	    GdkBitmap *m;
	    GdkPixmap *p;

	    /* Load the icons */
	    for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	    {
		p = GDK_PIXMAP_NEW_FROM_XPM_DATA(&m, data[i]);
		if(p == NULL)
		    continue;

		if(allow_resize)
		{
		    EDVResizePixmap(
			p, m,
			req_width, req_height,
			&pixmap[i], &mask[i]
		    );
		    GDK_PIXMAP_UNREF(p);
		    GDK_BITMAP_UNREF(m);
		}
		else
		{
		    pixmap[i] = p;
		    mask[i] = m;
		}
	    }
	}
	/* Load icons if the XPM files are specified */
	else if(file != NULL)
	{
	    GdkBitmap *m;
	    GdkPixmap *p;

	    /* Load the icons */
	    for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	    {
		p = GDK_PIXMAP_NEW_FROM_XPM_FILE(&m, file[i]);
		if(p == NULL)
		    continue;

		if(allow_resize)
		{
		    EDVResizePixmap(
			p, m,
			req_width, req_height,
			&pixmap[i], &mask[i]
		    );
		    GDK_PIXMAP_UNREF(p);
		    GDK_BITMAP_UNREF(m);
		}
		else
		{
		    pixmap[i] = p;
		    mask[i] = m;
		}
	    }
	}
}

/*
 *	Loads the MIME Type's small sized icons.
 *
 *	The given icon data must be an array of
 *	EDV_MIME_TYPE_TOTAL_ICON_STATES guint8 ** pointers.
 *
 *	Passing data as NULL will only unref the existing icons (if any)
 *	and not load new ones.
 */
void EDVMimeTypeLoadSmallIconsData(
	edv_mime_type_struct *m, guint8 ***data
)
{
	if(m == NULL)
	    return;

	EDVMimeTypeLoadIconsNexus(
	    m->small_pixmap, m->small_mask,
	    data, NULL,
	    20, 20, TRUE
	);
}

/*
 *	Loads the MIME Type's medium sized icons.
 *
 *	The given icon data must be an array of
 *	EDV_MIME_TYPE_TOTAL_ICON_STATES guint8 ** pointers.
 *
 *	Passing data as NULL will only unref the existing icons (if any)
 *	and not load new ones.
 */
void EDVMimeTypeLoadMediumIconsData(
	edv_mime_type_struct *m, guint8 ***data
)
{
	if(m == NULL)
	    return;

	EDVMimeTypeLoadIconsNexus(
	    m->medium_pixmap, m->medium_mask,
	    data, NULL,
	    32, 32, TRUE
	);
}

/*
 *	Loads the MIME Type's large sized icons.
 *
 *	The given icon data must be an array of
 *	EDV_MIME_TYPE_TOTAL_ICON_STATES guint8 ** pointers.
 *
 *	Passing data as NULL will only unref the existing icons (if any)
 *	and not load new ones.
 */
void EDVMimeTypeLoadLargeIconsData(
	edv_mime_type_struct *m, guint8 ***data
)
{
	if(m == NULL)
	    return;

	EDVMimeTypeLoadIconsNexus(
	    m->large_pixmap, m->large_mask,
	    data, NULL,
	    48, 48, TRUE
	);
}


/*
 *	Realizes the MIME Type, loading the icons and related resources.
 *
 *	If the MIME Type is already realized (member realized is TRUE)
 *	then nothing will be done, however if force_rerealize is TRUE
 *	then the above check will be ignored.
 */
void EDVMimeTypeRealize(
	edv_mime_type_struct *m, const gboolean force_rerealize
)
{
	const gchar **path;

	if(m == NULL)
	    return;

	/* Not forcing a re-realize? */
	if(!force_rerealize)
	{
	    /* MIME Type already realized? */
	    if(m->realized)
		return;
	}

	/* Begin realizing this MIME Type */

	/* Get paths for small sized icons and load them */
	path = (const gchar **)m->small_icon_file;
	if(!STRISEMPTY(path))
	    EDVMimeTypeLoadIconsNexus(
		m->small_pixmap, m->small_mask,
		NULL, path,
		20, 20, TRUE
	    );

	/* Get paths for medium sized icons and load them */
	path = (const gchar **)m->medium_icon_file;
	if(!STRISEMPTY(path))
	    EDVMimeTypeLoadIconsNexus(
		m->medium_pixmap, m->medium_mask,
		NULL, path,
		32, 32, TRUE
	    );

	/* Get paths for large sized icons and load them */
	path = (const gchar **)m->large_icon_file;
	if(!STRISEMPTY(path))
	    EDVMimeTypeLoadIconsNexus(
		m->large_pixmap, m->large_mask,
		NULL, path,
		48, 48, TRUE
	    );

	m->realized = TRUE;		/* Mark as realized */
}

/*
 *	Deletes the MIME Type.
 */
void EDVMimeTypeDelete(edv_mime_type_struct *m)
{
	gint i;

	if(m == NULL)
	    return;

	/* Call the icon loaders with NULL as the argument, this will
	 * only unload the icons
	 */
	EDVMimeTypeLoadSmallIconsData(m, NULL);
	EDVMimeTypeLoadMediumIconsData(m, NULL);
	EDVMimeTypeLoadLargeIconsData(m, NULL);

	/* Delete icon file paths */
	for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	    g_free(m->small_icon_file[i]);

	for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	    g_free(m->medium_icon_file[i]);

	for(i = 0; i < EDV_MIME_TYPE_TOTAL_ICON_STATES; i++)
	    g_free(m->large_icon_file[i]);

	/* Delete the commands */
	if(m->commands_list != NULL)
	{
	    g_list_foreach(
		m->commands_list, (GFunc)EDVMimeTypeCommandDelete, NULL
	    );
	    g_list_free(m->commands_list);
	}

	g_free(m->value);

	g_free(m->type);

	g_free(m->description);

	g_free(m);
}
