/*
			    Thumbs List
 */

#ifndef TLIST_H
#define TLIST_H

#include <gtk/gtk.h>


typedef struct _tlist_struct		tlist_struct;
#define TLIST(p)			((tlist_struct *)(p))
typedef struct _tlist_thumb_struct	tlist_thumb_struct;
#define TLIST_THUMB(p)			((tlist_thumb_struct *)(p))


/*
 *	Widget Name:
 */
#define TLIST_WIDGET_NAME		"thumbs-list"


/*
 *	Flags:
 */
typedef enum {
	TLIST_HORIZONTAL		= (1 << 0),	/* Tile thumbs horizontally */
	TLIST_DOUBLE_BUFFER		= (1 << 1),	/* Use double buffer */
	TLIST_SHOW_THUMB_FRAMES		= (1 << 2),	/* Show Thumb Frames */
 	TLIST_SHOW_THUMB_LABELS		= (1 << 3),	/* Show Thumb Labels */
	TLIST_SHOW_TEXTTIPS		= (1 << 4),	/* Show Texttips */
	TLIST_DRAG_ACTIVE		= (1 << 16),	/* A drag is in progress */
	TLIST_DRAG_ACTIVE_MOVE		= (1 << 17)	/* A drag move is in progress */
} tlist_flags;


/*
 *	Thumb Flags:
 */
typedef enum {
	TLIST_THUMB_SENSITIVE		= (1 << 0),
	TLIST_THUMB_SELECTABLE		= (1 << 1)
} tlist_thumb_flags;


/*
 *	Thumb Load States:
 */
typedef enum {
	TLIST_LOAD_STATE_NOT_LOADED,
	TLIST_LOAD_STATE_LOADED,
	TLIST_LOAD_STATE_FAILED
} tlist_load_state;


/*
 *	Thumb:
 */
struct _tlist_thumb_struct {

	tlist_thumb_flags	flags;
	tlist_load_state	load_state;
	GdkColormap	*colormap;

	GdkPixmap	*pixmap;
	GdkBitmap	*mask;

	gchar		*text;
	GtkVisibility	text_visibility;
	GdkColor	*text_color_fg,
			*text_color_bg;

	gulong		loaded_time;	/* In systime seconds (user set) */

	gpointer	data;

	/* Destroy Callback */
	void		(*destroy_cb)(
		gpointer		/* Data */
	);

};
#define TLIST_THUMB_IS_SELECTABLE(_thumb_)			\
 (((_thumb_) != NULL) ?						\
  ((_thumb_)->flags & TLIST_THUMB_SELECTABLE) : FALSE		\
 )


/*
 *	Thumbs List:
 */
struct _tlist_struct {

	GtkWidget	*toplevel;
	gboolean	map_state;
	gint		freeze_count;
	tlist_flags	flags;

	GdkColormap	*colormap;
	GdkGC		*gc;

	GtkWidget	*list_da,	/* List GtkDrawingArea */
			*vsb,		/* Vertical GtkScrollBar */
			*hsb;		/* Horizontal GtkScrollBar */

	GdkPixmap	*list_pm;	/* List GdkPixmap */

	GtkAdjustment	*vadj,		/* Scroll adjustments */
			*hadj;

	GdkBitmap	*transparent_stipple_bm;

	/* Thumbs */
	tlist_thumb_struct	**thumb;
	gint			total_thumbs;

	/* Selected thumbs, the glist->data is a (gint) that refers to
	 * a thumb's index */
	GList		*selection,
			*selection_end;

	gint		focus_thumb,	/* Thumb in focus or -1 for none */
			pointer_over_thumb;	/* Pointer is over this
						 * thumb or -1 for none */
	GtkSelectionMode	selection_mode;

	/* Thumb geometry */
	gint		thumb_width,	/* Size of each thumb */
			thumb_height,
			thumb_border,	/* Padding between a thumb's contents
					 * and its frame */
			thumbs_per_line;	/* Thumbs visible on one "line" */

	/* Callbacks */
	void	(*select_cb)(
		tlist_struct *,		/* Thumbs List */
		GdkEventButton *,	/* GdkEventButton */
		gint,			/* Thumb */
		gpointer		/* Data */
	);
	gpointer	select_data;

	void	(*unselect_cb)(
		tlist_struct *,		/* Thumbs List */
		GdkEventButton *,	/* GdkEventButton */
		gint,			/* Thumb */
		gpointer		/* Data */
	);
	gpointer	unselect_data;

};


/* Utils */
extern void TListQueryThumbPixmapSize(
	tlist_struct *tlist,
	gint img_width, gint img_height,
	gint *width, gint *height
);


/* Drawing */
extern void TListDraw(tlist_struct *tlist);
extern void TListQueueDraw(tlist_struct *tlist);


/* Resize */
extern void TListResize(tlist_struct *tlist, gint width, gint height);


/* Freeze/Thaw */
extern void TListFreeze(tlist_struct *tlist);
extern void TListThaw(tlist_struct *tlist);


/* Thumbs Add, Set, and Delete */
extern gint TListInsert(
	tlist_struct *tlist, const gint thumb_num, const gchar *text
);
extern gint TListAppend(
	tlist_struct *tlist, const gchar *text
);
extern void TListSetLoadState(
	tlist_struct *tlist, const gint thumb_num,
	const tlist_load_state load_state
);
extern void TListSetText(
	tlist_struct *tlist, const gint thumb_num,
	const gchar *text
);
extern void TListSetTextColor(
	tlist_struct *tlist, const gint thumb_num,
	GdkColor *fg, GdkColor *bg
);
extern void TListSetPixmap(
	tlist_struct *tlist, const gint thumb_num,
	GdkPixmap *pixmap, GdkBitmap *mask
);
extern void TListSetRGBA(
	tlist_struct *tlist, const gint thumb_num,
	gint width, gint height,        /* Of image data */
	gint bpl,                       /* Bytes per line, can be -1 */
	GdkRgbDither dith,
	const guint8 *rgba,
	gboolean no_enlarge		/* Do not upscale if image smaller
					 * than thumb */
);
extern void TListSetThumbData(
	tlist_struct *tlist, const gint thumb_num,
	gpointer data
);
extern void TListSetThumbDataFull(
	tlist_struct *tlist, const gint thumb_num,
	gpointer data,
	void (*destroy_cb)(gpointer)
);

extern void TListRemove(tlist_struct *tlist, const gint thumb_num);
extern void TListClear(tlist_struct *tlist);

/* Thumbs Get */
extern tlist_load_state TListGetLoadState(
	tlist_struct *tlist, const gint thumb_num
);
extern gboolean TListGetText(
	tlist_struct *tlist, gint thumb_num, gchar **text
);
extern gboolean TListGetPixmap(
	tlist_struct *tlist, gint thumb_num,
	GdkPixmap **pixmap, GdkBitmap **mask
);
extern gpointer TListGetThumbData(tlist_struct *tlist, gint thumb_num);

/* Thumbs Find */
extern gint TListFindThumbFromData(tlist_struct *tlist, gpointer data);

/* Selecting */
extern void TListSelectThumb(tlist_struct *tlist, gint thumb_num);
extern void TListUnselectThumb(tlist_struct *tlist, gint thumb_num);
extern void TListSelectAll(tlist_struct *tlist);
extern void TListUnselectAll(tlist_struct *tlist);

extern gboolean TListGetSelection(
	tlist_struct *tlist, gint x, gint y,
	gint *thumb_num, gint *thumb_ix, gint *thumb_iy
);
extern gboolean TListGetThumbPosition(
	tlist_struct *tlist, gint thumb_num,
	gint *x, gint *y
);
extern gboolean TListGetThumbPixmapGeometry(
	tlist_struct *tlist, gint thumb_num,
	gint *x, gint *y, gint *width, gint *height
);
extern gboolean TListGetThumbLabelGeometry(
	tlist_struct *tlist, gint thumb_num,
	gint *x, gint *y, gint *width, gint *height
);


/* Visibility */
extern GtkVisibility TListIsThumbVisible(tlist_struct *tlist, gint thumb_num);


/* Scrolling */
extern void TListMoveTo(
	tlist_struct *tlist, gint thumb_num, gfloat coeff
);


/* Thumbs List */
extern tlist_struct *TListNew(
	const GtkOrientation orientation,
	const gint thumb_width, const gint thumb_height,
	const gint thumb_border,
	void (*select_cb)(tlist_struct *, GdkEventButton *, gint, gpointer),
	gpointer select_data,
	void (*unselect_cb)(tlist_struct *, GdkEventButton *, gint, gpointer),
	gpointer unselect_data
);
extern void TListThumbGeometry(
	tlist_struct *tlist,
	gint thumb_width, gint thumb_height, gint thumb_border
);
extern void TListSelectionMode(
	tlist_struct *tlist, GtkSelectionMode selection_mode
);
extern void TListDoubleBuffer(tlist_struct *tlist, gboolean double_buffer);
extern void TListOrientation(tlist_struct *tlist, const GtkOrientation orientation);
extern void TListShowThumbFrames(tlist_struct *tlist, gboolean show);
extern void TListShowThumbLabels(tlist_struct *tlist, gboolean show);
extern void TListShowTextTips(tlist_struct *tlist, gboolean show);
extern void TListMap(tlist_struct *tlist);
extern void TListUnmap(tlist_struct *tlist);
extern void TListDelete(tlist_struct *tlist);


#endif	/* TLIST_H */
