#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "guiutils.h"
#include "toolbar.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_list_cb.h"
#include "edv_obj.h"
#include "edv_mime_type.h"
#include "edv_device.h"
#include "edv_status_bar.h"
#include "browser.h"
#include "browser_cb.h"
#include "browser_op_cb.h"
#include "browser_dir_tree.h"
#include "browser_contents_list.h"
#include "browser_dnd.h"
#include "endeavour2.h"
#include "edv_cb.h"
#include "edv_open_to_menu.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_hsep_20x20.xpm"
#include "images/icon_terminal2_20x20.xpm"
#include "images/icon_copy_20x20.xpm"
#include "images/icon_paste_20x20.xpm"
#include "images/icon_new_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_trace_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_exit_20x20.xpm"
#include "images/icon_mount_20x20.xpm"
#include "images/icon_unmount_20x20.xpm"
#include "images/icon_eject_20x20.xpm"
#include "images/icon_move_file_20x20.xpm"
#include "images/icon_copy_file_20x20.xpm"
#include "images/icon_rename_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_chmod_20x20.xpm"
#include "images/icon_owned_20x20.xpm"
#include "images/icon_time_stamp_20x20.xpm"
#include "images/icon_properties2_20x20.xpm"
#include "images/icon_run_20x20.xpm"
#include "images/icon_select_all_20x20.xpm"
#include "images/icon_unselect_all_20x20.xpm"
#include "images/icon_invert_selection_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_download_file_20x20.xpm"
#include "images/icon_search_20x20.xpm"
#include "images/icon_mimetypes_20x20.xpm"
#include "images/icon_devices_list_20x20.xpm"
#include "images/icon_tool_bar_20x20.xpm"
#include "images/icon_status_bar_20x20.xpm"
#include "images/icon_wildcards_20x20.xpm"
#include "images/icon_resize_column_20x20.xpm"
#include "images/icon_options2_20x20.xpm"
#include "images/icon_customize_20x20.xpm"
#include "images/icon_tree_origin_20x20.xpm"
#include "images/icon_insecure_20x20.xpm"
#include "images/icon_secure_20x20.xpm"
#include "images/icon_fsck_20x20.xpm"
#include "images/icon_tools_20x20.xpm"
#include "images/icon_floppy_20x20.xpm"

#include "images/icon_open_20x20.xpm"
#include "images/icon_file_20x20.xpm"
#include "images/icon_folder_closed_20x20.xpm"
#include "images/icon_folder_parent_20x20.xpm"
#include "images/icon_folder_home_20x20.xpm"
#include "images/icon_link2_20x20.xpm"
#include "images/icon_pipe_20x20.xpm"
#include "images/icon_device_block_20x20.xpm"
#include "images/icon_device_character_20x20.xpm"
#include "images/icon_socket_20x20.xpm"

#include "images/icon_trash_empty_20x20.xpm"
#include "images/icon_trash_20x20.xpm"
#include "images/icon_purge_20x20.xpm"
#include "images/icon_help_20x20.xpm"
#include "images/icon_about_20x20.xpm"

#include "images/endeavour_20x20.xpm"
#include "images/endeavour_48x48.xpm"
#include "images/image_browser_20x20.xpm"
#include "images/archiver_20x20.xpm"


/* Utilities */
edv_browser_opid_struct *EDVBrowserMatchOPID(
	edv_browser_struct *browser,
	const edv_browser_op op
);
gboolean EDVBrowserIsPathFromDirTreeOrigin(
	edv_browser_struct *browser,
	const gchar *path
);

/* Title */
void EDVBrowserSetTitle(
	edv_browser_struct *browser,
	const gchar *path
);

/* Location */
const gchar *EDVBrowserCurrentLocation(edv_browser_struct *browser);
void EDVBrowserSetLocation(
	edv_browser_struct *browser,
	const gchar *path,
	const gboolean record_history
);
void EDVBrowserUpdateLocationIcon(
	edv_browser_struct *browser,
	const gchar *path
);

/* Selected Objects & Paths */
GList *EDVBrowserGetSelectedObjectsList(
	edv_browser_struct *browser,
	const gboolean copy
);
GList *EDVBrowserGetSelectedPathsList(edv_browser_struct *browser);

/* Regenerate Menus, Styles, and Bars */
void EDVBrowserAccelkeysRegenerate(edv_browser_struct *browser);
void EDVBrowserListStylesRegenerate(edv_browser_struct *browser);
void EDVBrowserToolBarRegenerate(edv_browser_struct *browser);
void EDVBrowserFileMenuItemsRegenerate(edv_browser_struct *browser);
void EDVBrowserNewObjectMenuItemsRegenerate(edv_browser_struct *browser);
void EDVBrowserOpenToMenuRegenerate(
	edv_browser_struct *browser,
	edv_mime_type_struct *m
);
void EDVBrowserDirectoryMenuRegenerate(edv_browser_struct *browser);
void EDVBrowserContentsMenuRegenerate(edv_browser_struct *browser);

/* Sync Configuration */
void EDVBrowserSyncConfiguration(edv_browser_struct *browser);

/* Operation IDs */
GList *EDVBrowserOPIDListNew(edv_browser_struct *browser);
void EDVBrowserOPIDListDelete(GList *glist);

/* Accel Keys */
cfg_accelkey_list_struct *EDVBrowserAccelkeysListNewDefault(void);

/* File Browser */
static void EDVBrowserCreateMenuBar(
	edv_browser_struct *browser, GtkWidget *parent
);
edv_browser_struct *EDVBrowserNew(edv_core_struct *core);
void EDVBrowserSyncData(edv_browser_struct *browser);
void EDVBrowserUpdateMenus(edv_browser_struct *browser);
void EDVBrowserSetBusy(edv_browser_struct *browser, const gboolean busy);
gboolean EDVBrowserIsMapped(edv_browser_struct *browser);
void EDVBrowserMap(edv_browser_struct *browser);
void EDVBrowserUnmap(edv_browser_struct *browser);
void EDVBrowserDelete(edv_browser_struct *browser);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns the File Browser Operation ID that matches the given
 *	operation.
 */
edv_browser_opid_struct *EDVBrowserMatchOPID(
	edv_browser_struct *browser,
	const edv_browser_op op
)
{
	GList *glist;
	edv_browser_opid_struct *opid;

	if(browser == NULL)
	    return(NULL);

	for(glist = browser->opid_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    opid = EDV_BROWSER_OPID(glist->data);
	    if(opid == NULL)
		continue;

	    if(opid->op == op)
		return(opid);
	}

	return(NULL);
}

/*
 *	Checks if the path is within the current directory origin.
 *
 *	The path specifies the string describing the full path to
 *	test against the current directory tree origin.
 *
 *	Returns TRUE if the path falls within the current directory
 *	tree origin or FALSE if it falls outside of the current
 *	directory tree origin or error has occured.
 */
gboolean EDVBrowserIsPathFromDirTreeOrigin(
	edv_browser_struct *browser,
	const gchar *path
)
{
	gchar	*origin_path, *test_path,
		*origin_path_ptr, *test_path_ptr;

	if((browser == NULL) || (path == NULL))
	    return(FALSE);

	/* Get/copy the origin path and simplify it */
	origin_path = STRDUP(browser->directory_ctree_origin_path);
	if(origin_path == NULL)
	    return(FALSE);

	EDVSimplifyPath(origin_path);

	/* Get/copy the specified path and simplify it */
	test_path = STRDUP(path);
	if(test_path == NULL)
	{
	    g_free(origin_path);
	    return(FALSE);
	}

	EDVSimplifyPath(test_path);

	/* Seek through the origin path and the specified path to see
	 * if any characters differ
	 */
	origin_path_ptr = origin_path;
	test_path_ptr = test_path;
	while((*origin_path_ptr != '\0') && (*test_path_ptr != '\0'))
	{
	    /* If the characters differ then the specified path falls
	     * outside of the origin path
	     */
	    if(*origin_path_ptr != *test_path_ptr)
	    {
		g_free(origin_path);
		g_free(test_path);
		return(FALSE);
	    }

	    origin_path_ptr++;
	    test_path_ptr++;
	}

	/* Reached the end of the specified path in the above
	 * iteration?
	 */
	if(*test_path_ptr == '\0')
	{
	    /* If the origin path and the specified path are the
	     * same then the specified path falls within it otherwise
	     * it falls outside of it since the specified path is
	     * shorter than the origin path
	     */
	    if(*origin_path_ptr == '\0')
	    {
		/* Paths are the same */
		g_free(origin_path);
		g_free(test_path);
                return(TRUE);
	    }
	    else
	    {
		/* Paths are different and the origin path is longer
		 * than the specified path so therefore the specified
		 * path falls outside of the origin path
		 */
		g_free(origin_path);
		g_free(test_path);
		return(FALSE);
	    }
	}

	/* At this point the specified path is longer than the origin
	 * path
	 *
	 * Check if the specified path is a child/grandchild of the
	 * origin path by checking if the specified path's character
	 * at this length is a deliminator character
	 */
	if(*test_path_ptr == G_DIR_SEPARATOR)
	{
	    g_free(origin_path);
	    g_free(test_path);
	    return(TRUE);
	}

	/* Delete the coppies of the origin and specified paths */
	g_free(origin_path);
	g_free(test_path);

	/* The specified path is not a child of the origin path
	 * and is not the same as the origin path so therefore the
	 * specified path falls outside of it
	 */
	return(FALSE);
}

/*
 *	Sets the title.
 *
 *	The path specifies the full path to the current location
 *	that is to be displayed on the title.
 */
void EDVBrowserSetTitle(
	edv_browser_struct *browser,
	const gchar *path
)
{
	gboolean include_user_name, include_location;
	const gchar *prog_name = PROG_NAME_FULL;
	gchar *title;
	GtkWidget *w;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	w = browser->toplevel;
	core = browser->core;
	cfg_list = core->cfg_list;

	if(!STRISEMPTY(core->title))
	    prog_name = core->title;

	include_user_name = EDV_GET_B(EDV_CFG_PARM_BROWSER_TITLE_WITH_USER);
	include_location = EDV_GET_B(EDV_CFG_PARM_BROWSER_TITLE_WITH_LOCATION);

	/* Generate the new title string */
	if(include_user_name && include_location &&
	   (path != NULL)
	)
	    title = g_strdup_printf(
		"%s: %s %s",
		prog_name,
		core->effective_user_id_host_str,
		path
	    );
	else if(include_location && (path != NULL))
	    title = g_strdup_printf(
		"%s: %s",
		prog_name, path
	    );
	else if(include_user_name)
	    title = g_strdup_printf(
		"%s: %s",
		prog_name,
		core->effective_user_id_host_str
	    );
	else
	    title = STRDUP(prog_name);

	/* Set the new title */
	gtk_window_set_title(GTK_WINDOW(w), title);

	g_free(title);
}

/*
 *	Returns the File Browser's current location.
 */
const gchar *EDVBrowserCurrentLocation(edv_browser_struct *browser)
{
	GtkCombo *combo;

	if(browser == NULL)
	    return(NULL);

	combo = GTK_COMBO(browser->location_combo);
	return(gtk_entry_get_text(GTK_ENTRY(combo->entry)));
}

/*
 *	Sets the File Browser's location to the specified path.
 *
 *	If record_history is TRUE then the new location will be
 *	recorded on the locations history.
 */
void EDVBrowserSetLocation(
	edv_browser_struct *browser,
	const gchar *path,
	const gboolean record_history
)
{
	const gchar *s;
	gchar *new_path;
	GtkEntry *entry;
	GtkCombo *combo;

	if((browser == NULL) || STRISEMPTY(path))
	    return;

	combo = GTK_COMBO(browser->location_combo);
	entry = GTK_ENTRY(combo->entry);

	/* Make a copy of the specified path */
	new_path = STRDUP(path);
	if(new_path == NULL)
	    return;

	/* Simplify the new path */
	EDVSimplifyPath(new_path);

	/* No change? */
	s = gtk_entry_get_text(entry);
	if(s != NULL)
	{
	    if(!strcmp((const char *)s, (const char *)new_path))
	    {
		g_free(new_path);
		return;
	    }
	}

	/* Record new location on the location history? */
	if(record_history)
	    GUIComboAddItem(GTK_WIDGET(combo), new_path);

	/* Set the new location */
	gtk_entry_set_text(entry, new_path);
	gtk_entry_set_position(entry, -1);

	g_free(new_path);
}

/*
 *	Updates the File Browser's location bar icon with the object
 *	specified by path.
 *
 *	If the given path is NULL then the icon will be set as unknown.
 */
void EDVBrowserUpdateLocationIcon(
	edv_browser_struct *browser,
	const gchar *path
)
{
	edv_object_struct *obj = NULL;
	GdkPixmap *pixmap, *pixmap_ext, *pixmap_hid;
	GdkBitmap *mask, *mask_ext, *mask_hid;
	GtkWidget *parent, *w;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	parent = browser->location_icon_fixed;
	core = browser->core;

	if(!STRISEMPTY(path))
	{
	    struct stat lstat_buf;
	    if(!lstat((const char *)path, &lstat_buf))
	    {
		obj = EDVObjectNew();
		EDVObjectSetPath(obj, path);
		EDVObjectSetStat(obj, &lstat_buf);
		EDVObjectUpdateLinkFlags(obj);
	    }
	}
	if(obj == NULL)
	{
	    obj = EDVObjectNew();
	    if(obj != NULL)
	    {
		obj->type = EDV_OBJECT_TYPE_DIRECTORY;
		obj->permissions = 0x00000000;
		if(!STRISEMPTY(path))
		{
		    struct stat lstat_buf;
		    EDVObjectSetPath(obj, path);
		    if(!lstat((const char *)path, &lstat_buf))
		    {
			EDVObjectSetStat(obj, &lstat_buf);
			EDVObjectUpdateLinkFlags(obj);
		    }
		}
	    }
	}
	if(obj == NULL)
	    return;

	/* Get pixmap and mask for the icon */
	EDVMatchObjectIcon(
	    core->device, core->total_devices,
	    core->mimetype, core->total_mimetypes,
	    obj->type,
	    obj->full_path,
	    EDV_OBJECT_IS_LINK_VALID(obj),
	    obj->permissions,
	    0,			/* Small icons */
	    &pixmap, &mask,
	    NULL, NULL,
	    &pixmap_ext, &mask_ext,
	    &pixmap_hid, &mask_hid
	);
	/* Hidden? */
	if(EDVIsObjectHidden(obj))
	{
	    if(pixmap_hid != NULL)
	    {   
		pixmap = pixmap_hid;
		mask = mask_hid;
	    }
	}
        /* No access? */
	if(!EDVIsObjectAccessable(core, obj))
        {
            if(pixmap_ext != NULL)
            {
                pixmap = pixmap_ext;
                mask = mask_ext;
            }
        }
	if(pixmap == NULL)
	    return;

	GDK_PIXMAP_REF(pixmap);
	GDK_BITMAP_REF(mask);

	EDVObjectDelete(obj);

	/* Create new icon pixmap widget as needed */
	w = browser->location_icon_pm;
	if(w == NULL)
	{
	    browser->location_icon_pm = w = gtk_pixmap_new(pixmap, mask);
	    gtk_fixed_put(GTK_FIXED(parent), w, 0, 0);
	    gtk_widget_show(w);
	}
	else
	{
	    gtk_pixmap_set(GTK_PIXMAP(w), pixmap, mask);
	}

	GDK_PIXMAP_UNREF(pixmap);
	GDK_BITMAP_UNREF(mask);
}


/*
 *	Gets a list of the selected objects.
 *
 *	If copy is TRUE then each object in the returned list will
 *	be a copy of the original object.
 *
 *	Returns a GList of edv_object_struct * objects. If copy
 *	was TRUE then the calling function must delete each object
 *	in the list and the list itself, otherwise the calling
 *	function need only delete the returned list.
 */
GList *EDVBrowserGetSelectedObjectsList(
	edv_browser_struct *browser,
	const gboolean copy
)
{
	GList *objs_list;

	if(browser == NULL)
	    return(NULL);

	objs_list = NULL;

	/* Contents GtkCLiist was last selected? */
	if(browser->contents_clist_selected_row > -1)
	{
	    gint row;
	    GList *glist;
	    GtkCList *clist = GTK_CLIST(browser->contents_clist);
	    edv_object_struct *obj;

	    for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		row = (gint)glist->data;
		obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
		if(obj == NULL)
		    continue;

		objs_list = g_list_append(
		    objs_list,
		    (copy) ? EDVObjectCopy(obj) : obj
		);
	    }    
	}
	/* Directory GtkCTree was last selected? */
	else if(browser->directory_ctree_selected_node != NULL)
	{
	    GList *glist;
	    GtkCTreeNode *node;
	    GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	    GtkCList *clist = GTK_CLIST(ctree);
	    edv_object_struct *obj;

	    for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		node = (GtkCTreeNode *)glist->data;
		obj = EDV_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
		if(obj == NULL)
		    continue;

		objs_list = g_list_append(
		    objs_list,
		    (copy) ? EDVObjectCopy(obj) : obj
		);
	    }
	}

	return(objs_list);
}

/*
 *	Gets a list of selected object paths.
 *
 *	Returns a GList of gchar * paths. The calling function must
 *	delete each path and the list.
 */
GList *EDVBrowserGetSelectedPathsList(edv_browser_struct *browser)
{
	GList *paths_list;

	if(browser == NULL)
	    return(NULL);

	paths_list = NULL;

	/* Contents GtkCLiist was last selected? */
	if(browser->contents_clist_selected_row > -1)
	{
	    gint row;
	    GList *glist;
	    GtkCList *clist = GTK_CLIST(browser->contents_clist);
	    edv_object_struct *obj;

	    for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		row = (gint)glist->data;
		obj = EDV_OBJECT(gtk_clist_get_row_data(clist, row));
		if(obj == NULL)
		    continue;

		if(obj->full_path == NULL)
		    continue;

		paths_list = g_list_append(
		    paths_list,
		    STRDUP(obj->full_path)
		);
	    }    
	}
	/* Directory GtkCTree was last selected? */
	else if(browser->directory_ctree_selected_node != NULL)
	{
	    GList *glist;
	    GtkCTreeNode *node;
	    GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);
	    GtkCList *clist = GTK_CLIST(ctree);
	    edv_object_struct *obj;

	    for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		node = (GtkCTreeNode *)glist->data;
		obj = EDV_OBJECT(gtk_ctree_node_get_row_data(ctree, node));
		if(obj == NULL)
		    continue;

		if(obj->full_path == NULL)
		    continue;

		paths_list = g_list_append(
		    paths_list,
		    STRDUP(obj->full_path)
		);
	    }
	}

	return(paths_list);
}


/*
 *	Resets the File Browser's accelkeys displayed on the widgets.
 */
void EDVBrowserAccelkeysRegenerate(edv_browser_struct *browser)
{
	GtkAccelGroup *accelgrp = NULL;
	cfg_accelkey_list_struct *ak_list;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	core = browser->core;
	if(core == NULL)
	    return;

	cfg_list = core->cfg_list;

	ak_list = EDV_GET_ACCELKEY_LIST(EDV_CFG_PARM_BROWSER_ACCELERATOR_KEYS);
	if(ak_list == NULL)
	    return;

#define UPDATE_ACCELKEY(_w_,_opid_)	{	\
 if(((_w_) != NULL) && ((_opid_) > 0)) {	\
  GList *glist;					\
  cfg_accelkey_struct *ak;			\
						\
  for(glist = ak_list->list;			\
      glist != NULL;				\
      glist = g_list_next(glist)		\
  ) {						\
   ak = CFG_ACCELKEY(glist->data);		\
   if(ak == NULL)				\
    continue;					\
						\
   if(ak->opid == (gint)(_opid_)) {		\
    GUIMenuItemSetAccelKey(			\
     (_w_), accelgrp, ak->key, ak->modifiers	\
    );						\
    break;					\
   }						\
  }						\
 }						\
}

	UPDATE_ACCELKEY(
	    browser->open_mi,
	    EDV_BROWSER_OP_OPEN
	);
	UPDATE_ACCELKEY(
	    browser->open_to_mi,
	    EDV_BROWSER_OP_OPEN_TO
	);
	UPDATE_ACCELKEY(
	    browser->open_with_mi,
	    EDV_BROWSER_OP_OPEN_WITH
	);
	UPDATE_ACCELKEY(
	    browser->new_file_mi,
	    EDV_BROWSER_OP_NEW_FILE
	);
	UPDATE_ACCELKEY(
	    browser->new_directory_mi,
	    EDV_BROWSER_OP_NEW_DIRECTORY
	);
	UPDATE_ACCELKEY(
	    browser->new_link_mi,
	    EDV_BROWSER_OP_NEW_LINK
	);
	UPDATE_ACCELKEY(
	    browser->new_fifo_mi,
	    EDV_BROWSER_OP_NEW_FIFO
	);
	UPDATE_ACCELKEY(
	    browser->new_device_block_mi,
	    EDV_BROWSER_OP_NEW_DEVICE_BLOCK
	);
	UPDATE_ACCELKEY(
	    browser->new_device_character_mi,
	    EDV_BROWSER_OP_NEW_DEVICE_CHARACTER
	);
	UPDATE_ACCELKEY(
	    browser->new_socket_mi,
	    EDV_BROWSER_OP_NEW_SOCKET
	);
	UPDATE_ACCELKEY(
	    browser->sync_disks_mi,
	    EDV_BROWSER_OP_SYNC_DISKS
	);
	UPDATE_ACCELKEY(
	    browser->history_mi,
	    EDV_BROWSER_OP_HISTORY
	);
	UPDATE_ACCELKEY(
	    browser->run_mi,
	    EDV_BROWSER_OP_RUN
	);
	UPDATE_ACCELKEY(
	    browser->close_mi,
	    EDV_BROWSER_OP_CLOSE
	);

	UPDATE_ACCELKEY(
	    browser->move_mi,
	    EDV_BROWSER_OP_MOVE
	);
	UPDATE_ACCELKEY(
	    browser->copy_mi,
	    EDV_BROWSER_OP_COPY
	);
	UPDATE_ACCELKEY(
	    browser->link_mi,
	    EDV_BROWSER_OP_LINK
	);
	UPDATE_ACCELKEY(
	    browser->rename_mi,
	    EDV_BROWSER_OP_RENAME
	);
	UPDATE_ACCELKEY(
	    browser->chmod_mi,
	    EDV_BROWSER_OP_CHMOD
	);
	UPDATE_ACCELKEY(
	    browser->chown_mi,
	    EDV_BROWSER_OP_CHOWN
	);
	UPDATE_ACCELKEY(
	    browser->chtime_mi,
	    EDV_BROWSER_OP_CHTIME
	);
	UPDATE_ACCELKEY(
	    browser->delete_mi,
	    EDV_BROWSER_OP_DELETE
	);
	UPDATE_ACCELKEY(
	    browser->copy_path_mi,
	    EDV_BROWSER_OP_COPY_PATH
	);
	UPDATE_ACCELKEY(
	    browser->copy_url_mi,
	    EDV_BROWSER_OP_COPY_URL
	);
	UPDATE_ACCELKEY(
	    browser->paste_mi,
	    EDV_BROWSER_OP_PASTE
	);
	UPDATE_ACCELKEY(
	    browser->select_all_mi,
	    EDV_BROWSER_OP_SELECT_ALL
	);
	UPDATE_ACCELKEY(
	    browser->unselect_all_mi,
	    EDV_BROWSER_OP_UNSELECT_ALL
	);
	UPDATE_ACCELKEY(
	    browser->invert_selection_mi,
	    EDV_BROWSER_OP_INVERT_SELECTION
	);
	UPDATE_ACCELKEY(
	    browser->find_mi,
	    EDV_BROWSER_OP_FIND
	);
	UPDATE_ACCELKEY(
	    browser->properties_mi,
	    EDV_BROWSER_OP_PROPERTIES
	);

	UPDATE_ACCELKEY(
	    browser->mount_mi,
	    EDV_BROWSER_OP_MOUNT
	);
	UPDATE_ACCELKEY(
	    browser->eject_mi,
	    EDV_BROWSER_OP_EJECT
	);
	UPDATE_ACCELKEY(
	    browser->device_check_mi,
	    EDV_BROWSER_OP_DEVICE_CHECK
	);
	UPDATE_ACCELKEY(
	    browser->device_tools_mi,
	    EDV_BROWSER_OP_DEVICE_TOOLS
	);
	UPDATE_ACCELKEY(
	    browser->device_format_mi,
	    EDV_BROWSER_OP_DEVICE_FORMAT
	);
	UPDATE_ACCELKEY(
	    browser->devices_mi,
	    EDV_BROWSER_OP_DEVICES
	);

	UPDATE_ACCELKEY(
	    browser->refresh_mi,
	    EDV_BROWSER_OP_REFRESH
	);
	UPDATE_ACCELKEY(
	    browser->refresh_all_mi,
	    EDV_BROWSER_OP_REFRESH_ALL
	);
	UPDATE_ACCELKEY(
	    browser->show_tool_bar_micheck,
	    EDV_BROWSER_OP_SHOW_TOOL_BAR
	);
	UPDATE_ACCELKEY(
	    browser->show_location_bar_micheck,
	    EDV_BROWSER_OP_SHOW_LOCATION_BAR
	);
	UPDATE_ACCELKEY(
	    browser->show_mount_bar_micheck,
	    EDV_BROWSER_OP_SHOW_MOUNT_BAR
	);
	UPDATE_ACCELKEY(
	    browser->show_find_bar_micheck,
	    EDV_BROWSER_OP_SHOW_FIND_BAR
	);
	UPDATE_ACCELKEY(
	    browser->show_status_bar_micheck,
	    EDV_BROWSER_OP_SHOW_STATUS_BAR
	);
	UPDATE_ACCELKEY(
	    browser->directory_tree_origin_mi,
	    EDV_BROWSER_OP_DIRECTORY_TREE_ORIGIN
	);
	UPDATE_ACCELKEY(
	    browser->show_hidden_objects_micheck,
	    EDV_BROWSER_OP_SHOW_HIDDEN_OBJECTS
	);
	UPDATE_ACCELKEY(
	    browser->show_noaccess_objects_micheck,
	    EDV_BROWSER_OP_SHOW_NOACCESS_OBJECTS
	);
	UPDATE_ACCELKEY(
	    browser->contents_list_filter_mi,
	    EDV_BROWSER_OP_CONTENTS_LIST_FILTER
	);
	UPDATE_ACCELKEY(
	    browser->contents_list_auto_resize_columns_micheck,
	    EDV_BROWSER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS
	);
	UPDATE_ACCELKEY(
	    browser->mime_types_mi,
	    EDV_BROWSER_OP_MIME_TYPES
	);

	UPDATE_ACCELKEY(
	    browser->write_protect_mi,
	    EDV_BROWSER_OP_WRITE_PROTECT
	);
	UPDATE_ACCELKEY(
	    browser->delete_method_recycle_micheck,
	    EDV_BROWSER_OP_DELETE_METHOD_RECYCLE
	);
	UPDATE_ACCELKEY(
	    browser->delete_method_purge_micheck,
	    EDV_BROWSER_OP_DELETE_METHOD_PURGE
	);
	UPDATE_ACCELKEY(
	    browser->options_mi,
	    EDV_BROWSER_OP_OPTIONS
	);
	UPDATE_ACCELKEY(
	    browser->customize_mi,
	    EDV_BROWSER_OP_CUSTOMIZE
	);

	UPDATE_ACCELKEY(
	    browser->window_new_browser_mi,
	    EDV_BROWSER_OP_NEW_BROWSER
	);
	UPDATE_ACCELKEY(
	    browser->window_new_imbr_mi,
	    EDV_BROWSER_OP_NEW_IMBR
	);
	UPDATE_ACCELKEY(
	    browser->window_new_archiver_mi,
	    EDV_BROWSER_OP_NEW_ARCHIVER
	);
	UPDATE_ACCELKEY(
	    browser->window_recycle_bin_mi,
	    EDV_BROWSER_OP_RECYCLE_BIN
	);

	UPDATE_ACCELKEY(
	    browser->help_contents_mi,
	    EDV_BROWSER_OP_HELP_CONTENTS
	);
	UPDATE_ACCELKEY(
	    browser->help_file_browser_mi,
	    EDV_BROWSER_OP_HELP_FILE_BROWSER
	);
	UPDATE_ACCELKEY(
	    browser->help_keys_list_mi,
	    EDV_BROWSER_OP_HELP_KEYS_LIST
	);
	UPDATE_ACCELKEY(
	    browser->help_mime_types_mi,
	    EDV_BROWSER_OP_HELP_MIME_TYPES
	);
	UPDATE_ACCELKEY(
	    browser->help_devices_mi,
	    EDV_BROWSER_OP_HELP_DEVICES
	);
	UPDATE_ACCELKEY(
	    browser->help_common_operations_mi,
	    EDV_BROWSER_OP_HELP_COMMON_OPERATIONS
	);
	UPDATE_ACCELKEY(
	    browser->help_about_mi,
	    EDV_BROWSER_OP_HELP_ABOUT
	);


	UPDATE_ACCELKEY(
	    browser->directory_ctree_expand_mi,
	    EDV_BROWSER_OP_EXPAND
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_mount_mi,
	    EDV_BROWSER_OP_MOUNT
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_eject_mi,
	    EDV_BROWSER_OP_EJECT
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_move_mi,
	    EDV_BROWSER_OP_MOVE
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_copy_mi,
	    EDV_BROWSER_OP_COPY
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_link_mi,
	    EDV_BROWSER_OP_LINK
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_rename_mi,
	    EDV_BROWSER_OP_RENAME
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_chmod_mi,
	    EDV_BROWSER_OP_CHMOD
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_chown_mi,
	    EDV_BROWSER_OP_CHOWN
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_chtime_mi,
	    EDV_BROWSER_OP_CHTIME
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_delete_mi,
	    EDV_BROWSER_OP_DELETE
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_copy_path_mi,
	    EDV_BROWSER_OP_COPY_PATH
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_copy_url_mi,
	    EDV_BROWSER_OP_COPY_URL
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_paste_mi,
	    EDV_BROWSER_OP_PASTE
	);
	UPDATE_ACCELKEY(
	    browser->directory_ctree_properties_mi,
	    EDV_BROWSER_OP_PROPERTIES
	);


	UPDATE_ACCELKEY(
	    browser->contents_clist_open_mi,
	    EDV_BROWSER_OP_OPEN
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_open_to_mi,
	    EDV_BROWSER_OP_OPEN_TO
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_open_with_mi,
	    EDV_BROWSER_OP_OPEN_WITH
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_move_mi,
	    EDV_BROWSER_OP_MOVE
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_copy_mi,
	    EDV_BROWSER_OP_COPY
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_link_mi,
	    EDV_BROWSER_OP_LINK
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_rename_mi,
	    EDV_BROWSER_OP_RENAME
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_chmod_mi,
	    EDV_BROWSER_OP_CHMOD
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_chown_mi,
	    EDV_BROWSER_OP_CHOWN
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_chtime_mi,
	    EDV_BROWSER_OP_CHTIME
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_delete_mi,
	    EDV_BROWSER_OP_DELETE
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_copy_path_mi,
	    EDV_BROWSER_OP_COPY_PATH
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_copy_url_mi,
	    EDV_BROWSER_OP_COPY_URL
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_paste_mi,
	    EDV_BROWSER_OP_PASTE
	);
	UPDATE_ACCELKEY(
	    browser->contents_clist_properties_mi,
	    EDV_BROWSER_OP_PROPERTIES
	);


#undef UPDATE_ACCELKEY
}

/*
 *	Recreates the File Browser's list styles.
 */
void EDVBrowserListStylesRegenerate(edv_browser_struct *browser)
{
	gint i;
	edv_browser_cell_style n;
	GdkColormap *colormap;
	GtkStyle *style, *src_style;
	GtkWidget *w;

	if(browser == NULL)
	    return;

	w = browser->contents_clist;
	if(w == NULL)
	    return;

	/* Unref existing cell styles */
	for(i = 0; i < EDV_BROWSER_TOTAL_CELL_STYLES; i++)
	{
	    GTK_STYLE_UNREF(browser->cell_style[i]);
	    browser->cell_style[i] = NULL;
	}

	/* Get source style from contents clist as a template */
	colormap = gtk_widget_get_colormap(w);
	src_style = gtk_widget_get_style(w);
	if(src_style == NULL)
	    return;


	/* Begin creating cell styles */

#define REALLOC_COLOR(_ct_,_cs_)		{		\
 GDK_COLORMAP_FREE_COLOR(colormap, (_ct_));			\
 memcpy((_ct_), (_cs_), sizeof(GdkColor));			\
 GDK_COLORMAP_ALLOC_COLOR(colormap, (_ct_));			\
}

	/* No Access */
	n = EDV_BROWSER_CELL_STYLE_NO_ACCESS;
	browser->cell_style[n] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
	    const GdkColor	cs = EDV_GDKCOLOR_NO_ACCESS_FG,
				cs_sel = EDV_GDKCOLOR_NO_ACCESS_FG_SELECTED,
				cs_insen = EDV_GDKCOLOR_NO_ACCESS_FG_INSENSITIVE;

	    GtkStateType state = GTK_STATE_NORMAL;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_ACTIVE;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_PRELIGHT;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_SELECTED;
	    REALLOC_COLOR(&style->fg[state], &cs_sel)

	    state = GTK_STATE_INSENSITIVE;
	    REALLOC_COLOR(&style->fg[state], &cs_insen)
	}

	/* Dangling Link */
	n = EDV_BROWSER_CELL_STYLE_DANGLING_LINK;
	browser->cell_style[n] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
	    const GdkColor	cs = EDV_GDKCOLOR_DANGLING_LINK_FG,
				cs_sel = EDV_GDKCOLOR_DANGLING_LINK_FG_SELECTED,
				cs_insen = EDV_GDKCOLOR_DANGLING_LINK_FG_INSENSITIVE;

	    GtkStateType state = GTK_STATE_NORMAL;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_ACTIVE;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_PRELIGHT;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_SELECTED;
	    REALLOC_COLOR(&style->fg[state], &cs_sel)

	    state = GTK_STATE_INSENSITIVE;
	    REALLOC_COLOR(&style->fg[state], &cs_insen)
	}

	/* Infinately Recursive Link */
	n = EDV_BROWSER_CELL_STYLE_RECURSIVE_LINK;
	browser->cell_style[n] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
	    const GdkColor      cs = EDV_GDKCOLOR_RECURSIVE_LINK_FG,
				cs_sel = EDV_GDKCOLOR_RECURSIVE_LINK_FG_SELECTED,
				cs_insen = EDV_GDKCOLOR_RECURSIVE_LINK_FG_INSENSITIVE;

	    GtkStateType state = GTK_STATE_NORMAL;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_ACTIVE;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_PRELIGHT; 
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_SELECTED;
	    REALLOC_COLOR(&style->fg[state], &cs_sel)

	    state = GTK_STATE_INSENSITIVE;
	    REALLOC_COLOR(&style->fg[state], &cs_insen)
	}

	/* Hidden Object */
	n = EDV_BROWSER_CELL_STYLE_HIDDEN;
	browser->cell_style[n] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
	    const GdkColor	cs = EDV_GDKCOLOR_HIDDEN_OBJECT_FG,
				cs_sel = EDV_GDKCOLOR_HIDDEN_OBJECT_FG_SELECTED,
				cs_insen = EDV_GDKCOLOR_HIDDEN_OBJECT_FG_INSENSITIVE;

	    GtkStateType state = GTK_STATE_NORMAL;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_ACTIVE;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_PRELIGHT;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_SELECTED;
	    REALLOC_COLOR(&style->fg[state], &cs_sel)

	    state = GTK_STATE_INSENSITIVE;
	    REALLOC_COLOR(&style->fg[state], &cs_insen)
	}

	/* Executable */
	n = EDV_BROWSER_CELL_STYLE_EXECUTABLE;
	browser->cell_style[n] = style = gtk_style_copy(src_style);
	if(style != NULL)
	{
	    const GdkColor	cs = EDV_GDKCOLOR_EXECUTABLE_FG,
				cs_sel = EDV_GDKCOLOR_EXECUTABLE_FG_SELECTED,
				cs_insen = EDV_GDKCOLOR_EXECUTABLE_FG_INSENSITIVE;

	    GtkStateType state = GTK_STATE_NORMAL;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_ACTIVE;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_PRELIGHT;
	    REALLOC_COLOR(&style->fg[state], &cs)

	    state = GTK_STATE_SELECTED;
	    REALLOC_COLOR(&style->fg[state], &cs_sel)

	    state = GTK_STATE_INSENSITIVE;
	    REALLOC_COLOR(&style->fg[state], &cs_insen)
	}

#undef REALLOC_COLOR
}

/*
 *	Recreates the File Browser's Tool Bar.
 */
void EDVBrowserToolBarRegenerate(edv_browser_struct *browser)
{
	GList *glist, *items_list;
	GtkWidget *parent;
	toolbar_display tb_display;
	toolbar_relief tb_relief;
	toolbar_item_struct *item;
	const cfg_intlist_struct *intlist;
	const cfg_item_struct *cfg_list;
	edv_browser_opid_struct *opid;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	parent = browser->tool_bar_handle;
	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get the Tool Bar configuration */
	tb_display = (toolbar_display)EDV_GET_I(
	    EDV_CFG_PARM_BROWSER_TOOL_BAR_DISPLAY
	);
	tb_relief = (toolbar_relief)EDV_GET_I(
	    EDV_CFG_PARM_BROWSER_TOOL_BAR_RELIEF
	);
	intlist = EDV_GET_INTLIST(EDV_CFG_PARM_BROWSER_TOOL_BAR);
	if(intlist == NULL)
	    return;

	/* Delete the existing Tool Bar */
	ToolBarDelete(browser->tool_bar);
	browser->tool_bar = NULL;

	/* Create the Tool Bar items list */
	items_list = NULL;
	for(glist = intlist->list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    /* Get the Operation ID */ 
	    opid = EDVBrowserMatchOPID(
		browser,
		(edv_browser_op)glist->data
	    );
	    if(opid == NULL)
		continue;

	    switch(opid->op)
	    {
	      case EDV_BROWSER_OP_NONE:
		break;

	      default:
		item = ToolBarItemNew(
		    opid->tb_item_type,
		    NULL,
		    opid->button_name,
		    opid->button_icon_data,
		    opid->tooltip,
		    opid->op,
		    EDVBrowserOPCB, opid,
		    EDVBrowserOPEnterCB, opid,
		    EDVBrowserOPLeaveCB, opid
		);
		items_list = g_list_append(items_list, item);
		break;
	    }
	}

	/* Create the Tool Bar */
	browser->tool_bar = ToolBarNew(
	    parent,
	    items_list,
	    tb_display, tb_relief, FALSE
	);
	ToolBarMap(browser->tool_bar);

	/* Delete the Tool Bar items list */
	if(items_list != NULL)
	{
	    g_list_foreach(items_list, (GFunc)ToolBarItemDelete, NULL);
	    g_list_free(items_list);
	}
}

/*
 *	Recreates the File Browser's File menu items.
 */
void EDVBrowserFileMenuItemsRegenerate(edv_browser_struct *browser)
{
	guint8 **icon_data = NULL;
	GList *glist;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu, *submenu;
	GtkMenuShell *menushell;
	const cfg_item_struct *cfg_list;
	cfg_menu_struct *cfg_menu;
	edv_browser_op op;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get the File menu */
	menu = browser->file_menu;
	if(menu == NULL)
	    return;

	menushell = GTK_MENU_SHELL(menu);

	/* Delete the existing menu items */
	glist = g_list_copy(menushell->children);
	if(glist != NULL)
	{
	    g_list_foreach(glist, (GFunc)gtk_widget_destroy, NULL);
	    g_list_free(glist);
	}

#define DO_ADD_MENU_ITEM_LABEL	{		\
 edv_browser_opid_struct *opid =		\
  EDVBrowserMatchOPID(browser, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   NULL,					\
   opid, EDVBrowserMenuItemCB			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemEnterCB, opid,	\
   (gpointer)EDVBrowserMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_ITEM_CMD(_item_) {		\
 if((_item_) != NULL) {				\
  edv_browser_cmd_struct *d;			\
  const gchar *icon_file = (_item_)->icon_file;	\
  GdkBitmap *mask;				\
  GdkPixmap *pixmap;				\
  if(icon_file != NULL) {  			\
   pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(	\
    &mask, icon_file				\
   );						\
  } else {					\
   pixmap = NULL;				\
   mask = NULL;					\
  }						\
						\
  d = EDV_BROWSER_CMD(g_malloc0(		\
   sizeof(edv_browser_cmd_struct)		\
  ));						\
  d->browser = browser;				\
  d->label = STRDUP((_item_)->label);		\
  d->command = STRDUP((_item_)->command);	\
  d->description = STRDUP((_item_)->description);\
  d->ext_data = STRDUP((_item_)->ext_data);	\
						\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   NULL,					\
   (_item_)->label,				\
   0, 0,					\
   NULL,					\
   d, EDVBrowserMenuItemCmdObjectOpCB		\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemCmdEnterCB, d,	\
   (gpointer)EDVBrowserMenuItemCmdLeaveCB, d	\
  );						\
  if(pixmap != NULL)				\
   GUIMenuItemSetPixmap2(			\
    w,						\
    pixmap, mask				\
   );						\
  gtk_object_set_data_full(			\
   GTK_OBJECT(w), EDV_BROWSER_CMD_DATA_KEY,	\
   d, EDVBrowserCmdDestroyCB			\
  );						\
						\
  GDK_PIXMAP_UNREF(pixmap);			\
  GDK_BITMAP_UNREF(mask);			\
						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_SUBMENU_LABEL	{	\
 edv_browser_opid_struct *opid =		\
  EDVBrowserMatchOPID(browser, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, NULL,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   0, 0, NULL, NULL, NULL			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemEnterCB, opid,	\
   (gpointer)EDVBrowserMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_SEP	{			\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}
	op = EDV_BROWSER_OP_OPEN;
	DO_ADD_MENU_ITEM_LABEL
	browser->open_mi = w;

	op = EDV_BROWSER_OP_OPEN_TO;
	DO_ADD_MENU_SUBMENU_LABEL
	browser->open_to_mi = w;
	browser->open_to_menu = NULL;

	op = EDV_BROWSER_OP_OPEN_WITH;
	DO_ADD_MENU_ITEM_LABEL
	browser->open_with_mi = w;

	DO_ADD_MENU_SEP

	/* Customized Object Ops menu items */
	cfg_menu = EDV_GET_MENU(EDV_CFG_PARM_OBJECT_OPS_MENU);
	if((cfg_menu != NULL) ? (cfg_menu->list != NULL) : FALSE)
	{
	    GList *glist;
	    cfg_menu_item_struct *item;

	    for(glist = cfg_menu->list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		item = CFG_MENU_ITEM(glist->data);
		DO_ADD_MENU_ITEM_CMD(item);
	    }

	    DO_ADD_MENU_SEP
	}

	op = EDV_BROWSER_OP_NEW;
	DO_ADD_MENU_SUBMENU_LABEL
	browser->new_object_mi = w;
	browser->new_object_menu = submenu = GUIMenuCreate();
	GUIMenuItemSetSubMenu(w, submenu);

	DO_ADD_MENU_SEP

	op = EDV_BROWSER_OP_SYNC_DISKS;
	DO_ADD_MENU_ITEM_LABEL
	browser->sync_disks_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_BROWSER_OP_HISTORY;
	DO_ADD_MENU_ITEM_LABEL
	browser->history_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_BROWSER_OP_RUN;
	DO_ADD_MENU_ITEM_LABEL
	browser->run_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_BROWSER_OP_CLOSE;
	DO_ADD_MENU_ITEM_LABEL
	browser->close_mi = w;

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_ITEM_CMD
#undef DO_ADD_MENU_SUBMENU_LABEL
#undef DO_ADD_MENU_SEP
}

/*
 *	Recreates the File Browser's New Object Submenu.
 */
void EDVBrowserNewObjectMenuItemsRegenerate(edv_browser_struct *browser)
{
	guint8 **icon_data = NULL;
	GList *glist;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu;
	GtkMenuShell *menushell;
	const cfg_item_struct *cfg_list;
	cfg_menu_struct *cfg_menu;
	edv_browser_op op;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	core = browser->core;
	cfg_list = core->cfg_list;

	/* Get the New Object submenu */
	menu = browser->new_object_menu;
	if(menu == NULL)
	    return;

	menushell = GTK_MENU_SHELL(menu);

	/* Delete the existing menu items */
	glist = g_list_copy(menushell->children);
	if(glist != NULL)
	{
	    g_list_foreach(glist, (GFunc)gtk_widget_destroy, NULL);
	    g_list_free(glist);
	}

#define DO_ADD_MENU_ITEM_LABEL	{		\
 edv_browser_opid_struct *opid =		\
  EDVBrowserMatchOPID(browser, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   NULL,					\
   opid, EDVBrowserMenuItemCB			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemEnterCB, opid,	\
   (gpointer)EDVBrowserMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_ITEM_CMD(_item_) {		\
 if((_item_) != NULL) {				\
  edv_browser_cmd_struct *d;			\
  const gchar *icon_file = (_item_)->icon_file;	\
  GdkBitmap *mask;				\
  GdkPixmap *pixmap;				\
  if(icon_file != NULL) {  			\
   pixmap = GDK_PIXMAP_NEW_FROM_XPM_FILE(	\
    &mask, icon_file				\
   );						\
  } else {					\
   pixmap = NULL;				\
   mask = NULL;					\
  }						\
						\
  d = EDV_BROWSER_CMD(g_malloc0(		\
   sizeof(edv_browser_cmd_struct)		\
  ));						\
  d->browser = browser;				\
  d->label = STRDUP((_item_)->label);		\
  d->command = STRDUP((_item_)->command);	\
  d->description = STRDUP((_item_)->description);\
  d->ext_data = STRDUP((_item_)->ext_data);	\
						\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   NULL,					\
   (_item_)->label,				\
   0, 0,					\
   NULL,					\
   d, EDVBrowserMenuItemCmdNewObjectCB		\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemCmdEnterCB, d,	\
   (gpointer)EDVBrowserMenuItemCmdLeaveCB, d	\
  );						\
  if(pixmap != NULL)				\
   GUIMenuItemSetPixmap2(			\
    w,						\
    pixmap, mask				\
   );						\
  gtk_object_set_data_full(			\
   GTK_OBJECT(w), EDV_BROWSER_CMD_DATA_KEY,	\
   d, EDVBrowserCmdDestroyCB			\
  );						\
						\
  GDK_PIXMAP_UNREF(pixmap);			\
  GDK_BITMAP_UNREF(mask);			\
						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_SEP	{			\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}

	op = EDV_BROWSER_OP_NEW_FILE;
	DO_ADD_MENU_ITEM_LABEL
	browser->new_file_mi = w;

	op = EDV_BROWSER_OP_NEW_DIRECTORY;
	DO_ADD_MENU_ITEM_LABEL
	browser->new_directory_mi = w;

	op = EDV_BROWSER_OP_NEW_LINK;
	DO_ADD_MENU_ITEM_LABEL
	browser->new_link_mi = w;

	/* Customized New Object menu items */
	cfg_menu = EDV_GET_MENU(EDV_CFG_PARM_NEW_OBJECTS_MENU);
	if((cfg_menu != NULL) ? (cfg_menu->list != NULL) : FALSE)
	{
	    GList *glist;
	    cfg_menu_item_struct *item;

	    DO_ADD_MENU_SEP

	    for(glist = cfg_menu->list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		item = CFG_MENU_ITEM(glist->data);
		DO_ADD_MENU_ITEM_CMD(item);
	    }

	    DO_ADD_MENU_SEP
	}

	op = EDV_BROWSER_OP_NEW_FIFO;
	DO_ADD_MENU_ITEM_LABEL
	browser->new_fifo_mi = w;

	op = EDV_BROWSER_OP_NEW_DEVICE_BLOCK;
	DO_ADD_MENU_ITEM_LABEL
	browser->new_device_block_mi = w;

	op = EDV_BROWSER_OP_NEW_DEVICE_CHARACTER;
	DO_ADD_MENU_ITEM_LABEL
	browser->new_device_character_mi = w;

	op = EDV_BROWSER_OP_NEW_SOCKET;
	DO_ADD_MENU_ITEM_LABEL
	browser->new_socket_mi = w;

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_ITEM_CMD
#undef DO_ADD_MENU_SEP         
}

/* 
 *	Regenerates the Open To menu.
 */
void EDVBrowserOpenToMenuRegenerate(
	edv_browser_struct *browser,
	edv_mime_type_struct *m
)
{
	GtkWidget *submenu, *toplevel;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	toplevel = browser->toplevel;
	core = browser->core;

	submenu = EDVOpenToMenuNewFromMIMEType(
	    core, m,
	    EDVBrowserGetSelectedPathsListCB, browser,
	    EDVBrowserGotoDirectoryCB, browser,
	    toplevel, TRUE
	);
	GUIMenuItemSetSubMenu(browser->open_to_mi, submenu);
	browser->open_to_menu = submenu;

	submenu = EDVOpenToMenuNewFromMIMEType(
	    core, m,
	    EDVBrowserGetSelectedPathsListCB, browser,
	    EDVBrowserGotoDirectoryCB, browser,
	    toplevel, TRUE
	);
	GUIMenuItemSetSubMenu(browser->contents_clist_open_to_mi, submenu);
	browser->contents_clist_open_to_menu = submenu;
}

/*
 *	Recreates the File Browser Directory Tree's right-click menu.
 */
void EDVBrowserDirectoryMenuRegenerate(edv_browser_struct *browser)
{
	guint8 **icon_data = NULL;
	edv_browser_op op;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu;

	if(browser == NULL)
	    return;

	/* Create/recreate the menu */
	GTK_WIDGET_DESTROY(browser->directory_ctree_menu);
	browser->directory_ctree_menu = menu = GUIMenuCreate();

#define DO_ADD_MENU_ITEM_LABEL		{	\
 edv_browser_opid_struct *opid =		\
  EDVBrowserMatchOPID(browser, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   NULL,					\
   opid, EDVBrowserMenuItemCB			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemEnterCB, opid,	\
   (gpointer)EDVBrowserMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}
	op = EDV_BROWSER_OP_EXPAND;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_expand_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_BROWSER_OP_MOUNT;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_mount_mi = w;

	op = EDV_BROWSER_OP_EJECT;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_eject_mi = w;

	DO_ADD_MENU_SEP
	browser->directory_ctree_mount_misep = w;

	op = EDV_BROWSER_OP_MOVE;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_move_mi = w;

	op = EDV_BROWSER_OP_COPY;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_copy_mi = w;

	op = EDV_BROWSER_OP_LINK;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_link_mi = w;

	op = EDV_BROWSER_OP_RENAME;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_rename_mi = w;

	op = EDV_BROWSER_OP_CHMOD;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_chmod_mi = w;

	op = EDV_BROWSER_OP_CHOWN;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_chown_mi = w;

	op = EDV_BROWSER_OP_CHTIME;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_chtime_mi = w;

	op = EDV_BROWSER_OP_DELETE;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_delete_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_BROWSER_OP_COPY_PATH;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_copy_path_mi = w;

	op = EDV_BROWSER_OP_COPY_URL;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_copy_url_mi = w;

	op = EDV_BROWSER_OP_PASTE;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_paste_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_BROWSER_OP_PROPERTIES;
	DO_ADD_MENU_ITEM_LABEL
	browser->directory_ctree_properties_mi = w;

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP
}

/*
 *	Recreates the File Browser Contents List's right-click menu.
 */
void EDVBrowserContentsMenuRegenerate(edv_browser_struct *browser)
{
	guint8 **icon_data = NULL;
	edv_browser_op op;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu;

	if(browser == NULL)
	    return;

	/* Create/recreate the menu */
	GTK_WIDGET_DESTROY(browser->contents_clist_menu);
	browser->contents_clist_menu = menu = GUIMenuCreate();

#define DO_ADD_MENU_ITEM_LABEL	{		\
 edv_browser_opid_struct *opid =		\
  EDVBrowserMatchOPID(browser, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   NULL,					\
   opid, EDVBrowserMenuItemCB			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemEnterCB, opid,	\
   (gpointer)EDVBrowserMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_SUBMENU_LABEL	{	\
 edv_browser_opid_struct *opid =		\
  EDVBrowserMatchOPID(browser, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, NULL,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   0, 0, NULL, NULL, NULL			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemEnterCB, opid,	\
   (gpointer)EDVBrowserMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_SEP	{			\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}

	op = EDV_BROWSER_OP_OPEN;
	DO_ADD_MENU_ITEM_LABEL
	GUISetMenuItemDefault(w);
	browser->contents_clist_open_mi = w;

	op = EDV_BROWSER_OP_OPEN_TO;
	DO_ADD_MENU_SUBMENU_LABEL
	browser->contents_clist_open_to_mi = w;

	op = EDV_BROWSER_OP_OPEN_WITH;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_open_with_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_BROWSER_OP_MOVE;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_move_mi = w;

	op = EDV_BROWSER_OP_COPY;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_copy_mi = w;

	op = EDV_BROWSER_OP_LINK;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_link_mi = w;

	op = EDV_BROWSER_OP_RENAME;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_rename_mi = w;

	op = EDV_BROWSER_OP_CHMOD;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_chmod_mi = w;

	op = EDV_BROWSER_OP_CHOWN;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_chown_mi = w;

	op = EDV_BROWSER_OP_CHTIME;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_chtime_mi = w;

	op = EDV_BROWSER_OP_DELETE;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_delete_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_BROWSER_OP_COPY_PATH;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_copy_path_mi = w;

	op = EDV_BROWSER_OP_COPY_URL;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_copy_url_mi = w;

	op = EDV_BROWSER_OP_PASTE;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_paste_mi = w;

	DO_ADD_MENU_SEP

	op = EDV_BROWSER_OP_PROPERTIES;
	DO_ADD_MENU_ITEM_LABEL
	browser->contents_clist_properties_mi = w;

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SUBMENU_LABEL
#undef DO_ADD_MENU_SEP
}


/*
 *	Sets the File Browser's values to the configuration.
 */
void EDVBrowserSyncConfiguration(edv_browser_struct *browser)
{
	GtkWidget *w;
	cfg_item_struct *cfg_list;
	edv_core_struct *core = (browser != NULL) ?
	    browser->core : NULL;
	if(core == NULL)
	    return;

	cfg_list = core->cfg_list;

	/* Location history */
	w = browser->location_combo;
	if(w != NULL)
	    EDVSaveFileGList(
		EDV_GET_S(EDV_CFG_PARM_FILE_BROWSER_LOCATION_HISTORY),
		GUIComboGetList(w)
	    );

	/* Toplevel geometry */
	w = browser->toplevel;
	if(w != NULL)
	{
	    gint x = 0, y = 0;
	    GdkWindow *window = w->window;

	    if(window != NULL)
		gdk_window_get_root_origin(window, &x, &y);

	    EDV_SET_I(EDV_CFG_PARM_BROWSER_X, x);
	    EDV_SET_I(EDV_CFG_PARM_BROWSER_Y, y);
	    EDV_SET_I(EDV_CFG_PARM_BROWSER_WIDTH, w->allocation.width);
	    EDV_SET_I(EDV_CFG_PARM_BROWSER_HEIGHT, w->allocation.height);
	}

	/* Main paned position */
	w = browser->main_paned;
	if(w != NULL)
	{
	    EDV_SET_I(
		EDV_CFG_PARM_BROWSER_PANED_POSITION,
		GTK_PANED(w)->child1_size
	    );
	}

	/* EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN_WIDTH parameter is updated
	 * as they change in EDVBrowserResizeColumnCB() or when the
	 * configuration list is changed directly
	 */

	/* EDV_CFG_PARM_BROWSER_CONTENTS_COLUMN parameter is updated
	 * when the configuration list is changed directly
	 */

	/* Set new startup directory as the last directory of this
	 * window's location
	 */
	if(EDV_GET_B(EDV_CFG_PARM_RECORD_START_UP_DIR))
	{
	    EDV_SET_S(
		EDV_CFG_PARM_DIR_START_UP,
		EDVBrowserCurrentLocation(browser)
	    );
	}

	/* Directory tree origin path */
	if(!STRISEMPTY(browser->directory_ctree_origin_path))
	{
	    EDV_SET_S(
		EDV_CFG_PARM_BROWSER_DIR_TREE_ORIGIN,
		browser->directory_ctree_origin_path
	    );
	}

	/* Contents list filter */
	if(STRISEMPTY(browser->contents_list_filter))
	{
	    EDV_SET_S(
		EDV_CFG_PARM_BROWSER_CONTENTS_LIST_FILTER,
		"*"
	    );
	}
	else
	{
	    EDV_SET_S(
		EDV_CFG_PARM_BROWSER_CONTENTS_LIST_FILTER,
		browser->contents_list_filter
	    );
	}
}

/*
 *	Creates a new File Browser Operaton IDs list.
 *
 *	The specified File Browser may be NULL.
 */
GList *EDVBrowserOPIDListNew(edv_browser_struct *browser)
{
	const gchar *button_name, *menu_name, *tooltip;
	guint accel_key, accel_mods;
	guint8 **button_icon_data, **menu_icon_data;
	GList *opid_list = NULL;
	toolbar_item_type tb_item_type;
	edv_opid_flags flags;
	edv_browser_op op;
	edv_browser_opid_struct *opid;
	void (*func_cb)(toolbar_item_struct *, gint, gpointer);
	void (*enter_func_cb)(toolbar_item_struct *, gint, gpointer);

#define DO_APPEND_OPID	{			\
 opid = EDV_BROWSER_OPID(			\
  g_malloc0(sizeof(edv_browser_opid_struct))	\
 );						\
 if(opid != NULL) {				\
  opid->op = op;				\
  opid->flags = flags;				\
  opid->tb_item_type = tb_item_type;		\
  opid->func_cb = func_cb;			\
  opid->enter_func_cb = enter_func_cb;		\
  opid->button_name = STRDUP(button_name);	\
  opid->menu_name = STRDUP(menu_name);		\
  opid->tooltip = STRDUP(tooltip);		\
  opid->accel_key = accel_key;			\
  opid->accel_mods = accel_mods;		\
  opid->button_icon_data = button_icon_data;	\
  opid->menu_icon_data = menu_icon_data;	\
  opid->browser = browser;			\
  opid_list = g_list_append(opid_list, opid);	\
 }						\
}

	func_cb = EDVBrowserOPCB;
	enter_func_cb = EDVBrowserOPEnterCB;


	op = EDV_BROWSER_OP_SEPARATOR;
	flags = EDV_OPID_ALLOW_MULTIPLE | EDV_OPID_NO_OP;
	tb_item_type = TOOLBAR_ITEM_SEPARATOR;
	button_icon_data = (guint8 **)icon_hsep_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Sparateur";
	menu_name = "Sparateur";
	tooltip = "Sparateur";
#elif defined(PROG_LANGUAGE_GERMAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_ITALIAN)
	button_name = "Separatore";
	menu_name = "Separatore";
	tooltip = "Separatore";
#elif defined(PROG_LANGUAGE_DUTCH)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	button_name = "Separador";
	menu_name = "Separador";
	tooltip = "Separador";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#else
	button_name = "Separator";
	menu_name = "Separator";
	tooltip = "Separator";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_CLOSE;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_close_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Cierre";
	menu_name = "Cierre";
	tooltip = "Cierre esta ventana";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Proche";
	menu_name = "Proche";
	tooltip = "Fermer cette fentre";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zamknij";
	menu_name = "Zamknij";
	tooltip = "Zamknij to okno";
#else
	button_name = "Close";
	menu_name = "Close";
	tooltip = "Close this window";
#endif
	accel_key = GDK_w;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_EXIT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_exit_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Salida";
	menu_name = "La Salida";
	tooltip = "Cierra todas ventanas en la aplicacin y la";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Sortie";
	menu_name = "Sortie";
	tooltip = "Ferme toutes fentres dans l'application et sortie";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Wyjcie";
	menu_name = "Wyjcie";
	tooltip = "Zamknij wszystkie okna aplikacji i wyjd";
#else
	button_name = "Exit";
	menu_name = "Exit";
	tooltip = "Close all windows in the application and exit";
#endif
	accel_key = GDK_q;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_SYNC_DISKS;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_save_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Sincronizar Discos";
	menu_name = "Sincronizar Discos";
	tooltip = "Limpan cualquiera transferencias pendientes de datos al disco";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Disq Syn";
	menu_name = "Les Disques De Syn";
	tooltip = "Rincer des transferts de donnes en attente au disque";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Synchr.dyski";
	menu_name = "Synchronizuj dyski";
	tooltip = "Wykonaj wszystkie oczekujce transfery na dysk";
#else
	button_name = "Sync Disks";
	menu_name = "Sync Disks";
	tooltip = "Flush any pending data transfers to disk";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_HISTORY;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_trace_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Historia";
	menu_name = "La Historia...";
	tooltip = "La Historia de rdenes procesadas por esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "L'Histoire";
	menu_name = "L'Histoire...";
	tooltip = "L'Histoire d'ordres traits par cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Historia";
	menu_name = "Historia...";
	tooltip = "Historia komend wykonanych przez t aplikacj";
#else
	button_name = "History";
	menu_name = "History...";
	tooltip = "History of commands processed by this application";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_RUN;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_run_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Corra";
	menu_name = "Corra...";
	tooltip = "Corra un programa";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Courir";
	menu_name = "Courir...";
	tooltip = "Excute un programme";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Uruchom";
	menu_name = "Uruchom...";
	tooltip = "Uruchom program";
#else
	button_name = "Run";
	menu_name = "Run...";
	tooltip = "Run a program";
#endif
	accel_key = GDK_r;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_RUN_TERMINAL;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_terminal2_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Terminal";
	menu_name = "La Terminal";
	tooltip = "Corre una terminal";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Le Terminal";
	menu_name = "Le Terminal";
	tooltip = "Courir un terminal";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Uruchom terminal";
#else
	button_name = "Terminal";
	menu_name = "Terminal";
	tooltip = "Run a terminal";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_WRITE_PROTECT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_secure_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Proteja";
	menu_name = "Proteja contra escritura";
	tooltip = "Habilite/inhabilita la proteccion contra escritura";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Protge";
	menu_name = "Protge En criture";
	tooltip = "Rend capable/rend infirme protge en criture";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zabezpieczenie";
	menu_name = "Zabezpieczenie przed zapisem";
	tooltip = "Wcz/wycz zabezpieczenie przed zapisem";
#else
	button_name = "Protect";
	menu_name = "Write Protect";
	tooltip = "Enable/disable write protect";
#endif
	accel_key = GDK_y;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_DELETE_METHOD_RECYCLE;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_trash_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Recycle";
	menu_name = "Delete Method: Recycle";
	tooltip = "Set the delete method to recycle";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_DELETE_METHOD_PURGE;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_purge_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Purge";
	menu_name = "Delete Method: Purge";
	tooltip = "Set the delete method to purge";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_NEW;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_new_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Nuevo";
	menu_name = "Nuevo";
	tooltip = "Crea un objeto nuevo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Nouveau";
	menu_name = "Nouveau";
	tooltip = "Crer un nouvel objet";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowy";
	menu_name = "Nowy";
	tooltip = "Stwrz nowy obiekt";
#else
	button_name = "New";
	menu_name = "New";
	tooltip = "Create a new object";
#endif
	accel_key = GDK_n;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_NEW_FILE;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Archivo";
	menu_name = "El Archivo";
	tooltip = "Crea un objeto nuevo del archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Fichier";
	menu_name = "Fichier";
	tooltip = "Crer un nouvel objet de fichier";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowy plik";
	menu_name = "Plik";
	tooltip = "Stwrz nowy plik";
#else
	button_name = "New File";
	menu_name = "File";
	tooltip = "Create a new file";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_NEW_DIRECTORY;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_folder_closed_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Carpeta";
	menu_name = "Carpeta";
	tooltip = "Crea una nueva carpeta";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Annuaire";
	menu_name = "Annuaire";
	tooltip = "Crer un nouvel objet d'annuaire";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowy katalog";
	menu_name = "Katalog";
	tooltip = "Stwrz nowy katalog";
#else
	button_name = "New Dir";
	menu_name = "Directory";
	tooltip = "Create a new directory";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_NEW_LINK;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_link2_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Enlace";
	menu_name = "Enlace";
	tooltip = "Crea un nuevo enlace simblico";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Lien";
	menu_name = "Lien";
	tooltip = "Cre un nouveau symbolique lien";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowy odnonik";
	menu_name = "Odnonik";
	tooltip = "Stwrz nowy odnonik symboliczny";
#else
	button_name = "New Lnk";
	menu_name = "Link";
	tooltip = "Create a new symbolic link";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_NEW_FIFO;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_pipe_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FIFO";
	menu_name = "FIFO Tubo";
	tooltip = "Crea un objeto nuevo de tubo de FIFO";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FIFO";
	menu_name = "FIFO Tuyau";
	tooltip = "Crer un nouvel objet de tuyau de FIFO";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowe FIFO";
	menu_name = "Rura FIFO";
	tooltip = "Stwrz now rur FIFO";
#else
	button_name = "New FIFO";
	menu_name = "FIFO Pipe";
	tooltip = "Create a new FIFO pipe";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_NEW_DEVICE_BLOCK;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_device_block_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "BDispositivo";
	menu_name = "El Dispositivo Del Bloque";
	tooltip = "Crea un objeto nuevo de nodo de Dispositivo de bloque (buffered)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "BL'Appareil";
	menu_name = "Le Priphrique  Blocs";
	tooltip = "Crer un nouveau bloc (buffered) l'objet de noeu";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowe urz.blok.";
	menu_name = "Urzdzenie blokowe";
	tooltip = "Stwrz nowy wze urzdzenia blokowego (buforowanego)";
#else
	button_name = "New BDev";
	menu_name = "Block Device";
	tooltip = "Create a new block (buffered) device node";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_NEW_DEVICE_CHARACTER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_device_character_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "CDispositivo";
	menu_name = "El Dispositivo Del Carcter";
	tooltip = "Crea un objeto nuevo de nodo de Dispositivo de carcter (unbuffered)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "CL'Appareil";
	menu_name = "Le Priphrique  Caractres";
	tooltip = "Crer un nouveau caractre (unbuffered) l'objet de noeud d'appareil";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowe urz.znak.";
	menu_name = "Urzdzenie znakowe";
	tooltip = "Stwrz nowy wze urzdzenia znakowego (niebuforowanego)";
#else
	button_name = "New CDev";
	menu_name = "Character Device";
	tooltip = "Create a new character (unbuffered) device node";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_NEW_SOCKET;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_socket_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Enchufe";
	menu_name = "El Enchufe";
	tooltip = "Crea un objeto nuevo del enchufe";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Douille";
	menu_name = "La Douille";
	tooltip = "Crer un nouvel objet de douille";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nowe gniazdo";
	menu_name = "Gniazdo";
	tooltip = "Stwrz nowe gniazdo";
#else
	button_name = "New Sock";
	menu_name = "Socket";
	tooltip = "Create a new socket";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_OPEN;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_open_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Abrir";
	menu_name = "Abrir";
	tooltip = "Abre archivo escogido";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Ouvert";
	menu_name = "Ouvert";
	tooltip = "Ouvrir l'objet choisi";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Otwrz";
	menu_name = "Otwrz";
	tooltip = "Otwrz wybrany obiekt";
#else
	button_name = "Open";
	menu_name = "Open";
	tooltip = "Open the selected object(s)";
#endif
	accel_key = GDK_Return;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_OPEN_TO;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_open_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Open To";
	menu_name = "Open To...";
	tooltip = "Open the selected object(s) using a specific method";
	accel_key = GDK_Return;
	accel_mods = GDK_CONTROL_MASK | GDK_SHIFT_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_OPEN_WITH;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_open_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Abrir Con";
	menu_name = "Abrir Con...";
	tooltip = "Abrir archivo escogido usando un mtodo especfico";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Ouvrir Avec";
	menu_name = "Ouvrir Avec...";
	tooltip = "Ouvrir l'objet choisi pour utiliser une mthode spcifique";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Otwrz z...";
	menu_name = "Otwrz z...";
	tooltip = "Otwrz wybrany obiekt uywajc sbosobu wybranego przez uytkownika";
#else
	button_name = "Open With";
	menu_name = "Open With...";
	tooltip = "Open the selected object(s) using a specific application";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_SHIFT_MASK;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_COPY_PATH;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_copy_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "CopiSende";
	menu_name = "Copie Sendero";
	tooltip = "Copia la ubicacin del objeto(s) escogido como sendero(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "CopiSenti";
	menu_name = "Copier Le Sentier";
	tooltip = "Copier l'emplacement de l'objet choisi(s) comme le sentier(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Kopiuj ciek";
	menu_name = "Kopiuj ciek";
	tooltip = "Kopiuj lokacj zaznaczonego (zaznaczonych) obiektu (obiektw) jako ciek (cieki)";
#else
	button_name = "CopyPath";
	menu_name = "Copy Path";
	tooltip = "Copy the location of the selected object(s) as path(s)";
#endif
	accel_key = GDK_c;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_COPY_URL;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_copy_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "CopiURL";
	menu_name = "Copie URL";
	tooltip = "Copie la ubicacin del objeto(s) escogido como url(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "CopiURL";
	menu_name = "Copier URL";
	tooltip = "Copier l'emplacement de l'objet choisi(s) comme url(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Kopiuj URL";
	menu_name = "Kopiuj URL";
	tooltip = "Kopiuj lokacj zaznaczonego (zaznaczonych) obiektu (obiektw) jako URL";
#else
	button_name = "CopyURL";
	menu_name = "Copy URL";
	tooltip = "Copy the location of the selected object(s) as url(s)";
#endif
	accel_key = GDK_c;
	accel_mods = GDK_CONTROL_MASK | GDK_SHIFT_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_PASTE;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_paste_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "Coller";
	menu_name = "Coller";
	tooltip = "Coller object(s) depuis l'objet source(s)";
#else
	button_name = "Paste";
	menu_name = "Paste";
	tooltip = "Paste object(s) from coppied location source(s)";
#endif
	accel_key = GDK_v;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_MOVE;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_move_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Mueva";
	menu_name = "Mueva...";
	tooltip = "Mueva los objeto(s) escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Mouvement";
	menu_name = "Le mouvement...";
	tooltip = "Le Mouvement a choisi l'objet(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Przenie";
	menu_name = "Przenie...";
	tooltip = "Przenie zaznaczony(e) obiekt(y)";
#else
	button_name = "Move";
	menu_name = "Move...";
	tooltip = "Move the selected object(s)";
#endif
	accel_key = GDK_F7;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_COPY;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_copy_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Copia";
	menu_name = "Copia...";
	tooltip = "Copia los objeto(s) escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Copia";
	menu_name = "La Copie...";
	tooltip = "La Copie a choisi l'objet(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Kopiuj";
	menu_name = "Kopiuj...";
	tooltip = "Kopiuj zaznaczony(e) obiekt(y)";
#else
	button_name = "Copy";
	menu_name = "Copy...";
	tooltip = "Copy the selected object(s)";
#endif
	accel_key = GDK_F8;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_LINK;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_link2_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Enlace";
	menu_name = "Enlace...";
	tooltip = "Enlace los objeto(s) escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Le Lien";
	menu_name = "Le Lien...";
	tooltip = "Le Lien a choisi l'objet(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Odnonik";
	menu_name = "Odnonik...";
	tooltip = "Utwrz odnonik do zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "Link";
	menu_name = "Link...";
	tooltip = "Link the selected object(s)";
#endif
	accel_key = GDK_F6;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_RENAME;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_rename_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Renombrar";
	menu_name = "Renombrar";
	tooltip = "Renombrar archivo escogido";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Renommer";
	menu_name = "Renommer";
	tooltip = "Renomme l'objet choisi";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Zmie nazw";
	menu_name = "Zmie nazw";
	tooltip = "Zmie nazw zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "Rename";
	menu_name = "Rename";
	tooltip = "Rename the selected object";
#endif
	accel_key = GDK_F2;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_CHMOD;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_chmod_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ChMod";
	menu_name = "Cambie permisos...";
	tooltip = "El Cambio escogi los permisos de objetos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ChMod";
	menu_name = "Changer Le Mode...";
	tooltip = "Le Changement a choisi des permissions des objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "ChMod";
	menu_name = "Zmiana uprawnie...";
	tooltip = "Zmie uprawnienia dostpu do zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "ChMod";
	menu_name = "Change Permissions...";
	tooltip = "Change the permissions of the selected object(s)";
#endif
	accel_key = GDK_F9;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_CHOWN;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_owned_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Chown escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Chown a choisi l'objet(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "ChOwn";
	menu_name = "Zmiana waciciela...";
	tooltip = "Zmie wacicela zaznaczonego (zaznaczonych) obiektu (obiektw)";
#else
	button_name = "ChOwn";
	menu_name = "Change Ownership...";
	tooltip = "Change the ownership of the selected object(s)";
#endif
	accel_key = GDK_F10;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_CHTIME;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_time_stamp_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "ChTime";
	menu_name = "Change Time Stamps...";
	tooltip = "Change the time stamps of the selected object(s)";
	accel_key = GDK_F11;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_DELETE;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_cancel_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Borre";
	menu_name = "Borre";
	tooltip = "Borre los objeto(s)escogido(s)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Effacer";
	menu_name = "Effacer";
	tooltip = "Efface l'objet choisi(s)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Usu";
	menu_name = "Usu";
	tooltip = "Usu zaznaczony(e) obiekt(y)";
#else
	button_name = "Delete";
	menu_name = "Delete";
	tooltip = "Delete the selected object(s)";
#endif
	accel_key = GDK_Delete;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_PROPERTIES;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_properties2_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Props";
	menu_name = "Las Propiedades";
	tooltip = "Modifican escogi las propiedades de objeto";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Props";
	menu_name = "Les Proprits";
	tooltip = "Modifier a choisi les proprits de l'objet";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Waciwoci";
	menu_name = "Waciwoci...";
	tooltip = "Modyfikuj waciwoci zaznaczonego obiektu";
#else
	button_name = "Properties";
	menu_name = "Properties...";
	tooltip = "Modify the selected object's properties";
#endif
	accel_key = GDK_Return;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_SELECT_ALL;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_select_all_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "SelAll";
	menu_name = "Escoja Todo";
	tooltip = "Escoge todos los objetos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "SelAll";
	menu_name = "Choisir Tout";
	tooltip = "Choisir tous objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "ZaznWszyst";
	menu_name = "Zaznacz wszystko";
	tooltip = "Zaznacz wszystkie obiekty";
#else
	button_name = "SelAll";
	menu_name = "Select All";
	tooltip = "Select all objects";
#endif
	accel_key = GDK_a;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_UNSELECT_ALL;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_unselect_all_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "UnSelAll";
	menu_name = "Unescoja Todo";
	tooltip = "Unescoge todo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "UnSelAll";
	menu_name = "Unchoisir Tout";
	tooltip = "Unchoisir tous objets";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "OdznWszyst";
	menu_name = "Odznacz wszystko";
	tooltip = "Usu zaznaczenie ze wszystkich obiektw";
#else
	button_name = "UnSelAll";
	menu_name = "Unselect All";
	tooltip = "Unselect all objects";
#endif
	accel_key = GDK_u;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_INVERT_SELECTION;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_invert_selection_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "InvSel";
	menu_name = "Invierta Seleccin";
	tooltip =
 "Selectos todos objetos y unselect no seleccionados todos objetos escogidos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "InvSel";
	menu_name = "Inverser Slection";
	tooltip =
 "Choisir tous objets dslections et deslectionner tous objets choisis";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "OdwrZazn";
	menu_name = "Odwr zaznaczenie";
	tooltip =
 "Zaznacz wszystkie niezaznaczone obiekty i usu zaznaczenie ze wszystkich zaznaczonych obiektw";
#else
	button_name = "InvSel";
	menu_name = "Invert Selection";
	tooltip =
 "Select all unselected objects and unselect all selected objects";
#endif
	accel_key = GDK_i;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_FIND;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_search_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Buscar";
	menu_name = "Buscar";
	tooltip = "Buscar objeto(s) por atributos especificados";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Dcouverte";
	menu_name = "La Dcouverte";
	tooltip = "L'objet de Dcouverte (s) par les attributs spcifis ";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Znajd";
	menu_name = "Znajd...";
	tooltip = "Znajd obiekt(y) o podanych atrybutach";
#else
	button_name = "Find";
	menu_name = "Find...";
	tooltip = "Find object(s) by specific attributes";
#endif
	accel_key = GDK_f;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_DOWNLOAD;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_download_file_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Download";
	menu_name = "Download";
	tooltip = "Download del objeto";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Download";
	menu_name = "Download";
	tooltip = "Download de l'objet";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pobierz";
	menu_name = "Pobierz";
	tooltip = "Pobierz obiekt";
#else
	button_name = "Download";
	menu_name = "Download";
	tooltip = "Download an object";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_REFRESH;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Refresque";
	menu_name = "Refresque";
	tooltip = "Refresque carpeta escogida para listar";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Rafrachir";
	menu_name = "Rafrachir";
	tooltip = "Rafrachit a choisi l'numrer de l'annuaire";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Odwie";
	menu_name = "Odwie";
	tooltip = "Odwie list obiektw biecego katalogu";
#else
	button_name = "Refresh";
	menu_name = "Refresh";
	tooltip = "Refresh the selected directory's listing";
#endif
	accel_key = GDK_F5;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_REFRESH_ALL;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Refresque Todo";
	menu_name = "Refresque Todo";
	tooltip = "Refresca todas listas";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Rafrachir Tout";
	menu_name = "Rafrachir Tout";
	tooltip = "Rafrachir toutes listes";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Odwie wszystko";
	menu_name = "Odwie wszystko";
	tooltip = "Odwie wszystkie listy";
#else
	button_name = "Refresh All";
	menu_name = "Refresh All";
	tooltip = "Refresh all listings";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_GOTO_PARENT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_folder_parent_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Carpeta padre";
	menu_name = "Ir a carpeta padre";
	tooltip = "Para ir a la carpeta padre";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Parent";
	menu_name = "Aller Au Parent";
	tooltip = "Aller  l'annuaire de parent";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Nadrzdny";
	menu_name = "Id do nadrzdnego";
	tooltip = "Id do katalogu nadrzdnego";
#else
	button_name = "Parent";
	menu_name = "Go To Parent";
	tooltip = "Go to the parent directory";
#endif
	accel_key = GDK_BackSpace;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_GOTO_HOME;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_folder_home_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "Accueil";
	menu_name = "Aller  l'accueil";
	tooltip = "Aller au rpertoire d'accueil (HOME)";
#else
	button_name = "Home";
	menu_name = "Go To Home";
	tooltip = "Go to the home directory";
#endif
	accel_key = GDK_grave;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_EXPAND;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = NULL;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Ensanchar";
	menu_name = "Ensanchar";
	tooltip = "Ensanchar/el desplome escogi rama de rbol";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Augmenter";
	menu_name = "Augmenter";
	tooltip = "L'Augmente/Effondrement branche d'arbre choisie";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Rozwi";
	menu_name = "Rozwi";
	tooltip = "Rozwi/zwi wybran ga drzewa katalogw";
#else
	button_name = "Expand";
	menu_name = "Expand";
	tooltip = "Expand/collapse selected tree branch";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_SHOW_TOOL_BAR;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "La Exposicin/la barra de herramienta de cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "La barre d'outil de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasNarz";
	menu_name = "Pasek narzdzi";
	tooltip = "Poka/ukryj pasek narzdzi";
#else
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "Show/hide tool bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_SHOW_LOCATION_BAR;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "La exposicin/la barra de la ubicacin del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "La barre d'emplacement de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pascieki";
	menu_name = "Pasek cieki";
	tooltip = "Poka/ukryj pasek cieki";
#else
	button_name = "LocBar";
	menu_name = "Location Bar";
	tooltip = "Show/hide location bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_SHOW_MOUNT_BAR;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "MountBar";
	menu_name = "Mount Bar";
	tooltip = "La Exposicin/la barra del monte del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "MountBar";
	menu_name = "Mount Bar";
	tooltip = "La barre de mont de Show/Peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasMontowania";
	menu_name = "Pasek montowania";
	tooltip = "Poka/ukryj pasek montowania";
#else
	button_name = "MountBar";
	menu_name = "Mount Bar";
	tooltip = "Show/hide mount bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_SHOW_FIND_BAR;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "La exposicin/la barra del hallazgo del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "La barre de dcouverte de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasSzuk";
	menu_name = "Pasek wyszukiwania";
	tooltip = "Poka/ukryj pasek wyszukiwania";
#else
	button_name = "FindBar";
	menu_name = "Find Bar";
	tooltip = "Show/hide find bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_SHOW_STATUS_BAR;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_status_bar_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "La exposicin/la barra de la posicin del cuero";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "La barre de statut de montre/peau";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PasStatusu";
	menu_name = "Pasek statusu";
	tooltip = "Poka/ukryj pasek statusu";
#else
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "Show/hide status bar";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_CONTENTS_LIST_FILTER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_wildcards_20x20_xpm;
	menu_icon_data = button_icon_data;
	button_name = "Filter";
	menu_name = "Set Filter...";
	tooltip = "Set the name filter for listing objects";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_resize_column_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "ReSzCol";
	menu_name = "Auto Resize Columns";
	tooltip = "Automatically resize the columns each time the list is updated";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_SHOW_HIDDEN_OBJECTS;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_file_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "HidObj";
	menu_name = "Objets Cachs";
	tooltip = "Monter/cacher objets cachs(.)";
#else
	button_name = "HidObj";
	menu_name = "Hidden Objects";
	tooltip = "Show/hide hidden objects";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_SHOW_NOACCESS_OBJECTS;
	tb_item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_file_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "NoAccObj";
	menu_name = "Objet  accs rserv";
	tooltip = "Monter/cacher les objets  accs rserv";
#else
	button_name = "NoAccObj";
	menu_name = "No Access Objects";
	tooltip = "Show/hide no access objects";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_DIRECTORY_TREE_ORIGIN;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_tree_origin_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_FRENCH)
	button_name = "TreeOrig";
	menu_name = "Rpertoire Initial...";
	tooltip = "Positionner au rpertoire initial";
#else
	button_name = "TreeOrig";
	menu_name = "Directory Tree Origin...";
	tooltip = "Set the directory tree origin";
#endif
	accel_key = GDK_j;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_MOUNT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_mount_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Montar";
	menu_name = "Montar";
	tooltip = "Montar/desmontar carpeta de punto de montaje seleccionada";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Le Mont";
	menu_name = "Le Mont";
	tooltip = "Le Mont/unmount a choisi l'annuaire de point de mont";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Montowanie";
	menu_name = "Montowanie";
	tooltip = "Zamontuj odmontuj wybrany punkt montowania";
#else
	button_name = "Mount";
	menu_name = "Mount";
	tooltip = "Mount/unmount selected mount point directory";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_EJECT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_eject_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Expulse";
	menu_name = "Expulse";
	tooltip = "Expulse medios del Dispositivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Ejecter";
	menu_name = "Ejecter";
	tooltip = "Ejecter le presse de l'appareil";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Wysu";
	menu_name = "Wysu";
	tooltip = "Wysu nonik z napdu";
#else
	button_name = "Eject";
	menu_name = "Eject";
	tooltip = "Eject media from device";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_DEVICE_CHECK;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_fsck_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FSCK";
	menu_name = "Archive Cheque De Sistema...";
	tooltip = "Verifique el sistema del archivo de Dispositivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FSCK";
	menu_name = "Classer Le Chque De Systme...";
	tooltip = "Vrifier le systme de fichier de l'appareil";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "FSCK";
	menu_name = "Sprawd system plikw...";
	tooltip = "Sprawd system plikw urzdzenia pod ktem bdw";
#else
	button_name = "FSCK";
	menu_name = "File System Check...";
	tooltip = "Check the device's file system";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_DEVICE_TOOLS;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_tools_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "DevTool";
	menu_name = "Herramienta Equipar Dispositivo  ...";
	tooltip = "Corra el programa de herramientas de Dispositivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "DevTool";
	menu_name = "Outils D'Appareil...";
	tooltip = "Courir le programme d'outils de l'appareil";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "NarzUrzdzenia";
	menu_name = "Narzdzia urzdzenia...";
	tooltip = "Uruchom program narzdzi urzdzenia";
#else
	button_name = "DevTool";
	menu_name = "Device Tools...";
	tooltip = "Run the device's tools program";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_DEVICE_FORMAT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_floppy_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Format";
	menu_name = "Formatear Medios...";
	tooltip = "Formatear los medios en el Dispositivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Format";
	menu_name = "Met Format Presse...";
	tooltip = "Met en format le presse dans l'appareil";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Formatuj";
	menu_name = "Formatuj nonik...";
	tooltip = "Formatuj nonik w napdzie";
#else
	button_name = "Format";
	menu_name = "Format Media...";
	tooltip = "Format the media in the device";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_MIME_TYPES;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_mimetypes_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "El panorama/modifica MIME Types";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "Vue/modifie MIME Types";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Typy MIME";
	menu_name = "Typy MIME...";
	tooltip = "Obejrzyj/zmodyfikuj typy MIME";
#else
	button_name = "MIMETypes";
	menu_name = "MIME Types...";
	tooltip = "View/modify MIME Types";
#endif
	accel_key = GDK_m;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_DEVICES;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_devices_list_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Dispositivos";
	menu_name = "Los Dispositivos...";
	tooltip = "El Panorama/modifica las referencias de Dispositivo de sistema de archivo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Appareils";
	menu_name = "Les Appareils...";
	tooltip = "Les rfrences de priphrique-systme de fichier de Vue/Modifie ";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Urzdzenia";
	menu_name = "Urzdzenia...";
	tooltip = "Obejrzyj/zmodyfikuj systemowe ustawienia urzdze";
#else
	button_name = "Devices";
	menu_name = "Devices...";
	tooltip = "View/modify file system device references";
#endif
	accel_key = GDK_d;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_NEW_BROWSER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)endeavour_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "FBrowser";
	menu_name = "Nuevo Examinador De Archivos";
	tooltip = "Muestra un nuevo examinador de archivos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "FBrowser";
	menu_name = "Nouveau Browser de Fichier";
	tooltip = "Crer un nouveau browser de fichier";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PrzeglPlik";
	menu_name = "Nowa przegldarka plikw";
	tooltip = "Utwrz now przegldark plikw";
#else
	button_name = "FBrowser";
	menu_name = "New File Browser";
	tooltip = "Create a new file browser";
#endif
	accel_key = GDK_1;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_NEW_IMBR;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)image_browser_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "IBrowser";
	menu_name = "Nuevo Examinador De Imagenes";
	tooltip = "Muestra un nuevo examinador de imagenes";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "IBrowser";
	menu_name = "Nouveau Browser D'Image";
	tooltip = "Crer un nouveau browser d'image";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PrzeglObr";
	menu_name = "Nowa przegldarka obrazw";
	tooltip = "Utwrz now przegldark obrazw";
#else
	button_name = "IBrowser";
	menu_name = "New Image Browser";
	tooltip = "Create a new image browser";
#endif
	accel_key = GDK_2;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_NEW_ARCHIVER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)archiver_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Archiver";
	menu_name = "Nuevo Compresor De Archivos";
	tooltip = "Muestra un archivo comprimido nuevo";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Archiver";
	menu_name = "Nouveau Archiver";
	tooltip = "Crer un nouveau archiver";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Archiwizator";
	menu_name = "Nowy archiwizator";
	tooltip = "Utwrz nowego archwizatora";
#else
	button_name = "Archiver";
	menu_name = "New Archiver";
	tooltip = "Create a new archiver";
#endif
	accel_key = GDK_3;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_RECYCLE_BIN;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_trash_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "RecBin";
	menu_name = "El Cajn De Reciclaje";
	tooltip = "Abrir Cajn de Reciclaje (basura)";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "RecBin";
	menu_name = "Recycler L'Huche";
	tooltip = "Recycler l'huche (la poubelle)";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Kosz";
	menu_name = "Kosz";
	tooltip = "Otwrz kosz";
#else
	button_name = "RecBin";
	menu_name = "Recycle Bin";
	tooltip = "Open Recycle Bin";
#endif
	accel_key = GDK_4;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_OPTIONS;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_options2_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Las Opciones";
	menu_name = "Las Opciones...";
	tooltip = "Modifican estas opciones de la aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Les Options";
	menu_name = "Les Options";
	tooltip = "Modifier ces options de l'application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Opcje";
	menu_name = "Opcje...";
	tooltip = "Modyfikuj opcje tej aplikacji";
#else
	button_name = "Options";
	menu_name = "Options...";
	tooltip = "Modify this application's options";
#endif
	accel_key = GDK_o;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_CUSTOMIZE;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_customize_20x20_xpm;
	menu_icon_data = button_icon_data;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Personal";
	menu_name = "Personaliza...";
	tooltip = "Personaliza la mirada y se siente de esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "Personnal";
	menu_name = "Personnalise...";
	tooltip = "Personnalise le regard et sens de cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Dostosuj";
	menu_name = "Dostosuj...";
	tooltip = "Dostosuj wygld i zachowanie tej aplikacji";
#else
	button_name = "Custom";
	menu_name = "Customize...";
	tooltip = "Customize the look and feel of this application";
#endif
	accel_key = GDK_t;
	accel_mods = GDK_CONTROL_MASK;
	DO_APPEND_OPID


	op = EDV_BROWSER_OP_HELP_CONTENTS;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "La Ayuda";
	menu_name = "El Contenido";
	tooltip = "El Indice de toda ayuda pagina";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "L'Aide";
	menu_name = "Les Contenus";
	tooltip = "L'Index de toutes pages d'aide";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pomoc";
	menu_name = "Tematy pomocy";
	tooltip = "Spis treci wszystkich stron pomocy";
#else
	button_name = "Help";
	menu_name = "Contents";
	tooltip = "Index of all help pages";
#endif
	accel_key = GDK_F1;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_HELP_FILE_BROWSER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "File Browser";
	tooltip = "Help on using the file browser";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_HELP_IMAGE_BROWSER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Image Browser";
	tooltip = "Help on using the image browser";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_HELP_ARCHIVER;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Archiver";
	tooltip = "Help on using the archiver";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_HELP_RECYCLE_BIN;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	button_name = "Help";
	menu_name = "Recycle Bin";
	tooltip = "Help on using the recycle bin";
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_HELP_KEYS_LIST;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpKeys";
	menu_name = "La Lista De Llaves";
	tooltip = "La Lista de todo teclado las operaciones claves";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpKeys";
	menu_name = "La Liste De Clefs";
	tooltip = "La Liste de tout clavier toutes oprations principales";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomocKlaw";
	menu_name = "Lista klawiszy";
	tooltip = "Lista skrtw klawiaturowych";
#else
	button_name = "HelpKeys";
	menu_name = "Keys List";
	tooltip = "List of all keyboard key operations";

#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_HELP_MIME_TYPES;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Ayude a establecer MIME Types";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Aider sur montant MIME Types";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "Pomoc MIME";
	menu_name = "Typy MIME";
	tooltip = "Pomoc w ustawianiu typw MIME";
#else
	button_name = "HelpMIME";
	menu_name = "Mime Types";
	tooltip = "Help on setting up MIME Types";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_HELP_DEVICES;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpDev";
	menu_name = "Los Dispositivos";
	tooltip = "La Ayuda a establecer Dispositivos";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpDev";
	menu_name = "Les Appareils";
	tooltip = "L'Aide sur montant des appareils";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomUrzdzenia";
	menu_name = "Urzdzenia";
	tooltip = "Pomoc w ustawianiu urzdze";
#else
	button_name = "HelpDev";
	menu_name = "Devices";
	tooltip = "Help on setting up devices";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_HELP_COMMON_OPERATIONS;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "HelpOp";
	menu_name = "Las Operaciones Comunes";
	tooltip = "La Ayuda en las operaciones ms comunes";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "HelpOp";
	menu_name = "Les Oprations Communes";
	tooltip = "L'Aide sur montant des appareils";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "PomOperacje";
	menu_name = "Najczstsze operacje";
	tooltip = "Pomoc dotyczca najczstszych operacji";
#else
	button_name = "HelpOp";
	menu_name = "Common Operations";
	tooltip = "Help on the most common operations";

#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID

	op = EDV_BROWSER_OP_HELP_ABOUT;
	tb_item_type = TOOLBAR_ITEM_BUTTON;
	flags = 0;
	button_icon_data = (guint8 **)icon_about_20x20_xpm;
	menu_icon_data = (guint8 **)icon_about_20x20_xpm;
#if defined(PROG_LANGUAGE_SPANISH)
	button_name = "Acerca";
	menu_name = "Acerca De";
	tooltip = "Acerca de esta aplicacin";
#elif defined(PROG_LANGUAGE_FRENCH)
	button_name = "De";
	menu_name = "De";
	tooltip = "De cette application";
#elif defined(PROG_LANGUAGE_POLISH)
	button_name = "O programie";
	menu_name = "O programie";
	tooltip = "Informacje o programie";
#else
	button_name = "About";
	menu_name = "About";
	tooltip = "About this application";
#endif
	accel_key = 0;
	accel_mods = 0;
	DO_APPEND_OPID


	return(opid_list);

#undef DO_APPEND_OPID
}

/*
 *	Deletes the File Browser Operation IDs list.
 */
void EDVBrowserOPIDListDelete(GList *glist)
{
	GList *opid_list = glist;
	edv_browser_opid_struct *opid;

	for(;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    opid = EDV_BROWSER_OPID(glist->data);
	    if(opid == NULL)
		continue;

	    g_free(opid->button_name);
	    g_free(opid->menu_name);
	    g_free(opid->tooltip);

	    g_free(opid);
	}

	g_list_free(opid_list);
}


/*
 *	Creates a new Accel Key List with default values.
 */
cfg_accelkey_list_struct *EDVBrowserAccelkeysListNewDefault(void)
{
	GList *glist, *opid_list, *ak_glist = NULL;
	cfg_accelkey_list_struct *ak_list;
	edv_browser_opid_struct *opid;

	opid_list = EDVBrowserOPIDListNew(NULL);
	for(glist = opid_list; glist != NULL; glist = g_list_next(glist))
	{
	    opid = EDV_BROWSER_OPID(glist->data);
	    if(opid == NULL)
		continue;

	    if(opid->flags & EDV_OPID_NO_OP)
		continue;

	    ak_glist = g_list_append(
		ak_glist,
		CFGAccelkeyNew(
		    (gint)opid->op,
		    opid->accel_key,
		    opid->accel_mods
		)
	    );
	}
	EDVBrowserOPIDListDelete(opid_list);

	ak_list = CFGAccelkeyListNew(ak_glist);

	g_list_foreach(ak_glist, (GFunc)CFGAccelkeyDelete, NULL);
	g_list_free(ak_glist);

	return(ak_list);
}


/*
 *	Creates the File Browser's menu bar.
 *
 *	This function should be called by EDVBrowserNew().
 */
static void EDVBrowserCreateMenuBar(
	edv_browser_struct *browser, GtkWidget *parent
)
{
	guint8 **icon_data = NULL;
	edv_browser_op op;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *w, *menu, *menu_bar;

	/* Create Menu Bar */
	browser->menu_bar = menu_bar = GUIMenuBarCreate(NULL);
	gtk_container_add(GTK_CONTAINER(parent), menu_bar);
	gtk_widget_show(menu_bar);

#define DO_ADD_MENU_ITEM_LABEL	{		\
 edv_browser_opid_struct *opid =		\
  EDVBrowserMatchOPID(browser, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   NULL,					\
   opid, EDVBrowserMenuItemCB			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemEnterCB, opid,	\
   (gpointer)EDVBrowserMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_ITEM_CHECK	{		\
 edv_browser_opid_struct *opid =		\
  EDVBrowserMatchOPID(browser, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_CHECK, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   NULL,					\
   opid, EDVBrowserMenuItemCB			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemEnterCB, opid,	\
   (gpointer)EDVBrowserMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_SUBMENU_LABEL	{	\
 edv_browser_opid_struct *opid =		\
  EDVBrowserMatchOPID(browser, op);		\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, NULL,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   0, 0, NULL, NULL, NULL			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)EDVBrowserMenuItemEnterCB, opid,	\
   (gpointer)EDVBrowserMenuItemLeaveCB, opid	\
  );						\
 } else {					\
  w = NULL;					\
 }						\
}

#define DO_ADD_MENU_SEP	{			\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}
	/* Create menus */

	/* File Menu */
	browser->file_menu = menu = GUIMenuCreateTearOff();
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fichier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Akte"
#elif defined(PROG_LANGUAGE_ITALIAN)
"File"
#elif defined(PROG_LANGUAGE_DUTCH)
"Dossier"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiv"
#elif defined(PROG_LANGUAGE_POLISH)
"Plik"
#else
"File"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);
	EDVBrowserFileMenuItemsRegenerate(browser);

	/* Edit Menu */
	browser->edit_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_BROWSER_OP_MOVE;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->move_mi = w;

	    op = EDV_BROWSER_OP_COPY;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->copy_mi = w;

	    op = EDV_BROWSER_OP_LINK;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->link_mi = w;

	    op = EDV_BROWSER_OP_RENAME;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->rename_mi = w;

	    op = EDV_BROWSER_OP_CHMOD;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->chmod_mi = w;

	    op = EDV_BROWSER_OP_CHOWN;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->chown_mi = w;

	    op = EDV_BROWSER_OP_CHTIME;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->chtime_mi = w;

	    op = EDV_BROWSER_OP_DELETE;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->delete_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_COPY_PATH;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->copy_path_mi = w;

	    op = EDV_BROWSER_OP_COPY_URL;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->copy_url_mi = w;

	    op = EDV_BROWSER_OP_PASTE;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->paste_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_SELECT_ALL;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->select_all_mi = w;

	    op = EDV_BROWSER_OP_UNSELECT_ALL;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->unselect_all_mi = w;

	    op = EDV_BROWSER_OP_INVERT_SELECTION;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->invert_selection_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_FIND;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->find_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_PROPERTIES;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->properties_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Editar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#elif defined(PROG_LANGUAGE_POLISH)
"Edycja"
#else
"Edit"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);


	/* View Menu */
	browser->view_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_BROWSER_OP_REFRESH;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->refresh_mi = w;

	    op = EDV_BROWSER_OP_REFRESH_ALL;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->refresh_all_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_SHOW_TOOL_BAR;
	    DO_ADD_MENU_ITEM_CHECK
	    browser->show_tool_bar_micheck = w;

	    op = EDV_BROWSER_OP_SHOW_LOCATION_BAR;
	    DO_ADD_MENU_ITEM_CHECK
	    browser->show_location_bar_micheck = w;

	    op = EDV_BROWSER_OP_SHOW_MOUNT_BAR;
	    DO_ADD_MENU_ITEM_CHECK
	    browser->show_mount_bar_micheck = w;

	    op = EDV_BROWSER_OP_SHOW_FIND_BAR;
	    DO_ADD_MENU_ITEM_CHECK
	    browser->show_find_bar_micheck = w;

	    op = EDV_BROWSER_OP_SHOW_STATUS_BAR;
	    DO_ADD_MENU_ITEM_CHECK
	    browser->show_status_bar_micheck = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_DIRECTORY_TREE_ORIGIN;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->directory_tree_origin_mi = w;

	    DO_ADD_MENU_SEP       

	    op = EDV_BROWSER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS;
	    DO_ADD_MENU_ITEM_CHECK
	    browser->contents_list_auto_resize_columns_micheck = w;

	    op = EDV_BROWSER_OP_SHOW_HIDDEN_OBJECTS;
	    DO_ADD_MENU_ITEM_CHECK
	    browser->show_hidden_objects_micheck = w;

	    op = EDV_BROWSER_OP_SHOW_NOACCESS_OBJECTS;
	    DO_ADD_MENU_ITEM_CHECK
	    browser->show_noaccess_objects_micheck = w;

	    op = EDV_BROWSER_OP_CONTENTS_LIST_FILTER;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->contents_list_filter_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_MIME_TYPES;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->mime_types_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Vista"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vue"
#elif defined(PROG_LANGUAGE_GERMAN)
"Blick"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Veduta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Overzicht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Vista"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sikt"
#elif defined(PROG_LANGUAGE_POLISH)
"Widok"
#else
"View"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);

	/* Device Menu */
	browser->device_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_BROWSER_OP_MOUNT;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->mount_mi = w;

	    op = EDV_BROWSER_OP_EJECT;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->eject_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_DEVICE_CHECK;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->device_check_mi = w;

	    op = EDV_BROWSER_OP_DEVICE_TOOLS;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->device_tools_mi = w;

	    op = EDV_BROWSER_OP_DEVICE_FORMAT;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->device_format_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_DEVICES;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->devices_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Dispositivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Appareil"
#elif defined(PROG_LANGUAGE_GERMAN)
"Vorrichtung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Congegno"
#elif defined(PROG_LANGUAGE_DUTCH)
"Apparaat"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Artifcio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innretning"
#elif defined(PROG_LANGUAGE_POLISH)
"Urzdzenie"
#else
"Device"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);


	/* Settings Menu */
	browser->settings_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_BROWSER_OP_WRITE_PROTECT;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->write_protect_mi = w;

	    op = EDV_BROWSER_OP_DELETE_METHOD_RECYCLE;
	    DO_ADD_MENU_ITEM_CHECK
	    browser->delete_method_recycle_micheck = w;

	    op = EDV_BROWSER_OP_DELETE_METHOD_PURGE;
	    DO_ADD_MENU_ITEM_CHECK
	    browser->delete_method_purge_micheck = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_OPTIONS;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->options_mi = w;

	    op = EDV_BROWSER_OP_CUSTOMIZE;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->customize_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Configuracion"
#elif defined(PROG_LANGUAGE_FRENCH)
"Montages"
#elif defined(PROG_LANGUAGE_GERMAN)
"Setzen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Montaggi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Zetten"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Por"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Innstillinger"
#elif defined(PROG_LANGUAGE_POLISH)
"Ustawienia"
#else
"Settings"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);


	/* Windows Menu */
	browser->windows_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_BROWSER_OP_NEW_BROWSER;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->window_new_browser_mi = w;

	    op = EDV_BROWSER_OP_NEW_IMBR;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->window_new_imbr_mi = w;

	    op = EDV_BROWSER_OP_NEW_ARCHIVER;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->window_new_archiver_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_RECYCLE_BIN;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->window_recycle_bin_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ventanas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Windows"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fenster"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Finestre"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ramen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Janelas"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Vinduer"
#elif defined(PROG_LANGUAGE_POLISH)
"Okna"
#else
"Windows"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);


	/* Help Menu */
	browser->help_menu = menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    op = EDV_BROWSER_OP_HELP_CONTENTS;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->help_contents_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_HELP_FILE_BROWSER;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->help_file_browser_mi = w;

	    op = EDV_BROWSER_OP_HELP_KEYS_LIST;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->help_keys_list_mi = w;

	    op = EDV_BROWSER_OP_HELP_MIME_TYPES;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->help_mime_types_mi = w;

	    op = EDV_BROWSER_OP_HELP_DEVICES;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->help_devices_mi = w;

	    op = EDV_BROWSER_OP_HELP_COMMON_OPERATIONS;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->help_common_operations_mi = w;

	    DO_ADD_MENU_SEP

	    op = EDV_BROWSER_OP_HELP_ABOUT;
	    DO_ADD_MENU_ITEM_LABEL
	    browser->help_about_mi = w;
	}
	GUIMenuAddToMenuBar(
	    menu_bar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ayuda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aide"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hilfe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aiuto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Hulp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ajuda"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Hjelp"
#elif defined(PROG_LANGUAGE_POLISH)
"Pomoc"
#else
"Help"
#endif
	    , GUI_MENU_BAR_ALIGN_RIGHT
	);


#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_ITEM_CHECK
#undef DO_ADD_MENU_SUBMENU_LABEL
#undef DO_ADD_MENU_SEP
}


/*
 *	Creates a new File Browser.
 */
edv_browser_struct *EDVBrowserNew(edv_core_struct *core)
{
	const gchar *location_history_file = NULL;
	gint	toplevel_x, toplevel_y,
		toplevel_width, toplevel_height,
		paned_position;
	gboolean	tool_bar_map_state,
			location_bar_map_state,
			mount_bar_map_state,
			find_bar_map_state,
			status_bar_map_state;
	const gchar *directory_ctree_origin_path;
	gint i;
	const gchar *wm_name, *wm_class;
	GdkCursor *cursor;
	GList *glist;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkStyle *style;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*toplevel;
	gpointer combo_rtn;
	GtkCombo *combo;
	GtkCList *clist;
	GtkCTree *ctree;
	GdkWindow *window;
	const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING},
{EDV_DND_TARGET_RECYCLED_OBJECT, 0,	EDV_DND_INFO_RECYCLED_OBJECT},
{EDV_DND_TARGET_ARCHIVE_OBJECT,	0,	EDV_DND_INFO_ARCHIVE_OBJECT}
	};
	const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	};
	const cfg_item_struct *cfg_list;
	edv_status_bar_struct *status_bar;
	edv_browser_struct *browser;

	if(core == NULL)
	    return(NULL);

	cfg_list = core->cfg_list;

	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	wm_name = core->wm_name;
	wm_class = core->wm_class;

	location_history_file = EDV_GET_S(
	    EDV_CFG_PARM_FILE_BROWSER_LOCATION_HISTORY
	);

	if(core->geometry_set)
	{
	    const GdkRectangle *rect = &core->geometry;
	    toplevel_x = rect->x;
	    toplevel_y = rect->y;
	    toplevel_width = rect->width;
	    toplevel_height = rect->height;
	}
	else
	{
	    toplevel_x = EDV_GET_I(EDV_CFG_PARM_BROWSER_X);
	    toplevel_y = EDV_GET_I(EDV_CFG_PARM_BROWSER_Y);
	    toplevel_width = EDV_GET_I(EDV_CFG_PARM_BROWSER_WIDTH);
	    toplevel_height = EDV_GET_I(EDV_CFG_PARM_BROWSER_HEIGHT);
	}

	paned_position = EDV_GET_I(EDV_CFG_PARM_BROWSER_PANED_POSITION);

	tool_bar_map_state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_TOOL_BAR);
	location_bar_map_state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_LOCATION_BAR);
	mount_bar_map_state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_MOUNT_BAR);
	find_bar_map_state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_FIND_BAR);
	status_bar_map_state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_STATUS_BAR);

	directory_ctree_origin_path = EDV_GET_S(
	    EDV_CFG_PARM_BROWSER_DIR_TREE_ORIGIN
	);


	/* Create the File Browser */
	browser = EDV_BROWSER(g_malloc0(sizeof(edv_browser_struct)));
	if(browser == NULL)
	    return(NULL);

	browser->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	browser->processing = FALSE;
	browser->freeze_count = 0;
	browser->busy_count = 0;
	browser->core = core;

	browser->opid_list = EDVBrowserOPIDListNew(browser);

	browser->tool_bar_map_state = tool_bar_map_state;
	browser->location_bar_map_state = location_bar_map_state;
	browser->mount_bar_map_state = mount_bar_map_state;
	browser->find_bar_map_state = find_bar_map_state;
	browser->status_bar_map_state = status_bar_map_state;
	browser->contents_list_filter = STRDUP(EDV_GET_S(
	    EDV_CFG_PARM_BROWSER_CONTENTS_LIST_FILTER
	));
	browser->directory_ctree_expand_optimize_idleid = 0;
	browser->directory_ctree_selected_node = NULL;
	browser->contents_clist_selected_row = -1;
	browser->selected_dev_num = -1;
	browser->directory_ctree_origin_path = STRDUP(directory_ctree_origin_path);
	browser->last_recbin_items = -1;
	browser->last_write_protect_state = -1;
	browser->status_bar = NULL;

	browser->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_widget_set_name(w, EDV_BROWSER_TOPLEVEL_WIDGET_NAME);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_set_uposition(
	    w, toplevel_x, toplevel_y
	);
	gtk_widget_set_usize(
	    w, toplevel_width, toplevel_height
	);
	if(!STRISEMPTY(wm_name) && !STRISEMPTY(wm_class))
	    gtk_window_set_wmclass(GTK_WINDOW(w), wm_name, wm_class);
	else
	    gtk_window_set_wmclass(
		GTK_WINDOW(w), "filebrowser", PROG_NAME
	    );
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    GdkGeometry geo;

	    geo.min_width = 100;
	    geo.min_height = 70;
	    geo.base_width = 0;
	    geo.base_height = 0;
	    geo.width_inc = 1;
	    geo.height_inc = 1;
	    gdk_window_set_geometry_hints(
		window, &geo,
		GDK_HINT_MIN_SIZE |
		GDK_HINT_BASE_SIZE |
		GDK_HINT_RESIZE_INC
	    );

	    GUISetWMIcon(window, (guint8 **)endeavour_48x48_xpm);
	}
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVBrowserDeleteEventCB), browser
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVBrowserKeyEventCB), browser
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVBrowserKeyEventCB), browser
	);
	style = gtk_widget_get_style(w);
	parent = w;


	/* Main GtkVBox */
	browser->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Bars GtkVBox */
	browser->bars_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Menu Bar GtkHandleBox */
	browser->menu_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_attached",
	    GTK_SIGNAL_FUNC(EDVBrowserHandleChildAttachedCB), browser
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_detached",
	    GTK_SIGNAL_FUNC(EDVBrowserHandleChildDetachedCB), browser
	);
	gtk_widget_show(w);
	/* Menu Bar */
	EDVBrowserCreateMenuBar(browser, w);


	/* Tool Bar GtkHandleBox */
	browser->tool_bar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_attached",
	    GTK_SIGNAL_FUNC(EDVBrowserHandleChildAttachedCB), browser
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_detached",
	    GTK_SIGNAL_FUNC(EDVBrowserHandleChildDetachedCB), browser
	);
	if(browser->tool_bar_map_state)
	    gtk_widget_show(w);
	/* Tool Bar will be built later */
	browser->tool_bar = NULL;


	/* Location Bar GtkHandleBox */
	browser->location_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_attached",
	    GTK_SIGNAL_FUNC(EDVBrowserHandleChildAttachedCB), browser
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_detached",
	    GTK_SIGNAL_FUNC(EDVBrowserHandleChildDetachedCB), browser
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(browser->location_bar_map_state)
	    gtk_widget_show(w);
	parent2 = w;

	/* Location Bar GtkHBox */
	browser->location_bar_hbox = w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;


	/* Location Icon and Combo GtkHBox */
	w = gtk_hbox_new(FALSE, 0);	/* Don't need spacing between icon
					 * and combo because combo has it
					 * already.
					 */
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* GtkVBox to orient the location icon fixed parent */
	w = gtk_vbox_new(TRUE, 0);
	gtk_widget_set_usize(w, 20, 20);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Location Icon GtkFixed */
	browser->location_icon_fixed = w = gtk_fixed_new();
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVBrowserLocBarIconCrossingCB), browser
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVBrowserLocBarIconCrossingCB), browser
	);
	gtk_widget_realize(w);
	cursor = EDVGetCursor(core, EDV_CURSOR_CODE_HAND);
	gdk_window_set_cursor(w->window, cursor);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Arrastre esto para crear un enlace en esta ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Traner ceci pour crer un lien  cet emplacement"
#elif defined(PROG_LANGUAGE_POLISH)
"Przecignij to by utworzy odnonik do tej lokacji"
#else
"Drag this to create a link to this location"
#endif
	);
	gtk_widget_show(w);

	/* Reset the location icon pixmap */
	browser->location_icon_pm = NULL;

	/* Location GtkCombo */
	glist = EDVOpenFileGList(location_history_file, 0);
	w = GUIComboCreate(
#if defined(PROG_LANGUAGE_SPANISH)
	    "La Ubicacin:",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Emplacement:",
#elif defined(PROG_LANGUAGE_POLISH)
	    "cieka:",
#else
	    "Location:",
#endif
	    "/",
	    glist,
	    20,		/* Maximum items */
	    &combo_rtn,
	    browser,
	    EDVBrowserComboActivateCB,
	    NULL
	);
	if(glist != NULL)
	{
	    g_list_foreach(glist, (GFunc)g_free, NULL);
	    g_list_free(glist);
	    glist = NULL;
	}
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	browser->location_combo = w = (GtkWidget *)combo_rtn;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);
	GUIDNDSetSrc(
	    browser->location_icon_fixed,
	    dnd_src_types,
	    sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY | GDK_ACTION_MOVE |
	    GDK_ACTION_LINK,			/* Actions */
	    GDK_BUTTON1_MASK,			/* Buttons */
	    NULL,
	    EDVLocBarIconDragDataGet,
	    NULL,
	    NULL,
	    combo
	);
	w = combo->entry;
	EDVEntrySetDND(core, w);
	EDVEntrySetCompletePath(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* GtkHandleBox for the Mount Bar */
	browser->mount_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_attached",
	    GTK_SIGNAL_FUNC(EDVBrowserHandleChildAttachedCB), browser
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_detached",
	    GTK_SIGNAL_FUNC(EDVBrowserHandleChildDetachedCB), browser
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(browser->mount_bar_map_state)
	    gtk_widget_show(w);
	parent2 = w;

	/* Create the Mount Bar */
	browser->mount_bar = EDVMountBarNew(
	    core, parent2,
	    EDVBrowserMountBarMountCB,
	    EDVBrowserMountBarEjectCB,
	    EDVBrowserMountBarGoToCB,
	    EDVBrowserStatusMessageCB,
	    browser
	);
	EDVMountBarMap(browser->mount_bar);


	/* GtkHandleBox for the Find Bar */
	browser->find_bar_handle = w = gtk_handle_box_new();
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_attached",
	    GTK_SIGNAL_FUNC(EDVBrowserHandleChildAttachedCB), browser
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "child_detached",
	    GTK_SIGNAL_FUNC(EDVBrowserHandleChildDetachedCB), browser
	);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	if(browser->find_bar_map_state)
	    gtk_widget_show(w);
	parent2 = w;

	/* Create the Find Bar */
	browser->find_bar = EDVFindBarNew(
	    core, parent2,
	    EDV_LOCATION_TYPE_VFS,
	    EDVBrowserFindBarLocationCB,
	    EDVBrowserFindBarStartCB,
	    EDVBrowserFindBarEndCB,
	    EDVBrowserFindBarMatchCB,
	    EDVBrowserStatusMessageCB,
	    EDVBrowserStatusProgressCB,
	    browser
	);
	EDVFindBarMap(browser->find_bar);


	/* Main GtkHPaned */
	browser->main_paned = w = gtk_hpaned_new();
	gtk_paned_set_position(GTK_PANED(w), paned_position);
	gtk_paned_set_handle_size(
	    GTK_PANED(w), EDV_DEF_PANED_HANDLE_SIZE
	);
	gtk_paned_set_gutter_size(
	    GTK_PANED(w), EDV_DEF_PANED_GUTTER_SIZE
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* GtkScrolledWindow for the directory GtkCTree */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_paned_add1(GTK_PANED(parent2), w);
	gtk_widget_show(w);
	parent3 = w;

	/* Directory GtkCTree */
	browser->directory_ctree = w = gtk_ctree_new(1, 0);
	clist = GTK_CLIST(w);
	ctree = GTK_CTREE(w);
	gtk_widget_set_name(w, EDV_BROWSER_CTREE_WIDGET_NAME);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(EDVCListMotionEventCB), core
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVBrowserKeyEventCB), browser
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVBrowserKeyEventCB), browser
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVBrowserButtonPressEventCB), browser
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_realize(w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_ctree_set_line_style(ctree, GTK_CTREE_LINES_DOTTED);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_signal_connect(
	    GTK_OBJECT(w), "tree_select_row",
	    GTK_SIGNAL_FUNC(EDVBrowserTreeSelectRowCB), browser
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "tree_unselect_row",
	    GTK_SIGNAL_FUNC(EDVBrowserTreeUnselectRowCB), browser
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "tree_expand",
	    GTK_SIGNAL_FUNC(EDVBrowserTreeExpandCB), browser
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "tree_collapse",
	    GTK_SIGNAL_FUNC(EDVBrowserTreeCollapseCB), browser
	);
	/* Set up DND for the directory ctree */
	GUIDNDSetSrc(
	    w,
	    dnd_src_types,
	    sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY | GDK_ACTION_MOVE |
		GDK_ACTION_LINK,		/* Actions */
	    GDK_BUTTON1_MASK,			/* Buttons */
	    NULL,
	    EDVBrowserDirTreeDragDataGetCB,
	    EDVBrowserDirTreeDragDataDeleteCB,
	    NULL,
	    browser
	);
	GUIDNDSetTar(
	    w,
	    dnd_tar_types,
	    sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY | GDK_ACTION_MOVE |
		GDK_ACTION_LINK,		/* Actions */
	    GDK_ACTION_MOVE,			/* Default action if same */
	    GDK_ACTION_MOVE,			/* Default action */
	    EDVBrowserDirTreeDragDataReceivedCB,
	    browser
	);
	gtk_widget_show(w);


	/* Scrolled window for contents clist */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_paned_add2(GTK_PANED(parent2), w);
	gtk_widget_show(w);
	parent3 = w;

	/* Contents clist */
	browser->contents_clist = w = gtk_clist_new(EDV_BROWSER_CONTENTS_CLIST_COLUMNS_MAX);
	clist = GTK_CLIST(w);
	gtk_widget_set_name(w, EDV_BROWSER_CLIST_WIDGET_NAME);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(EDVCListMotionEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVBrowserKeyEventCB), browser
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVBrowserKeyEventCB), browser
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVBrowserButtonPressEventCB), browser
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_realize(w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	for(i = 0; i < clist->columns; i++)
	    gtk_clist_set_column_width(clist, i, 10);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_signal_connect(
	    GTK_OBJECT(w), "resize_column",
	    GTK_SIGNAL_FUNC(EDVBrowserResizeColumnCB), browser
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "click_column",
	    GTK_SIGNAL_FUNC(EDVBrowserClickColumnCB), browser
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(EDVBrowserSelectRowCB), browser
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(EDVBrowserUnselectRowCB), browser
	);
	/* Set up DND for the contents list */
	GUIDNDSetSrc(
	    w,
	    dnd_src_types,
	    sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY | GDK_ACTION_MOVE |
		GDK_ACTION_LINK,		/* Actions */
	    GDK_BUTTON1_MASK,			/* Buttons */
	    NULL,
	    EDVBrowserContentsDragDataGetCB,
	    EDVBrowserContentsDragDataDeleteCB,
	    NULL,
	    browser
	);
	GUIDNDSetTar(
	    w,
	    dnd_tar_types,
	    sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY | GDK_ACTION_MOVE |
		GDK_ACTION_LINK,		/* Actions */
	    GDK_ACTION_MOVE,			/* Default action if same */
	    GDK_ACTION_MOVE,			/* Default action */
	    EDVBrowserContentsDragDataReceivedCB,
	    browser
	);
	gtk_widget_show(w);


	/* Create Tool Bar */
	EDVBrowserToolBarRegenerate(browser);

	/* Create Directory Tree Menu */
	EDVBrowserDirectoryMenuRegenerate(browser);

	/* Create Contents List Menu */
	EDVBrowserContentsMenuRegenerate(browser);


	/* Create Status Bar */
	browser->status_bar = status_bar = EDVStatusBarNew(
	    core, browser->main_vbox
	);
	if(status_bar != NULL)
	{
	    if(browser->status_bar_map_state)
		EDVStatusBarMap(status_bar);
	    else
		EDVStatusBarUnmap(status_bar);
	}


	/* Create the New Objects submenu */
	EDVBrowserNewObjectMenuItemsRegenerate(browser);

	/* Create the Open To submenu */
	EDVBrowserOpenToMenuRegenerate(browser, NULL);


	/* Set initial RC styles */
	if(standard_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		browser->toplevel, standard_rcstyle
	    );
	if(lists_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		browser->directory_ctree, lists_rcstyle
	    );
	if(lists_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		browser->contents_clist, lists_rcstyle
	    );
	if(standard_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		browser->directory_ctree_menu, standard_rcstyle
	    );
	if(standard_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		browser->contents_clist_menu, standard_rcstyle
	    );
	EDVBrowserListStylesRegenerate(browser);

	/* Load special icons */
	if(TRUE)
	{
	    gchar *icons_dir = g_strconcat(
		EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
		G_DIR_SEPARATOR_S,
		EDV_DEF_ICONS_SUBDIR,
		NULL
	    );

#define LOAD_ICON(_f_,_p_,_m_)	{			\
 gchar *icon_path = g_strconcat(			\
  icons_dir,						\
  G_DIR_SEPARATOR_S,					\
  (_f_),						\
  NULL							\
 );							\
 *(_p_) = GDK_PIXMAP_NEW_FROM_XPM_FILE((_m_), icon_path); \
 g_free(icon_path);					\
}
	    LOAD_ICON(
		"icon_folder_parent_20x20.xpm",
		&browser->folder_parent_pixmap,
		&browser->folder_parent_mask
	    );
	    if(browser->folder_parent_pixmap == NULL)
		EDVLoadPixmap(
		    core,
		    (guint8 **)icon_folder_parent_20x20_xpm,
		    "icon_folder_parent_20x20_xpm",
		    &browser->folder_parent_pixmap,
		    &browser->folder_parent_mask
		);

#undef LOAD_ICON

	    g_free(icons_dir);
	}

	/* Set the initial values */
	EDVBrowserSetTitle(browser, NULL);
	EDVBrowserAccelkeysRegenerate(browser);
	EDVBrowserUpdateMenus(browser);

	browser->freeze_count--;

	return(browser);
}

/*
 *	Syncronizes the File Browser's widget values with other values.
 */
void EDVBrowserSyncData(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

/* This function may be removed in the future, since there really is
 * no floating data from widgets to be synced to disk on the browser
 */

}


/*
 *	Updates the File Browser's widgets to reflect current values.
 */
void EDVBrowserUpdateMenus(edv_browser_struct *browser)
{
	gboolean	write_protect,
			write_protect_changed = FALSE,
			sensitive,
			state,
			sel_dir_node_expanded = FALSE,
			sel_dir_node_has_child = FALSE;
	edv_delete_method delete_method;
	gint sel_dev_num, sel_obj_row;
	guint8 **icon_data, **recbin_icon_data;
	GtkWidget *w;
	GtkCTreeNode *sel_dir_node;
	const cfg_item_struct *cfg_list;
	toolbar_struct *tb;
	edv_device_struct *sel_dev_ptr;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	browser->freeze_count++;

	tb = browser->tool_bar;
	core = browser->core;
	cfg_list = core->cfg_list;

#define SHOW_WIDGET(_w_)			\
{ if((_w_) != NULL) gtk_widget_show(_w_); }
#define HIDE_WIDGET(_w_)			\
{ if((_w_) != NULL) gtk_widget_hide(_w_); }

	/* Get global write protect state */
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	if(browser->last_write_protect_state < 0)
	{
	    write_protect_changed = TRUE;
	}
	else
	{
	    if((write_protect && !browser->last_write_protect_state) ||
	       (!write_protect && browser->last_write_protect_state)
	    )
		write_protect_changed = TRUE;
	}
	browser->last_write_protect_state = write_protect ? 1 : 0;

	/* Delete method */
	delete_method = (edv_delete_method)EDV_GET_I(
	    EDV_CFG_PARM_DELETE_METHOD
	);

	/* Get last selected directory node and contents row, note
	 * that even not NULL or positive does not mean they are
	 * valid. Check if they actually exist on the list before
	 * referencing them
	 */
	sel_dir_node = browser->directory_ctree_selected_node;
	sel_obj_row = browser->contents_clist_selected_row;

	/* Get last selected device (if any) */
	sel_dev_num = browser->selected_dev_num;
	if((sel_dev_num >= 0) && (sel_dev_num < core->total_devices))
	    sel_dev_ptr = core->device[sel_dev_num];
	else
	    sel_dev_ptr = NULL;

	/* Update recycle bin icon data, setting it to not NULL if it
	 * needs to be changed and update the browser's
	 * last_recbin_items to match the value on the core
	 */
	icon_data = NULL;
	if(browser->last_recbin_items < 0)
	{
	    if(core->last_recbin_items > 0)
		icon_data = (guint8 **)icon_trash_20x20_xpm;
	    else
		icon_data = (guint8 **)icon_trash_empty_20x20_xpm;
	}
	else
	{
	    if(!browser->last_recbin_items &&
	       core->last_recbin_items
	    )
		icon_data = (guint8 **)icon_trash_20x20_xpm;
	    else if(browser->last_recbin_items &&
		    !core->last_recbin_items
	    )
		icon_data = (guint8 **)icon_trash_empty_20x20_xpm;
	}
	browser->last_recbin_items = core->last_recbin_items;
	recbin_icon_data = icon_data;


	/* Get selected directory node expand state (if any) */
	if(browser->directory_ctree != NULL)
	{
	    GtkCTreeNode *node;
	    GtkCTreeRow *row_ptr;
	    GtkCTree *ctree = GTK_CTREE(browser->directory_ctree);

	    node = EDVCTreeGetSelectedLast(ctree, &row_ptr);
	    if(row_ptr != NULL)
	    {
		sel_dir_node_expanded = row_ptr->expanded;
		sel_dir_node_has_child = (row_ptr->children != NULL) ?
		    TRUE : FALSE;
	    }
	}


	/* Update the mount bar */
	EDVMountBarUpdateMenus(browser->mount_bar);

	/* Update the find bar */
	EDVFindBarUpdateMenus(browser->find_bar);



	/* Begin updating menu items and tool bar items */

	/* File Menu */
	/* Open & Open With */
	sensitive = ((sel_dir_node != NULL) ||
	    (sel_obj_row > -1)) ? TRUE : FALSE;
	/* Open */
	GTK_WIDGET_SET_SENSITIVE(browser->open_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_OPEN, sensitive
	);
	/* Open To */
	GTK_WIDGET_SET_SENSITIVE(browser->open_to_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_OPEN_TO, sensitive
	);
	/* Open with */
	GTK_WIDGET_SET_SENSITIVE(browser->open_with_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_OPEN_WITH, sensitive
	);

	/* New Object Submenu Item */
	sensitive = ((sel_dir_node != NULL) ||
	    (sel_obj_row > -1)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(browser->new_object_mi, sensitive);
	/* New File */
	GTK_WIDGET_SET_SENSITIVE(browser->new_file_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_NEW_FILE, sensitive
	);
	/* New Directory */
	GTK_WIDGET_SET_SENSITIVE(browser->new_directory_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_NEW_DIRECTORY, sensitive
	);
	/* New Link */
	GTK_WIDGET_SET_SENSITIVE(browser->new_link_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_NEW_LINK, sensitive
	);
	/* New FIFO */
	GTK_WIDGET_SET_SENSITIVE(browser->new_fifo_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_NEW_FIFO, sensitive
	);
	/* New Block Device */
	GTK_WIDGET_SET_SENSITIVE(browser->new_device_block_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_NEW_DEVICE_BLOCK, sensitive
	);
	/* New Character Device */
	GTK_WIDGET_SET_SENSITIVE(browser->new_device_character_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_NEW_DEVICE_CHARACTER, sensitive
	);
	/* New Socket */
	GTK_WIDGET_SET_SENSITIVE(browser->new_socket_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_NEW_SOCKET, sensitive
	);

	/* Go To Parent */
	sensitive = TRUE;
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_GOTO_PARENT, sensitive
	);


	/* Edit Menu */
	sensitive = ((sel_dir_node != NULL) ||
	    (sel_obj_row > -1)) ? TRUE : FALSE;
	/* Move */
	GTK_WIDGET_SET_SENSITIVE(browser->move_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_MOVE, sensitive
	);
	/* Copy */
	GTK_WIDGET_SET_SENSITIVE(browser->copy_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_COPY, sensitive
	);
	/* Link */
	GTK_WIDGET_SET_SENSITIVE(browser->link_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_LINK, sensitive
	);
	/* Rename */
	GTK_WIDGET_SET_SENSITIVE(browser->rename_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_RENAME, sensitive
	);
	/* Change Permissions */
	GTK_WIDGET_SET_SENSITIVE(browser->chmod_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_CHMOD, sensitive
	);
	/* Change Ownership */
	GTK_WIDGET_SET_SENSITIVE(browser->chown_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_CHOWN, sensitive
	);
	/* Set Time Stamps */
	GTK_WIDGET_SET_SENSITIVE(browser->chtime_mi, sensitive);
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_CHTIME, sensitive
	);
	/* Delete */
	GTK_WIDGET_SET_SENSITIVE(browser->delete_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_DELETE, sensitive
	);
	/* Copy Path */
	GTK_WIDGET_SET_SENSITIVE(browser->copy_path_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_COPY_PATH, sensitive
	);
	/* Copy URL */
	GTK_WIDGET_SET_SENSITIVE(browser->copy_url_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_COPY_URL, sensitive
	);
	/* Paste */
	GTK_WIDGET_SET_SENSITIVE(browser->paste_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_PASTE, sensitive
	);
	/* Properties */
	GTK_WIDGET_SET_SENSITIVE(browser->properties_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_PROPERTIES, sensitive
	);

	/* View Menu */
	/* Show Tool Bar */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_TOOL_BAR);
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(browser->show_tool_bar_micheck, state);
	ToolBarItemSetToggleID(
	    tb, EDV_BROWSER_OP_SHOW_TOOL_BAR, state
	);
	/* Show Location Bar */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_LOCATION_BAR);
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(browser->show_location_bar_micheck, state);
	ToolBarItemSetToggleID(
	    tb, EDV_BROWSER_OP_SHOW_LOCATION_BAR, state
	);
	/* Show Mount Bar */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_MOUNT_BAR);
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(browser->show_mount_bar_micheck, state);
	ToolBarItemSetToggleID(
	    tb, EDV_BROWSER_OP_SHOW_MOUNT_BAR, state
	);
	/* Show Find Bar */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_FIND_BAR);
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(browser->show_find_bar_micheck, state);
	ToolBarItemSetToggleID(
	    tb, EDV_BROWSER_OP_SHOW_FIND_BAR, state
	);
	/* Show Status Bar */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_STATUS_BAR);
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(browser->show_status_bar_micheck, state);
	ToolBarItemSetToggleID(
	    tb, EDV_BROWSER_OP_SHOW_STATUS_BAR, state
	);
	/* Show Hidden Objects */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_OBJECT_HIDDEN);
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(browser->show_hidden_objects_micheck, state);
	ToolBarItemSetToggleID(
	    tb, EDV_BROWSER_OP_SHOW_HIDDEN_OBJECTS, state
	);
	/* Show No Access Objects */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_SHOW_OBJECT_NOACCESS);
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(browser->show_noaccess_objects_micheck, state);
	ToolBarItemSetToggleID(
	    tb, EDV_BROWSER_OP_SHOW_NOACCESS_OBJECTS, state
	);
	/* Auto Resize Columns */
	state = EDV_GET_B(EDV_CFG_PARM_BROWSER_CONTENTS_LIST_AUTO_RESIZE_COLUMNS);
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(browser->contents_list_auto_resize_columns_micheck, state);
	ToolBarItemSetToggleID(
	    tb, EDV_BROWSER_OP_CONTENTS_LIST_AUTO_RESIZE_COLUMNS, state
	);


	/* Device Menu */
	sensitive = (sel_dev_ptr != NULL) ?
	    !EDV_DEVICE_IS_NO_UNMOUNT(sel_dev_ptr) : FALSE;
	w = browser->mount_mi;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive)
	if(w != NULL)
	{
	    const gchar *text;
	    guint8 **icon_data;
	    if(sel_dev_ptr != NULL)
	    {
		if(EDV_DEVICE_IS_MOUNTED(sel_dev_ptr))
		{
		    text = "Unmount";
		    icon_data = (guint8 **)icon_unmount_20x20_xpm;
		}
		else
		{
		    text = "Mount";
		    icon_data = (guint8 **)icon_mount_20x20_xpm;
		}
	    }
	    else
	    {
		text = "Mount/Unmount";
		icon_data = (guint8 **)icon_unmount_20x20_xpm;
	    }
	    GUIMenuItemSetPixmap(w, icon_data);
	    GUIMenuItemSetLabel(w, text);
	    ToolBarItemUpdateByID(
		tb, EDV_BROWSER_OP_MOUNT, text, icon_data, NULL
	    );
	}
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_MOUNT, sensitive
	);

	/* Eject */
	if(sel_dev_ptr != NULL)
	    sensitive = !STRISEMPTY(sel_dev_ptr->command_eject);
	else
	    sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(browser->eject_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_EJECT, sensitive
	);

	/* Device check, tools, and format commands */
	if(sel_dev_ptr != NULL)
	    sensitive = !STRISEMPTY(sel_dev_ptr->command_check);
	else
	    sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(browser->device_check_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_DEVICE_CHECK, sensitive
	);

	if(sel_dev_ptr != NULL)
	    sensitive = !STRISEMPTY(sel_dev_ptr->command_tools);
	else
	    sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(browser->device_tools_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_DEVICE_TOOLS, sensitive
	);

	if(sel_dev_ptr != NULL)
	    sensitive = !STRISEMPTY(sel_dev_ptr->command_format);
	else
	    sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(browser->device_format_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    tb, EDV_BROWSER_OP_DEVICE_FORMAT, sensitive
	);


	/* Settings Menu */
	/* Write Protect */
	if(write_protect_changed)
	{
	    w = browser->write_protect_mi;
	    if(w != NULL)
		GUIMenuItemSetPixmap(
		    w,
		    (guint8 **)(write_protect ?
			icon_secure_20x20_xpm : icon_insecure_20x20_xpm)
		);
	    ToolBarItemUpdateByID(
		tb, EDV_BROWSER_OP_WRITE_PROTECT,
		NULL,
		(guint8 **)(write_protect ?
		    icon_secure_20x20_xpm : icon_insecure_20x20_xpm),
		NULL
	    );
	}

	/* Delete Method: Recycle */
	w = browser->delete_method_recycle_micheck;
	GUIMenuItemSetPixmap(w, recbin_icon_data);
	state = (delete_method == EDV_DELETE_METHOD_RECYCLE) ?
	    TRUE : FALSE;
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(w, state);
	ToolBarItemUpdateByID(
	    tb, EDV_BROWSER_OP_DELETE_METHOD_RECYCLE,
	    NULL, recbin_icon_data, NULL
	);
	ToolBarItemSetToggleID(
	    tb, EDV_BROWSER_OP_DELETE_METHOD_RECYCLE, state
	);

	/* Delete Method: Purge */
	w = browser->delete_method_purge_micheck;
	state = (delete_method == EDV_DELETE_METHOD_PURGE) ?
	    TRUE : FALSE;
	GTK_CHECK_MENU_ITEM_SET_ACTIVE(w, state);
	ToolBarItemSetToggleID(
	    tb, EDV_BROWSER_OP_DELETE_METHOD_PURGE, state
	);


	/* Windows Menu */
	/* Recycle Bin */
	if(recbin_icon_data != NULL)
	{
	    GUIMenuItemSetPixmap(
		browser->window_recycle_bin_mi, recbin_icon_data
	    );
	    ToolBarItemUpdateByID(
		tb, EDV_BROWSER_OP_RECYCLE_BIN,
		NULL, recbin_icon_data, NULL
	    );
	}


	/* Directory Tree right-click menu */
	sensitive = (sel_dir_node != NULL) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_move_mi, sensitive
	)
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_copy_mi, sensitive
	)
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_link_mi, sensitive
	)
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_rename_mi, sensitive
	)
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_chmod_mi, sensitive
	)
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_chown_mi, sensitive
	)
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_delete_mi, sensitive
	)
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_copy_path_mi, sensitive
	)
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_copy_url_mi, sensitive
	)
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_paste_mi, sensitive
	) 
	GTK_WIDGET_SET_SENSITIVE(
	    browser->directory_ctree_properties_mi, sensitive
	)

	w = browser->directory_ctree_expand_mi;
	if(sel_dir_node != NULL)
	    sensitive = sel_dir_node_has_child;
	else
	    sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive)
	if(w != NULL)
	{
	    const gchar *text;
	    if(sensitive)
		text = sel_dir_node_expanded ?
		    "Collapse" : "Expand";
	    else
		text = "Expand/Collapse";
	    GUIMenuItemSetLabel(w, text);
	}

	w = browser->directory_ctree_mount_mi;
	if(sel_dev_ptr != NULL)
	{
	    SHOW_WIDGET(w);
	}
	else
	{
	    HIDE_WIDGET(w);
	}
	sensitive = (sel_dev_ptr != NULL) ?
	    !EDV_DEVICE_IS_NO_UNMOUNT(sel_dev_ptr) : FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive)
	if(sel_dev_ptr != NULL)
	{
	    const gchar *text;
	    guint8 **icon_data;
	    if(EDV_DEVICE_IS_MOUNTED(sel_dev_ptr))
	    {
		text = "Unmount";
		icon_data = (guint8 **)icon_unmount_20x20_xpm;
	    }
	    else
	    {   
		text = "Mount";
		icon_data = (guint8 **)icon_mount_20x20_xpm;
	    }
	    GUIMenuItemSetPixmap(w, icon_data);
	    GUIMenuItemSetLabel(w, text);
	}

	w = browser->directory_ctree_mount_misep;
	if(sel_dev_ptr != NULL)
	{
	    SHOW_WIDGET(w);
	}
	else
	{
	    HIDE_WIDGET(w);
	}

	/* Eject */
	w = browser->directory_ctree_eject_mi;
	if(sel_dev_ptr != NULL)
	{
	    SHOW_WIDGET(w);
	}
	else
	{
	    HIDE_WIDGET(w);
	}
	if(sel_dev_ptr != NULL)
	    sensitive = (sel_dev_ptr->command_eject != NULL) ?
		(*sel_dev_ptr->command_eject != '\0') : FALSE;
	else
	    sensitive = FALSE;
	GTK_WIDGET_SET_SENSITIVE(w, sensitive);


	/* Contents List right-click menu */
	sensitive = (sel_obj_row > -1) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_open_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_open_to_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_open_with_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_move_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_copy_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_link_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_rename_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_chmod_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_chown_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_delete_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_copy_path_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_copy_url_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_paste_mi, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(
	    browser->contents_clist_properties_mi, sensitive
	);

	/* Update the Status Bar */
	EDVStatusBarUpdate(browser->status_bar);

#undef SHOW_WIDGET
#undef HIDE_WIDGET

	browser->freeze_count--;
}

/*
 *	Sets the File Browser as busy or ready.
 */
void EDVBrowserSetBusy(edv_browser_struct *browser, const gboolean busy)
{
	GdkCursor *cur;
	GtkWidget *w;
	edv_core_struct *core;

	if(browser == NULL)
	    return;

	w = browser->toplevel;
	core = browser->core;

	if(busy)
	{
	    browser->busy_count++;
	    if(browser->busy_count > 1)
		return;
	    cur = EDVGetCursor(core, EDV_CURSOR_CODE_BUSY);
	}
	else
	{
	    browser->busy_count--;
	    if(browser->busy_count < 0)
		browser->busy_count = 0;
	    if(browser->busy_count > 0)
		return;
	    cur = NULL;			/* Use default cursor */
	}

	if(w->window != NULL)
	{
	    gdk_window_set_cursor(w->window, cur);
	    gdk_flush();
	}
}

/*
 *	Checks if the File Browser is mapped.
 */
gboolean EDVBrowserIsMapped(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(browser->toplevel));
}

/*
 *	Maps the File Browser.
 */
void EDVBrowserMap(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	gtk_widget_show_raise(browser->toplevel);
}

/*
 *	Unmaps the File Browser.
 */
void EDVBrowserUnmap(edv_browser_struct *browser)
{
	if(browser == NULL)
	    return;

	gtk_widget_hide(browser->toplevel);
}

/*
 *	Deletes the File Browser.
 */
void EDVBrowserDelete(edv_browser_struct *browser)
{
	gint i;

	if(browser == NULL)
	    return;

	/* Clear the Contents List */
	EDVBrowserContentsClear(browser);

	/* Clear the Directory Tree */
	EDVBrowserDirTreeClear(browser);

	/* Remove the idle & timeout callbacks */
	GTK_IDLE_REMOVE(browser->directory_ctree_expand_optimize_idleid);
	browser->directory_ctree_expand_optimize_idleid = 0;

	/* Reset the last selected item references */
	browser->directory_ctree_selected_node = NULL;
	browser->contents_clist_selected_row = -1;


	EDVBrowserUnmap(browser);

	browser->freeze_count++;

	GTK_WIDGET_DESTROY(browser->directory_ctree_menu);
	GTK_WIDGET_DESTROY(browser->contents_clist_open_to_menu);
	GTK_WIDGET_DESTROY(browser->contents_clist_menu);

	EDVStatusBarDelete(browser->status_bar);
	EDVFindBarDelete(browser->find_bar);
	EDVMountBarDelete(browser->mount_bar);
	ToolBarDelete(browser->tool_bar);

	GTK_WIDGET_DESTROY(browser->new_object_menu);
	GTK_WIDGET_DESTROY(browser->open_to_menu);

	GTK_WIDGET_DESTROY(browser->toplevel);

	/* Unref all cell styles */
	for(i = 0; i < EDV_BROWSER_TOTAL_CELL_STYLES; i++)
	    GTK_STYLE_UNREF(browser->cell_style[i]);

	/* Unref special icons */
	GDK_PIXMAP_UNREF(browser->folder_parent_pixmap);
	GDK_BITMAP_UNREF(browser->folder_parent_mask);

	/* Operation IDs */
	EDVBrowserOPIDListDelete(browser->opid_list);

	g_free(browser->directory_ctree_origin_path);
	g_free(browser->contents_list_filter);

	browser->freeze_count--;

	g_free(browser);
}
