#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <glib.h>

#include "../include/string.h"

#include "edv_types.h"
#include "edv_date.h"
#include "config.h"


const gchar *EDVDateFormatString(
	const gulong t,
	const gchar *format,
	const edv_date_relativity relativity
);
const gchar *EDVDateStringDuration(gulong dt);

void EDVDateParseDateDMY(
	const gchar *s,
	gint *day, gint *month, gint *year
);
void EDVDateParseDateMDY(
	const gchar *s,
	gint *month, gint *day, gint *year
);
void EDVDateParseDateYMD(
	const gchar *s,
	gint *year, gint *month, gint *day
);
void EDVDateParseDate(
	const gchar *s,
	gint *year, gint *month, gint *day
);
void EDVDateParseTime(
	const gchar *s,
	gint *hour, gint *minutes, gint *seconds
);
gulong EDVDateParseEPOCH(const gchar *s);


#define MONTH_NAMES_CONICAL	{		\
 "Jan", "Feb", "Mar", "Apr", "May", "Jun",	\
 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"	\
}

#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Format a date and time string.
 *
 *	The t specifies the time since EPOCH in seconds. If t is 0
 *	then an empty string will be returned.
 *
 *	The format specifies the date and time format string. If format
 *	is NULL then ctime() will be used to format the date and time
 *	string.
 *
 *	The relativity can be either EDV_DATE_RELATIVITY_ABSOLUTE or
 *	EDV_DATE_RELATIVITY_CURRENT. If the relativity is
 *	EDV_DATE_RELATIVITY_CURRENT then the format will be
 *	"<n> <month(s)/day(s)/hour(s)/minute(s)/second(s)> ago",
 *	except if t >= 6 months then EDV_DATE_RELATIVITY_ABSOLUTE
 *	will be used as the relativity.
 *
 *	Returns a statically allocated string describing the date and
 *	time, the returned string must not be modified or deleted.
 *
 *	This function never returns NULL.
 *
 *	Reminder: There is a sister function in lib/edv_date.c which
 *	needs to perform equvilently to this function.
 */
const gchar *EDVDateFormatString(
	const gulong t,
	const gchar *format,
	const edv_date_relativity relativity
)
{
	static gchar buf[80];

	/* Handle by relativity */
	switch(relativity)
	{
	  case EDV_DATE_RELATIVITY_CURRENT:
	    if(t > 0l)
	    {
		const gulong dt = (gulong)time(NULL) - t;
		gulong ct;

		/* Less than one second? */
		if(dt < 1l)
		{
		    strncpy(
			(char *)buf,
			"less than a second ago",
			sizeof(buf)
		    );
		    buf[sizeof(buf) - 1] = '\0';
		}
		/* Less than one minute? */
		else if(dt < (1l * 60l))
		{
		    ct = MAX((dt / 1l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "second" : "seconds")
		    );
		}
		/* Less than one hour? */
		else if(dt < (60l * 60l))
		{
		    ct = MAX((dt / 60l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "minute" : "minutes")
		    );
		}
		/* Less than one day? */
		else if(dt < (24l * 60l * 60l))
		{
		    ct = MAX((dt / 60l / 60l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "hour" : "hours")
		    );
		}
		/* Less than one week? */
		else if(dt < (7l * 24l * 60l * 60l))
		{
		    ct = MAX((dt / 60l / 60l / 24l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "day" : "days")
		    );
		}
		/* Less than one month (30 days)? */
		else if(dt < (30l * 24l * 60l * 60l))
		{
		    ct = MAX((dt / 60l / 60l / 24l / 7l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "week" : "weeks")
		    );
		}
		/* Less than 6 months ago? */
		else if(dt < (6l * 30l * 24l * 60l * 60l))
		{
		    ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
		    g_snprintf(
			buf, sizeof(buf),
			"%ld %s ago",
			ct, ((ct == 1l) ? "month" : "months")
		    );
		}
		/* 6 months or longer ago */
		else
		{
		    /* Recurse to return a date and time string
		     * describing absolute relativity
		     */
		    return(EDVDateFormatString(
			t,
			format,
			EDV_DATE_RELATIVITY_ABSOLUTE
		    ));
		}

		return(buf);
	    }
	    break;

	  case EDV_DATE_RELATIVITY_ABSOLUTE:
	    /* Date and time format string specified? */
	    if(format != NULL)
	    {
		time_t tv = (time_t)t;
		const struct tm *tm_ptr;

		*buf = '\0';
		tm_ptr = localtime(&tv);

		if(tm_ptr != NULL)
		    strftime(
			(char *)buf, sizeof(buf),
			(const char *)format, tm_ptr
		    );
		buf[sizeof(buf) - 1] = '\0';

		return(buf);
	    }
	    else
	    {
		/* No format string specified so use ctime() */
		time_t t2 = (time_t)t;
		return((const gchar *)ctime(&t2));
	    }
	    break;
	}

	return("");
}

/*
 *	Formats a date and time string describing the time lapsed in
 *	the format of "<n1> <units1> <n2> <units2>".
 *
 *	The dt specifies the time lapsed in seconds. If dt is 0 then
 *	"less than a second" will be returned.
 *
 *	Returns a statically allocated string describing the
 *	time lapsed.
 *
 *	Reminder: There is a sister function in lib/edv_date.c which
 *	needs to perform equvilently to this function.
 */
const gchar *EDVDateStringDuration(const gulong dt)
{
	gulong ct, ct2;
	static gchar buf[80];

	/* Less than one second? */
	if(dt < 1l)
	{
	    g_snprintf(
		buf, sizeof(buf),
		"less than a second"
	    );
	}
	/* Less than one minute? */
	else if(dt < (1l * 60l))
	{
	    ct = MAX((dt / 1l), 1l);
	    g_snprintf(
		buf, sizeof(buf),
		"%ld %s",
		ct, ((ct == 1l) ? "second" : "seconds")
	    );
	}
	/* Less than one hour? */
	else if(dt < (60l * 60l))
	{
	    ct = MAX((dt / 60l), 1l);
	    ct2 = MAX((dt / 1l), 1l) % 60l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "minute" : "minutes"),
		    ct2, ((ct2 == 1l) ? "second" : "seconds")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "minute" : "minutes") 
		);
	}
	/* Less than one day? */
	else if(dt < (24l * 60l * 60l))
	{
	    ct = MAX((dt / 60l / 60l), 1l);
	    ct2 = MAX((dt / 60l), 1l) % 60l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "hour" : "hours"),
		    ct2, ((ct2 == 1l) ? "minute" : "minutes")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "hour" : "hours")
		);
	}
	/* Less than one week? */
	else if(dt < (7l * 24l * 60l * 60l))
	{
	    ct = MAX((dt / 60l / 60l / 24l), 1l);
	    ct2 = MAX((dt / 60l / 60l), 1l) % 24l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "day" : "days"),
		    ct2, ((ct2 == 1l) ? "hour" : "hours")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "day" : "days") 
		);
	}
	/* Less than one month (30 days)? */
	else if(dt < (30l * 24l * 60l * 60l))
	{
	    ct = MAX((dt / 60l / 60l / 24l / 7l), 1l);
	    ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 7l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "week" : "weeks"),
		    ct2, ((ct2 == 1l) ? "day" : "days")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "week" : "weeks")
		);
	}
#if 0
	/* Less than 6 months ago? */
	else if(dt < (6l * 30l * 24l * 60l * 60l))
	{
	    ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
	    ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 30l;
	    g_snprintf(
		buf, sizeof(buf),
		"%ld %s %ld %s",
		ct, ((ct == 1l) ? "month" : "months"),
		ct2, ((ct2 == 1l) ? "day" : "days")
	    );
	}
#endif
	/* Less than a year (365 days)? */
	else if(dt < (12l * 30l * 24l * 60l * 60l))
	{
	    ct = MAX((dt / 60l / 60l / 24l / 30l), 1l);
	    ct2 = MAX((dt / 60l / 60l / 24l), 1l) % 30l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "month" : "months"),
		    ct2, ((ct2 == 1l) ? "day" : "days")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "month" : "months")
		);
	}
	/* A year or longer */
	else
	{
	    ct = MAX((dt / 60l / 60l / 24l / 30l / 12l), 1l);
	    ct2 = MAX((dt / 60l / 60l / 24l / 30l), 1l) % 12l;
	    if(ct2 != 0l)
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s %ld %s",
		    ct, ((ct == 1l) ? "year" : "years"),
		    ct2, ((ct2 == 1l) ? "month" : "months")
		);
	    else
		g_snprintf(
		    buf, sizeof(buf),
		    "%ld %s",
		    ct, ((ct == 1l) ? "year" : "years") 
		);
	}

	return(buf);
}


/*
 *	Parses the date string.
 *
 *	The s specifies the date string which must be in one of the
 *	following formats:
 *
 *	"D/M/Y"
 *	"D-M-Y"
 *	"D,M,Y"
 *	"D.M.Y"
 *	"D M Y"
 *
 *	The month M may be either the month's number (1 to 12) or the
 *	month's name.
 *
 *	The year Y must be a 4-digit year.
 *
 *	The day specifies the return value for day in the range of 1
 *	to 32.
 *
 *	The mon specifies the return value for the month in the range
 *	of 1 to 12.
 *
 *	The year specifies the return value for the year.
 */
void EDVDateParseDateDMY(
	const gchar *s,
	gint *day, gint *month, gint *year
)
{
#define ISDATEDELIM(c)	(		\
 ((c) == '/') || ((c) == '-') ||	\
 ((c) == ',') || ((c) == '.') ||	\
 ((c) == ':') || ((c) == ';') ||	\
 ((c) == ' ') || ((c) == '\t')		\
)

	if(STRISEMPTY(s))
	{
	    if(day != NULL)
		*day = 0;
	    if(month != NULL)
		*month = 0;
	    if(year != NULL)
		*year = 0;
	    return;
	}

	/* Parse day */
	while(ISBLANK(*s))
	    s++;
	if(day != NULL)
	    *day = CLIP(ATOI(s), 1, 32);

	/* Parse month */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(isalpha(*s))
	{
	    gint i;
	    const gchar *month_name[] = MONTH_NAMES_CONICAL;

	    for(i = 0; i < 12; i++)
	    {
		if(strcasepfx((const char *)s, (const char *)month_name[i]))
		    break;
	    }
	    if(i >= 12)
		i = 0;

	    if(month != NULL)
		*month = i + 1;
	}
	else
	{
	    if(month != NULL)
		*month = CLIP(ATOI(s), 1, 12);
	}

	/* Parse year */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(year != NULL)
	    *year = ATOI(s);
#undef ISDATEDELIM
}

/*
 *	Same as EDVDateParseDateDMY() except that the order is parsed
 *	month, day (of month), (4-digit) year
 */
void EDVDateParseDateMDY(
	const gchar *s,
	gint *month, gint *day, gint *year
)
{
#define ISDATEDELIM(c)	(		\
 ((c) == '/') || ((c) == '-') ||	\
 ((c) == ',') || ((c) == '.') ||	\
 ((c) == ':') || ((c) == ';') ||	\
 ((c) == ' ') || ((c) == '\t')		\
)

	if(STRISEMPTY(s))
	{
	    if(day != NULL)
		*day = 0;
	    if(month != NULL)
		*month = 0;
	    if(year != NULL)
		*year = 0;
	    return;
	}

	/* Parse month */
	while(ISBLANK(*s))
	    s++;
	if(isalpha(*s))
	{
	    gint i;
	    const gchar *month_name[] = MONTH_NAMES_CONICAL;

	    for(i = 0; i < 12; i++)
	    {
		if(strcasepfx((const char *)s, (const char *)month_name[i]))
		    break;
	    }
	    if(i >= 12)
		i = 0;

	    if(month != NULL)
		*month = i + 1;
	}
	else
	{
	    if(month != NULL)
		*month = CLIP(ATOI(s), 1, 12);
	}

	/* Parse day */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(day != NULL)
	    *day = CLIP(ATOI(s), 1, 32);

	/* Parse year */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(year != NULL)
	    *year = ATOI(s);
#undef ISDATEDELIM
}

/*
 *	Same as EDVDateParseDateDMY() except that the order is parsed
 *	as (4-digit) year, month, day (of month).
 */
void EDVDateParseDateYMD(
	const gchar *s,
	gint *year, gint *month, gint *day
)
{
#define ISDATEDELIM(c)	(		\
 ((c) == '/') || ((c) == '-') ||	\
 ((c) == ',') || ((c) == '.') ||	\
 ((c) == ':') || ((c) == ';') ||	\
 ((c) == ' ') || ((c) == '\t')		\
)

	if(STRISEMPTY(s))
	{
	    if(day != NULL)
		*day = 0;
	    if(month != NULL)
		*month = 0;
	    if(year != NULL)
		*year = 0;
	    return;
	}

	/* Parse year */
	while(ISBLANK(*s))
	    s++;
	if(year != NULL)
	    *year = ATOI(s);

	/* Parse month */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(isalpha(*s))
	{
	    gint i;
	    const gchar *month_name[] = MONTH_NAMES_CONICAL;

	    for(i = 0; i < 12; i++)
	    {
		if(strcasepfx((const char *)s, (const char *)month_name[i]))
		    break;
	    }
	    if(i >= 12)
		i = 0;

	    if(month != NULL)
		*month = i + 1;
	}
	else
	{
	    if(month != NULL)
		*month = CLIP(ATOI(s), 1, 12);
	}

	/* Parse day */
	for(; *s != '\0'; s++)
	{
	    if(ISDATEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	if(day != NULL)
	    *day = CLIP(ATOI(s), 1, 32);
#undef ISDATEDELIM
}

/*
 *	Parses the date string in any format.
 *
 *	Calls one of EDVDateParseDate*() by determining the date
 *	format of the date string.
 *
 *	The s specifies the date string.
 */
void EDVDateParseDate(
	const gchar *s,
	gint *year, gint *month, gint *day
)
{
	const gchar *s2;

	if(day != NULL)
	    *day = 0;
	if(month != NULL)
	    *month = 0;
	if(year != NULL)
	    *year = 0;

	if(STRISEMPTY(s))
	    return;

	while(ISBLANK(*s))
	    s++;

	/* M D Y
	 *
	 * Check if the first value is a month's name
	 */
	if(isalpha(*s))
	{
	    EDVDateParseDateMDY(s, month, day, year);
	    return;
	}

	/* Y M D
	 *
	 * Check if the first value is a 4-digit year
	 */
	s2 = (const gchar *)strpbrk(s, " \t/-,.:;");
	if(s2 == NULL)
	    return;
	if((s2 - s) >= 4)
	{
	    EDVDateParseDateYMD(s, year, month, day);
	    return;
	}

	/* D M Y */
	EDVDateParseDateDMY(s, day, month, year);
}

/*
 *	Parses the time string.
 *
 *	The s specifies the time string which must be in one of the
 *	following formats:
 *
 *      "H:M[:S][AM|PM]"
 *	"H M[ S][AM|PM]"
 *
 *	To indicate 12-hour time a "AM" or "PM" may be postfixed to
 *	the time string.
 *
 */
void EDVDateParseTime(
	const gchar *s,
	gint *hour, gint *minutes, gint *seconds
)
{
#define ISTIMEDELIM(c)	(		\
 ((c) == '/') || ((c) == '-') ||	\
 ((c) == ',') || ((c) == '.') ||	\
 ((c) == ':') || ((c) == ';') ||	\
 ((c) == ' ') || ((c) == '\t')		\
)
	const gchar *s_start = s;

	if(STRISEMPTY(s))
	{
	    if(hour != NULL)
		*hour = 0;
	    if(minutes != NULL)
		*minutes = 0;
	    if(seconds != NULL)
		*seconds = 0;
	    return;
	}

	/* Parse hour */
	while(ISBLANK(*s))
	    s++;
	/* Seek past the leading 0 */
	if(*s == '0')
	{
	    if(isdigit(s[1]))
		s++;
	}
	if(hour != NULL)
	    *hour = CLIP(ATOI(s), 0, 23);

	/* Parse minutes */
	for(; *s != '\0'; s++)
	{
	    if(ISTIMEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	/* Seek past leading 0 */
	if(*s == '0')
	{
	    if(isdigit(s[1]))
		s++;
	}
	if(minutes != NULL)
	    *minutes = CLIP(ATOI(s), 0, 59);

	/* Parse seconds */
	for(; *s != '\0'; s++)
	{
	    if(ISTIMEDELIM(*s))
	    {
		s++;
		break;
	    }
	}
	while(ISBLANK(*s))
	    s++;
	/* Seek past leading 0 */
	if(*s == '0')
	{
	    if(isdigit(s[1]))
		s++;
	}
	if(seconds != NULL)
	    *seconds = CLIP(ATOI(s), 0, 59);

	/* Is there a PM postfix? */
	while(isdigit(*s))
	    s++;

	if((strchr((const char *)s_start, 'P') != NULL) ||
	   (strchr((const char *)s_start, 'p') != NULL)
	)
	{
	    /* There is a PM postfix, add 12 hours */
	    if(hour != NULL)
		*hour += 12;
	}
#undef ISTIMEDELIM
}


/*
 *	Parses the date and time string in any format.
 *
 *	The s specifies the date and time string.
 *
 *	Returns the number of seconds since EPOCH.
 */
gulong EDVDateParseEPOCH(const gchar *s)
{
	struct tm t;
	const gchar *s2;
	gint	hour = 0, minutes = 0, seconds = 0,
		year = 0, month = 0, day = 0;

	if(STRISEMPTY(s))
	    return(0l);

	/* Check if the time was specified */
	s2 = (const gchar *)strchr((const char *)s, ':');
	if(s2 != NULL)
	{
	    /* The time was specified */
	    gchar *s_time;

	    /* Seek s2 to the start of the time */
	    if((s2 - s) >= 2)
		s2 -= 2;
	    else if((s2 - s) >= 1)
		s2 -= 1;
	    while(ISBLANK(*s2))
		s2++;

	    /* Get/copy just the time string from s2 */
	    s_time = STRDUP(s2);
	    if(s_time != NULL)
	    {
		/* Null terminate the time string */
		gchar *s3 = (gchar *)strpbrk((char *)s_time, " \t");
		if(s3 != NULL)
		    *s3 = '\0';

		/* Parse the time */
		EDVDateParseTime(s_time, &hour, &minutes, &seconds);
		g_free(s_time);
	    }

	    /* Was the time at the beginning? */
	    if(s2 == s)
	    {
		/* Seek past the time */
		while(*s2 != '\0')
		{
		    if(ISBLANK(*s2))
			break;

		    s2++;
		}

		/* Parse the date */
		EDVDateParseDate(s2, &year, &month, &day);
	    }
	    else
	    {
		/* Time was at the end */
		EDVDateParseDate(s, &year, &month, &day);
	    }
	}
	else
	{
	    /* No time, parse only the date */
	    EDVDateParseDate(s, &year, &month, &day);
	}

	t.tm_sec = seconds;
	t.tm_min = minutes;
	t.tm_hour = hour;
	t.tm_mday = day;
	t.tm_mon = month - 1;
	t.tm_year = year - 1900;
	t.tm_wday = 0;
	t.tm_yday = 0;
	t.tm_isdst = -1;

	return((gulong)mktime(&t));
}
