#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "guirgbimg.h"

#include "edv_types.h"
#include "edv_mime_type.h"
#include "edv_utils_gtk.h"


/* MIME Type Commands */
const gchar *EDVMimeTypeGetCommandValueByName(
	edv_mime_type_struct *m, const gchar *name
);
edv_mime_type_command_struct *EDVMimeTypeGetCommandByName(
	edv_mime_type_struct *m, const gchar *name
);

edv_mime_type_command_struct *EDVMimeTypeCommandNew(void);
edv_mime_type_command_struct *EDVMimeTypeCommandCopy(
	edv_mime_type_command_struct *cmd
);
void EDVMimeTypeCommandDelete(edv_mime_type_command_struct *cmd);


/* MIME Types */
edv_mime_type_struct *EDVMimeTypeNew(void);
edv_mime_type_struct *EDVMimeTypeCopy(
	edv_mime_type_struct *m
);
static void EDVMimeTypeLoadIconsNexus(
	GdkPixmap **pixmap, GdkBitmap **mask,
	guint8 ***data, const gchar **paths_list,
	const gint req_width, const gint req_height,
	const gboolean allow_resize
);
void EDVMimeTypeLoadSmallIconsData(
	edv_mime_type_struct *m, guint8 ***data
);
void EDVMimeTypeLoadMediumIconsData(
	edv_mime_type_struct *m, guint8 ***data
);
void EDVMimeTypeLoadLargeIconsData(
	edv_mime_type_struct *m, guint8 ***data
);

void EDVMimeTypeRealize(
	edv_mime_type_struct *m, const gboolean force_rerealize
);

void EDVMimeTypeDelete(edv_mime_type_struct *m);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the command value of the MIME Type command that matches
 *	the specified command name.
 */
const gchar *EDVMimeTypeGetCommandValueByName(
	edv_mime_type_struct *m, const gchar *name
)
{
	GList *glist;
	edv_mime_type_command_struct *cmd;

	if((m == NULL) || STRISEMPTY(name))
	    return(NULL);

	for(glist = m->commands_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    cmd = EDV_MIME_TYPE_COMMAND(glist->data);
	    if(cmd == NULL)
		continue;

	    if(cmd->name == NULL)
		continue;

	    if(!g_strcasecmp(cmd->name, name))
		return(cmd->command);
	}

	return(NULL);
}

/*
 *	Gets the MIME Type command that matches the specified command
 *	name.
 */
edv_mime_type_command_struct *EDVMimeTypeGetCommandByName(
	edv_mime_type_struct *m, const gchar *name
)
{
	GList *glist;
	edv_mime_type_command_struct *cmd;

	if((m == NULL) || STRISEMPTY(name))
	    return(NULL);

	for(glist = m->commands_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    cmd = EDV_MIME_TYPE_COMMAND(glist->data);
	    if(cmd == NULL)
		continue;

	    if(cmd->name == NULL)
		continue;

	    if(!g_strcasecmp(cmd->name, name))
		return(cmd);
	}

	return(NULL);
}


/*
 *	Creates a new MIME Type command.
 */
edv_mime_type_command_struct *EDVMimeTypeCommandNew(void)
{
	return(EDV_MIME_TYPE_COMMAND(g_malloc0(
	    sizeof(edv_mime_type_command_struct)
	)));
}

/*
 *	Coppies the MIME Type command.
 */
edv_mime_type_command_struct *EDVMimeTypeCommandCopy(
	edv_mime_type_command_struct *cmd
)
{
	edv_mime_type_command_struct	*tar,
					*src = cmd;
	if(src == NULL)
	    return(NULL);

	tar = EDVMimeTypeCommandNew();
	if(tar == NULL)
	    return(NULL);

	tar->name = STRDUP(src->name);
	tar->command = STRDUP(src->command);

	return(tar);
}

/*
 *	Deletes the MIME Type command.
 */
void EDVMimeTypeCommandDelete(edv_mime_type_command_struct *cmd)
{
	if(cmd == NULL)
	    return;

	g_free(cmd->name);
	g_free(cmd->command);
	g_free(cmd);
}


/*
 *	Creates a new MIME Type.
 */
edv_mime_type_struct *EDVMimeTypeNew(void)
{
	return(EDV_MIME_TYPE(g_malloc0(sizeof(edv_mime_type_struct))));
}

/*
 *	Coppies the MIME Type.
 */
edv_mime_type_struct *EDVMimeTypeCopy(
	edv_mime_type_struct *m
)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	gint i;
	GList *glist;
	edv_mime_type_struct	*tar,
				*src = m;
	if(src == NULL)
	    return(NULL);

	tar = EDVMimeTypeNew();
	if(tar == NULL)
	    return(NULL);

	tar->mt_class = src->mt_class;
	tar->value = STRDUP(src->value);
	tar->type = STRDUP(src->type);
	tar->description = STRDUP(src->description);

	tar->realized = src->realized;
	tar->read_only = src->read_only;

	for(i = 0; i < nicon_states; i++)
	{
	    /* Small */
	    GDK_PIXMAP_REF(src->small_pixmap[i]);
	    tar->small_pixmap[i] = src->small_pixmap[i];

	    GDK_BITMAP_REF(src->small_mask[i]);
	    tar->small_mask[i] = src->small_mask[i];

	    tar->small_icon_file[i] = STRDUP(src->small_icon_file[i]);

	    /* Medium */
	    GDK_PIXMAP_REF(src->medium_pixmap[i]);
	    tar->medium_pixmap[i] = src->medium_pixmap[i];

	    GDK_BITMAP_REF(src->medium_mask[i]);
	    tar->medium_mask[i] = src->medium_mask[i];

	    tar->medium_icon_file[i] = STRDUP(src->medium_icon_file[i]);

	    /* Large */
	    GDK_PIXMAP_REF(src->large_pixmap[i]);
	    tar->large_pixmap[i] = src->large_pixmap[i];

	    GDK_BITMAP_REF(src->large_mask[i]);
	    tar->large_mask[i] = src->large_mask[i];

	    tar->large_icon_file[i] = STRDUP(src->large_icon_file[i]);
	}

	tar->handler = src->handler;

	tar->commands_list = NULL;
	for(glist = src->commands_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	    tar->commands_list = g_list_append(
		tar->commands_list,
		EDVMimeTypeCommandCopy(EDV_MIME_TYPE_COMMAND(
		    glist->data
		))
	    );

	tar->access_time = src->access_time;
	tar->modify_time = src->modify_time;
	tar->change_time = src->change_time;

	return(tar);
}

/*
 *	Loads the pixmap and mask pairs list.
 *
 *	The pixmap and mask specifies the pixmap and mask pairs list,
 *	any existing (non NULL) pixmap and mask pairs in this list will
 *	be unref'ed.
 *
 *	The data specifies the XPM data. If data is not NULL then the
 *	pixmap and mask pair will be loaded from the XPM data.
 *
 *	The file specifies the path to the XPM file. If file is not NULL
 *	then the pixmap and mask pair will be loaded from the XPM file.
 *
 *	If both data and file are NULL then the existing pixmap and
 *	mask pair will be unref'ed and no new pixmap and mask pair will
 *	be loaded.
 *
 *	The req_width and req_height specifies the requested size in
 *	pixels.
 *
 *	If allow_resize is TRHE then the pixmap and mask pair will be
 *	resized to the requested size as needed.
 */
static void EDVMimeTypeLoadIconsNexus(
	GdkPixmap **pixmap, GdkBitmap **mask,
	guint8 ***data, const gchar **paths_list,
	const gint req_width, const gint req_height,
	const gboolean allow_resize
)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	gint i;

	/* Unload the current icon pixmaps and masks on the MIME
	 * type first
	 */
	for(i = 0; i < nicon_states; i++)
	{
	    GDK_PIXMAP_UNREF(pixmap[i]);
	    pixmap[i] = NULL;
	    GDK_BITMAP_UNREF(mask[i]);
	    mask[i] = NULL;
	}

	/* Load the icons if the XPM datas are specified */
	if(data != NULL)
	{
	    GdkBitmap *m;
	    GdkPixmap *p;

	    /* Load the icons */
	    for(i = 0; i < nicon_states; i++)
	    {
		p = GDK_PIXMAP_NEW_FROM_XPM_DATA(&m, data[i]);
		if(p == NULL)
		    continue;

		if(allow_resize)
		{
		    EDVResizePixmap(
			p, m,
			req_width, req_height,
			&pixmap[i], &mask[i]
		    );
		    GDK_PIXMAP_UNREF(p);
		    GDK_BITMAP_UNREF(m);
		}
		else
		{
		    pixmap[i] = p;
		    mask[i] = m;
		}
	    }
	}
	/* Load the icons if the XPM files are specified */
	else if(paths_list != NULL)
	{
	    GdkBitmap *m;
	    GdkPixmap *p;

	    /* Load the icons */
	    for(i = 0; i < nicon_states; i++)
	    {
		p = GDK_PIXMAP_NEW_FROM_XPM_FILE(&m, paths_list[i]);
		if(p == NULL)
		    continue;

		if(allow_resize)
		{
		    EDVResizePixmap(
			p, m,
			req_width, req_height,
			&pixmap[i], &mask[i]
		    );
		    GDK_PIXMAP_UNREF(p);
		    GDK_BITMAP_UNREF(m);
		}
		else
		{
		    pixmap[i] = p;
		    mask[i] = m;
		}
	    }
	}
}

/*
 *	Loads the MIME Type's small sized icons.
 *
 *	The given icon data must be an array of
 *	EDV_MIME_TYPE_TOTAL_ICON_STATES guint8 ** pointers.
 *
 *	Passing data as NULL will only unref the existing icons (if any)
 *	and not load new ones.
 */
void EDVMimeTypeLoadSmallIconsData(
	edv_mime_type_struct *m, guint8 ***data
)
{
	if(m == NULL)
	    return;

	EDVMimeTypeLoadIconsNexus(
	    m->small_pixmap, m->small_mask,
	    data, NULL,
	    20, 20, TRUE
	);
}

/*
 *	Loads the MIME Type's medium sized icons.
 *
 *	The given icon data must be an array of
 *	EDV_MIME_TYPE_TOTAL_ICON_STATES guint8 ** pointers.
 *
 *	Passing data as NULL will only unref the existing icons (if any)
 *	and not load new ones.
 */
void EDVMimeTypeLoadMediumIconsData(
	edv_mime_type_struct *m, guint8 ***data
)
{
	if(m == NULL)
	    return;

	EDVMimeTypeLoadIconsNexus(
	    m->medium_pixmap, m->medium_mask,
	    data, NULL,
	    32, 32, TRUE
	);
}

/*
 *	Loads the MIME Type's large sized icons.
 *
 *	The given icon data must be an array of
 *	EDV_MIME_TYPE_TOTAL_ICON_STATES guint8 ** pointers.
 *
 *	Passing data as NULL will only unref the existing icons (if any)
 *	and not load new ones.
 */
void EDVMimeTypeLoadLargeIconsData(
	edv_mime_type_struct *m, guint8 ***data
)
{
	if(m == NULL)
	    return;

	EDVMimeTypeLoadIconsNexus(
	    m->large_pixmap, m->large_mask,
	    data, NULL,
	    48, 48, TRUE
	);
}


/*
 *	Realizes the MIME Type, loading the icons and related resources.
 *
 *	If the MIME Type is already realized (member realized is TRUE)
 *	then nothing will be done, however if force_rerealize is TRUE
 *	then the above check will be ignored.
 */
void EDVMimeTypeRealize(
	edv_mime_type_struct *m, const gboolean force_rerealize
)
{
	const gchar **paths_list;

	if(m == NULL)
	    return;

	/* Not forcing a re-realize? */
	if(!force_rerealize)
	{
	    /* MIME Type already realized? */
	    if(m->realized)
		return;
	}

	/* Begin realizing this MIME Type
	 *
	 * Load the icons
	 *
	 * Get the paths list for the small sized icons and load them
	 */
	paths_list = (const gchar **)m->small_icon_file;
	if(paths_list != NULL)
	    EDVMimeTypeLoadIconsNexus(
		m->small_pixmap, m->small_mask,
		NULL, paths_list,
		20, 20, TRUE
	    );

	/* Get the paths list for the medium sized icons and load them */
	paths_list = (const gchar **)m->medium_icon_file;
	if(paths_list != NULL)
	    EDVMimeTypeLoadIconsNexus(
		m->medium_pixmap, m->medium_mask,
		NULL, paths_list,
		32, 32, TRUE
	    );

	/* Get the paths list for the large sized icons and load them */
	paths_list = (const gchar **)m->large_icon_file;
	if(paths_list != NULL)
	    EDVMimeTypeLoadIconsNexus(
		m->large_pixmap, m->large_mask,
		NULL, paths_list,
		48, 48, TRUE
	    );

	m->realized = TRUE;		/* Mark as realized */
}

/*
 *	Deletes the MIME Type.
 */
void EDVMimeTypeDelete(edv_mime_type_struct *m)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	gint i;

	if(m == NULL)
	    return;

	/* Call the icon loaders with NULL as the argument to unload
	 * the icons
	 */
	EDVMimeTypeLoadSmallIconsData(m, NULL);
	EDVMimeTypeLoadMediumIconsData(m, NULL);
	EDVMimeTypeLoadLargeIconsData(m, NULL);

	/* Delete the icon file paths */
	for(i = 0; i < nicon_states; i++)
	{
	    g_free(m->small_icon_file[i]);
	    g_free(m->medium_icon_file[i]);
	    g_free(m->large_icon_file[i]);
	}

	/* Delete the commands */
	if(m->commands_list != NULL)
	{
	    g_list_foreach(
		m->commands_list, (GFunc)EDVMimeTypeCommandDelete, NULL
	    );
	    g_list_free(m->commands_list);
	}

	g_free(m->value);
	g_free(m->type);
	g_free(m->description);

	g_free(m);
}
