#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "cfg.h"
#include "edv_recycled_obj.h"
#include "edv_recbin_index.h"
#include "edv_recbin_delete.h"
#include "endeavour2.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "animations/file01_20x20.xpm"
#include "animations/file02_20x20.xpm"
#include "animations/file03_20x20.xpm"
#include "animations/file04_20x20.xpm"
#include "animations/file05_20x20.xpm"
#include "animations/file06_20x20.xpm"
#include "animations/folder_32x32.xpm"
#include "animations/folderfile_32x32.xpm"
#include "animations/trash01_20x20.xpm"
#include "animations/trash02_20x20.xpm"
#include "animations/trash03_20x20.xpm"
#include "animations/trash04_20x20.xpm"
#include "animations/trash05_20x20.xpm"
#include "animations/trash06_20x20.xpm"

#include "images/icon_trash_32x32.xpm"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


const gchar *EDVRecBinDeleteObjectGetError(edv_core_struct *core);
static void EDVRecBinDeleteCopyErrorMessage(
	edv_core_struct *core, const gchar *msg
);

static gboolean EDVRecBinNoDeleteCheck(
	edv_core_struct *core,
	GtkWidget *toplevel,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gchar *path,
	struct stat *stat_buf
);

static void EDVRecBinDeleteMapProgressDialog(
	const gchar *label,
	const gfloat progress_value,
	GtkWidget *toplevel,
	const gboolean force_remap,
	const edv_delete_method delete_method
);
static gint EDVRecBinDeleteProgressCB(
	gpointer data, const gulong pos, const gulong total
);

static guint EDVRecBinDeleteObjectIterate(
	edv_core_struct *core,
	const gchar *recycled_index_file,
	const gchar *path,
	GList **index_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *status,
	const edv_delete_method delete_method
);
gint EDVRecBinDeleteObject(
	edv_core_struct *core,
	const gchar *path,
	GList **index_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *      Returns a statically allocated string describing the last
 *	error message (or NULL if there was no error) since the last
 *	call to a EDVRecBinFOP*() function.
 */
const gchar *EDVRecBinDeleteObjectGetError(edv_core_struct *core)
{
	return((core != NULL) ? core->recbin_last_error : NULL);
}

/*
 *	Coppies the error message to the core's recbin_last_error_buf
 *	and sets the core's recbin_last_error to point to it.
 */
static void EDVRecBinDeleteCopyErrorMessage(
	edv_core_struct *core, const gchar *msg
)
{
	if(core == NULL)
	    return;

	core->recbin_last_error = NULL;

	g_free(core->recbin_last_error_buf);
	core->recbin_last_error_buf = STRDUP(msg);

	core->recbin_last_error = core->recbin_last_error_buf;
}


/*
 *	Checks if the object specified by path and stat_buf may not be
 *	deleted for some reason.
 *
 *	Returns TRUE if the object may not be deleted.
 */
static gboolean EDVRecBinNoDeleteCheck(
	edv_core_struct *core,
	GtkWidget *toplevel,
	const gboolean interactive,
	gboolean *yes_to_all,
	const gchar *path,
	struct stat *stat_buf
)
{
typedef struct {
	gchar		*path;
	gchar		*message;
	gint		error_level;
} no_delete_struct;
	gint i;
	gboolean no_delete;
	const gchar *s;
	gchar *parent_path;
	const cfg_item_struct *cfg_list;
	no_delete_struct *nd_ptr = NULL;
	no_delete_struct no_delete_list[] = EDV_NO_DELETE_LIST;

	/* Check for valid inputs, return TRUE if they are invalid to
	 * prevent calling operation from proceeding
	 */
	if((core == NULL) || STRISEMPTY(path))
	    return(TRUE);

	cfg_list = core->cfg_list;

	/* Iterate through no delete list, checking if an entry has a
	 * path that matches the given path
	 */
	i = 0;
	no_delete = FALSE;
	while(no_delete_list[i].path != NULL)
	{
	    nd_ptr = &no_delete_list[i];

	    /* Given object path matches one in the no delete list? */
	    if(!strcmp((const char *)nd_ptr->path, (const char *)path))
	    {
		/* Handle by error level of this no delete entry */
		switch(nd_ptr->error_level)
		{
		  case 0:	/* Only warn user */
		    if(interactive)
		    {


		    }
		    break;

		  case 1:	/* Query user for no delete */
		    if(interactive && !(*yes_to_all))
		    {
			gint status;
			gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El \"%s\" objetivo no se debe borrar, la razn:\n\
\n\
%s.\n\
\n\
Usted est seguro que usted quiere borrar este objeto?"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le \"%s\" d'objet ne devrait pas tre effac, la raison:\n\
\n\
%s.\n\
\n\
Etes-vous sr que vous voulez effacer cet objet? "
#elif defined(PROG_LANGUAGE_GERMAN)
"Das Objekt \"%s\" sollte, Grund nicht gelscht werden:\n\
\n\
%s.\n\
\n\
Sind Sie sicher Sie dieses Objekt wollen lschen?"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lo \"%s\" di oggetto non dovrebbe essere cancellato, la ragione:\n\
\n\
%s.\n\
\n\
Lei sono sicuro che lei vuole cancellare quest'oggetto?"
#elif defined(PROG_LANGUAGE_DUTCH)
"Het voorwerp \"%s\" zou, reden niet moeten geschrappet worden:\n\
\n\
%s.\n\
\n\
Bent u zeker u deze voorwerp wil schrappen?"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O \"%s\" de objeto nao deve ser anulado, razo:\n\
\n\
%s.\n\
\n\
Esto seguro quer anular este objeto?"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Objektet \"%s\" stryker ikke, grunn:\n\
\n\
%s.\n\
\n\
Er De sikker De stryker dette objektet?"
#else
"The object \"%s\" should not be deleted, reason:\n\
\n\
%s.\n\
\n\
Are you sure you want to delete this object?"
#endif
			    , path,
			    (nd_ptr->message != NULL) ?
				nd_ptr->message :
				"Object is marked \"no_delete\" by this process"
			);
			EDVPlaySoundQuestion(core);
			CDialogSetTransientFor(toplevel);
			status = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Borre Advertir"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer L'Avertissement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Warnen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare L'Avvertimento"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap Waarschuwen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule Aviso"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk Warning"
#else
"Delete Warning"
#endif
			    , buf, NULL,
			    CDIALOG_ICON_WARNING,
			    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			    CDIALOG_BTNFLAG_NO
			);
			CDialogSetTransientFor(NULL);
			g_free(buf);

			switch(status)
			{
			  case CDIALOG_RESPONSE_YES_TO_ALL:
			    *yes_to_all = TRUE;
			  case CDIALOG_RESPONSE_YES:
			    break;

			  default:
			    no_delete = TRUE;
			    break;
			}
		    }
		    else
		    {
			no_delete = TRUE;
		    }
		    break;

		  default:	/* No delete under any case */
		    no_delete = TRUE;
		    if(interactive)
		    {
			gchar *buf = g_strdup_printf(
"Unable to delete object \"%s\".\n\
\n\
%s.\n",
			    path,
			    (nd_ptr->message != NULL) ?
				nd_ptr->message :
				"Object is marked \"no_delete\" by this process"
			);
			EDVPlaySoundError(core);
			EDVMessageError(
#if defined(PROG_LANGUAGE_SPANISH)
"Borre Fallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap Geverzuimenene"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule Fracassado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk Failed"
#else
"Delete Failed"
#endif
			    , buf, NULL,
			    toplevel
			);
			g_free(buf);
		    }
		    break;
		}
		/* Stop iterating through no delete list and return
		 * response
		 */
		return(no_delete);
	    }

	    i++;
	}


	/* Check for other paths that are defined at run time and may
	 * not be deleted
	 */

	/* Special notations? */
	s = g_basename(path);
	if(!STRISEMPTY(s))
	{
	    /* Current or parent directory? */
	    if(!strcmp((const char *)s, ".") ||
	       !strcmp((const char *)s, "..")
	    )
	    {
		if(interactive)
		{
		    gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Las anotaciones objetivas especiales tal como \"%s\"\n\
no se estn acostumbrados a reffer a un objeto que\n\
deber ser borrado."
#elif defined(PROG_LANGUAGE_FRENCH)
"Les notations spciales d'objet telles que \"%s\"\n\
ne peut pas tre utilis  reffer  un objet qui\n\
va tre effac."
#elif defined(PROG_LANGUAGE_GERMAN)
"Besondere Objekt Aufzeichnungen kann wie zum\n\
Beispiel \"%s\" zu reffer zu einem Objekt nicht benutzt\n\
werden, das gelscht werden soll."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Le notazioni di oggetto speciali come \"%s\" non\n\
possono essere usate al reffer a un oggetto che \n\
essere cancellato."
#elif defined(PROG_LANGUAGE_DUTCH)
"Speciale voorwerp aantekeningen zal zoals \"%s\"\n\
te reffer aan een voorwerp niet misschien gebruikt\n\
worden dat geschrappet worden zal."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anotaes especiais de objeto tal como \"%s\"\n\
nao pode ser usado a reffer a um objeto que  ser\n\
anulado."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Spesiell objektnotasjoner slik som \"%s\" ikke\n\
brukt til reffer til et objekt stryker som."
#else
"Special object notations such as \"%s\" may not be\n\
used to reffer to an object that is to be deleted."
#endif
			, s
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
#if defined(PROG_LANGUAGE_SPANISH)
"Borre Fallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap Geverzuimenene"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule Fracassado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk Failed"
#else
"Delete Failed"
#endif
			, buf, NULL,
			toplevel
		    );
		    g_free(buf);
		}
		return(TRUE);
	    }
	}

	/* Home directory? */
	s = core->home_dir;
	if(!STRISEMPTY(s) ?
	    !strcmp((const char *)s, (const char *)path) : FALSE
	)
	{
	    if(interactive)
	    {
		gchar *buf = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El \"%s\" objetivo no se puede borrar, la razn:\n\
\n\
La gua buscadora del usuario."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le \"%s\" d'objet ne peut pas tre effac, la raison:\n\
\n\
L'annuaire de maison de l'utilisateur."
#elif defined(PROG_LANGUAGE_GERMAN)
"Das Objekt \"%s\" kann, Grund nicht gelscht werden:\n\
\n\
Das Heim von Verbraucher Verzeichnis."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lo \"%s\" di oggetto non pu essere cancellato, la ragione:\n\
\n\
L'elenco di casa dell'operatore."
#elif defined(PROG_LANGUAGE_DUTCH)
"Het voorwerp \"%s\" kan niet, reden is geschrappet:\n\
\n\
Het huis van gebruiker gids."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O \"%s\" de objeto nao pode ser anulado, razo:\n\
\n\
O guia de lar do operador."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Objektet \"%s\" strket ikke, grunn:\n\
\n\
Brukers hjemkatalog."
#else
"Unable to delete object \"%s\".\n\
\n\
Object is the user's home directory.\n"
#endif
		    , path
		);
		EDVPlaySoundError(core);
		EDVMessageError(
#if defined(PROG_LANGUAGE_SPANISH)
"Borre Fallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap Geverzuimenene"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule Fracassado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk Failed"
#else
"Delete Failed"
#endif
		    , buf, NULL,
		    toplevel
		);
		g_free(buf);
	    }
	    return(TRUE);
	}

	/* Is the object part of the recycled objects system?
	 *
	 * Check this by getting the parent of the recycled objects
	 * index file, this parent directory will be checked to see if
	 * it is the parent of the specified object
	 *
	 * If it is then we know that the object is in within the
	 * recycled objects directory and therefore is a part of the
	 * recycled objects system (which may not be deleted)
	 */
	s = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	parent_path = !STRISEMPTY(s) ? g_dirname(s) : NULL;
	if(!STRISEMPTY(parent_path) ? EDVIsParentPath(parent_path, path) : FALSE)
	{
	    if(interactive)
	    {
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El \"%s\" objetivo no se puede borrar, la razn:\n\
\n\
El objeto forma parte del sistema de objetos de\n\
recycled (purga de uso en lugar)."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le \"%s\" d'objet ne peut pas tre effac, la raison:\n\
\n\
L'objet fait partie du systme d'objets recycl\n\
(la purge d'usage plutt)."
#elif defined(PROG_LANGUAGE_GERMAN)
"Das Objekt \"%s\" kann, Grund nicht gelscht werden:\n\
\n\
Objekt ist teil des wiederverwerteten objekte systems\b\
(gebrauch reinigt anstatt)."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lo \"%s\" di oggetto non pu essere cancellato, la ragione:\n\
\n\
L'oggetto  la parte del sistema di oggetti riciclato\n\
(la purga di uso invece)."
#elif defined(PROG_LANGUAGE_DUTCH)
"Het voorwerp \"%s\" kan niet, reden is geschrappet:\n\
\n\
Voorwerp is deel van de gerecyclde voorwerpen systeem\n\
(gebruik zuivering in plaats van)."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O \"%s\" de objeto nao pode ser anulado, razo:\n\
\n\
O objeto  parte do sistema de objetos de recycled\n\
(purge de uso contrriamente)."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Objektet \"%s\" strket ikke, grunn:\n\
\n\
Objekt er del av det resirkuleredde objektsystem renser\n\
(bruk istedenfor)."
#else
"Unable to delete object \"%s\".\n\
\n\
Object is part of the recycled objects system.\n"
#endif
		    , path
		);
		EDVPlaySoundError(core);
		EDVMessageError(
#if defined(PROG_LANGUAGE_SPANISH)
"Borre Fallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen Sie Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrap Geverzuimenene"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anule Fracassado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryk Failed"
#else
"Delete Failed"
#endif
		    , msg, NULL,
		    toplevel
		);
		g_free(msg);
	    }

	    g_free(parent_path);

	    return(TRUE);
	}

	g_free(parent_path);

	return(FALSE);
}


/*
 *      Maps the progress dialog as needed in animation mode for deleting.
 */
static void EDVRecBinDeleteMapProgressDialog(
	const gchar *label,
	const gfloat progress_value,
	GtkWidget *toplevel,
	const gboolean force_remap,
	const edv_delete_method delete_method
)
{
	guint8	**start_icon_data[3],
		**icon_data[6],
		**end_icon_data[3];

	/* Already mapped? */
	if(ProgressDialogIsQuery())
	{
	    /* Check if the progress dialog needs to be unmapped and
	     * remapped again.
	     */
	    if(force_remap)
	    {
		ProgressDialogBreakQuery(FALSE);
	    }
	    else
	    {
		/* Already mapped and does not need unmapping, so just
		 * update the progress message.
		 */
		ProgressDialogUpdate(
		    NULL, label, NULL, NULL,
		    progress_value, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
		return;
	    }
	}

	ProgressDialogSetTransientFor(toplevel);

	if(delete_method == EDV_DELETE_METHOD_PURGE)
	{
	    start_icon_data[0] = (guint8 **)folderfile_32x32_xpm;
	    start_icon_data[1] = (guint8 **)folder_32x32_xpm;
	    start_icon_data[2] = (guint8 **)folder_32x32_xpm;
	    icon_data[0] = (guint8 **)trash01_20x20_xpm;
	    icon_data[1] = (guint8 **)trash02_20x20_xpm;
	    icon_data[2] = (guint8 **)trash03_20x20_xpm;
	    icon_data[3] = (guint8 **)trash04_20x20_xpm;
	    icon_data[4] = (guint8 **)trash05_20x20_xpm;
	    icon_data[5] = (guint8 **)trash06_20x20_xpm;
	    end_icon_data[0] = (guint8 **)NULL;
	    end_icon_data[1] = (guint8 **)NULL;
	    end_icon_data[2] = (guint8 **)NULL;
	}
	else
	{
	    start_icon_data[0] = (guint8 **)folderfile_32x32_xpm;
	    start_icon_data[1] = (guint8 **)folder_32x32_xpm;
	    start_icon_data[2] = (guint8 **)folder_32x32_xpm;
	    icon_data[0] = (guint8 **)file01_20x20_xpm;
	    icon_data[1] = (guint8 **)file02_20x20_xpm;
	    icon_data[2] = (guint8 **)file03_20x20_xpm;
	    icon_data[3] = (guint8 **)file04_20x20_xpm;
	    icon_data[4] = (guint8 **)file05_20x20_xpm;
	    icon_data[5] = (guint8 **)file06_20x20_xpm;
	    end_icon_data[0] = (guint8 **)icon_trash_32x32_xpm;
	    end_icon_data[1] = (guint8 **)icon_trash_32x32_xpm;
	    end_icon_data[2] = (guint8 **)icon_trash_32x32_xpm;
	}

	ProgressDialogMapAnimation(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Borrar",
	    label,
	    "Parada",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Effacer",
	    label,
	    "Arrt",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Lschen",
	    label,
	    "Halt",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Cancellare",
	    label,
	    "Fermata",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Schrappen",
	    label,
	    "Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Anular",
	    label,
	    "Parada",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Stryking",
	    label,
	    "Stans",
#else
	    "Deleting",
	    label,
	    "Stop",
#endif
	    start_icon_data, 3,
	    icon_data, 6,
	    end_icon_data, 3,
	    EDV_DEF_PROGRESS_DLG_ANIM_INT,
	    EDV_DEF_PROGRESS_DLG_ANIM_INC
	);
	ProgressDialogUpdate(
	    NULL, NULL, NULL, NULL,
	    progress_value, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	);

	/* Flush output so dialog gets mapped and we catch the beginning
	 * of the operation (some WM need this)
	 */
	gdk_flush();
}

/*
 *	Recycle recover or delete progress callback.
 */
static gint EDVRecBinDeleteProgressCB(
	gpointer data, const gulong pos, const gulong total
)
{
	gint status = 0;
/*	edv_core_struct *core = EDV_CORE(data); */

	if(ProgressDialogIsQuery())
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		((total > 0) && (pos >= 0)) ?
		    ((gfloat)pos / (gfloat)total) : -1.0f,
		EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -1;
	}

	return(status);
}


/*
 *	Called by EDVRecBinDeleteObject() to delete the object specified by
 *	path to the recycled objects directory.
 *
 *	Returns the index of the recycled object or 0 if no recycled
 *	object index is available, this can occure if the delete
 *	method is set to purge.
 */
static guint EDVRecBinDeleteObjectIterate(
	edv_core_struct *core,
	const gchar *recycled_index_file,
	const gchar *path,
	GList **index_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all,
	gint *status,
	const edv_delete_method delete_method
)
{
	gboolean no_delete;
	guint index;
	gint strc = 0;
	gchar **strv;
	struct stat lstat_buf, stat_buf;

	/* Error or user aborted from prior call? */
	if(*status != 0)
	    return(0);

	/* Get the local stats of the object to be deleted */
	if(lstat((const char *)path, &lstat_buf))
	{
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
		g_strerror(error_code), path
	    );
	    EDVRecBinDeleteCopyErrorMessage(core, msg);
	    g_free(msg);
	    *status = -1;
	    return(0);
	}

#if 0
/* Let the system determine this */
	/* If the effective user ID of this process does not own
	 * object then the object unable to be deleted
	 */
	if(!EDVCheckIsOwner(core, lstat_buf.st_uid))
	{
	    *status = -1;
	    core->recbin_last_error =
"Effective user id of this process does not own the object.";
	    return(0);
	}
#endif

	/* Get destination statistics of this object and check if it
	 * may not be deleted
	 */
	if(stat((const char *)path, &stat_buf))
	    no_delete = EDVRecBinNoDeleteCheck(
		core, toplevel, interactive, yes_to_all, path, NULL
	    );
	else
	    no_delete = EDVRecBinNoDeleteCheck(
		core, toplevel, interactive, yes_to_all, path, &stat_buf
	    );
	if(no_delete)
	{
	    *status = -1;
	    /* Set last error only if not interactive since calling
	     * EDVRecBinNoDeleteCheck() would have already displayed 
	     * appropriate error message
	     */
	    if(!interactive)
	    {
		gchar *msg = g_strdup_printf(
"The object:\n\
\n\
    %s\n\
\n\
Is part of the function of this process or system\n\
and may not be deleted.",
		    path
		);
		EDVRecBinDeleteCopyErrorMessage(core, msg);
		g_free(msg);
	    }
	    return(0);
	}


	/* If the specified object to be deleted is locally a directory,
	 * then get contents of the directory and delete recursively
	 */
#ifdef S_ISDIR
	strv = S_ISDIR(lstat_buf.st_mode) ?
	    GetDirEntNames2(path, &strc) : NULL;
#else
	strv = NULL;
#endif
	if(strv != NULL)
	{
	    gint i;
	    gchar *child_path, *name;

	    /* Sort strings */
	    strv = StringQSort(strv, strc);
	    if(strv == NULL)
	    {
		*status = -1;
		core->recbin_last_error = "Unable to sort directory entries.";
		return(0);
	    }

	    /* Iterate through directory entry */
	    child_path = NULL;
	    for(i = 0; i < strc; i++)
	    {
#define FREE_AND_CONTINUE	{	\
 g_free(child_path);			\
 child_path = NULL;			\
 g_free(strv[i]);			\
 strv[i] = NULL;			\
 continue;				\
}
		name = strv[i];
		if(name == NULL)
		    FREE_AND_CONTINUE

		/* Error encountered or user abort? */
		if(*status != 0)
		    FREE_AND_CONTINUE

		/* Skip special dir notations */
		if(!strcmp((const char *)name, "..") ||
		   !strcmp((const char *)name, ".")
		)
		    FREE_AND_CONTINUE

		/* Formulate full path to child */
		child_path = STRDUP(PrefixPaths(
		    (const char *)path, (const char *)name
		));
		if(child_path == NULL)
		    FREE_AND_CONTINUE

		/* Delete this object and all its children */
		index = EDVRecBinDeleteObjectIterate(
		    core,
		    recycled_index_file,
		    child_path,
		    index_list_rtn,
		    toplevel,
		    show_progress,
		    interactive,
		    yes_to_all,
		    status,
		    delete_method
		);
		/* Above return index is ignored */

		FREE_AND_CONTINUE
#undef FREE_AND_CONTINUE
	    }

	    /* Delete pointer array */
	    g_free(strv);
	}

	/* Error or user aborted while deleting any child objects? */
	if(*status != 0)
	    return(0);

	/* Map the progress dialog? */
	if(show_progress)
	{
	    gchar *p1 = EDVShortenPath(
		path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
	    );
	    gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Borrar:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Effacer:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Lschen:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cancellare:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schrappen:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Anular:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stryking:\n\
\n\
    %s\n"
#else
"Deleting:\n\
\n\
    %s\n"
#endif
		, p1
	    );

	    EDVRecBinDeleteMapProgressDialog(
		msg, 0.0f, toplevel, FALSE, delete_method
	    );

	    g_free(msg);
	    g_free(p1);

	    if(ProgressDialogStopCount() > 0)
	    {
		*status = -4;
		return(0);
	    }
	}

	/* At this point, check if we need to delete the object by
	 * moving it to the recycled objects directory or permanently
	 * purging the object
	 */
	if(delete_method == EDV_DELETE_METHOD_PURGE)
	{
	    /* Permanently purge this object */
	    const mode_t m = lstat_buf.st_mode;

	    /* Delete by object type */
	    /* Directory? */
#ifdef S_ISDIR
	    if(S_ISDIR(m))
#else
	    if(FALSE)
#endif
	    {
		if(rmdir((const char *)path))
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to delete the directory:\n\
\n\
    %s\n\
\n\
%s.",
			path, g_strerror(error_code)
		    );
		    EDVRecBinDeleteCopyErrorMessage(core, msg);
		    g_free(msg);
		    *status = -1;
		}
	    }
	    /* All else, assume it can be unlink'ed */
	    else
	    {
		if(unlink((const char *)path))
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to delete:\n\
\n\
    %s\n\
\n\
%s.",
			path, g_strerror(error_code)
		    );
		    EDVRecBinDeleteCopyErrorMessage(core, msg);
		    g_free(msg);
		    *status = -1;
	        }
	    }

	    index = 0;

	    if(*status == 0)
	    {
		if(index_list_rtn != NULL)
		    *index_list_rtn = g_list_append(
			*index_list_rtn, (gpointer)index
		    );
	    }
	}
	else
	{
	    /* All else, transfer this object to the recycled objects
	     * directory
	     */
	    gchar *parent_path;
	    gulong cur_time = time(NULL);
	    const mode_t m = lstat_buf.st_mode;

	    /* Create a new recycled object */
	    edv_recycled_object_struct *obj = EDVRecycledObjectNew();
	    if(obj == NULL)
		return(0);

	    /* Get this object's location */
	    parent_path = g_dirname(path);
	    if(parent_path == NULL)
		parent_path = STRDUP("/");

	    /* Set the values from this object to the new recycled object */
	    obj->name = STRDUP(g_basename(path));
	    obj->original_path = STRDUP(parent_path);
	    obj->deleted_time = cur_time;

	    obj->type = EDVStatModeToObjectType(m);
	    obj->size = (gulong)lstat_buf.st_size;
#ifdef S_ISLNK
	    if(S_ISLNK(m))
		obj->link_target = EDVGetLinkTarget(path);
#endif
	    obj->permissions = EDVStatModeToEDVPermissions(m);
	    obj->access_time = (gulong)lstat_buf.st_atime;
	    obj->modify_time = (gulong)lstat_buf.st_mtime;
	    obj->change_time = (gulong)lstat_buf.st_ctime;
	    obj->owner_id = (gint)lstat_buf.st_uid;
	    obj->group_id = (gint)lstat_buf.st_gid;

	    /* Add an entry to the recycled objects index file */
	    index = EDVRecBinIndexAdd(recycled_index_file, obj);
	    if(index != 0)
	    {
		/* Recycle this object */
		*status = EDVRecBinDiskObjectDelete(
		    recycled_index_file,
		    index,
		    path,
		    show_progress ? EDVRecBinDeleteProgressCB : NULL,
		    core
		);
		/* User aborted operation? */
		if(*status == -4)
		{
		    EDVRecBinIndexRemove(recycled_index_file, index);
		    index = 0;
		}
		/* Other error occured? */
		else if(*status != 0)
		{
		    gchar *msg = g_strdup_printf(
"Unable to delete:\n\
\n\
    %s\n\
\n\
%s.",
			path, EDVRecBinIndexGetError()
		    );
		    EDVRecBinDeleteCopyErrorMessage(core, msg);
		    g_free(msg);
		    EDVRecBinIndexRemove(recycled_index_file, index);
		    index = 0;
		}
		/* Successfully deleted the object */
		else
		{
		    /* Append this index to the index list return */
		    if(index_list_rtn != NULL)
			*index_list_rtn = g_list_append(
			    *index_list_rtn, (gpointer)index
			);
		}
	    }
	    else
	    {
		core->recbin_last_error =
"Unable to generate a new recycled object index.";
		*status = -1;
	    }

	    /* Delete the recycled object */
	    EDVRecycledObjectDelete(obj);

	    g_free(parent_path);
	}

	return(index);
}

/*
 *	Deletes the object.
 *
 *	If the global configuration specifies purge on delete then
 *	the object will be deleted instead of being put in to the
 *	recycle bin, otherwise the object will be put into the recycle
 *	bin.
 *
 *	The path specifies the full path to the object to delete.
 *
 *	The index_list_rtn specifies a pointer to a GList * that will
 *	be set to a list of guint indices describing all of the
 *	objects deleted. If index_list_rtn is NULL then no list will
 *	be returned. The calling function must delete the returned
 *	list.
 *
 *	The toplevel specifies the toplevel GtkWidget.
 *
 *	If show_progress is TRUE then the progress dialog will be
 *	shown during this operation.
 *
 *	If interactive is TRUE then the user will be notified and
 *	queried if any problems arise.
 *
 *	The yes_to_all specifies a pointer to the preinitialized
 *	current gboolean "yes to all" value. This value may be
 *	set to TRUE during this call if the user responds with
 *	"yes to all" to a query.
 */
gint EDVRecBinDeleteObject(
	edv_core_struct *core,
	const gchar *path,
	GList **index_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress,
	const gboolean interactive,
	gboolean *yes_to_all
)
{
	static gint status;
	guint index;
	const gchar *recycled_index_file;
	edv_delete_method delete_method;
	gulong time_start = (gulong)time(NULL);
	gchar *lsrc_obj = NULL;
	const cfg_item_struct *cfg_list;

	/* Clear the last error message */
	EDVRecBinDeleteCopyErrorMessage(core, NULL);

	if(index_list_rtn != NULL)
	    *index_list_rtn = NULL;

	if((core == NULL) || STRISEMPTY(path))
	{
	    EDVRecBinDeleteCopyErrorMessage(core, "Invalid value.");
	    return(-1);
	}

#define CLEANUP_RETURN(_v_)	{	\
 g_free(lsrc_obj);			\
					\
 return(_v_);				\
}

	cfg_list = core->cfg_list;

	/* Get path to recycled objects index file */
	recycled_index_file = EDV_GET_S(
	    EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX
	);
	if(recycled_index_file == NULL)
	{
	    core->recbin_last_error = "Unable to get the recycled objects index file.";
	    CLEANUP_RETURN(-1);
	}

	/* Get delete method, so we know if we should just move the
	 * object to the recycled objects directory or permanently
	 * purge it
	 */
	delete_method = (edv_delete_method)EDV_GET_I(
	    EDV_CFG_PARM_DELETE_METHOD
	);


	/* Get copy of path of object to be deleted */
	lsrc_obj = STRDUP(path);
	if(lsrc_obj == NULL)
	{
	    core->recbin_last_error = "Memory allocation error.";
	    CLEANUP_RETURN(-3);
	}


	status = 0;

	/* Delete this object */
	index = EDVRecBinDeleteObjectIterate(
	    core,
	    recycled_index_file,
	    lsrc_obj,
	    index_list_rtn,
	    toplevel,
	    show_progress,
	    interactive,
	    yes_to_all,
	    &status,
	    delete_method
	);

	/* Record history */
	EDVAppendHistory(
	    core,
	    EDV_HISTORY_VFS_OBJECT_DELETE,
	    time_start,
	    (gulong)time(NULL),
	    status,
	    lsrc_obj,			/* Source */
	    NULL,			/* No target */
	    core->recbin_last_error	/* Comments */
	);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
