#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/disk.h"

#include "guiutils.h"
#include "pulist.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_list_cb.h"
#include "edv_date.h"
#include "edv_obj.h"
#include "edv_recycled_obj.h"
#include "edv_recbin_stat.h"
#include "edv_status_bar.h"
#include "find_win.h"
#include "find_win_cb.h"
#include "endeavour2.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_search_20x20.xpm"
#include "images/icon_open_20x20.xpm"
#include "images/icon_stop_20x20.xpm"
#include "images/icon_browse_20x20.xpm"
#include "images/icon_clear_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_goto_20x20.xpm"

#include "images/icon_search_48x48.xpm"


const gchar *EDVFindWinCurrentSearch(edv_find_win_struct *fw);
void EDVFindWinSetSearch(
	edv_find_win_struct *fw,
	const gchar *s,
	const gboolean record_history
);

const gchar *EDVFindWinCurrentLocation(edv_find_win_struct *fw);
void EDVFindWinSetLocation(
	edv_find_win_struct *fw,
	const gchar *path,
	const gboolean record_history
);

edv_find_win_find_by EDVFindWinCurrentFindBy(edv_find_win_struct *fw); 
void EDVFindWinSetReferenceWindow(
	edv_find_win_struct *fw,
	const gint browser_num,
	const gint imbr_num,
	const gint recbin_num,
	const gint archiver_num
);

void EDVFindWinListResetColumns(
	edv_find_win_struct *fw,
	const edv_find_win_find_by find_by
);
void EDVFindWinListAppend(
	edv_find_win_struct *fw,
	const gchar *path, struct stat *lstat_buf,
	const gchar *excerpt, const gint line_index
);
void EDVFindWinListClear(edv_find_win_struct *fw);

void EDVFindWinSyncConfiguration(edv_find_win_struct *fw);

edv_find_win_struct *EDVFindWinNew(edv_core_struct *core);
void EDVFindWinUpdateMenus(edv_find_win_struct *fw);
void EDVFindWinSetBusy(edv_find_win_struct *fw, const gboolean busy);
gboolean EDVFindWinIsMapped(edv_find_win_struct *fw);
void EDVFindWinMap(edv_find_win_struct *fw);
void EDVFindWinUnmap(edv_find_win_struct *fw);
void EDVFindWinDelete(edv_find_win_struct *fw);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the current search string.
 */
const gchar *EDVFindWinCurrentSearch(edv_find_win_struct *fw)
{
	GtkCombo *combo;

	if(fw == NULL)
	    return(NULL);

	combo = GTK_COMBO(fw->search_combo);

	return(gtk_entry_get_text(GTK_ENTRY(combo->entry)));
}

/*
 *	Sets the search string.
 *
 *	The s specifies the new search string.
 *
 *	If record_history is TRUE then the new search string will be
 *	recorded on the search history.
 */
void EDVFindWinSetSearch(
	edv_find_win_struct *fw,
	const gchar *s,
	const gboolean record_history
)
{
	gchar *new_s;
	GtkCombo *combo;

	if((fw == NULL) || STRISEMPTY(s))
	    return;

	combo = GTK_COMBO(fw->search_combo);

	/* Make a copy of the new search string */
	new_s = STRDUP(s);
	if(new_s == NULL)
	    return;

#if 0
/* This does not work out if we need to record history */
	/* Check for no change in value */
	s = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if((s != NULL) ? !strcmp(s, new_s) : FALSE)
	{
	    g_free(new_s);
	    return;
	}
#endif

	/* Record history? */
	if(record_history)
	    GUIComboAddItem(GTK_WIDGET(combo), new_s);

	/* Set the new search string */
	gtk_entry_set_text(GTK_ENTRY(combo->entry), new_s);

	g_free(new_s);
}

/*
 *	Gets the current location.
 */
const gchar *EDVFindWinCurrentLocation(edv_find_win_struct *fw)
{
	GtkCombo *combo;

	if(fw == NULL)
	    return(NULL);

	combo = GTK_COMBO(fw->location_combo);

	return(gtk_entry_get_text(GTK_ENTRY(combo->entry)));
}

/*
 *	Sets the location.
 *
 *	The path specifies the new location.
 *
 *	If record_history is TRUE then the new location will be
 *	recorded in the locations history.
 */
void EDVFindWinSetLocation(
	edv_find_win_struct *fw,
	const gchar *path,
	const gboolean record_history
)
{
	gchar *new_path;
	GtkCombo *combo;

	if((fw == NULL) || STRISEMPTY(path))
	    return;

	combo = GTK_COMBO(fw->location_combo);

	/* Make a copy of the new location */
	new_path = STRDUP(path);
	if(new_path == NULL)
	    return;

	/* Simplify the path */
	SimplifyPath(new_path);

#if 0
/* This does not work out if we need to record history */
	/* Check for no change in value */
	path = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	if((path != NULL) ? !strcmp(path, new_path) : FALSE)
	{
	    g_free(new_path);
	    return;
	}
#endif

	/* Record history? */
	if(record_history)
	    GUIComboAddItem(GTK_WIDGET(combo), new_path);

	/* Set the new location */
	gtk_entry_set_text(GTK_ENTRY(combo->entry), new_path);

	g_free(new_path);
}

/*
 *	Gets the current operation.
 */
edv_find_win_find_by EDVFindWinCurrentFindBy(edv_find_win_struct *fw)
{
	pulist_struct *pulist;

	if(fw == NULL)
	    return(EDV_FIND_WIN_FIND_BY_NAME);

	pulist = PUListBoxGetPUList(fw->find_by_pulistbox);
	if(pulist == NULL)
	    return(EDV_FIND_WIN_FIND_BY_NAME);

	return((edv_find_win_find_by)PUListGetItemData(
	    pulist,
	    PUListGetSelectedLast(pulist)
	));
}

/*
 *	Sets the reference window and location type.
 *
 *	Only one of the window indices may be non-negative.
 */
void EDVFindWinSetReferenceWindow(
	edv_find_win_struct *fw,
	const gint browser_num,
	const gint imbr_num,
	const gint recbin_num,
	const gint archiver_num
)
{
	edv_location_type location_type;

	if(fw == NULL)
	    return;

	fw->browser_num = browser_num;
	fw->imbr_num = imbr_num;
	fw->recbin_num = recbin_num;
	fw->archiver_num = archiver_num;

	/* Set the location type based on which reference window was
	 * specified
	 */
	if(browser_num > -1)
	    location_type = EDV_LOCATION_TYPE_VFS;
	else if(imbr_num > -1)
	    location_type = EDV_LOCATION_TYPE_VFS;
	else if(recbin_num > -1)
	    location_type = EDV_LOCATION_TYPE_RECBIN;
	else if(archiver_num > -1)
	    location_type = EDV_LOCATION_TYPE_ARCHIVE;
	else
	    location_type = EDV_LOCATION_TYPE_VFS;
	fw->location_type = location_type;

	/* Update the results list title and DND source types due to
	 * change in the location type
	 */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    gtk_window_set_title(
		GTK_WINDOW(fw->toplevel),
		"Find Objects"
	    );
	    if(fw->results_clist != NULL)
	    {
		const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,    0,      EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST, 0,      EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,        0,      EDV_DND_INFO_STRING}
		};
		GUIDNDSetSrcTypes(
		    fw->results_clist,
		    dnd_src_types,
		    sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		    GDK_ACTION_COPY | GDK_ACTION_MOVE |
			GDK_ACTION_LINK,
		    GDK_BUTTON1_MASK
		);
	    }
	    break;
	  case EDV_LOCATION_TYPE_RECBIN:
	    gtk_window_set_title(
		GTK_WINDOW(fw->toplevel),
		"Find Recycled Objects"
	    );
	    if(fw->results_clist != NULL)
	    {
		const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TARGET_RECYCLED_OBJECT, 0,	EDV_DND_INFO_RECYCLED_OBJECT}
		};
		GUIDNDSetSrcTypes(
		    fw->results_clist,
		    dnd_src_types,
		    sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		    GDK_ACTION_MOVE,
		    GDK_BUTTON1_MASK
		);
	    }
	    break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    gtk_window_set_title(
		GTK_WINDOW(fw->toplevel),
		"Find Archive Objects"
	    );
	    if(fw->results_clist != NULL)
	    {
		const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TARGET_ARCHIVE_OBJECT, 0,      EDV_DND_INFO_ARCHIVE_OBJECT}
		};
		GUIDNDSetSrcTypes(
		    fw->results_clist,
		    dnd_src_types,
		    sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		    GDK_ACTION_COPY,
		    GDK_BUTTON1_MASK
		);
	    }
	    break;
	}
}


/*
 *	Resets the Results List column attributes.
 *
 *	The find_by specifies the find by method.
 */
void EDVFindWinListResetColumns(
	edv_find_win_struct *fw,
	const edv_find_win_find_by find_by
)
{
	gint i, ncolumns;
	gchar **title;
	gint *justify;
	GtkRcStyle *lists_rcstyle;
	GtkWidget *w;
	GtkCList *clist;
	edv_core_struct *core;

	if(fw == NULL)
	    return;

	w = fw->results_clist;
	clist = GTK_CLIST(w);
	ncolumns = MAX(clist->columns, 1);
	core = fw->core;
	lists_rcstyle = core->lists_rcstyle;

	/* Allocate the column property values */
	title = (gchar **)g_malloc0(ncolumns * sizeof(gchar *));
	justify = (gint *)g_malloc0(ncolumns * sizeof(gint));
	if((title == NULL) || (justify == NULL))
	{
	    g_free(title);
	    g_free(justify);
	    return;
	}

	/* Begin resetting the columns */

	gtk_clist_freeze(clist);

	/* Update the column settings */
	gtk_clist_column_titles_show(clist);
	gtk_clist_column_titles_active(clist);

	/* Set up each column */
	for(i = 0; i < ncolumns; i++)
	{
	    /* Set column attributes by the find operation code */
	    switch(find_by)
	    {
	      case EDV_FIND_WIN_FIND_BY_NAME:
		switch(i)
		{
		  case 0:
		    title[i] = "Name";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  case 1:
		    title[i] = "Size";
		    justify[i] = GTK_JUSTIFY_RIGHT;
		    break;
		  case 2:
		    title[i] = "Date Modified";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  case 3:
		    title[i] = "Location";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  default:
		    title[i] = NULL;
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		}
		break;
	      case EDV_FIND_WIN_FIND_BY_CONTENT:
		switch(i)
		{
		  case 0:
		    title[i] = "Name";	/* And location */
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  case 1:
		    title[i] = "Size";
		    justify[i] = GTK_JUSTIFY_RIGHT;
		    break;
		  case 2:
		    title[i] = "Line";
		    justify[i] = GTK_JUSTIFY_RIGHT;
		    break;
		  case 3:
		    title[i] = "Excerpt";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  case 4:
		    title[i] = "Date Modified";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  case 5:
		    title[i] = "Location";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  default:
		    title[i] = NULL;
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		}
		break;
	      case EDV_FIND_WIN_FIND_BY_SIZE_EQUAL_TO:
	      case EDV_FIND_WIN_FIND_BY_SIZE_NOT_EQUAL_TO:
	      case EDV_FIND_WIN_FIND_BY_SIZE_LESS_THAN:
	      case EDV_FIND_WIN_FIND_BY_SIZE_GREATER_THAN:
		switch(i)
		{
		  case 0:
		    title[i] = "Name";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  case 1:
		    title[i] = "Size";
		    justify[i] = GTK_JUSTIFY_RIGHT;
		    break;
		  case 2:
		    title[i] = "Date Modified";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  case 3:
		    title[i] = "Location";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  default:
		    title[i] = NULL;
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		}
		break;
	      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_EQUAL_TO:
	      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
	      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_LESS_THAN:
	      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_GREATER_THAN:
		switch(i)
		{
		  case 0:
		    title[i] = "Name";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  case 1:
		    title[i] = "Size";
		    justify[i] = GTK_JUSTIFY_RIGHT;
		    break;
		  case 2:
		    title[i] = "Date Modified";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  case 3:
		    title[i] = "Location";
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		  default:
		    title[i] = NULL;
		    justify[i] = GTK_JUSTIFY_LEFT;
		    break;
		}
		break;
	    }

	    /* Is this column's properties defined? */
	    if(title[i] != NULL)
	    {
		/* Show this column and set its attributes */
		gtk_clist_set_column_visibility(
		    clist, i, TRUE
		);
		gtk_clist_set_column_resizeable(
		    clist, i, TRUE
		);
		gtk_clist_set_column_auto_resize(
		    clist, i, FALSE
		);
		gtk_clist_set_column_title(
		    clist, i, title[i]
		);
		gtk_clist_set_column_justification(
		    clist, i, justify[i]
		);
	    }
	    else
	    {
		/* Hide this column */
		gtk_clist_set_column_visibility(
		    clist, i, FALSE
		);
	    }
	}

	gtk_clist_thaw(clist);

	/* Set the RC style */
	if(lists_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(w, lists_rcstyle);

	g_free(title);
	g_free(justify);
}

/*
 *	Appends an item to the Results List.
 *
 *	The path specifies the object's path. It can be either a full
 *	path, a recycled object index string, or a path within an
 *	archive depending on the current location type.
 *
 *	The lstat_buf specifies the object's statistics.
 *
 *	The excerpt specifies the content excerpt or NULL if this
 *	is not applicable.
 *
 *	The line_index specifies the line index of the excerpt.
 */
void EDVFindWinListAppend(
	edv_find_win_struct *fw,
	const gchar *path,
	struct stat *lstat_buf,
	const gchar *excerpt,
	const gint line_index
)
{
	gint ncolumns;
	const gchar *format;
	gulong block_size;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_date_relativity relativity;
	edv_size_format size_format;
	edv_find_win_find_by find_by;
	edv_core_struct *core;
	if((fw == NULL) || STRISEMPTY(path))
	    return;

	clist = GTK_CLIST(fw->results_clist);
	ncolumns = MAX(clist->columns, 1);
	core = fw->core;
	cfg_list = core->cfg_list;

	/* Get the current find by method */
	find_by = EDVFindWinCurrentFindBy(fw);

	/* Get the configuration */
	relativity = (edv_date_relativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	size_format = (edv_size_format)EDV_GET_I(
	    EDV_CFG_PARM_SIZE_FORMAT
	);
	block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE);

	/* Begin appending the item */
	if(TRUE)
	{
	    gint i, row;

	    /* Allocate the row cell values */
	    gchar **strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	    for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	    gtk_clist_freeze(clist);

	    /* Append a new row */
	    row = gtk_clist_append(clist, strv);
	    if(row > -1)
	    {
		edv_object_struct *obj = NULL;

		/* Create the object data based on the location type */
		switch(fw->location_type)
		{
		  case EDV_LOCATION_TYPE_VFS:
		    obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			EDVObjectSetPath(obj, path);
			EDVObjectSetStat(obj, lstat_buf);
			EDVObjectUpdateLinkFlags(obj);
		    }
		    break;

		  case EDV_LOCATION_TYPE_RECBIN:
		    obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			guint index;
			edv_recycled_object_struct *rec_obj;

			/* Get the recycled object's index from the path */
			if(*path == '#')
			    index = (guint)ATOI(path + 1);
			else
			    index = (guint)ATOI(path);

			/* Get the recycled object's statistics */
			rec_obj = EDVRecBinObjectStat(
			    EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
			    index
			);
			if(rec_obj != NULL)
			{
			    obj->name = STRDUP(rec_obj->name);
			    obj->full_path = STRDUP(PrefixPaths(
				(const char *)rec_obj->original_path,
				(const char *)rec_obj->name
			    ));
			    obj->index = (guint)rec_obj->index;
			    obj->type = rec_obj->type;
			    obj->size = rec_obj->size;
			    obj->link_target = STRDUP(rec_obj->link_target);
			    obj->permissions = rec_obj->permissions;
			    obj->access_time = rec_obj->access_time;
			    obj->modify_time = rec_obj->modify_time;
			    obj->change_time = rec_obj->change_time;
			    obj->owner_id = rec_obj->owner_id;
			    obj->group_id = rec_obj->group_id;

			    EDVRecycledObjectDelete(rec_obj);
			}
		    }
		    break;

		  case EDV_LOCATION_TYPE_ARCHIVE:
		    obj = EDVObjectNew();
		    if(obj != NULL)
		    {
			obj->name = STRDUP(g_basename(path));
			obj->full_path = STRDUP(path);
			EDVObjectSetStat(obj, lstat_buf);
		    }
		    break;
		}
		/* Object created? */
		if(obj != NULL)
		{
		    const gchar *name = obj->name;
		    GdkPixmap *pixmap, *pixmap_hid;
		    GdkBitmap *mask, *mask_hid;

		    /* Get pixmap and mask for the given object */
		    EDVMatchObjectIcon(
			core->device, core->total_devices,
			core->mimetype, core->total_mimetypes,
			obj->type,
			obj->full_path,
			EDV_OBJECT_IS_LINK_VALID(obj),
			obj->permissions,
			0,			/* Small icons */
			&pixmap, &mask,
			NULL, NULL,
			NULL, NULL,
			&pixmap_hid, &mask_hid
		    );
		    /* Hidden? */
		    if(EDVIsObjectNameHidden(name))
		    {
			pixmap = pixmap_hid;
			mask = mask_hid;
		    }

		    /* Begin setting cell values */

		    /* Set the name, which is always the first column
		     * regardless of the find by method
		     */
		    i = 0;
		    if(i < ncolumns)
		    {
			if(pixmap != NULL)
			    gtk_clist_set_pixtext(
				clist, row, i,
				(name != NULL) ? name : "(null)",
				EDV_LIST_PIXMAP_TEXT_SPACING,
				pixmap, mask
			    );
			else
			    gtk_clist_set_text(
				clist, row, i,
				(name != NULL) ? name : "(null)"
			    );
		    }
		    /* Set the subsequent cells by the find by method */
		    switch(find_by)
		    {
		      case EDV_FIND_WIN_FIND_BY_NAME:
			/* Size */
			i = 1;
			if(i < ncolumns)
			{
			    const gulong size = obj->size;
			    const gchar *s = EDVSizeStrFormat(
				size, size_format, block_size, ',', TRUE
			    );
			    gtk_clist_set_text(
				clist, row, i,
				(s != NULL) ? s : ""
			    );
			}
			/* Modify Time */
			i = 2;
			if(i < ncolumns)
			{
			    const gulong modify_time = obj->modify_time;
			    const gchar *s = (modify_time > 0l) ?
				EDVDateFormatString(
				    modify_time, format, relativity
				) : NULL;
			    gtk_clist_set_text(
				clist, row, i,
				(s != NULL) ? s : ""
			    );
			}
			/* Location */
			i = 3;
			if(i < ncolumns)
			{
			    gchar *location = (obj->full_path != NULL) ?
				g_dirname(obj->full_path) : NULL;
			    if(!STRISEMPTY(location))
			    {
				struct stat lstat_buf;
				edv_object_struct *loc_obj = EDVObjectNew();
				if(loc_obj != NULL)
				{
				    EDVObjectSetPath(loc_obj, location);
				    if(lstat((const char *)location, &lstat_buf))
				    {
					loc_obj->type = EDV_OBJECT_TYPE_DIRECTORY;
				    }
				    else
				    {
					EDVObjectSetStat(loc_obj, &lstat_buf);
				    }
				    EDVMatchObjectIcon(
					core->device, core->total_devices,
					core->mimetype, core->total_mimetypes,
					loc_obj->type,
					loc_obj->full_path,
					EDV_OBJECT_IS_LINK_VALID(loc_obj),
					loc_obj->permissions,
					0,          /* Small icons */
					&pixmap, &mask,
					NULL, NULL,
					NULL, NULL,
					&pixmap_hid, &mask_hid
				    );
				    /* Hidden? */
				    if(EDVIsObjectNameHidden(loc_obj->name))
				    {
					pixmap = pixmap_hid;
					mask = mask_hid;
				    }
				    if(pixmap != NULL)
					gtk_clist_set_pixtext(
					    clist, row, i,
					    location,
					    EDV_LIST_PIXMAP_TEXT_SPACING,
					    pixmap, mask
					);
				    else
					gtk_clist_set_text(
					    clist, row, i,
					    location
					);
				    EDVObjectDelete(loc_obj);
				}
			    }
			    g_free(location);
			}
			break;

		      case EDV_FIND_WIN_FIND_BY_CONTENT:
			/* Size */
			i = 1;
			if(i < ncolumns)
			{
			    const gulong size = obj->size;
			    const gchar *s = EDVSizeStrFormat(
				size, size_format, block_size, ',', TRUE
			    );
			    gtk_clist_set_text(
				clist, row, i,
				(s != NULL) ? s : ""
			    );
			}
			/* Line */
			i = 2;
			if(i < ncolumns)
			{
			    gchar *s;
			    if(line_index > -1)
				s = g_strdup_printf(
				    "%i", line_index + 1
				);
			    else
				s = STRDUP("");
			    gtk_clist_set_text(
				clist, row, i, s
			    );
			    g_free(s);
			}
			/* Excerpt */
			i = 3;
			if(i < ncolumns)
			{
			    gtk_clist_set_text(
				clist, row, i,
				(excerpt != NULL) ? excerpt : ""
			    );
			}
			/* Modify Time */
			i = 4;
			if(i < ncolumns)
			{
			    const gulong modify_time = obj->modify_time;
			    const gchar *s = (modify_time > 0l) ?
				EDVDateFormatString(
				    modify_time, format, relativity
				) : NULL;
			    gtk_clist_set_text(
				clist, row, i,
				(s != NULL) ? s : ""
			    );
			}
			/* Location */
			i = 5;
			if(i < ncolumns)
			{
			    gchar *location = (obj->full_path != NULL) ?
				g_dirname(obj->full_path) : NULL;
			    if(!STRISEMPTY(location))
			    {
				struct stat lstat_buf;
				edv_object_struct *loc_obj = EDVObjectNew();
				if(loc_obj != NULL)
				{
				    EDVObjectSetPath(loc_obj, location);
				    if(lstat((const char *)location, &lstat_buf))
				    {
					loc_obj->type = EDV_OBJECT_TYPE_DIRECTORY;
				    }
				    else
				    {
					EDVObjectSetStat(loc_obj, &lstat_buf);
				    }
				    EDVMatchObjectIcon(
					core->device, core->total_devices,
					core->mimetype, core->total_mimetypes,
					loc_obj->type,
					loc_obj->full_path,
					EDV_OBJECT_IS_LINK_VALID(loc_obj),
					loc_obj->permissions,
					0,          /* Small icons */
					&pixmap, &mask,
					NULL, NULL,
					NULL, NULL,
					&pixmap_hid, &mask_hid
				    );
				    /* Hidden? */
				    if(EDVIsObjectNameHidden(loc_obj->name))
				    {
					pixmap = pixmap_hid;
					mask = mask_hid;
				    }
				    if(pixmap != NULL)
					gtk_clist_set_pixtext(
					    clist, row, i,
					    location,
					    EDV_LIST_PIXMAP_TEXT_SPACING,
					    pixmap, mask
					);
				    else
					gtk_clist_set_text(
					    clist, row, i,
					    location
					);
				    EDVObjectDelete(loc_obj);
				}
			    }
			    g_free(location);
			}
			break;

		      case EDV_FIND_WIN_FIND_BY_SIZE_EQUAL_TO:
		      case EDV_FIND_WIN_FIND_BY_SIZE_NOT_EQUAL_TO:
		      case EDV_FIND_WIN_FIND_BY_SIZE_LESS_THAN:
		      case EDV_FIND_WIN_FIND_BY_SIZE_GREATER_THAN:
			/* Size */
			i = 1;
			if(i < ncolumns)
			{
			    const gulong size = obj->size;
			    const gchar *s = EDVSizeStrFormat(
				size, size_format, block_size, ',', TRUE
			    );
			    gtk_clist_set_text(
				clist, row, i,
				(s != NULL) ? s : ""
			    );
			}
			/* Modify Time */
			i = 2;
			if(i < ncolumns)
			{
			    const gulong modify_time = obj->modify_time;
			    const gchar *s = (modify_time > 0l) ?
				EDVDateFormatString(
				    modify_time, format, relativity
				) : NULL;
			    gtk_clist_set_text(
				clist, row, i,
				(s != NULL) ? s : ""
			    );
		 	}
			/* Location */
			i = 3;
			if(i < ncolumns)
			{
			    gchar *location = (obj->full_path != NULL) ?
				g_dirname(obj->full_path) : NULL;
			    if(!STRISEMPTY(location))
			    {
				struct stat lstat_buf;
				edv_object_struct *loc_obj = EDVObjectNew();
				if(loc_obj != NULL)
				{
				    EDVObjectSetPath(loc_obj, location);
				    if(lstat((const char *)location, &lstat_buf))
				    {
					loc_obj->type = EDV_OBJECT_TYPE_DIRECTORY;
				    }
				    else
				    {
					EDVObjectSetStat(loc_obj, &lstat_buf);
				    }
				    EDVMatchObjectIcon(
					core->device, core->total_devices,
					core->mimetype, core->total_mimetypes,
					loc_obj->type,
					loc_obj->full_path,
					EDV_OBJECT_IS_LINK_VALID(loc_obj),
					loc_obj->permissions,
					0,          /* Small icons */
					&pixmap, &mask,
					NULL, NULL,
					NULL, NULL,
					&pixmap_hid, &mask_hid
				    );
				    /* Hidden? */
				    if(EDVIsObjectNameHidden(loc_obj->name))
				    {
					pixmap = pixmap_hid;
					mask = mask_hid;
				    }
				    if(pixmap != NULL)
					gtk_clist_set_pixtext(
					    clist, row, i,
					    location,
					    EDV_LIST_PIXMAP_TEXT_SPACING,
					    pixmap, mask
					);
				    else
					gtk_clist_set_text(
					    clist, row, i,
					    location
					);
				    EDVObjectDelete(loc_obj);
				}
			    }
			    g_free(location);
			}
			break;

		      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_EQUAL_TO:
		      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
		      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_LESS_THAN:
		      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_GREATER_THAN:
			/* Size */
			i = 1;
			if(i < ncolumns)
			{
			    const gulong size = obj->size;
			    const gchar *s = EDVSizeStrFormat(
				size, size_format, block_size, ',', TRUE
			    );
			    gtk_clist_set_text(
				clist, row, i,
				(s != NULL) ? s : ""
			    );
			}
			/* Modify Time */
			i = 2;
			if(i < ncolumns)
			{
			    const gulong modify_time = obj->modify_time;
			    const gchar *s = (modify_time > 0l) ?
				EDVDateFormatString(
				    modify_time, format, relativity
				) : NULL;
			    gtk_clist_set_text(
				clist, row, i,
				(s != NULL) ? s : ""
			    );
		 	}
			/* Location */
			i = 3;
			if(i < ncolumns)
			{
			    gchar *location = (obj->full_path != NULL) ?
				g_dirname(obj->full_path) : NULL;
			    if(!STRISEMPTY(location))
			    {
				struct stat lstat_buf;
				edv_object_struct *loc_obj = EDVObjectNew();
				if(loc_obj != NULL)
				{
				    EDVObjectSetPath(loc_obj, location);
				    if(lstat((const char *)location, &lstat_buf))
				    {
					loc_obj->type = EDV_OBJECT_TYPE_DIRECTORY;
				    }
				    else
				    {
					EDVObjectSetStat(loc_obj, &lstat_buf);
				    }
				    EDVMatchObjectIcon(
					core->device, core->total_devices,
					core->mimetype, core->total_mimetypes,
					loc_obj->type,
					loc_obj->full_path,
					EDV_OBJECT_IS_LINK_VALID(loc_obj),
					loc_obj->permissions,
					0,          /* Small icons */
					&pixmap, &mask,
					NULL, NULL,
					NULL, NULL,
					&pixmap_hid, &mask_hid
				    );
				    /* Hidden? */
				    if(EDVIsObjectNameHidden(loc_obj->name))
				    {
					pixmap = pixmap_hid;
					mask = mask_hid;
				    }
				    if(pixmap != NULL)
					gtk_clist_set_pixtext(
					    clist, row, i,
					    location,
					    EDV_LIST_PIXMAP_TEXT_SPACING,
					    pixmap, mask
					);
				    else
					gtk_clist_set_text(
					    clist, row, i,
					    location
					);
				    EDVObjectDelete(loc_obj);
				}
			    }
			    g_free(location);
			}
			break;
		    }

		    /* Set the new row's data as the object */
		    gtk_clist_set_row_data_full(
			clist, row,
			obj, EDVFindWinListItemDestroyCB
		    );
/*		    obj = NULL; */
		}
	    }

	    gtk_clist_thaw(clist);

	    gtk_clist_columns_autosize(clist);

	    /* Delete the row's cell values */
	    g_free(strv);
	}

}

/*
 *	Clears the results clist.
 */
void EDVFindWinListClear(edv_find_win_struct *fw)
{
	GtkCList *clist;

	if(fw == NULL)
	    return;

	clist = GTK_CLIST(fw->results_clist);

	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
}



/*
 *	Sets values on the given find window to the configuration list on
 *	the core structure of the given find window.
 */
void EDVFindWinSyncConfiguration(edv_find_win_struct *fw)
{
	GtkWidget *w;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(fw == NULL)
	    return;

	core = fw->core;
	cfg_list = core->cfg_list;

	/* Get size of toplevel */
	w = fw->toplevel;
	if(w != NULL)
	{
	    GdkWindow *window = w->window;
	    gint x = 0, y = 0;

	    if(window != NULL)
		gdk_window_get_root_origin(window, &x, &y);

	    EDV_SET_I(EDV_CFG_PARM_FINDWIN_X, x);
	    EDV_SET_I(EDV_CFG_PARM_FINDWIN_Y, y);
	    EDV_SET_I(EDV_CFG_PARM_FINDWIN_WIDTH, w->allocation.width);
	    EDV_SET_I(EDV_CFG_PARM_FINDWIN_HEIGHT, w->allocation.height);
	}

	/* Find By Method */
	EDV_SET_I(
	    EDV_CFG_PARM_FIND_WIN_FIND_BY,
	    EDVFindWinCurrentFindBy(fw)
	);

	/* Search String */
	w = fw->search_combo;
	if(w != NULL)
	{
	    GtkCombo *combo = GTK_COMBO(w);
	    GtkEntry *entry = GTK_ENTRY(combo->entry);
	    EDV_SET_S(
		EDV_CFG_PARM_FINDWIN_SEARCH_STRING,
		gtk_entry_get_text(entry)
	    );
	}


	/* Case sensitive */
	w = fw->case_sensitive_check;
	if(w != NULL)
	{
	    EDV_SET_B(
		EDV_CFG_PARM_FINDWIN_CASE_SENSITIVE,
		GTK_TOGGLE_BUTTON_GET_ACTIVE(w)
	    );
	}

	/* Recursive */
	w = fw->recursive_check;
	if(w != NULL)
	{
	    EDV_SET_B(
		EDV_CFG_PARM_FINDWIN_RECURSIVE,
		GTK_TOGGLE_BUTTON_GET_ACTIVE(w)
	    );
	}
}



/*
 *	Creates a new Find Window.
 */
edv_find_win_struct *EDVFindWinNew(edv_core_struct *core)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint	i,
		toplevel_x = 0,
		toplevel_y = 0,
		toplevel_width = EDV_DEF_FINDWIN_WIDTH,
		toplevel_height = EDV_DEF_FINDWIN_HEIGHT;
	gboolean status_bar_map_state = TRUE;
	edv_find_win_find_by find_op;
	const gchar *search_string;
	gboolean case_sensitive, recursive;
	GList *glist;
	GdkWindow *window;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4, *parent5,
			*toplevel;
	gpointer combo_rtn;
	GtkCombo *combo;
	GtkCList *clist;
	pulist_struct *pulist;
	pulistbox_struct *pulistbox;
	const cfg_item_struct *cfg_list;
	edv_status_bar_struct *status_bar;
	edv_find_win_struct *fw;

	if(core == NULL)
	    return(NULL);

	cfg_list = core->cfg_list;

	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	toplevel_x = EDV_GET_I(EDV_CFG_PARM_FINDWIN_X);
	toplevel_y = EDV_GET_I(EDV_CFG_PARM_FINDWIN_Y);
	toplevel_width = EDV_GET_I(EDV_CFG_PARM_FINDWIN_WIDTH);
	toplevel_height = EDV_GET_I(EDV_CFG_PARM_FINDWIN_HEIGHT);
	find_op = (edv_find_win_find_by)EDV_GET_I(
	    EDV_CFG_PARM_FIND_WIN_FIND_BY
	);
	search_string = EDV_GET_S(EDV_CFG_PARM_FINDWIN_SEARCH_STRING);
	case_sensitive = EDV_GET_B(EDV_CFG_PARM_FINDWIN_CASE_SENSITIVE);
	recursive = EDV_GET_B(EDV_CFG_PARM_FINDWIN_RECURSIVE);


	/* Create the Find Window */
	fw = EDV_FIND_WIN(g_malloc0(sizeof(edv_find_win_struct)));
	if(fw == NULL)
	    return(NULL);

	fw->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	fw->accelgrp = accelgrp = gtk_accel_group_new();
	fw->processing = FALSE;
	fw->freeze_count = 0;
	fw->busy_count = 0;
	fw->stop_count = 0;
	fw->core = core;

	fw->status_bar_map_state = status_bar_map_state;

	fw->browser_num = -1;
	fw->imbr_num = -1;
	fw->recbin_num = -1;
	fw->archiver_num = -1;

	fw->location_type = EDV_LOCATION_TYPE_VFS;
	fw->find_op = EDV_FIND_WIN_FIND_BY_NAME;
	fw->last_write_protect_state = -1;

	fw->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_set_uposition(w, toplevel_x, toplevel_y);
	gtk_widget_set_usize(w, toplevel_width, toplevel_height);
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "find", PROG_NAME
	);
	gtk_window_set_title(GTK_WINDOW(w), "Find Objects");
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    GdkGeometry geo;
	    geo.min_width = 100;
	    geo.min_height = 70;
	    geo.base_width = 0;
	    geo.base_height = 0;
	    geo.width_inc = 1;
	    geo.height_inc = 1;
	    gdk_window_set_geometry_hints(
		window, &geo,
		GDK_HINT_MIN_SIZE | GDK_HINT_BASE_SIZE |
		GDK_HINT_RESIZE_INC
	    );
	    GUISetWMIcon(window, (guint8 **)icon_search_48x48_xpm);
	}
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVFindWinDeleteEventCB), fw
	);
	gtk_container_set_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;


	/* Main vbox */
	fw->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Upper section hbox to separate two columns */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_set_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left panel vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Begin creating search criteria widgets */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* GtkHBox for find the find by method and search string */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* GtkHBox for the find by method */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	w = gtk_label_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Buscar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fund"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trovare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vondst"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ache"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Funn"
#else
"Find"
#endif
	    ":"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	fw->find_by_pulistbox = pulistbox = PUListBoxNew(
	    parent5,
	    170, -1
	);
	PUListBoxSetChangedCB(
	    pulistbox,
	    EDVFindWinFindByChangedCB,
	    fw
	);
	pulist = PUListBoxGetPUList(pulistbox);
	if(core->run_flags & EDV_RUN_SAFE_MODE)
	    PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
	i = PUListAddItem(pulist, "Name");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_WIN_FIND_BY_NAME);
	i = PUListAddItem(pulist, "Content");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_WIN_FIND_BY_CONTENT);
	i = PUListAddItem(pulist, "Size Equal To");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_WIN_FIND_BY_SIZE_EQUAL_TO);
	i = PUListAddItem(pulist, "Size Not Equal To");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_WIN_FIND_BY_SIZE_NOT_EQUAL_TO);
	i = PUListAddItem(pulist, "Size Less Than");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_WIN_FIND_BY_SIZE_LESS_THAN);
	i = PUListAddItem(pulist, "Size Greater Than");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_WIN_FIND_BY_SIZE_GREATER_THAN);
	i = PUListAddItem(pulist, "Date Modified On");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_WIN_FIND_BY_MODIFY_TIME_EQUAL_TO);
	i = PUListAddItem(pulist, "Date Modified Not On");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_WIN_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO);
	i = PUListAddItem(pulist, "Date Modified Before");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_WIN_FIND_BY_MODIFY_TIME_LESS_THAN);
	i = PUListAddItem(pulist, "Date Modified After");
	PUListSetItemData(pulist, i, (gpointer)EDV_FIND_WIN_FIND_BY_MODIFY_TIME_GREATER_THAN);
	PUListBoxSetLinesVisible(pulistbox, MIN((i + 1), 10));
	PUListBoxMap(pulistbox);


	/* Search string GtkCombo */
	glist = NULL;
	w = GUIComboCreate(
#if defined(PROG_LANGUAGE_SPANISH)
"Palabra"
#elif defined(PROG_LANGUAGE_FRENCH)
"Egaler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Anpassend"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Uguagliare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Passend"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Combinar"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Passende"
#else
"Matching"
#endif
	    ":",
	    "",			/* No initial value */
	    glist,		/* List */
	    20,			/* Max items */
	    &combo_rtn,
	    fw,
	    EDVFindWinComboActivateCB,
	    NULL
	);
	fw->search_combo = (GtkWidget *)combo_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = fw->search_combo;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);

	w = combo->entry;
	EDVEntrySetDND(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	if(search_string != NULL)
	    gtk_entry_set_text(GTK_ENTRY(w), search_string);

	/* Location GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Location GtkCombo */
	glist = NULL;
	w = GUIComboCreate(
#if defined(PROG_LANGUAGE_SPANISH)
"Empezar En"
#elif defined(PROG_LANGUAGE_FRENCH)
"Commencer A"
#elif defined(PROG_LANGUAGE_GERMAN)
"Anfangen An"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cominciare A"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beginnen Aan"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comear Em"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Start P"
#else
"Starting At"
#endif
	    ":",
	    "",
	    glist,
	    25,		/* Maximum items */
	    &combo_rtn,
	    fw,
	    EDVFindWinComboActivateCB,
	    NULL
	);
	fw->location_combo = (GtkWidget *)combo_rtn;
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = fw->location_combo;
	combo = GTK_COMBO(w);
	gtk_combo_set_case_sensitive(combo, TRUE);

	w = combo->entry;
	EDVEntrySetDND(core, w);
	EDVEntrySetCompletePath(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* Browse button */
	fw->browse_location_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_browse_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVFindWinBrowseLocationCB), fw
	);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Hojee"
#elif defined(PROG_LANGUAGE_FRENCH)
"Parcourir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Brausen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Curiosare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Grasduin"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Olhe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Browse"
#else
"Browse"
#endif
	);
	gtk_widget_show(w);


	/* Options GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Case sensitive GtkCheckButton */
	fw->case_sensitive_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
"Sensible May/min."
#elif defined(PROG_LANGUAGE_FRENCH)
"Reconnatre Sensible"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fall Empfindlich"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Imballare Sensibile"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sluit Gevoelig In"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Caso Sensvel"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfelle Sensitiv"
#else
"Case Sensitive"
#endif
	);
	GTK_TOGGLE_BUTTON_SET_ACTIVE(w, case_sensitive);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique esto para discriminar Mayusculas y minusculas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vrifier ceci pour galer des ficelles reconnaissent sensiblement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Prfen sie dies, schnren fall empfindlich anzupassen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Controllare questo per uguagliare le cordicelle imballano\
 sensibilmente"
#elif defined(PROG_LANGUAGE_DUTCH)
"Controleer dit om bij koorden geval gevoelig te passen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verifique isto combinar caso de barbantes sensivelmente"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sjekk dette passe snortilfelle sensitivt"
#else
"Check this to match strings case sensitively"
#endif
	);
	gtk_widget_show(w);

	/* Recursive check */
	fw->recursive_check = w = gtk_check_button_new_with_label(
#if defined(PROG_LANGUAGE_SPANISH)
"Recursivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Recursive"
#elif defined(PROG_LANGUAGE_GERMAN)
"Rekursiv"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ricorsivo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Recursief"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Recursive"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rekursiv"
#else
"Recursive"
#endif
	);
	GTK_TOGGLE_BUTTON_SET_ACTIVE(w, recursive);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Verifique esto buscar recursivamente en sub guas"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vrifier ceci pour chercher recursively dans sous les annuaires"
#elif defined(PROG_LANGUAGE_GERMAN)
"Prfen sie dies, rekursiv in u-boot verzeichnisse zu suchen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Controllare questo per ricercare ricorsivamente in sotto gli elenchi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Controleer dit om recursief in sub gidzen te zoeken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Verifique isto procurar recursively em guias de submarino"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sjekk dette gjennomske rekursivt inn i sub kataloger"
#else
"Check this to search recursively into sub directories"
#endif
	);
	gtk_widget_show(w);


	/* Separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Scrolled window for results clist */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Results clist */
	fw->results_clist = w = gtk_clist_new(
	    DEV_FINDWIN_RESULTS_LIST_COLUMNS_MAX
	);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVCListKeyEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(EDVCListButtonEventCB), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(EDVCListMotionEventCB), core
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVFindWinButtonCB), fw
	);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_realize(w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_column_titles_hide(clist);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_signal_connect(
	    GTK_OBJECT(w), "click_column",
	    GTK_SIGNAL_FUNC(EDVFindWinClickColumnCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(EDVFindWinSelectRowCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(EDVFindWinUnselectRowCB), fw
	);
	if(w != NULL)
	{
	    const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	    };
	    GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE |
		    GDK_ACTION_LINK,		/* Actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		EDVFindWinDragDataGetCB,
		EDVFindWinDragDataDeleteCB,
		NULL,
		fw
	    );
	}
	gtk_widget_show(w);


	/* Right click menu */
	fw->results_clist_menu = w = GUIMenuCreate();
	if(w != NULL)
	{
	    GtkWidget *menu = w;
	    guint8 **icon;
	    const gchar *label;
	    guint accel_key, accel_mods;
	    void (*func_cb)(GtkWidget *, gpointer);
	    gpointer mclient_data = fw;

#define DO_ADD_MENU_ITEM_LABEL  {                       \
 w = GUIMenuItemCreate(                                 \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,             \
  icon, label, accel_key, accel_mods, NULL,		\
  mclient_data, func_cb                                 \
 );                                                     \
}
#define DO_ADD_MENU_SEP         {                       \
 w = GUIMenuItemCreate(                                 \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,             \
  NULL, NULL, 0, 0, NULL,                               \
  NULL, NULL                                            \
 );                                                     \
}
	    icon = (guint8 **)icon_open_20x20_xpm;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen"
#else
"Open"
#endif
	    ;
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVFindWinOpenCB;
	    DO_ADD_MENU_ITEM_LABEL
	    fw->results_clist_open_mi = w;

	    icon = NULL;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir Con"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir Avec"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Mit"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto Con"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Met"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto Com"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn Med"
#else
"Open With"
#endif
	    "...";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVFindWinOpenWithCB;
	    DO_ADD_MENU_ITEM_LABEL
	    fw->results_clist_open_with_mi = w;

	    DO_ADD_MENU_SEP

	    icon = (guint8 **)icon_goto_20x20_xpm;
	    label =
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya A"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller A"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen Zu"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare A"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga Te"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V A"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra Til"
#else
"GoTo"
#endif
	    ;
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVFindWinGotoCB;
	    DO_ADD_MENU_ITEM_LABEL
	    fw->results_clist_goto_mi = w;

#undef DO_ADD_MENU_SEP
#undef DO_ADD_MENU_ITEM_LABEL
	}

	/* Buttons hbox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Open button */
	fw->open_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_open_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pen"
#else
"Open"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVFindWinOpenCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir archivo escogido"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir l'objet choisi"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen ausgewhlten objekt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto scelto oggetto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open geselecteerd voorwerp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Objeto selecionado aberto"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn valgt ut objekt"
#else
"Open selected object"
#endif
	);
	gtk_widget_show(w);

	/* Open With button */
	fw->open_with_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_open_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir Con"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir Avec"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen Mit"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto Con"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open Met"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aberto Com"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn Med"
#else
"Open With"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVFindWinOpenWithCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_w, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_w);
	GUISetWidgetTip(
	    w,
#if defined(PROG_LANGUAGE_SPANISH)
"Abrir archivo escogido utilizando un mtodo especfico"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ouvrir l'objet choisi utilisant une mthode spcifique"
#elif defined(PROG_LANGUAGE_GERMAN)
"Offen ausgewhlten objekt, eine spezifische methode zu benutzen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aperto scelto oggetto usando uno specifico metodo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Open geselecteerd voorwerp een specifieke methode te gebruiken"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Objeto selecionado aberto usando um mtodo especfico"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"pn valgt ut objekt bruke en spesifikk metode"
#else
"Open selected object using a specific method"
#endif
	);
	gtk_widget_show(w);

	/* Goto button */
	fw->goto_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_goto_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya A"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller A"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen Zu"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare A"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga Te"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V A"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra Til"
#else
"GoTo"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVFindWinGotoCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_g, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_g);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Vaya a escogido se opone"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aller  l'objet choisi"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gehen sie zu ausgewhltem objekt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Andare all'oggetto scelto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Ga te geselecteerd voorwerp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"V a objeto selecionado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Dra til valgt ut objekt"
#else
"Go to selected object"
#endif
	);
	gtk_widget_show(w);


	/* Right column vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Buttons set vbox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Search button */
	fw->search_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_search_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Bsqueda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Recherche"
#elif defined(PROG_LANGUAGE_GERMAN)
"Suche"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Ricerca"
#elif defined(PROG_LANGUAGE_DUTCH)
"Zoektocht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Busca"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Leting"
#else
"Search"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVFindWinSearchCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Empiece la bsqueda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Commencer la recherche"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fangen sie die suche an"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cominciare la ricerca"
#elif defined(PROG_LANGUAGE_DUTCH)
"Begin de zoektocht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comece a busca"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Start letingen"
#else
"Start the search"
#endif
	);
	gtk_widget_show(w);

	/* Stop button */
	fw->stop_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_stop_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Parada"
#elif defined(PROG_LANGUAGE_FRENCH)
"Arrt"
#elif defined(PROG_LANGUAGE_GERMAN)
"Halt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Fermata"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Parada"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stans"
#else
"Stop"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVFindWinStopCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_t, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_t);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Pare el procedimiento actual del hallazgo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Arrter la procdure actuelle de dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Halten sie das jetzige fund verfahren auf"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Fermare la corrente trova la procedura"
#elif defined(PROG_LANGUAGE_DUTCH)
"Stop de huidig vondst procedure"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pare a corrente achar procedimento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stans den nvrendee funnprosedyre"
#else
"Stop the current find procedure"
#endif
	);
	gtk_widget_show(w);

	/* Clear button */
	fw->clear_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_clear_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Claro"
#elif defined(PROG_LANGUAGE_FRENCH)
"Clair"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klar"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Claro"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar"
#else
"Clear"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVFindWinClearCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_l, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_l);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"La cuerda clara de la bsqueda y la lista de resultados"
#elif defined(PROG_LANGUAGE_FRENCH)
"La ficelle clairs de recherche et la liste de rsultats"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klare suche schnur und ergebnisse fhren auf"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'elenco di cordicella di ricerca e risultati chiaro"
#elif defined(PROG_LANGUAGE_DUTCH)
"Helder zoektocht koord en resultaten sommen op"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Barbante clara de busca e lista de resultados"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klar letingssnor og resultater lister opp"
#else
"Clear search string and results list"
#endif
	);
	gtk_widget_show(w);


	/* Buttons set GtkVBox */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Close button */
	fw->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fin"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVFindWinCloseCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(EDVFindWinCrossingCB), fw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre esta ventana"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fermer cette fentre"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schlieen sie dieses fenster"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino questa finestra"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sluit deze raam"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo esta janela"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stenge dette vinduet"
#else
"Close this window"
#endif
	);
	gtk_widget_show(w);


	/* Status Bar */
	fw->status_bar = status_bar = EDVStatusBarNew(
	    core, fw->main_vbox
	);
	if(status_bar != NULL)
	{
	    if(fw->status_bar_map_state)
		EDVStatusBarMap(status_bar);
	    else
		EDVStatusBarUnmap(status_bar);
	}


	/* Set initial RC styles */
	if(standard_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		fw->toplevel, standard_rcstyle
	    );
	if(lists_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		fw->results_clist, lists_rcstyle
	    );
	if(standard_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		fw->results_clist_menu, standard_rcstyle
	    );


	EDVFindWinUpdateMenus(fw);

	fw->freeze_count--;

	return(fw);
}

/*
 *	Updates the Find Window's widgets to reflect current values.
 */
void EDVFindWinUpdateMenus(edv_find_win_struct *fw)
{
	gboolean	write_protect,
			write_protect_changed = FALSE,
			sensitive,
			processing;
	gint sel_row;
	GList *glist;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_location_type location_type;
	edv_find_win_find_by find_by;
	edv_core_struct *core;

	if(fw == NULL)
	    return;

	processing = fw->processing;
	location_type = fw->location_type;
	find_by = EDVFindWinCurrentFindBy(fw);
	clist = GTK_CLIST(fw->results_clist);
	core = fw->core;
	cfg_list = core->cfg_list;

	/* Get the last selected row on the results GtkCList */
	glist = clist->selection_end;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;

	/* Get the global write protect state */
	write_protect = EDV_GET_B(EDV_CFG_PARM_WRITE_PROTECT);
	if(fw->last_write_protect_state < 0)
	{
	    write_protect_changed = TRUE;
	}
	else
	{
	    if((write_protect && !fw->last_write_protect_state) ||
	       (!write_protect && fw->last_write_protect_state)
	    )
		write_protect_changed = TRUE;
	}
	fw->last_write_protect_state = write_protect ? 1 : 0;

	/* Find By */
	sensitive = !processing;
	gtk_widget_set_sensitive(
	    PUListBoxGetToplevel(fw->find_by_pulistbox),
	    sensitive
	);

	/* Search String */
	sensitive = !processing;
	gtk_widget_set_sensitive(fw->search_combo, sensitive);

	/* Location */
	if(processing)
	{
	    sensitive = FALSE;
	}
	else
	{
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		sensitive = TRUE;
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		sensitive = FALSE;
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		sensitive = TRUE;
		break;
	    }
	}
	gtk_widget_set_sensitive(fw->location_combo, sensitive);
	gtk_widget_set_sensitive(fw->browse_location_btn, sensitive);

	/* Case Sensitive */
	if(processing)
	{
	    sensitive = FALSE;
	}
	else
	{
	    switch(find_by)
	    {
	      case EDV_FIND_WIN_FIND_BY_NAME:
	      case EDV_FIND_WIN_FIND_BY_CONTENT:
		sensitive = TRUE;
		break;
	      case EDV_FIND_WIN_FIND_BY_SIZE_EQUAL_TO:
	      case EDV_FIND_WIN_FIND_BY_SIZE_NOT_EQUAL_TO:
	      case EDV_FIND_WIN_FIND_BY_SIZE_LESS_THAN:
	      case EDV_FIND_WIN_FIND_BY_SIZE_GREATER_THAN:
	      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_EQUAL_TO:
	      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_NOT_EQUAL_TO:
	      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_LESS_THAN:
	      case EDV_FIND_WIN_FIND_BY_MODIFY_TIME_GREATER_THAN:
		sensitive = FALSE;
		break;
	    }
	}
	gtk_widget_set_sensitive(fw->case_sensitive_check, sensitive);

	/* Recursive */
	if(processing)
	{
	    sensitive = FALSE;
	}
	else
	{
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		sensitive = TRUE;
		break;
	      case EDV_LOCATION_TYPE_RECBIN:
		sensitive = FALSE;
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		sensitive = FALSE;
		break;
	    }
	}
	gtk_widget_set_sensitive(fw->recursive_check, sensitive);

	/* Stop */
	sensitive = processing;
	gtk_widget_set_sensitive(fw->stop_btn, sensitive);

	/* Search */
	sensitive = !processing;
	gtk_widget_set_sensitive(fw->search_btn, sensitive);
	gtk_widget_set_sensitive(fw->clear_btn, sensitive);
	gtk_widget_set_sensitive(fw->close_btn, sensitive);

	/* Open */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    sensitive = (!processing && (sel_row > -1)) ? TRUE : FALSE;
	    break;
	  case EDV_LOCATION_TYPE_RECBIN:
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    sensitive = FALSE;
	    break;
	}
	gtk_widget_set_sensitive(fw->open_btn, sensitive);
	gtk_widget_set_sensitive(fw->results_clist_open_mi, sensitive);
	gtk_widget_set_sensitive(fw->open_with_btn, sensitive);
	gtk_widget_set_sensitive(fw->results_clist_open_with_mi, sensitive);

	/* Goto */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	  case EDV_LOCATION_TYPE_RECBIN:
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    sensitive = (!processing && (sel_row > -1)) ? TRUE : FALSE;
	    break;
	}
	gtk_widget_set_sensitive(fw->goto_btn, sensitive);
	gtk_widget_set_sensitive(fw->results_clist_goto_mi, sensitive);

	/* Status Bar */
	EDVStatusBarUpdate(fw->status_bar);
}

/*
 *	Sets the Find Window as busy or ready.
 */
void EDVFindWinSetBusy(edv_find_win_struct *fw, const gboolean busy)
{
	GdkCursor *cursor;
	GtkWidget *w;
	edv_core_struct *core;

	if(fw == NULL)
	    return;

	w = fw->toplevel;
	core = fw->core;

	if(w != NULL)
	{
	    if(busy)
	    {
		/* Increase busy count */
		fw->busy_count++;

		/* If already busy then don't change anything */
		if(fw->busy_count > 1)
		    return;

		cursor = EDVGetCursor(core, EDV_CURSOR_CODE_BUSY);
	    }
	    else
	    {
		/* Reduce busy count */
		fw->busy_count--;
		if(fw->busy_count < 0)
		    fw->busy_count = 0;

		/* If still busy do not change anything */
		if(fw->busy_count > 0)
		    return;

		cursor = NULL;  /* Use default cursor */
	    }

	    /* Update toplevel window's cursor */
	    if(w->window != NULL)
	    {
		gdk_window_set_cursor(w->window, cursor);
		gdk_flush();
	    }
	}
}

/*
 *	Checks if the Find Window is mapped.
 */
gboolean EDVFindWinIsMapped(edv_find_win_struct *fw)
{
	if(fw == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(fw->toplevel));
}

/*
 *	Maps the Find Window.
 */
void EDVFindWinMap(edv_find_win_struct *fw)
{
	GtkWidget *w;

	if(fw == NULL)
	    return;

	gtk_widget_show_raise(fw->toplevel);

	w = fw->search_combo;
	w = GTK_COMBO(w)->entry;
	gtk_widget_grab_focus(w);
}

/*
 *	Unmaps the Find Window.
 */
void EDVFindWinUnmap(edv_find_win_struct *fw)
{
	if(fw == NULL)
	    return;

	gtk_widget_hide(fw->toplevel);
}

/*
 *	Deletes the Find Window.
 */
void EDVFindWinDelete(edv_find_win_struct *fw)
{
	if(fw == NULL)
	    return;

	EDVFindWinUnmap(fw);

	fw->freeze_count++;

	GTK_WIDGET_DESTROY(fw->results_clist_menu);
	EDVStatusBarDelete(fw->status_bar);
	PUListBoxDelete(fw->find_by_pulistbox);
	GTK_WIDGET_DESTROY(fw->toplevel);
	GTK_ACCEL_GROUP_UNREF(fw->accelgrp);

	fw->freeze_count--;

	g_free(fw);
}
