#include <string.h>
#include <sys/types.h>
#include <limits.h>
#include <time.h>
#include <sys/stat.h>
#include <glib.h>
#include <unistd.h>

#include "edv_types.h"
#include "../edv_obj.h"
#include "../edv_recycled_obj.h"
#include "../edv_recbin_index.h"
#include "../edv_recbin_stat.h"
#include "edv_notify.h"
#include "edv_recycle.h"
#include "edv_utils.h"
#include "config.h"


const gchar *EDVRecycleGetError(edv_context_struct *ctx);

gint EDVRecycledObjectStat(
	edv_context_struct *ctx,
	const guint index,
	gchar **path_rtn,
	struct stat *stat_buf
);
gint EDVRecycledObjectStatAll(
	edv_context_struct *ctx,
	gchar ***path_rtn,
	guint **index_rtn,
	struct stat ***stat_buf,
	gint *total
);

guint EDVRecycle(
	edv_context_struct *ctx,
	const gchar *path,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer data, const gulong i, const gulong m
	),
	gpointer data
);

gint EDVRecover(
	edv_context_struct *ctx,
	const guint index,
	const gchar *alternate_recovery_path,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer data, const gulong i, const gulong m
	),
	gpointer data
);

gint EDVPurge(
	edv_context_struct *ctx,
	const guint index,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer data, const gulong i, const gulong m
	),
	gpointer data
);


static const gchar *last_error = NULL;


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)



/*
 *	Returns a statically allocated string describing the last
 *	error that occured when calling EDVRecycle(), EDVRecover(),
 *	or EDVPurge(), or NULL if there was no error.
 */
const gchar *EDVRecycleGetError(edv_context_struct *ctx)
{
	return((ctx != NULL) ? last_error : NULL);
}


/*
 *	Gets the statistics of the recycled object.
 *
 *	The index specifies the recycled object.
 *
 *	The path_rtn specifies the pointer to the return full path
 *	describing the original location of the object. You may pass
 *	NULL if the return value is not requested.
 *
 *	The stat_buf specifies the pointer to the return struct stat
 *	describing the original object's statistics. You may pass
 *	NULL if the return value is not requested.
 *
 *	The struct stat members st_rdev, st_blksize, and st_blocks
 *	members will be undefined.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint EDVRecycledObjectStat(
	edv_context_struct *ctx,
	const guint index,
	gchar **path_rtn,
	struct stat *stat_buf
)
{
	edv_recycled_object_struct *obj;
	const gchar *index_path = (ctx != NULL) ?
	    ctx->recycled_index_file : NULL;

	last_error = NULL;

	if(path_rtn != NULL)
	    *path_rtn = NULL;
	if(stat_buf != NULL)
	    memset(stat_buf, 0x00, sizeof(stat_buf));

	if(STRISEMPTY(index_path))
	{
	    last_error = "Bad value";
	    return(-1);
	}

	/* Get the recycled object's statistics */
	obj = EDVRecBinObjectStat(index_path, index);
	if(obj == NULL)
	{
	    last_error = "Unable to obtain the recycled object's statistics";
	    return(-1);
	}

	/* Transfer values from recycled object structure to path_rtn
	 * and stat_buf
	 */
	if(path_rtn != NULL)
	{
	    /* Format the full path to the original object */
	    *path_rtn = g_strconcat(
		obj->original_path,
		G_DIR_SEPARATOR_S,
		obj->name,
		NULL
	    );
	}
	if(stat_buf != NULL)
	{
	    /* Copy the recycled object's values to the stat_buf */
	    EDVRecycledObjectSetToStat(stat_buf, obj);
	}

	EDVRecycledObjectDelete(obj);

	return(0);
}

/*
 *	Gets a listing of all recycled objects.
 *
 *	The path_rtn specifies the pointer to the return list for full
 *	paths to the original object. You may pass NULL if the return
 *	value is not requested.
 *
 *	The index_rtn specifies the pointer to the return list of
 *	recycled object indices. You may pass NULL if the return value
 *	is not requested.
 *
 *	The stat_buf specifies the pointer to the return list of
 *	recycled object stats. You may pass NULL if the return value  
 *	is not requested.
 *
 *	The returned pointer arrays and each pointer to location must
 *	be deleted by the calling function.
 *
 *	The struct stat members st_rdev, st_blksize, and st_blocks
 *	members will be undefined.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint EDVRecycledObjectStatAll(
	edv_context_struct *ctx,
	gchar ***path_rtn,
	guint **index_rtn,
	struct stat ***stat_buf,
	gint *total
)
{
	struct stat *stat_buf_ptr;
	gint i;
	edv_recycled_object_struct *obj;
	edv_recbin_index_struct *rbi_ptr;
	const gchar *index_path = (ctx != NULL) ?
	    ctx->recycled_index_file : NULL;

	if(path_rtn != NULL)
	    *path_rtn = NULL;
	if(index_rtn != NULL)
	    *index_rtn = NULL;
	if(stat_buf != NULL)
	    *stat_buf = NULL;

	if(total != NULL)
	    *total = 0;

	if(STRISEMPTY(index_path) || (total == NULL))
	    return(-1);

	/* Open recycled objects index file for subsequent reading
	 * of each recycled object's stats
	 */
	rbi_ptr = EDVRecBinIndexOpen(index_path);
	if(rbi_ptr == NULL)
	    return(-1);

	/* Read each recycled object */
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    obj = rbi_ptr->obj;
	    if(obj != NULL)
	    {
		/* Increase total */
		i = MAX(*total, 0);
		*total = i + 1;

		/* Append this object's path to the path return list */
		if(path_rtn != NULL)
		{
		    *path_rtn = (gchar **)g_realloc(
			*path_rtn,
			(*total) * sizeof(gchar *)
		    );
		    if(*path_rtn == NULL)
		    {
			*total = 0;
			break;
		    }
		    (*path_rtn)[i] = g_strdup_printf(
			"%s%c%s",
			obj->original_path,
			G_DIR_SEPARATOR,
			obj->name
		    );
		}

		/* Append this object's index to the index return list */
		if(index_rtn != NULL)
		{
		    *index_rtn = (guint *)g_realloc(
			*index_rtn,
			(*total) * sizeof(guint)
		    );
		    if(*index_rtn == NULL)
		    {
			*total = 0;
			break;
		    }
		    (*index_rtn)[i] = obj->index;
		}

		/* Append this object's stats to the stats return list */
		if(stat_buf != NULL)
		{
		    *stat_buf = (struct stat **)g_realloc(
			*stat_buf,
			(*total) * sizeof(struct stat *)
		    );
		    if(*stat_buf == NULL)
		    {
			*total = 0;
			break;
		    }
		    (*stat_buf)[i] = stat_buf_ptr = g_malloc0(sizeof(struct stat));
		    EDVRecycledObjectSetToStat(stat_buf_ptr, obj);
		}
	    }
	}	/* Read each recycled object */

	EDVRecBinIndexClose(rbi_ptr);

	return(0);
}

/*
 *	Deletes an object.
 *
 *	The path specifies the full path to the object to delete.
 *
 *	If notify is set to TRUE then a "object_removed_notify" and
 *	"recycled_object_added_notify" will be queued on the ctx
 *	(only on success).
 *
 *	Returns the recycled object's index number on success or 0
 *	on error.
 */
guint EDVRecycle(
	edv_context_struct *ctx,
	const gchar *path,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer data, const gulong i, const gulong m
	),
	gpointer data
)
{
	mode_t m;
	struct stat lstat_buf;
	gchar *cwd, *dpath, *parent_path;
	guint index;
	gulong deleted_time;
	edv_recycled_object_struct *obj;
	const gchar *index_path = (ctx != NULL) ?
	    ctx->recycled_index_file : NULL;

	last_error = NULL;

	if(STRISEMPTY(index_path) || STRISEMPTY(path))
	{
	    last_error = "Bad value";
	    return(0);
	}

	cwd = EDVGetCWD();
	dpath = EDVEvaluatePath(cwd, path);
	deleted_time = (gulong)time(NULL);

	/* Get this object's local statistics */
	if(lstat((const char *)dpath, &lstat_buf))
	{
	    g_free(dpath);
	    g_free(cwd);
	    last_error = "Unable to get the object's local stats";
	    return(0);
	}

	m = lstat_buf.st_mode;

	/* Get this object's location */
        parent_path = g_dirname(dpath);
        if(parent_path == NULL)
            parent_path = STRDUP("/");

	/* Create a new recycled object */
	obj = EDVRecycledObjectNew();
	if(obj == NULL)
	{
	    g_free(parent_path);
	    g_free(dpath);
	    g_free(cwd);
	    last_error = "Memory allocation error";
	    return(0);
	}

	/* Convert the struct stat values to the recycled object */
	EDVRecycledObjectSetStat(obj, &lstat_buf);

	/* Set the additional values not provided by lstat_buf */
	g_free(obj->name);
	obj->name = STRDUP(g_basename(dpath));

	g_free(obj->original_path);
	obj->original_path = STRDUP(parent_path);

	obj->deleted_time = deleted_time;

#ifdef S_ISLNK
        if(S_ISLNK(m))
            obj->link_target = EDVGetLinkTarget(dpath);
#endif

	/* Add an entry to the recycled objects index file */
	index = EDVRecBinIndexAdd(index_path, obj);
	if(index != 0)
	{
	    /* Recycle this object */
	    if(EDVRecBinDiskObjectDelete(
		index_path,
		index,
		dpath,
		progress_cb, data
	    ))
	    {
		/* An error occured while recycling the object */
		last_error = EDVRecBinIndexGetError();
		EDVRecBinIndexRemove(
		    index_path, index
		);
		index = 0;
	    }
	    /* Notify Endeavour about the recycled object being
	     * recycled?
	     */
	    else if(notify)
	    {
		EDVNotifyQueueObjectRemoved(ctx, dpath);
		EDVNotifyQueueRecycledObjectAdded(ctx, index);
	    }
	}
	else
	{
	    last_error = EDVRecBinIndexGetError();
	}

	EDVRecycledObjectDelete(obj);
	g_free(parent_path);
	g_free(dpath);
	g_free(cwd);

	return(index);
}

/*
 *	Recovers a recycled object.
 *
 *	The index specifies the index of the recycled object to
 *	recover.
 *
 *	The alternate_recovery_path specifies the full path to the
 *	directory that is to be used as the alternate recovery location
 *	for the recycled object. If alternate_recovery_path is NULL
 *	then the recycled object's original location will be used
 *	as the recovery location.
 *
 *	If notify is set to TRUE then a "object_added_notify" and
 *	"recycled_object_removed_notify" will be queued on the ctx
 *	(only on success).
 *
 *	Returns 0 on success or non-zero on error.
 */
gint EDVRecover(
	edv_context_struct *ctx, const guint index,
	const gchar *alternate_recovery_path,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer data, const gulong i, const gulong m
	),
	gpointer data
)
{
	gint status;
	gchar *dpath, *original_path;
	const gchar *index_path = (ctx != NULL) ?
	    ctx->recycled_index_file : NULL;

	last_error = NULL;

	if(STRISEMPTY(index_path))
	{
	    last_error = "Bad value";
	    return(-1);
	}

	if(EDVRecycledObjectStat(
	    ctx, index, &original_path, NULL
	))
	    return(-1);

	/* Alternate recovery path specified? */
	if(alternate_recovery_path != NULL)
	{
	    gchar	*cwd = EDVGetCWD(),
			*parent = EDVEvaluatePath(cwd, alternate_recovery_path);
	    if(parent == NULL)
	    {
		last_error = "Unable to evaluate the alternate recovery location";
		g_free(cwd);
		return(-1);
	    }
	    dpath = g_strconcat(
		parent,
		G_DIR_SEPARATOR_S,
		g_basename(original_path),
		NULL
	    );
	    g_free(cwd);
	    g_free(parent);
	}
	else
	{
	    /* Use the original location and name */
	    dpath = NULL;
	}

	/* Recover the recycled object */
	status = EDVRecBinDiskObjectRecover(
	    index_path,
	    index,		/* Index of the recycled object to recover */
	    dpath,		/* Full path to the recovered object */
	    progress_cb, data
	);
	if(status != 0)
	{
	    /* An error occured while recovering the object */
	    last_error = EDVRecBinIndexGetError();
	}
	else
	{
	    /* Success, now remove the recycled object entry from the
	     * recycled objects index file
	     */
	    EDVRecBinIndexRemove(
		index_path, index
	    );

	    /* Notify Endeavour about the recycled object being
	     * recovered?
	     */
	    if(notify)
	    {
		const gchar *new_path = (dpath != NULL) ? dpath : original_path;
		EDVNotifyQueueObjectAdded(ctx, new_path);
		EDVNotifyQueueRecycledObjectRemoved(ctx, index);
	    }
	}

	g_free(original_path);
	g_free(dpath);

	return(status);
}

/*
 *	Purges a recycled object.
 *
 *	The index specifies the recycled object to purge.
 *
 *	If notify is set to TRUE then a "recycled_object_removed_notify"
 *	will be queued on the ctx (only on success).
 *
 *	Returns 0 on success or non-zero on error.
 */
gint EDVPurge(
	edv_context_struct *ctx,
	const guint index,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer data, const gulong i, const gulong m
	),
	gpointer data
)
{
	gint status;
	const gchar *index_path = (ctx != NULL) ?
	    ctx->recycled_index_file : NULL;

	last_error = NULL;

	if(STRISEMPTY(index_path))
	{
	    last_error = "Bad value";
	    return(-1);
	}

	/* Purge the recycled object */
	status = EDVRecBinDiskObjectPurge(
	    index_path,
	    index,
	    progress_cb, data
	);
	if(status != 0)
	{
	    /* An error occured while purging the object */
	    last_error = EDVRecBinIndexGetError();
	}
	else
	{
	    /* Success, now remove the recycled object entry from the
	     * recycled objects index file
	     */
	    EDVRecBinIndexRemove(
		index_path, index
	    );

	    /* Notify Endeavour about the recycled object being
	     * purged?
	     */
	    if(notify)
		EDVNotifyQueueRecycledObjectRemoved(ctx, index);
	}

	return(status);
}
