#include <string.h>
#include <math.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "pie_chart.h"


static void PieChartDrawArc(
	GdkPixmap *pixmap, GdkGC *gc,
	const gint x, const gint y, const gint width, const gint height,
	const gfloat start_angle,     /* 0.0 to 1.0 */
	const gfloat end_angle,       /* 0.0 to 1.0 */
	const gboolean outline_circumferance,
	const gboolean outline_radius, const gboolean outline_height
);
static void PieChartDrawPieIterate(
	pie_chart_struct *pc, GtkStyle *style, GdkPixmap *pixmap,
	const GtkStateType state,
	const gint x, const gint y, const gint width, const gint height,
	const gboolean outline_circumference, const gboolean outline_radius,
	const gboolean outline_height,
	const gboolean shaded, const gboolean shadow
);
static void PieChartCreateValueLabel(
	pie_chart_struct *pc,
	GdkColor *c, const gchar *type_label, const gchar *value_label,
	GtkWidget **type_label_hbox_rtn, GtkWidget **value_label_hbox_rtn,
	GtkWidget **drawing_area_rtn
);
static void PieChartRealize(pie_chart_struct *pc);
static void PieChartRedrawPixmap(pie_chart_struct *pc);

static gint PieChartValueLabelExposeCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
);
static gint PieChartExposeCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
);


/* Pie Chart Value */
pie_chart_value_struct *PieChartValueNew(void);
void PieChartValueDelete(pie_chart_value_struct *v);
gint PieChartValueAdd(
	pie_chart_struct *pc,
	GtkAdjustment *adj, GdkColor *c,
	const gchar *type_label, const gchar *value_label
);
void PieChartValueSet(
	pie_chart_struct *pc, const gint value_num,
	GtkAdjustment *adj, GdkColor *c,
	const gchar *type_label, const gchar *value_label
);
void PieChartValueRemove(
	pie_chart_struct *pc, const gint value_num
);
void PieChartClear(pie_chart_struct *pc);

/* Pie Chart */
pie_chart_struct *PieChartNew(
	const gint width, const gint height,
	GtkAdjustment *adj,
	GdkColor *c,
	const gchar *title, const gchar *footer,
	const gchar *base_type_label, const gchar *base_value_label
);
void PieChartMap(pie_chart_struct *pc);
void PieChartUnmap(pie_chart_struct *pc);
void PieChartDelete(pie_chart_struct *pc);


#ifndef PI
# define PI	3.14159265
#endif

#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define PIE_CHART_PIE_HEIGHT	10


/*
 *	Draws a segment of a pie arc.
 *
 *	Angles are in units from 0.0 to 1.0 starting at the 12 o'clock
 *	position being 0.0.
 */
static void PieChartDrawArc(
	GdkPixmap *pixmap, GdkGC *gc,
	const gint x, const gint y, const gint width, const gint height,
	const gfloat start_angle,	/* 0.0 to 1.0 */
	const gfloat end_angle,		/* 0.0 to 1.0 */
	const gboolean outline_circumferance,
	const gboolean outline_radius, const gboolean outline_height
)
{
	gfloat	_start_angle = start_angle,
		delta_angle;

	if((pixmap == NULL) || (gc == NULL))
	    return;

	/* Calculate delta angle */
	delta_angle = _start_angle - end_angle;
	if(delta_angle == 0.0f)
	    return;

	/* Outline radius? */
	if(outline_radius &&
	   ((_start_angle != 0.0f) || (end_angle != 1.0f))
	)
	{
	    gfloat	ostart_angle = _start_angle,
			oend_angle = end_angle;
	    gint	rx = (width / 2),
			ry = (height / 2),
			cx = x + rx,
			cy = y + ry;

	    /* Sanitize outline start and end angles */
	    while(ostart_angle > 1.0f)
		ostart_angle -= 1.0f;
	    while(ostart_angle < 0.0f)
		ostart_angle += 1.0f;

	    while(oend_angle > 1.0f)
		oend_angle -= 1.0f;
	    while(oend_angle < 0.0f)
		oend_angle += 1.0f;

	    gdk_draw_line(
		pixmap, gc,
		cx, cy,
		cx + (rx * sin(ostart_angle * 2.0f * PI)),
		cy + (ry * -cos(ostart_angle * 2.0f * PI))
	    );
	    gdk_draw_line(
		pixmap, gc,
		cx, cy,
		cx + (rx * sin(oend_angle * 2.0f * PI)),
		cy + (ry * -cos(oend_angle * 2.0f * PI))
	    );
	}

	/* Outline height? */
	if(outline_height)
	{
	    gfloat      ostart_angle = _start_angle,
			oend_angle = end_angle;
	    gint        rx = (width / 2),
			ry = (height / 2),
			cx = x + rx,
			cy = y + ry;

	    /* Sanitize outline start and end angles */
	    while(ostart_angle > 1.0f)
		ostart_angle -= 1.0f;
	    while(ostart_angle < 0.0f)
		ostart_angle += 1.0f;

	    while(oend_angle > 1.0f)
		oend_angle -= 1.0f;
	    while(oend_angle < 0.0f)
		oend_angle += 1.0f;

	    /* Draw height lines dividing pie segments */
	    if((ostart_angle > 0.25f) && (ostart_angle < 0.75f))
		gdk_draw_line(
		    pixmap, gc,
		    cx + (rx * sin(ostart_angle * 2.0f * PI)),
		    cy + (ry * -cos(ostart_angle * 2.0f * PI)),
		    cx + (rx * sin(ostart_angle * 2.0f * PI)),
		    cy + (ry * -cos(ostart_angle * 2.0f * PI)) +
			PIE_CHART_PIE_HEIGHT
		);
	    if((oend_angle > 0.25f) && (oend_angle < 0.75f))
		gdk_draw_line(
		    pixmap, gc,
		    cx + (rx * sin(oend_angle * 2.0f * PI)),
		    cy + (ry * -cos(oend_angle * 2.0f * PI)),
		    cx + (rx * sin(oend_angle * 2.0f * PI)),
		    cy + (ry * -cos(oend_angle * 2.0f * PI)) +
			PIE_CHART_PIE_HEIGHT
		);

	    /* Draw height lines at far left and right edges */
	    gdk_draw_line(
		pixmap, gc,
		x, y + ry,
		x, y + ry + PIE_CHART_PIE_HEIGHT - 1
	    );
	    gdk_draw_line(
		pixmap, gc,
		x + width, y + ry,
		x + width, y + ry + PIE_CHART_PIE_HEIGHT - 1
	    );
	}

	/* Convert angles to degrees and make it relative to the 3
	 * o'clock position
	 */
	_start_angle = 90.0f - (_start_angle * 360.0f);
	while(_start_angle < 0.0f)
	    _start_angle += 360.0f;
	while(_start_angle >= 360.0f)
	    _start_angle -= 360.0f;

	delta_angle *= 360.0f;

	gdk_draw_arc(
	    pixmap, gc, !outline_circumferance,
	    x, y, width, height,
	    (gint)(_start_angle * 64.0f),
	    (gint)(delta_angle * 64.0f)
	);
}

/*
 *	Creates a new value label and parents it to the given pie chart's
 *	value_labels_vbox.
 *
 *	Returns the pointer to the hbox containing the new label.
 */
static void PieChartCreateValueLabel(
	pie_chart_struct *pc,
	GdkColor *c, const gchar *type_label, const gchar *value_label,
	GtkWidget **type_label_hbox_rtn, GtkWidget **value_label_hbox_rtn,
	GtkWidget **drawing_area_rtn
)
{
	const gint	border_minor = 2;
	gint da_width = 12, da_height = 12;
	GdkFont *font;
	GtkStyle *style;
	GtkWidget *w, *parent, *parent2;
	GtkWidget *type_labels_vbox, *value_labels_vbox;


	/* Reste returns */
	if(type_label_hbox_rtn != NULL)
	    *type_label_hbox_rtn = NULL;
	if(value_label_hbox_rtn != NULL)
	    *value_label_hbox_rtn = NULL;
	if(drawing_area_rtn != NULL)
	    *drawing_area_rtn = NULL;

	if((pc == NULL) || STRISEMPTY(type_label))
	    return;

	w = pc->toplevel;
	style = (w != NULL) ? gtk_widget_get_style(w) : NULL;
	if(style == NULL)
	    return;

	font = style->font;

	/* Get parent for labels */
	type_labels_vbox = parent = pc->type_labels_vbox;
	if(parent == NULL)
	    return;

	/* Hbox for labels */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;
	if(type_label_hbox_rtn != NULL)
	    *type_label_hbox_rtn = w;

	/* Frame for color */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Drawing area to display color */
	if(font != NULL)
	{
	    da_height = MAX(font->ascent + font->descent - 2, 2);
	    da_width = da_height;
	}
	w = gtk_drawing_area_new();
	gtk_widget_set_usize(w, da_width, da_height);
	gtk_widget_add_events(w, GDK_EXPOSURE_MASK);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(PieChartValueLabelExposeCB), pc
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	if(drawing_area_rtn != NULL)
	    *drawing_area_rtn = w;

	/* Type label */
	w = gtk_label_new(type_label);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Value label */
	value_labels_vbox = parent = pc->value_labels_vbox;
	if((value_label != NULL) && (parent != NULL))
	{
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;
	    if(value_label_hbox_rtn != NULL)
		*value_label_hbox_rtn = w;

	    w = gtk_alignment_new(0.0f, 0.5f, 0.0f, 0.0f);
	    gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    w = gtk_label_new(value_label);
	    gtk_container_add(GTK_CONTAINER(parent2), w);
	    gtk_widget_show(w);
	}
}

/*
 *	Checks if the given pie chart is not realized, if it is not
 *	realized then all colors and pixmap will be created and the pie
 *	chart will be marked as realized.
 */
static void PieChartRealize(pie_chart_struct *pc)
{
	gint width, height;
	GdkColor *c;
	GdkColormap *colormap;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GList *glist;
	GtkWidget *w;
	pie_chart_value_struct *v;

	if(pc == NULL)
	    return;

	w = pc->drawing_area;
	if(w == NULL)
	    return;

	/* Pie chart already realized or drawing area GtkWidget
	 * not realized?
	 */
	if(pc->realized || !GTK_WIDGET_REALIZED(w))
	    return;

	window = w->window;
	if(window == NULL)
	    return;

	gdk_window_get_size(window, &width, &height);

	/* Get the colormap */
	colormap = pc->colormap;
	if(colormap == NULL)
	{
	    pc->colormap = colormap = gdk_window_get_colormap(window);
	    if(colormap != NULL)
		gdk_colormap_ref(colormap);
	}
	if(colormap == NULL)
	{
	    pc->colormap = colormap = gtk_widget_get_default_colormap();
	    if(colormap != NULL)
		gdk_colormap_ref(colormap);
	}
	if(colormap == NULL)
	    return;

	/* Create the graphics context */
	if(pc->gc == NULL)
	    pc->gc = GDK_GC_NEW();

	/* Create the offscreen GdkPixmap buffer */
	pixmap = pc->pixmap;
	if(pixmap == NULL)
	    pc->pixmap = pixmap = gdk_pixmap_new(
		window, width, height, -1
	    );

	/* Set the pie base colors */
	c = &pc->c;
	GDK_COLORMAP_ALLOC_COLOR(colormap, c)
	c = &pc->c_shade;
	GDK_COLORMAP_ALLOC_COLOR(colormap, c)
	c = &pc->c_shadow;
	GDK_COLORMAP_ALLOC_COLOR(colormap, c)
	/* Create base value label */
	PieChartCreateValueLabel(
	    pc, &pc->c, pc->base_type_label, pc->base_value_label,
	    NULL, NULL, NULL
	);

	/* Realize all the values */
	for(glist = pc->values_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    v = PIE_CHART_VALUE(glist->data);
	    if(v == NULL)
		continue;

	    /* Allocate color of this value */
	    c = &v->c;
	    GDK_COLORMAP_ALLOC_COLOR(colormap, c);
	    c = &v->c_shade;
	    GDK_COLORMAP_ALLOC_COLOR(colormap, c);

	    /* Create label widget for this value */
	    PieChartCreateValueLabel(
		pc, &v->c, v->type_label, v->value_label,
		&v->type_label_hbox, &v->value_label_hbox,
		&v->drawing_area
	    );
	}

	/* Mark as realized */
	pc->realized = TRUE;
}


/*
 *	Draws one iteration of a complete pie disk at the given
 *	coordinates.
 */
static void PieChartDrawPieIterate(
	pie_chart_struct *pc, GtkStyle *style, GdkPixmap *pixmap,
	const GtkStateType state,
	const gint x, const gint y, const gint width, const gint height,
	const gboolean outline_circumference, const gboolean outline_radius,
	const gboolean outline_height,
	const gboolean shaded, const gboolean shadow
)
{
	const gboolean outline = (outline_circumference || outline_radius || outline_height) ?
	    TRUE : FALSE;
	gfloat values_range, total_range, values_range_coeff;
	GList *glist;
	GdkGC *gc;
	GtkAdjustment *adj;
	pie_chart_value_struct *v;

	if((pc == NULL) || (style == NULL) || (pixmap == NULL) ||
	   (width <= 0) || (height <= 0)
	)
	    return;

	if(outline)
	    gc = style->fg_gc[state];
	else
	    gc = pc->gc;
	if(gc == NULL)
	    return;

	/* Set foreground color only if not drawing for an outline */
	if(!outline)
	{
	    if(shaded)
		gdk_gc_set_foreground(gc, &pc->c_shade);
	    else if(shadow)
		gdk_gc_set_foreground(gc, &pc->c_shadow);
	    else
		gdk_gc_set_foreground(gc, &pc->c);
	    gdk_gc_set_fill(gc, GDK_SOLID);
	}

	/* Calculate total range of all values */
	values_range = 0.0f;
	for(glist = pc->values_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    v = PIE_CHART_VALUE(glist->data);
	    if(v == NULL)
		continue;

	    adj = v->adj;
	    values_range += (adj != NULL) ? (adj->upper - adj->lower) : 0.0f;
	}

	/* Calculate absolute total range */
	adj = pc->adj;
	total_range = (adj != NULL) ? (adj->upper - adj->lower) : 0.0f;

	/* Check if more of the values range is visible on the `lower'
	 * side
	 *
	 * Note that angles are in units from 0.0 to 1.0 starting at
	 * the 12 o'clock position being 0.0 going clockwise
	 */
	values_range_coeff = (total_range > 0.0f) ?
	    (values_range / total_range) : 0.0f;
	if(values_range_coeff < 0.5f)
	{
	    gfloat value_pos_coeff = 0.5 + (values_range_coeff / 2.0);
	    gfloat value_pos_delta;


	    /* Draw base disk */
	    PieChartDrawArc(
		pixmap, gc, x, y, width, height,
		0.0, 1.0,
		outline_circumference, outline_radius,
		outline_height
	    );
	    /* Draw each value only if this disk is not a shadow */
	    if(!shadow)
	    {
		/* Draw each value */
		for(glist = pc->values_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    v = PIE_CHART_VALUE(glist->data);
		    if(v == NULL)
			continue;

		    adj = v->adj;
		    if((adj != NULL) && (total_range > 0.0f))
			value_pos_delta = (adj->upper - adj->lower) /
			    total_range;
		    else
			value_pos_delta = 0.0f;

		    if(!outline)
		    {
			if(shaded)
			    gdk_gc_set_foreground(gc, &v->c_shade);
			else
			    gdk_gc_set_foreground(gc, &v->c);
			gdk_gc_set_fill(gc, GDK_SOLID);
		    }

		    PieChartDrawArc(
			pixmap, gc, x, y, width, height,
			value_pos_coeff, value_pos_coeff - value_pos_delta,
			outline_circumference, outline_radius,
			outline_height
		    );

		    value_pos_coeff -= value_pos_delta;
		}
	    }
	}
	else
	{
	    gfloat	value_pos_coeff = 1.0f - (values_range_coeff / 2.0f),
			value_pos_delta;

	    /* Draw base disk */
	    PieChartDrawArc(
		pixmap, gc, x, y, width, height,
		0.0f, 1.0f,
		outline_circumference, outline_radius,
		outline_height
	    );

	    /* Draw each value only if this disk is not a shadow */
	    if(!shadow)
	    {
		/* Draw each value */
		for(glist = pc->values_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    v = PIE_CHART_VALUE(glist->data);
		    if(v == NULL)
			continue;

		    adj = v->adj;
		    if((adj != NULL) && (total_range > 0.0f))
			value_pos_delta = (adj->upper - adj->lower) /
			    total_range;
		    else
			value_pos_delta = 0.0f;

		    if(!outline)
		    {
			if(shaded)
			    gdk_gc_set_foreground(gc, &v->c_shade);
			else
			    gdk_gc_set_foreground(gc, &v->c);
			gdk_gc_set_fill(gc, GDK_SOLID);
		    }

		    PieChartDrawArc(
			pixmap, gc, x, y, width, height,
			value_pos_coeff, value_pos_coeff + value_pos_delta,
			outline_circumference, outline_radius,
			outline_height
		    );

		    value_pos_coeff += value_pos_delta;
		}

	    }
	}
}

/*
 *	Redraws the off screen buffer pixmap.
 *
 *	The pie chart will be realized as needed.
 */
static void PieChartRedrawPixmap(pie_chart_struct *pc)
{
	GtkStateType state;
	gint x, y, width, height, pwidth, pheight;
	GdkGC *gc;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GtkStyle *style;
	GtkWidget *w;

	if(pc == NULL)
	    return;

	w = pc->drawing_area;
	if(w == NULL)
	    return;

	if(!GTK_WIDGET_REALIZED(w) || !GTK_WIDGET_VISIBLE(w))
	    return;

	window = w->window;
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return;

	/* Realize the pie chart as needed */
	if(!pc->realized)
	    PieChartRealize(pc);

	/* Get the offscreen buffer pixmap that we will be drawing to,
	 * this pixmap should be created in PieChartRealize()
	 */
	pixmap = pc->pixmap;
	if(pixmap == NULL)
	    return;

	state = GTK_WIDGET_STATE(w);
	gdk_window_get_size(pixmap, &width, &height);

	/* Get the graphics context */
	gc = pc->gc;
	if(gc == NULL)
	    return;

	/* Begin drawing the pie chart */

	/* Draw background */
	gtk_style_apply_default_background(
	    style, pixmap, FALSE, state,
	    NULL,
	    0, 0, width, height
	);


	/* Begin drawing pie chart */

	/* Shadow */
	x = 4;
	y = PIE_CHART_PIE_HEIGHT + 2;
	pwidth = width - 6;
	pheight = height - PIE_CHART_PIE_HEIGHT - 4;
	if(pc->shadows)
	    PieChartDrawPieIterate(
		pc, style, pixmap,
		state, x, y, pwidth, pheight,
		FALSE,		/* Outline circumference */
		FALSE, FALSE,	/* Outline radius and height */
		FALSE,		/* Is shaded */
		TRUE		/* Is shadow */
	    );

	/* Base outline */
	x = 0;
	y = PIE_CHART_PIE_HEIGHT;
	if(pc->outline)
	{
	    PieChartDrawPieIterate(
		pc, style, pixmap,
		state, x, y, pwidth, pheight,
		FALSE,		/* Outline circumference */
		FALSE, FALSE,	/* Outline radius and height */
		TRUE,		/* Is shaded */
		FALSE		/* Is shadow */
	    );
	    PieChartDrawPieIterate(
		pc, style, pixmap,
		state, x, y, pwidth, pheight,
		TRUE,		/* Outline circumference */
		FALSE, FALSE,	/* Outline radius and height */
		TRUE,		/* Is shaded */
		FALSE		/* Is shadow */
	    );
	    y--;
	}

	while(y > 0)
	{
	    PieChartDrawPieIterate(
		pc, style, pixmap,
		state, x, y, pwidth, pheight,
		FALSE, 		/* Outline circumference */
		FALSE, FALSE,	/* Outline radius and height */
		TRUE,		/* Is shaded */
		FALSE		/* Is shadow */
	    );
	    y--;
	}

	y = 0;
	PieChartDrawPieIterate(
	    pc, style, pixmap,
	    state, x, y, pwidth, pheight,
	    FALSE,		/* Outline circumference */
	    FALSE, FALSE,	/* Outline radius and height */
	    FALSE,		/* Is shaded */
	    FALSE		/* Is shadow */
	);

	if(pc->outline)
	    PieChartDrawPieIterate(
		pc, style, pixmap,
		state, x, y, pwidth, pheight,
		TRUE,		/* Outline circumference */
		TRUE, TRUE,	/* Outline radius and height */
		FALSE,		/* Is shaded */
		FALSE		/* Is shadow */
	    );


}

/*
 *	Value label drawing area "expose_event" signal callback.
 */
static gint PieChartValueLabelExposeCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	gint width, height;
	GList *glist;
	GdkColor *c_bg;
	GdkGC *gc;
	GdkWindow *window;
	GtkStateType state;
	GtkStyle *style;
	pie_chart_value_struct *v;
	pie_chart_struct *pc = PIE_CHART(data);
	if((widget == NULL) || (pc == NULL))
	    return(FALSE);

	/* Find the value that contains this GtkDrawingArea */
	v = NULL;
	for(glist = pc->values_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    v = PIE_CHART_VALUE(glist->data);
	    if(v == NULL)
		continue;

	    if(v->drawing_area == widget)
		break;
	}
	if(glist == NULL)
	    v = NULL;
	if(v != NULL)
	{
	    c_bg = &v->c;
	}
	else
	{
	    c_bg = &pc->c;
	}

	state = GTK_WIDGET_STATE(widget);
	window = widget->window;
	style = gtk_widget_get_style(widget);
	if((window == NULL) || (style == NULL))
	    return(FALSE);

	gdk_window_get_size(window, &width, &height);
	
	gc = pc->gc;
	if(gc != NULL)
	{
	    gdk_gc_set_foreground(gc, c_bg);
	    gdk_gc_set_fill(gc, GDK_SOLID);
	    gdk_draw_rectangle(
		(GdkDrawable *)window, gc, TRUE,
		0, 0, width, height
	    );
	}
#if 0
	gc = style->fg_gc[state];
	if(gc != NULL)
	{
	    gdk_draw_rectangle(
		(GdkDrawable *)window, gc, FALSE,
		0, 0, width - 1, height - 1
	    );
	}
#endif

	return(TRUE);
}

/*
 *	Drawing area "expose_event" signal callback.
 */
static gint PieChartExposeCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	gint status = FALSE;
	GtkStateType state;
	gint width, height;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GdkDrawable *drawable;
	GtkStyle *style;
	pie_chart_struct *pc = PIE_CHART(data);
	if((widget == NULL) || (pc == NULL))
	    return(status);

	state = GTK_WIDGET_STATE(widget);
	window = widget->window;
	style = gtk_widget_get_style(widget);
	if((window == NULL) || (style == NULL))
	    return(status);

	/* Get pixmap for drawing area
	 *
	 * If the pixmap is NULL then it implies it has not been
	 * realized or redrawn, so realize and redraw it as needed
	 */
	pixmap = pc->pixmap;
	if(pixmap == NULL)
	{
	    PieChartRedrawPixmap(pc);
	    pixmap = pc->pixmap;
	}
	if(pixmap == NULL)
	    return(status);

	drawable = (GdkDrawable *)pixmap;
	gdk_window_get_size(drawable, &width, &height);

	/* Send drawable to window if drawable is not the window */
	if(drawable != window)
	    gdk_draw_pixmap(
		window, style->fg_gc[state], drawable,
		0, 0, 0, 0, width, height
	    );

	status = TRUE;
	return(status);
}


/*
 *	Creates a new pie chart value.
 */
pie_chart_value_struct *PieChartValueNew(void)
{
	return(PIE_CHART_VALUE(g_malloc0(sizeof(pie_chart_value_struct))));
}

/*
 *	Deletes the pie chart value.
 */
void PieChartValueDelete(pie_chart_value_struct *v)
{
	GdkColormap *colormap;
	pie_chart_struct *pc;

	if(v == NULL)
	    return;

	/* Get the pie chart that this value is on */
	pc = v->pc;

	/* Unref the GtkAdjustment */
	if(v->adj != NULL)
	{
	    gtk_object_unref(GTK_OBJECT(v->adj));
	    v->adj = NULL;
	}

	/* Delete the GdkColors */
	colormap = (pc != NULL) ? pc->colormap : NULL;
	if(colormap != NULL)
	{
	    GDK_COLORMAP_FREE_COLOR(colormap, &v->c);
/*	    memset(&v->c, 0x00, sizeof(GdkColor)); */

	    GDK_COLORMAP_FREE_COLOR(colormap, &v->c_shade);
/*	    memset(&v->c_shade, 0x00, sizeof(GdkColor)); */
	}

	/* Label GtkHBox */
	if(v->type_label_hbox != NULL)
	{
	    gtk_widget_destroy(v->type_label_hbox);
	    v->type_label_hbox = NULL;
	}
	if(v->value_label_hbox != NULL)
	{
	    gtk_widget_destroy(v->value_label_hbox);
	    v->value_label_hbox = NULL;
	}
/*	v->drawing_area = NULL; */

	g_free(v->type_label);
	g_free(v->value_label);

	g_free(v);
}


/*
 *	Adds a new value to the pie chart.
 */
gint PieChartValueAdd(
	pie_chart_struct *pc,
	GtkAdjustment *adj, GdkColor *c,
	const gchar *type_label, const gchar *value_label
)
{
	gint i;
	GdkColor *tc;
	GdkColormap *colormap;
	pie_chart_value_struct *v;

	if((pc == NULL) || (adj == NULL) || (c == NULL))
	    return(-2);

	/* Create a new pie chart value */
	v = PieChartValueNew();
	if(v == NULL)
	    return(-3);

	i = g_list_length(pc->values_list);

	/* Add the new pie chart value to the values list */
	pc->values_list = g_list_append(pc->values_list, v);

	/* Set the pie chart that this value is on */
	v->pc = pc;

	/* Set the GtkAdjustment */
	gtk_object_ref(GTK_OBJECT(adj));
	v->adj = adj;

	/* Set the GdkColor */
	tc = &v->c;
	memcpy(tc, c, sizeof(GdkColor));

	tc = &v->c_shade;
	tc->red = c->red / 2;
	tc->green = c->green / 2;
	tc->blue = c->blue / 2;

	colormap = pc->colormap;
	if(colormap != NULL)
	{
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &v->c);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &v->c_shade);
	}

	v->type_label = STRDUP(type_label);
	v->value_label = STRDUP(value_label);

	return(i);
}

/*
 *	Sets the value on the pie chart.
 */
void PieChartValueSet(
	pie_chart_struct *pc, const gint value_num,
	GtkAdjustment *adj, GdkColor *c,
	const gchar *type_label, const gchar *value_label
)
{
	GdkColormap *colormap;
	pie_chart_value_struct *v;

	if((pc == NULL) || (adj == NULL) || (c == NULL))
	    return;

	v = PIE_CHART_VALUE(g_list_nth_data(pc->values_list, value_num));
	if(v == NULL)
	    return;

	/* Add refcount to given adjustment, remove refcount from existing
	 * adjustment and assign given adjustment.
	 */
	gtk_object_ref(GTK_OBJECT(adj));
	if(v->adj != NULL)
	    gtk_object_unref(GTK_OBJECT(v->adj));
	v->adj = adj;

	/* Delete old color if colormap is available, set new color
	 * and allocate it if colormap is available
	 */
	colormap = pc->colormap;
	if(colormap != NULL)
	{
	    GdkColor *tc;

	    tc = &v->c;
	    GDK_COLORMAP_FREE_COLOR(colormap, tc);
	    memcpy(tc, c, sizeof(GdkColor));
	    GDK_COLORMAP_ALLOC_COLOR(colormap, tc);

	    tc = &v->c_shade;
	    GDK_COLORMAP_FREE_COLOR(colormap, tc);
	    tc->red = c->red / 2;
	    tc->green = c->green / 2;
	    tc->blue = c->blue / 2;
	    GDK_COLORMAP_ALLOC_COLOR(colormap, tc);
	}
	else
	{
	    GdkColor *tc;

	    tc = &v->c;
	    memcpy(tc, c, sizeof(GdkColor));

	    tc = &v->c_shade;
	    tc->red = c->red / 2;
	    tc->green = c->green / 2;
	    tc->blue = c->blue / 2;
	}

	/* Set the new label text */
	g_free(v->type_label);
	v->type_label = STRDUP(type_label);

	g_free(v->value_label);
	v->value_label = STRDUP(value_label);

/* Need to recreate pixmap */
}

/*
 *	Removes the value from the pie chart.
 */
void PieChartValueRemove(
	pie_chart_struct *pc, const gint value_num
)
{
	pie_chart_value_struct *v;

	if(pc == NULL)
	    return;

	v = PIE_CHART_VALUE(g_list_nth_data(pc->values_list, value_num));
	if(v == NULL)
	    return;

	PieChartValueDelete(v);
	g_list_remove(pc->values_list, v);
}

/*
 *	Removes all values on the given pie chart.
 */
void PieChartClear(pie_chart_struct *pc)
{
	if(pc == NULL)
	    return;

	if(pc->values_list != NULL)
	{
	    g_list_foreach(pc->values_list, (GFunc)PieChartValueDelete, NULL);
	    g_list_free(pc->values_list);
	    pc->values_list = NULL;
	}
}


/*
 *	Creates a new pie chart widget.
 */
pie_chart_struct *PieChartNew(
	const gint width, const gint height,
	GtkAdjustment *adj,
	GdkColor *c,
	const gchar *title, const gchar *footer,
	const gchar *base_type_label, const gchar *base_value_label
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GtkWidget *w, *parent, *parent2, *toplevel;
	pie_chart_struct *pc = PIE_CHART(g_malloc0(
	    sizeof(pie_chart_struct)
	));
	if(pc == NULL)
	    return(pc);

	pc->toplevel = toplevel = gtk_vbox_new(FALSE, border_minor);
	pc->realized = FALSE;
	pc->map_state = FALSE;
	pc->show_labels = TRUE;
	pc->shadows = TRUE;
	pc->outline = TRUE;
	pc->freeze_count = 0;
	pc->width = width;
	pc->height = height;
	pc->pixmap = NULL;
	if(c != NULL)
	{
	    GdkColor *tc;
	    tc = &pc->c;
	    memcpy(tc, c, sizeof(GdkColor));
	    tc = &pc->c_shade;
	    tc->red	= c->red	/ 2;
	    tc->green	= c->green	/ 2;
	    tc->blue	= c->blue	/ 2;
	    tc = &pc->c_shadow;
	    tc->red	= 0.5f * (gushort)-1;
	    tc->green	= 0.5f * (gushort)-1;
	    tc->blue	= 0.5f * (gushort)-1;
	}
	pc->base_type_label = STRDUP(base_type_label);
	pc->base_value_label = STRDUP(base_value_label);
	pc->gc = NULL;
	pc->colormap = NULL;
	pc->values_list = NULL;

	/* Claim the specified GtkAdjustment */
	pc->adj = adj;
	if(adj != NULL)
	    gtk_object_ref(GTK_OBJECT(adj));

	pc->freeze_count++;

	/* Toplevel GtkVBox */
	w = toplevel;
	parent = w;

	/* Create title label if title string is given */
	if(title != NULL)
	{
	    pc->title_label = w = gtk_label_new(title);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}


	/* Type and value labels GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Type labels GtkVBox */
	pc->type_labels_vbox = w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Center right column GtkHBox */
	w = gtk_hbox_new(TRUE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;
	/* Value labels GtkVBox */
	pc->value_labels_vbox = w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);


	/* Pie chart GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Pie chart GtkDrawingArea */
	pc->drawing_area = w = gtk_drawing_area_new();
	gtk_widget_set_usize(w, width, height);
	gtk_widget_add_events(w, GDK_EXPOSURE_MASK);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(PieChartExposeCB), pc
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);

	/* Create footer label if the title is specified */
	if(footer != NULL)
	{
	    pc->footer_label = w = gtk_label_new(footer);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}

	pc->freeze_count--;

	return(pc);
}

/*
 *	Maps the pie chart.
 */
void PieChartMap(pie_chart_struct *pc)
{
	if(pc == NULL)
	    return;

	gtk_widget_show(pc->toplevel);
	pc->map_state = TRUE;
}

/*
 *	Unmaps the pie chart.
 */
void PieChartUnmap(pie_chart_struct *pc)
{
	if(pc == NULL)
	    return;

	gtk_widget_hide(pc->toplevel);
	pc->map_state = FALSE;
}

/*
 *	Deletes the pie chart.
 */
void PieChartDelete(pie_chart_struct *pc)
{
	GdkColormap *colormap;

	if(pc == NULL)
	    return;

	pc->freeze_count++;

	/* Delete all the values */
	PieChartClear(pc);

	/* Delete all the colors */
	colormap = pc->colormap;
	GDK_COLORMAP_FREE_COLOR(colormap, &pc->c);
	GDK_COLORMAP_FREE_COLOR(colormap, &pc->c_shade);
	GDK_COLORMAP_FREE_COLOR(colormap, &pc->c_shadow);

	GTK_OBJECT_UNREF(pc->adj);

	GDK_PIXMAP_UNREF(pc->pixmap);

	gtk_widget_destroy(pc->toplevel);

	GDK_GC_UNREF(pc->gc);
	GDK_COLORMAP_UNREF(pc->colormap);

	g_free(pc->base_type_label);
	g_free(pc->base_value_label);

	pc->freeze_count--;

	g_free(pc);
}
