/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: options.cc,v 1.22.2.2 2003/09/28 20:41:54 dheck Exp $
 */
#include "enigma.hh"
#ifdef __MINGW32__
    #include <windows.h>
#endif
#include "lua.hh"
#include "options.hh"
#include "system.hh"
#include "game.hh"
#include "px/dict.hh"

#include <cassert>
#include <cstdio>
#include <cstdlib>
#include <string>

using namespace options;
using namespace std;

namespace options
{
    int    BitsPerPixel      = 16;
    double MouseSpeed        = 5.0;
    double MouseDamping      = 25.0;
    bool   WizardMode        = false;
    bool   FullScreen        = true;
    bool   ShowFPS           = false;
    bool   UseAlpha          = true;
    bool   Nograb            = false;
    bool   Nozoom            = true;
    bool   InGameMusic       = false;
    int    Difficulty        = enigma::DIFFICULTY_HARD ;
    int    VideoMode         = 0;
    int    LevelMenuPosition = 0;

    double SoundVolume      = 1.0;
    double MusicVolume      = 1.0;
    double StereoSeparation = 10.0;

    string MenuMusicFile  = "/sound/menu.s3m";
    string LevelMusicFile = "/sound/Emilie.xm";

    int SoundSet = 0;
    // 0   = 'enigma' for enigma, appropriate oxyd sound sets for diff. oxyd versions
    // 1   = 'enigma'
    // 2.. = OxydVersion-2

    bool SkipSolvedLevels = false;

    // not stored in ~/.enigmarc :

    bool LevelStatusChanged = false;
    bool MustRestart        = false;
    bool MustRestartLevel   = false;
}

namespace
{
    px::Dict<LevelStatus> level_status_dict;
}


LevelStatus::LevelStatus(int easy, int hard, int fin, time_t solv)
: par_easy(easy),
  par_hard(hard),
  finished(fin),
  solved_at(solv)
{
    if (!solved_at && fin>0)
        solved_at = time(0); // default to now
}

bool LevelStatus::operator != (const LevelStatus& other) const
{
    return !(par_easy == other.par_easy &&
             par_hard == other.par_hard &&
             finished == other.finished &&
             solved_at == other.solved_at);
}


static string
levelkey(const string &/*levelpack*/, const string &levelname)
{
    return string("#")+levelname;
}

/* Get the status of a particular level in a particular pack.
   Returns 0 if no record for this level exists. */
LevelStatus *
options::GetLevelStatus(const string &levelpack, const string &levelname)
{
    px::Dict<LevelStatus>::iterator i;
    i = level_status_dict.find (levelkey (levelpack, levelname));
    return (i != level_status_dict.end()) ? &i->second : 0;
}

/* Set the status of a particular level.  The previous entry (if
   available) for this level is discarded. */
void
options::SetLevelStatus(const string &levelpack, const string &levelname,
                        const LevelStatus &stat)
{
    string                          key = levelkey (levelpack, levelname);
    px::Dict<LevelStatus>::iterator i   = level_status_dict.find (key);

    if (i != level_status_dict.end()) { // status exists -> overwrite
        if (i->second != stat) {
            LevelStatusChanged = true;
        }
        i->second = stat;
    }
    else {
        level_status_dict.insert(key,stat);
        LevelStatusChanged = true;
    }
}

bool
options::SetLevelTime(const string &pack, const string &level, int difficulty, int time)
{
    SetLevelFinished(pack, level, difficulty);
    LevelStatus *stat = GetLevelStatus(pack, level);

    assert(stat);

    bool new_record = false;

    if (difficulty == enigma::DIFFICULTY_EASY) {
        if (stat->par_easy > time || stat->par_easy == -1) {
            stat->par_easy     = time;
            LevelStatusChanged = true;
            new_record = true;
        }
    }
    else if (difficulty == enigma::DIFFICULTY_HARD) {
        if (stat->par_hard > time || stat->par_hard == -1) {
            stat->par_hard     = time;
            LevelStatusChanged = true;
            new_record = true;
        }
    }
    return new_record;
}

void
options::SetLevelFinished(const string &pack, const string &level, int difficulty)
{
    assert(difficulty==enigma::DIFFICULTY_EASY || difficulty==enigma::DIFFICULTY_HARD);

    if (LevelStatus *stat = GetLevelStatus(pack, level)) {
        stat->finished  |= difficulty;
        stat->solved_at  = time(0);
    }
    else {
        SetLevelStatus(pack, level, LevelStatus(-1, -1, difficulty));
    }

    LevelStatusChanged = true;
}

/* Determine name of the user's personal configuration file. */
static string
Personal_ConfigurationFileName()
{
    string fname = "enigmarc.lua";

    if (getenv ("HOME") != 0)
        fname = sysdep::expand_path ("~/.enigmarc");

    return fname;
}

static string
System_ConfigurationFileName()
{
    return enigma::FindDataFile ("enigma_conf.lua");
}

#ifdef __MINGW32__

static string
Windows_ConfigurationFileName()
{
    typedef HRESULT (WINAPI *SHGETFOLDERPATH)( HWND, int, HANDLE, DWORD, LPTSTR );
#   define CSIDL_FLAG_CREATE 0x8000
#   define CSIDL_APPDATA 0x1A
#   define SHGFP_TYPE_CURRENT 0

    HINSTANCE shfolder_dll;
    SHGETFOLDERPATH SHGetFolderPath ;

    /* load the shfolder.dll to retreive SHGetFolderPath */
    if( ( shfolder_dll = LoadLibrary("shfolder.dll") ) != NULL )
    {
        SHGetFolderPath = (SHGETFOLDERPATH)GetProcAddress( shfolder_dll, "SHGetFolderPathA");
        if ( SHGetFolderPath != NULL )
        {
            TCHAR szPath[MAX_PATH] = "";

            /* get the "Application Data" folder for the current user */
            if( S_OK == SHGetFolderPath( NULL, CSIDL_APPDATA | CSIDL_FLAG_CREATE,
                                         NULL, SHGFP_TYPE_CURRENT, szPath) )
            {
                FreeLibrary( shfolder_dll);
                return string(szPath) + "/enigmarc.lua";
            }
        }
        FreeLibrary( shfolder_dll);
    }

    return Personal_ConfigurationFileName();
}

#endif


bool
options::Save ()
{
#ifdef __MINGW32__
    string fname = Windows_ConfigurationFileName();
#else
    string fname = Personal_ConfigurationFileName();
#endif
    FILE *fp = fopen(fname.c_str(), "wt");

    if (fp) {
        fprintf (fp, "options.MouseSpeed = %f\n",           MouseSpeed);
        fprintf (fp, "options.MouseDamping = %f\n",         MouseDamping);
        fprintf (fp, "options.FullScreen = %d\n",           FullScreen);
        fprintf (fp, "options.UseAlpha = %d\n",             UseAlpha);
        fprintf (fp, "options.SoundVolume = %f\n",          SoundVolume);
        fprintf (fp, "options.MusicVolume = %f\n",          MusicVolume);
        fprintf (fp, "options.InGameMusic = %d\n",          InGameMusic);
        fprintf (fp, "options.StereoSeparation = %f\n",     StereoSeparation);
        fprintf (fp, "options.MenuMusicFile = \"%s\"\n",    MenuMusicFile.c_str());
        fprintf (fp, "options.LevelMusicFile = \"%s\"\n",   LevelMusicFile.c_str());
        fprintf (fp, "options.SoundSet = \"%i\"\n",         SoundSet);
        fprintf (fp, "options.SkipSolvedLevels = %d\n",     SkipSolvedLevels);
        fprintf (fp, "options.Difficulty = %d\n",           Difficulty);
        fprintf (fp, "options.VideoMode = %d\n",            VideoMode);
        fprintf (fp, "options.LevelMenuPosition = %d\n",    LevelMenuPosition);

        // Save level information
        px::Dict<LevelStatus>::iterator i=level_status_dict.begin();
        for (; i!=level_status_dict.end(); ++i) {
            LevelStatus& ls = i->second;

            fprintf (fp, "options.LevelStat(\"%s\", {%d, %d, %d, %ld})\n",
                     i->first.c_str(),
                     ls.par_easy,
                     ls.par_hard,
                     ls.finished,
                     ls.solved_at);
        }

        fclose(fp);

        LevelStatusChanged = false;
        return true;
    }
    return false;
}

static bool
load_from_file (const char *fname)
{
    switch (lua_dofile (lua::state, fname)) {
    case 0:                     // Everything ok
    case LUA_ERRFILE:           // File not found is not fatal
        return true;
    default:
        return false;           // All other errors are
    }
}

bool
options::Load ()
{
    string fname;
    bool error = false;

    fname = System_ConfigurationFileName();
    error |= load_from_file (fname.c_str());

    fname = Personal_ConfigurationFileName();
    error |= load_from_file (fname.c_str());

#ifdef __MINGW32__
    fname  = Windows_ConfigurationFileName();
    error |= load_from_file (fname.c_str());
#endif

    LevelStatusChanged = false;

    return error;
}

