/*
 * GeeXboX Enna Media Center.
 * Copyright (C) 2005-2009 The Enna Project
 *
 * This file is part of Enna.
 *
 * Enna is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * Enna is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with Enna; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <dirent.h>

#include <Ecore_File.h>
#include <Ecore_Str.h>
#include <Edje.h>
#include <efreet/Efreet.h>
#include <Elementary.h>

#include "enna.h"
#include "enna_config.h"
#include "module.h"
#include "logs.h"
#include "view_list.h"
#include "utils.h"
#include "content.h"
#include "mainmenu.h"
#include "input.h"

#define ENNA_MODULE_NAME "games"

static void _play(void *data);
static void _parse_directory();
static void _create_menu();
static void _create_gui();

typedef enum _GAMES_STATE GAMES_STATE;

enum _GAMES_STATE
{
    MENU_VIEW,
    GAME_VIEW
};

typedef struct _Enna_Module_Games
{
    Evas *e;
    Evas_Object *o_edje;
    Evas_Object *o_menu;
    GAMES_STATE state;
    Enna_Module *em;
} Enna_Module_Games;

static Enna_Module_Games *mod;

/*****************************************************************************/
/*                              Games Helpers                                */
/*****************************************************************************/

static void _play(void *data)
{
    int ret;
    char* game = data;

    mod->state = GAME_VIEW;
    enna_log(ENNA_MSG_INFO, ENNA_MODULE_NAME, "starting game: %s", game);
    ret = system(game);
    enna_log(ENNA_MSG_INFO, ENNA_MODULE_NAME, "game stopped: %s", game);
    mod->state = MENU_VIEW;
}

static void _parse_directory(Evas_Object *list, const char *dir_path)
{
    struct dirent *dp;
    DIR *dir;

    if (!(dir = opendir(dir_path))) return;

    while ((dp = readdir(dir)))
    {
        Efreet_Desktop *desktop;
        char dsfile[4096];

        if (!ecore_str_has_extension(dp->d_name, "desktop")) continue;
        sprintf(dsfile, "%s/%s", dir_path, dp->d_name);
        desktop = efreet_desktop_get(dsfile);
        if ((desktop = efreet_desktop_get(dsfile)))
        {
            Eina_List *l;
            const char *cat;

            EINA_LIST_FOREACH(desktop->categories, l, cat)
            {
                if(!strncmp(cat, "Game", strlen("Game")))
                {
                    char *iconpath = NULL;
                    Enna_Vfs_File *item;

                    if (ecore_file_can_read(desktop->icon))
                    {
                        iconpath = desktop->icon;
                    } else {
                        Eina_List *theme_list;
                        Eina_List *l;
                        Efreet_Icon_Theme *theme;

                        theme_list = efreet_icon_theme_list_get();
                        EINA_LIST_FOREACH(theme_list, l, theme)
                        {
                            iconpath = efreet_icon_path_find((theme->name).name, desktop->icon, 64);
                            if(iconpath)
                                break;
                        }
                    }

                    item = calloc(1, sizeof(Enna_Vfs_File));
                    item->icon = (char*)eina_stringshare_add(iconpath);
                    item->label = (char*)eina_stringshare_add(desktop->name);
                    item->is_menu = 1;
                    enna_list_file_append(list, item, _play, desktop->exec);
                    break;
                }
            }
        }
        efreet_desktop_free(desktop);
    }
    closedir(dir);
}

static void _create_menu()
{
    Evas_Object *o;
    char gamesdir[4096];

    /* Create List */
    o = enna_list_add(enna->evas);
    edje_object_signal_emit(mod->o_edje, "menu,show", "enna");

    sprintf(gamesdir, "%s/.enna/games", enna_util_user_home_get());

    /* Populate list */
    _parse_directory(o, gamesdir);
    _parse_directory(o, "/usr/share/applications");

    enna_list_select_nth(o, 0);
    mod->o_menu = o;
    edje_object_part_swallow(mod->o_edje, "menu.swallow", o);
    edje_object_signal_emit(mod->o_edje, "menu,show", "enna");
}

static void _create_gui(void)
{
    /* Set default state */
    mod->state = MENU_VIEW;

    /* Create main edje object */
    mod->o_edje = edje_object_add(enna->evas);
    edje_object_file_set(mod->o_edje, enna_config_theme_get(), "activity/games");

    _create_menu();
}

/*****************************************************************************/
/*                         Private Module API                                */
/*****************************************************************************/

static void _class_init(int dummy)
{
    efreet_init();
}

static void _class_shutdown(int dummy)
{
    efreet_shutdown();
}

static void _class_show(int dummy)
{
    if (!mod->o_edje)
    {
        _create_gui();
        enna_content_append(ENNA_MODULE_NAME, mod->o_edje);
    }

    enna_content_select(ENNA_MODULE_NAME);
    edje_object_signal_emit(mod->o_edje, "module,show", "enna");
}

static void _class_hide(int dummy)
{
    edje_object_signal_emit(mod->o_edje, "module,hide", "enna");
}

static void _class_event(enna_input event)
{
    switch (mod->state)
    {
        case MENU_VIEW:
            switch (event)
            {
                case ENNA_INPUT_LEFT:
                case ENNA_INPUT_EXIT:
                    enna_content_hide();
                    enna_mainmenu_show();
                    break;
                case ENNA_INPUT_RIGHT:
                case ENNA_INPUT_OK:
                    _play(enna_list_selected_data_get(mod->o_menu));
                   break;
                default:
                   enna_list_input_feed(mod->o_menu, event);
            }
            break;
        default:
            break;
    }

}

static Enna_Class_Activity
class =
{
    ENNA_MODULE_NAME,
    9,
    N_("Games"),
    NULL,
    "icon/games",
    "background/games",
    {
        _class_init,
        NULL,
        _class_shutdown,
        _class_show,
        _class_hide,
        _class_event
    },
    NULL
};

/*****************************************************************************/
/*                          Public Module API                                */
/*****************************************************************************/

#ifdef USE_STATIC_MODULES
#undef MOD_PREFIX
#define MOD_PREFIX enna_mod_activity_games
#endif /* USE_STATIC_MODULES */

Enna_Module_Api ENNA_MODULE_API =
{
    ENNA_MODULE_VERSION,
    "activity_games",
    N_("System games"),
    "icon/games",
    N_("With this module you can play your games directly from enna"),
    "bla bla bla<br><b>bla bla bla</b><br><br>bla."
};

void
ENNA_MODULE_INIT(Enna_Module *em)
{
    if (!em)
        return;

    mod = calloc(1, sizeof(Enna_Module_Games));
    mod->em = em;
    em->mod = mod;

    /* Add activity */
    enna_activity_add(&class);
}

void
ENNA_MODULE_SHUTDOWN(Enna_Module *em)
{
    enna_activity_del(ENNA_MODULE_NAME);
    evas_object_del(mod->o_edje);
    ENNA_OBJECT_DEL(mod->o_menu);
    free(mod);
}
