#-------------------------------------------------------------------------------
#  
#  Defines the OMContactController class of the Enable 'om' (Object Model) 
#  package. 
#
#  The OMContactController class defines the base class for creating 
#  controllers used to connect and control the relationship between a canvas 
#  component contact (i.e. OMContact object) and the corresponding object 
#  element in the underlying object model. The controller mediates all 
#  interaction between events that occur on the canvas component contact and the 
#  object model object element.
#  
#  Written by: David C. Morrill
#  
#  Date: 01/27/2005
#  
#  (c) Copyright 2005 by Enthought, Inc.
#  
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:  
#-------------------------------------------------------------------------------

from om_contact \
    import OMContact
    
from om_component_controller \
    import OMComponentController

from om_menu_controller \
    import menu_controller

from enthought.traits.api \
    import HasPrivateTraits, Instance

#-------------------------------------------------------------------------------
#  'OMContactController' class:
#-------------------------------------------------------------------------------

class OMContactController ( HasPrivateTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------

    # The associated component controller:
    component_controller = Instance( OMComponentController )
    
    #---------------------------------------------------------------------------
    #  Returns whether or not a Contact can accept a specified Link object:  
    #---------------------------------------------------------------------------

    def can_accept_link ( self, contact, link ):
        """ Returns whether or not a Contact can accept a specified Link object.
        """
        return True

    #---------------------------------------------------------------------------
    #  Returns whether or not a Contact can accept a specified Contact object:  
    #---------------------------------------------------------------------------
    
    def can_accept_contact ( self, contact, other_contact ):
        """ Returns whether or not a Contact can accept a specified Contact 
            object.
        """
        return True
        
    #---------------------------------------------------------------------------
    #  Returns whether or not a Contact is allowed to begin a drag Link 
    #  operation:  
    #---------------------------------------------------------------------------
                
    def can_drag_link ( self, contact ):
        """ Returns whether or not a Contact is allowed to begin a drag Link 
            operation.
        """
        return True
        
    #---------------------------------------------------------------------------
    #  Begins a drag link operation by returning a Link for the specified
    #  Contact:
    #---------------------------------------------------------------------------
                
    def begin_drag_link ( self, contact ):
        """ Begins a drag link operation by returning a Link for the specified 
            Contact.
        """
        return None  # fixme: Implement this...
        
    #---------------------------------------------------------------------------
    #  Ends a drag link operation by completing the link between the two
    #  specified contacts:
    #---------------------------------------------------------------------------
    
    def end_drag_link ( self, contact1, contact2 ):
        """ Ends a drag link operation by completing the link between the two
            specified contacts.
        """
        pass
        
    #---------------------------------------------------------------------------
    #  Handles an update event while performing a drag link operation:  
    #---------------------------------------------------------------------------
                
    def drag_update ( self, contact, event ):
        return None
        
    #---------------------------------------------------------------------------
    #  Handles the completion of an OMDragContact dragging operation:  
    #---------------------------------------------------------------------------
                
    def drag_contact_done ( self, drag_contact, old, event ):
        """ Handles the completion of an OMDragContact dragging operation.
        """
        contact         = drag_contact.contact
        contact.pointer = 'arrow'
        contact2        = drag_contact.drag_contact
        if contact2 is not None:
            x, y = drag_contact.anchor2
            if len( contact2._components_at( x, y, 3 ) ) == 0:  
                contact2 = None
        self.end_drag_link( contact, contact2 )
            
    #---------------------------------------------------------------------------
    #  Begins a drag operation when the 'control' key is pressed:  
    #---------------------------------------------------------------------------
                
    def begin_control_drag ( self, contact, event ):
        """ Begins a drag operation when the 'control' key is pressed.
        """
        return False
        
    #---------------------------------------------------------------------------
    #  Begins a drag operation when the 'shift' key is pressed:  
    #---------------------------------------------------------------------------
                
    def begin_shift_drag ( self, contact, event ):
        """ Begins a drag operation when the 'shift' key is pressed.
        """
        return False
        
    #---------------------------------------------------------------------------
    #  Begins a drag operation when the 'alt' key is pressed:  
    #---------------------------------------------------------------------------
                
    def begin_alt_drag ( self, contact, event ):
        """ Begins a drag operation when the 'alt' key is pressed.
        """
        return False
        
    #---------------------------------------------------------------------------
    #  Returns the name of the image to use for a specified Contact state:  
    #---------------------------------------------------------------------------
                
    def get_contact_image ( self, contact, state ):
        """ Returns the name of the image to use for a specified Contact state.
        """
        return contact.image_map.get( state, contact.image )
        
    #---------------------------------------------------------------------------
    #  Edits the properties of a specified contact:  
    #---------------------------------------------------------------------------
                
    def edit_contact ( self, contact ):
        """ Edits the properties of a specified contact.
        """
        contact.edit()
        
    #---------------------------------------------------------------------------
    #  Handles a contact being clicked:  
    #---------------------------------------------------------------------------
                
    def clicked_contact ( self, event ):
        """ Handles a contact being clicked.
        """
        pass
        
    #---------------------------------------------------------------------------
    #  Display the contact's context menu:
    #
    #  Note: The 'GetSizeTuple/PopupMenuXY' calls make this code wxPython
    #        specific
    #---------------------------------------------------------------------------
    
    def popup_menu ( self, contact, event ):
        menu = contact.menu
        if menu is not None:
            self._contact              = contact
            self._event                = event
            menu_controller.controller = self
            window                     = contact.window.control
            dx, dy                     = window.GetSizeTuple()
            window.PopupMenuXY( menu.create_menu( window, menu_controller ),
                                event.x - 10, dy - event.y - 10 )
            self._contact = self._event = None
                
