/***************************************************************************
 *   Copyright (C) 2006 by Niklas Knutsson                                 *
 *   nq@altern.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef SECURITY_H
#define SECURITY_H

#include <qstring.h>
#include <qdatetime.h>
#include <qmap.h>
#include <qptrlist.h>

#include "transaction.h"

class AssetsAccount;
class QDomElement;
class Security;
class Budget;

typedef enum {
	SECURITY_TYPE_BOND,
	SECURITY_TYPE_STOCK,
	SECURITY_TYPE_MUTUAL_FUND
} SecurityType;

class ReinvestedDividend {
	public:
		ReinvestedDividend(const QDate &date_, double shares_) : date(date_), shares(shares_) {}
		QDate date;
		double shares;
};

template<class type> class SecurityTransactionList : public QPtrList<type> {
	public:
		SecurityTransactionList() : QPtrList<type>() {};
	protected:
		int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2) {
			Transaction *t1 = (Transaction*) item1;
			Transaction *t2 = (Transaction*) item2;
			if(t1->date() > t2->date()) return 1;
			if(t1->date() < t2->date()) return -1;
			return 0;
		}
};

template<class type> class ScheduledSecurityTransactionList : public QPtrList<type> {
	public:
		ScheduledSecurityTransactionList() : QPtrList<type>() {};
	protected:
		int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2) {
			Transaction *t1 = ((ScheduledTransaction*) item1)->transaction();
			Transaction *t2 = ((ScheduledTransaction*) item2)->transaction();
			if(t1->date() > t2->date()) return 1;
			if(t1->date() < t2->date()) return -1;
			return 0;
		}
};

template<class type> class ReinvestedDividendList : public QPtrList<type> {
	public:
		ReinvestedDividendList() : QPtrList<type>() {};
	protected:
		int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2) {
			ReinvestedDividend *t1 = (ReinvestedDividend*) item1;
			ReinvestedDividend *t2 = (ReinvestedDividend*) item2;
			if(t1->date > t2->date) return 1;
			if(t1->date < t2->date) return -1;
			return 0;
		}
};

template<class type> class TradedSharesList : public QPtrList<type> {
	public:
		TradedSharesList() : QPtrList<type>() {};
	protected:
		int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2) {
			SecurityTrade *t1 = (SecurityTrade*) item1;
			SecurityTrade *t2 = (SecurityTrade*) item2;
			if(t1->date > t2->date) return 1;
			if(t1->date < t2->date) return -1;
			return 0;
		}
};

class Security {

	protected:

		Budget *o_budget;

		int i_id;

		AssetsAccount *o_account;

		SecurityType st_type;

		double d_initial_shares;

		int i_decimals;

		QString s_name;
		QString s_description;

		void init();

	public:

		Security(Budget *parent_budget, AssetsAccount *parent_account, SecurityType initial_type, double initial_shares = 0.0, int initial_decimals = 4, QString initial_name = QString::null, QString initial_description = QString::null);
		Security(Budget *parent_budget, QDomElement *e, bool *valid);
		Security();
		Security(const Security *security);
		~Security();

		const QString &name() const;
		void setName(QString new_name);
		const QString &description() const;
		void setDescription(QString new_description);
		Budget *budget() const;
		double initialBalance() const;
		double initialShares() const;
		void setInitialShares(double initial_shares);
		virtual SecurityType type() const;
		void setType(SecurityType new_type);
		int id() const;
		void setId(int new_id);
		bool hasQuotation(const QDate &date) const;
		void setQuotation(const QDate &date, double value, bool auto_added = false);
		void removeQuotation(const QDate &date, bool auto_added = false);
		void clearQuotations();
		double getQuotation(const QDate &date, QDate *actual_date = NULL) const;
		AssetsAccount *account() const;
		void addReinvestedDividend(const QDate &date, double added_shares);
		int decimals() const;
		void setDecimals(int new_decimals);
		void setAccount(AssetsAccount *new_account);
		virtual void save(QDomElement *e) const;

		double shares();
		double shares(const QDate &date, bool estimate = false, bool no_scheduled_shares = false);
		double value();
		double value(const QDate &date, bool estimate = false, bool no_scheduled_shares = false);
		double cost();
		double cost(const QDate &date, bool no_scheduled_shares = false);
		double profit();
		double profit(const QDate &date, bool estimate = false, bool no_scheduled_shares = false);
		double profit(const QDate &date1, const QDate &date2, bool estimate = false, bool no_scheduled_shares = false);
		double yearlyRate();
		double yearlyRate(const QDate &date);
		double yearlyRate(const QDate &date_from, const QDate &date_to);
		double expectedQuotation(const QDate &date);

		QMap<QDate, double> quotations;
		QMap<QDate, bool> quotations_auto;
		TradedSharesList<SecurityTrade> tradedShares;
		ReinvestedDividendList<ReinvestedDividend> reinvestedDividends;
		SecurityTransactionList<SecurityTransaction> transactions;
		SecurityTransactionList<Income> dividends;
		ScheduledSecurityTransactionList<ScheduledTransaction> scheduledTransactions;
		ScheduledSecurityTransactionList<ScheduledTransaction> scheduledDividends;

};

#endif
