/***************************************************************************
 *   Copyright (C) 2006 by Niklas Knutsson   *
 *   nq@altern.org   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "transactionlistwidget.h"
#include "budget.h"
#include "recurrence.h"
#include "transactioneditwidget.h"
#include "transactionfilterwidget.h"
#include "editscheduledtransactiondialog.h"
#include "eqonomize.h"

#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kdeversion.h>
#include <kstatusbar.h>
#include <kaccel.h>
#include <kio/netaccess.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <kurl.h>
#include <kurldrag.h>
#include <qheader.h>
#include <kmessagebox.h>
#include <qlayout.h>
#include <klineedit.h>
#include <kcombobox.h>
#include <qlabel.h>
#include <kbuttonbox.h>
#include <kstdguiitem.h>
#include <kstdaccel.h>
#include <kaction.h>
#include <kstdaction.h>
#include <qobject.h>
#include <qpushbutton.h>
#include <ktempfile.h>
#include <qlabel.h>
#include <kseparator.h>
#include <qcheckbox.h>
#include <kjanuswidget.h>
#include <qradiobutton.h>
#include <qgroupbox.h>
#include <kpushbutton.h>
#include <ktextedit.h>
#include <qbuttongroup.h>
#include <klistview.h>
#include <qpopupmenu.h>
#include <ksavefile.h>

#include <math.h>

#if KDE_VERSION_MAJOR < 4 && KDE_VERSION_MINOR < 2
#include <qtabwidget.h>
#else
#include <ktabwidget.h>
#endif

extern double monthsBetweenDates(const QDate &date1, const QDate &date2);

class TransactionListViewItem : public KListViewItem {
	protected:
		Transaction *o_trans;
		ScheduledTransaction *o_strans;
		QDate d_date;
	public:
		TransactionListViewItem(const QDate &trans_date, Transaction *trans, ScheduledTransaction *strans, QListView *parent, QString, QString=QString::null, QString=QString::null, QString=QString::null, QString=QString::null, QString=QString::null, QString=QString::null, QString=QString::null);
		int compare(QListViewItem *i, int col, bool ascending) const;
		Transaction *transaction() const;
		ScheduledTransaction *scheduledTransaction() const;
		const QDate &date() const;
		void setDate(const QDate &newdate);
};

TransactionListWidget::TransactionListWidget(bool extra_parameters, int transaction_type, Budget *budg, Eqonomize *main_win, QWidget *parent, const char *name) : QWidget(parent, name), transtype(transaction_type), budget(budg), mainWin(main_win), b_extra(extra_parameters) {

	current_value = 0.0;
	current_quantity = 0.0;

	listPopupMenu = NULL;
	
	QVBoxLayout *transactionsLayout = new QVBoxLayout(this, 0, 6);

	QVBoxLayout *transactionsViewLayout = new QVBoxLayout(transactionsLayout, 2);
	
	transactionsView = new KListView(this);
	transactionsView->addColumn(i18n("Date"));
	transactionsView->addColumn(i18n("Description"));
	comments_col = 5;
	switch(transtype) {
		case TRANSACTION_TYPE_EXPENSE: {
			transactionsView->addColumn(i18n("Cost"));
			transactionsView->addColumn(i18n("Category"));
			transactionsView->addColumn(i18n("From Account"));
			if(b_extra) {
				transactionsView->addColumn(i18n("Payee"));
				comments_col = 6;
			}
			from_col = 4; to_col = 3;
			break;
		}
		case TRANSACTION_TYPE_INCOME: {
			transactionsView->addColumn(i18n("Income"));
			transactionsView->addColumn(i18n("Category"));
			transactionsView->addColumn(i18n("To Account"));
			if(b_extra) {
				transactionsView->addColumn(i18n("Payer"));
				comments_col = 6;
			}
			from_col = 3; to_col = 4;
			break;
		}
		default: {
			transactionsView->addColumn(i18n("Amount"));
			transactionsView->addColumn(i18n("From"));
			transactionsView->addColumn(i18n("To"));
			from_col = 3; to_col = 4;
			break;
		}
	}	
	transactionsView->addColumn(i18n("Comments"));
	transactionsView->setRootIsDecorated(false);
	transactionsView->setAllColumnsShowFocus(true);
	transactionsView->setSorting(0, false);
	transactionsView->setShowSortIndicator(true);
	transactionsView->setColumnAlignment(2, Qt::AlignRight);
	transactionsView->setColumnAlignment(3, Qt::AlignCenter);
	transactionsView->setColumnAlignment(4, Qt::AlignCenter);
	transactionsView->setSelectionMode(QListView::Extended);
	transactionsViewLayout->addWidget(transactionsView);
	statLabel = new QLabel(this);
	transactionsViewLayout->addWidget(statLabel);
	QSizePolicy sp = transactionsView->sizePolicy();
	sp.setVerData(QSizePolicy::MinimumExpanding);
	transactionsView->setSizePolicy(sp);

#if KDE_VERSION_MAJOR < 4 && KDE_VERSION_MINOR < 2
	tabs = new QTabWidget(this);
#else
	tabs = new KTabWidget(this);
#endif

	tabs->setMargin(6);
	tabs->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);

	editWidget = new TransactionEditWidget(true, b_extra, transtype, NULL, SECURITY_SHARES_AND_QUOTATION, budget, this);
	editInfoLabel = new QLabel(QString::null, editWidget->bottomWidget());
	KButtonBox *buttons = new KButtonBox(editWidget->bottomWidget());
	buttons->addStretch();
#if KDE_VERSION_MAJOR < 4 && KDE_VERSION_MINOR < 4	
	addButton = buttons->addButton(i18n("Add"));
#else
	addButton = buttons->addButton(KStdGuiItem::guiItem(KStdGuiItem::Add));
#endif
	modifyButton = buttons->addButton(i18n("Change"));
	modifyButton->setEnabled(false);
#if KDE_VERSION_MAJOR > 3 || KDE_VERSION_MINOR >= 3
	removeButton = buttons->addButton(KStdGuiItem::guiItem(KStdGuiItem::Delete));
#else
	removeButton = buttons->addButton(i18n("Delete"));
#endif
	removeButton->setEnabled(false);
	buttons->layout();

	filterWidget = new TransactionFilterWidget(b_extra, transtype, budget, this);
	tabs->addTab(editWidget, i18n("New/Edit Expense"));
	tabs->addTab(filterWidget, i18n("Filter"));
	transactionsLayout->addWidget(tabs);

	updateStatistics();

	connect(addButton, SIGNAL(clicked()), this, SLOT(addTransaction()));
	connect(modifyButton, SIGNAL(clicked()), this, SLOT(modifyTransaction()));
	connect(editWidget, SIGNAL(addmodify()), this, SLOT(addModifyTransaction()));
	connect(removeButton, SIGNAL(clicked()), this, SLOT(removeTransaction()));
	connect(filterWidget, SIGNAL(filter()), this, SLOT(filterTransactions()));
	connect(filterWidget, SIGNAL(toActivated(int)), this, SLOT(filterCategoryActivated(int)));
	connect(filterWidget, SIGNAL(fromActivated(int)), this, SLOT(filterFromActivated(int)));
	connect(transactionsView, SIGNAL(selectionChanged()), this, SLOT(transactionSelectionChanged()));
	connect(transactionsView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)), this, SLOT(popupListMenu(KListView*, QListViewItem*, const QPoint&)));

}

//QSize TransactionListWidget::minimumSizeHint() const {return filterWidget->minimumSizeHint().expandedTo(editWidget->minimumSizeHint());}
//QSize TransactionListWidget::sizeHint() const {return QSize(filterWidget->sizeHint().expandedTo(editWidget->sizeHint()).width() + 12, QWidget::sizeHint().height());}

QSize TransactionListWidget::minimumSizeHint() const {return QWidget::minimumSizeHint();}
QSize TransactionListWidget::sizeHint() const {return minimumSizeHint();}

void TransactionListWidget::updateStatistics() {
	int i_count_frac = 0;
	double intpart = 0.0;
	if(modf(current_quantity, &intpart) != 0.0) i_count_frac = 2;
	statLabel->setText(QString("<div align=\"right\"><b>%1</b> %5 &nbsp; <b>%2</b> %6 &nbsp; <b>%3</b> %7 &nbsp; <b>%4</b> %8</div>").arg(i18n("Quantity:")).arg(i18n("Total:")).arg(i18n("Average:")).arg(i18n("Monthly:")).arg(KGlobal::locale()->formatNumber(current_quantity, i_count_frac)).arg(KGlobal::locale()->formatMoney(current_value)).arg(KGlobal::locale()->formatMoney(current_quantity == 0.0 ? 0.0 : current_value / current_quantity)).arg(KGlobal::locale()->formatMoney(current_value == 0.0 ? current_value : current_value / filterWidget->countMonths())));
}

void TransactionListWidget::popupListMenu(KListView*, QListViewItem*, const QPoint &p) {
	if(!listPopupMenu) {
		listPopupMenu = new QPopupMenu(this);
		switch(transtype) {
			case TRANSACTION_TYPE_EXPENSE: {mainWin->ActionNewExpense->plug(listPopupMenu); break;}
			case TRANSACTION_TYPE_INCOME: {mainWin->ActionNewIncome->plug(listPopupMenu); break;}
			case TRANSACTION_TYPE_TRANSFER: {mainWin->ActionNewTransfer->plug(listPopupMenu); break;}
		}
		listPopupMenu->insertSeparator();
		mainWin->ActionEditTransaction->plug(listPopupMenu);
		mainWin->ActionEditScheduledTransaction->plug(listPopupMenu);
		listPopupMenu->insertSeparator();
		mainWin->ActionDeleteTransaction->plug(listPopupMenu);
		mainWin->ActionDeleteScheduledTransaction->plug(listPopupMenu);
	}
	listPopupMenu->popup(p);
}

extern QString htmlize_string(QString str);

bool TransactionListWidget::isEmpty() {
	return transactionsView->childCount() == 0;
}

bool TransactionListWidget::exportList(QTextStream &outf, int fileformat) {

	switch(fileformat) {
		case 'h': {
			outf.setEncoding(QTextStream::UnicodeUTF8);
			outf << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \"http://www.w3.org/TR/html4/loose.dtd\">" << '\n';
			outf << "<html>" << '\n';
			outf << "\t<head>" << '\n';
			outf << "\t\t<title>";
			switch(transtype) {
				case TRANSACTION_TYPE_EXPENSE: {outf << htmlize_string(i18n("Expenses")); break;}
				case TRANSACTION_TYPE_INCOME: {outf << htmlize_string(i18n("Incomes")); break;}
				default: {outf << htmlize_string(i18n("Transfers")); break;}
			}
			outf << "</title>" << '\n';
			outf << "\t\t<meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">" << '\n';
			outf << "\t\t<meta name=\"GENERATOR\" content=\"Eqonomize!\">" << '\n';
			outf << "\t</head>" << '\n';
			outf << "\t<body>" << '\n';
			outf << "\t\t<table border=\"1\" cellspacing=\"0\" cellpadding=\"5\">" << '\n';
			outf << "\t\t\t<caption>";
			switch(transtype) {
				case TRANSACTION_TYPE_EXPENSE: {outf << htmlize_string(i18n("Expenses")); break;}
				case TRANSACTION_TYPE_INCOME: {outf << htmlize_string(i18n("Incomes")); break;}
				default: {outf << htmlize_string(i18n("Transfers")); break;}
			}
			outf << "</caption>" << '\n';
			outf << "\t\t\t<thead>" << '\n';
			outf << "\t\t\t\t<tr>" << '\n';
			outf << "\t\t\t\t\t<th>" << htmlize_string(transactionsView->columnText(0)) << "</th>";
			outf << "<th>" << htmlize_string(transactionsView->columnText(1)) << "</th>";
			outf << "<th>" << htmlize_string(transactionsView->columnText(2)) << "</th>";
			outf << "<th>" << htmlize_string(transactionsView->columnText(3)) << "</th>";
			outf << "<th>" << htmlize_string(transactionsView->columnText(4)) << "</th>";			
			outf << "<th>" << htmlize_string(transactionsView->columnText(5)) << "</th>";
			if(comments_col == 6) outf << "<th>" << htmlize_string(transactionsView->columnText(6)) << "</th>";
			outf << "\n";
			outf << "\t\t\t\t</tr>" << '\n';
			outf << "\t\t\t</thead>" << '\n';
			outf << "\t\t\t<tbody>" << '\n';
			TransactionListViewItem *i = (TransactionListViewItem*) transactionsView->firstChild();
			while(i) {
				Transaction *trans = i->transaction();
				outf << "\t\t\t\t<tr>" << '\n';
				outf << "\t\t\t\t\t<td nowrap align=\"right\">" << htmlize_string(KGlobal::locale()->formatDate(trans->date(), true)) << "</td>";
				outf << "<td>" << htmlize_string(i->text(1)) << "</td>";
				outf << "<td nowrap align=\"right\">" << htmlize_string(i->text(2)) << "</td>";
				outf << "<td align=\"center\">" << htmlize_string(i->text(3)) << "</td>";
				outf << "<td align=\"center\">" << htmlize_string(i->text(4)) << "</td>";
				outf << "<td>" << htmlize_string(i->text(5)) << "</td>";
				if(comments_col == 6) outf << "<td>" << htmlize_string(i->text(6)) << "</td>";
				outf << "\n";
				outf << "\t\t\t\t</tr>" << '\n';
				i = (TransactionListViewItem*) i->nextSibling();
			}
			outf << "\t\t\t</tbody>" << '\n';
			outf << "\t\t</table>" << '\n';
			outf << "\t\t<div>";
			int i_count_frac = 0;
			double intpart = 0.0;
			if(modf(current_quantity, &intpart) != 0.0) i_count_frac = 2;
			outf << i18n("Quantity:") << " " << KGlobal::locale()->formatNumber(current_quantity, i_count_frac);
			outf << ", ";
			switch(transtype) {
				case TRANSACTION_TYPE_EXPENSE: {
					outf << i18n("Total cost:") << " ";
					break;
				}
				case TRANSACTION_TYPE_INCOME: {
					outf << i18n("Total income:") << " ";
					break;
				}
				default: {
					outf << i18n("Total amount:") << " ";
					break;
				}
			}
			outf << KGlobal::locale()->formatMoney(current_value);
			outf << ", ";
			outf << i18n("Monthly average:") << " " << KGlobal::locale()->formatMoney(current_value == 0.0 ? current_value : current_value / filterWidget->countMonths());
			outf << ", ";
			outf << i18n("Daily average:") << " " << KGlobal::locale()->formatMoney(current_value == 0.0 ? current_value : current_value / filterWidget->countDays());
			outf << "</div>\n";
			outf << "\t</body>" << '\n';
			outf << "</html>" << '\n';
			break;
		}
		case 'c': {
			outf.setEncoding(QTextStream::Locale);
			outf << "\"" << transactionsView->columnText(0) << "\",\"" << transactionsView->columnText(1) << "\",\"" << transactionsView->columnText(2) << "\",\"" << transactionsView->columnText(3) << "\",\"" << transactionsView->columnText(4) << "\",\"" << transactionsView->columnText(5);
			if(comments_col == 6) outf << "\",\"" << transactionsView->columnText(5);
			outf << "\"\n";
			TransactionListViewItem *i = (TransactionListViewItem*) transactionsView->firstChild();
			while(i) {
				Transaction *trans = i->transaction();
				outf << "\"" << KGlobal::locale()->formatDate(trans->date(), true) << "\",\"" << i->text(1) << "\",\"" << i->text(2) << "\",\"" << i->text(3) << "\",\"" << i->text(4) << "\",\"" << i->text(5);
				if(comments_col == 6) outf << "\",\"" << i->text(6);
				outf << "\"\n";
				i = (TransactionListViewItem*) i->nextSibling();
			}
			break;
		}
	}

	return true;
	
}

void TransactionListWidget::transactionsReset() {
	filterWidget->transactionsReset();
	editWidget->transactionsReset();
	filterTransactions();
}
void TransactionListWidget::addTransaction() {
	Transaction *trans = editWidget->createTransaction();
	if(!trans) return;
	transactionsView->selectAll(false);
	editWidget->setTransaction(NULL);
	editInfoLabel->setText(QString::null);
	ScheduledTransaction *strans = NULL;
	if(trans->date() > QDate::currentDate()) {
		strans = new ScheduledTransaction(budget, trans, NULL);
		budget->addScheduledTransaction(strans);
	} else {
		budget->addTransaction(trans);
	}
	if(strans) mainWin->scheduledTransactionAdded(strans);
	else mainWin->transactionAdded(trans);
}
void TransactionListWidget::editScheduledTransaction() {
#if KDE_VERSION_MAJOR > 3 || KDE_VERSION_MINOR >= 4
	QPtrList<QListViewItem> selection = transactionsView->selectedItems(true);
#else
	QPtrList<QListViewItem> selection = transactionsView->selectedItems();
#endif
	if(selection.count() == 1) {
		TransactionListViewItem *i = (TransactionListViewItem*) selection.getFirst();
		if(i->scheduledTransaction()) {
			mainWin->editScheduledTransaction(i->scheduledTransaction());
		}
	}
}
void TransactionListWidget::editTransaction() {
#if KDE_VERSION_MAJOR > 3 || KDE_VERSION_MINOR >= 4
	QPtrList<QListViewItem> selection = transactionsView->selectedItems(true);
#else
	QPtrList<QListViewItem> selection = transactionsView->selectedItems();
#endif
	if(selection.count() == 1) {
		TransactionListViewItem *i = (TransactionListViewItem*) selection.getFirst();
		if(i->scheduledTransaction()) {
			if(i->scheduledTransaction()->isOneTimeTransaction()) {
				mainWin->editScheduledTransaction(i->scheduledTransaction());
			} else {
				mainWin->editOccurrence(i->scheduledTransaction(), i->date());
			}
		} else {
			mainWin->editTransaction(i->transaction());
		}
		transactionsView->selectAll(false);
	} else if(selection.count() > 1) {
		bool warned1 = false, warned2 = false, warned3 = false;
		MultipleTransactionsEditDialog *dialog = new MultipleTransactionsEditDialog(b_extra, transtype, budget, this);
		TransactionListViewItem *i = (TransactionListViewItem*) transactionsView->currentItem();
		if(!i->isSelected()) i = (TransactionListViewItem*) selection.first();
		if(i) {
			if(i->scheduledTransaction()) dialog->setScheduledTransaction(i->scheduledTransaction(), i->date());
			else dialog->setTransaction(i->transaction());
		}
		bool equal_date = true, equal_description = true, equal_value = true, equal_category = (transtype != TRANSACTION_TYPE_TRANSFER), equal_payee = (dialog->payeeButton != NULL);
		Transaction *comptrans = NULL;
		Account *compcat = NULL;
		QDate compdate;
		i = (TransactionListViewItem*) selection.first();
		while(i) {
			if(!comptrans) {
				comptrans = i->transaction();
				compdate = i->date();
				if(i->transaction()->type() != TRANSACTION_TYPE_EXPENSE && i->transaction()->type() != TRANSACTION_TYPE_INCOME) equal_payee = false;
				if(i->transaction()->type() == TRANSACTION_TYPE_INCOME) {
					compcat = ((Income*) i->transaction())->category();
				} else if(i->transaction()->type() == TRANSACTION_TYPE_EXPENSE) {
					compcat = ((Expense*) i->transaction())->category();
				} else if(i->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || i->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL) {
					equal_value = false;
					equal_description = false;
					compcat = ((SecurityTransaction*) i->transaction())->account();
					if(compcat->type() == ACCOUNT_TYPE_ASSETS) {
						equal_category = false;
					}
				}
			} else {
				if(equal_date && compdate != i->date()) {
					equal_date = false;
				}
				if(equal_description && (i->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || i->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL || comptrans->description() != i->transaction()->description())) {
					equal_description = false;
				}
				if(equal_payee && (i->transaction()->type() != comptrans->type() || (comptrans->type() == TRANSACTION_TYPE_EXPENSE && ((Expense*) comptrans)->payee() != ((Expense*) i->transaction())->payee()) || (comptrans->type() == TRANSACTION_TYPE_EXPENSE && ((Income*) comptrans)->payer() != ((Income*) i->transaction())->payer()))) {
					equal_payee = false;
				}
				if(equal_value && (i->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || i->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL || comptrans->value() != i->transaction()->value())) {
					equal_value = false;
				}
				if(equal_category) {
					if(i->transaction()->type() == TRANSACTION_TYPE_INCOME) {
						if(compcat != ((Income*) i->transaction())->category()) {
							equal_category = false;
						}
					} else if(i->transaction()->type() == TRANSACTION_TYPE_EXPENSE) {
						if(compcat != ((Expense*) i->transaction())->category()) {
							equal_category = false;
						}
					} else if(i->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || i->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL) {
						if(compcat != ((SecurityTransaction*) i->transaction())->account()) {
							equal_category = false;
						}
					}
				}
			}
			i = (TransactionListViewItem*) selection.next();
		}
		if(equal_description) dialog->descriptionButton->setChecked(true);
		if(equal_payee) dialog->payeeButton->setChecked(true);
		if(equal_value) dialog->valueButton->setChecked(true);
		if(equal_date) dialog->dateButton->setChecked(true);
		if(equal_category && dialog->categoryButton) dialog->categoryButton->setChecked(true);
		if(dialog->exec() == QDialog::Accepted) {
			QDate date = dialog->date();
			bool future = !date.isNull() && date > QDate::currentDate();
			i = (TransactionListViewItem*) selection.first();
			while(i) {
				if(!warned1 && (i->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || i->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL)) {
					if(dialog->valueButton->isChecked()) {
						KMessageBox::error(this, i18n("Cannot set the value of security transactions using the dialog for modifying multiple transactions."));
						warned1 = true;
					}
				}
				if(!warned2 && (i->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || i->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL || (i->transaction()->type() == TRANSACTION_TYPE_INCOME && ((Income*) i->transaction())->security()))) {
					if(dialog->descriptionButton->isChecked()) {
						KMessageBox::error(this, i18n("Cannot change description of dividends and security transactions."));
						warned2 = true;
					}
				}
				if(!warned3 && dialog->payeeButton && (i->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || i->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL || (i->transaction()->type() == TRANSACTION_TYPE_INCOME && ((Income*) i->transaction())->security()))) {
					if(dialog->payeeButton->isChecked()) {
						KMessageBox::error(this, i18n("Cannot change payer of dividends and security transactions."));
						warned3 = true;
					}
				}
				ScheduledTransaction *strans = i->scheduledTransaction();
				if(strans && strans->isOneTimeTransaction() && (date.isNull() || date == strans->transaction()->date())) {
					ScheduledTransaction *old_strans = strans->copy();
					Transaction *trans = strans->transaction();
					if(dialog->modifyTransaction(trans)) {
						mainWin->scheduledTransactionModified(strans, old_strans);
					}
					delete old_strans;
				} else if(strans) {
					date = i->date();
					Transaction *trans = strans->transaction()->copy();
					trans->setDate(date);
					if(dialog->modifyTransaction(trans)) {
						if(future) {
							ScheduledTransaction *strans_new = new ScheduledTransaction(budget, trans, NULL);
							budget->addScheduledTransaction(strans_new);
							mainWin->scheduledTransactionAdded(strans_new);
						} else {
							budget->addTransaction(trans);
							mainWin->transactionAdded(trans);
						}
						if(strans->isOneTimeTransaction()) {
							budget->removeScheduledTransaction(strans, true);
							mainWin->scheduledTransactionRemoved(strans);
							delete strans;
						} else {
							ScheduledTransaction *old_strans = strans->copy();
							strans->addException(date);
							mainWin->scheduledTransactionModified(strans, old_strans);
							delete old_strans;
						}
					} else {
						delete trans;
					}
				} else {
					Transaction *trans = i->transaction();
					Transaction *oldtrans = trans->copy();
					if(dialog->modifyTransaction(trans)) {
						if(future) {
							budget->removeTransaction(trans, true);
							mainWin->transactionRemoved(trans);
							strans = new ScheduledTransaction(budget, trans, NULL);
							budget->addScheduledTransaction(strans);
							mainWin->scheduledTransactionAdded(strans);
						} else {
							mainWin->transactionModified(trans, oldtrans);
						}
					}
					delete oldtrans;
				}
				i = (TransactionListViewItem*) selection.next();
			}
		}
		delete dialog;
	}
}
void TransactionListWidget::modifyTransaction() {
#if KDE_VERSION_MAJOR > 3 || KDE_VERSION_MINOR >= 4
	QPtrList<QListViewItem> selection = transactionsView->selectedItems(true);
#else
	QPtrList<QListViewItem> selection = transactionsView->selectedItems();
#endif
	if(selection.count() == 0) return;
	if(selection.count() > 1) {
		editTransaction();
		return;
	}
	TransactionListViewItem *i = (TransactionListViewItem*) selection.getFirst();
	if(!i) return;
	if(((TransactionListViewItem*) i)->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || ((TransactionListViewItem*) i)->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL) {
		editTransaction();
		return;
	}
	if(!editWidget->validValues()) return;
	ScheduledTransaction *oldstrans = NULL;
	if(i->scheduledTransaction()) {
		if(i->scheduledTransaction()->isOneTimeTransaction() && i->scheduledTransaction()->firstOccurrence() > QDate::currentDate()) {
			oldstrans = i->scheduledTransaction()->copy();
		} else {
			if(editWidget->validValues()) {
				ScheduledTransaction *curstranscopy = i->scheduledTransaction();
				QDate curdate_copy = i->date();
				addTransaction();
				oldstrans = curstranscopy->copy();
				curstranscopy->addException(curdate_copy);
				mainWin->scheduledTransactionModified(curstranscopy, oldstrans);
				delete oldstrans;
			}
			return;
		}
	}
	if(i->scheduledTransaction()) {
		if(editWidget->modifyTransaction(i->transaction())) {
			i->scheduledTransaction()->setDate(i->transaction()->date());
			mainWin->scheduledTransactionModified(i->scheduledTransaction(), oldstrans);
		}
		delete oldstrans;
	} else if(editWidget->date() > QDate::currentDate()) {
		Transaction *newtrans = i->transaction()->copy();
		if(editWidget->modifyTransaction(newtrans)) {
			ScheduledTransaction *strans = new ScheduledTransaction(budget, newtrans, NULL);
			removeTransaction();
			budget->addScheduledTransaction(strans);
			mainWin->scheduledTransactionAdded(strans);
		} else {
			delete newtrans;
		}
	} else {
		Transaction *oldtrans = i->transaction()->copy();
		if(editWidget->modifyTransaction(i->transaction())) {
			mainWin->transactionModified(i->transaction(), oldtrans);
		}
		delete oldtrans;
	}
}
void TransactionListWidget::removeScheduledTransaction() {
#if KDE_VERSION_MAJOR > 3 || KDE_VERSION_MINOR >= 4
	QPtrList<QListViewItem> selection = transactionsView->selectedItems(true);
#else
	QPtrList<QListViewItem> selection = transactionsView->selectedItems();
#endif
	if(selection.count() == 1) {
		TransactionListViewItem *i = (TransactionListViewItem*) selection.getFirst();
		if(i->scheduledTransaction()) {
			mainWin->removeScheduledTransaction(i->scheduledTransaction());
		}
	}
}
void TransactionListWidget::removeTransaction() {
#if KDE_VERSION_MAJOR > 3 || KDE_VERSION_MINOR >= 4
	QPtrList<QListViewItem> selection = transactionsView->selectedItems(true);
#else
	QPtrList<QListViewItem> selection = transactionsView->selectedItems();
#endif
	if(selection.count() > 1) {
		if(!KMessageBox::warningContinueCancel(this, i18n("Are you sure you want to delete all (%1) selected transactions?").arg(selection.count()))) {
			return;
		}
	}
	transactionsView->selectAll(false);
	TransactionListViewItem *i = (TransactionListViewItem*) selection.first();
	while(i) {
		if(i->scheduledTransaction()) {
			ScheduledTransaction *strans = i->scheduledTransaction();
			if(strans->isOneTimeTransaction()) {
				budget->removeScheduledTransaction(strans, true);
				mainWin->scheduledTransactionRemoved(strans);
				delete strans;
			} else {
				ScheduledTransaction *oldstrans = strans->copy();
				strans->addException(i->date());
				mainWin->scheduledTransactionModified(strans, oldstrans);
				delete oldstrans;
			}
		} else {
			Transaction *trans = i->transaction();
			budget->removeTransaction(trans, true);
			mainWin->transactionRemoved(trans);
			delete trans;
		}
		i = (TransactionListViewItem*) selection.next();
	}
}
void TransactionListWidget::addModifyTransaction() {
	addTransaction();
}

void TransactionListWidget::appendFilterTransaction(Transaction *trans, bool update_total_value) {
	if(!filterWidget->filterTransaction(trans)) {
		QListViewItem *i = new TransactionListViewItem(trans->date(), trans, NULL, transactionsView, KGlobal::locale()->formatDate(trans->date(), true), trans->description(), KGlobal::locale()->formatMoney(trans->value()));
		i->setText(from_col, trans->fromAccount()->name());
		i->setText(to_col, trans->toAccount()->name());
		if(comments_col == 6 && trans->type() == TRANSACTION_TYPE_EXPENSE) i->setText(5, ((Expense*) trans)->payee());
		else if(comments_col == 6 && trans->type() == TRANSACTION_TYPE_INCOME) i->setText(5, ((Income*) trans)->payer());
		i->setText(comments_col, trans->comment());
		current_value += trans->value();
		current_quantity += trans->quantity();
		if(update_total_value) {
			updateStatistics();
		}
	}
}
void TransactionListWidget::appendFilterScheduledTransaction(ScheduledTransaction *strans, bool update_total_value) {
	if(!filterWidget->filterTransaction(strans->transaction(), false)) {
		QDate date = filterWidget->startDate();
		QDate enddate = filterWidget->endDate();
		if(strans->isOneTimeTransaction()) {
			if(date.isNull()) date = strans->firstOccurrence();
			else if(date > strans->firstOccurrence()) date = QDate();
			else date = strans->firstOccurrence();
		} else {
			if(date.isNull()) date = strans->recurrence()->firstOccurrence();
			else date = strans->recurrence()->nextOccurrence(date, true);
		}
		if(date.isNull() || date > enddate) update_total_value = false;
		Transaction *trans = strans->transaction();
		while(!date.isNull() && date <= enddate) {
			QListViewItem *i = NULL;
			if(strans->recurrence()) i = new TransactionListViewItem(date, trans, strans, transactionsView, KGlobal::locale()->formatDate(date, true) + "*", trans->description(), KGlobal::locale()->formatMoney(trans->value()));
			else i = new TransactionListViewItem(date, trans, strans, transactionsView, KGlobal::locale()->formatDate(date, true), trans->description(), KGlobal::locale()->formatMoney(trans->value()));
			i->setText(from_col, trans->fromAccount()->name());
			i->setText(to_col, trans->toAccount()->name());
			if(comments_col == 6 && trans->type() == TRANSACTION_TYPE_EXPENSE) i->setText(5, ((Expense*) trans)->payee());
			else if(comments_col == 6 && trans->type() == TRANSACTION_TYPE_INCOME) i->setText(5, ((Income*) trans)->payer());
			i->setText(comments_col, trans->comment());
			current_value += trans->value();
			current_quantity += trans->quantity();
			if(!strans->isOneTimeTransaction()) date = strans->recurrence()->nextOccurrence(date);
			else break;
		}
		if(update_total_value) {
			updateStatistics();
		}
	}
}

void TransactionListWidget::onTransactionAdded(Transaction *trans) {
	appendFilterTransaction(trans, true);
	filterWidget->transactionAdded(trans);
	editWidget->transactionAdded(trans);
}
void TransactionListWidget::onTransactionModified(Transaction *trans, Transaction *oldtrans) {
	current_value -= oldtrans->value();
	current_quantity -= oldtrans->quantity();
	bool b_filter = filterWidget->filterTransaction(trans);
	if(!b_filter) {
		current_value += trans->value();
		current_quantity += trans->quantity();
	}
	TransactionListViewItem *i = (TransactionListViewItem*) transactionsView->firstChild();
	while(i) {
		if(i->transaction() == trans) {
			break;
		}
		i = (TransactionListViewItem*) i->nextSibling();
	}
	if(i && b_filter) {
		delete i;
		i = NULL;
	}
	updateStatistics();
	if(i) {
		i->setDate(trans->date());
		i->setText(0, KGlobal::locale()->formatDate(trans->date(), true));
		i->setText(1, trans->description());
		i->setText(2, KGlobal::locale()->formatMoney(trans->value()));
		i->setText(from_col, trans->fromAccount()->name());
		i->setText(to_col, trans->toAccount()->name());
		if(comments_col == 6 && trans->type() == TRANSACTION_TYPE_EXPENSE) i->setText(5, ((Expense*) trans)->payee());
		else if(comments_col == 6 && trans->type() == TRANSACTION_TYPE_INCOME) i->setText(5, ((Income*) trans)->payer());
		i->setText(comments_col, trans->comment());
	}
	if(oldtrans->description() != trans->description()) filterWidget->transactionModified(trans);
	editWidget->transactionModified(trans);
}
void TransactionListWidget::onTransactionRemoved(Transaction *trans) {
	TransactionListViewItem *i = (TransactionListViewItem*) transactionsView->firstChild();
	while(i) {
		if(i->transaction() == trans) {
			delete i;
			current_value -= trans->value();
			current_quantity -= trans->quantity();
			updateStatistics();
			break;
		}
		i = (TransactionListViewItem*) i->nextSibling();
	}
	editWidget->transactionRemoved(trans);
}
void TransactionListWidget::onScheduledTransactionAdded(ScheduledTransaction *strans) {
	appendFilterScheduledTransaction(strans, true);
	filterWidget->transactionAdded(strans->transaction());
	editWidget->transactionAdded(strans->transaction());
}
void TransactionListWidget::onScheduledTransactionModified(ScheduledTransaction *strans, ScheduledTransaction *oldstrans) {
	TransactionListViewItem *i = (TransactionListViewItem*) transactionsView->firstChild();
	while(i) {
		if(i->scheduledTransaction() == strans) {
			current_value -= oldstrans->transaction()->value();
			current_quantity -= oldstrans->transaction()->quantity();
			QListViewItem *i_del = i;
			i = (TransactionListViewItem*) i->nextSibling();
			delete i_del;
		} else {
			i = (TransactionListViewItem*) i->nextSibling();
		}
	}
	appendFilterScheduledTransaction(strans, true);
	updateStatistics();
	if(oldstrans->transaction()->description() != strans->transaction()->description()) filterWidget->transactionModified(strans->transaction());
	editWidget->transactionModified(strans->transaction());
}
void TransactionListWidget::onScheduledTransactionRemoved(ScheduledTransaction *strans) {
	TransactionListViewItem *i = (TransactionListViewItem*) transactionsView->firstChild();
	while(i) {
		if(i->scheduledTransaction() == strans) {
			current_value -= strans->transaction()->value();
			current_quantity -= strans->transaction()->quantity();
			QListViewItem *i_del = i;
			i = (TransactionListViewItem*) i->nextSibling();
			delete i_del;
		} else {
			i = (TransactionListViewItem*) i->nextSibling();
		}
	}
	updateStatistics();
	editWidget->transactionRemoved(strans->transaction());
}
void TransactionListWidget::filterTransactions() {
	transactionsView->clear();
	current_value = 0.0;
	current_quantity = 0.0;
	switch(transtype) {
		case TRANSACTION_TYPE_EXPENSE: {
			Expense *expense = budget->expenses.first();
			while(expense) {
				appendFilterTransaction(expense, false);
				expense = budget->expenses.next();
			}
			SecurityTransaction *sectrans = budget->securityTransactions.first();
			while(sectrans) {
				if(sectrans->account()->type() == ACCOUNT_TYPE_EXPENSES) {
					appendFilterTransaction(sectrans, false);
				}
				sectrans = budget->securityTransactions.next();
			}
			ScheduledTransaction *strans = budget->scheduledTransactions.first();
			while(strans) {
				if(strans->transaction()->type() == TRANSACTION_TYPE_EXPENSE || ((strans->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || strans->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL) && ((SecurityTransaction*) strans->transaction())->account()->type() == ACCOUNT_TYPE_EXPENSES)) {
					appendFilterScheduledTransaction(strans, false);
				}
				strans = budget->scheduledTransactions.next();
			}
			break;
		}
		case TRANSACTION_TYPE_INCOME: {
			Income *income = budget->incomes.first();
			while(income) {
				appendFilterTransaction(income, false);
				income = budget->incomes.next();
			}
			SecurityTransaction *sectrans = budget->securityTransactions.first();
			while(sectrans) {
				if(sectrans->account()->type() == ACCOUNT_TYPE_INCOMES) {
					appendFilterTransaction(sectrans, false);
				}
				sectrans = budget->securityTransactions.next();
			}
			ScheduledTransaction *strans = budget->scheduledTransactions.first();
			while(strans) {
				if(strans->transaction()->type() == TRANSACTION_TYPE_INCOME || ((strans->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || strans->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL) && ((SecurityTransaction*) strans->transaction())->account()->type() == ACCOUNT_TYPE_INCOMES)) {
					appendFilterScheduledTransaction(strans, false);
				}
				strans = budget->scheduledTransactions.next();
			}
			break;
		}
		default: {
			Transfer *transfer = budget->transfers.first();
			while(transfer) {
				appendFilterTransaction(transfer, false);
				transfer = budget->transfers.next();
			}
			SecurityTransaction *sectrans = budget->securityTransactions.first();
			while(sectrans) {
				if(sectrans->account()->type() == ACCOUNT_TYPE_ASSETS) {
					appendFilterTransaction(sectrans, false);
				}
				sectrans = budget->securityTransactions.next();
			}
			ScheduledTransaction *strans = budget->scheduledTransactions.first();
			while(strans) {
				if(strans->transaction()->type() == TRANSACTION_TYPE_TRANSFER || ((strans->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || strans->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL) && ((SecurityTransaction*) strans->transaction())->account()->type() == ACCOUNT_TYPE_ASSETS)) {
					appendFilterScheduledTransaction(strans, false);
				}
				strans = budget->scheduledTransactions.next();
			}
			break;
		}
	}
	updateStatistics();
}

void TransactionListWidget::currentTransactionChanged(QListViewItem *i) {
	if(i == NULL) {
		editWidget->setTransaction(NULL);
		editInfoLabel->setText(QString::null);
	} else if(((TransactionListViewItem*) i)->scheduledTransaction()) {
		editWidget->setScheduledTransaction(((TransactionListViewItem*) i)->scheduledTransaction(), ((TransactionListViewItem*) i)->date());
		if(((TransactionListViewItem*) i)->scheduledTransaction()->isOneTimeTransaction()) editInfoLabel->setText(QString::null);
		else editInfoLabel->setText(i18n("* Recurring (editing occurrance)"));
	} else {
		editWidget->setTransaction(((TransactionListViewItem*) i)->transaction());
		editInfoLabel->setText(QString::null);
	}
}
void TransactionListWidget::transactionSelectionChanged() {
#if KDE_VERSION_MAJOR > 3 || KDE_VERSION_MINOR >= 4
	QPtrList<QListViewItem> selection = transactionsView->selectedItems(true);
#else
	QPtrList<QListViewItem> selection = transactionsView->selectedItems();
#endif
	if(selection.count() == 0) {
		editInfoLabel->setText(QString::null);
		modifyButton->setEnabled(false);
		removeButton->setEnabled(false);
		currentTransactionChanged(NULL);
	} else {
		QListViewItem *i = selection.getFirst();
		if(selection.count() > 1) {
			modifyButton->setText(i18n("Modify..."));
		} else if(((TransactionListViewItem*) i)->transaction()->type() == TRANSACTION_TYPE_SECURITY_BUY || ((TransactionListViewItem*) i)->transaction()->type() == TRANSACTION_TYPE_SECURITY_SELL) {
			modifyButton->setText(i18n("Edit..."));
		} else {
			modifyButton->setText(i18n("Change"));
		}
		modifyButton->setEnabled(true);
		removeButton->setEnabled(true);
		if(transactionsView->currentItem()->isSelected()) {
			currentTransactionChanged(transactionsView->currentItem());
		}
	}
	updateTransactionActions();
}
void TransactionListWidget::updateTransactionActions() {
#if KDE_VERSION_MAJOR > 3 || KDE_VERSION_MINOR >= 4
	QPtrList<QListViewItem> selection = transactionsView->selectedItems(true);
#else
	QPtrList<QListViewItem> selection = transactionsView->selectedItems();
#endif
	bool b_transaction = false, b_scheduledtransaction = false;
	if(selection.count() == 1) {
		TransactionListViewItem *i = (TransactionListViewItem*) selection.getFirst();
		b_scheduledtransaction = i->scheduledTransaction() && i->scheduledTransaction()->recurrence();
		b_transaction = !b_scheduledtransaction || !i->scheduledTransaction()->isOneTimeTransaction();
	} else {
		b_transaction = true;
	}
	mainWin->ActionEditTransaction->setEnabled(b_transaction);
	mainWin->ActionDeleteTransaction->setEnabled(b_transaction);
	mainWin->ActionEditScheduledTransaction->setEnabled(b_scheduledtransaction);
	mainWin->ActionDeleteScheduledTransaction->setEnabled(b_scheduledtransaction);
}
void TransactionListWidget::filterCategoryActivated(int index) {
	if(index > 0) editWidget->setCurrentToItem(index - 1);
}
void TransactionListWidget::filterFromActivated(int index) {
	if(index > 0) editWidget->setCurrentFromItem(index - 1);
}
void TransactionListWidget::onDisplay() {
	if(tabs->currentPageIndex() == 0) editWidget->focusDescription();
}
void TransactionListWidget::updateFromAccounts() {
	editWidget->updateFromAccounts();
	filterWidget->updateFromAccounts();
}
void TransactionListWidget::updateToAccounts() {
	editWidget->updateToAccounts();
	filterWidget->updateToAccounts();
}
void TransactionListWidget::updateAccounts() {
	editWidget->updateAccounts();
	filterWidget->updateAccounts();
}
void TransactionListWidget::setCurrentEditToItem(int index) {
	editWidget->setCurrentToItem(index);
}
void TransactionListWidget::setCurrentEditFromItem(int index) {
	editWidget->setCurrentFromItem(index);
}
int TransactionListWidget::currentEditToItem() {
	return editWidget->currentToItem();
}
int TransactionListWidget::currentEditFromItem() {
	return editWidget->currentFromItem();
}
void TransactionListWidget::showFilter() {tabs->showPage(filterWidget);}
void TransactionListWidget::showEdit() {tabs->showPage(editWidget);}
void TransactionListWidget::setFilter(QDate fromdate, QDate todate, double min, double max, Account *from_account, Account *to_account, QString description, QString payee, bool exclude) {
	filterWidget->setFilter(fromdate, todate, min, max, from_account, to_account, description, payee, exclude);
}

TransactionListViewItem::TransactionListViewItem(const QDate &trans_date, Transaction *trans, ScheduledTransaction *strans, QListView *parent, QString s1, QString s2, QString s3, QString s4, QString s5, QString s6, QString s7, QString s8) : KListViewItem(parent, s1, s2, s3, s4, s5, s6, s7, s8), o_trans(trans), o_strans(strans), d_date(trans_date) {}
int TransactionListViewItem::compare(QListViewItem *i, int col, bool ascending) const {
	if(col == 0) {
		if(d_date > ((TransactionListViewItem*) i)->date()) return 1;
		if(d_date < ((TransactionListViewItem*) i)->date()) return -1;
		return 0;
	} else if(col == 2) {
		if(o_trans->value() > ((TransactionListViewItem*) i)->transaction()->value()) return 1;
		if(o_trans->value() < ((TransactionListViewItem*) i)->transaction()->value()) return -1;
		return 0;
	}
	return QListViewItem::compare(i, col, ascending);
}
Transaction *TransactionListViewItem::transaction() const {
	return o_trans;
}
ScheduledTransaction *TransactionListViewItem::scheduledTransaction() const {
	return o_strans;
}
const QDate &TransactionListViewItem::date() const {
	return d_date;
}
void TransactionListViewItem::setDate(const QDate &newdate) {
	d_date = newdate;
}

#include "transactionlistwidget.moc"

