# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to edit breakpoint properties.
"""

import os.path

from qt import *

from KdeQt import KQFileDialog

from EditBreakpointForm import EditBreakpointForm


class EditBreakpointDialog(EditBreakpointForm):
    """
    Class implementing a dialog to edit breakpoint properties.
    """
    def __init__(self, id, properties, condHistory, parent = None, name = None, 
                 modal = 0, fl = 0, addMode = 0, filenameHistory = None):
        """
        Constructor
        
        @param id id of the breakpoint (tuple)
                (filename, linenumber)
        @param properties properties for the breakpoint (tuple)
                (condition, temporary flag, enabled flag, ignore count)
        @param condHistory the list of conditionals history (QStringList)
        @param parent the parent of this dialog
        @param name the widget name of this dialog
        @param modal flag indicating a modal dialog
        @param fl the window flags
        """
        EditBreakpointForm.__init__(self,parent,name,modal,fl)
        
        dummy = self.trUtf8("Dummy")
        
        fn, lineno = id
        
        if not addMode:
            cond, temp, enabled, count = properties
            
            # set the filename
            if fn is not None:
                self.filenameCombo.setCurrentText(fn)
            
            # set the line number
            self.linenoSpinBox.setValue(lineno)
            
            # set the condition
            if cond is None:
                cond = ''
            curr = condHistory.findIndex(cond)
            if curr == -1:
                condHistory.prepend(cond)
                curr = 0
            self.conditionCombo.insertStringList(condHistory)
            self.conditionCombo.setCurrentItem(curr)
            
            # set the ignore count
            self.ignoreSpinBox.setValue(count)
            
            # set the checkboxes
            self.temporaryCheckBox.setChecked(temp)
            self.enabledCheckBox.setChecked(enabled)
            
            self.filenameCombo.setEnabled(0)
            self.fileButton.setEnabled(0)
            self.linenoSpinBox.setEnabled(0)
            self.conditionCombo.setFocus()
        else:
            # set the filename
            if fn is None:
                fn = QString('')
            curr = filenameHistory.findIndex(fn)
            if curr == -1:
                filenameHistory.prepend(fn)
                curr = 0
            self.filenameCombo.insertStringList(filenameHistory)
            self.filenameCombo.setCurrentItem(curr)
            
            # set the condition
            cond = ''
            curr = condHistory.findIndex(cond)
            if curr == -1:
                condHistory.prepend(cond)
                curr = 0
            self.conditionCombo.insertStringList(condHistory)
            self.conditionCombo.setCurrentItem(curr)
            
            if fn.isEmpty():
                self.okButton.setEnabled(0)
        
    def handleSelectFile(self):
        """
        Private slot to select a file via a file selection dialog.
        """
        file = KQFileDialog.getOpenFileName(self.filenameCombo.currentText(),
            None, self, None, self.trUtf8("Select filename of the breakpoint"))
            
        if not file.isNull():
            self.filenameCombo.setCurrentText(QDir.convertSeparators(file))
        
    def filenameChanged(self, fn):
        """
        Private slot to handle the change of the filename.
        
        @param fn text of the filename edit (QString)
        """
        if fn.isEmpty():
            self.okButton.setEnabled(0)
        else:
            self.okButton.setEnabled(1)
        
    def getData(self):
        """
        Public method to retrieve the entered data.
        
        @return a tuple containing the breakpoints new properties
            (condition, temporary flag, enabled flag, ignore count)
        """
        cond = self.conditionCombo.currentText()
        if cond.isEmpty():
            cond = None
            
        return (cond, self.temporaryCheckBox.isChecked(),
                self.enabledCheckBox.isChecked(), self.ignoreSpinBox.value())
        
    def getAddData(self):
        """
        Public method to retrieve the entered data for an add.
        
        @return a tuple containing the new breakpoints properties
            (filename, lineno, condition, temporary flag, enabled flag, ignore count)
        """
        fn = self.filenameCombo.currentText()
        if fn.isEmpty():
            fn  =  None
        else:
            fn = os.path.expanduser(os.path.expandvars(unicode(fn)))
        
        cond = self.conditionCombo.currentText()
        if cond.isEmpty():
            cond = None
        
        return (fn, self.linenoSpinBox.value(), cond, self.temporaryCheckBox.isChecked(),
                self.enabledCheckBox.isChecked(), self.ignoreSpinBox.value())
