# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the happydoc process.
"""

import os.path

from qt import *

from KdeQt import KQMessageBox

from DocGeneratorExecForm import DocGeneratorExecForm

class DocGeneratorExecDialog(DocGeneratorExecForm):
    """
    Module implementing a dialog to show the output of the happydoc process.
    
    This class starts a QProcess and displays a dialog that
    shows the output of the happydoc command process.
    """
    def __init__(self, cmdname, parent = None):
        """
        Constructor
        
        @param cmdname name of the documentation generator (string)
        @param parent parent widget of this dialog (QWidget)
        """
        DocGeneratorExecForm.__init__(self, parent, None, 1)
        
        self.process = None
        self.cmdname = cmdname
        
    def start(self, args, fn):
        """
        Public slot to start the happydoc command.
        
        @param args commandline arguments for happydoc (QStringList)
        @param fn filename or dirname to be processed by happydoc
        @return flag indicating the successful start of the process
        """
        self.filename = unicode(fn)
        if os.path.isdir(self.filename):
            dname = os.path.abspath(self.filename)
            fname = "."
            if os.path.exists(os.path.join(dname, "__init__.py")):
                fname = os.path.basename(dname)
                dname = os.path.dirname(dname)
        else:
            dname = os.path.dirname(self.filename)
            fname = os.path.basename(self.filename)
        
        self.contents.clear()
        self.errors.clear()
        
        self.process = QProcess()
        self.process.setArguments(args)
        self.process.addArgument(fname)
        self.process.setWorkingDirectory(QDir(dname))
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.finish)
            
        self.setCaption(self.trUtf8('%1 - %2').arg(self.cmdname).arg(self.filename))
        procStarted = self.process.start()
        if not procStarted:
            arg0 = self.process.arguments()[0]
            KQMessageBox.critical(None,
                self.trUtf8('Process Generation Error'),
                self.trUtf8(
                    'The process %1 could not be started.'
                    'Ensure, that it is in the search path.'
                ).arg(arg0),
                self.trUtf8('&OK'))
        return procStarted
        
    def buttonPressed(self):
        """
        Private slot connected to the button clicked signal.
        """
        if self.process is None:
            self.accept()
        else:
            self.finish()
            
    def finish(self):
        """
        Private slot called when the process finished.
        
        It is called when the process finished or
        the user pressed the button.
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        self.button.setText(self.trUtf8('&Close'))
        self.button.setDefault(1)
        
        self.process = None
        
        self.contents.append(self.trUtf8('\n%1 finished.\n').arg(self.cmdname))
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal. 
        
        It reads the output of the process, formats it and inserts it into
        the contents pane.
        """
        self.contents.setTextFormat(QTextBrowser.PlainText)
        
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            self.contents.append(s)
                
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal. 
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        while self.process.canReadLineStderr():
            s = self.process.readLineStderr()
            self.errors.append(s)
