# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the cvs diff command process.
"""

import os
import types

from qt import *

from KdeQt import KQFileDialog, KQMessageBox

from LogForm import LogForm

class CvsDiffDialog(LogForm):
    """
    Class implementing a dialog to show the output of the cvs diff command process.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs reference to the vcs object
        @param parent parent widget (QWidget)
        """
        LogForm.__init__(self, parent)
        
        self.contentsGroup.setTitle(self.trUtf8('Difference:'))
        self.setCaption(self.trUtf8('CVS Diff'))
        QWhatsThis.add(self.contents,self.tr("<b>CVS Diff</b>\n"
            "<p>This shows the output of the cvs diff command.</p>"
        ))
        QWhatsThis.add(self.errors,self.tr("<b>CVS diff errors</b>\n"
            "<p>This shows possible error messages of the cvs diff"
            " command.</p>"
        ))
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.process = QProcess()
        self.vcs = vcs
        
        self.contents.setTextFormat(QTextEdit.LogText)
        
        self.cAdded = QColor(190, 237, 190)
        self.cRemoved = QColor(237, 190, 190)
        self.cNormal = self.contents.paletteBackgroundColor()
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.handleProcessExited)
            
    def closeEvent(self, e):
        """
        Private slot implementing a close event handler.
        
        @param e close event (QCloseEvent)
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        e.accept()
        
    def start(self, fn, versions=None):
        """
        Public slot to start the cvs diff command.
        
        @param fn filename to be diffed (string)
        @param versions list of versions to be diffed (list of 2 int or None)
        """
        self.intercept = 0
        self.filename = fn
        
        self.process.kill()
        self.contents.clear()
        self.paras = 0
        
        self.process.clearArguments()
        self.process.addArgument('cvs')
        self.vcs.addArguments(self.process, self.vcs.options['global'])
        self.process.addArgument('diff')
        self.vcs.addArguments(self.process, self.vcs.options['diff'])
        if versions is not None:
            self.process.addArgument('-r')
            self.process.addArgument(versions[0])
            self.process.addArgument('-r')
            self.process.addArgument(versions[1])
        if type(fn) is types.ListType:
            dname, fnames = self.vcs.splitPathList(fn)
            self.vcs.addArguments(self.process, fnames)
        else:
            dname, fname = self.vcs.splitPath(fn)
            self.process.addArgument(fname)
        self.process.setWorkingDirectory(QDir(dname))
        
        self.process.start()
        self.setCaption(self.trUtf8('CVS Diff'))
        
    def handleProcessExited(self):
        """
        Private slot to handle the processExited signal.
        
        After the process has exited, the contents pane is colored.
        """
        self.closeButton.setFocus()
        self.inputGroup.setEnabled(0)
        
        paras = self.contents.paragraphs()
        if paras == 1:
            self.contents.append(\
                self.trUtf8('There is no difference.'))
            return
            
        self.saveButton.setEnabled(1)
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal. 
        
        It reads the output of the process, formats it and inserts it into
        the contents pane.
        """
        self.contents.setTextFormat(QTextBrowser.PlainText)
        
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            self.contents.append(s)
            if s.startsWith('+') or s.startsWith('>'):
                self.contents.setParagraphBackgroundColor(self.paras, self.cAdded)
            elif s.startsWith('-') or s.startsWith('<'):
                self.contents.setParagraphBackgroundColor(self.paras, self.cRemoved)
            else:
                self.contents.setParagraphBackgroundColor(self.paras, self.cNormal)
            self.paras += 1
        
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal.
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        if self.process is not None:
            s = unicode(self.process.readStderr())
            self.errors.moveCursor(QTextEdit.MoveEnd, 0)
            self.errors.insert(s)
        
    def handleSave(self):
        """
        Private slot to handle the Save button press.
        
        It saves the diff shown in the dialog to a file in the local
        filesystem.
        """
        if type(self.filename) is types.ListType:
            if len(self.filename) > 1:
                fname = self.vcs.splitPathList(self.filename)[0]
            else:
                dname, fname = self.vcs.splitPath(self.filename[0])
                if fname != '.':
                    fname = "%s.diff" % self.filename[0]
                else:
                    fname = dname
        else:
            fname = self.vcs.splitPath(self.filename)[0]
            
        selectedFilter = QString('')
        fname = KQFileDialog.getSaveFileName(\
            fname,
            self.trUtf8("Patch Files (*.diff)"),
            self, None,
            self.trUtf8("Save Diff"),
            selectedFilter, 0)
        
        if fname.isEmpty():
            return
            
        ext = QFileInfo(fname).extension()
        if ext.isEmpty():
            ex = selectedFilter.section('(*',1,1).section(')',0,0)
            if not ex.isEmpty():
                fname.append(ex)
        if QFileInfo(fname).exists():
            abort = KQMessageBox.warning(self,
                self.trUtf8("Save Diff"),
                self.trUtf8("<p>The patch file <b>%1</b> already exists.</p>")
                    .arg(fname),
                self.trUtf8("&Overwrite"),
                self.trUtf8("&Abort"), None, 1)
            if abort:
                return
        fname = unicode(QDir.convertSeparators(fname))
        
        try:
            f = open(fname, "wb")
            paras = self.contents.paragraphs()
            for i in range(paras):
                txt = self.contents.text(i)
                try:
                    f.write("%s%s" % (unicode(txt), os.linesep))
                except UnicodeError:
                    pass
            f.close()
        except IOError, why:
            KQMessageBox.critical(self, self.trUtf8('Save Diff'),
                self.trUtf8('<p>The patch file <b>%1</b> could not be saved.<br>Reason: %2</p>')
                    .arg(fn).arg(unicode(why)))
        
    def passwordMode(self, isOn):
        """
        Private slot to handle the password checkbox toggled.
        
        @param isOn flag indicating the status of the check box (boolean)
        """
        if isOn:
            self.input.setEchoMode(QLineEdit.Password)
        else:
            self.input.setEchoMode(QLineEdit.Normal)
        
    def sendInput(self):
        """
        Private slot to send the input to the subversion process.
        """
        input = self.input.text()
        input.append(os.linesep)
        
        self.errors.moveCursor(QTextEdit.MoveEnd, 0)
        if self.passwordCheckBox.isChecked():
            self.errors.insert(os.linesep)
        else:
            self.errors.insert(input)
        
        self.proc.writeToStdin(input)
        
        self.passwordCheckBox.setChecked(0)
        self.input.clear()
        
    def returnPressed(self):
        """
        Private slot to handle the press of the return key in the input field.
        """
        self.intercept = 1
        self.sendInput()
        
    def keyPressEvent(self, evt):
        """
        Protected slot to handle a key press event.
        
        @param evt the key press event (QKeyEvent)
        """
        if self.intercept:
            self.intercept = 0
            evt.accept()
            return
        LogForm.keyPressEvent(self, evt)
