#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2002-2005 Detlev Offenbach <detlev@die-offenbachs.de>
#
# This is the install script for eric3. It is based on the
# build script found in the PyQt distribution. The original is 
# Copyright (c) 2002 Riverbank Computing Limited <info@riverbankcomputing.co.uk>

"""
Installation script for the eric3 IDE and all eric3 related tools.
"""

import sys
import os
import string
import re
import compileall
import glob
import shutil
import fnmatch
import distutils.sysconfig

# Define the globals.
progName = None
modDir = None
pyModDir = None
platBinDir = None
distDir = None
doCleanup = 1
cfg = {}
qtcfg = {}

def usage(rcode = 2):
    """
    Display a usage message and exit.

    @param rcode the return code passed back to the calling process.
    """
    global progName, platBinDir, modDir, distDir

    print
    print "Usage:"
    print "    %s [-hxc] [-b dir] [-d dir] [-f file] [-i dir]" % (progName)
    print "where:"
    print "    -h        display this help message"
    print "    -b dir    where the binaries will be installed"
    print "              (default: %s)" % (platBinDir)
    print "    -d dir    where eric3 python files will be installed"
    print "              (default: %s)" % (modDir)
    print "    -f file   configuration file naming the various installation paths"
    if sys.platform <> "win32":
        print "    -i dir    temporary install prefix"
        print "              (default: %s)" % (distDir)
    print "    -x        don't perform dependency checks (use on your own risk)"
    print "    -c        don't cleanup old installation first"
    print
    print "The file given to the -f option must be valid Python code defining a"
    print "dictionary called 'cfg' with the keys 'ericDir', 'ericPixDir', 'ericIconDir',"
    print "'ericDTDDir', 'ericCSSDir', 'ericDocDir', 'ericExamplesDir', 'ericTranslationsDir',"
    print "'ericWizardsDir', 'ericTemplatesDir', 'ericOthersDir', 'bindir' and 'mdir'."
    print "These define the directories for the installation of the various parts of eric3."

    sys.exit(rcode)


def initGlobals():
    """
    Sets the values of globals that need more than a simple assignment.
    """
    global platBinDir, modDir, pyModDir

    if sys.platform == "win32":
        platBinDir = sys.exec_prefix
    else:
        platBinDir = "/usr/local/bin"

    modDir = distutils.sysconfig.get_python_lib()
    pyModDir = modDir


def copyToFile(name,text):
    """
    Copy a string to a file.

    @param name the name of the file.
    @param text the contents to copy to the file.
    """
    f = open(name,"w")
    f.write(text)
    f.close()


def wrapperName(dname,wfile):
    """
    Create the platform specific name for the wrapper script.
    
    @param dname name of the directory to place the wrapper into
    @param wfile basename (without extension) of the wrapper script
    @return the name of the wrapper script
    """
    if sys.platform == "win32":
        wname = dname + "\\" + wfile + ".bat"
    else:
        wname = dname + "/" + wfile

    return wname


def createPyWrapper(pydir,wfile):
    """
    Create an executable wrapper for a Python script.

    @param pydir the name of the directory where the Python script will eventually
        be installed
    @param wfile the basename of the wrapper
    @return the platform specific name of the wrapper
    """
    if sys.platform == "win32":
        wname = wfile + ".bat"
        wrapper = \
'''@"%s\\pythonw" "%s\\%s.py" %%1 %%2 %%3 %%4 %%5 %%6 %%7 %%8 %%9
''' % (platBinDir,pydir,wfile)
    elif sys.platform == "darwin":
        wname = wfile
        wrapper = \
'''#!/bin/sh

exec "%sw" "%s/%s.py" $*
''' % (sys.executable,pydir,wfile)
    else:
        wname = wfile
        wrapper = \
'''#!/bin/sh

exec "%s" "%s/%s.py" $*
''' % (sys.executable,pydir,wfile)

    copyToFile(wname,wrapper)
    os.chmod(wname,0755)

    return wname


def copyTree(src, dst, filters, excludeDirs=[]):
    """
    Copy Python, translation, documentation, wizards configuration,
    designer template files and DTDs of a directory tree.
    
    @param src name of the source directory
    @param dst name of the destination directory
    @param filters list of filter pattern determining the files to be copied
    @param excludeDirs list of (sub)directories to exclude from copying
    """
    try:
        names = os.listdir(src)
    except OSError:
        return      # ignore missing directories (most probably the i18n directory)
    
    for name in names:
        srcname = os.path.join(src, name)
        dstname = os.path.join(dst, name)
        for filter in filters:
            if fnmatch.fnmatch(srcname, filter):
                if not os.path.isdir(dst):
                    os.makedirs(dst)
                shutil.copy2(srcname, dstname)
                break
        else:
            if os.path.isdir(srcname) and not srcname in excludeDirs:
                copyTree(srcname, dstname, filters)


def cleanUp():
    """
    Uninstall the old eric files.
    """
    from eric3config import getConfig
    global pyModDir
    
    # Remove the wrapper scripts
    rem_wnames = ["eric3", "eric3-unittest", "eric3-helpviewer",
        "eric3-doc", "eric3-qregexp", "eric3-re", "eric3-api",
        "eric3-uipreviewer", "eric3-trpreviewer",
        "qtunittest", "helpviewer"]
    for rem_wname in rem_wnames:
        rwname = wrapperName(getConfig('bindir'),rem_wname)
        if os.path.exists(rwname):
            os.remove(rwname)
    
    # Cleanup our config file
    for name in ['eric3config.py', 'eric3config.pyc']:
        e3cfile = os.path.join(pyModDir, name)
        if os.path.exists(e3cfile):
            os.remove(e3cfile)
        
    # Cleanup the install directories
    for name in ['ericExamplesDir', 'ericDocDir', 'ericDTDDir', 'ericCSSDir',
                 'ericIconDir', 'ericPixDir', 'ericDir', 'ericTemplatesDir',
                 'ericOthersDir']:
        try:
            shutil.rmtree(getConfig(name), 1)
        except:
            pass
    
    # Cleanup translations
    for name in glob.glob(os.path.join(getConfig('ericTranslationsDir'), 'eric3_*.qm')):
        try:
            os.remove(name)
        except:
            pass


def installEric():
    """
    Actually perform the installation steps.
    """
    global distDir, doCleanup, cfg
    
    # Create the platform specific wrappers.
    wnames = []
    wnames.append(createPyWrapper(cfg['ericDir'],"eric3"))
    wnames.append(createPyWrapper(cfg['ericDir'],"eric3-unittest"))
    wnames.append(createPyWrapper(cfg['ericDir'],"eric3-helpviewer"))
    wnames.append(createPyWrapper(cfg['ericDir'],"eric3-doc"))
    wnames.append(createPyWrapper(cfg['ericDir'],"eric3-qregexp"))
    wnames.append(createPyWrapper(cfg['ericDir'],"eric3-re"))
    wnames.append(createPyWrapper(cfg['ericDir'],"eric3-api"))
    wnames.append(createPyWrapper(cfg['ericDir'],"eric3-uipreviewer"))
    wnames.append(createPyWrapper(cfg['ericDir'],"eric3-trpreviewer"))
    
    # Create a config file and delete the default one
    createConfig()

    # set install prefix, if not None
    #
    # ToDo: This has to be reworked, if there is a need
    #       for distDir on "win32".
    #       Don't use os.path.join because absolute paths
    #       seem to be assumed, but not assured.
    #       At least for linux, double slashes make no problems.
    #
    if distDir:
        for key in cfg.keys():
            cfg[key] = distDir + os.sep + cfg[key]
    
    if doCleanup:
        if distDir:
            shutil.rmtree(distDir, 1)
        else:
            cleanUp()

    # Install the files
    # make the install directories
    for key in cfg.keys():
        if not os.path.isdir(cfg[key]):
            os.makedirs(cfg[key])
    
    # copy the new sitecustomize.py
    if distDir:
        shutil.copy('sitecustomize.py', cfg['mdir'])
    else:
        shutil.copy('sitecustomize.py', pyModDir)
    
    # copy the eric3 config file
    if distDir:
        shutil.copy('eric3config.py', cfg['mdir'])
    else:
        shutil.copy('eric3config.py', pyModDir)
    
    # copy the various parts of eric3
    copyTree('eric', cfg['ericDir'], ['*.py', '*.pyc', '*.pyo', '*.pyw'], 
        ['eric%sExamples' % os.sep])
    copyTree('eric', cfg['ericDir'], ['*.rb'], 
        ['eric%sExamples' % os.sep])
    copyTree('eric%sDocumentation' % os.sep, cfg['ericDocDir'], ['*.html'])
    copyTree('eric%sWizards' % os.sep, cfg['ericWizardsDir'], ['*.e3w'])
    copyTree('eric%sDTDs' % os.sep, cfg['ericDTDDir'], ['*.dtd'])
    copyTree('eric%sCSSs' % os.sep, cfg['ericCSSDir'], ['*.css'])
    copyTree('eric%si18n' % os.sep, cfg['ericTranslationsDir'], ['*.qm'])
    copyTree('eric%sicons' % os.sep, cfg['ericIconDir'], ['*.png', 'LICENSE*.*'])
    copyTree('eric%spixmaps' % os.sep, cfg['ericPixDir'], ['*.png', '*.xpm', '*.ico'])
    copyTree('eric%sDesignerTemplates' % os.sep, cfg['ericTemplatesDir'], ['*.tmpl'])
    copyTree('eric%sExamples' % os.sep, cfg['ericExamplesDir'], ['*.py', '*.pyc', '*.pyo'])
    
    # copy the wrappers
    for wname in wnames:
        shutil.copy(wname, cfg['bindir'])
    
    # copy the license file
    shutil.copy('eric%sLICENSE.GPL' % os.sep, cfg['ericDir'])
    
    # copy some text files to the doc area
    for name in ["LICENSE.GPL", "THANKS", "HISTORY"]:
        shutil.copy('eric%s%s' % (os.sep, name), cfg['ericDocDir'])
    for name in glob.glob(os.path.join('eric', 'README*.*')):
        shutil.copy(name, cfg['ericDocDir'])
        
    # copy some more stuff
    for name in ['default.e3k']:
        shutil.copy('eric%s%s' % (os.sep, name), cfg['ericOthersDir'])
    for name in ['eric3.api']:
        shutil.copy('eric%sDocumentation%s%s' % (os.sep, os.sep, name), cfg['ericOthersDir'])
        
    # copy some text files of BRM
    brmDocDir = os.path.join(cfg['ericDocDir'], 'brm')
    if not os.path.isdir(brmDocDir):
        os.makedirs(brmDocDir)
    for name in ['AUTHORS', 'COPYING', 'ChangeLog', 'DESIGN.html', 'NEWS', 'README']:
        shutil.copy(os.path.join('eric', 'ThirdParty', 'brm', name), brmDocDir)


def createSitecustomize():
    """
    Generate the sitecustomize.py script merging an already installed script with our additions.
    """
    global modDir, distDir

    exists = 1

    if not distDir:
        try:
            f = open(os.path.join(modDir, 'sitecustomize.py'), 'r')
        except:
            exists = 0
    else:
        exists = 0

    if exists:
        lines = f.readlines()
        f.close()
        s = open("sitecustomize.py", "w")
        if not "sys.setappdefaultencoding = sys.setdefaultencoding\n" in lines:
            s.write("#\n# added by the eric install script\n#\n\n")
            s.write("import sys\n\n")
            s.write("sys.setappdefaultencoding = sys.setdefaultencoding\n\n")
        for line in lines:
            s.write(line)
        s.close()
    else:
        s = open("sitecustomize.py", "w")
        s.write("#\n# sitecustomize.py - saving a useful function.\n")
        s.write("# generated by the eric install script\n#\n\n")
        s.write("import sys\n\n")
        s.write("sys.setappdefaultencoding = sys.setdefaultencoding\n\n")
        s.close()

def createInstallConfig():
    """
    Create the installation config dictionary.
    """
    global modDir, platBinDir, cfg
        
    ericdir = os.path.join(modDir, "eric3")
    cfg = {
        'ericDir'             : ericdir,
        'ericPixDir'          : os.path.join(ericdir, "pixmaps"),
        'ericIconDir'         : os.path.join(ericdir, "icons"),
        'ericDTDDir'          : os.path.join(ericdir, "DTDs"),
        'ericCSSDir'          : os.path.join(ericdir, "CSSs"),
        'ericDocDir'          : os.path.join(ericdir, "Documentation"),
        'ericExamplesDir'     : os.path.join(ericdir, "Examples"),
        'ericTranslationsDir' : os.path.join(ericdir, "i18n"),
        'ericWizardsDir'      : os.path.join(ericdir, "Wizards"),
        'ericTemplatesDir'    : os.path.join(ericdir, "DesignerTemplates"),
        'ericOthersDir'       : ericdir,
        'bindir'              : platBinDir,
        'mdir'                : modDir,
    }
configLength = 13
    
def createConfig():
    """
    Create a config file with the respective config entries.
    """
    global cfg, qtcfg
    
    fn = 'eric3config.py'
    config = \
"""# -*- coding: utf-8 -*-
#
# This module contains the configuration of the individual eric installation
#

_pkg_config = {
    'ericDir'             : r'%s',
    'ericPixDir'          : r'%s',
    'ericIconDir'         : r'%s',
    'ericDTDDir'          : r'%s',
    'ericCSSDir'          : r'%s',
    'ericDocDir'          : r'%s',
    'ericExamplesDir'     : r'%s',
    'ericTranslationsDir' : r'%s',
    'ericWizardsDir'      : r'%s',
    'ericTemplatesDir'    : r'%s',
    'ericOthersDir'       : r'%s',
    'bindir'              : r'%s',
    'mdir'                : r'%s',
    
    'qtui'                : %s,
    'qttable'             : %s,
    'qtcanvas'            : %s,
}

def getConfig(name):
    '''
    Module function to get a configuration value.

    @param name the name of the configuration value (string).
    '''
    try:
        return _pkg_config[name]
    except KeyError:
        pass

    raise AttributeError, '"%%s" is not a valid configuration value' %% name
""" % (cfg['ericDir'], cfg['ericPixDir'], cfg['ericIconDir'], 
       cfg['ericDTDDir'], cfg['ericCSSDir'], cfg['ericDocDir'],
       cfg['ericExamplesDir'], cfg['ericTranslationsDir'],
       cfg['ericWizardsDir'], cfg['ericTemplatesDir'],
       cfg['ericOthersDir'], cfg['bindir'], cfg['mdir'],
       qtcfg['qtui'], qtcfg['qttable'], qtcfg['qtcanvas'])
    copyToFile(fn, config)
    
def main(argv):
    """
    The main function of the script.

    @param argv the list of command line arguments.
    """
    import getopt

    # Parse the command line.
    global progName, modDir, doCleanup, cfg, qtcfg
    progName = os.path.basename(argv[0])

    initGlobals()

    try:
        if sys.platform == "win32":
            optlist, args = getopt.getopt(argv[1:],"chxb:d:f:")
        else:
            optlist, args = getopt.getopt(argv[1:],"chxb:d:f:i:")
    except getopt.GetoptError:
        usage()

    global platBinDir
    
    depChecks = 1

    for opt, arg in optlist:
        if opt == "-h":
            usage(0)
        elif opt == "-b":
            platBinDir = arg
        elif opt == "-d":
            global modDir
            modDir = arg
        elif opt == "-i":
            global distDir
            distDir = os.path.normpath(arg)
        elif opt == "-x":
            depChecks = 0
        elif opt == "-c":
            doCleanup = 0
        elif opt == "-f":
            try:
                execfile(arg, globals())
                if len(cfg) != configLength:
                    print "The configuration dictionary in '%s' is incorrect. Aborting" % arg
                    sys.exit(6)
            except:
                cfg = {}
    
    if len(cfg) == 0:
        createInstallConfig()
    
    qtcfg['qtui'] = True
    qtcfg['qttable'] = True
    qtcfg['qtcanvas'] = True
    
    if depChecks:
        # perform dependency checks
        if sys.version_info < (2,2,0):
            print 'Sorry, you must have Python 2.2.0 or higher.'
            sys.exit(5)
            
        try:
            from qt import qVersion, PYQT_VERSION
        except ImportError, msg:
            print 'Sorry, please install PyQt.'
            print 'Error: %s' % msg
            sys.exit(1)
            
        try:
            from qtnetwork import QServerSocket
        except ImportError, msg:
            print 'Sorry, PyQt is missing QtNetwork support.'
            print 'Error: %s' % msg
            sys.exit(1)
            
        try:
            from qtcanvas import QCanvas
        except ImportError, msg:
            print 'PyQt is missing QtCanvas support.'
            print 'Error: %s' % msg
            qtcfg['qtcanvas'] = False
            
        try:
            from qttable import QTable
        except ImportError, msg:
            print 'PyQt is missing QtTable support.'
            print 'Error: %s' % msg
            qtcfg['qttable'] = False
            
        try:
            from qtui import QWidgetFactory
        except ImportError, msg:
            print 'PyQt is missing QtUI support.'
            print 'Error: %s' % msg
            qtcfg['qtui'] = False
            
        try:
            from qtext import QextScintilla
        except ImportError, msg:
            print 'Sorry, please install QScintilla and/or reinstall'
            print 'PyQt with QScintilla support.'
            print 'Error: %s' % msg
            sys.exit(1)
            
        # check version of Qt
        qtMajor = int(qVersion().split('.')[0])
        qtMinor = int(qVersion().split('.')[1])
        if qtMajor < 3 or (qtMajor == 3 and qtMinor < 1):
            print 'Sorry, you must have Qt version 3.1.0 or higher.'
            sys.exit(2)
            
        #check version of PyQt
        try:
            from qt import PYQT_VERSION_STR
            pyqtVersion = PYQT_VERSION_STR
        except:
            pyqtVersion = PYQT_VERSION
        if pyqtVersion[:9] == "snapshot-":
            if long(pyqtVersion[9:]) < 20030418L:
                print 'Sorry, you must have PyQt snapshot-20030418 or higher.'
                sys.exit(3)
        else:
            while pyqtVersion.count('.') < 2:
                pyqtVersion += '.0'
            (maj, min, pat) = pyqtVersion.split('.')
            maj = int(maj)
            min = int(min)
            pat = int(pat)
            if maj < 3 or (maj == 3 and min < 6):
                print 'Sorry, you must have PyQt 3.6.0 or snapshot-20030418 or higher.'
                sys.exit(3)
            if maj == 3 and min == 9:
                print 'PyQt 3.9 contains a bug, which prevents eric3 from functioning.'
                print 'Please get PyQt 3.8 or 3.10 or newer (or snapshot-20031226 or newer).'
                sys.exit(3)
            
        #check version of QScintilla
        try:
            from qtext import QSCINTILLA_VERSION_STR
            scintillaVersion = QSCINTILLA_VERSION_STR
            if scintillaVersion[:9] == "snapshot-":
                if long(scintillaVersion[9:]) < 20041001L:
                    print 'Sorry, you must have QScintilla snapshot-20041001 or higher.'
                    sys.exit(4)
            else:
                while scintillaVersion.count('.') < 2:
                    scintillaVersion += '.0'
                (maj, min, pat) = scintillaVersion.split('.')
                maj = int(maj)
                min = int(min)
                pat = int(pat)
                if maj < 1 or (maj == 1 and min < 2):
                    print 'Sorry, you must have QScintilla 1.2 or snapshot-20041001 or higher.'
                    sys.exit(4)
        except:
            print 'Sorry, you must have QScintilla 1.2 or snapshot-20041001 or higher.'
            sys.exit(4)
        
    # get rid of development config file, if it exists
    try:
        os.remove(os.path.join("eric", "eric3config.py"))
    except:
        pass
    
    createSitecustomize()
    print "Compiling debugger files."
    if distDir:
        compileall.compile_dir("eric", ddir=os.path.join(distDir, modDir, cfg['ericDir']))
    else:
        compileall.compile_dir("eric", ddir=os.path.join(modDir, cfg['ericDir']))
    print "\nInstalling eric3 ..."
    installEric()
    print "\nInstallation complete."
    print "\nIf you would like to use the autocompletion feature please call"
    print "the gen_sip_api.py and/or gen_python_api.py script available"
    print "on the eric3 contribution page or get ready made api-files"
    print "from this contribution page."
    print
        
    # check PyXML version and output a message for broken PyXML 0.8.3
    try:
        import _xmlplus
        v = _xmlplus.version_info
        if v < (0, 8, 5):
            print "NOTE:"
            print "PyXML < 0.8.5 needs a patch to work correctly with foreign characters."
            print "Please see 'README-PyXML-0.8.3.txt' for details."
            print
    except:
        pass
    
    
if __name__ == "__main__":
    try:
        main(sys.argv)
    except SystemExit:
        raise
    except:
        print \
"""An internal error occured.  Please report all the output of the program,
including the following traceback, to eric-bugs@die-offenbachs.de.
"""
        raise

