"""label displaying a compacted filepath (Qt3)"""

import qt
from compactpath import compactPath, DEFAULT

#********************************************************************************
# lightweight wrapper class for QLabels to display a compactable filepath or url
# the class is designed as a wrapper not a derrived class so it does not get in the 
# way with QtDesigner
#
# problem to tackle is that every setText() triggers a resize of the whole layout
# so the filepath can't be adjusted in resizeEvent() (recursive). We handle 
# paintEvent() instead and silently draw the label from scratch.
#
# similar wrappers can be easily implemented for QMenuItem() and QListView() header 
# controls (...).
#
# Usage hint: 
# if used on statusbars you should adjust the stretch factor, otherwise
# the label might appear invisible, 'cos it may not get resized.
#
# The label has a minimum size of 0 by default. To force a minimum
# size set any initial text to the label wrapped. Effect is 1. the text will not be 
# displayed ++ the label will not get resized any smaller than this text.
#
#*******************************************************************************
class PathLabelWrap(object):
    """
    class wrapping a QLabel to display a filepath or url that is compacted on resizing
    of the label
    """
    
    def __init__(self, label, fpath="", prefix="", max_pardirs=1, type=DEFAULT):
        """
        constructor
        
        @param label an initialized QLabel to wrap compactPath funktionality around
        @param fpath the filepath or url the label should display
        @keyparam prefix: chars to be used as fixed prefix of the path like: "file: my/path/here"
        @keyparam max_pardirs maximum number of compacted parent dirs ("../") allowed 
            in the compacted path. Must be > 0.
        @keyparam sep use this to force a separator. Default is os.sep for filepaths 
            and "/" for urls.
       
        Note: currently the wrapper does not know how to display text for disabled state.
        So disabling the underlaying QLabel will not result in the desired effect.
        """
        
        
        self.label = label
        self.fpath = fpath
        self.prefix = prefix
        self.type = type
        self.max_pardirs= max_pardirs  
        
        self.label.paintEvent = self.onPaint    # overwrite
        #self.label.setPaletteBackgroundColor(qt.QColor(10, 255, 22))
        
    
    def onPaint(self, event):
        """
        overwritten method to handle painting of the displayed filepath
        """
        
        # draw the label from scratch
        # TODOS: check if there is an easier processing via QStyle.drawControl()
        fm = self.label.fontMetrics()
        rc = self.label.rect()
        frameW = self.label.frameWidth()
        indent = self.label.indent()
        if indent < 0:          # see Qt docs: label.indent()
            if frameW > 0:
                indent = fm.width("x") /2
            else:
                indent = 0
        rc.addCoords(frameW + indent, frameW , -(frameW + indent), -frameW)
        
        w = rc.width()
        if self.prefix:
            w -= fm.width(self.prefix)
        
        fpath = compactPath(
                        self.fpath, 
                        w, 
                        measure=fm.width, 
                        max_pardirs=self.max_pardirs,
                       type=self.type
                        )
                
        painter = qt.QPainter(self.label)
        painter.eraseRect(self.label.rect())
        self.label.drawFrame(painter)
        # TODOS: draw enabled/disabled text
        # textColor is already ok but I haven't found a way to draw embossed text (if required) 
        # if self.label.isEnabled():
        # else:
        if self.prefix:
            painter.drawText(rc, self.label.alignment(), '%s%s' % (self.prefix, fpath))
        else:
            painter.drawText(rc, self.label.alignment(), fpath)
        
    
    def setPath(self, fpath):
        """
        sets the filepath or url to be displayed
        """
        self.fpath = fpath
        self.label.update()
        
    def getPath(self):
        """
        retrieves the (uncompacted) filepath or url
        """
        return self.fpath

    
    #def printRc(self, rc):
    #    """just a helper for debugging"""
    #    print "(%s, %s, %s, %s)" % (rc.left(), rc.top(), rc.right(), rc.bottom())  



