# -*- coding: utf-8 -*-

# Copyright (c) 2004 - 2007 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the writer class for writing an XML project file.
"""

import os
import time
import pickle

from qt import qApp

from XMLWriterBase import XMLWriterBase
from Config import projectFileFormatVersion

import Preferences

class ProjectWriter(XMLWriterBase):
    """
    Class implementing the writer class for writing an XML project file.
    """
    def __init__(self, file, projectName):
        """
        Constructor
        
        @param file open file (like) object for writing
        @param projectName name of the project (string)
        """
        XMLWriterBase.__init__(self, file)
        
        self.pdata = qApp.mainWidget().getProject().pdata
        self.name = projectName
        
    def writeXML(self):
        """
        Public method to write the XML to the file.
        """
        XMLWriterBase.writeXML(self)
        
        self._write('<!DOCTYPE Project SYSTEM "Project-%s.dtd">' % projectFileFormatVersion)
        
        # add some generation comments
        self._write("<!-- Project file for project %s -->" % self.name)
        if Preferences.getProject("XMLTimestamp"):
            self._write("<!-- Saved: %s -->" % time.strftime('%Y-%m-%d, %H:%M:%S'))
            self._write("<!-- Copyright (C) %s %s, %s -->" % \
                    (time.strftime('%Y'), 
                     self.escape(self.pdata["AUTHOR"][0]), 
                     self.escape(self.pdata["EMAIL"][0])))
        
        # add the main tag
        self._write('<Project version="%s">' % projectFileFormatVersion)
        
        # do the programming language
        self._write('  <ProgLanguage mixed="%d">%s</ProgLanguage>' % \
            (self.pdata["MIXEDLANGUAGE"][0], self.pdata["PROGLANGUAGE"][0]))
        
        # do the UI type
        self._write('  <UIType>%s</UIType>' % self.pdata["UITYPE"][0])
        
        # do description
        if self.pdata["DESCRIPTION"]:
            self._write("  <Description>", 0)
            for d in self.pdata["DESCRIPTION"][:-1]:
                self._write(self.escape("%s" % d.strip()))
            self._write("%s</Description>" % self.escape(self.pdata["DESCRIPTION"][-1].strip()))
        
        # do version, author and email
        for key in ["VERSION", "AUTHOR", "EMAIL"]:
            element = key.capitalize()
            if self.pdata[key]:
                self._write("  <%s>%s</%s>" % (element, self.escape(self.pdata[key][0]), element))
            
        # do the translation prefix
        if self.pdata["TRANSLATIONPREFIX"]:
            self._write("  <TranslationPrefix>")
            parts = self.pdata["TRANSLATIONPREFIX"][0].split(os.sep)
            for p in parts[:-1]:
                self._write("    <Dir>%s</Dir>" % p)
            self._write("    <Name>%s</Name>" % parts[-1])
            self._write("  </TranslationPrefix>")
        
        # do the sources
        self._write("  <Sources>")
        for name in self.pdata["SOURCES"]:
            self._write("    <Source>")
            parts = name.split(os.sep)
            for p in parts[:-1]:
                self._write("      <Dir>%s</Dir>" % p)
            self._write("      <Name>%s</Name>" % parts[-1])
            self._write("    </Source>")
        self._write("  </Sources>")
        
        # do the forms
        self._write("  <Forms>")
        for name in self.pdata["FORMS"]:
            self._write("    <Form>")
            parts = name.split(os.sep)
            for p in parts[:-1]:
                self._write("      <Dir>%s</Dir>" % p)
            self._write("      <Name>%s</Name>" % parts[-1])
            self._write("    </Form>")
        self._write("  </Forms>")
        
        # do the translations
        self._write("  <Translations>")
        for name in self.pdata["TRANSLATIONS"]:
            self._write("    <Translation>")
            parts = name.split(os.sep)
            for p in parts[:-1]:
                self._write("      <Dir>%s</Dir>" % p)
            self._write("      <Name>%s</Name>" % parts[-1])
            self._write("    </Translation>")
        self._write("  </Translations>")
        
        # do the interfaces (IDL)
        self._write("  <Interfaces>")
        for name in self.pdata["INTERFACES"]:
            self._write("    <Interface>")
            parts = name.split(os.sep)
            for p in parts[:-1]:
                self._write("      <Dir>%s</Dir>" % p)
            self._write("      <Name>%s</Name>" % parts[-1])
            self._write("    </Interface>")
        self._write("  </Interfaces>")
        
        # do the others
        self._write("  <Others>")
        for name in self.pdata["OTHERS"]:
            self._write("    <Other>")
            parts = name.split(os.sep)
            for p in parts[:-1]:
                self._write("      <Dir>%s</Dir>" % p)
            self._write("      <Name>%s</Name>" % parts[-1])
            self._write("    </Other>")
        self._write("  </Others>")
        
        # do the main script
        if self.pdata["MAINSCRIPT"]:
            self._write("  <MainScript>")
            parts = self.pdata["MAINSCRIPT"][0].split(os.sep)
            for p in parts[:-1]:
                self._write("    <Dir>%s</Dir>" % p)
            self._write("    <Name>%s</Name>" % parts[-1])
            self._write("  </MainScript>")
        
        # do the vcs stuff
        self._write("  <Vcs>")
        if self.pdata["VCS"]:
            self._write("    <VcsType>%s</VcsType>" % self.pdata["VCS"][0])
        if self.pdata["VCSOPTIONS"]:
            self._write("    <VcsOptions>%s</VcsOptions>" % \
                self.escape(pickle.dumps(self.pdata["VCSOPTIONS"][0])))
        if self.pdata["VCSOTHERDATA"]:
            self._write("    <VcsOtherData>%s</VcsOtherData>" % \
                self.escape(pickle.dumps(self.pdata["VCSOTHERDATA"][0])))
        self._write("  </Vcs>")
        
        # do the eric3-doc stuff
        if len(self.pdata["ERIC3DOCPARMS"]):
            self._write("  <Eric3Doc>")
            if self.pdata["ERIC3DOCPARMS"]:
                self._write("    <Eric3DocParams>%s</Eric3DocParams>" % \
                    self.escape(pickle.dumps(self.pdata["ERIC3DOCPARMS"][0])))
            self._write("  </Eric3Doc>")
        
        # do the eric3-api stuff
        if len(self.pdata["ERIC3APIPARMS"]):
            self._write("  <Eric3Api>")
            if self.pdata["ERIC3APIPARMS"]:
                self._write("    <Eric3ApiParams>%s</Eric3ApiParams>" % \
                    self.escape(pickle.dumps(self.pdata["ERIC3APIPARMS"][0])))
            self._write("  </Eric3Api>")
        
        # do the filetype associations
        self._write("  <FiletypeAssociations>")
        for pattern, filetype in self.pdata["FILETYPES"].items():
            self._write('    <FiletypeAssociation pattern="%s" type="%s" />' % \
                (pattern, filetype))
        self._write("  </FiletypeAssociations>")
        
        # do the cx_Freeze stuff
        if len(self.pdata["CXFREEZEPARMS"]):
            self._write("  <Cxfreeze>")
            if self.pdata["CXFREEZEPARMS"]:
                self._write("    <CxfreezeParams>%s</CxfreezeParams>" % \
                    self.escape(pickle.dumps(self.pdata["CXFREEZEPARMS"][0])))
            self._write("  </Cxfreeze>")
        
        # do the pylint stuff
        if len(self.pdata["PYLINTPARMS"]):
            self._write("  <PyLint>")
            if self.pdata["PYLINTPARMS"]:
                self._write("    <PyLintParams>%s</PyLintParams>" % \
                    self.escape(pickle.dumps(self.pdata["PYLINTPARMS"][0])))
            self._write("  </PyLint>")
        
        self._write("</Project>")
