# -*- coding: utf-8 -*-

# Copyright (c) 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the handler class for reading an XML project file.
"""

import codecs
import sys
import os
from types import UnicodeType

from ThirdParty.XMLFilter import XMLSAXHandler

class ProjectHandler(XMLSAXHandler):
    """
    Class implementing a sax handler to read an XML project file.
    """
    def __init__(self, project):
        """
        Constructor
        
        @param project Reference to the project object to store the
                information into.
        """
        self.elements = {
            'Description' : (self.defaultStartElement, self.endDescription),
            'Version' : (self.defaultStartElement, self.endVersion),
            'Author' : (self.defaultStartElement, self.endAuthor),
            'Email' : (self.defaultStartElement, self.endEmail),
            'VcsType' : (self.defaultStartElement, self.endVcsType),
            'VcsOptions' : (self.defaultStartElement, self.endVcsOptions),
            'Eric3DocParams' : (self.defaultStartElement, self.endEric3DocParams),
            'HappyDocParams' : (self.defaultStartElement, self.endHappyDocParams),
            'Dir' : (self.defaultStartElement, self.endDir),
            'Name' : (self.defaultStartElement, self.endName),
            'Source' : (self.startSource, self.endSource),
            'Form' : (self.startForm, self.endForm),
            'Translation' : (self.startTranslation, self.endTranslation),
            'Interface' : (self.startInterface, self.endInterface),
            'Other' : (self.startOther, self.endOther),
            'MainScript' : (self.startMainScript, self.endMainScript),
        }
    
        self.project = project
        self.encoder = codecs.lookup(sys.getdefaultencoding())[0] # encode,decode,reader,writer
        
    def utf8_to_code(self, text):
        """
        Private method to convert a string to unicode and encode it for XML.
        
        @param text the text to encode (string)
        """
        if type(text) is not UnicodeType:
            text = unicode(text, "utf-8")
        return self.encoder(text)[0] # result,size
        
    def startDocument(self):
        """
        Handler called, when the document parsing is started.
        """
        self.buffer = ""
        self.pathStack = []
        
    def startElement(self, name, attrs):
        """
        Handler called, when a starting tag is found.
        
        @param name name of the tag (string)
        @param attrs list of tag attributes
        """
        try:
            self.elements[name][0](attrs)
        except KeyError:
            pass
            
    def endElement(self, name):
        """
        Handler called, when an ending tag is found.
        
        @param name name of the tag (string)
        """
        try:
            self.elements[name][1]()
        except KeyError:
            pass
            
    def characters(self, chars):
        """
        Handler called for ordinary text.
        
        @param chars the scanned text (string)
        """
        self.buffer += chars
        
    ###################################################
    ## below follow the individual handler functions
    ###################################################
    
    def defaultStartElement(self, attrs):
        """
        Handler method for common start tags.
        
        @param attrs list of tag attributes
        """
        self.buffer = ""
        
    def endDescription(self):
        """
        Handler method for the "Description" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.project.pdata["DESCRIPTION"] = self.buffer.split(os.linesep)
        
    def endVersion(self):
        """
        Handler method for the "Version" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.project.pdata["VERSION"] = [self.buffer]
        
    def endAuthor(self):
        """
        Handler method for the "Author" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.project.pdata["AUTHOR"] = [self.buffer]
        
    def endEmail(self):
        """
        Handler method for the "Email" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.project.pdata["EMAIL"] = [self.buffer]
        
    def endVcsType(self):
        """
        Handler method for the "VcsType" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.project.pdata["VCS"] = [self.buffer]
        
    def endVcsOptions(self):
        """
        Handler method for the "VcsOptions" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.project.pdata["VCSOPTIONS"] = [self.buffer]
        
    def endEric3DocParams(self):
        """
        Handler method for the "Eric3DocParams" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.project.pdata["ERIC3DOCPARMS"] = [self.buffer]
        
    def endHappyDocParams(self):
        """
        Handler method for the "HappyDocParams" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.project.pdata["HAPPYDOCPARMS"] = [self.buffer]
        
    def endDir(self):
        """
        Handler method for the "Dir" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.pathStack.append(self.buffer)
        
    def endName(self):
        """
        Handler method for the "Name" end tag.
        """
        self.buffer = self.utf8_to_code(self.buffer)
        self.pathStack.append(self.buffer)
        
    def startSource(self, attrs):
        """
        Handler method for the "Source" start tag.
        
        @param attrs list of tag attributes
        """
        self.pathStack = []
        
    def endSource(self):
        """
        Handler method for the "Source" end tag.
        """
        path = self.buildPath()
        self.project.pdata["SOURCES"].append(path)
        
    def startForm(self, attrs):
        """
        Handler method for the "Form" start tag.
        
        @param attrs list of tag attributes
        """
        self.pathStack = []
        
    def endForm(self):
        """
        Handler method for the "Form" end tag.
        """
        path = self.buildPath()
        self.project.pdata["FORMS"].append(path)
        
    def startTranslation(self, attrs):
        """
        Handler method for the "Translation" start tag.
        
        @param attrs list of tag attributes
        """
        self.pathStack = []
        
    def endTranslation(self):
        """
        Handler method for the "Translation" end tag.
        """
        path = self.buildPath()
        self.project.pdata["TRANSLATIONS"].append(path)
        
    def startInterface(self, attrs):
        """
        Handler method for the "Interface" start tag.
        
        @param attrs list of tag attributes
        """
        self.pathStack = []
        
    def endInterface(self):
        """
        Handler method for the "Interface" end tag.
        """
        path = self.buildPath()
        self.project.pdata["INTERFACES"].append(path)
        
    def startOther(self, attrs):
        """
        Handler method for the "Other" start tag.
        
        @param attrs list of tag attributes
        """
        self.pathStack = []
        
    def endOther(self):
        """
        Handler method for the "Other" end tag.
        """
        path = self.buildPath()
        self.project.pdata["OTHERS"].append(path)
        
    def startMainScript(self, attrs):
        """
        Handler method for the "MainScript" start tag.
        
        @param attrs list of tag attributes
        """
        self.pathStack = []
        
    def endMainScript(self):
        """
        Handler method for the "MainScript" end tag.
        """
        path = self.buildPath()
        self.project.pdata["MAINSCRIPT"] = [path]
        
    def buildPath(self):
        """
        Private method to assemble a path.
        
        @return The ready assembled path. (string)
        """
        path = ""
        if self.pathStack and not self.pathStack[0]:
            self.pathStack[0] = os.sep
        for p in self.pathStack:
            path = os.path.join(path, p)
        return path
