# -*- coding: utf-8 -*-

# Copyright (c) 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a dialog to show the output of the svn status command process.
"""

from qt import *

from StatusForm import StatusForm

class SvnStatusDialog(StatusForm):
    """
    Module implementing a dialog to show the output of the svn status command process.
    """
    def __init__(self, vcs, parent = None):
        """
        Constructor
        
        @param vcs -- reference to the vcs object
        @param parent -- parent widget (QWidget)
        """
        StatusForm.__init__(self, parent)
        
        self.setWFlags(self.getWFlags() | Qt.WDestructiveClose)
        self.process = QProcess(self)
        self.vcs = vcs
        
        self.statusList.setSorting(8)
        
        self.connect(self.process, SIGNAL('readyReadStdout()'),
            self.handleReadStdout)
        self.connect(self.process, SIGNAL('readyReadStderr()'),
            self.handleReadStderr)
        self.connect(self.process, SIGNAL('processExited()'),
            self.handleProcessExited)
            
        self.status = {
            ' ' : self.trUtf8('normal'),
            'A' : self.trUtf8('added'),
            'D' : self.trUtf8('deleted'),
            'M' : self.trUtf8('modified'),
            'C' : self.trUtf8('conflict'),
            '?' : self.trUtf8('unversioned'),
            '!' : self.trUtf8('missing'),
            '~' : self.trUtf8('type error')
        }
        self.propStatus = {
            ' ' : self.trUtf8('normal'),
            'M' : self.trUtf8('modified'),
            'C' : self.trUtf8('conflict')
        }
        self.locked = {
            ' ' : self.trUtf8('no'),
            'L' : self.trUtf8('yes')
        }
        self.history = {
            ' ' : self.trUtf8('no'),
            '+' : self.trUtf8('yes')
        }
        self.switched = {
            ' ' : self.trUtf8('no'),
            'S' : self.trUtf8('yes')
        }
        
        self.rx_status = QRegExp('(.{8})\\s+([0-9]+)\\s+([0-9?]+)\\s+([\\w?]+)\\s+(.+)')
            
    def closeEvent(self, e):
        """
        Private slot implementing a close event handler.
        
        @param e -- close event (QCloseEvent)
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        e.accept()
        
    def start(self, fn):
        """
        Public slot to start the svn status command.
        
        @param fn -- filename to be diffed (string)
        """
        self.filename = fn
        dname, fname = self.vcs.splitPath(fn)
        
        self.process.kill()
        
        self.process.clearArguments()
        self.process.addArgument('svn')
        self.process.addArgument('status')
        self.vcs.addArguments(self.process, self.vcs.options['global'])
        self.vcs.addArguments(self.process, self.vcs.options['status'])
        self.process.addArgument('--verbose')
        if '--show-updates' in self.vcs.options['status'] or \
           '-u' in self.vcs.options['status']:
            try:
                if self.vcs.svnGetReposName(dname).startswith('http'):
                    self.vcs.addArguments(self.process, self.vcs.authData())
                self.setActiveWindow()
                self.raiseW()
            except:
                pass
        self.process.addArgument(fname)
        self.process.setWorkingDirectory(QDir(dname))
        
        self.process.start()
        self.setCaption(self.trUtf8('Subversion Status %1').arg(self.filename))
        
    def finish(self):
        """
        Private slot called when the process finished or the user pressed the button.
        """
        if self.process is not None:
            self.process.tryTerminate()
            QTimer.singleShot(2000, self.process, SLOT('kill()'))
            
        self.cancelButton.setText(self.trUtf8('OK'))
        self.cancelButton.setDefault(1)
        
        self.process = None
        
    def buttonPressed(self):
        """
        Private slot connected to the button clicked signal.
        """
        if self.process is None:
            self.close()
        else:
            self.finish()
            
    def handleProcessExited(self):
        """
        Private slot to handle the processExited signal.
        
        Just call finish().
        """
        self.finish()
        
    def handleReadStdout(self):
        """
        Private slot to handle the readyReadStdout signal.
        
        It reads the output of the process, formats it and inserts it into
        the contents pane.
        """
        while self.process.canReadLineStdout():
            s = self.process.readLineStdout()
            if self.rx_status.exactMatch(s):
                flags = str(self.rx_status.cap(1))
                rev = self.rx_status.cap(2)
                change = self.rx_status.cap(3)
                author = self.rx_status.cap(4)
                path = self.rx_status.cap(5)
                itm = QListViewItem(self.statusList,
                    self.status[flags[0]], self.propStatus[flags[1]],
                    self.locked[flags[2]], self.history[flags[3]],
                    self.switched[flags[4]], rev, change, author)
                itm.setText(8, path)
                
    def handleReadStderr(self):
        """
        Private slot to handle the readyReadStderr signal.
        
        It reads the error output of the process and inserts it into the
        error pane.
        """
        while self.process.canReadLineStderr():
            s = self.process.readLineStderr()
            self.errors.append(s)
