# -*- coding: utf-8 -*-

# Copyright (c) 2003 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing the message box wizard.
"""

from qt import QAction, QMessageBox, QDialog, SIGNAL

from Wizards.WizardIFace import WizardIFace
from MessageBoxWizardDialog import MessageBoxWizardDialog

class MessageBoxWizard(WizardIFace):
    """
    Class implementing the message box wizard.
    """
    def __init__(self, parent = None):
        """
        Constructor
        
        @param parent parent widget (QWidget)
        """
        WizardIFace.__init__(self, parent)
        
    def callForm(self, editor):
        """
        Private method to display a dialog and get the code.
        
        @param editor reference to the current editor
        @return the generated code (string)
        """
        dlg = MessageBoxWizardDialog(None)
        if dlg.exec_loop() == QDialog.Accepted:
            line, index = editor.getCursorPosition()
            indLevel = editor.indentation(line)/editor.indentationWidth()
            if editor.indentationsUseTabs():
                indString = '\t'
            else:
                indString = editor.indentationWidth() * ' '
            return (dlg.getCode(indLevel, indString), 1)
        else:
            return (None, 0)
        
    def handle(self):
        """
        Worker method associated with the wizard action 
        """
        editor = self.parent().getCurrentEditor()
        
        if editor == None:
                QMessageBox.critical(None, 
                self.trUtf8('No current editor'),
                self.trUtf8('Please open or create a file first.'))

        else:
            code, ok = self.callForm(editor)
            if ok:
                line, index = editor.getCursorPosition()
                # It should be done on this way to allow undo
                editor.beginUndoAction()
                editor.insertAt(code, line, index)
                editor.endUndoAction()

        
    def getAction(self):
        """
        Public method to get the action associated with this class wizard. 
        
        @return the associated action
        """
        self.action = QAction(self.trUtf8('QMessageBox Wizard'),
             self.trUtf8('Q&MessageBox Wizard...'), 0, self)
        self.action.setStatusTip(self.trUtf8('QMessageBox Wizard'))
        self.action.setWhatsThis(self.trUtf8(
            """<b>QMessageBox Wizard</b>"""
            """<p>This wizard opens a dialog for entering all the parameters"""
            """ needed to create a QMessageBox. The generated code is inserted"""
            """ at the current cursor position.</p>"""
        ))
        self.connect(self.action, SIGNAL('activated()'), self.handle)
        return self.action
