#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2002 - 2004 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Eric3 Python IDE

This is the main Python script that performs the neccessary initialization
of the IDE and starts the Qt event loop.
"""

import sys
import os
from qt import QApplication, QTranslator, QTextCodec, QPixmap, QPoint, \
    QLabel, Qt, QFrame, QRect, QPainter, QFont, QMimeSourceFactory, \
    SIGNAL, SLOT, QVBox, qApp

from UI.Info import Program, Version
from UI.SplashScreen import SplashScreen, NoneSplashScreen
import Preferences

def loadTranslator(dirs, tn):
    """
    Global function to find and load a specific translation.

    @param dirs Searchpath for the translations. (list of strings)
    @param tn The translation to be loaded. (string)
    @return Tuple of a status flag and the loaded translator. (int, QTranslator)
    """
    trans = QTranslator(None)
    for dir in dirs:
        loaded = trans.load(tn, dir)
        if loaded:
            return (trans, 1)
            
    print "eric: no translation file '" + tn + "'found."
    print "Using default."
    return (None, 0)

def initializeMimeSourceFactory(ericDir):
    """
    Global function to initialize the default mime source factory.
    
    @param ericDir directory name of the eric installation
    """
    defaultFactory = QMimeSourceFactory.defaultFactory()
    defaultIconPath = os.path.join(ericDir, "icons", "default")
    iconPaths = Preferences.getIcons("Path")
    for iconPath in iconPaths:
        defaultFactory.addFilePath(iconPath)
    if not defaultIconPath in iconPaths:
        defaultFactory.addFilePath(defaultIconPath)

def main():
    """
    Main entry point into the application.
    """
    ericDir = os.path.dirname(sys.argv[0])
    if ericDir == '':
        ericDir = '.'
    app = QApplication(sys.argv)
    
    # set the searchpath for icons
    initializeMimeSourceFactory(ericDir)

    # generate and show a splash window, if not suppressed
    if "--nosplash" in sys.argv:
        del sys.argv[sys.argv.index("--nosplash")]
        splash = NoneSplashScreen(ericDir)
    else:
        splash = SplashScreen(ericDir)

    # is there a set of filenames or options on the command line,
    # if so, pass them to the UI
    args = None
    if len(sys.argv) > 1:
        args = sys.argv[1:]
    
    # Set the applications string encoding
    try:
        sys.setappdefaultencoding(str(Preferences.getSystem("StringEncoding")))
    except:
        pass
    
    # get the Qt installation directory
    qtdir = Preferences.getQtDir()
    if qtdir is not None:
        if Preferences.getQt("ExportQtDir"):
            os.environ["QTDIR"] = qtdir
    
    # Load translation files and install them
    loaded = 0
    loc = Preferences.getUILanguage()
    if loc is not None:
        if loc == "System":
            loc = str(QTextCodec.locale())
        
        if loc != "C":
            # setup the searchpath for qt and qscintilla
            dirs = [ericDir, "."]
            if qtdir is not None:
                dirs.insert(-1, os.path.join(qtdir, "translations"))
                
            # 1) load translations for qt
            qtTrans, ok = loadTranslator(dirs, "qt_" + loc)
            if ok:
                app.installTranslator(qtTrans)
                
            # 2) load translations for qscintilla
            scintillaTrans, ok = loadTranslator(dirs, "qscintilla_" + loc)
            if ok:
                app.installTranslator(scintillaTrans)
                
            # 3) load translations for eric
            dirs = [".", ericDir]
            ericTrans, ok = loadTranslator(dirs, "eric3_" + loc)
            if ok:
                app.installTranslator(ericTrans)
            else:
                print "Please report to <eric-bugs@die-offenbachs.de>."
        else:
            loc = None
        
    splash.message(qApp.translate("eric3", "Importing packages..."))
    # We can only import these after creating the QApplication because they
    # make Qt calls that need the QApplication to exist.
    from UI.UserInterface import UserInterface

    splash.message(qApp.translate("eric3", "Generating Main Window..."))
    mw = UserInterface(loc, splash)
    app.setMainWidget(mw)
    app.connect(app, SIGNAL("lastWindowClosed()"), app, SLOT("quit()"))
    mw.show()

    splash.message(qApp.translate("eric3", "Loading files..."))
    mw.processArgs(args)
    
    if splash:
        del splash
        
    res = app.exec_loop()
    sys.exit(res)

if __name__ == '__main__':
    main()
