%%--------------------------------------------------------------------
%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.1, (the "License"); you may not use this file except in
%% compliance with the License. You should have received a copy of the
%% Erlang Public License along with this software. If not, it can be
%% retrieved via the world wide web at http://www.erlang.org/.
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Initial Developer of the Original Code is Ericsson Utvecklings AB.
%% Portions created by Ericsson are Copyright 1999, Ericsson Utvecklings
%% AB. All Rights Reserved.''
%% 
%%     $Id$
%%
%%-----------------------------------------------------------------
%% File: orber_iiop_pm.erl
%% Description:
%%    This file contains the mapping of addresses on the format {Host, Port} 
%%    to a proxy pid.
%%
%% Creation date: 990615
%%
%%-----------------------------------------------------------------
-module(orber_iiop_pm).

-behaviour(gen_server).

-include_lib("orber/src/orber_iiop.hrl").
-include_lib("orber/include/corba.hrl").
-include_lib("kernel/include/inet.hrl").

%%-----------------------------------------------------------------
%% External exports
%%-----------------------------------------------------------------
-export([start/0, start/1]).

%%-----------------------------------------------------------------
%% Internal exports
%%-----------------------------------------------------------------
-export([connect/6, disconnect/2, list_existing_connections/0, 
	 list_setup_connections/0, list_all_connections/0,
	 init/1, handle_call/3, handle_cast/2, handle_info/2,
	 code_change/3, terminate/2, stop/0, setup_connection/7]).

%%-----------------------------------------------------------------
%% Macros/Defines
%%-----------------------------------------------------------------
-define(DEBUG_LEVEL, 7).

-define(PM_CONNECTION_DB, orber_iiop_pm_db).

-record(state, {connections, queue}).

-record(connection, {hp, child, interceptors, slave, flags = 0, alias = 0}).

%%-----------------------------------------------------------------
%% External interface functions
%%-----------------------------------------------------------------
start() ->
    ignore.
start(Opts) ->
    gen_server:start_link({local, 'orber_iiop_pm'}, ?MODULE, Opts, []).


connect(Host, Port, SocketType, Timeout, Chars, Wchars) when SocketType == normal ->
    case ets:lookup(?PM_CONNECTION_DB, {Host, Port}) of
	[#connection{child = connecting}] ->
	    gen_server:call(orber_iiop_pm, {connect, Host, Port, SocketType, 
					    [], Chars, Wchars}, Timeout);
	[] ->
	    gen_server:call(orber_iiop_pm, {connect, Host, Port, SocketType, 
					    [], Chars, Wchars}, Timeout);
	[#connection{child = P, interceptors = I}] ->
	    {P, [], I}
    end;
connect(Host, Port, SocketType, Timeout, Chars, Wchars) 
  when SocketType == ssl ->
    case ets:lookup(?PM_CONNECTION_DB, {Host, Port}) of
	[#connection{child = connecting}] ->
	    SocketOptions = get_ssl_socket_options(),
	    gen_server:call(orber_iiop_pm, {connect, Host, Port, SocketType, 
					    SocketOptions, Chars, Wchars}, Timeout);
	[] ->
	    SocketOptions = get_ssl_socket_options(),
	    gen_server:call(orber_iiop_pm, {connect, Host, Port, SocketType, 
					    SocketOptions, Chars, Wchars}, Timeout);
	[#connection{child = P, interceptors = I}] ->
	    {P, [], I}
    end.

get_ssl_socket_options() ->
    [{verify, orber:ssl_client_verify()},
     {depth, orber:ssl_client_depth()} |
     ssl_client_extra_options([{certfile, orber:ssl_client_certfile()},
			       {cacertfile, orber:ssl_client_cacertfile()},
			       {password, orber:ssl_client_password()},
			       {keyfile, orber:ssl_client_keyfile()},
			       {ciphers, orber:ssl_client_ciphers()},
			       {cachetimeout, orber:ssl_client_cachetimeout()}], [])].

ssl_client_extra_options([], Acc) ->
    Acc;
ssl_client_extra_options([{_Type, []}|T], Acc) ->
    ssl_client_extra_options(T, Acc);
ssl_client_extra_options([{_Type, infinity}|T], Acc) ->
    ssl_client_extra_options(T, Acc);
ssl_client_extra_options([{Type, Value}|T], Acc) ->
    ssl_client_extra_options(T, [{Type, Value}|Acc]).


disconnect(Host, Port) ->
    gen_server:call(orber_iiop_pm, {disconnect, Host, Port}).

list_existing_connections() ->
    do_select([{#connection{hp = '$2', child = '$1', _='_'}, 
		[{is_pid, '$1'}], ['$2']}]).

list_setup_connections() ->
    do_select([{#connection{hp = '$1', child = connecting, _='_'}, [], ['$1']}]).

list_all_connections() ->
    do_select([{#connection{hp = '$1', _='_'}, [], ['$1']}]).

do_select(Pattern) ->   
    case catch ets:select(?PM_CONNECTION_DB, Pattern) of
	{'EXIT', _What} ->
	    [];
	Result ->
	    Result
    end.

%%-----------------------------------------------------------------
%% Internal interface functions
%%-----------------------------------------------------------------
%%-----------------------------------------------------------------
%% Func: stop/0 (Only used for test purpose !!!!!!)
%%-----------------------------------------------------------------
stop() ->
    gen_server:call(orber_iiop_pm, stop).

%%-----------------------------------------------------------------
%% Server functions
%%-----------------------------------------------------------------
%%-----------------------------------------------------------------
%% Func: init/1
%%-----------------------------------------------------------------
init(_Opts) ->
    process_flag(trap_exit, true),
    {ok, #state{connections = ets:new(orber_iiop_pm_db, 
				      [{keypos, 2}, set, protected, named_table]),
		queue = ets:new(orber_iiop_pm_queue, [bag])}}.

%%-----------------------------------------------------------------
%% Func: terminate/2
%%-----------------------------------------------------------------
terminate(_Reason, #state{queue = Q}) ->
    %% Kill all proxies and close table before terminating
    stop_all_proxies(ets:first(?PM_CONNECTION_DB)),
    ets:delete(?PM_CONNECTION_DB),
    ets:delete(Q),
    ok.

stop_all_proxies('$end_of_table') ->
    ok;
stop_all_proxies(Key) ->
    case ets:lookup(?PM_CONNECTION_DB, Key) of
	[] ->
	    ok;
	[#connection{child = connecting, interceptors = I}] ->
	    catch invoke_connection_closed(I);
	[#connection{child = P, interceptors = I}] ->
	    catch invoke_connection_closed(I),
	    catch orber_iiop_outproxy:stop(P)
    end,
    stop_all_proxies(ets:next(?PM_CONNECTION_DB, Key)).

%%-----------------------------------------------------------------
%% Func: handle_call/3
%%-----------------------------------------------------------------
handle_call({connect, Host, Port, SocketType, SocketOptions, Chars, Wchars}, From, State) ->
    case ets:lookup(?PM_CONNECTION_DB, {Host, Port}) of
	[#connection{child = connecting}] ->
	    %% Another client already requested a connection to the given host/port. 
	    %% Just add this client to the queue.
	    ets:insert(State#state.queue, {{Host, Port}, From}),
	    {noreply, State};
	[#connection{child = P, interceptors = I}] ->
	    %% This case will occur if the PortMapper completed a connection
	    %% between the client's ets:lookup and receiving this request.
	    {reply, {P, [], I}, State};
	[] ->
	    %% The first time a connection is requested to the given host/port.
	    case catch spawn_link(?MODULE, setup_connection, 
				  [self(), Host, Port, SocketType, 
				   SocketOptions, Chars, Wchars]) of
		Slave when pid(Slave) ->
		    ets:insert(?PM_CONNECTION_DB, 
			       #connection{hp = {Host, Port}, child = connecting, 
					   interceptors = false, slave = Slave}),
		    ets:insert(State#state.queue, {{Host, Port}, From}),
		    {noreply, State};
		What ->
		    orber:dbg("[~p] orber_iiop_pm:handle_call(connect);~n"
			      "Unable to invoke setup_connection due to: ~n~p~n", 
			      [?LINE, What], ?DEBUG_LEVEL),
		    {reply, 
		     {'EXCEPTION', #'INTERNAL'{completion_status=?COMPLETED_NO}}, 
		     State}
	    end
    end;
handle_call({disconnect, Host, Port}, _From, State) ->
    case ets:lookup(?PM_CONNECTION_DB, {Host, Port}) of
	[] ->
	    ok;
	[#connection{child = connecting, interceptors = I}] ->
	    ets:delete(?PM_CONNECTION_DB, {Host, Port}),
	    Exc = {'EXCEPTION',#'INTERNAL'{completion_status = ?COMPLETED_NO}},
	    send_reply_to_queue(ets:lookup(State#state.queue, {Host, Port}), Exc),
	    ets:delete(State#state.queue, {Host, Port}),
	    catch invoke_connection_closed(I);
	[#connection{child = P, interceptors = I}] ->
	    unlink(P),
	    catch orber_iiop_outproxy:stop(P),
	    ets:delete(?PM_CONNECTION_DB, {Host, Port}),
	    catch invoke_connection_closed(I)
    end,
    {reply, ok, State};
handle_call(stop, _From, State) ->
    {stop, normal, ok, State};
handle_call(_, _, State) ->
    {noreply, State}.

%%-----------------------------------------------------------------
%% Func: handle_cast/2
%%-----------------------------------------------------------------
handle_cast(stop, State) ->
    {stop, normal, State};
handle_cast(_, State) ->
    {noreply, State}.

%%-----------------------------------------------------------------
%% Func: handle_info/2
%%-----------------------------------------------------------------
%% Trapping exits 
handle_info({'EXIT', Pid, Reason}, State) ->
    %% Check the most common scenario first, i.e., a proxy terminates.
    case ets:match_object(?PM_CONNECTION_DB, #connection{child = Pid, _='_'}) of
	[#connection{hp = K, interceptors = I}] ->
	    ets:delete(?PM_CONNECTION_DB, K),
	    invoke_connection_closed(I),
	    {noreply, State};
	[] when Reason == normal ->
	    %% This might have been a spawned 'setup_connection' which terminated
	    %% after sucessfully setting up a new connection.
	    {noreply, State};
	[] ->
	    %% Wasn't a proxy. Hence, we must test if it was a spawned
	    %% 'setup_connection' that failed.
	    case ets:match_object(?PM_CONNECTION_DB, #connection{slave = Pid, _='_'}) of
		[#connection{hp = K, child = connecting, interceptors = I}] ->
		    ets:delete(?PM_CONNECTION_DB, K),
		    invoke_connection_closed(I),
		    Exc = {'EXCEPTION',#'INTERNAL'{completion_status = ?COMPLETED_NO}},
		    send_reply_to_queue(ets:lookup(State#state.queue, K), Exc),
		    ets:delete(State#state.queue, K),
		    orber:dbg("[~p] orber_iiop_pm:handle_info(setup_failed ~p);~n"
			      "It was not possible to create a connection to the given host/port.",
			      [?LINE, K], ?DEBUG_LEVEL),
		    {noreply, State};
		_ ->
		    {noreply, State}
	    end
    end;
handle_info({setup_failed, {Host, Port}, Exc}, State) ->
    %% Deletet the data from the connection DB first to avoid clients from
    %% trying to access it again.
    ets:delete(?PM_CONNECTION_DB, {Host, Port}),
    %% Now we can send whatever exception received.
    send_reply_to_queue(ets:lookup(State#state.queue, {Host, Port}), Exc),
    ets:delete(State#state.queue, {Host, Port}),
    orber:dbg("[~p] orber_iiop_pm:handle_info(setup_failed ~p ~p);~n"
	      "It was not possible to create a connection to the given host/port.", 
	      [?LINE, Host, Port], ?DEBUG_LEVEL),
    {noreply, State};
handle_info({setup_successfull, {Host, Port}, {Child, Ctx, Int}}, State) ->
    %% Create a link to the proxy and store it in the connection DB.
    link(Child),
    ets:insert(?PM_CONNECTION_DB, #connection{hp = {Host, Port}, child = Child, 
					      interceptors = Int, 
					      slave = undefined}),
    %% Send the Proxy reference to all waiting clients.
    send_reply_to_queue(ets:lookup(State#state.queue, {Host, Port}),{Child, Ctx, Int}),
    %% Reset the queue.
    ets:delete(State#state.queue, {Host, Port}),
    {noreply, State};
handle_info(_, State) ->
    {noreply, State}.


send_reply_to_queue([], _) ->
    ok;
send_reply_to_queue([{_, Client}|T], Reply) ->
    gen_server:reply(Client, Reply),
    send_reply_to_queue(T, Reply). 

%%-----------------------------------------------------------------
%% Func: code_change/3
%%-----------------------------------------------------------------
code_change(_OldVsn, State, _Extra) ->
    {ok, State}.

%%-----------------------------------------------------------------
%% Internal functions
%%-----------------------------------------------------------------
setup_connection(PMPid, Host, Port, SocketType, SocketOptions, Chars, Wchars) ->
    case catch access_allowed(Host, Port, SocketType) of
	ok ->
	    do_setup_connection(PMPid, Host, Port, SocketType, SocketOptions, 
				Chars, Wchars);
	{ok, Interface} ->
	    do_setup_connection(PMPid, Host, Port, SocketType, 
				[{ip, Interface}|SocketOptions], 
				Chars, Wchars);
	_ ->
	    orber_tb:info("Blocked connect attempt to ~s - ~p", [Host, Port]),
	    PMPid ! {setup_failed, {Host, Port}, 
		     {'EXCEPTION', #'NO_PERMISSION'{completion_status=?COMPLETED_NO}}},
	    ok
    end.

do_setup_connection(PMPid, Host, Port, SocketType, SocketOptions, Chars, Wchars) ->
    case init_interceptors(Host, Port) of
	{'EXCEPTION', E} ->
	    PMPid ! {setup_failed, {Host, Port}, {'EXCEPTION', E}},
	    ok;
	Interceptors ->
	    case catch orber_iiop_outsup:connect(Host, Port, SocketType, 
						 SocketOptions, PMPid) of
		{'error', {'EXCEPTION', E}} ->
		    orber:dbg("[~p] orber_iiop_pm:handle_call(connect ~p ~p);~n"
			      "Raised Exc: ~p", 
			      [?LINE, Host, Port, E], ?DEBUG_LEVEL),
		    PMPid ! {setup_failed, {Host, Port}, {'EXCEPTION', E}},
		    ok;
		{'error', Reason} ->
		    orber:dbg("[~p] orber_iiop_pm:handle_call(connect ~p ~p);~n"
			      "Got EXIT: ~p", 
			      [?LINE, Host, Port, Reason], ?DEBUG_LEVEL),
		    PMPid ! {setup_failed, {Host, Port}, 
			     {'EXCEPTION', #'INTERNAL'{completion_status=?COMPLETED_NO}}},
		    ok;
		{ok, undefined} ->
		    orber:dbg("[~p] orber_iiop_pm:handle_call(connect ~p ~p);~n"
			      "Probably no listener on the given Node/Port or timedout.",
			      [?LINE, Host, Port], ?DEBUG_LEVEL),
		    PMPid ! {setup_failed, {Host, Port}, 
			     {'EXCEPTION', #'COMM_FAILURE'{minor=(?ORBER_VMCID bor 1),
							   completion_status=?COMPLETED_NO}}},
		    ok;
		{ok, Child} ->
		    BiDirCtx = orber:bidir_context(),
		    Ctx = case orber:exclude_codeset_ctx() of
			      true ->
				  BiDirCtx;
			      _ ->
				  CodeSetCtx = 
				      #'CONV_FRAME_CodeSetContext'
				    {char_data =  Chars, 
				     wchar_data = Wchars},
				  [#'IOP_ServiceContext'
				   {context_id=?IOP_CodeSets, 
				    context_data = CodeSetCtx} | BiDirCtx]
			  end,
		    PMPid ! {setup_successfull, {Host, Port}, {Child, Ctx, Interceptors}},
		    ok
	    end
    end.

access_allowed(Host, Port, Type) ->
    Flags = orber:get_flags(),
    case ?ORB_FLAG_TEST(Flags, ?ORB_ENV_USE_ACL_OUTGOING) of
	false ->
	    ok;
	true ->
	    SearchFor = 
		case Type of
		    normal ->
			tcp_out;
		    ssl ->
			ssl_out
		end,
	    Family = orber_env:ip_version(),
	    {ok, Ip} = inet:getaddr(Host, Family),
	    case orber_acl:match(Ip, SearchFor, true) of
		{true, [], 0} ->
		    ok;
		{true, [], Port} ->
		    ok;
		{true, [], {Min, Max}} when Port >= Min, Port =< Max ->
		    ok;
		{true, [Interface], 0} ->
		    inet:getaddr(Interface, Family);
		{true, [Interface], Port} ->
		    inet:getaddr(Interface, Family);
		{true, [Interface], {Min, Max}} when Port >= Min, Port =< Max ->
		    inet:getaddr(Interface, Family);
		_ ->
		    false
	    end
    end.



invoke_connection_closed(false) ->
    ok;
invoke_connection_closed({native, Ref, PIs}) ->
    orber_pi:closed_out_connection(PIs, Ref);
invoke_connection_closed({_Type, _PIs}) ->
    ok.


init_interceptors(Host, Port) ->
    case orber:get_interceptors() of
	{native, PIs} ->
	    case catch orber_pi:new_out_connection(PIs, Host, Port) of
		{'EXIT', R} ->
		    orber:dbg("[~p] orber_iiop_pm:init_interceptors(~p); Got Exit: ~p.~n"
			      "One or more Interceptor incorrect or undefined?", 
			      [?LINE, PIs, R], ?DEBUG_LEVEL),
		    {'EXCEPTION', #'COMM_FAILURE'{minor=(?ORBER_VMCID bor 2), 
						  completion_status=?COMPLETED_NO}};
		IntRef ->
		    {native, IntRef, PIs}
	    end;
	Other ->
            %% Either 'false' or {Type, PIs}.
	    Other
    end.
    

%%-----------------------------------------------------------------
%% END OF MODULE
%%-----------------------------------------------------------------
