#if !defined(ESVN_UTILS_H)
#define ESVN_UTILS_H

#include <qlistview.h> 
#include <qdir.h>
#include <vector>

namespace EsvnUtils
{
	inline QDateTime UTC2Local(const QDateTime& utc)
    {
		QDateTime result(utc);

		// If daylight global variable is non-zero than timezone provides
		// daylight saving time period. It does NOT mean that saving is active NOW.
		// So the correct way is to call localtime(utc time) then correct
		// utc time by adding tm_isdst*3600

		// For older Windows versions there is a problem with localtime()
		// setting tm_isdst based on US timezone info, not system timezone

#if defined(__GLIBC__)

		time_t a; a = time(NULL);
		struct tm tm_local = *localtime(&a);
		result = result.addSecs(tm_local.tm_gmtoff);

#elif (__FreeBSD__ || __NetBSD__ || __OpenBSD__ || ESVN_BSD || __APPLE__)

		time_t a; a = time(NULL);
		struct tm bsd_hack = *localtime(&a);
		result = result.addSecs(-(bsd_hack.tm_gmtoff - bsd_hack.tm_isdst*3600) );

#elif (_WINDOWS || ESVN_WIN)

		result = result.addSecs(-(_timezone - _daylight*3600));

#else

		result = result.addSecs(-(timezone - daylight*3600));

#endif

		return result;
	}
}

namespace EsvnLVI
{
    enum Item {
        ITEM_NAME = 0,
        ITEM_FULL_PATH = 16,
        ITEM_WORKING_DIRECTORY = 20,
        ITEM_RELATIVE_PATH = 21,
    };

    QListViewItem* getFirstSelectedItem(QListView *view);
}

// Simple two-component path representation : base and name

class C_Path
{
public:
    C_Path();
    virtual ~C_Path();

public:
    virtual QString Name() const = 0;
    virtual QString Path() const = 0;
    virtual bool isDir() const = 0;

    virtual QString FullPath() const {
        if (Path().isEmpty()) {
            return "";
        } else if (Name().isEmpty()) {
            return Path();
        } else {
            return Path() + "/" + Name();
        }
    }

public:
    static QString Concat(const QString& prefix, const QString& suffix)
    {
        if (prefix.isEmpty()) {
            return suffix;
        } else if (suffix.isEmpty()) {
            return prefix;
        }

        return prefix + "/" + suffix;
    }
};

// Directory path implementation

class C_DirPath : public C_Path
{
public:
    C_DirPath();
    C_DirPath(const QDir& path) {
        operator=(path);
    }
    ~C_DirPath();

public:
    virtual QString Name() const;
    virtual QString Path() const;
    virtual bool isDir() const;

public:
    void SetName(const QString& name);
    void SetPath(const QString& name);
    C_DirPath& operator=(const QDir& path);

private:
    QString name_;
    QString pathname_;
};

// Standalone path object

class C_PathCopy : public C_Path
{
public:
    C_PathCopy();
    C_PathCopy(const C_Path& from);
    ~C_PathCopy();

public:
    C_PathCopy& operator=(const C_Path& from);

public:
    virtual QString Name() const;
    virtual QString Path() const;
    virtual bool isDir() const;

private:
    QString name_;
    QString pathname_;
    bool is_dir_;
};

// File path implementation

class C_RelativePath : public C_Path
{
public:
    C_RelativePath(const C_Path& parent, const QString& name);
    ~C_RelativePath();

public:
    virtual QString Name() const;
    virtual QString Path() const;
    virtual bool isDir() const;

public:
    void SetIsDir(bool is_dir);

private:
//    const C_Path& parent_;
    const C_PathCopy parent_;
    QString name_;
    bool is_dir_;
};

// Selection containers

typedef std::vector<const C_Path*> PathSelection;

class C_SinglePathSelection
{
public:
    C_SinglePathSelection(const C_Path& selected)
    {
        selection_.push_back(&selected);
    }
    operator const PathSelection& () const
    {
        return selection_;
    }

private:
    PathSelection selection_;
};

#endif // ESVN_UTILS_H

