/////////////////////////////////////////////////////////////
//                                                         //
// Copyright (c) 2003-2011 by The University of Queensland //
// Earth Systems Science Computational Centre (ESSCC)      //
// http://www.uq.edu.au/esscc                              //
//                                                         //
// Primary Business: Brisbane, Queensland, Australia       //
// Licensed under the Open Software License version 3.0    //
// http://www.opensource.org/licenses/osl-3.0.php          //
//                                                         //
/////////////////////////////////////////////////////////////

#include <mpi.h>
#include <boost/version.hpp>
#include <stdexcept>
#include <boost/python.hpp>
#include "Python/esys/lsm/RotParticlePy.h"

namespace esys
{
  namespace lsm
  {
    RotParticlePy::RotParticlePy() : CRotParticle()
    {
    }

    RotParticlePy::RotParticlePy(const RotParticlePy &p) : CRotParticle(p)
    {
    }

    RotParticlePy::RotParticlePy(const CRotParticle &p) : CRotParticle(p)
    {
    }

    // dynamic & rotational default to true
    RotParticlePy::RotParticlePy(int id, const Vec3Py &posn, double radius, double mass)
      : CRotParticle(radius, mass, posn, Vec3(), Vec3(), id,true,true)
    {
    }

    Vec3Py RotParticlePy::getInitialPosn() const
    {
      return Vec3Py(getInitPos());
    }

    Vec3Py RotParticlePy::getPosn() const
    {
      return Vec3Py(getPos());
    }

    void RotParticlePy::setPosn(const Vec3Py& posn)
    {
      setPos(posn);
    }

    QuaternionPy RotParticlePy::getOrientation() const
    {
      return QuaternionPy(getQuat());
    }

    void RotParticlePy::setOrientation(const QuaternionPy& quat)
    {
      setQuat(quat);
    }

    Vec3Py RotParticlePy::getVelocity() const
    {
      return Vec3Py(getVel());
    }

    Vec3Py RotParticlePy::getLinearVelocity() const
    {
      return Vec3Py(getVel());
    }

    void RotParticlePy::setLinearVelocity(const Vec3Py& vel)
    {
      return setVel(vel);
    }

    Vec3Py RotParticlePy::getAngularVelocity() const
    {
      return Vec3Py(getAngVel());
    }

    void RotParticlePy::setAngularVelocity(const Vec3Py& vel)
    {
      setAngVel(vel);
    }

    Vec3Py RotParticlePy::getAcceleration() const
    {
      return Vec3Py(getForce()*getInvMass());
    }

    Vec3Py RotParticlePy::getLinearAcceleration() const
    {
      return Vec3Py(getForce()*getInvMass());
    }

    void RotParticlePy::setLinearAcceleration(const Vec3Py& accel)
    {
      setForce(accel*getMass());
    }

    Vec3Py RotParticlePy::getAngularAcceleration() const
    {
      return getMoment()*getInvInertRot();
    }

    void RotParticlePy::setAngularAcceleration(const Vec3Py& accel)
    {
      setMoment(accel*getInertRot());
    }

    Vec3Py RotParticlePy::getLinearForce() const
    {
      return Vec3Py(m_force);
    }

    void RotParticlePy::setLinearForce(const Vec3Py& force)
    {
      setForce(force);
    }

    Vec3Py RotParticlePy::getAngularForce() const
    {
      return Vec3Py(m_moment);
    }

    void RotParticlePy::setAngularForce(const Vec3Py& moment)
    {
      setMoment(moment);
    }

    using boost::python::arg;
    void exportRotParticle()
    {
      // Check that Boost 1.34.0 or higher is being used.
      // If so, disable auto-generation of C++ signatures for Epydoc
      // (which stumbles over indentation in the auto-generated strings).
      #if ((BOOST_VERSION / 100000 >= 1) \
          && (BOOST_VERSION / 100 % 1000 >= 34)) \
          || (BOOST_VERSION / 100000 >= 2)
        boost::python::docstring_options no_autogen(true,false);
      #endif

      boost::python::class_<RotParticlePy>("RotSphere")
        .def(boost::python::init<>())
        .def(boost::python::init<const RotParticlePy &>())
        .def(boost::python::init<int,const Vec3Py &, double, double>(
          (
            arg("id"),
            arg("posn"),
            arg("radius"),
            arg("mass")
          ),
          "Construct a rotational spherical particle.\n"
          "@type id: int\n"
          "@kwarg id: Unique identifier for particle.\n"
          "@type posn: L{Vec3<esys.lsm.util.FoundationPy.Vec3>}\n"
          "@kwarg posn: Initial position of particle, centre-point of sphere.\n"
          "@type radius: float\n"
          "@kwarg radius: The radius of the sphere.\n"
          "@type mass: float\n"
          "@kwarg mass: Mass of particle.\n"
        ))
        .def("getId",                   &RotParticlePy::getID)
        .def("getTag",                  &RotParticlePy::getTag)
        .def("setTag",                  &RotParticlePy::setTag)
        .def("getInitialPosn",          &RotParticlePy::getInitialPosn)
        .def("getInitialPosition",      &RotParticlePy::getInitialPosn)
        .def("getPosn",                 &RotParticlePy::getPosn)
        .def("getPosition",             &RotParticlePy::getPosn)
        .def("setPosn",                 &RotParticlePy::setPosn)
        .def("setPosition",             &RotParticlePy::setPosn)
        .def("getOrientation",          &RotParticlePy::getOrientation)
        .def("setOrientation",          &RotParticlePy::setOrientation)
        .def("getVelocity",             &RotParticlePy::getVelocity)
        .def("getLinearVelocity",       &RotParticlePy::getLinearVelocity)
        .def("setLinearVelocity",       &RotParticlePy::setLinearVelocity)
        .def("getAngularVelocity",      &RotParticlePy::getAngularVelocity)
        .def("setAngularVelocity",      &RotParticlePy::setAngularVelocity)
        .def("getAcceleration",         &RotParticlePy::getAcceleration)
        .def("getLinearAcceleration",   &RotParticlePy::getLinearAcceleration)
        .def("setLinearAcceleration",   &RotParticlePy::setLinearAcceleration)
        .def("getAngularAcceleration",  &RotParticlePy::getAngularAcceleration)
        .def("setAngularAcceleration",  &RotParticlePy::setAngularAcceleration)
        .def("getRad",                  &RotParticlePy::getRad)
        .def("getRadius",               &RotParticlePy::getRad)
        .def("getCentre",               &RotParticlePy::getPosn)
        .def("getCenter",               &RotParticlePy::getPosn)
        .def("getMass",                 &RotParticlePy::getMass)
      ;
    }
  }
}
