/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * evms_query
 *
 * A utility for displaying information about EVMS and its volumes, objects,
 * containers, and filesystems.
 */

#include <stdio.h>
#include <string.h>
#include <getopt.h>
#include <libgen.h>
#include <frontend.h>
#include <locale.h>

#define REQUIRED_ENGINE_API_MAJOR 10
#define REQUIRED_ENGINE_API_MINOR 0
#define REQUIRED_ENGINE_API_PATCH 0

static char *prog_name;				
static char *log_file = NULL;
static char *node_name = NULL;
static debug_level_t debug_level = -1;
static boolean verbose = FALSE;

static int info(object_type_t type, int argc, char **argv);
static int details(object_type_t type, int argc, char **argv);
static int display(object_type_t type, int argc, char **argv);

/**
 * query_commands
 *
 * Structure defining the available query commands and which function
 * each command maps to.
 **/
static struct query_commands_t {
	char *name;
	object_type_t type;
	int (*fn)(object_type_t type, int argc, char **argv);
} query_commands[] = {
	{ "info",		0,		info },
	{ "details",		0,		details },
	{ "plugins",		PLUGIN,		display },
	{ "disks",		DISK,		display },
	{ "segments",		SEGMENT,	display },
	{ "regions",		REGION,		display },
	{ "feature_objects",	EVMS_OBJECT,	display },
	{ "containers",		CONTAINER,	display },
	{ "volumes",		VOLUME,		display },
	{ "objects",		DISK | SEGMENT |
				REGION | EVMS_OBJECT,	display },
};

#define NR_COMMANDS (sizeof(query_commands) / sizeof(query_commands[0]))

#define LOG(msg, args...)		\
	do { fprintf(stdout, msg, ## args); } while (0)
#define LOG_VERBOSE(msg, args...)	\
	do { if (verbose) { printf(msg, ## args); } } while (0)
#define LOG_ERROR(msg, args...)		\
	do { fprintf(stderr, msg, ## args); } while (0)

/**
 * EVMS User-Interface Callbacks
 **/

static int message_callback(char *message_text, int *answer, char **choices)
{
	LOG_VERBOSE("%s", message_text);
	LOG_VERBOSE(_("%s: Responding with default selection \"%s\".\n"),
		    prog_name, choices[*answer]);
	return 0;
}

static void status_callback(char *message_text)
{
	LOG_VERBOSE("%s", message_text);
}

static ui_callbacks_t callbacks = {
	user_message: message_callback,
	user_communication: NULL,
	progress: NULL,
	status: status_callback
};

/**
 * Option parsing code.
 **/

static boolean is_numeric(char * str)
{
	while (*str != '\0') {
		if ((*str < '0') || (*str > '9')) {
			return FALSE;
		}
		str++;
	}
	return TRUE;
}

static void show_help(void)
{
	int i;
	LOG(_("Usage: %s [options] <command> [command_options] [name]*\n"), prog_name);
	LOG(_("Display information about EVMS.\n"
	      "   options:\n"
	      "      [-d debug_level, --debug-level debug_level]\n"
	      "         debug_level = [0-9] |\n"
	      "                       [critical | serious | error | warning | default |\n"
	      "                        details | entry_exit | debug | extra | everything]\n"
	      "      [-l log_file_name, --log-file log_file_name]\n"
	      "      [-n node_name, --node-name node_name]\n"
	      "      [-v, --verbose] verbose mode\n"
	      "      [-h | -? | --help] display this help\n"
	      "   commands:\n"));
	for (i = 0; i < NR_COMMANDS; i++) {
		LOG("      %s\n", query_commands[i].name);
	}
	LOG(_("   command options:\n"
	      "      [-i, --info]\n"
	      "      [-a, --above]\n"
	      "      [-b, --below]\n"));
}

static void parse_display_options(int argc, char **argv,
				  int *name_only, int *above)
{
	int c;
	char *short_opts = "abi";
	struct option long_opts[] = { { "above", no_argument, NULL, 'a' },
				      { "below", no_argument, NULL, 'b' },
				      { "info",  no_argument, NULL, 'i' },
				      { NULL,    0,           NULL,  0  } };

	while ((c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF) {
		switch (c) {
		case 'a':
			*above = TRUE;
			break;
		case 'b':
			*above = FALSE;
			break;
		case 'i':
			*name_only = FALSE;
			break;
		default:
			break;
		}
	}
}

static int parse_options(int argc, char ** argv)
{
	int c, rc = 0;
	char *short_opts = "+d:l:n:vh?";
	struct option long_opts[] = { { "debug-level",  required_argument, NULL, 'd'},
				      { "log-file",     required_argument, NULL, 'l'},
				      { "node-name",    required_argument, NULL, 'n'},
				      { "help",         no_argument,       NULL, 'h'},
				      { "verbose",      no_argument,       NULL, 'v'},
				      { NULL,           0,                 NULL,   0}  };

	while ((c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF) {
		switch (c) {
		case 'd':
			if (is_numeric(optarg)) {
				int level = atoi(optarg);

				if ((level < CRITICAL) ||
				    (level > EVERYTHING)) {
					LOG(_("%s is not a valid debug level.\n"),
					    optarg);
					/* Display the help. */
					rc = 1;
				}
				debug_level = level;
			} else if (strcasecmp(optarg, "critical") == 0) {
				debug_level = CRITICAL;
			} else if (strcasecmp(optarg, "serious") == 0) {
				debug_level = SERIOUS;
			} else if (strcasecmp(optarg, "error") == 0) {
				debug_level = ERROR;
			} else if (strcasecmp(optarg, "warning") == 0) {
				debug_level = WARNING;
			} else if (strcasecmp(optarg, "default") == 0) {
				debug_level = DEFAULT;
			} else if (strcasecmp(optarg, "details") == 0) {
				debug_level = DETAILS;
			} else if (strcasecmp(optarg, "entry_exit") == 0) {
				debug_level = ENTRY_EXIT;
			} else if (strcasecmp(optarg, "debug") == 0) {
				debug_level = DEBUG;
			} else if (strcasecmp(optarg, "extra") == 0) {
				debug_level = EXTRA;
			} else if (strcasecmp(optarg, "everything") == 0) {
				debug_level = EVERYTHING;
			} else {
				LOG(_("%s is not a valid debug level.\n"),
				    optarg);
				/* Display the help. */
				rc = EINVAL;
			}

			break;

		case 'l':
			log_file = strdup(optarg);
			break;

		case 'n':
			node_name = strdup(optarg);
			break;

		case 'v':
			verbose = TRUE;
			break;

		case 'h':
		case '?':
			/* Display the help. */
			rc = EINVAL;
			break;

		default:
			LOG(_("%s -- unrecognized option \"%c\"\n\n"),
			    prog_name, c);
			/* Display the help. */
			rc = EINVAL;
			break;
		}
	}

	return rc;
}

/**
 * get_command
 *
 * Search the query_commands array for the specified name and return the
 * index. If not found, return -1.
 **/
static int get_command(char *command_name)
{
	int i, rc;

	for (i = 0; i < NR_COMMANDS; i++) {
		rc = strncmp(command_name, query_commands[i].name,
			     strlen(query_commands[i].name));
		if (!rc) {
			LOG_VERBOSE(_("Found command: %s (%d)\n"), command_name, i);
			return i;
		}
	}

	return -1;
}

/**
 * get_handle
 *
 * Get the engine handle for the given name. The name could be for a storage-
 * object, container, volume, or plugin.
 **/
static int get_handle(char *name, engine_handle_t *handle)
{
	int rc;

	rc = evms_get_object_handle_for_name(DISK | SEGMENT | REGION | EVMS_OBJECT |
					     CONTAINER | VOLUME, name, handle);
	if (rc) {
		rc = evms_get_plugin_by_name(name, handle);
		if (rc) {
			LOG_ERROR(_("Error getting handle for \"%s\"\n"), name);
		}
	}

	return rc;
}

/**
 * Managing lists of engine handles.
 **/

typedef struct handle_list {
	struct handle_list *next;
	engine_handle_t handle;
} handle_list_t;

static int exclusive_insert_handle(engine_handle_t handle, handle_list_t **list)
{
	handle_list_t *item;

	for (item = *list; item; item = item->next) {
		if (item->handle == handle) {
			return EEXIST;
		}
	}

	item = malloc(sizeof(*item));
	if (!item) {
		return ENOMEM;
	}

	item->handle = handle;
	item->next = *list;
	*list = item;

	return 0;
}

static void delete_handle_list(handle_list_t *list)
{
	handle_list_t *item, *next;

	for (item = list; item; item = next) {
		next = item->next;
		free(item);
	}
}

/**
 * Converting constants to strings for common EVMS things.
 **/

static char *plugin_type_string(evms_plugin_code_t type)
{
	static char *name = NULL;

	switch (type) {
	case EVMS_DEVICE_MANAGER:
		name = "Device Manager";
		break;
	case EVMS_SEGMENT_MANAGER:
		name = "Segment Manager";
		break;
	case EVMS_REGION_MANAGER:
		name = "Region Manager";
		break;
	case EVMS_FEATURE:
		name = "EVMS Feature";
		break;
	case EVMS_ASSOCIATIVE_FEATURE:
		name = "EVMS Associative Feature";
		break;
	case EVMS_FILESYSTEM_INTERFACE_MODULE:
		name = "Filesystem Interface Module";
		break;
	case EVMS_CLUSTER_MANAGER_INTERFACE_MODULE:
		name = "Cluster Manager";
		break;
	case EVMS_DISTRIBUTED_LOCK_MANAGER_INTERFACE_MODULE:
		name = "Distributed Lock Manager";
		break;
	default:
		name = NULL;
	}

	return name;
}
static char *object_type_string(object_type_t type)
{
	static char *name = NULL;

	switch (type) {
	case PLUGIN:
		name = "Plugin";
		break;
	case DISK:
		name = "Disk";
		break;
	case SEGMENT:
		name = "Segment";
		break;
	case REGION:
		name = "Region";
		break;
	case EVMS_OBJECT:
		name = "EVMS Object";
		break;
	case CONTAINER:
		name = "Container";
		break;
	case VOLUME:
		name = "Volume";
		break;
	default:
		name = NULL;
	}

	return name;
}

static char *data_type_string(data_type_t type)
{
	static char *name = NULL;

	switch (type) {
	case META_DATA_TYPE:
		name = "Metadata";
		break;
	case DATA_TYPE:
		name = "Data";
		break;
	case FREE_SPACE_TYPE:
		name = "Freespace";
		break;
	default:
		name = NULL;
	}

	return name;
}

static char *value_unit_string(value_unit_t type)
{
	static char *name = "";

	switch (type) {
	case EVMS_Unit_Disks:
		name = "disks";
		break;
	case EVMS_Unit_Sectors:
		name = "sectors";
		break;
	case EVMS_Unit_Segments:
		name = "segments";
		break;
	case EVMS_Unit_Regions:
		name = "regions";
		break;
	case EVMS_Unit_Percent:
		name = "%";
		break;
	case EVMS_Unit_Milliseconds:
		name = "milliseconds";
		break;
	case EVMS_Unit_Microseconds:
		name = "microseconds";
		break;
	case EVMS_Unit_Bytes:
		name = "bytes";
		break;
	case EVMS_Unit_Kilobytes:
		name = "KB";
		break;
	case EVMS_Unit_Megabytes:
		name = "MB";
		break;
	case EVMS_Unit_Gigabytes:
		name = "GB";
		break;
	case EVMS_Unit_Terabytes:
		name = "TB";
		break;
	case EVMS_Unit_Petabytes:
		name = "PB";
		break;
	case EVMS_Unit_None:
	default:
		name = "";
	}

	return name;
}

/**
 * display_info_plugin
 *
 * Display the basic info for a plugin.
 **/
static void display_info_plugin(plugin_info_t *info, int name_only)
{
	if (name_only) {
		LOG("%s\n", info->short_name);
	} else {
		LOG(_("Name: %s\n"), info->short_name);
		LOG(_("Type: %s\n"), object_type_string(PLUGIN));
		LOG(_("Long Name: %s\n"), info->long_name);
		LOG(_("OEM Name: %s\n"), info->oem_name);
		LOG(_("Plugin ID: %d.%d.%d\n"), GetPluginOEM(info->id),
		    GetPluginType(info->id), GetPluginID(info->id));
		LOG(_("Plugin Type: %s\n"),
		    plugin_type_string(GetPluginType(info->id)));
		LOG(_("Plugin Version: %d.%d.%d\n"), info->version.major,
		    info->version.minor, info->version.patchlevel);
		LOG(_("Required Engine Services API Version: %d.%d.%d\n"),
		    info->required_engine_api_version.major,
		    info->required_engine_api_version.minor,
		    info->required_engine_api_version.patchlevel);
	}
}

/**
 * display_info_object
 *
 * Display the basic info for a storage object.
 **/
static void display_info_object(storage_object_info_t *info, int name_only)
{
	handle_object_info_t *plugin_info;
	int rc;
	
	if (name_only) {
		LOG("%s\n", info->name);
	} else {
		char number_buffer[64];

		LOG(_("Name: %s\n"), info->name);
		LOG(_("Type: %s\n"), object_type_string(info->object_type));
		LOG(_("Data Type: %s\n"), data_type_string(info->data_type));
		LOG(_("Device Number: %d,%d\n"),
		    info->dev_major, info->dev_minor);
		sprintf(number_buffer, "%"PRIu64, info->size);
		LOG(_("Size (sectors): %s\n"), number_buffer);
		rc = evms_get_info(info->plugin, &plugin_info);
		if (rc) {
			LOG(_("Plugin: Unknown\n"));
		} else {
			LOG(_("Plugin: %s\n"),
			    plugin_info->info.plugin.short_name);
			evms_free(plugin_info);
		}
	}
}

/**
 * display_info_container
 *
 * Display the basic info for a storage container.
 **/
static void display_info_container(storage_container_info_t *info, int name_only)
{
	handle_object_info_t *plugin_info;
	int rc;

	if (name_only) {
		LOG("%s\n", info->name);
	} else {
		char number_buffer[64];

		LOG(_("Name: %s\n"), info->name);
		LOG(_("Type: %s\n"), object_type_string(CONTAINER));
		sprintf(number_buffer, "%"PRIu64, info->size);
		LOG(_("Size (sectors): %s\n"), number_buffer);
		rc = evms_get_info(info->plugin, &plugin_info);
		if (rc) {
			LOG(_("Plugin: Unknown\n"));
		} else {
			LOG(_("Plugin: %s\n"),
			    plugin_info->info.plugin.short_name);
			evms_free(plugin_info);
		}
	}
}

/**
 * display_info_volume
 *
 * Display the basic info for a logical volume.
 **/
static void display_info_volume(logical_volume_info_t *info, int name_only)
{
	handle_object_info_t *fsim_info;
	int rc;

	if (name_only) {
		LOG("%s\n", info->name);
	} else {
		char number_buffer[64];

		LOG(_("Name: %s\n"), info->name);
		LOG(_("Type: %s\n"), object_type_string(VOLUME));
		LOG(_("Device Number: %d:%d\n"),
		    info->dev_major, info->dev_minor);
		sprintf(number_buffer, "%"PRIu64, info->vol_size);
		LOG(_("Size (sectors): %s\n"), number_buffer);
		rc = evms_get_info(info->file_system_manager, &fsim_info);
		if (rc) {
			LOG(_("Filesystem: Unknown\n"));
		} else {
			LOG(_("Filesystem: %s\n"),
			    fsim_info->info.plugin.short_name);
			evms_free(fsim_info);
		}
	}
}

/**
 * display_info_evms
 *
 * Display the basic info about EVMS.
 **/
static void display_info_evms(void)
{
	evms_version_t engine_api;

	LOG(_("EVMS Version: %d.%d.%d\n"),
	    MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
	evms_get_api_version(&engine_api);
	LOG(_("EVMS Engine API Version: %d.%d.%d\n"),
	    engine_api.major, engine_api.minor, engine_api.patchlevel);
}

/**
 * display_info
 *
 * Determine the type of info structure, and call the appropriate routine
 * to display the info.
 **/
static int display_info(handle_object_info_t *info, int name_only)
{
	int rc = 0;

	switch (info->type) {
	case PLUGIN:
		display_info_plugin(&(info->info.plugin), name_only);
		break;
	case DISK:
	case SEGMENT:
	case REGION:
	case EVMS_OBJECT:
		display_info_object(&(info->info.object), name_only);
		break;
	case CONTAINER:
		display_info_container(&(info->info.container), name_only);
		break;
	case VOLUME:
		display_info_volume(&(info->info.volume), name_only);
		break;
	default:
		rc = EINVAL;
		break;
	}

	return rc;
}

static void display_value(value_t value, value_type_t type)
{
	switch (type) {
	case EVMS_Type_String:
		if (value.s) {
			LOG("%s", value.s);
		}
		break;
	case EVMS_Type_Boolean:
		LOG("%s", value.b ? "True" : "False");
		break;
	case EVMS_Type_Char:
		LOG("%c", value.c);
		break;
	case EVMS_Type_Unsigned_Char:
		LOG("%uc", value.uc);
		break;
	case EVMS_Type_Real32:
		LOG("%g", value.r32);
		break;
	case EVMS_Type_Real64:
		LOG("%lg", value.r64);
		break;
	case EVMS_Type_Int:
		LOG("%d", value.i);
		break;
	case EVMS_Type_Int8:
		LOG("%"PRIi8, value.i8);
		break;
	case EVMS_Type_Int16:
		LOG("%"PRIi16, value.i16);
		break;
	case EVMS_Type_Int32:
		LOG("%"PRIi32, value.i32);
		break;
	case EVMS_Type_Int64:
		LOG("%"PRIi64, value.i64);
		break;
	case EVMS_Type_Unsigned_Int:
		LOG("%ud", value.ui);
		break;
	case EVMS_Type_Unsigned_Int8:
		LOG("%"PRIu8, value.ui8);
		break;
	case EVMS_Type_Unsigned_Int16:
		LOG("%"PRIu16, value.ui16);
		break;
	case EVMS_Type_Unsigned_Int32:
		LOG("%"PRIu32, value.ui32);
		break;
	case EVMS_Type_Unsigned_Int64:
		LOG("%"PRIu64, value.ui64);
		break;
	default:
		break;
	}
}

static void display_extended_info_value(extended_info_t *info)
{
	int i;

	switch (info->collection_type) {
	case EVMS_Collection_None:
		LOG(_("Value: "));
		display_value(info->value, info->type);
		LOG(" %s\n", value_unit_string(info->unit));
		break;

	case EVMS_Collection_List:
		LOG(_("Value List:\n"));
		for (i = 0; i < info->collection.list->count; i++) {
			LOG("\t");
			display_value(info->collection.list->value[i], info->type);
			LOG(" %s\n", value_unit_string(info->unit));
		}
		break;

	case EVMS_Collection_Range:
		LOG(_("Value Range:\n"));
		LOG(_("\tMinimum: "));
		display_value(info->collection.range->min, info->type);
		LOG(" %s\n", value_unit_string(info->unit));

		LOG(_("\tMaximum: "));
		display_value(info->collection.range->max, info->type);
		LOG(" %s\n", value_unit_string(info->unit));

		LOG(_("\tIncrement: "));
		display_value(info->collection.range->increment, info->type);
		LOG(" %s\n", value_unit_string(info->unit));
		break;
	default:
		break;
	}
}

/**
 * display_extended_info_item
 *
 * Display a single item from an extended_info_array.
 **/
static void display_extended_info_item(extended_info_t *info)
{
	if (info->name) {
		LOG(_("Name: %s"), info->name);
		if (info->flags & EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE) {
			LOG(_(" (extra information available)"));
		}
		LOG("\n");
	}
	if (info->title)
		LOG(_("Title: %s\n"), info->title);
	if (info->desc)
		LOG(_("Description: %s\n"), info->desc);

	display_extended_info_value(info);
}

static int display_extended_info(extended_info_array_t *info)
{
	int i;

	for (i = 0; i < info->count; i++) {
		display_extended_info_item(info->info + i);

		if (i < info->count - 1) {
			LOG("\n");
		}
	}

	return 0;
}

/**
 * get_handle_array
 *
 * Get the array of handles for all "things" of the given type.
 **/
static int get_handle_array(object_type_t type, handle_array_t **handles)
{
	int rc;

	if (type & PLUGIN) {
		rc = evms_get_plugin_list(0, 0, handles);
	} else if (type & DISK ||
		   type & SEGMENT ||
		   type & REGION ||
		   type & EVMS_OBJECT) {
		rc = evms_get_object_list(type, 0, 0, 0, 0, handles);
	} else if (type & CONTAINER) {
		rc = evms_get_container_list(0, 0, 0, handles);
	} else if (type & VOLUME) {
		rc = evms_get_volume_list(0, 0, 0, handles);
	} else {
		rc = EINVAL;
	}

	return rc;
}

/**
 * display_handle
 *
 * Get and display the info for the given handle.
 **/
static int display_handle(engine_handle_t handle, int name_only)
{
	handle_object_info_t *info = NULL;
	int rc;

	rc = evms_get_info(handle, &info);
	if (rc) {
		LOG_ERROR(_("Error getting info for handle 0x%x\n"), handle);
		goto out;
	}

	rc = display_info(info, name_only);
	if (rc) {
		LOG_ERROR(_("Error displaying info for handle 0x%x\n"), handle);
		goto out;
	}

out:
	evms_free(info);
	return rc;
}

/**
 * display_handle_array
 *
 * Get and display the info for each handle in the handle-array.
 **/
static int display_handle_array(handle_array_t *handles, int name_only)
{
	int i, rc = 0;

	for (i = 0; i < handles->count; i++) {
		rc = display_handle(handles->handle[i], name_only);
		if (rc) {
			break;
		}

		/* If displaying info with each handle, insert blank lines. */
		if (!name_only && i < handles->count - 1) {
			LOG("\n");
		}
	}

	return rc;
}

/**
 * display_list
 *
 * Get the handle-array for the specified type of "thing", and display
 * info for all handles in the array.
 **/
static int display_list(object_type_t type, int name_only)
{
	handle_array_t *handles = NULL;
	int rc;

	rc = get_handle_array(type, &handles);
	if (rc) {
		goto out;
	}

	rc = display_handle_array(handles, name_only);
	if (rc) {
		goto out;
	}

out:
	evms_free(handles);
	return rc;
}

/**
 * display_plugin
 *
 * Display the name/info for the plugin that owns this handle.
 **/
static int display_plugin(engine_handle_t handle, int name_only)
{
	handle_object_info_t *info = NULL;
	int rc = 0;

	if (!handle) {
		return 0;
	}

	rc = evms_get_info(handle, &info);
	if (rc) {
		LOG_ERROR(_("Error getting info for handle 0x%x\n"), handle);
		goto out;
	}

	switch (info->type) {
	case DISK:
	case SEGMENT:
	case REGION:
	case EVMS_OBJECT:
		rc = display_handle(info->info.object.plugin, name_only);
		break;
	case CONTAINER:
		rc = display_handle(info->info.container.plugin, name_only);
		break;
	case VOLUME:
		rc = display_handle(info->info.volume.file_system_manager,
				    name_only);
		break;
	default:
		break;
	}

out:
	evms_free(info);
	return rc;
}

/**
 * display_children
 *
 * Recursively walk down the volume stack, displaying names/info for the
 * desired type of objects.
 **/
static int display_children(engine_handle_t handle,
			    object_type_t display_type,
			    int name_only,
			    handle_list_t **display_list)
{
	handle_object_info_t *info = NULL;
	int i, rc;

	if (!handle) {
		return 0;
	}

	rc = evms_get_info(handle, &info);
	if (rc) {
		LOG_ERROR(_("Error getting info for handle 0x%x\n"), handle);
		goto out;
	}

	if (info->type & display_type) {
		rc = exclusive_insert_handle(handle, display_list);
		if (!rc) {
			display_info(info, name_only);
			if (!name_only) {
				LOG("\n");
			}
		}
		rc = 0;
	}

	if (info->type & VOLUME) {
		display_children(info->info.volume.object,
				 display_type, name_only, display_list);
	} else if (info->type & CONTAINER) {
		if (info->info.container.objects_consumed) {
			for (i = 0; i < info->info.container.objects_consumed->count; i++) {
				display_children(info->info.container.objects_consumed->handle[i],
						 display_type, name_only, display_list);
			}
		}
	} else if (info->type & DISK    ||
		   info->type & SEGMENT ||
		   info->type & REGION  ||
		   info->type & EVMS_OBJECT) {
		if (info->info.object.producing_container) {
			display_children(info->info.object.producing_container,
					 display_type, name_only, display_list);
		} else if (info->info.object.child_objects) {
			for (i = 0; i < info->info.object.child_objects->count; i++) {
				display_children(info->info.object.child_objects->handle[i],
						 display_type, name_only, display_list);
			}
		}
	}

out:
	evms_free(info);
	return rc;
}

/**
 * display_parents
 *
 * Recursively walk up the volume stack, displaying names/info for the
 * desired type of objects.
 **/
static int display_parents(engine_handle_t handle,
			   object_type_t display_type,
			   int name_only,
			   handle_list_t **display_list)
{
	handle_object_info_t *info = NULL;
	int i, rc;

	if (!handle) {
		return 0;
	}

	rc = evms_get_info(handle, &info);
	if (rc) {
		LOG_ERROR(_("Error getting info for handle 0x%x\n"), handle);
		goto out;
	}

	if (info->type & display_type) {
		rc = exclusive_insert_handle(handle, display_list);
		if (!rc) {
			display_info(info, name_only);
			if (!name_only) {
				LOG("\n");
			}
		}
		rc = 0;
	}

	if (info->type & CONTAINER) {
		if (info->info.container.objects_produced) {
			for (i = 0; i < info->info.container.objects_produced->count; i++) {
				display_parents(info->info.container.objects_produced->handle[i],
						display_type, name_only, display_list);
			}
		}
	} else if (info->type & DISK    ||
		   info->type & SEGMENT ||
		   info->type & REGION  ||
		   info->type & EVMS_OBJECT) {
		if (info->info.object.volume) {
			display_parents(info->info.object.volume,
					display_type, name_only, display_list);
		} else if (info->info.object.parent_objects) {
			for (i = 0; i < info->info.object.parent_objects->count; i++) {
				display_parents(info->info.object.parent_objects->handle[i],
						display_type, name_only, display_list);
			}
		}
	}

out:
	evms_free(info);
	return rc;
}

/**
 * walk_tree
 *
 * Starting at the specified "thing", walk up or down the
 * stack, displaying names/info for the desired type of list.
 **/
static int walk_tree(char *source_name, object_type_t type,
		     int name_only, int above)
{
	engine_handle_t source_handle;
	object_type_t source_type;
	handle_list_t *display_list = NULL;
	int rc;

	rc = get_handle(source_name, &source_handle);
	if (rc) {
		goto out;
	}

	rc = evms_get_handle_object_type(source_handle, &source_type);
	if (rc) {
		LOG_ERROR(_("Error getting type for \"%s\"\n"), source_name);
		goto out;
	}

	rc = exclusive_insert_handle(source_handle, &display_list);
	if (rc) {
		goto out;
	}

	switch (source_type) {
	case PLUGIN:
		/* FIXME: List the things owned by this plugin. */
		break;

	case DISK:
		if (type & PLUGIN) {
			rc = display_plugin(source_handle, name_only);
		} else {
			rc = display_parents(source_handle, type,
					     name_only, &display_list);
		}
		break;

	case SEGMENT:
		if (type & PLUGIN) {
			rc = display_plugin(source_handle, name_only);
		} else if ((type & DISK) ||
			   ((type & (SEGMENT | CONTAINER)) && !above)) {
			rc = display_children(source_handle, type,
					      name_only, &display_list);
		} else if ((type & (REGION | EVMS_OBJECT | VOLUME)) ||
			   ((type & (SEGMENT | CONTAINER)) && above)) {
			rc = display_parents(source_handle, type,
					     name_only, &display_list);
		}
		break;

	case REGION:
		if (type & PLUGIN) {
			rc = display_plugin(source_handle, name_only);
		} else if ((type & (DISK | SEGMENT)) ||
			   ((type & (REGION | CONTAINER)) && !above)) {
			rc = display_children(source_handle, type,
					      name_only, &display_list);
		} else if ((type & (EVMS_OBJECT | VOLUME)) ||
			   ((type & (REGION | CONTAINER)) && above)) {
			rc = display_parents(source_handle, type,
					     name_only, &display_list);
		}
		break;

	case EVMS_OBJECT:
		if (type & PLUGIN) {
			rc = display_plugin(source_handle, name_only);
		} else if ((type & (DISK | SEGMENT | REGION | CONTAINER)) ||
			   ((type & EVMS_OBJECT) && !above)) {
			rc = display_children(source_handle, type,
					      name_only, &display_list);
		} else if ((type & VOLUME) ||
			   ((type & EVMS_OBJECT) && above)) {
			rc = display_parents(source_handle, type,
					     name_only, &display_list);
		}
		break;

	case CONTAINER:
		/* FIXME: Can we do anything better than this? */
		if (type & PLUGIN) {
			rc = display_plugin(source_handle, name_only);
		} else if (above) {
			rc = display_parents(source_handle, type,
					     name_only, &display_list);
		} else {
			rc = display_children(source_handle, type,
					      name_only, &display_list);
		}
		break;

	case VOLUME:
		if (type & PLUGIN) {
			rc = display_plugin(source_handle, name_only);
		} else {
			rc = display_children(source_handle, type,
					      name_only, &display_list);
		}
		break;

	default:
		break;
	}

	delete_handle_list(display_list);

out:
	return rc;
}

/**
 * display
 *
 * Create and display a list of items.
 **/
static int display(object_type_t type, int argc, char **argv)
{
	int name_only = TRUE;
	int above = FALSE;
	int rc;

	parse_display_options(argc, argv, &name_only, &above);

	if (optind < argc) {
		rc = walk_tree(argv[optind], type, name_only, above);
	} else {
		rc = display_list(type, name_only);
	}

	return rc;
}

/**
 * info
 *
 * Display info about an EVMS "thing".
 **/
static int info(object_type_t type, int argc, char **argv)
{
	object_handle_t handle;
	char *name;
	int rc = 0;

	if (argc > 1) {
		/* Get the handle and basic info for the specified name. */
		name = argv[1];
		rc = get_handle(name, &handle);
		if (rc) {
			goto out;
		}

		rc = display_handle(handle, FALSE);
		if (rc) {
			goto out;
		}
	} else {
		/* Display general EVMS information. */
		display_info_evms();
	}

out:
	return rc;
}

/**
 * details
 *
 * Display detailed info about an EVMS "thing".
 **/
static int details(object_type_t type, int argc, char **argv)
{
	object_handle_t handle;
	extended_info_array_t *info;
	char *name, *descriptor = NULL;
	int rc = 0;

	if (argc > 1) {
		/* Get the handle and extended info for the specified name. */
		name = argv[1];
		if (argc > 2) {
			descriptor = argv[2];
		}

		rc = get_handle(name, &handle);
		if (rc) {
			goto out;
		}

		rc = evms_get_extended_info(handle, descriptor, &info);
		if (rc) {
			LOG_ERROR(_("Error getting details for \"%s\"\n"), name);
			goto out;
		}

		rc = display_extended_info(info);
		if (rc) {
			LOG_ERROR(_("Error displaying details for \"%s\"\n"), name);
		}

		evms_free(info);
	} else {
		/* Display general EVMS information. */
		display_info_evms();
	}

out:
	return rc;
}

int main(int argc, char **argv)
{
	evms_version_t engine_api;
	int command;
	int rc = 0;

	setlocale(LC_MESSAGES, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	prog_name = basename(argv[0]);

	rc = parse_options(argc, argv);
	if (rc) {
		show_help();
		return rc;
	}

	if (optind >= argc) {
		show_help();
		return 0;
	}

	evms_get_api_version(&engine_api);

	if ((engine_api.major != REQUIRED_ENGINE_API_MAJOR) ||
	    (engine_api.major == REQUIRED_ENGINE_API_MAJOR &&
	     engine_api.minor <  REQUIRED_ENGINE_API_MINOR)) {
		LOG_ERROR(_("EVMS API version mismatch.\n"));
		LOG_ERROR(_("\tEVMS provides %d.%d.%d\n"),
			  engine_api.major, engine_api.minor,
			  engine_api.patchlevel);
		LOG_ERROR(_("\t%s requires %d.%d.%d\n"),
			  prog_name, REQUIRED_ENGINE_API_MAJOR,
			  REQUIRED_ENGINE_API_MINOR, REQUIRED_ENGINE_API_PATCH);
		return EPERM;
	}

	command = get_command(argv[optind]);
	if (command < 0) {
		show_help();
		return EINVAL;
	}

	rc = evms_open_engine(node_name, ENGINE_READWRITE,
			      &callbacks, debug_level, log_file);
	if (rc) {
		LOG_ERROR(_("evms_open_engine() failed with error code %d: %s\n"),
			  rc, evms_strerror(rc));
		return rc;
	}

	rc = query_commands[command].fn(query_commands[command].type,
					argc - optind, argv + optind);
	if (rc) {
		LOG_ERROR(_("\"%s\" command failed with error code %d: %s\n"),
			  query_commands[command].name, rc, evms_strerror(rc));
	}

	evms_close_engine();

	return rc;
}

