/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/***********************************************************************
*
*  File: check.c
*
*  Contents:  Various integrity and consistency checks.
*
*/

#include "include.h"

/* maximum errors of any type before aborting */
#define MAXERR 10
int wrap_check ARGS((void));

/*********************************************************************
*
*  Function: run_checks()
*
*  Purpose:  Overall control of checking.
*/

int run_checks()
{
  int numerr = 0;
  if ( memdebug ) memory_report();
  numerr += list_check();
  if ( web.representation != SIMPLEX )
    { numerr += facetedge_check(REGCHECK);
      if ( web.skel[BODY].count > 0 )
        numerr += facet_body_check();
      numerr += collapse_check();
    }
  numerr += method_check();
  if ( web.symmetry_flag ) numerr += wrap_check(); 
  check_count = numerr;
  return numerr;
} /* end run_checks() */

/**********************************************************************
*
* Function: method_check()
*
* purpose: see if any method applied twice to an element.
*
* return: number of duplications.
*/

int method_check()
{ int type;
  element_id id;
  int count = 0;
  int k,n;

  for ( type = VERTEX; type < FACETEDGE ; type++ )
  { int meth_offset =  get_meth_offset(type);
    FOR_ALL_ELEMENTS(type,id)
     { struct element *e_ptr = elptr(id);
       int *instlist = (int*)((char*)e_ptr + meth_offset);
       for ( k = 0; k < (int)e_ptr->method_count ; k++ )
         for ( n = k+1; n < (int)e_ptr->method_count ; n++ )
           if ( abs(instlist[k]) == abs(instlist[n]) )
           { sprintf(msg,"%s %d has method %s twice.\n",
               typenames[type],ordinal(id)+1,METH_INSTANCE(abs(instlist[k]))->name);
             outstring(msg);
             count++;
           }
      }
  }
  return count;
} /* end method_check() */

/**********************************************************************
*
*  Function: list_check()
*
*  Purpose:  Check element linked lists.
*
*/

int list_check()
{
  int type;
  int usedcount,maxused,discards;
  element_id id,prev_id;
  int numerr = 0;

  free_discards(DISCARDS_ALL);
  for ( type = 0 ; type < 5 ; type++ )
  { 
    int freecount,maxfree;
    if ( (web.representation == SIMPLEX)  && (type == EDGE) ) continue;
    /* check free list */
    maxfree = web.skel[type].maxcount - web.skel[type].count
                 - web.skel[type].discard_count;
    id = web.skel[type].free;
    freecount = 0;
    while ( id != NULLID )
    { freecount++;
      if ( freecount > maxfree )
      { sprintf(msg,"Type %d freelist has too many elements: %d.\n",
            type,freecount);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
        {erroutstring("Too many errors.\n"); return numerr;}
        break;
      }
      if ( id_type(id) != type )
      { sprintf(msg,"Type %d freelist has bad id %lX\n",type,id);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
        {erroutstring("Too many errors.\n"); return numerr;}
        break;
      }
      id = elptr(id)->forechain;
    } /* end while */

    if ( freecount != maxfree )
    { sprintf(msg,"Type %d freelist has %d elements instead of %d.\n",
          type,freecount,maxfree);
      erroutstring(msg);
    }
    if ( !equal_id(id,NULLID) )
    { sprintf(msg,"Type %d freelist last id is non-null: %lX\n",type,id);
      erroutstring(msg);
    }

    /* check used list */
    maxused = web.skel[type].count;
    id = web.skel[type].used;
    prev_id = NULLID;
    usedcount = 0;
    discards = 0;
    while ( valid_id(id) )
    { 
      if ( valid_element(id) )
        usedcount++;
      else discards++;

      if ( usedcount > maxused )
      { sprintf(msg,"Type %d usedlist has too many elements: %d.\n",
            type,usedcount);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
        {erroutstring("Too many errors.\n"); return numerr;}
        break;
      }
      if ( id_type(id) != type )
      { sprintf(msg,"Type %d usedlist has bad id %lX of type %d\n",
            type,id,id_type(id));
        erroutstring(msg);
        if ( ++numerr > MAXERR )
        { erroutstring("Too many errors.\n"); return numerr;}
        break;
      }
      if ( !equal_id(prev_id,elptr(id)->backchain) )
      { sprintf(msg,"Type %d used list id %lX has backchain %lX instead of %lX\n",
              type,id,elptr(id)->backchain,prev_id);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
        {erroutstring("Too many errors.\n"); return numerr;}
      }
      prev_id = id;
      id = elptr(id)->forechain;
    } /* end while */
    if ( usedcount != maxused )
    { sprintf(msg,"Type %d usedlist has %d elements.\n",type,usedcount);
      erroutstring(msg);
    }
    if ( discards != web.skel[type].discard_count )
    { sprintf(msg,"Type %d usedlist has %d elements.\n",type,usedcount);
      erroutstring(msg);
    }
    if ( !equal_id(id,NULLID) )
    { sprintf(msg,"Type %d usedlist last id is non-null: %lX\n",type,id);
      erroutstring(msg);
    }
  } /* end for loop */

  return numerr;
} /* end list_check() */



/**********************************************************************
*
*  Function: facetedge_check()
*
*  Purpose:  Check integrity of facetedge chains both ways.
*/

int facetedge_check(flag)
int flag;  /* PRELIMCHECK for check before subdivision, else REGCHECK */
{
  vertex_id v_id;
  edge_id e_id;
  facet_id f_id;
  facetedge_id fe_id,last_fe;
  int count;
  int numerr = 0;
  int n;

  /* check facetedge chain consistencies */
  if ( numerr >= MAXERR ) numerr = 0;
  count = 0;
  FOR_ALL_FACETEDGES(fe_id)
  { facetedge_id fe;

    f_id = get_fe_facet(fe_id);
    if ( valid_id(f_id) && !valid_element(f_id) )
     { sprintf(msg,"Facetedge  %d links to invalid facet %08lX.\n",
          ordinal(fe_id)+1,f_id);
        erroutstring(msg);
     }
    if ( valid_id(f_id) && !valid_element(get_facet_fe(f_id)) )
     { sprintf(msg,"Facetedge %d links to facet %d with no facetedge.\n",
        ordinal(fe_id)+1,ordinal(f_id)+1);
        erroutstring(msg);
     }
    e_id = get_fe_edge(fe_id);
    if ( valid_id(e_id) && !valid_element(e_id) )
     { sprintf(msg,"Facetedge %d links to invalid edge %08lX.\n",
             ordinal(fe_id)+1,e_id);
        erroutstring(msg);
     }
    if ( valid_id(e_id) && !valid_element(get_edge_fe(e_id)) )
     { sprintf(msg,"Facetedge %d links to edge %d with no facetedge.\n",
         ordinal(fe_id)+1,ordinal(e_id)+1);
        erroutstring(msg);
     }
    fe = get_prev_edge(fe_id);
    if ( valid_id(fe) && !equal_id(get_next_edge(fe),fe_id) )
      { sprintf(msg,"Facetedge %d has bad prev edge link\n",ordinal(fe_id)+1);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); return numerr;}
      }
    fe = get_next_edge(fe_id);
    if ( valid_id(fe) && !equal_id(get_prev_edge(fe),fe_id) )
      { sprintf(msg,"Facetedge %d has bad next edge link\n",ordinal(fe_id)+1);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); return numerr;}
      }
    fe = get_next_facet(fe_id);
    if ( valid_id(fe) && !equal_id(get_prev_facet(fe),fe_id) )
      { sprintf(msg,"Facetedge %d has bad next facet link\n",ordinal(fe_id)+1);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); return numerr;}
      }
    fe = get_prev_facet(fe_id);
    if ( valid_id(fe) && !equal_id(get_next_facet(fe),fe_id) )
      { sprintf(msg,"Facetedge %d has bad prev facet link\n",ordinal(fe_id)+1);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); return numerr;}
      }
    fe = get_next_edge(fe_id);
    if ( valid_id(fe) && !equal_id(get_fe_headv(fe_id),get_fe_tailv(fe)) )
      { sprintf(msg,"Facetedge %d head vertex disagrees with next tail.\n",
        ordinal(fe_id)+1);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); return numerr;}
      }
    fe = get_prev_edge(fe_id);
    if ( valid_id(fe) && !equal_id(get_fe_tailv(fe_id),get_fe_headv(fe)) )
      { sprintf(msg,"Facetedge %d tail vertex disagrees with prev head.\n",
        ordinal(fe_id)+1);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); return numerr;}
      }
    count++;
     }
  if ( count != web.skel[FACETEDGE].count )
    { sprintf(msg,"Only %d facetedges out of %ld generated.\n",
         count,web.skel[FACETEDGE].count);
      erroutstring(msg);
    }

    /* check that vertices have legit edge link */
  FOR_ALL_VERTICES(v_id)
  {
    e_id = get_vertex_edge(v_id);
    if ( !valid_id(e_id) ) continue; 
    if ( !valid_element(e_id) ) 
     { sprintf(errmsg,"Vertex %d has invalid edge link.\n",
          ordinal(v_id)+1);
       kb_error(1829,errmsg,WARNING);
       continue;
     }
    else if (get_vattr(v_id) & Q_MIDPOINT)
     { if ( !equal_id(v_id,get_edge_midv(e_id)) )
        { sprintf(errmsg,"Vertex %d has bad midpoint edge link.\n",ordinal(v_id)+1);
          kb_error(1830,errmsg,WARNING);
          continue;
        }
     }
    else if (get_vattr(v_id) & Q_MIDEDGE)
    { vertex_id *v = get_edge_vertices(e_id);
      int i;
      for ( i = 0 ; i <= web.lagrange_order ; i++ )
         if ( equal_id(v_id,v[i]) ) break;
      if ( i > web.lagrange_order )
      { sprintf(errmsg,"Vertex %d has bad edge link.\n",ordinal(v_id)+1);
        kb_error(1831,errmsg,WARNING);
      }
    }
    else if (get_vattr(v_id) & Q_MIDFACET) { }
    else if ( !equal_id(v_id,get_edge_tailv(e_id)) )
     { sprintf(errmsg,"Vertex %d has bad edge link.\n",ordinal(v_id)+1);
       check_vertex_fe(v_id);
       kb_error(1832,errmsg,WARNING);
       continue;
     }
  } /* end vertices */

  FOR_ALL_EDGES(e_id)
  { edge_id ee_id;

    v_id = get_edge_tailv(e_id);
    n = 0; ee_id = e_id;
    do { ee_id = get_next_tail_edge(ee_id); 
         if ( ++n > 2*web.skel[EDGE].count )
          { sprintf(errmsg,"Vertex %d has bad edge loop from edge %d.\n",
            ordinal(v_id)+1,ordinal(e_id)+1);
            kb_error(1833,errmsg,WARNING);
            break;
          }
       }
    while ( !equal_id(ee_id,e_id) );
    v_id = get_edge_headv(e_id);
    n = 0; ee_id = e_id;
    do { ee_id = get_next_head_edge(ee_id); 
         if ( ++n > 2*web.skel[EDGE].count )
          { sprintf(errmsg,"Vertex %d has bad edge loop from edge %d.\n",
            ordinal(v_id)+1,ordinal(e_id)+1);
            kb_error(1834,errmsg,WARNING);
            break;
          }
       }
    while ( !equal_id(ee_id,e_id) );
  } /* end edges */

  /* some more checks on edges */
  count = 0;
  bare_edge_count = 0;
  FOR_ALL_EDGES(e_id)
  { vertex_id v_id = get_edge_headv(e_id);
    facetedge_id first_fe;

    if ( get_vattr(v_id) & AXIAL_POINT )
      { sprintf(msg,"Vertex %d is axial and not first vertex of edge %d.\n",
            ordinal(v_id)+1,ordinal(e_id)+1);
        erroutstring(msg);
      }
    last_fe = NULLID;
    fe_id = first_fe = get_edge_fe(e_id);
    if ( valid_id(fe_id) ) do
    { edge_id ee_id = get_fe_edge(fe_id);
      if ( !equal_id(e_id,ee_id) )
         { sprintf(msg,"Facetedge %d on edge %d instead of %d.\n",
             ordinal(fe_id)+1,(inverted(ee_id)?-1:1)*(ordinal(ee_id)+1),
                 (inverted(e_id)?-1:1)*(ordinal(e_id)+1));
           erroutstring(msg);
           if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); goto facet_check;}
         }
      count++;
      if ( count > web.skel[FACETEDGE].count )
      { sprintf(msg,"Bad chain of facetedges around edge %d.\n",
        ordinal(e_id)+1);
        erroutstring(msg);
        if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); goto facet_check;}
      }
      last_fe = fe_id;
      fe_id = get_next_facet(fe_id);
    } while ( valid_id(fe_id) && !equal_id(fe_id,first_fe) );
    if ( last_fe == NULLID )
     { 
       if ( (web.representation == SOAPFILM) && (flag == REGCHECK) ) 
       {
         bare_edge_count++;
         if ( get_eattr(e_id) & BARE_NAKED ) continue;
         sprintf(msg,"Edge %d has no facets.\n",ordinal(e_id)+1);
         erroutstring(msg);
         if ( get_original(e_id)  != ordinal(e_id)+1 )
            { sprintf(msg,"      (originally edge %d)\n",get_original(e_id));
              erroutstring(msg);
            }
       }
     }
    else if ( !equal_id(get_edge_fe(e_id),get_next_facet(last_fe)) )
     { sprintf(msg,"Facets around edge %d do not link up.\n",
          ordinal(e_id)+1);
       erroutstring(msg);
       if ( get_original(e_id)  != ordinal(e_id)+1 )
          { sprintf(msg,"      (originally edge %d)\n",get_original(e_id));
            erroutstring(msg);
          }
     }
  }
  if ( count != web.skel[FACETEDGE].count )
     { sprintf(msg,"Edges have %d facetedges out of %ld used.\n",
          count,web.skel[FACETEDGE].count);
       erroutstring(msg);
       ++numerr;
     }

facet_check:
  if ( numerr >= MAXERR ) numerr = 0;
  if ( web.representation == SOAPFILM ) 
  { int i;
    count = 0;
    FOR_ALL_FACETS(f_id)
    {
      int thiscount = 0;
      last_fe = NULLID;
      fe_id = get_facet_fe(f_id);
      for ( i = 0 ; i < FACET_EDGES ; i++, fe_id = get_next_edge(fe_id) )
      { vertex_id v_id = get_fe_tailv(fe_id);
        facet_id ff_id;
        if ( (thiscount != 0) && (get_vattr(v_id) & AXIAL_POINT))
        { sprintf(msg,
            "Vertex %d is axial and not the first vertex of facet %d (original %d).\n",
              ordinal(v_id)+1,ordinal(f_id)+1,get_original(f_id)+1);
          erroutstring(msg);
        }
        ff_id = get_fe_facet(fe_id);
        if ( !equal_id(f_id,ff_id) )
         { sprintf(msg,"Facetedge %d on facet %d instead of %d.\n",
             ordinal(fe_id)+1,(inverted(ff_id)?-1:1)*(ordinal(ff_id)+1),
               ordinal(f_id)+1);
              erroutstring(msg);
           if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); break;}
         }
         count++;
         if ( ++thiscount > web.skel[FACETEDGE].count )
         { sprintf(msg,"Facetedge loop not closed on facet %d.\n",
             ordinal(f_id)+1);
           erroutstring(msg);
           if ( get_original(f_id) != ordinal(f_id)+1  )
             { sprintf(msg,"      (originally facet %d)\n",get_original(f_id));
                erroutstring(msg);
             }
           break;
         }
       last_fe = fe_id;
     } /* end while */
     if ( !equal_id(get_facet_fe(f_id),get_next_edge(last_fe)) )
     { sprintf(msg,"Edges around facet %d do not link up.\n",
          ordinal(f_id)+1);
       erroutstring(msg);
       if ( get_original(f_id) != ordinal(f_id)+1  )
          { sprintf(msg,"      (originally facet %d)\n",get_original(f_id));
            erroutstring(msg);
          }
       if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); break;}
     }
     if ( (thiscount != 3) && (flag == REGCHECK)  )
     { sprintf(msg,"Facet %d has %d edges.\n",ordinal(f_id)+1,thiscount);
       erroutstring(msg);
       if ( get_original(f_id) != ordinal(f_id)+1 )
          { sprintf(msg,"      (originally facet %d)\n",get_original(f_id));
            erroutstring(msg);
          }
       if ( ++numerr > MAXERR )
          {erroutstring("Too many errors.\n"); break;}
     }
    }
    if ( count != web.skel[FACETEDGE].count )
     { sprintf(msg,"Facets have %d facetedges out of %ld used.\n",
          count,web.skel[FACETEDGE].count);
       erroutstring(msg);
       ++numerr;
     }
  } /* end SOAPFILM */
  return numerr;
} /* end facetedge_check() */


/************************************************************************
*
*  Function: facet_body_check()
*
*  Purpose:  Check whether adjacent facets have same bodies.
*        Should be used only for complete bodies.
*/

int facet_body_check()
{
  facetedge_id fe;
  facet_id f_id;
  body_id b_id;
  int numerr = 0;
  int orig;
      

  FOR_ALL_BODIES(b_id)
  { fe = get_body_fe(b_id);
    if ( !valid_id(fe) ) continue;
    if ( !valid_element(fe) )
    { sprintf(msg,"Body %d has invalid facetedge link to edge %d facet %d.\n",
       ordinal(b_id)+1,ordinal(get_fe_edge(fe))+1,ordinal(get_fe_facet(fe))+1);
      erroutstring(msg);
      numerr++;
      continue;
     }
     f_id = get_fe_facet(fe);
     if ( !equal_id(b_id,get_facet_body(f_id) ) )
     { sprintf(msg,"Body %d has link to facet %d, which is on body %d.\n",
         ordinal(b_id)+1,inverted(f_id)?-(ordinal(f_id)+1):ordinal(f_id)+1,
         ordinal(get_facet_body(f_id))+1);
       erroutstring(msg);
       numerr++;
     }
  }

  if ( web.representation != SOAPFILM ) return numerr;

  FOR_ALL_FACETEDGES(fe)
  { body_id bb_id;
    facet_id ff_id;
    edge_id e_id;
    int k;

    for ( k = 0 ; k < 2 ; k++ )
    { fe = inverse_id(fe);
     
      f_id = get_fe_facet(fe);
      b_id = get_facet_body(f_id);
      if ( equal_id(fe,get_prev_facet(fe)) ) continue;
      ff_id = get_fe_facet(fe_inverse(get_prev_facet(fe)));
      bb_id = get_facet_body(ff_id);
      e_id = get_fe_edge(fe);
      if ( !equal_id(b_id,bb_id) 
             && !(get_attr(e_id)&(CONSTRAINT|FIXED|BOUNDARY)) )
      { sprintf(msg,"Inconsistent bodies for facets on edge %d.",
              ordinal(e_id)+1 );
        erroutstring(msg);
        numerr++;
        orig = get_original(e_id);
        if ( orig && (orig != ordinal(e_id)+1) && (orig != -1) )
        { sprintf(msg,"      (originally edge %d)",orig);
          erroutstring(msg);
        }
        sprintf(msg,"\n     facet %d",oid(f_id));
        erroutstring(msg);
        orig = get_original(f_id);
        if ( orig && (orig != ordinal(f_id)+1) )
        { sprintf(msg," (orig. %d)",orig); erroutstring(msg); }
        if ( valid_id(b_id) )
        {
          sprintf(msg," has body %d",ordinal(b_id)+1);
            erroutstring(msg);
          orig = get_original(b_id);
          if ( orig && (orig != ordinal(b_id)+1) )
             { sprintf(msg," (orig. %d)",orig);
               erroutstring(msg);
             }
        }
        else erroutstring(" has no body");
        sprintf(msg,"; facet %d",oid(ff_id));
        erroutstring(msg);
        orig = get_original(ff_id);
        if ( orig && (orig != ordinal(ff_id)+1) )
        { sprintf(msg," (orig. %d)",orig);
          erroutstring(msg);
        }
        if ( valid_id(bb_id) )
        {
          sprintf(msg," has body %d",ordinal(bb_id)+1);
          erroutstring(msg);
          orig = get_original(bb_id);
          if ( orig && (orig != ordinal(bb_id)+1) )
             { sprintf(msg," (orig. %d)",orig);
               erroutstring(msg);
             }
        }
        else erroutstring(" has no body");
        erroutstring(".\n");
        /* just a warning, not an error */
      }
    }
   } /* end FACETEDGES */
  return numerr;
} /* end facet_body_check() */


/**********************************************************************
*
*  Function: collapse_check()
*
*  Purpose:  Checks whether adjacent facets have collapsed.
*        Tests whether they have the same third vertex.
*        Also checks for edges that have the same endpoints.
*/

int vvvvcomp(a,b)
struct vvvv *a,*b;
{ int i;
  for ( i = 0 ; i < 3 ; i++ )
  { if ( a->vord[i] < b->vord[i] ) return -1;
    if ( a->vord[i] > b->vord[i] ) return 1;
  }
  return 0;
}

int collapse_check()
{
  struct vvvv *vvlist,*current;
  facet_id f_id;
  edge_id e_id;
  int ord1,ord2,ord3,tmp;
  int i;
  int count;
  int numerr = 0;
  int orig;

  /* check edges */
  if ( web.skel[EDGE].count == 0 ) return numerr;
  vvlist = (struct vvvv *)temp_calloc(web.skel[EDGE].count,sizeof(struct vvvv));

  current = vvlist;
  count = 0;
  FOR_ALL_EDGES(e_id)
  { 
    ord1 = ordinal(get_edge_tailv(e_id));
    ord2 = ordinal(get_edge_headv(e_id));
    if ( ord1 == ord2 )
    { sprintf(msg,"Edge %d",ordinal(e_id)+1);
      erroutstring(msg);
      orig = get_original(e_id);
      if ( orig && (orig != ordinal(e_id)+1) )
        { sprintf(msg," (orig. %d)\n",orig);
          erroutstring(msg);
        }
      sprintf(msg," is loop on vertex %d",ord1+1);
      erroutstring(msg);
      orig = get_original(get_edge_tailv(e_id));
      if ( orig && (orig != ord1+1) )
        { sprintf(msg," (orig. %d)\n",orig);
          erroutstring(msg);
        }
      erroutstring(".\n");
    }
    if ( ord1 <= ord2 )
      { current->id = e_id;
        current->vord[0] = ord1;
        current->vord[1] = ord2;
      }
    else
      { current->id = inverse_id(e_id);
        current->vord[0] = ord2;
        current->vord[1] = ord1;
      }
    if ( ++count > web.skel[EDGE].count )
      { erroutstring("Edge count disagrees with supposed value.");
        count--;
        break;
      }
    current++;
  }
  qsort((char *)vvlist,count,sizeof(struct vvvv),FCAST vvvvcomp);

  /* scan list for duplicates */
  for ( i = 0 ; i < count-1 ; i++ )
   { if ( vvvvcomp(vvlist+i,vvlist+i+1) == 0 )
     if ( web.symmetry_flag &&
        (get_edge_wrap(vvlist[i].id) == get_edge_wrap(vvlist[i+1].id)) )
      { sprintf(msg,"Edges %d and %d have same endpoints: %d %d.\n",
         ordinal(vvlist[i].id)+1,ordinal(vvlist[i+1].id)+1,vvlist[i].vord[0]+1,
         vvlist[i].vord[1]+1);
         erroutstring(msg);
      }
   }
  temp_free((char*)vvlist);

  if ( web.representation == STRING ) return numerr;

  /* check facets */
  if ( web.skel[FACET].count == 0 ) return numerr;
  vvlist = (struct vvvv *)temp_calloc(web.skel[FACET].count,sizeof(struct vvvv));

  current = vvlist;
  count = 0;
  FOR_ALL_FACETS(f_id)
  { 
    facetedge_id fe;
    fe = get_facet_fe(f_id);
    ord1 = ordinal(get_fe_tailv(fe));
    ord2 = ordinal(get_fe_headv(fe));
    ord3 = ordinal(get_fe_headv(get_next_edge(fe)));
    /* bubble sort */
    if ( ord1 > ord2 )
      { tmp = ord1; ord1 = ord2; ord2 = tmp; invert(f_id); }
    if ( ord2 > ord3 )
      { tmp = ord2; ord2 = ord3; ord3 = tmp; invert(f_id); }
    if ( ord1 > ord2 )
      { tmp = ord1; ord1 = ord2; ord2 = tmp; invert(f_id); }
    current->id = f_id;
    current->vord[0] = ord1;
    current->vord[1] = ord2;
    current->vord[2] = ord3;
    if ( ++count > web.skel[FACET].count )
      { erroutstring("Facet count disagrees with supposed value.");
        count--;
        if ( ++numerr > MAXERR )
              {erroutstring("Too many errors.\n"); return numerr;}
        break;
      }
    current++;
  }
  qsort((char *)vvlist,count,sizeof(struct vvvv),FCAST vvvvcomp);

  /* scan list for duplicates */
  for ( i = 0 ; i < count-1 ; i++ )
  { if ( vvvvcomp(vvlist+i,vvlist+i+1) == 0 )
      { sprintf(msg,"Facets %d and %d have same vertices: %d %d %d.\n",
        ordinal(vvlist[i].id)+1,ordinal(vvlist[i+1].id)+1,vvlist[i].vord[0]+1,
        vvlist[i].vord[1]+1,vvlist[i].vord[2]+1);
        erroutstring(msg);
      }
  }
  temp_free((char*)vvlist);
  return numerr;
} /* end collapse_check() */



/**********************************************************************
*
*  Function:  normal_change_check()
*
*  Purpose:    Checks how much the normal of facets have changed
*         during motion. Returns the largest (delta normal)/(normal)
*/

REAL normal_change_check()
{
  facet_id f_id;
  facetedge_id fe;
  vertex_id v_id;
  REAL side[2][MAXCOORD];
  REAL max_delta;
  REAL old_normal[MAXCOORD];
  REAL new_normal[MAXCOORD];
  REAL x[3][MAXCOORD];
  int ord_v;
  REAL diff;
  struct boundary *bdry;
  int i,n;

  max_delta = 0.0;

  FOR_ALL_FACETS(f_id)
  { int ii;
    if ( get_fattr(f_id) & FIXED ) continue;

    /* get old normal */
    fe = get_facet_fe(f_id);
    for ( ii = 0 ; ii < FACET_EDGES ; ii++, fe = get_next_edge(fe) )
    {
      v_id = get_fe_tailv(fe);
      ord_v = ordinal(v_id);
      if ( get_vattr(v_id) & BOUNDARY )     
      { bdry = get_boundary(v_id);
        for ( i = 0 ; i < SDIM ; i++ )
          x[n][i] = eval(bdry->coordf[i],saved.coord[ord_v],v_id,NULL);
      }
      else
      {    
        for ( i = 0 ; i < SDIM ; i++ )
           x[n][i] = saved.coord[ord_v][i];
      }
    }
    for ( i = 0 ; i < SDIM ; i++ )
    { side[0][i] = x[1][i] - x[0][i];
      side[1][i] = x[2][i] - x[0][i];
    }
    cross_prod(side[0],side[1],old_normal);

    /* get new normal */
    fe = get_facet_fe(f_id);
    for ( n = 0 ; n < FACET_EDGES ; n++, fe = get_next_edge(fe) )
    {
      v_id = get_fe_tailv(fe);
      for ( i = 0 ; i < SDIM ; i++ )
          x[n][i] = get_coord(v_id)[i];
    }
    for ( i = 0 ; i < SDIM ; i++ )
    { side[0][i] = x[1][i] - x[0][i];
      side[1][i] = x[2][i] - x[0][i];
    }
    cross_prod(side[0],side[1],new_normal);

    /* test difference */
    for ( i = 0 ; i < SDIM ; i++ )
     side[0][i] = new_normal[i] - old_normal[i];
    diff = sqrt(SDIM_dot(side[0],side[0])/
                         SDIM_dot(old_normal,old_normal));
    if ( diff > max_delta ) 
    max_delta = diff;
  }

  return max_delta;
} /* end normal_change_check() */

/********************************************************************
*
* function: wrap_check()
*
* purpose: check wraps around facet make identity, except for axial
*         point.
*
* return: number of bad facets.
*/

int wrap_check()
{
  facet_id f_id;
  int numerr = 0;

  FOR_ALL_FACETS(f_id)
  {
     facetedge_id fe,start_fe;
     int wrap = 0;
     fe = start_fe = get_facet_fe(f_id);
     do
     { if ( get_vattr(get_fe_tailv(fe)) & AXIAL_POINT )
       { wrap = 0; break; }
       wrap = (*sym_compose)(wrap,get_fe_wrap(fe));
       fe = get_next_edge(fe);
     } while ( valid_id(fe) && !equal_id(fe,start_fe) );
     if ( valid_id(fe) && (wrap != 0) )
       { sprintf(errmsg,"Wraps around facet %d not consistent.\n",ordinal(f_id)+1);
         kb_error(2000,errmsg,WARNING);
         numerr++;
       }
  }
  return numerr;
} /* end wrap_check() */



