/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/


/**********************************************************************
*
*  File: graphgen.c
*
*  Purpose: Generates triangles with normals and colors for    
*              feeding to device-specific graphing routines.
*
*/

#include "include.h"
#include "ytab.h"

/* prototypes to keep some compilers happy */
int INDEX_TO_RGBA ARGS((int));
int get_edge_color_2 ARGS((edge_id));
int get_facet_color_2 ARGS((facet_id));
int get_facet_backcolor_2 ARGS((facet_id));
int get_facet_frontcolor_2 ARGS((facet_id));

/*************************************************************************
*  Some preliminary stuff for handling RGBA colors.
*/
int INDEX_TO_RGBA(c)
int c;
{ return 
    ((int)(rgb_colors[c][0]*255)<<24) +
    ((int)(rgb_colors[c][1]*255)<<16) +
    ((int)(rgb_colors[c][2]*255)<<8)  +
    ((int)(rgb_colors[c][3]*255));
}

int get_edge_color_2(e_id)
edge_id e_id;
{ if ( edge_rgb_color_attr > 0 )
  { REAL *c = (REAL*)get_extra(e_id,edge_rgb_color_attr);
    return 
    ((int)(c[0]*255)<<24) +
    ((int)(c[1]*255)<<16) +
    ((int)(c[2]*255)<<8)  +
    (edge_alpha_flag ? ((int)(c[3]*255)) : 255);
  } 
  else return get_edge_color(e_id);
}

int get_facet_frontcolor_2(f_id)
facet_id f_id;
{ if ( facet_rgb_color_attr > 0 )
  { REAL *c = (REAL*)get_extra(f_id,facet_rgb_color_attr);
    return 
    ((int)(c[0]*255)<<24) +
    ((int)(c[1]*255)<<16) +
    ((int)(c[2]*255)<<8)  +
    (facet_alpha_flag ? ((int)(c[3]*255)) : 255);
  } 
  else return get_facet_frontcolor(f_id);
}

int get_facet_color_2(f_id)
facet_id f_id;
{ return get_facet_frontcolor_2(f_id);
}

int get_facet_backcolor_2(f_id)
facet_id f_id;
{
  if ( facet_rgb_color_attr > 0 )
  {
    if ( facet_rgb_backcolor_attr > 0 )
    { REAL *c = (REAL*)get_extra(f_id,facet_rgb_backcolor_attr);
      return 
      ((int)(c[0]*255)<<24) +
      ((int)(c[1]*255)<<16) +
      ((int)(c[2]*255)<<8)  +
      (facetback_alpha_flag ? ((int)(c[3]*255)) : 255);
    } 
    else return INDEX_TO_RGBA(get_facet_backcolor(f_id));
  }
  else return get_facet_backcolor(f_id);
}

/*****************************************************************************
*
*  Function: graphgen()
*
*  purpose:  Generates data for each triangle and calls the display
*                function graph_facet().
*
*
*  Return value: number of triangles plotted
*/

int graphgen()
{
  int  graphcount = 0;  /* number of facets done */
  int b,i;
  REAL *c;
  FILE *mapfd = NULL;
  int dummy;
  struct extra *ex;

  if ( rgb_colors_flag )
  { int three = 3;
    /* see if using RGBA colors for elements */
    edge_rgb_color_attr = find_extra("ergb",&dummy);
    if ( edge_rgb_color_attr > 0 )
    { ex = EXTRAS(EDGE)+edge_rgb_color_attr;
      if ( ex->datacount < 3 ) 
        expand_attribute(EDGE,edge_rgb_color_attr,&three);
      edge_alpha_flag = (ex->datacount >= 4);
    }
    facet_rgb_color_attr = find_extra("frgb",&dummy);
    if ( facet_rgb_color_attr > 0 )
    { ex = EXTRAS(FACET)+facet_rgb_color_attr;
      if ( ex->datacount < 3 )
        expand_attribute(FACET,facet_rgb_color_attr,&three);
      facet_alpha_flag = (ex->datacount >= 4);
    }
    facet_rgb_backcolor_attr = find_extra("fbrgb",&dummy);
    if ( facet_rgb_backcolor_attr > 0 )
    { ex = EXTRAS(FACET)+facet_rgb_backcolor_attr;
      if ( ex->datacount < 3 ) 
        expand_attribute(FACET,facet_rgb_backcolor_attr,&three);
      facetback_alpha_flag = (ex->datacount >= 4);
    }
  } else /* turn off rgb */
  { edge_rgb_color_attr = -1;
    facet_rgb_color_attr = -1;
    facet_rgb_backcolor_attr = -1;
  }
  (*graph_start)();  /* device-specific initialization */

  if ( markedgedrawflag )
  { edge_id e_id;
    FOR_ALL_EDGES(e_id) unset_attr(e_id,EDGE_DRAWN);
  }

  iterate_flag = 2;
  if ( box_flag )
    for ( i = 0 ; i < SDIM ; i++ ) 
      bounding_box[i][0] = bounding_box[i][1] = 0.0;

  if ( colorflag )
  do 
  {
    if ( strlen(cmapname) == 0 )
      prompt("Enter name of colormap file: ",cmapname,sizeof(cmapname));
    if ( cmapname[0] == 0 )
    { outstring("No colormap used.\n"); colorflag = 0; }
    mapfd = path_open(cmapname,NOTDATAFILENAME);
    if ( mapfd )
    {
      colormap = (maprow *)temp_calloc(4*web.bodycount,sizeof(REAL));
      for ( b = 0 ; b < web.bodycount ; b++ )
      { c = colormap[b];
#ifdef LONGDOUBLE
        if ( fscanf(mapfd,"%Lf %Lf %Lf %Lf",
#else
        if ( fscanf(mapfd,"%lf %lf %lf %lf",
#endif
            c,c+1,c+2,c+3) != 4 ) break;
      }
      if ( b < web.bodycount )
      { sprintf(errmsg,
             "Colormap file has only %d entries for %d bodies.\n",
              b,web.bodycount);
        kb_error(1046,errmsg,WARNING);
        for ( ; b < web.bodycount ; b++ )
        { c = colormap[b];
          c[0] = c[1] = c[2] = c[3] = 0.5;
        } 
      }
    }
    else perror(cmapname);
  }  while ( mapfd == NULL );
  /* call appropriate facet generator */
  if ( web.symmetry_flag && (torus_display_mode==TORUS_CONNECTED_MODE) )
  { if ( web.skel[BODY].count <= 0 )
    { kb_error(1049,
      "There are no bodies to display connectedly.  Reverting to raw mode.\n",
      WARNING);
      torus_display_mode = TORUS_RAW_MODE;
    }
  }
  if ( web.symmetry_flag && (torus_display_mode==TORUS_CONNECTED_MODE) )
  { if ( web.representation == STRING ) torus_cells();
    else torus_bodies();
    bare_edges();
  }
  else 
  { if ( web.representation == SIMPLEX ) hi_dim_graph();
    else if ( web.representation == STRING ) plain_edges();
    else if ( web.representation == SOAPFILM ) 
    { plain_facets();  bare_edges();
    }
  }


  /* bounding box (torus only) */
  if ( box_flag && web.torus_flag )
  { int k,m,n,a[MAXCOORD];
    struct graphdata gdata[2];
    gdata[0].color = gdata[0].ecolor = 
      (edge_rgb_color_attr > 0 ) ? INDEX_TO_RGBA(BLUE) : BLUE;
    gdata[0].etype = REGULAR_EDGE;

    if ( SDIM == 2 )
     for ( a[0] = 0 ; a[0] <= 1 ; a[0]++ )
      for ( a[1] = 0 ; a[1] <= 1 ; a[1]++ )
       for ( k = 0 ; k < 2 ; k++ )
       { if ( a[k] == 0 )
         { for ( m = 0 ; m < SDIM ; m++ )
           { for ( n = 0, gdata[0].x[m] = 0.0 ; n < SDIM ; n++ )
               gdata[0].x[m] += a[n]*web.torus_period[n][m];
             gdata[1].x[m] = gdata[0].x[m] + web.torus_period[k][m];
           }
           (*graph_edge_transforms)(gdata,NULLID);
         }
       }
    else
    for ( a[0] = 0 ; a[0] <= 1 ; a[0]++ )
     for ( a[1] = 0 ; a[1] <= 1 ; a[1]++ )
      for ( a[2] = 0 ; a[2] <= 1 ; a[2]++ )
       for ( k = 0 ; k < 3 ; k++ )
        if ( a[k] == 0 )
        { for ( m = 0 ; m < SDIM ; m++ )
          { for ( n = 0, gdata[0].x[m] = 0.0 ; n < SDIM ; n++ )
              gdata[0].x[m] += a[n]*web.torus_period[n][m];
            gdata[1].x[m] = gdata[0].x[m] + web.torus_period[k][m];
          }
          (*graph_edge_transforms)(gdata,NULLID);
        }
  }
  else if ( box_flag )
  { int k,m,a[MAXCOORD];
    struct graphdata gdata[2];
    gdata[0].color = gdata[0].ecolor = 
       (edge_rgb_color_attr > 0 ) ? INDEX_TO_RGBA(BLACK) : BLACK;
    gdata[0].etype = REGULAR_EDGE;

    if ( SDIM == 2 )
    { for ( a[0] = 0 ; a[0] <= 1 ; a[0]++ )
      for ( a[1] = 0 ; a[1] <= 1 ; a[1]++ )
       for ( k = 0 ; k < 2 ; k++ )
         if ( a[k] == 0 )
         { for ( m = 0 ; m < SDIM ; m++ )
           { gdata[0].x[m] = bounding_box[m][a[m]];
             gdata[1].x[m] = bounding_box[m][m==k?1:a[m]];
           }
           (*graph_edge)(gdata,NULLID);
         }
    }
    else
    for ( a[0] = 0 ; a[0] <= 1 ; a[0]++ )
     for ( a[1] = 0 ; a[1] <= 1 ; a[1]++ )
      for ( a[2] = 0 ; a[2] <= 1 ; a[2]++ )
       for ( k = 0 ; k < 3 ; k++ )
        if ( a[k] == 0 )
        { for ( m = 0 ; m < SDIM ; m++ )
          { gdata[0].x[m] = bounding_box[m][a[m]];
            gdata[1].x[m] = bounding_box[m][m==k?1:a[m]];
          }
          (*graph_edge)(gdata,NULLID);       
        }
  }
 


 
  (*graph_end)();  /* device-specific termination */

  if ( colorflag ) temp_free((char *)colormap);

  return  graphcount;
} /* end graphgen() */

/*****************************************************************
*
*  function: plain_facets()
*
*  purpose:  plots all facets one by one.
*
*/

void plain_facets()
{
  int i,j,ii,jj,k;
  facetedge_id fe,fe_id;
  edge_id e_id;
  facet_id f_id;
  body_id b0_id,b1_id;
  REAL **verts; /* for adjusted triangle vertices */
  struct graphdata *gdata;
  int ctrlpts = web.skel[FACET].ctrlpts;

  gdata = (struct graphdata *)temp_calloc(ctrlpts+1,sizeof(struct graphdata));
  verts = (REAL **)temp_calloc(ctrlpts,sizeof(REAL *));
  if ( web.modeltype == LAGRANGE )
    for ( i = 0 ; i < ctrlpts ; i++ ) 
      verts[i] = gdata[i].x;
  else
  for ( i = 0 ; i < FACET_VERTS ; i++ ) 
  { if ( web.modeltype == QUADRATIC ) /* mdpts after verts in gdata */
    { verts[2*i] = gdata[i].x;
      verts[2*i+1] = gdata[FACET_VERTS+i].x;  
    }
    else 
      verts[i] = gdata[i].x;
  }

  for ( i = 0 ; i <= ctrlpts ; i++ ) 
      gdata[i].x[3] = 1.0;  /* homogeneous coord */
     
  FOR_ALL_FACETS(f_id)
  { 
    int nbrs;  /* number of neighboring bodies */
    int fattr = get_fattr(f_id);
    if ( breakflag ) break;

    if ( (fattr & (BOUNDARY|CONSTRAINT)) && !bdry_showflag )
        continue;
    if ( fattr & NODISPLAY )
        continue;
    gdata[0].color = get_facet_color_2(f_id);
    gdata[0].backcolor = get_facet_backcolor_2(f_id);

    if ( no_wall_flag )
    { /* skip facets with all three vertices on walls */
      fe = get_facet_fe(f_id);

      if ( get_vattr(get_fe_headv(fe)) & (HIT_WALL|CONSTRAINT) )
         if ( get_vattr(get_fe_tailv(fe)) & (HIT_WALL|CONSTRAINT) ) 
          { fe = get_next_edge(fe);
            if ( get_vattr(get_fe_headv(fe)) & (HIT_WALL|CONSTRAINT) )
               continue;
          }
    }      

    if ( show_expr[FACET] && show_expr[FACET]->start )
    { 
      if ( !eval(show_expr[FACET],NULL,f_id,NULL) ) 
         gdata[0].color = gdata[0].backcolor = UNSHOWN; /* maybe do edges */
    }

    nbrs =  (valid_id(get_facet_body(f_id)) ? 1 : 0) 
            + (valid_id(get_facet_body(facet_inverse(f_id))) ? 1 : 0);
    if ( (nbrs >= 2) && !innerflag ) continue;
    if ( (nbrs < 2) && !outerflag ) continue;
    if ( colorflag ) /* get vertex color */
    { 
      b0_id = get_facet_body(f_id);
      b1_id = get_facet_body(facet_inverse(f_id));
      for ( i = 0 ; i < ctrlpts ; i++ )    /* vertex loop */
      { if ( valid_id(b0_id) ) 
         gdata[i].backcolor = 
           (facet_rgb_color_attr > 0 )? INDEX_TO_RGBA(ordinal(b0_id)) : 
                 ordinal(b0_id);
        else  gdata[i].backcolor = 0;
        if ( valid_id(b1_id) ) 
          gdata[i].color =
           ( facet_rgb_color_attr > 0 ) ? INDEX_TO_RGBA(ordinal(b1_id)): 
                ordinal(b1_id);
        else  gdata[i].color = 0;
      }
    }
    /* get vertices */
    if ( labelflag ) get_facet_verts(f_id,verts,NULL);
    else get_facet_verts_special(f_id,verts,NULL);
    if ( web.modeltype == LAGRANGE )
    { vertex_id *v = get_facet_vertices(f_id);
      for ( i = 0 ; i < ctrlpts ; i++ ) gdata[i].v_id = v[i];
    }
    else 
    { fe = get_facet_fe(f_id);
      for ( i = 0 ; i < FACET_VERTS ; i++, fe = get_next_edge(fe) )
      { e_id = get_fe_edge(fe);
        if ( web.modeltype == LINEAR ) gdata[i].v_id = get_edge_tailv(e_id);
        else if ( web.modeltype == QUADRATIC )
        { gdata[i].v_id = get_edge_tailv(e_id);
          gdata[FACET_VERTS+i].v_id = get_edge_midv(e_id); 
        }
      }
    }
    /* do inner clipping, if called for */
    if ( inner_clip_flag )
    { 
      for ( i = 0 ; i < ctrlpts ; i++ )
      { REAL dist = 0.0;
        REAL *x = get_coord(web.zoom_v);

        for ( j = 0 ; j < SDIM ; j++ )
           dist += (x[j]-verts[i][j])*(x[j]-verts[i][j]);

        if ( sqrt(dist) > inner_clip_rad ) break; /* it's a keeper */
      }
      if ( i == ctrlpts ) continue; /* entirely inside */
    }
  
    if ( gdata[0].color != gdata[0].backcolor )
    { /* need normal for separation */ 
      REAL dd;
      if ( web.modeltype == LAGRANGE )
         vnormal(gdata[0].x,gdata[web.lagrange_order].x,gdata[ctrlpts-1].x,
                   gdata[0].norm);
      else vnormal(gdata[0].x,gdata[1].x,gdata[2].x,gdata[0].norm);
      dd = sqrt(SDIM_dot(gdata[0].norm,gdata[0].norm));
      if ( dd > 0.0 )
         for ( i = 0 ; i < SDIM ; i++ )
           gdata[0].norm[i] /= dd;
    }
    if ( normflag || thickenflag ) 
    {
      fe = get_facet_fe(f_id);
      for ( i = 0 ; i < FACET_VERTS ; i++ )
      { calc_vertex_smooth_normal(get_fe_tailv(fe),fe,gdata[i].norm);
        fe = get_next_edge(fe);
      }
    }
    /* check for special edges */
    fe_id = get_facet_fe(f_id);
    if ( web.representation != SIMPLEX )
     for ( i = 0 ; i < 3 ; i++, fe_id = get_next_edge(fe_id) )
     { int eattr;
       gdata[i].etype = INVISIBLE_EDGE; /* default */
       e_id = get_fe_edge(fe_id);
       gdata[i].ecolor = get_edge_color_2(e_id);
       gdata[i].id = e_id;
       eattr = get_eattr(e_id);
       if ( markedgedrawflag )
       { if ( eattr & EDGE_DRAWN ) continue; }             
             
       if ( get_edge_color(e_id) == CLEAR ) continue;
       if ( eattr & BOUNDARY ) gdata[i].etype |= BOUNDARY_EDGE;
       if ( equal_id(get_next_facet(fe_id),fe_id) ) /* valence 1 */
          gdata[i].etype |= SINGLE_EDGE;
       else if  ( !equal_id(get_next_facet(fe_id),get_prev_facet(fe_id)) )
          gdata[i].etype |= TRIPLE_EDGE; /* triple line at least */
       if ( (eattr & HIT_WALL) && !(fattr & CONSTRAINT) )
          gdata[i].etype |= CONSTRAINT_EDGE;
       if ( (eattr & FIXED) && !(fattr & FIXED) )
          gdata[i].etype |= FIXED_EDGE;
       if ( show_expr[EDGE] && show_expr[EDGE]->start )
       { if ( eval(show_expr[EDGE],NULL,e_id,NULL) ) 
            gdata[i].etype |= REGULAR_EDGE;
         else gdata[i].etype = INVISIBLE_EDGE;
       }
       if ( markedgedrawflag && (gdata[i].etype != INVISIBLE_EDGE) )
         set_attr(e_id,EDGE_DRAWN);
     }

   if ( ridge_color_flag )
   {
     REAL side[FACET_EDGES][MAXCOORD];
     REAL otherside[FACET_EDGES][MAXCOORD];
            
     fe = get_facet_fe(f_id);
     for ( i = 0 ; i < FACET_EDGES ; i++ )
     { get_fe_side(fe,side[i]);
       get_fe_side(get_next_edge(get_next_facet(fe)),otherside[i]);
       fe = get_next_edge(fe);
     }
     for ( i = 0 ; i < FACET_EDGES ; i++ )
       if ( triple_prod(side[i],side[(i+1)%FACET_EDGES],otherside[i]) < 0.0 )
         gdata[i].ecolor = 
          ( edge_rgb_color_attr > 0 ) ? INDEX_TO_RGBA(RIDGE) : RIDGE;
       else gdata[i].ecolor = 
          ( edge_rgb_color_attr > 0 ) ? INDEX_TO_RGBA(VALLEY) : VALLEY; 
    }
    if ( web.modeltype == LINEAR )
    { /* call option-specific routine */
      gdata->flags |= LABEL_FACET|LABEL_EDGE|LABEL_HEAD|LABEL_TAIL;
      option_facet(gdata,f_id);         
    }
    else if ( web.modeltype == QUADRATIC )
    { 
      /* quadratic, plot as four subtriangles */
      struct graphdata qdata[FACET_VERTS+1];
      int flags = (gdata[0].flags | LIST_FACET) & 
           ~(LABEL_FACET|LABEL_EDGE|LABEL_HEAD|LABEL_TAIL);
      memcpy((char*)qdata,(char*)gdata,3*sizeof(struct graphdata));

      for( j=0 ; j<SDIM ; j++ )
         { qdata[1].x[j] = gdata[3].x[j]; qdata[2].x[j]=gdata[5].x[j]; }
      qdata[1].etype=INVISIBLE_EDGE;
      qdata[1].v_id = gdata[3].v_id;
      qdata[2].v_id = gdata[5].v_id;
      qdata[0].id = gdata[0].id; 
      qdata[1].id = NULLEDGE; 
      qdata[2].id = gdata[2].id;
      qdata[0].flags = flags;
      qdata[1].ecolor = BLACK;
      option_facet(qdata,f_id);

      for(j=0;j<SDIM;j++)
      { qdata[0].x[j]=gdata[3].x[j];
        qdata[1].x[j]=gdata[1].x[j];
        qdata[2].x[j]=gdata[4].x[j];
      }
      qdata[1].etype=gdata[1].etype;
      qdata[2].etype=INVISIBLE_EDGE;
      qdata[0].v_id = gdata[3].v_id;
      qdata[1].v_id = gdata[1].v_id;
      qdata[2].v_id = gdata[4].v_id;
      qdata[0].id = gdata[0].id; 
      qdata[1].id = gdata[1].id; 
      qdata[2].id = NULLEDGE;
      qdata[2].ecolor = BLACK;
      qdata[0].flags = flags;
      qdata[1].ecolor = gdata[1].ecolor;
      option_facet(qdata,f_id);

      for(j=0;j<SDIM;j++)
      { qdata[0].x[j]=gdata[5].x[j];
        qdata[1].x[j]=gdata[4].x[j];
        qdata[2].x[j]=gdata[2].x[j];
      }
      qdata[2].etype=gdata[2].etype;
      qdata[0].etype=INVISIBLE_EDGE;
      qdata[0].v_id = gdata[5].v_id;
      qdata[1].v_id = gdata[4].v_id;
      qdata[2].v_id = gdata[2].v_id;
      qdata[0].id = NULLEDGE; 
      qdata[1].id = gdata[1].id; 
      qdata[2].id = gdata[2].id;
      qdata[0].flags = flags;
      qdata[0].ecolor = BLACK;
      qdata[2].ecolor = gdata[2].ecolor;
      option_facet(qdata,f_id);

      for(j=0;j<SDIM;j++)
      { qdata[0].x[j]=gdata[3].x[j];
        qdata[1].x[j]=gdata[4].x[j];
        qdata[2].x[j]=gdata[5].x[j];
      }
      qdata[0].etype=INVISIBLE_EDGE;
      qdata[1].etype=INVISIBLE_EDGE;
      qdata[2].etype=INVISIBLE_EDGE;
      qdata[0].v_id = gdata[3].v_id;
      qdata[1].v_id = gdata[4].v_id;
      qdata[2].v_id = gdata[5].v_id;
      qdata[0].id = NULLEDGE; qdata[1].id = NULLEDGE;
      qdata[2].id = NULLEDGE;
      qdata[0].flags = flags | LABEL_FACET;
      qdata[1].ecolor = BLACK;
      qdata[2].ecolor = BLACK;
      option_facet(qdata,f_id);
    }
    else if ( web.modeltype == LAGRANGE )
    {
      /* plot as subtriangles */
      int flags = (gdata[0].flags|LIST_FACET) & 
           ~(LABEL_FACET|LABEL_EDGE|LABEL_HEAD|LABEL_TAIL);
      struct graphdata qdata[FACET_VERTS+1];
      memcpy((char*)qdata,(char*)gdata,3*sizeof(struct graphdata));
      for ( j = 0,jj=0,ii=web.lagrange_order+1 ; 
                j < web.lagrange_order ; j++,jj++ )
        for ( i = 0 ; i < web.lagrange_order-j ; i++,jj++,ii++ )
        { 
          for ( k = 0 ; k < SDIM ; k++ )
          { qdata[0].x[k]=gdata[jj].x[k];
            qdata[1].x[k]=gdata[jj+1].x[k];
            qdata[2].x[k]=gdata[ii].x[k];
          }
          qdata[0].v_id = gdata[jj].v_id;
          qdata[1].v_id = gdata[jj+1].v_id;
          qdata[2].v_id = gdata[ii].v_id;
          qdata[0].id = (j == 0) ? gdata[0].id : NULLEDGE;
          qdata[1].id = (i == web.lagrange_order-j-1) ? gdata[1].id : NULLEDGE;
          qdata[2].id = (i == 0) ? gdata[2].id : NULLEDGE;
          qdata[0].etype = (j==0) ? gdata[0].etype : INVISIBLE_EDGE;
          qdata[1].etype = (i+j+1==web.lagrange_order) ? gdata[1].etype : INVISIBLE_EDGE;
          qdata[2].etype = (i==0) ? gdata[2].etype : INVISIBLE_EDGE;
          qdata[0].ecolor = (j==0) ? gdata[0].ecolor : BLACK;
          qdata[1].ecolor = (i+j+1==web.lagrange_order) ? gdata[1].ecolor : BLACK;
          qdata[2].ecolor = (i==0) ? gdata[2].ecolor : BLACK;
          qdata[0].flags = flags;
          option_facet(qdata,f_id);

          if ( i < web.lagrange_order-j-1 )
          { int n;
            for ( k = 0 ; k < SDIM ; k++ )
            { qdata[0].x[k]=gdata[jj+1].x[k];
              qdata[1].x[k]=gdata[ii+1].x[k];
              qdata[2].x[k]=gdata[ii].x[k];
            }
            qdata[0].v_id = gdata[jj+1].v_id;
            qdata[1].v_id = gdata[ii+1].v_id;
            qdata[2].v_id = gdata[ii].v_id;
            for ( k = 0 ; k < FACET_EDGES ; k++ ) qdata[k].id = NULLEDGE;
            qdata[0].flags = flags |
              (i==web.lagrange_order/3 && j==web.lagrange_order/3 ? LABEL_FACET:0);
            for ( n = 0 ; n < FACET_VERTS ; n++ ) 
            { qdata[n].etype = INVISIBLE_EDGE;
              qdata[n].ecolor = BLACK;
            }
            option_facet(qdata,f_id);
          }
       }
    }
  } /* end FOR_ALL_FACETS */

  temp_free((char*)gdata);
  temp_free((char*)verts);
} /* end plain_facets() */

/*************************************************************************
*
* function: option_facet()
*
* purpose: call facet plotter for specific options in effect
*/
void option_facet(gdata,f_id)
struct graphdata *gdata;
facet_id f_id;
{
  if ( web.torus_flag )
  { gdata[0].flags &= ~(LIST_FACET | SIMPLE_FACET);
  }
  if ( web.torus_flag && (torus_display_mode == TORUS_CLIPPED_MODE) )
  {  torus_clip(gdata,0,f_id); }
  else
  { if ( !web.symmetry_flag ) gdata[0].flags |= SIMPLE_FACET;
    graph_facet_transforms(gdata,f_id);
  }
}

/*****************************************************************
*
* function: graph_edge_transforms()
*
* purpose: graph edge and all transforms specified in datafile
*
*/

void graph_edge_transforms(gdata,e_id)
struct graphdata *gdata;
edge_id e_id;
{ int i,j,k;
  struct graphdata gt[2];

  for ( i = 0 ; i < 2 ; i++ ) 
     gdata[i].x[SDIM] = 1.0;  /* homogeneous coord */
  if ( box_flag )
     for ( j = 0 ; j < 2 ; j++ )
       for  ( k = 0 ; k < SDIM ; k++ )
       { if ( gdata[j].x[k] < bounding_box[k][0] )
           bounding_box[k][0] = gdata[j].x[k];
         if ( gdata[j].x[k] > bounding_box[k][1] )
           bounding_box[k][1] = gdata[j].x[k];
       }
  (*graph_edge)(gdata,e_id);

  if ( transforms_flag && !lazy_transforms_flag &&
             (transform_colors_flag || (graph_facet != geomview_facet) )  )
    for ( i = 1 ;  i < transform_count ; i++ )
    { gt[0] = gdata[0];
      for ( j = 0 ; j < 2 ; j++ )
        { matvec_mul(view_transforms[i],gdata[j].x,gt[j].x,
                  SDIM+1,SDIM+1); 
          for ( k = 0 ; k <= SDIM ; k++ )
          { gt[j].x[k] /= gt[j].x[SDIM]; /* normalize */
            if ( box_flag )
            { if ( gt[j].x[k] < bounding_box[k][0] )
                 bounding_box[k][0] = gt[j].x[k];
              if ( gt[j].x[k] > bounding_box[k][1] )
                 bounding_box[k][1] = gt[j].x[k];
            }
          }
        }
        gt[0].flags = 0;
        if ( web.representation == STRING )
          gt[0].ecolor = transform_colors[i];
        (*graph_edge)(gt,e_id);
     }
}

/*****************************************************************
*
* function: graph_facet_transforms()
*
* purpose: graph facet and all transforms specified in datafile
*
*/

void graph_facet_transforms(gdata,f_id)
struct graphdata *gdata;
facet_id f_id;
{ struct graphdata gt[FACET_VERTS+1];
  int i,j,k;

  for ( i = 0 ; i < FACET_VERTS ; i++ ) 
     gdata[i].x[SDIM] = 1.0;  /* homogeneous coord */
  if ( box_flag )
    for ( j = 0 ; j < FACET_VERTS ; j++ )
      for ( k = 0 ; k < SDIM ; k++ )
      { if ( gdata[j].x[k] < bounding_box[k][0] )
          bounding_box[k][0] = gdata[j].x[k];
        if ( gdata[j].x[k] > bounding_box[k][1] )
          bounding_box[k][1] = gdata[j].x[k];
      }
  (*graph_facet)(gdata,f_id);
  if ( transforms_flag && !lazy_transforms_flag &&
             (transform_colors_flag || (graph_facet != geomview_facet) )  )
  {
    for ( i = 1 ;  i < transform_count ; i++ )
    { if ( view_transform_det[i] > 0 )
      { for ( j = 0 ; j < FACET_VERTS ; j++ )
        { gt[j] = gdata[j];
          matvec_mul(view_transforms[i],gdata[j].x,gt[j].x,SDIM+1,SDIM+1); 
          for ( k = 0 ; k <= SDIM ; k++ )
             gt[j].x[k] /= gt[j].x[SDIM]; /* normalize */
          matvec_mul(view_transforms[i],gdata[j].norm,gt[j].norm,
             SDIM,SDIM); 
        }
      }
      else
      { /* reverse orientation */
        for ( j = 0 ; j < FACET_VERTS ; j++ )
        { gt[j] = gdata[2-j];
          matvec_mul(view_transforms[i],gdata[2-j].x,gt[j].x,SDIM+1,SDIM+1); 
          for ( k = 0 ; k <= SDIM ; k++ )
            gt[j].x[k] /= gt[j].x[SDIM]; /* normalize */
          matvec_mul(view_transforms[i],gdata[2-j].norm,gt[j].norm,SDIM,SDIM); 
        }
        gt[0].color = gdata[0].color;
        gt[0].backcolor = gdata[0].backcolor;
        gt[0].ecolor = gdata[1].ecolor; gt[1].ecolor = gdata[0].ecolor;
        gt[2].ecolor = gdata[2].ecolor;
        gt[0].etype = gdata[1].etype; gt[1].etype = gdata[0].etype;
        gt[2].etype = gdata[2].etype;
        gt[0].flags = gdata[0].flags;
      }

      if ( transform_colors[i] == SWAP_COLORS )
      { int temp = gt[0].color;
        gt[0].color = gt[0].backcolor;
        gt[0].backcolor = temp;
      }
      else if ( transform_colors_flag && (transform_colors[i] != SAME_COLOR) ) 
        gt[0].color = gt[0].backcolor = transform_colors[i];
      if ( box_flag )
      for ( j = 0 ; j < FACET_VERTS ; j++ )
        for ( k = 0 ; k < SDIM ; k++ )
        { if ( gt[j].x[k] < bounding_box[k][0] )
            bounding_box[k][0] = gt[j].x[k];
          if ( gt[j].x[k] > bounding_box[k][1] )
            bounding_box[k][1] = gt[j].x[k];
        }
      (*graph_facet)(gt,f_id);
    }
  }
}

/*****************************************************************
*
*  function: plain_edges()
*
*  purpose:  plots all edges one by one.
*
*/

void plain_edges()
{ 
  int i,k;
  edge_id e_id;
  REAL **verts; /* for adjusted triangle vertices */
  struct graphdata *gdata;
  int ctrlpts = web.skel[EDGE].ctrlpts;

  gdata = (struct graphdata *)temp_calloc(ctrlpts,sizeof(struct graphdata));
  verts = dmatrix(0,ctrlpts,0,SDIM);
  gdata[0].etype = REGULAR_EDGE;
  FOR_ALL_EDGES(e_id)
  { if ( breakflag ) break;
    if ( show_expr[EDGE] && show_expr[EDGE]->start )
      if ( !eval(show_expr[EDGE],NULL,e_id,NULL) )
        continue;

    /* get vertices */
    get_edge_verts(e_id,verts,NULL);
    for ( i = 0 ; i < SDIM ; i++ )
    { gdata[1].x[i] = verts[1][i];
      gdata[0].x[i] = verts[0][i];
      if ( web.modeltype == QUADRATIC )
        gdata[2].x[i] = verts[2][i];
      else if ( web.modeltype == LAGRANGE )
      { for ( k = 0 ; k < ctrlpts ; k++ )
          gdata[k].x[i] = verts[k][i];
      }
    }
    gdata[0].color = gdata[0].ecolor = get_edge_color_2(e_id);
    gdata[0].id = e_id;

    /* call device-specific routine */
    if ( web.modeltype == LINEAR )
    { gdata[0].flags |= LABEL_EDGE|LABEL_HEAD|LABEL_TAIL;
      if ( web.torus_flag && (torus_display_mode == TORUS_CLIPPED_MODE) ) 
        torus_edge_clip(gdata,0);
      else
        (*graph_edge_transforms)(gdata,e_id);
    }
    else if ( (web.modeltype == QUADRATIC) && circular_arc_flag ) 
    { /* arc, so plot in 16 segments */
      REAL headx[MAXCOORD];
      REAL tailx[MAXCOORD];
      REAL midx[MAXCOORD];
      REAL w1[MAXCOORD],mag1;
      REAL w2[MAXCOORD],mag2;
      REAL w[MAXCOORD],mag;
      REAL xx[MAXCOORD];
      static REAL wlambda[17] = 
           { 1e30,10,5,4,3,2,1.5,1.2,1,.85,.7,.6,.5,.35,.25,.12,0.};
      for (i = 0 ; i < SDIM ; i++ ) 
      { headx[i] = gdata[2].x[i]; 
        tailx[i] = gdata[0].x[i];
        midx[i] = gdata[1].x[i];
        w1[i] = midx[i] - tailx[i];
        w2[i] = headx[i] - tailx[i];
      }
      mag1 = SDIM_dot(w1,w1); mag2 = SDIM_dot(w2,w2);
      for ( i = 0 ; i < SDIM  ; i++ )
      { w1[i] /= mag1; w2[i] /= mag2; }
      for ( i = 0 ; i < SDIM ; i++ ) gdata[0].x[i] = tailx[i];
        for ( k = 1 ; k <= 16 ; k++ )
        { for ( i = 0 ; i < SDIM ; i++ )
            w[i] = w2[i] + wlambda[k]*(w1[i]-w2[i]);
          mag = SDIM_dot(w,w);
          for ( i = 0 ; i < SDIM ; i++ )
            gdata[1].x[i] = xx[i] = tailx[i] + w[i]/mag;
          gdata[0].flags &= ~(LABEL_EDGE|LABEL_HEAD|LABEL_TAIL);
          if ( k == 1 ) gdata[0].flags |= LABEL_TAIL;
          if ( k == 8 ) gdata[0].flags |= LABEL_EDGE;
          if ( k == 16 ) gdata[0].flags |= LABEL_HEAD;
          if ( web.torus_flag && (torus_display_mode == TORUS_CLIPPED_MODE)) 
             torus_edge_clip(gdata,0);
          else (*graph_edge_transforms)(gdata,e_id);
          for ( i = 0 ; i < SDIM ; i++ )  /* gdata[1].x was messed up by clip */
             gdata[0].x[i] = xx[i];
        }
      }
      else if ( web.modeltype == QUADRATIC ) /* quadratic, so plot in 8 segments */
      { REAL headx[MAXCOORD];
        REAL tailx[MAXCOORD];
        REAL midx[MAXCOORD];
        for ( i = 0 ; i < SDIM ; i++ ) 
        { headx[i] = gdata[2].x[i]; tailx[i] = gdata[0].x[i];
          midx[i] = gdata[1].x[i];
        }
        for ( k = 1 ; k <= 8 ; k++ )
        { for ( i = 0 ; i < SDIM ; i++ )
            gdata[1].x[i] = (1-k/8.)*(1-k/4.)*tailx[i]
                                        + k/2.*(1-k/8.)*midx[i]
                                        - k/8.*(1-k/4.)*headx[i];
          gdata[0].flags &= ~(LABEL_EDGE|LABEL_HEAD|LABEL_TAIL);
          if ( k == 1 ) gdata[0].flags |= LABEL_TAIL;
          if ( k == 5 ) gdata[0].flags |= LABEL_EDGE;
          if ( k == 8 ) gdata[0].flags |= LABEL_HEAD;
          if ( web.torus_flag && (torus_display_mode == TORUS_CLIPPED_MODE)) 
             torus_edge_clip(gdata,0);
          else (*graph_edge_transforms)(gdata,e_id);
          for ( i = 0 ; i < SDIM ; i++ )  /* gdata[1].x was messed up by clip */
             gdata[0].x[i] = (1-k/8.)*(1-k/4.)*tailx[i]
                                        + k/2.*(1-k/8.)*midx[i]
                                        - k/8.*(1-k/4.)*headx[i];
        }
     }
     else if ( web.modeltype == LAGRANGE )
     { /* plot subsegments */
       for ( k = 0 ; k < ctrlpts-1 ; k++ )
       { gdata[k].ecolor = gdata[0].ecolor;
         gdata[k].etype = gdata[0].etype;
         for ( i = 0 ; i < SDIM ; i++ )  /* were messed up by clip */
         { gdata[k].x[i] = verts[k][i];
           gdata[k+1].x[i] = verts[k+1][i];
         }
         gdata[0].flags &= ~(LABEL_EDGE|LABEL_HEAD|LABEL_TAIL);
         if ( k == 0 ) gdata[0].flags |= LABEL_TAIL;
         if ( k == (ctrlpts-1)/2 ) gdata[0].flags |= LABEL_EDGE;
         if ( k == ctrlpts-2 ) gdata[0].flags |= LABEL_HEAD;
         if ( web.torus_flag && (torus_display_mode == TORUS_CLIPPED_MODE) ) 
              torus_edge_clip(gdata+k,0);
         else (*graph_edge_transforms)(gdata+k,e_id);
       }
     }
   }
  free_matrix(verts);
  temp_free((char*)gdata);

}  /* end plain_edges() */


/*****************************************************************
*
*  function: bare_edges()
*
*  purpose:  plots all facetless edges.
*
*/

void bare_edges()
{
  int i;
  edge_id e_id;
  REAL *verts[3]; /* for adjusted triangle vertices */
  struct graphdata gdata[3];     
  facetedge_id fe;

  gdata[0].etype = BARE_EDGE;
  for ( i = 0 ; i < 2 ; i++ ) verts[i] = gdata[i].x;
  FOR_ALL_EDGES(e_id)
     { 
        if ( breakflag ) break;
        gdata[0].id = e_id;
        fe = get_edge_fe(e_id);
        if ( valid_id(fe) && valid_id(get_fe_facet(fe)) ) continue;
        if ( show_expr[EDGE] && show_expr[EDGE]->start )
            if ( !eval(show_expr[EDGE],NULL,e_id,NULL) ) continue;

        /* get vertices */
        verts[0] = get_coord(get_edge_tailv(e_id));
        verts[1] = get_coord(get_edge_headv(e_id));
        for ( i = 0 ; i < SDIM ; i++ )
        { gdata[0].x[i] = verts[0][i];
        }
        gdata[0].color = gdata[0].ecolor = get_edge_color_2(e_id);
        if ( web.symmetry_flag )
          (*sym_wrap)(verts[1],gdata[1].x,get_edge_wrap(e_id));
        else for ( i = 0 ; i < SDIM ; i++ )
          gdata[1].x[i] = verts[1][i];

        /* call device-specific routine */
        gdata[0].flags |= (LABEL_EDGE|LABEL_HEAD|LABEL_TAIL);
        if ( web.torus_flag && (torus_display_mode == TORUS_CLIPPED_MODE) ) 
            torus_edge_clip(gdata,0);
        else (*graph_edge_transforms)(gdata,e_id);
     }
}


/*****************************************************************
*
*  function: triple_edges()
*
*  purpose:  plots all multiple-facet edges if triple_show_flag set.
*
*/

void triple_edges()
{
  int i;
  edge_id e_id;
  facetedge_id fe_id;
  REAL *verts[3]; /* for adjusted triangle vertices */
  struct graphdata gdata[3];

  gdata[0].etype = TRIPLE_EDGE;
  for ( i = 0 ; i < 2 ; i++ ) verts[i] = gdata[i].x;
  FOR_ALL_EDGES(e_id)
     { 
        if ( breakflag ) break;
        fe_id = get_edge_fe(e_id);
        if ( equal_id(get_next_facet(fe_id),get_prev_facet(fe_id)) ) continue;
        if ( show_expr[EDGE] && show_expr[EDGE]->start )
            if ( !eval(show_expr[EDGE],NULL,e_id,NULL) ) continue;

        /* get vertices */
        verts[0] = get_coord(get_edge_tailv(e_id));
        verts[1] = get_coord(get_edge_headv(e_id));
        verts[2] = get_coord(get_edge_midv(e_id));
        for ( i = 0 ; i < SDIM ; i++ )
        { gdata[0].x[i] = verts[0][i];
          gdata[2].x[i] = verts[2][i];
        }
        if ( web.symmetry_flag )
          (*sym_wrap)(verts[1],gdata[1].x,get_edge_wrap(e_id));
        else
          for ( i = 0 ; i < SDIM ; i++ )
             gdata[1].x[i] = verts[1][i];
        gdata[0].id = e_id;
        gdata[0].color = gdata[0].ecolor = get_edge_color_2(e_id);

        /* call device-specific routine */
        gdata[0].flags |= LABEL_EDGE|LABEL_HEAD|LABEL_TAIL;
        if ( web.torus_flag && (torus_display_mode == TORUS_CLIPPED_MODE) ) 
            torus_edge_clip(gdata,0);
        else (*graph_edge_transforms)(gdata,e_id);
     }
}

/**********************************************************************
* 
*  Function: torus_edge_clip()
*
*  Purpose:  Recursive routine to subdivide edges crossing 
*                torus cell faces.  Edges have been already 
*                unwrapped by torus periods.
*/

void torus_edge_clip(gdata,m)
struct graphdata *gdata;  /* triangle to check */
int m;                          /* coordinate number */
{
  struct graphdata gdata1[2];  /* for possible subdivided edge */
  int i,k;
  REAL t,a,b;
  int wrap[2];

  /* see if any vertices outside cell in this coordinate */
  for ( i = 0 ; i < 2 ; i++ )
     wrap[i] = (int)floor(SDIM_dot(web.inverse_periods[m],gdata[i].x));

  /* split, if necessary */
  if ( wrap[0] != wrap[1] )
     {
        int cut = (wrap[0] > wrap[1]) ? wrap[0] : wrap[1];

        /* set up head of new edge */
        gdata1[1] = gdata[1];
        gdata1[0].id = gdata[0].id;
        gdata1[0].etype = gdata[0].etype;
        gdata1[0].flags = gdata[0].flags & ~LABEL_TAIL;
        gdata[0].flags &= ~LABEL_HEAD;

        /* calculate new vertex */
        a = SDIM_dot(web.inverse_periods[m],gdata[0].x);
        b = SDIM_dot(web.inverse_periods[m],gdata1[1].x);
        t = (cut - a)/(b - a);
        for ( k = 0 ; k < SDIM ; k++ )
          gdata1[0].x[k] = gdata[1].x[k] =
              (1 - t)*gdata[0].x[k] + t*gdata1[1].x[k];

        /* wrap new edge vertices properly */
        for ( i = 0 ; i < 2 ; i++ )
         for ( k = 0 ; k < SDIM ; k++ )
             gdata1[i].x[k] -= wrap[1]*web.torus_period[m][k];
 
        /* send on for further check, or plot */
        gdata1[0].color = gdata1[0].ecolor = gdata[0].color;
        if ( m == SDIM-1 ) 
          (*graph_edge_transforms)(gdata1,gdata[0].id);
        else
          torus_edge_clip(gdata1,m+1);
      }

    /* wrap vertices properly */
    for ( i = 0 ; i < 2 ; i++ )
      for ( k = 0 ; k < SDIM ; k++ )
         gdata[i].x[k] -=  wrap[0]*web.torus_period[m][k];

    /* send on original edge structure */
    if ( m == SDIM-1 ) 
      (*graph_edge_transforms)(gdata,gdata[0].id);
    else
      torus_edge_clip(gdata,m+1);

} /* end torus_edge_clip() */

/**********************************************************************
* 
*  Function: torus_clip()
*
*  Purpose:  Recursive routine to subdivide triangles crossing 
*                torus cell faces 
*/

void torus_clip(gdata,m,f_id)
struct graphdata *gdata;  /* triangle to check */
int m;                    /* coordinate number */
facet_id f_id;            /* parent facet */
{
  struct graphdata gdata0[FACET_VERTS+1],gdata1[FACET_VERTS+1],
      gdata2[FACET_VERTS+1];
  int i,k;
  REAL t,d,a,b;
  int wrap[FACET_VERTS];
  int cut,oddman,oddflag;
  int oddwrap;
  REAL w;

  for ( i = 0 ; i < FACET_VERTS ; i++ ) gdata0[i] = gdata[i]; /* local copy */

  /* see if any vertices outside cell in this coordinate */

 for ( i = 0 ; i < FACET_VERTS ; i++ )
  { 
    w = SDIM_dot(web.inverse_periods[m],gdata0[i].x);
    wrap[i] = (w < .5) ? ((int)floor(w+.000000001)):((int)floor(w-.00000001));
    /* slight shift to prevent spurious wraps due to rounding */
  }

  /* find odd vertex */
  oddflag = 1;  
  if ( (wrap[0] == wrap[1]) && (wrap[1] == wrap[2]) )
     { oddflag = 0;  oddman = 0; }
  else if ( wrap[0] == wrap[1] ) oddman = 2;
  else if ( wrap[1] == wrap[2] ) oddman = 0;
  else if ( wrap[2] == wrap[0] ) oddman = 1;
  else
    { sprintf(msg,"Triangle spans too many periods. Wraps %d %d %d\n",
            wrap[0],wrap[1],wrap[2]);
      erroutstring(msg);
      oddman = 0;
    }

  /* wrap new triangle vertices properly */
  oddwrap = wrap[oddman];
  for ( i = 0 ; i < FACET_VERTS ; i++ )
    { 
      wrap[i] -= oddwrap;
      for ( k = 0 ; k < SDIM ; k++ )
        { d = oddwrap*web.torus_period[m][k];
          gdata0[i].x[k] -= d;
        }
    }


  /* split, if necessary */
  if ( oddflag )
     {
        int pair1 = (oddman+1)%FACET_VERTS;
        int pair2 = (oddman+2)%FACET_EDGES;

        gdata0[0].flags = 0;  /* vertices the hard way */
        /* find wrap multiple that cuts triangle */
        if ( wrap[oddman] < wrap[pair1] )
           cut = wrap[pair1];
        else
           cut = wrap[oddman];

        /* set up new triangles */
        for ( i = 0 ; i < FACET_VERTS ; i++ )
          { gdata2[i] = gdata0[i];
             gdata1[i] = gdata0[i];
          } 

        /* calculate new vertices */
        a = SDIM_dot(web.inverse_periods[m],gdata0[oddman].x);
        b = SDIM_dot(web.inverse_periods[m],gdata1[pair1].x);
        if ( fabs(a-b) < 0.00000000001 ) t = 0.5;
        else t = (cut - a)/(b - a);
        for ( k = 0 ; k < SDIM ; k++ )
          gdata1[oddman].x[k] = gdata0[pair1].x[k] =
              (1 - t)*gdata0[oddman].x[k] + t*gdata1[pair1].x[k];
        b = SDIM_dot(web.inverse_periods[m],gdata2[pair2].x);
        if ( fabs(a-b) < 0.00000000001 ) t = 0.5;
        else t = (cut - a)/(b - a);
        for ( k = 0 ; k < SDIM ; k++ )
          gdata2[oddman].x[k] = gdata0[pair2].x[k] = gdata1[pair2].x[k] =
              (1 - t)*gdata0[oddman].x[k] + t*gdata2[pair2].x[k];
        gdata0[pair1].v_id = NULLID;
        gdata0[pair2].v_id = NULLID;
        gdata1[oddman].v_id = NULLID;
        gdata1[pair2].v_id = NULLID;
        gdata2[oddman].v_id = NULLID;
        /* new edge colors are clear */
        gdata0[pair1].ecolor = CLEAR;
        gdata1[pair1].ecolor = gdata1[pair2].ecolor = CLEAR;
        gdata2[oddman].ecolor = CLEAR;
        gdata0[pair1].etype = INVISIBLE_EDGE;
        gdata1[pair1].etype = gdata1[pair2].etype = INVISIBLE_EDGE;
        gdata2[oddman].etype = INVISIBLE_EDGE;

        /* wrap new triangle vertices properly */
        for ( i = 0 ; i < FACET_VERTS ; i++ )
         for ( k = 0 ; k < SDIM ; k++ )
          { d = wrap[pair1]*web.torus_period[m][k];
             gdata1[i].x[k] -= d;
             gdata2[i].x[k] -= d;
          }
 
        /* send on for further check, or plot */
        if ( m == SDIM-1 ) 
          { (*graph_facet_transforms)(gdata1,f_id);
             (*graph_facet_transforms)(gdata2,f_id);
          }
        else
          { torus_clip(gdata1,m+1,f_id);
             torus_clip(gdata2,m+1,f_id);
          }
      }

    /* send on original triangle structure */
    if ( m == SDIM-1 ) 
      (*graph_facet_transforms)(gdata0,f_id);
    else
      torus_clip(gdata0,m+1,f_id);

} /* end torus_clip */

/********************************************************************
*
* function: bfcomp()
*
* purpose:  comparison for ordering structures in torus_bodies()
*
*/
 
int bfcomp(a,b)
struct bodyface *a,*b;
{
  if ( ordinal(a->b_id) < ordinal(b->b_id) ) return -1;
  if ( ordinal(a->b_id) > ordinal(b->b_id) ) return 1;
  if ( ordinal(a->f_id) < ordinal(b->f_id) ) return -1;
  if ( ordinal(a->f_id) > ordinal(b->f_id) ) return 1;
  if ( inverted(a->f_id) < inverted(b->f_id) ) return -1;
  if ( inverted(a->f_id) > inverted(b->f_id) ) return 1;

  return 0;
}

/**********************************************************************
*
*  Function: torus_bodies()
*
*  Purpose:  To display SOAPFILM torus model in terms of connected
*            bodies.  Finds all facets of each body and plots them
*            in a connected manner.
*/

void torus_bodies()
{
  struct bodyface *allfaces;
  int allfacetop;
  int facemax;
  facetedge_id fe,fen,ff,fa,fan;
  body_id b_id;
  int i,j,k,kk,bj;
  int numleft;
  int oldnumleft;
  int startflag;
  struct bodyface *bf;
  struct graphdata *gdata;
  facet_id f_id;
  WRAPTYPE xwrap; /* for centering body */
  int ctrlpts = web.skel[FACET].ctrlpts;
  REAL **xxx; /* for LAGRANGE unwrapped vertices */
  int *bodystarts; /* spots in bodyface list */

  gdata = (struct graphdata *)temp_calloc(ctrlpts+1,sizeof(struct graphdata));
  xxx = dmatrix(0,ctrlpts,0,SDIM);

  facemax = 2*web.skel[FACET].count;
  allfaces = (struct bodyface *) temp_calloc(facemax,sizeof(struct bodyface));
  memset((char*)gdata,0,sizeof(gdata));
  bodystarts = (int *)temp_calloc(web.skel[BODY].max_ord+3,sizeof(int));


  allfacetop = 0;

  /* first, get a list of all facets bounding the body,
     with proper orientation. Some may be included twice
     with opposite orientations in case the body wraps
     around the torus.
   */
  FOR_ALL_FACETS(f_id)
  { body_id front,back;

    front = get_facet_body(f_id);
    back = get_facet_body(facet_inverse(f_id));

    if ( valid_id(front) ) 
    { allfaces[allfacetop].b_id = front;
      allfaces[allfacetop++].f_id = f_id;        
    }
    if ( valid_id(back) )
    { allfaces[allfacetop].b_id = get_facet_body(facet_inverse(f_id));
      allfaces[allfacetop++].f_id = facet_inverse(f_id);         
    }
    if ( !valid_id(front) && !valid_id(back) )
    { allfaces[allfacetop].b_id = web.skel[BODY].max_ord+1; 
                                 /* big, so bare facets sort at end */
      allfaces[allfacetop++].f_id = f_id;         
    }
  }

  /* sort in facet order, so can find facets quickly */
  qsort((char *)allfaces,allfacetop,sizeof(struct bodyface),FCAST bfcomp);

  /* find ranges for particular bodies */
  bodystarts[0] = 0;
  i = ordinal(allfaces[0].b_id);  /* first body may not have ordinal 0 */
  for ( j = 0 ; j <= i ; j++ ) bodystarts[j] = 0;
  for ( k = 1 ; k < allfacetop ; k++ )
  { if ( allfaces[k].b_id != allfaces[k-1].b_id )
    { i = ordinal(allfaces[k].b_id); 
      for ( ; j <= i ; j++ ) bodystarts[j] = k;
    }
  }
  for ( ; j <= web.skel[BODY].max_ord+2 ; j++ )
    bodystarts[j] = k;

  /* Now go through list repeatedly, marking proper wraps
     for facet base vertices as can.
   */
  for ( bj = 0 ; bj <= web.skel[BODY].max_ord+1 ; bj++ )
  { int facetop;
    struct bodyface *faces;

    faces = allfaces + bodystarts[bj];
    facetop = bodystarts[bj+1] - bodystarts[bj];
    if ( facetop == 0 ) continue;

    if ( bj <= web.skel[BODY].max_ord ) b_id = get_ordinal_id(BODY,bj);
    else b_id = 0;

    if ( breakflag ) break;
    if ( valid_id(b_id) && show_expr[BODY] && show_expr[BODY]->start )
      if ( !eval(show_expr[BODY],NULL,b_id,NULL) ) continue;

    numleft = oldnumleft = facetop;
    do
    { startflag = (numleft == oldnumleft); /* see if need to start body piece */
      oldnumleft = numleft;
      for ( k = 0 ; k < facetop ; k++ )
      {
        if ( faces[k].wrapflag ) continue;
        if ( startflag ) /* for starting disconnected body pieces */
        { faces[k].wrap = 0; /* wrap to base cell */
          faces[k].wrapflag = 1;
          numleft--;
          startflag = 0;
          continue;
        }
        fe = get_facet_fe( faces[k].f_id );
        for ( i = 0 ; i < FACET_VERTS ; i++ )  /* check for wrapped neighbor */
        { struct bodyface key;
          fen = fe_inverse(get_prev_facet(fe));
          ff = get_fe_facet(fen);
          key.f_id = ff;
          key.b_id = b_id;
          bf = (struct bodyface *)bsearch((char *)&key,(char *)faces,
                   facetop, sizeof(struct bodyface),FCAST bfcomp);
          if ( bf == NULL )
          { sprintf(errmsg,
                "INTERNAL ERROR, torus_bodies(): missing face on body %d.\n",
                                   ordinal(b_id)+1);
            kb_error(1050,errmsg,WARNING);
            fe = get_next_edge(fe);
            continue;
          }
          if ( bf-> wrapflag == 0 ) 
          { fe = get_next_edge(fe);
            continue;
          }
                
          /* now have wrapped neighbor */ 
  
          /* start at base point of neighbor and follow edges
             accumulating wraps until new base point is reached */
  
          faces[k].wrap = bf->wrap; 
          fan = get_facet_fe(bf->f_id);
          fen = get_next_edge(fen);  /* so tail of fan winds up at tail of fe */
          while ( !equal_id(fan,fen) )
          { /* walk around neighbor */
            faces[k].wrap = (*sym_compose)(faces[k].wrap,get_fe_wrap(fan)); 
            fan = get_next_edge(fan);
          }
          fa = get_facet_fe(faces[k].f_id);
          while ( !equal_id(fa,fe) )
          { /* walk backward around new facet */
            fe = get_prev_edge(fe);
            faces[k].wrap = (*sym_compose)(faces[k].wrap,
                                              (*sym_inverse)(get_fe_wrap(fe))); 
          }
              
          faces[k].wrapflag = 1;
          numleft--;
          break; 
       } 
      }
    } while ( numleft > 0 );

    /* try to center body in cell by finding most common wrap */
    { struct { WRAPTYPE wrap; int count; } xxwrap[50];
      int wrapcount = 1;  /* number of different wraps */
      memset((char*)xxwrap,0,sizeof(xxwrap));
      for ( k = 0 ; k < facetop ; k++ )
      { for ( i = 0 ; i < wrapcount ; i++ )
          if ( xxwrap[i].wrap == faces[k].wrap )
            xxwrap[i].count++;
          if ( (i == wrapcount) && (i < 50-1) )
          { xxwrap[wrapcount].wrap = faces[k].wrap;
            xxwrap[wrapcount++].count = 1;
          }
      }
      for ( k = 0, i = 0 ; i < wrapcount ; i++ )
        if ( xxwrap[i].count > xxwrap[k].count ) k = i;
      xwrap = (*sym_inverse)(xxwrap[k].wrap);
    }
     
    /* now plot all the facets */
    for ( kk = 0 ; kk < facetop ; kk++ )
    { WRAPTYPE wrap;
      int fattr;
      facetedge_id fe_id;
      edge_id e_id;

      if ( breakflag ) break;

      f_id = faces[kk].f_id;
      fattr = get_fattr(f_id);
      if ( fattr & NODISPLAY ) continue;
      fe = get_facet_fe(f_id);
      wrap = (*sym_compose)(faces[kk].wrap,xwrap);
      if ( web.modeltype == LAGRANGE )
      { vertex_id *v = get_facet_vertices(f_id);
        WRAPTYPE lwrap;
        get_facet_verts(f_id,xxx,NULL);  /* for positive facet */
        if ( inverted(f_id) )
           lwrap = (*sym_compose)(get_fe_wrap(fe),wrap);
        else lwrap = wrap;
        for ( i = 0 ; i < ctrlpts ; i++ )
        { gdata[i].v_id = v[i];
          (*sym_wrap)(xxx[i],gdata[i].x,lwrap);
          if ( colorflag )  
            gdata[i].color = gdata[i].backcolor = 
                       (facet_rgb_color_attr > 0 ) ?
                         INDEX_TO_RGBA(ordinal(b_id)) : ordinal(b_id);
           else
           { if ( inverted(f_id) )
             { gdata[i].color = get_facet_backcolor_2(f_id);
               gdata[i].backcolor = get_facet_frontcolor_2(f_id);
             }
             else
             { gdata[i].color = get_facet_frontcolor_2(f_id);
               gdata[i].backcolor = get_facet_backcolor_2(f_id);
             }
           }
         }
       }
       else
       for ( i = 0 ; i < FACET_VERTS ; i++ )    /* vertex loop */
       { REAL *verts;
         vertex_id v_id;

          e_id = get_fe_edge(fe);
          gdata[i].v_id = v_id = get_edge_tailv(e_id);
          verts = get_coord(v_id);
          (*sym_wrap)(verts,gdata[i].x,wrap);
          if ( colorflag )  gdata[i].color = gdata[i].backcolor = 
             (facet_rgb_color_attr > 0 ) ? INDEX_TO_RGBA(ordinal(b_id)) :
                      ordinal(b_id);
          else
          { gdata[i].color = get_facet_frontcolor_2(f_id);
            gdata[i].backcolor = get_facet_backcolor_2(f_id);
          }

          /* wrap vertices that belong with base of edge */
          if ( web.modeltype == QUADRATIC )
          { if ( !inverted(e_id) )
             { verts = get_coord(get_edge_midv(e_id));
               (*sym_wrap)(verts,gdata[FACET_VERTS+i].x,wrap);
             }
             gdata[FACET_VERTS+i].color = gdata[i].color;
             gdata[FACET_VERTS+i].backcolor = gdata[i].backcolor;
             gdata[FACET_VERTS+i].v_id = get_edge_midv(e_id);
          }

          wrap = (*sym_compose)(wrap,get_fe_wrap(fe));
          if ( web.modeltype == QUADRATIC && inverted(e_id) )
          { verts = get_coord(get_edge_midv(e_id));
            (*sym_wrap)(verts,gdata[FACET_VERTS+i].x,wrap);
          }
          fe = get_next_edge(fe);
        }

        if ( show_expr[FACET] && show_expr[FACET]->start )
          if ( !eval(show_expr[FACET],NULL,f_id,NULL) ) 
            gdata[0].color = gdata[0].backcolor = UNSHOWN; /* maybe do edges */


        /* do inner clipping, if called for */
        if ( inner_clip_flag )
        {
          for ( i = 0 ; i < FACET_VERTS ; i++ )
          { REAL dist = 0.0;
            REAL *x = get_coord(web.zoom_v);

            for ( j = 0 ; j < SDIM ; j++ )
              dist += (x[j]-gdata[i].x[j])*(x[j]-gdata[i].x[j]);
    
            if ( sqrt(dist) > inner_clip_rad ) break; /* it's a keeper */
          }
             if ( i == FACET_VERTS ) continue; /* entirely inside */
        }

        if ( gdata[0].color != gdata[0].backcolor )
        { /* need normal for separation */ 
          REAL dd;
          if ( web.modeltype == LAGRANGE )
            vnormal(gdata[0].x,gdata[web.lagrange_order].x,gdata[ctrlpts-1].x,
                          gdata[0].norm);
          else vnormal(gdata[0].x,gdata[1].x,gdata[2].x,gdata[0].norm);
          dd = sqrt(SDIM_dot(gdata[0].norm,gdata[0].norm));
          for ( i = 0 ; i < SDIM ; i++ )
          gdata[0].norm[i] /= dd;
        }

        fe = get_facet_fe(f_id);
        if ( normflag || thickenflag  )
         { fe = get_facet_fe(f_id);
           for ( i = 0 ; i < FACET_VERTS ; i++ )
           { calc_vertex_smooth_normal(get_fe_tailv(fe),fe,gdata[i].norm);
             fe = get_next_edge(fe);
           }
         }
  
        /* check for special edges */
        fe_id = get_facet_fe(f_id);
        for ( i = 0 ; i < 3 ; i++, fe_id = get_next_edge(fe_id) )
          { int eattr;
             gdata[i].etype = INVISIBLE_EDGE; /* default */
             gdata[i].id = e_id = get_fe_edge(fe_id);
             gdata[i].ecolor = get_edge_color_2(e_id);
             eattr = get_eattr(e_id);
             if ( get_edge_color(e_id) == CLEAR ) continue;
             if ( eattr & BOUNDARY ) gdata[i].etype |= BOUNDARY_EDGE;
             if ( equal_id(get_next_facet(fe_id),fe_id) ) /* valence 1 */
                gdata[i].etype |= SINGLE_EDGE;
             else if  ( !equal_id(get_next_facet(fe_id),get_prev_facet(fe_id)) )
                gdata[i].etype |= TRIPLE_EDGE; /* triple line at least */
             if ( (eattr & HIT_WALL) && !(fattr & CONSTRAINT) )
                gdata[i].etype |= CONSTRAINT_EDGE;
             if ( (eattr & FIXED) && !(fattr & FIXED) )
                gdata[i].etype |= FIXED_EDGE;
             if ( show_expr[EDGE] && show_expr[EDGE]->start )
             { if ( eval(show_expr[EDGE],NULL,e_id,NULL) ) 
                  gdata[i].etype |= REGULAR_EDGE;
               else gdata[i].etype = INVISIBLE_EDGE;
             }

           }
        /* call device-specific routine */
        if ( web.modeltype == LINEAR )
        { /* call option-specific routine */
          gdata->flags |= LABEL_FACET;
          option_facet(gdata,f_id);         
        }
        else if ( web.modeltype == QUADRATIC )
        /* plot as four subtriangles */
        { 
          struct graphdata qdata[FACET_VERTS+1];
          memcpy((char*)qdata,(char*)gdata,3*sizeof(struct graphdata));
          qdata[0].flags = 0;

          for( j = 0 ; j < SDIM ; j++ )
          { qdata[1].x[j] = gdata[3].x[j];
            qdata[2].x[j] = gdata[5].x[j];
          }
          qdata[1].etype=INVISIBLE_EDGE;
          qdata[1].v_id = gdata[3].v_id;
          qdata[2].v_id = gdata[5].v_id;
          qdata[0].flags = 0;
          option_facet(qdata,f_id);

          for(j=0;j<SDIM;j++)
          { qdata[0].x[j]=gdata[3].x[j];qdata[1].x[j]=gdata[1].x[j];
            qdata[2].x[j]=gdata[4].x[j];
          }
          qdata[1].etype=gdata[1].etype;
          qdata[2].etype=INVISIBLE_EDGE;
          qdata[0].v_id = gdata[3].v_id;
          qdata[1].v_id = gdata[1].v_id;
          qdata[2].v_id = gdata[4].v_id;
          qdata[0].flags = 0;
          option_facet(qdata,f_id);

          for(j=0;j<SDIM;j++)
          { qdata[0].x[j]=gdata[5].x[j];qdata[1].x[j]=gdata[4].x[j];
            qdata[2].x[j]=gdata[2].x[j];
          }
          qdata[2].etype=gdata[2].etype;
          qdata[0].etype=INVISIBLE_EDGE;
          qdata[0].v_id = gdata[5].v_id;
          qdata[1].v_id = gdata[4].v_id;
          qdata[2].v_id = gdata[2].v_id;
          qdata[0].flags = 0;
          option_facet(qdata,f_id);

          for(j=0;j<SDIM;j++)
          { qdata[0].x[j]=gdata[3].x[j];qdata[1].x[j]=gdata[4].x[j];
            qdata[2].x[j]=gdata[5].x[j];
          }
          qdata[0].etype=INVISIBLE_EDGE;
          qdata[1].etype=INVISIBLE_EDGE;
          qdata[2].etype=INVISIBLE_EDGE;
          qdata[0].v_id = gdata[3].v_id;
          qdata[1].v_id = gdata[4].v_id;
          qdata[2].v_id = gdata[5].v_id;
          qdata[0].flags = LABEL_FACET;
          option_facet(qdata,f_id);
        }
        else if ( web.modeltype == LAGRANGE )
        { int ii,jj;
          /* plot as subtriangles */
          int flags = (gdata[0].flags|LIST_FACET) & ~LABEL_FACET;
          struct graphdata qdata[FACET_VERTS+1];
          memcpy((char*)qdata,(char*)gdata,3*sizeof(struct graphdata));
          for ( i = 0 ; i < FACET_VERTS ; i++ ) qdata[i].etype = INVISIBLE_EDGE;
          for ( j = 0,jj=0,ii=web.lagrange_order+1 ; 
                            j < web.lagrange_order ; j++,jj++ )
            for ( i = 0 ; i < web.lagrange_order-j ; i++,jj++,ii++ )
             { 
                for ( k = 0 ; k < 3 ; k++ )
                { qdata[0].x[k]=gdata[jj].x[k];
                  qdata[1].x[k]=gdata[jj+1].x[k];
                  qdata[2].x[k]=gdata[ii].x[k];
                }
                qdata[0].v_id = gdata[jj].v_id;
                qdata[1].v_id = gdata[jj+1].v_id;
                qdata[2].v_id = gdata[ii].v_id;
                qdata[0].etype = (j==0) ? gdata[0].etype : INVISIBLE_EDGE;
                qdata[1].etype = (i+j+1==web.lagrange_order) ? gdata[1].etype : INVISIBLE_EDGE;
                qdata[2].etype = (i==0) ? gdata[2].etype : INVISIBLE_EDGE;
                qdata[0].flags = flags;
                option_facet(qdata,f_id);

                if ( i < web.lagrange_order-j-1 )
                { int n;
                  for ( k = 0 ; k < 3 ; k++ )
                  { qdata[0].x[k]=gdata[jj+1].x[k];
                     qdata[1].x[k]=gdata[ii+1].x[k];
                     qdata[2].x[k]=gdata[ii].x[k];
                  }
                  qdata[0].v_id = gdata[jj+1].v_id;
                  qdata[1].v_id = gdata[ii+1].v_id;
                  qdata[2].v_id = gdata[ii].v_id;
                  qdata[0].flags = flags |
              (i==web.lagrange_order/3 && j==web.lagrange_order/3 ? LABEL_FACET:0);
                  for ( n = 0 ; n < FACET_VERTS ; n++ ) 
                     qdata[n].etype = INVISIBLE_EDGE;
                  option_facet(qdata,f_id);
                }
             }
        }

     }  /* end facets */
      
    }  /* end bodies */

    temp_free((char *)allfaces);
    temp_free((char *)gdata);
    temp_free((char *)bodystarts);
    free_matrix(xxx);
} /* end torus_bodies() */


/**********************************************************************
*
*  Function: torus_cells()
*
*  Purpose:  To display STRING torus model in terms of connected
*                cells. Assumes everything in x-y plane.  Graphs edges
*                around each facet in a connected manner.
*/

void torus_cells()
{
  facet_id f_id;
  struct graphdata gdata[2];
  REAL *x;
  facetedge_id fe_id,fe;
  int i; 
  WRAPTYPE wrap = 0;
  edge_id e_id;

  FOR_ALL_FACETS(f_id)
  {
    if ( show_expr[FACET] && show_expr[FACET]->start )
      if ( !eval(show_expr[FACET],NULL,f_id,NULL) ) continue;
    
    if ( breakflag ) break;
    fe_id = get_facet_fe(f_id);
    x = get_coord(get_fe_tailv(fe_id));
    for ( i = 0 ; i < SDIM ; i++ )
      gdata[0].x[i] = x[i];
    fe = fe_id;
    do
    { /* follow edges, keeping track of wrap */
      if ( !valid_id(fe) ) break;
      gdata[0].id = e_id = get_fe_edge(fe);
      gdata[0].color = gdata[0].ecolor = get_edge_color_2(gdata[0].id);
      gdata[0].etype = REGULAR_EDGE;
      gdata[0].flags &= ~(LABEL_EDGE|LABEL_HEAD|LABEL_TAIL);

      if ( web.modeltype == LINEAR )
      {
        wrap = (*sym_compose)(wrap,get_fe_wrap(fe)); 
      }
      else if ( web.modeltype == QUADRATIC )
      { gdata[1] = gdata[0];
        if ( inverted(e_id) )
           wrap = (*sym_compose)(wrap,get_fe_wrap(fe)); 
        x = get_coord(get_fe_midv(fe));
        (*sym_wrap)(x,gdata[1].x,wrap); 
        gdata[0].flags &= LABEL_TAIL;
        (*graph_edge_transforms)(gdata,gdata[0].id);
        gdata[0] = gdata[1];
        if ( !inverted(e_id) )
           wrap = (*sym_compose)(wrap,get_fe_wrap(fe)); 
      }
      else if ( web.modeltype == LAGRANGE )
      { int k;
        vertex_id *v = get_edge_vertices(e_id);
        gdata[1] = gdata[0];
        if ( inverted(e_id) )
        { wrap = (*sym_compose)(wrap,get_fe_wrap(fe)); 
          for ( k = web.lagrange_order - 2; k > 0 ; k-- )
          { x = get_coord(v[k]);
            (*sym_wrap)(x,gdata[1].x,wrap);
             if ( k == web.lagrange_order/2 ) 
                 gdata[0].flags |= LABEL_EDGE;
             if ( k == web.lagrange_order - 2 )
                 gdata[0].flags |= LABEL_HEAD;
             (*graph_edge_transforms)(gdata,gdata[0].id);
             gdata[0] = gdata[1];
          }
        }
        else
        {
          for ( k = 1;  k < web.lagrange_order - 1; k++ )
          { x = get_coord(v[k]);
            (*sym_wrap)(x,gdata[1].x,wrap);
            if ( k == web.lagrange_order/2 ) 
                gdata[0].flags |= LABEL_EDGE;
            if ( k == web.lagrange_order - 2 )
                gdata[0].flags |= LABEL_HEAD;
            (*graph_edge_transforms)(gdata,gdata[0].id);
            gdata[0] = gdata[1];
          }
          wrap = (*sym_compose)(wrap,get_fe_wrap(fe)); 
        }
      }
      /* now the head vertex for everybody */
      x = get_coord(get_fe_headv(fe));
      (*sym_wrap)(x,gdata[1].x,wrap); 
      gdata[0].flags |= LABEL_EDGE|LABEL_TAIL;
      (*graph_edge_transforms)(gdata,gdata[0].id);
      gdata[0] = gdata[1];
      fe = get_next_edge(fe);
    }
    while ( !equal_id(fe,fe_id) );
  }
} /* end  torus_cells() */

