/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/*****************************************************************
*
*  File: skeleton.c
*
*  Purpose: Variables and functions for skeleton element handling
*/

#include "include.h"

/* non-inlined versions of inlinable functions */
#ifndef INLINE
#define INLINE
#include "inline.h"
#endif

/* extra attribute type sizes */
int attr_type_size[NUMATTRTYPES] = 
  {0,sizeof(REAL),sizeof(int),sizeof(unsigned long), sizeof(unsigned char),
    sizeof(unsigned int)};
char *attr_type_name[NUMATTRTYPES] = { NULL,"real","integer","ulong","uchar","uint"};

/* quadratic interpolation coefficients */
/* partials of interpolation polynomials at midpoints of patch edges */
/* control points are numbered 0 to 5 counterclockwise */
/* midpoints are numbered 0 to 2 counterclockwise after control pt 0 */
/* dip[midpoint][control point][partial]  */

REAL dip[FACET_EDGES][FACET_CTRL][2] = {
    { { -0.5, -0.5 }, { 0.0, -1.0 }, { 0.5, 0.0 }, { 0.0, 1.0 },
      { 0.0, -0.5 }, { 0.0, 1.0 } },
    { { 0.5, 0.5 }, { -1.0, -1.0 }, { 0.5, 0.0  }, { 1.0, 1.0 },
      { 0.0, 0.5 }, { -1.0, -1.0 } },
    { { -0.5, -0.5 }, { 1.0, 0.0 }, { -0.5, 0.0 }, { 1.0, 0.0 },
      { 0.0, 0.5 }, { -1.0, 0.0 } }
  };

int Ord(id)    /* useful in debugger */
element_id id;
{ return ordinal(id); }

void vloop ARGS((vertex_id));

void vloop(v_id) /* prints vertex edge loop; useful in debugger */
element_id v_id;
{ edge_id e_id,ee_id;
  int n = 0;
  e_id = get_vertex_edge(v_id);
  if ( !valid_id(e_id) ) 
     { puts("No valid edge on vertex."); return; }
  ee_id = e_id;
  do { printf("%d ",inverted(ee_id)?-(Ord(ee_id)+1): (Ord(ee_id)+1)); 
       ee_id = get_next_tail_edge(ee_id);
       if ( ++n > web.skel[EDGE].count )
       { puts("Unclosed loop."); break;}
     } while ( ee_id != e_id );
  printf("\n");
}


void set_facet_body(f_id,b_id)
facet_id f_id;
body_id  b_id;
{
  if ( web.skel[BODY].count == 0 ) return;
  if ( !valid_id(f_id) ) return;

  if ( !valid_id(get_body_fe(b_id)) )
    set_body_fe(b_id,get_facet_fe(f_id));

  if ( everything_quantities_flag )
  { 
     if ( inverted(f_id) )  /* back facet */
     { 
        body_id bb_id = FULONG(f_id,F_BODY_LIST_ATTR)[1];
        if ( equal_id(bb_id,b_id) ) return;
        if ( valid_id(bb_id) )
          unapply_method(f_id,get_body_volmeth(bb_id)); 
        FULONG(f_id,F_BODY_LIST_ATTR)[1] = b_id;
        if ( valid_id(b_id) )
          apply_method_num(f_id,get_body_volmeth(b_id)); 
     }
     else /* front facet */
     { 
        body_id bb_id = FULONG(f_id,F_BODY_LIST_ATTR)[0];
        if ( equal_id(bb_id,b_id) ) return;
        if ( valid_id(bb_id) )
          unapply_method(f_id,get_body_volmeth(bb_id)); 
        FULONG(f_id,F_BODY_LIST_ATTR)[0] = b_id;
        if ( valid_id(b_id) )
          apply_method_num(f_id,get_body_volmeth(b_id)); 
     }
  }
  else
  {
     if ( inverted(f_id) )  FULONG(f_id,F_BODY_LIST_ATTR)[1] = b_id;
     else  FULONG(f_id,F_BODY_LIST_ATTR)[0] = b_id;
  }

}
/**************************************************************************
*
* Function: set_facet_fe()
*
* Purpose: Link facet to facetedge.
*
*/
void set_facet_fe(f_id,fe)
facet_id f_id;
facetedge_id fe;
{
  if ( inverted(f_id) ) { invert(fe); invert(f_id); }
  fptr(f_id)->fe_id = fe;
  if ( web.representation == STRING )
  { body_id b_id = get_facet_body(f_id);
    if ( valid_id(b_id) )
       set_body_fe(b_id,fe);
    b_id = get_facet_body(inverse_id(f_id));
    if ( valid_id(b_id) )
       set_body_fe(b_id,inverse_id(fe));
  }
}


REAL get_vertex_length_star(v_id) vertex_id v_id;
{ edge_id e_id = get_vertex_edge(v_id);
  edge_id firste = e_id;
  REAL star = 0.0;
  if ( get_vattr(v_id) & (Q_MIDPOINT|Q_MIDEDGE) ) return get_edge_length(e_id);
  if ( !valid_id(e_id) ) return 0.0;
  do { star += get_edge_length(e_id); e_id = get_next_tail_edge(e_id);}
  while ( !equal_element(e_id,firste) );
  return star;
}

REAL get_vertex_area_star(v_id) vertex_id v_id;
{ 
  REAL star = 0.0;

  if ( web.representation == STRING ) 
      return get_vertex_length_star(v_id);
  else
  {
     facet_id f_id = get_vertex_facet(v_id);
     facet_id firstf = f_id;
     if ( !valid_id(f_id) ) return 0.0;
     if ( vfacet_timestamp < top_timestamp ) make_vfacet_lists();
     if ( get_vattr(v_id) & Q_MIDPOINT )
     { facetedge_id fe,start_fe;
        fe = start_fe = get_vertex_fe(v_id);
        do 
        { star += get_facet_area(get_fe_facet(fe));
          fe = get_next_facet(fe);
        } while ( !equal_id(fe,start_fe) );
     }
     else if ( get_vattr(v_id) & Q_MIDFACET )
        star = get_facet_area(get_vertex_facet(v_id));
     else do 
      { star += get_facet_area(f_id); f_id = get_next_vertex_facet(v_id,f_id);}
     while ( !equal_element(f_id,firstf) );
  } 
  return star;
}

int get_vertex_fvalence(v_id) vertex_id v_id;
{ 
 int valence = 0;
 facet_id f_id = get_vertex_facet(v_id);
 facet_id firstf = f_id;

 if ( !valid_id(f_id) ) return 0;
 if ( vfacet_timestamp < top_timestamp ) make_vfacet_lists();
 if ( get_vattr(v_id) & Q_MIDPOINT )
 { facetedge_id fe,start_fe;
   fe = start_fe = get_vertex_fe(v_id);
   do 
   { valence++;
     fe = get_next_facet(fe);
   } while ( !equal_id(fe,start_fe) );
 }
 else if ( get_vattr(v_id) & Q_MIDFACET )
    valence = 1;
 else do 
  { valence++; f_id = get_next_vertex_facet(v_id,f_id);}
 while ( !equal_element(f_id,firstf) );
 return valence;
}

facet_id get_next_vertex_facet(v_id,f_id) vertex_id v_id; facet_id f_id;
{ int i; 
  vertex_id *v = get_facet_vertices(f_id);
  facet_id *f = FULONG(f_id,F_NEXT_VFACET_ATTR);
  int vmax = web.skel[FACET].ctrlpts; 

  for ( i = 0 ; i < vmax ; i++ )
     if (equal_id(v_id,v[i]))  return f[i] ; 
  sprintf(errmsg,"Internal error: get_next_vertex_facet failure v %d f %d\n",ordinal(v_id)+1,
     ordinal(f_id)+1);
  kb_error(1306,errmsg,RECOVERABLE);

  return NULLID;
}


void set_next_vertex_facet(v_id,f_id,ff_id)
      vertex_id v_id; facet_id f_id,ff_id;
{ 
   vertex_id *v = get_facet_vertices(f_id);
   facet_id *f = FULONG(f_id,F_NEXT_VFACET_ATTR);
   int vmax = web.skel[FACET].ctrlpts; 
   int i;
   for ( i = 0 ; i < vmax ; i++ )
   { if (equal_id(v_id,v[i]))
     { f[i] = ff_id ; 
       break;
     }
   }
}

void set_next_body_facet(f_id,ff_id)  facet_id f_id,ff_id;
  { FULONG(f_id,F_NEXT_BFACET_ATTR)[inverted(f_id)?1:0] = (ff_id) ; }

facet_id get_next_body_facet(f_id) facet_id f_id;
{ return    FULONG(f_id,F_NEXT_BFACET_ATTR)[inverted(f_id)?1:0]; }

void set_body_volume(b_id,v)  body_id b_id; REAL v;
{ if ( !valid_id(b_id) ) return;
  bptr(b_id)->volume = v; 
  bptr(b_id)->abstotal = fabs(v); 
  bptr(b_id)->voltimestamp = global_timestamp;
  if ( web.representation == STRING)
  { facet_id f_id = get_fe_facet(get_body_fe(b_id));
    if ( valid_id(f_id) ) set_facet_area(f_id,v);
  }
  if ( everything_quantities_flag )
  { struct gen_quant *q = GEN_QUANT(get_body_volquant(b_id));
    q->value = v;
  }
} 

void add_body_volume(b_id,v)  body_id b_id; REAL v;
{ struct body *b;
  if ( !valid_id(b_id) ) return;
  b = bptr(b_id);
  b->volume += v; 
  b->abstotal += fabs(v); 
  b->voltimestamp = global_timestamp;
  if ( web.representation == STRING)
  { facet_id f_id = get_fe_facet(b->fe_id);
    if ( valid_id(f_id) ) set_facet_area(f_id,b->volume);
  }
  if ( everything_quantities_flag )
  { struct gen_quant *q = GEN_QUANT(b->volquant);
    q->value = b->volume;
  }
} 


void set_body_fixvol(b_id,v) 
body_id b_id; 
REAL v;
{ if ( valid_id(b_id) )
  { bptr(b_id)->fixvol = (v); 
    if ( everything_quantities_flag )
    { struct gen_quant *q = GEN_QUANT(get_body_volquant(b_id));
      q->target = v;
      if ( !(q->flags & Q_FIXED) )
      { q->flags &= ~(Q_INFO|Q_ENERGY|Q_CONSERVED);
        q->flags |= Q_FIXED;
      }
      if ( web.pressure_flag )
      { q = GEN_QUANT(get_body_ambquant(b_id));
        q->flags &= ~(Q_INFO|Q_FIXED|Q_CONSERVED);
        q->flags |= Q_ENERGY;
      }
    }
  }
  else
  { sprintf(errmsg,"fix body volume: illegal body %d.\n",ordinal(b_id)+1);
    kb_error(1307,errmsg,RECOVERABLE);
  }
}

vertex_id new_vertex(x,parent)
REAL *x;
element_id parent; /* for inherited stuff */
{ 
  int i;
  vertex_id v_id;
  REAL *y;

  v_id = new_element(VERTEX,parent);
  if ( x )
  { y = get_coord(v_id);
    for ( i = 0 ; i < SDIM ; i++ ) y[i] = x[i];
  }
  set_vertex_edge(v_id,NULLID);
  set_vertex_facet(v_id,NULLID);

  set_v_global(v_id);

  /* interrupt conjugate gradient */
  if ( cg_hvector ) { myfree((char *)cg_hvector); cg_hvector = NULL; }

  return v_id;
}

vertex_id dup_vertex(old_v)
vertex_id old_v;
{ 
  vertex_id v_id;

  v_id = new_element(VERTEX,NULLID);
  memcpy((char *)&(elptr(v_id)->attr),(char *)&(elptr(old_v)->attr),
              web.sizes[VERTEX] - 2*sizeof(element_id));
  elptr(v_id)->self_id = v_id;  /* restore new id */
  vptr(v_id)->e_id = NULLID;
  return v_id;
}

edge_id    new_edge(tail_id,head_id,parent)
vertex_id tail_id,head_id;
element_id parent; /* for inherited stuff */
{ 
  edge_id e_id;
  vertex_id v_id;
  REAL *x,*h,*t;
  int i,k;

  e_id = new_element(EDGE,parent);
  set_edge_fe(e_id,NULLFACETEDGE);
  set_edge_color(e_id,DEFAULT_EDGE_COLOR);
  if ( valid_id(tail_id) && valid_id(head_id) )
  { set_edge_tailv(e_id,tail_id);
     set_edge_headv(e_id,head_id);
     if ( (web.modeltype == QUADRATIC) && valid_id(head_id) )
     { /* quadratic version; linear interpolation of midpoint */
        v_id = new_element(VERTEX,parent);
        set_edge_midv(e_id,v_id);
        h = get_coord(head_id);
        t = get_coord(tail_id);
        x = get_coord(v_id);
        for ( i = 0 ; i < SDIM ; i++ )
          x[i] = (h[i] + t[i])/2.0;
     }
     else if ( (web.modeltype == LAGRANGE) && valid_id(head_id) )
     { /* Lagrange version; linear interpolation of points */
        vertex_id *v = get_edge_vertices(e_id);
        h = get_coord(head_id);
        t = get_coord(tail_id);
        for ( k = 1 ; k < web.lagrange_order ; k++ )
        { v[k] = new_element(VERTEX,parent);
          set_attr(v[k],Q_MIDEDGE);
          set_vertex_edge(v[k],e_id);
          x = get_coord(v[k]);
          for ( i = 0 ; i < SDIM ; i++ )
             x[i] = (k*h[i] + (web.lagrange_order-k)*t[i])/web.lagrange_order;
        }
     }
  }
  return e_id;
}

edge_id dup_edge(old_e)
edge_id old_e;
{ 
  edge_id e_id;
  vertex_id newmid;

  e_id = new_element(EDGE,NULLID);
  memcpy((char *)&(elptr(e_id)->attr),(char *)&(elptr(old_e)->attr),
              web.sizes[EDGE] - 2*sizeof(element_id));
  elptr(e_id)->self_id = e_id; /* restore new id */
  eptr(e_id)->next_vedge[0] = NULLID;
  eptr(e_id)->next_vedge[1] = NULLID;
  if ( web.modeltype == QUADRATIC )
  { newmid = dup_vertex(get_edge_midv(old_e));
    set_edge_midv(e_id,newmid);
  } 
  else if ( web.modeltype == LAGRANGE )
  { int i;
    vertex_id *v = get_edge_vertices(e_id);
    vertex_id *oldv = get_edge_vertices(old_e);
    for ( i = 1 ; i < web.lagrange_order ; i++ )
       v[i] = dup_vertex(oldv[i]);
  } 
  if ( inverted(old_e) ) return inverse_id(e_id);
  return e_id;
}

void recalc_facet_area(f_id)
facet_id f_id;
{
  if ( everything_quantities_flag )
    quantity_attribute(f_id,default_area_quant_num);
  else
    (*calc_facet_energy)(f_id,AREA_ONLY);
}

facet_id  new_facet ()
{ 
  facet_id f_id;

  f_id = new_element(FACET,NULLID);
  set_facet_body(f_id,NULLBODY);
  set_facet_body(inverse_id(f_id),NULLBODY);
  set_facet_fe(f_id,NULLFACETEDGE);
  set_facet_color(f_id,DEFAULT_FACET_COLOR);
  set_facet_density(f_id,1.0);
  return f_id;
}

facet_id dup_facet(old_f)
facet_id old_f;
{ 
  facet_id f_id;

  f_id = new_element(FACET,NULLID);
  memcpy((char *)&(elptr(f_id)->attr),(char *)&(elptr(old_f)->attr),
              web.sizes[FACET] - 2*sizeof(element_id));
  set_attr(f_id,NEWELEMENT);
  elptr(f_id)->self_id = f_id; /* restore new id */
  return f_id;
}

body_id new_body()
{ int two = 2;
  body_id b_id;

  expand_attribute(FACET,F_BODY_LIST_ATTR,&two);
  b_id = new_element(BODY,NULLID);
  set_body_fe(b_id,NULLFACETEDGE);
  if ( everything_quantities_flag ) 
      convert_new_body_to_quantity(b_id);
  web.bodycount++;
  return b_id;
}

body_id dup_body(old_b)
body_id old_b;
{ 
  body_id b_id;

  b_id = new_element(BODY,NULLID);
  memcpy((char *)&(elptr(b_id)->attr),(char *)&(elptr(old_b)->attr),
              web.sizes[BODY] - 2*sizeof(element_id));
  elptr(b_id)->self_id = b_id; /* restore new id */
  web.bodycount++;
  return b_id;
}

facetedge_id new_facetedge(f_id,e_id)
facet_id f_id;
edge_id e_id;
{ 
  facetedge_id fe_id;
  vertex_id headv,tailv;

  fe_id = new_element(FACETEDGE,NULLID);
  set_fe_edge(fe_id,e_id);
  set_fe_facet(fe_id,f_id);
  set_prev_edge(fe_id,NULLFACETEDGE);
  set_next_edge(fe_id,NULLFACETEDGE);
  set_prev_facet(fe_id,NULLFACETEDGE);      
  set_prev_facet(fe_id,NULLFACETEDGE);
  tailv = get_edge_tailv(e_id);
  if ( !valid_id(get_vertex_edge(tailv)) )
    set_vertex_edge(tailv,e_id);
  headv = get_edge_headv(e_id);
  if ( !valid_id(get_vertex_edge(headv)) )
    set_vertex_edge(headv,e_id);

  if ( web.representation==STRING && everything_quantities_flag )
  { /* attach volume quantities */
     body_id b_id;
     b_id = get_facet_body(f_id);
     if ( valid_id(b_id) )
     { if ( same_sign(f_id,e_id) )
          apply_method_num(e_id,get_body_volmeth(b_id));
        else
          apply_method_num(inverse_id(e_id),get_body_volmeth(b_id));
     }
     b_id = get_facet_body(inverse_id(f_id));
     if ( valid_id(b_id) )
     { if ( same_sign(f_id,e_id) )
          apply_method_num(inverse_id(e_id),get_body_volmeth(b_id));
        else
          apply_method_num(e_id,get_body_volmeth(b_id));
     }
  }
  return fe_id;
}

REAL get_edge_length(e_id)
edge_id e_id;
{
  return    (eptr(e_id)->length);
}


REAL get_facet_pressure(f_id)
facet_id f_id;
{ 
  return  (get_body_pressure(get_facet_body(f_id)) - 
        get_body_pressure(get_facet_body(facet_inverse(f_id))));
}


/********************************************************************
*
* Function: compare_edge_attr(ea,eb)
* 
* Purpose: See if edge eb can be merged with edge ea.
*          Used in eliminate_edge().  Checks constraints
*          and named methods. Those of edge eb must be
*          a subset of those of edge ea.
*
* Return value: 0  Cannot
*               1  Can
*/

int compare_edge_attr(ea,eb)
edge_id ea,eb;
{ int i,j;
  conmap_t * con1,*con2;
  int meth_offset = get_meth_offset(EDGE); 
  struct edge *ea_ptr,*eb_ptr;

  /* check constraints */
  con1 = get_e_constraint_map(ea);
  con2 = get_e_constraint_map(eb);

  if ( con2[0] > con1[0] )
  { if ( verbose_flag ) 
    { sprintf(msg,"Edge %d has more constraints than edge %d.\n",
      ordinal(eb)+1,ordinal(ea)+1);
     outstring(msg);
    }
    return 0;
  }
  for ( i = 1 ; i <= (int)con2[0] ; i++ )
  { for ( j = 1 ; j <= (int)con1[0] ; j++ )
      if ( (con1[j]&CONMASK) == (con2[i]&CONMASK) ) break;
    if ( j > (int)con1[0] )
    { if ( verbose_flag )
      { sprintf(msg,"Edge %d is on constraint %s, but edge %d isn't.\n",
        ordinal(eb)+1,get_constraint((int)con2[i])->name,ordinal(ea)+1);
        outstring(msg);
      }          
      return 0;
    }
  }

  /* check methods */
  ea_ptr = eptr(ea);
  eb_ptr = eptr(eb);
  for ( i = 0 ; i < (int)eb_ptr->method_count ; i++ )
  { int m;
    m = abs(((int*)((char*)eb_ptr+meth_offset))[i]);
    for ( j = 0 ; j < (int)ea_ptr->method_count ; j++ )
    { int n;
      n = abs(((int*)((char*)ea_ptr+meth_offset))[j]);
    if ( n == m ) break;
    }
    if ( j >= (int)ea_ptr->method_count )
    {  if ( verbose_flag )
       { sprintf(msg,"Edge %d is on method %s but edge %d isn't.\n",
           ordinal(eb)+1,METH_INSTANCE(m)->name,ordinal(ea)+1);
         outstring(msg);
       }       
       return 0;
    }
  }

  return 1;
}

/**************************************************************
*
*  Function:  equal_constr()
*
*  Purpose:    See if two elements have the same set of constraints.
*
*/

int equal_constr(id1,id2)
element_id id1,id2;
{ int i,j;
  conmap_t * con1=NULL,*con2=NULL;

  switch ( id_type(id1) )
     {
        case VERTEX: 
                         con1        = get_v_constraint_map(id1);
                         break;

        case EDGE  : 
                         con1        = get_e_constraint_map(id1);
                         break;

        case FACET : 
                         con1        = get_f_constraint_map(id1);
                         break;
     }


  switch ( id_type(id2) )
     {
        case VERTEX: 
                         con2        = get_v_constraint_map(id2);
                         break;

        case EDGE  :
                         con2        = get_e_constraint_map(id2);
                         break;

        case FACET :
                         con2        = get_f_constraint_map(id2);
                         break;
     }
  if ( con2[0] != con1[0] ) return 0;
  for ( i = 1 ; i <= (int)con1[0] ; i++ )
  { for ( j = 1 ; j <= (int)con2[0] ; j++ )
        if ( (con1[i]&CONMASK) == (con2[j]&CONMASK) ) break;
     if ( j > (int)con2[0] ) return 0;
  }
  return 1;
}

/*************************************************************************
*
*  function: add_attribute()
*
*  purpose: add extra attribute to an element type
*
*  return:  index of attribute
*/

int add_attribute(e_type,name,attr_type,dim,dims,dumpflag,code)
int e_type; /* VERTEX ,... */
char *name;
int attr_type; /* REAL_ATTR or INTEGER_ATTR or ULONG_ATTR */
int dim; /* number of dimensions, 0 for scalar */
int *dims; /* sizes of dimensions, NULL for all sizes 0 */
            /* Note: scalar still needs size of 0 or 1 */ 
int dumpflag; /* whether appears in dump file */
struct expnode *code; /* nonnull for function attribute */
{ int newsize,newcount;
  struct extra *ex;
  int oldsize;

  if ( web.skel[e_type].extra_count >= web.skel[e_type].maxextra-1 )
  { web.skel[e_type].dy_extras = 
       dy_realloc(web.skel[e_type].dy_extras,
       (web.skel[e_type].maxextra+10),sizeof(struct extra));
    web.skel[e_type].maxextra += 10;
  }

  /* expand space */ 
  /* get old used space, minus any padding, and pad to proper size */
  if ( web.skel[e_type].extra_count > 0 )
  { ex = EXTRAS(e_type) + web.skel[e_type].extra_count-1;
    oldsize = ex->offset + ex->datacount*attr_type_size[ex->type];
  }
  else oldsize = web.sizes[e_type];
  if  (oldsize % attr_type_size[attr_type])
    oldsize += attr_type_size[attr_type] - (oldsize % attr_type_size[attr_type]);

  ex = EXTRAS(e_type) + web.skel[e_type].extra_count;

  if ( dim == 0 ) newcount = dims ? dims[0] : 0;
  else if ( dim > MAXEXTRADIM )
  { sprintf(errmsg,"Extra attribute \"%s\" has %d dimensions, exceeding limit of %d.\n", name,dim,MAXEXTRADIM);
    kb_error(2510,errmsg,RECOVERABLE);
  }
  else if ( dims == NULL ) newcount = 0;
  else
  { int i;
    for ( i = 0, newcount = 1 ; i < dim ; i++ )
    { newcount *= dims[i];
      ex->sizes[i] = dims[i];
    }
  }
  if ( (newcount > 0) || (oldsize > web.sizes[e_type]) )
  { newsize =  newcount*attr_type_size[attr_type];
    expand(e_type,oldsize + newsize); 
  }
  strncpy(ex->name,name,ATTR_NAME_SIZE);
  ex->type = attr_type;
  switch ( ex->type )
  { case INTEGER_ATTR: ex->itemsize = sizeof(int); break;
    case REAL_ATTR: ex->itemsize = sizeof(REAL); break;
    case ULONG_ATTR: ex->itemsize = sizeof(unsigned long); break;
    case UINT_ATTR: ex->itemsize = sizeof(unsigned int); break;
    default:
     sprintf(errmsg,"Internal: Unexpected attribute type: %d\n",ex->type);
     kb_error(2831,errmsg,WARNING);
     ex->itemsize = sizeof (REAL);
  }
 
  ex->offset = oldsize;
  ex->datacount = newcount;
  ex->adim = dim;
  if ( dim > 0 ) ex->flags |= DIMENSIONED_ATTR;
  if ( code ) ex->code = *code;
  if ( dumpflag ) ex->flags |= DUMP_ATTR;

  if ( stricmp(name,EXTRA_BDRY_NAME) == 0 )
  { extra_bdry_attr = web.skel[e_type].extra_count;
    if ( ex->type != INTEGER_ATTR )
      kb_error(2842,"Attribute extra_boundary must be of type integer.\n",
         RECOVERABLE);
    if ( e_type != VERTEX )
      kb_error(2843,"Attribute extra_boundary should be vertex attribute.\n",
         RECOVERABLE);
  }
  else if ( stricmp(name,EXTRA_BDRY_PARAM_NAME) == 0 )
  { extra_bdry_param_attr = web.skel[e_type].extra_count;
    if ( ex->type != REAL_ATTR )
      kb_error(2844,"Attribute extra_boundary_param must be of type real.\n",
         RECOVERABLE);
    if ( e_type != VERTEX )
      kb_error(2845,
        "Attribute extra_boundary_param should be vertex attribute.\n",
         RECOVERABLE);
  }
  web.skel[e_type].extra_count++;
  return web.skel[e_type].extra_count - 1; /* index */
}


/*************************************************************************
*
*  function: expand_attribute()
*
*  purpose: enlarge space for attribute of an element type
*              or shrink it.
*/

void expand_attribute(e_type,attr_num,newsizes)
int e_type; /* VERTEX ,... */
int attr_num; /* number of attribute */
int *newsizes; /* new numbers of components, even for scalar */
{ int newsize;  
  int diff;  /* difference between old and new total sizes */
  struct extra *ex;
  int chunksize,offset,available,needed;
  char *spot;
  element_id id;
  int k,d,n,dsize,dest,inx,blocksize;
  char *temp=NULL; /* for shuffling higher dim entries */

  ex = EXTRAS(e_type) + attr_num;
  dsize = ex->itemsize;

  if ( ex->adim == 0 ) newsize = newsizes[0];
  else
    for ( newsize = 1, k = 0 ; k < ex->adim ; k++ )
       newsize *= newsizes[k];

  if ( (ex->adim <= 1) && (newsize == ex->datacount) ) 
     return;

  /* expand or contract space */
  /* see how much extra space is needed */
  if ( attr_num < web.skel[e_type].extra_count-1 )
         available = ex[1].offset - ex[0].offset;
  else available = web.sizes[e_type] - ex->offset;
  needed = newsize*attr_type_size[ex->type];
  if ( ex->adim >= 2 ) 
     temp = (char*)temp_calloc(needed,1);
  if ( needed > available )
  { /* expand */
    /* check alignment of following fields */
    diff = needed - available;
    for ( k = attr_num+1 ; k < web.skel[e_type].extra_count ; k++ )
      while ( diff % attr_type_size[EXTRAS(e_type)[k].type] )
        diff++;
    expand(e_type,web.sizes[e_type] + diff); 
    /* move stuff above */
    if ( attr_num < web.skel[e_type].extra_count-1 )
      offset = EXTRAS(e_type)[attr_num+1].offset;
    else offset = web.sizes[e_type] - diff;
    chunksize = web.sizes[e_type] - offset - diff;
    if ( chunksize || ( ex->adim >= 2) )
    { element_id sentinel;
      id = NULLID; 
      if ( web.skel[e_type].count > 0 ) 
        while ( generate_all(e_type,&id,&sentinel) )
        { 
          spot = (char*)elptr(id) + offset;
          kb_memmove(spot + diff,spot,chunksize);
          if ( ex->adim >= 2 )
          { /* entry shuffle via temp */
            char *old = (char*)elptr(id) + EXTRAS(e_type)[attr_num].offset;
            for ( n = 0 ; n < ex->datacount ; n++ )
            { /* figure out indices and new spot */
              int oldinx = n;
              for ( d = ex->adim-1, dest = 0, blocksize = 1 ; d >= 0 ; d-- )
              { inx = oldinx % ex->sizes[d];
                if ( inx >= newsizes[d] ) goto skipentry; 
                dest += blocksize*inx;
                blocksize *= newsizes[d];
                oldinx = oldinx/ex->sizes[d];
              }
              kb_memmove(temp+dest*dsize,old+n*dsize,dsize);
skipentry:    ;
            }
            kb_memmove(old,temp,needed);
          }
          else
            memset(spot,0,diff);
        }
    }
    for ( k = attr_num+1 ; k < web.skel[e_type].extra_count ; k++ )
         EXTRAS(e_type)[k].offset += diff;
  }
  else if ( needed < available )
  { /* maybe shrink */
     /* check alignment of following fields */
     diff = available - needed;
     for ( k = attr_num+1 ; k < web.skel[e_type].extra_count ; k++ )
        while ( diff % attr_type_size[EXTRAS(e_type)[k].type] )
          diff--;
     /* move stuff above */
     if ( attr_num < web.skel[e_type].extra_count-1 )
       offset = EXTRAS(e_type)[attr_num+1].offset;
     else offset = web.sizes[e_type];
     chunksize = web.sizes[e_type] - offset;
     if ( chunksize || (ex->adim >= 2) )
     { element_id sentinel;
       id = NULLID;
       if ( web.skel[e_type].count > 0 ) 
        while ( generate_all(e_type,&id,&sentinel) )
        {
          if ( ex->adim >= 2 )
          { /* entry shuffle via temp */
            char *old = (char*)elptr(id)+EXTRAS(e_type)[attr_num].offset;
            for ( n = 0 ; n < ex->datacount ; n++ )
            { /* figure out indices and new spot */
              int oldinx = n;
              for ( d = ex->adim-1, dest = 0, blocksize = 1 ; d >= 0 ; d-- )
              { inx = oldinx % ex->sizes[d];
                if ( inx >= newsizes[d] ) goto skipentry2; 
                dest += blocksize*inx;
                blocksize *= newsizes[d];
                oldinx = oldinx/ex->sizes[d];
              }
              kb_memmove(temp+dest*dsize,old+n*dsize,dsize);
skipentry2:    ;
            }
            kb_memmove(old,temp,needed);
          }
          spot = (char*)elptr(id) + offset;
          kb_memmove(spot - diff,spot,chunksize);
        }
     }
     for ( k = attr_num+1 ; k < web.skel[e_type].extra_count ; k++ )
         EXTRAS(e_type)[k].offset -= diff;
     expand(e_type,web.sizes[e_type] - diff); 
  }
  if ( ex->adim >= 2 ) temp_free(temp);
  ex->datacount = newsize;
  for ( n = 0 ; n < ex->adim ; n++ )
	ex->sizes[n] = newsizes[n];
  parallel_update_flag[e_type] = 1;
}

/*************************************************************************
*
*  function: find_attribute()
*
*  purpose: find extra attribute by name, if it exists.
*  return: index number if found, -1 if not.
*/

int find_attribute(etype,name)
int etype;
char *name;
{ struct extra *ex;
  int n;
  ex = EXTRAS(etype);
  for ( n = 0 ; n < web.skel[etype].extra_count ; n++,ex++ )
    if ( stricmp(ex->name,name) == 0 ) break;
  if ( n == web.skel[etype].extra_count )
     return -1;
  return n;
}
/**************************************************************************
*
* function: find_extra()
*
* purpose: return index of named attribute, searching all element types.
*             return -1 if not found.
*/
int find_extra(name,etype)
char *name;
int *etype; /* for returning element type */
{ int el_type,qnum,n;
  struct extra *ex;

  for ( el_type = VERTEX, qnum = -1 ; el_type <= FACETEDGE ; el_type++ )
     { ex = EXTRAS(el_type);
        for ( n = 0 ; n < web.skel[el_type].extra_count ; n++,ex++ )
          if ( stricmp(ex->name,name) == 0 )
             {*etype = el_type;qnum = n;break;}
     }
  return qnum;
}

/***************************************************************************
  Constraint handling routines
****************************************************************************/

/* Methodology:
    Constraint map is array of conmap_t.
    First entry is number of constraints.
    Then follow constraint numbers, with high bit CON_HIT_BIT set
      if constraint is hit.
    Allocated as an extra attribute if needed.
*/

conmap_t nullcon[2]; /* default empty list */

void set_v_global(v_id)
vertex_id v_id;
{ int k;
  for ( k = 0 ; k < web.con_global_count ; k++ )
     set_v_constraint_map(v_id,web.con_global_map[k]);
}

void set_v_conmap(v_id,map) 
vertex_id v_id;
conmap_t *map;
{ int k, m=(int)map[0];
  for ( k = 1 ; k <= m ; k++ )
    set_v_constraint_map(v_id,map[k]);
  map = get_v_constraint_map(v_id);
  if ( map[0] == 0 ) unset_attr(v_id,CONSTRAINT);
}

void set_e_conmap(e_id,map)  
edge_id e_id;
conmap_t *map;
{ int k, m=(int)map[0];
  for ( k = 1 ; k <= m ; k++ )
    set_e_constraint_map(e_id,map[k]);
  map = get_e_constraint_map(e_id);
  if ( map[0] == 0 ) unset_attr(e_id,CONSTRAINT);
}

void set_f_conmap(f_id,map)  
facet_id f_id;
conmap_t *map;
{ int k, m=(int)map[0];
  for ( k = 1 ; k <= m ; k++ )
    set_f_constraint_map(f_id,map[k]);
  map = get_f_constraint_map(f_id);
  if ( map[0] == 0 ) unset_attr(f_id,CONSTRAINT);
}

void set_v_constraint_map(v_id,n)  
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(VERTEX)[V_CONSTR_LIST_ATTR].datacount;
  struct constraint *constr;
  int k;
  int four = 4;

  n &= CONMASK;
  if ( maxcon == 0 )
     expand_attribute(VERTEX,V_CONSTR_LIST_ATTR,&four);
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
     if ( (map[k] & CONMASK) == ((conmap_t)n & CONMASK) ) return;
  if ( k >= maxcon )
  { int newmax = maxcon+4;
    expand_attribute(VERTEX,V_CONSTR_LIST_ATTR,&newmax);
    map = get_v_constraint_map(v_id);
  }  
  constr = get_constraint(n);
  map[k] = (conmap_t)n; 
  if ( !(constr->attr & (NONPOSITIVE|NONNEGATIVE) )) 
    map[k] |= CON_HIT_BIT;
  map[0]++; /* counter */

  set_attr(v_id,CONSTRAINT);
  if ( (constr->attr & CON_ENERGY) && (web.representation == STRING) )
  { set_attr(v_id, BDRY_ENERGY);
     if ( everything_quantities_flag )
        apply_method_num(v_id,constr->energy_method);
  }
  if ( (constr->attr & CON_CONTENT) && (web.representation == STRING) )
  { set_attr(v_id, BDRY_CONTENT);
    if ( everything_quantities_flag )
    { edge_id e_id,first_e;
     first_e = e_id = get_vertex_edge(v_id);
     if ( valid_id(e_id) && !(get_eattr(e_id) & NONCONTENT) )
     do
     { char name[100];
       body_id b_id;
       facetedge_id first_fe,fe;
       first_fe = fe = get_edge_fe(e_id);
       if ( valid_id(fe) )  do
       { facet_id f_id = get_fe_facet(fe);
         fe = get_next_facet(fe);
         if ( !valid_id(f_id) ) continue;
         b_id = get_facet_body(f_id);
         sprintf(name,"body_%d_con_%d_meth",ordinal(b_id)+1,n);
         if ( valid_id(b_id) )
         { attach_method(get_body_volquant(b_id),name);
           apply_method(inverse_id(v_id),name);
         }
         b_id = get_facet_body(inverse_id(f_id));
         sprintf(name,"body_%d_con_%d_meth",ordinal(b_id)+1,n);
         if ( valid_id(b_id) )
         { attach_method(get_body_volquant(b_id),name);
           apply_method(v_id,name);
         }
       } while ( !equal_id(fe,first_fe) );
       e_id = get_next_tail_edge(e_id);
     } while ( !equal_id(first_e,e_id));
   }
  }
}

void unset_v_constraint_map(v_id,n)    
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(VERTEX)[V_CONSTR_LIST_ATTR].datacount;
  int k,j;

  n &= CONMASK;
  if ( maxcon == 0 ) return;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
     if ( (map[k] & CONMASK) == (conmap_t)n ) break;
  if ( k > (int)*map ) return;
  map[0]--;
  for ( j = k ; j <= (int)*map ; j++ ) map[j] = map[j+1];
  if ( map[0] == 0 ) unset_attr(v_id,CONSTRAINT);
  if ( everything_quantities_flag )
  { struct constraint *con = get_constraint(n);
    int i;
    if ( con->attr & CON_ENERGY )
            unapply_method(v_id,con->energy_method);
    if ( con->attr & CON_CONTENT )
    { int meth_offset = get_meth_offset(VERTEX);
      int *instlist = (int*)((char*)elptr(v_id) + meth_offset);
      int mcount = elptr(v_id)->method_count;
      for ( i = 0 ; i < mcount ; i++ )
        if ( METH_INSTANCE(abs(instlist[i]))->flags & BODY_INSTANCE )
           unapply_method(v_id,instlist[i]);
    }
  }
}

int v_on_constraint(v_id,n)  
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(VERTEX)[V_CONSTR_LIST_ATTR].datacount;
  int k;

  n &= CONMASK;
  if ( maxcon == 0 ) return 0;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
        return 1;
  }
  return 0;
}

int v_hit_constraint_count(v_id)  
vertex_id v_id;
{ conmap_t *map;
  int maxcon = EXTRAS(VERTEX)[V_CONSTR_LIST_ATTR].datacount;
  int count = 0;
  int k;

  if ( maxcon == 0 ) return 0;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CON_HIT_BIT) ) 
      count++;
  }
  return count;
}

void get_v_common_conmap(v1,v2,conmap)
vertex_id v1,v2;
conmap_t *conmap;
{ conmap_t *map1 = get_v_constraint_map(v1);
  unsigned int k;

  conmap[0] = 0;
  for ( k = 1; k <= map1[0] ; k++ )
     if ( v_on_constraint(v2,map1[k]) )
        conmap[++conmap[0]] = map1[k] & (conmap_t)CONMASK;
}

int get_v_constraint_status(v_id,n)  
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(VERTEX)[V_CONSTR_LIST_ATTR].datacount;
  int k;

  n &= CONMASK;
  if ( maxcon == 0 ) return 0;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
        return (map[k] & CON_HIT_BIT) ? 1 : 0;
  }
  return 0;
}

void clear_v_conmap(v_id)
vertex_id v_id;
{ conmap_t *map = get_v_constraint_map(v_id);
  map[0] = 0;
}

void set_v_constraint_status(v_id,n)  
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(VERTEX)[V_CONSTR_LIST_ATTR].datacount;

  int k;
  n &= CONMASK;
  if ( maxcon == 0 ) return;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
    { map[k] |= CON_HIT_BIT;
      set_attr(v_id,HIT_WALL);
      return;
    }
  }
}

void unset_v_constraint_status(v_id,n) 
vertex_id v_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(VERTEX)[V_CONSTR_LIST_ATTR].datacount;

  int k;
  n &= CONMASK;
  if ( maxcon == 0 ) return;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
     { map[k] &= ~CON_HIT_BIT;
        return;
     }
  }
}

void clear_v_constraint_status(v_id) 
vertex_id v_id;
{ conmap_t *map;
  int maxcon = EXTRAS(VERTEX)[V_CONSTR_LIST_ATTR].datacount;

  int k;
  if ( maxcon == 0 ) return;
  map = get_v_constraint_map(v_id);
  for ( k = 1; k <= (int)*map ; k++ )
      map[k] &= ~CON_HIT_BIT;
}

void set_e_constraint_map(e_id,n)  
edge_id e_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(EDGE)[E_CONSTR_LIST_ATTR].datacount;
  int k;
  struct constraint *constr;
  int four = 4;

  n &= CONMASK;
  if ( maxcon == 0 )
     expand_attribute(EDGE,E_CONSTR_LIST_ATTR,&four);
  map = get_e_constraint_map(e_id);
  for ( k = 1; k <= (int)*map ; k++ )
     if ( (map[k] & CONMASK) == (conmap_t)n ) return;
  if ( k >= maxcon )
  { int newmax = maxcon+4;
    expand_attribute(EDGE,E_CONSTR_LIST_ATTR,&newmax);
    map = get_e_constraint_map(e_id);
  } 
  map[k] = (conmap_t)n; 
  map[0]++; /* counter */

  set_attr(e_id,CONSTRAINT);
  constr = get_constraint(n);
  if ( constr->attr & CON_ENERGY )
  { set_attr(e_id, BDRY_ENERGY);
     if ( everything_quantities_flag )
        apply_method_num(e_id,constr->energy_method);
  }
  if ( constr->attr & CON_CONTENT )
  { set_attr(e_id, BDRY_CONTENT);  /* BIG PROBLEM HERE GETTING RIGHT BODY!!! */
    if ( everything_quantities_flag )
    { facetedge_id fe,first_fe = get_edge_fe(e_id);
      fe = first_fe;
      do
      { char name[100];
        body_id b_id;
        facet_id f_id;
        f_id = get_fe_facet(fe);
        fe = get_next_facet(fe);
        if ( !valid_id(f_id) ) continue;
        if ( get_fattr(f_id) & NONCONTENT ) continue;
        b_id = get_facet_body(f_id);
        sprintf(name,"body_%d_con_%d_meth",ordinal(b_id)+1,n);
        if ( valid_id(b_id) )
        { attach_method(get_body_volquant(b_id),name);
          apply_method(e_id,name);
        }
        b_id = get_facet_body(inverse_id(f_id));
        sprintf(name,"body_%d_con_%d_meth",ordinal(b_id)+1,n);
        if ( valid_id(b_id) )
        { attach_method(get_body_volquant(b_id),name);
          apply_method(inverse_id(e_id),name);
        }
      } while ( !equal_id(fe,first_fe) );

    }
  }
}

int e_on_constraint(e_id,n)  
edge_id e_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(EDGE)[E_CONSTR_LIST_ATTR].datacount;
  int k;

  n &= CONMASK;
  if ( maxcon == 0 ) return 0;
  map = get_e_constraint_map(e_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
        return 1;
  }
  return 0;
}

void unset_e_constraint_map(e_id,n) 
edge_id e_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(EDGE)[E_CONSTR_LIST_ATTR].datacount;
  int j,k;

  n &= CONMASK;
  if ( maxcon == 0 ) return;
  map = get_e_constraint_map(e_id);
  for ( k = 1; k <= (int)*map ; k++ )
    if ( (map[k] & CONMASK) == (conmap_t)n ) break;
  if ( k > (int)*map ) return;

  map[0]--;
  for ( j = k ; j <= (int)*map ; j++ ) map[j] = map[j+1];
  if ( map[0] == 0 ) unset_attr(e_id,CONSTRAINT);

  if ( everything_quantities_flag )
  { struct constraint *con = get_constraint(n);
    if ( con->attr & CON_ENERGY )
            unapply_method(e_id,con->energy_method);
    if ( con->attr & CON_CONTENT )
    { int meth_offset = get_meth_offset(EDGE);
      int i;
      int *instlist = (int*)((char*)elptr(e_id) + meth_offset);
      int mcount = elptr(e_id)->method_count;
      for ( i = 0 ; i < mcount ; i++ )
        if ( METH_INSTANCE(abs(instlist[i]))->flags & BODY_INSTANCE )
           unapply_method(e_id,instlist[i]);
    }
  }

}

void set_f_constraint_map(f_id,n)  
facet_id f_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(FACET)[F_CONSTR_LIST_ATTR].datacount;
  int k;
  int four = 4;

  n &= CONMASK;  /* get rid of hit bit */
  if ( maxcon == 0 )
     expand_attribute(FACET,F_CONSTR_LIST_ATTR,&four);
  map = get_f_constraint_map(f_id);
  for ( k = 1; k <= (int)*map ; k++ )
     if ( (map[k] & CONMASK) == (conmap_t)n ) return;
  if ( k >= maxcon )
  { int newmax = maxcon+4;
    expand_attribute(FACET,F_CONSTR_LIST_ATTR,&newmax);
    map = get_f_constraint_map(f_id);
  } 
  map[k] = (conmap_t)n; 
  map[0]++; /* counter */
}

int f_on_constraint(f_id,n)  
facet_id f_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(FACET)[F_CONSTR_LIST_ATTR].datacount;
  int k;

  n &= CONMASK;
  if ( maxcon == 0 ) return 0;
  map = get_f_constraint_map(f_id);
  for ( k = 1; k <= (int)*map ; k++ )
  { if ( (map[k] & CONMASK) == (conmap_t)n ) 
        return 1;
  }
  return 0;
}

void unset_f_constraint_map(f_id,n)    
facet_id f_id;
int n;
{ conmap_t *map;
  int maxcon = EXTRAS(FACET)[F_CONSTR_LIST_ATTR].datacount;
  int k,j;

  n &= CONMASK;
  if ( maxcon == 0 ) return;
  map = get_f_constraint_map(f_id);
  for ( k = 1; k <= (int)*map ; k++ )
     if ( (map[k] & CONMASK) == (conmap_t)n ) break;
  if ( k > (int)*map ) return;
  map[0]--;
  for ( j = k ; j <= (int)*map ; j++ ) map[j] = map[j+1];
  if ( map[0] == 0 ) unset_attr(f_id,CONSTRAINT);
  if ( everything_quantities_flag )
  { struct constraint *con = get_constraint(n);
     if ( con->attr & CON_ENERGY )
            unapply_method(f_id,con->energy_method);
     if ( con->attr & CON_CONTENT )
         { unapply_method(f_id,con->content_method);
         }
  }
}

int get_tag(f_id)
facet_id f_id;
{ if ( EXTRAS(FACET)[F_TAG_ATTR].datacount > 0 )
     return (*FINT(f_id,F_TAG_ATTR));
  else return 0;
}

void set_tag(f_id,t)
facet_id f_id;
int t;
{ int one=1;
  if ( EXTRAS(FACET)[F_TAG_ATTR].datacount == 0 )
     expand_attribute(FACET,F_TAG_ATTR,&one);
  *FINT(f_id,F_TAG_ATTR) = t;
}
