/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/********************************************************************
*
*  File: storage.c
*
*  Purpose:  File defining details of storage implementation.
*            All machine-dependent gory details should be here
*            (purely private details) or in storage.h
*            (for inclusion in other source files that need to know).
*                
*        This version has element ids implemented as longs.
*/
#ifdef WIN32
#include <windows.h>
#endif

#include "include.h"


struct blocklist_struct *blocklist[NUMELEMENTS]; /* list of allocated blocks */
int blockcount[NUMELEMENTS];  /* how many blocks allocated */
int blockmax[NUMELEMENTS];  /* length of blocklist */

/* individual indirect block pointer arrays, handy for debugging */
/* set to web.skel[].ibase */
INDIRECT_TYPE *vibase;
INDIRECT_TYPE *eibase;
INDIRECT_TYPE *fibase;
INDIRECT_TYPE *bibase;
INDIRECT_TYPE *feibase;

/* Starting web structure */
struct webstruct web = {
{ { VERTEX, 1, 0,  NULL, 0, NULLID, NULLID, 0, NULLID },
 { EDGE  , 2, 0,  NULL, 0, NULLID, NULLID, 0, NULLID },
 { FACET , 3, 0,  NULL, 0, NULLID, NULLID, 0, NULLID },
 { BODY  , 0, 0,   NULL, 0, NULLID, NULLID, 0, NULLID },
 { FACETEDGE, 0,0, NULL, 0, NULLID, NULLID, 0, NULLID } },
 {0,0,0,0,0},{0,0,0,0,0},3,2,LINEAR
     };


element_id
  NULLVERTEX  =  VERTEX  << TYPESHIFT, 
  NULLEDGE     =  EDGE  << TYPESHIFT,
  NULLFACET    =  FACET  << TYPESHIFT,
  NULLBODY     =  BODY  << TYPESHIFT,  
  NULLFACETEDGE =  (unsigned long)FACETEDGE << TYPESHIFT;


struct element *elptr(id)
element_id id;
{
  int type = id_type(id);
  return (struct element *)(web.skel[type].ibase[id & OFFSETMASK]);
}

int oid(id)
element_id id;
{ return inverted(id) ? -(ordinal(id)+1) : (ordinal(id) +1) ;
}
/* handy for debugging */
struct vertex * Vptr ARGS((element_id)); 
struct edge * Eptr ARGS((element_id));
struct facet * Fptr ARGS((element_id));
struct body * Bptr ARGS((element_id)); 
struct facetedge * Feptr ARGS((element_id));

struct vertex * Vptr(id) vertex_id id; { return vptr(id); }
struct edge * Eptr(id) edge_id id; { return eptr(id); }
struct facet * Fptr(id) facet_id id; { return fptr(id); }
struct body * Bptr(id) body_id id; { return bptr(id); }
struct facetedge * Feptr(id) facetedge_id id; { return feptr(id); }

/***********************************************************************
*
*  function: expand()
*
*  purpose:  Increase size of element structures.  Works only for
*                indexed or indirect id. Will also decrease size.
*/
void expand(type,newsize)
int type;  /* VERTEX, etc. */
int newsize; /* new size of structure */
{ char *newblock;
  int oldsize = web.sizes[type];
  int count = web.skel[type].maxcount;
  int i,n;                                   
  char *newptr,*oldptr;

  /* round up newsize to alignment for doubles */
  newsize = ((newsize+sizeof(REAL)-1)/sizeof(REAL))*sizeof(REAL);
  if ( newsize == web.sizes[type] ) return; /* don't have to expand */
  web.sizes[type] = newsize;
  if ( count == 0 ) return; /* don't have to reallocate space */

  ENTER_GRAPH_MUTEX;
  /* Don't mess with structures while graph thread using them */

  for ( n = 0 ; n < blockcount[type] ; n++ )
  { INDIRECT_TYPE *iptr;
     char *spot;
     struct blocklist_struct *b = blocklist[type] + n;
     int copysize;

     newblock = mycalloc(b->count,newsize);
     copysize = (newsize < oldsize) ? newsize : oldsize;
     for ( i = 0, oldptr = (char*)(b->blockptr), newptr = newblock; 
             i < b->count ; i++, oldptr += oldsize, newptr += newsize )
         memcpy(newptr,oldptr,copysize);
     myfree((char*)b->blockptr);
     b->blockptr = (struct element*)newblock;
     /* update indirect pointers */
     /* keeping in mind structures not necessarily in id order 
        due to reorder command */
     for ( i=0,spot=newblock, iptr=web.skel[type].ibase+b->start_ord ; 
                i<b->count ; i++,spot+=newsize,iptr++ )
        web.skel[type].ibase[ordinal(((struct element *)spot)->self_id)] = 
               (struct element *)spot;
  }

  LEAVE_GRAPH_MUTEX;

  parallel_update_flag[type] = 1;
} 

/***********************************************************************
*
*  function: extend()
*
*  purpose: allocate more empty element structures
*/

void extend(type,mode)
int type;
int mode; /* EXTEND_BATCH or EXTEND_FOR_REFINE */
{
  int number;  /* of new items to allocate */
  char *newblock;
  INDIRECT_TYPE *newiblock;
  element_id id,oldfreehead;
  struct element *newptr = NULL;
  int k;
  int allocsize;
  long  oldnum = web.skel[type].maxcount;
  long  newnum;
  int neword;

  ENTER_GRAPH_MUTEX;
  /* Don't mess with structures while graph thread using them */

  if ( blockcount[type] >= blockmax[type] )
  { blocklist[type] = (struct blocklist_struct*)
      kb_realloc((char*)(blocklist[type]),
         (blockcount[type]+BATCHSIZE)*sizeof(struct blocklist_struct),
         blockcount[type]*sizeof(struct blocklist_struct));
     blockmax[type] += BATCHSIZE;
  }
  if ( mode == EXTEND_BATCH )
  {
    /* calculate number of structures to fit in block size just under 2^n */
    allocsize = BATCHSIZE*web.sizes[type];
    k = 0x100 ; while ( k < allocsize ) k <<= 1 ;
    number = (k-16)/web.sizes[type]; /* maybe room for block header */
    newnum = web.skel[type].maxcount + number;
  }
  else if ( mode == EXTEND_FOR_REFINE )
  { /* increase by 2^surface_dimension factor */
    number = web.skel[type].count*(1<<web.dimension)
                - web.skel[type].maxcount + 100;
    newnum = web.skel[type].maxcount + number;
    if ( number <= 0 ) { LEAVE_GRAPH_MUTEX; return; }/* don't need any more */
  }
  else kb_error(2474,"Internal error: bad mode for extend().\n",RECOVERABLE);

  newblock = mycalloc(number,web.sizes[type]);
  blocklist[type][blockcount[type]].start_ord = oldnum ;
  blocklist[type][blockcount[type]].count = number ;
  blocklist[type][blockcount[type]++].blockptr = (struct element *)newblock;

  while ( newnum > web.skel[type].ialloc )
  {
    if ( web.skel[type].ibase == NULL )
    { newiblock = (INDIRECT_TYPE*)mycalloc(number,sizeof(INDIRECT_TYPE));
      web.skel[type].ialloc = number;
    }
    else
    { newiblock = (INDIRECT_TYPE*)kb_realloc((char*)(web.skel[type].ibase),
        2*web.skel[type].ialloc*sizeof(INDIRECT_TYPE),
         web.skel[type].ialloc*sizeof(INDIRECT_TYPE));
      web.skel[type].ialloc *= 2;
    }
    web.skel[type].ibase = newiblock;
    switch(type)
    {    case VERTEX: vibase = newiblock; break;
         case EDGE:    eibase = newiblock; break;
         case FACET:  fibase = newiblock; break;
         case BODY:    bibase = newiblock; break;
         case FACETEDGE: feibase = newiblock; break;
    }
  }

  LEAVE_GRAPH_MUTEX;

  /* add to end of freelist */
  neword = web.skel[type].maxcount;  
  id = ((long)type << TYPESHIFT) | VALIDMASK | neword; 
  oldfreehead = web.skel[type].free;
  if ( valid_id(oldfreehead) )
  { element_id nextid = oldfreehead;
    while ( valid_id(elptr(nextid)->forechain) )
      nextid = elptr(nextid)->forechain;
    elptr(nextid)->forechain = id;
  }
  else web.skel[type].free = id;
  for ( k = 0 ; k < number ; k++ )
  { newptr = (struct element *)(newblock + k*web.sizes[type]); 
    web.skel[type].ibase[neword] = newptr;
    newptr->self_id = id;
    if ( k < number-1 )
    { neword++;
      id = ((long)type << TYPESHIFT) | VALIDMASK | neword; 
      newptr->forechain = id;
    }
  }

  web.skel[type].maxcount += number;

}

/************************************************************************
*
* function: new_element()
*
* purpose: Allocate new element from freelist
*
*/

element_id new_element(type,parent)
int type;
element_id parent; /* for inherited stuff */
{
  element_id newid;
  struct element *newptr,*last;
  ORDTYPE ord;
  
  newid = web.skel[type].free;
  
  if ( !valid_id(newid) ) /* free list empty */
  { if ( aggregate_depth == 0 )
    { free_discards(DISCARDS_ALL);
      newid = web.skel[type].free;
    }
  }

  if ( !valid_id(newid) ) /* free list empty */
  { extend(type,EXTEND_BATCH);
    newid = web.skel[type].free;
  }
  newptr = elptr(newid);

  /* remove from free chain */
  web.skel[type].free = newptr->forechain;

  /* clean out old info */
  ord = ordinal(newid);
  memset((char *)newptr,0,web.sizes[type]);
  if ( ord > web.skel[type].max_ord ) web.skel[type].max_ord = ord;
  newptr->original = NO_ORIGINAL;


  if ( match_id_flag && datafile_flag )
  { /* link in numerical order */
     int i;
     struct element *prev,*next;
     for ( i = ord-1 ; i >= 0 ; i-- )
     { prev = web.skel[type].ibase[i];
        if ( prev->attr & ALLOCATED )
        { if ( valid_id(prev->forechain) )
          { next = elptr(prev->forechain);
            next->backchain = newid;
          }
          newptr->forechain = prev->forechain;
          prev->forechain = newid;
          newptr->backchain = prev->self_id;
          break;
        }
     }
     if ( i < 0 )  /* no predecessor */
     { newptr->forechain = web.skel[type].used;
       if ( valid_id(web.skel[type].used) )
       { next = elptr(web.skel[type].used);
         next->backchain = newid;
       }
       web.skel[type].used = newid;
     }
     if ( !valid_id(newptr->forechain) ) web.skel[type].last = newid;
  }
  else
  { /* add to end of in-use chain */
     newptr->forechain = NULLID;
     newptr->backchain = web.skel[type].last;
     if ( valid_id(web.skel[type].last) )
     {
        last = elptr(web.skel[type].last);     /* find end of in-use chain */
        last->forechain = newid;
     }
     else
     {        
        web.skel[type].used = newid;
     }
     web.skel[type].last = newid;
  }

  newptr->attr = ALLOCATED | NEWELEMENT;

  newptr->self_id = newid;
  web.skel[type].count++;  

  /* inherited named methods from parent */
  if ( valid_id(parent) )
  { struct element *e_ptr = elptr(parent);
     int *instlist = (int*)((char*)e_ptr + get_meth_offset(id_type(parent)));
     int i;
     for ( i = 0 ; i < (int)e_ptr->method_count ; i++ )
     { if ( METH_INSTANCE(abs(instlist[i]))->type <= type )
          apply_method_num(newid,instlist[i]);
     }
  }

  return newid;
}

void free_element(id) 
element_id id;
{
  struct element *ptr;
  int type = id_type(id);

  if ( !valid_id(id) )
  {
    sprintf(errmsg,
      "Internal error: Trying to free invalid element type %d id %d \n",
            type,ordinal(id)+1);
    kb_error(1311,errmsg,WARNING);
    return;
  }
     
  if ( type == EDGE ) /* remove from vertex lists */
  { vertex_id tailv = get_edge_tailv(id);
    vertex_id headv = get_edge_headv(id);
    int i; 
    remove_vertex_edge(tailv,id);
    remove_vertex_edge(headv,inverse_id(id));
    if ( web.modeltype == QUADRATIC )
             free_element(get_edge_midv(id));
    if ( web.modeltype == LAGRANGE )
    { vertex_id *v = get_edge_vertices(id);
      for ( i = 1 ; i < web.lagrange_order ; i++ )
          free_element(v[i]);
    }
  }
    
  if ( (type == FACET) && (web.representation==SIMPLEX) )
  { /* remove from vertex lists */
    int n,nmax = web.skel[FACET].ctrlpts;
    vertex_id *v = get_facet_vertices(id);
    for ( n = 0 ; n < nmax ; n++ )
      remove_vertex_facet(v[n],id);
  }
  ptr = elptr(id);
  if ( !(ptr->attr & ALLOCATED) )
  { sprintf(errmsg,
      "Internal error: Trying to free unallocated element type %d id %d \n",
            type,ordinal(id)+1);
    kb_error(1313,errmsg,WARNING);
    return;
  }
  ptr->attr &= ~ALLOCATED;

#ifdef OLDDISCARD
  /* remove from in-use list */
  if ( valid_id(ptr->forechain) )
     elptr(ptr->forechain)->backchain = ptr->backchain;
  else
     web.skel[type].last = ptr->backchain;

/**** DON'T DO THIS! Don't mess with forechains until free_discards()! */
  if ( valid_id(ptr->backchain) )
    {
      /*     elptr(ptr->backchain)->forechain = ptr->forechain;  */
    }
  else
     web.skel[type].used = ptr->forechain;
/*****/

  /* add to discard list */
  /* save forechain for generators */
  ptr->backchain = web.skel[type].discard;
  web.skel[type].discard = id & (TYPEMASK | VALIDMASK | OFFSETMASK);
                                 /* clear old bits, keep only needed */
#else
  web.skel[type].discard_count++;
#endif

  web.skel[type].count--; 

}

/* reclaim element from discard list */
void unfree_element(id)
element_id id;
{
  struct element *ptr;
  int type = id_type(id);

  if ( !valid_id(id) )
  { sprintf(errmsg,"Internal error: Trying to unfree invalid id %08lX \n",id);
    kb_error(1314,errmsg,RECOVERABLE);
  }
    
  ptr = elptr(id);
  if ( ptr->attr & ALLOCATED )
  { sprintf(errmsg,
     "Internal error: Trying to unfree allocated element id %08lX \n",id);
    kb_error(1315,errmsg,RECOVERABLE);
  }
  ptr->attr |= ALLOCATED;


  web.skel[type].discard_count--;

  web.skel[type].count++;  
}

/* index as id */
element_id get_ordinal_id(type,ord)
int type; /* type of element */
int ord;  /* ordinal of element */
{ element_id id;

  if ( (type < 0) || (type > NUMELEMENTS) ) return NULLID;
  if ( (ord < 0) || (ord > web.skel[type].max_ord)) return NULLID;
  id = (type << TYPESHIFT) | VALIDMASK | ord;
  if ( get_attr(id) & ALLOCATED ) return id;
  else return NULLID;
}

int generate_all(type,idptr,sentinel)  /* re-entrant */
int type;
element_id *idptr;
element_id *sentinel; /* to record original end of list */
{
  struct element *ptr;

  if ( !valid_id(*idptr) ) /* first time */
  { *idptr = web.skel[type].used;
    *sentinel = web.skel[type].last;   /* may be a discard */
    if ( !valid_id(*idptr) ) return 0;
  }
  else
  { ptr = elptr(*idptr);
    do
    { if ( equal_id(*idptr,*sentinel) ) return 0;
      *idptr = ptr->forechain;
      if ( !valid_id(*idptr) ) return 0;
      ptr = elptr(*idptr);
    }
    while ( !(ptr->attr & ALLOCATED) );
  }

  return 1;
}

void memory_report()
{
    long mem;
    int k;

    mem = 0;
    for ( k = 0 ; k < NUMELEMENTS ; k++ )
      mem += web.skel[k].count*web.sizes[k];

    sprintf(errmsg,
    "Vertices: %ld  Edges: %ld  Facets: %ld  Bodies: %ld  Facetedges: %ld\nElement memory: %ld  ",
	web.skel[0].count,web.skel[1].count,web.skel[2].count,
                web.skel[3].count,web.skel[4].count,
      mem);
    outstring(errmsg);

#if defined(_WIN32) && defined(_HEAPOK)
#ifndef _HEAPINFO
#define _HEAPINFO _heapinfo
#endif

    if ( _heapchk() != _HEAPOK )
      kb_error(1317,"Internal error: Corrupt heap! Memory is trashed.\n",WARNING);
    else if ( verbose_flag )
    { struct _HEAPINFO hinfo;
      int b_use=0,b_free=0;
      long mem_use=0,mem_free=0;
      char * heaptop = NULL; 
      char * heapstart = NULL;

      hinfo._pentry = NULL;
      while ( _heapwalk(&hinfo) == _HEAPOK )
      {   if ( heapstart == NULL )
             heapstart = (char*)hinfo._pentry; 
 
#ifdef HEAPLIST
         sprintf(errmsg,"%p %10ld %s end %p\n",hinfo._pentry,hinfo._size,
             hinfo._useflag ? "used" : "free",
                  (char*)hinfo._pentry+hinfo._size);
          outstring(errmsg);
#endif
       if (hinfo._useflag)
        {   
          b_use++; mem_use+= hinfo._size;
        }
        else { b_free++; mem_free += hinfo._size; }
        if ( (char*)hinfo._pentry + hinfo._size > heaptop )
          heaptop = (char*)hinfo._pentry + hinfo._size;
      }
      outstring("\n");
      sprintf(errmsg,"blocks in use: %d    memory in use: %ld \n",
                    b_use,mem_use);
      outstring(errmsg);
      sprintf(errmsg,"blocks free:    %d    memory free:    %ld \n",
                 b_free,mem_free);
      outstring(errmsg);
      sprintf(errmsg,"Heap top: %P\n",heaptop);
      outstring(errmsg);
      sprintf(errmsg,"Heap size: %4.2f MB\n",
             (heaptop-heapstart)/1024./1024.);
      outstring(errmsg);
    }
#endif

#if defined(_UNISTD_H)
  if ( verbose_flag )
  { /* do this only on unix systems with unistd.h */
    sprintf(msg,"\nTotal data memory arena %d\n",
      (char*)sbrk(0)-(char*)&evolver_version);
    outstring(msg); 
  }
#endif

#if defined(M_MXFAST) && defined(IRIS)
  if ( verbose_flag ) 
     { char *ptr;
        struct mallinfo m;
        /* using libmalloc.a for debugging */
        ptr = malloc(10);
        if ( ptr == NULL )
          erroutstring("Bad heap.\n");
        else myfree(ptr);
        m = mallinfo();
        sprintf(msg,"Arena %d     Ordblocks: %d    Orduse: %d     Ordfree: %d\n",
            m.arena,m.ordblks,m.uordblks,m.fordblks);
        outstring(msg);
        sprintf(msg,"Small blocks: %d Small use: %d Small free: %d\n",
            m.smblks,m.usmblks,m.fsmblks);
        outstring(msg);
}
#endif

#if defined(SUNXX)
    if ( memdebug )
      if ( malloc_verify() != 1 )
         kb_error(1318,"Internal error: Malloc_verify() failed.\n.",RECOVERABLE);

#endif

  mem_list_summary();
  dy_check();
}

/**************************************************************************
*
* function: reset_skeleton()
*
* purpose: Clean out old surface and initialize empty web.
*          Note all permanently allocated memory cleaned out en masse
*               previously.
*
*/

void reset_skeleton()
{
  int i;
  int type;
  int three = FACET_VERTS;
  int permcount = web.perm_global_count;
  int maxperm   = web.max_perm_globals;
  struct global *perm = perm_globals(0);

  gauss1Dpt = NULL; 
  gauss1Dwt = NULL;

  memset((char *)&web,0,sizeof(web));  /* total clean out */

  web.perm_global_count = permcount;
  web.max_perm_globals = maxperm;
  dy_perm_globals = perm;

  web.sizes[VERTEX] = sizeof(struct vertex);
  web.sizes[EDGE] = sizeof(struct edge);
  web.sizes[FACET] = sizeof(struct facet);
  web.sizes[BODY] = sizeof(struct body);
  web.sizes[FACETEDGE] = sizeof(struct facetedge);
  web.usedsizes[VERTEX] = sizeof(struct vertex);
  web.usedsizes[EDGE] = sizeof(struct edge);
  web.usedsizes[FACET] = sizeof(struct facet);
  web.usedsizes[BODY] = sizeof(struct body);
  web.usedsizes[FACETEDGE] = sizeof(struct facetedge);

  for ( i = 0 ; i < NUMELEMENTS ; i++ ) 
  { web.skel[i].max_ord = -1;
    blocklist[i] = NULL;
    blockmax[i] = blockcount[i] = 0;
  }
  vibase = NULL;
  eibase = NULL;
  fibase = NULL;
  bibase = NULL;
  feibase = NULL;

  web.skel[EDGE].dimension = 1;
  web.skel[FACET].dimension = 2;
  web.skel[BODY].dimension = 3;

  /* set up permanent attributes, empty to start with */
  /* have REAL attributes first, so can align on 8-byte */
  /* Be sure the order given here is same as order in skeleton.h */
  /* vertex */
  add_attribute(VERTEX,"x",REAL_ATTR,1,NULL,0,NULL);
  add_attribute(VERTEX,"__oldx",REAL_ATTR,1,NULL,0,NULL);
  add_attribute(VERTEX,"p",REAL_ATTR,1,NULL,0,NULL);
  add_attribute(VERTEX,"__force",REAL_ATTR,1,NULL,0,NULL);
  add_attribute(VERTEX,"__velocity",REAL_ATTR,1,NULL,0,NULL);
  add_attribute(VERTEX,"__v_boundary",INTEGER_ATTR,0,NULL,0,NULL);
  add_attribute(VERTEX,"star",INTEGER_ATTR,0,NULL,0,NULL);
  add_attribute(VERTEX,"__v_constraint_list",UINT_ATTR,1,NULL,0,NULL);
  /* edge */
  add_attribute(EDGE,"star",REAL_ATTR,0,NULL,0,NULL);
  add_attribute(EDGE,"density",REAL_ATTR,0,NULL,0,NULL);
  add_attribute(EDGE,"__e_vertices",ELEMENTID_ATTR,1,NULL,0,NULL);
  add_attribute(EDGE,"__e_boundary",INTEGER_ATTR,0,NULL,0,NULL);
  add_attribute(EDGE,"__wrap_list",INTEGER_ATTR,1,NULL,0,NULL);
  add_attribute(EDGE,"__e_constraint_list",UINT_ATTR,1,NULL,0,NULL);
  /* facet */
  add_attribute(FACET,"__f_constraint_list",UINT_ATTR,1,NULL,0,NULL);
  add_attribute(FACET,"__f_vertices",ELEMENTID_ATTR,1,&three,0,NULL);
  add_attribute(FACET,"__f_boundary",INTEGER_ATTR,0,NULL,0,NULL);
  add_attribute(FACET,"tag",INTEGER_ATTR,0,NULL,0,NULL);
  add_attribute(FACET,"__body_list",ELEMENTID_ATTR,1,NULL,0,NULL);
  add_attribute(FACET,"__next_vfacet_list",ELEMENTID_ATTR,0,NULL,0,NULL);
  add_attribute(FACET,"__next_bfacet_list",ELEMENTID_ATTR,0,NULL,0,NULL);
  add_attribute(FACET,"phase",INTEGER_ATTR,0,NULL,0,NULL);
  /* body */
  add_attribute(BODY,"__b_constraint_list",UINT_ATTR,1,NULL,0,NULL);

  /* all */
  for ( type = 0 ; type < NUMELEMENTS ; type++ )
     web.meth_attr[type] = 
        add_attribute(type,"__method_list",INTEGER_ATTR,1,NULL,0,NULL);
}

/***********************************************************************
*
* function: free_discards()
*
* purpose: totally free up discard list.  to be called only when
*          no lists are being used.
*/
void free_discards(mode)
int mode; /* DISCARDS_ALL or DISCARDS_SOME */
{ int type;

  for ( type = 0 ; type < NUMELEMENTS ; type++ )
  { element_id id,next_id;
    int small_potatoes = (mode==DISCARDS_ALL) ? 0 : web.skel[type].count/10; 
    if ( web.skel[type].discard_count <= small_potatoes )
       continue; /* don't bother with small potatoes */
    id = web.skel[type].used;
    while ( valid_id(id) )
    { struct element *ptr = elptr(id);
      next_id = ptr->forechain;
      if ( !(ptr->attr & ALLOCATED) )
      { /* move to free list */
        if ( valid_id(ptr->forechain) )
          elptr(ptr->forechain)->backchain = ptr->backchain;
        else web.skel[type].last = ptr->backchain;
        if ( valid_id(ptr->backchain) )
          elptr(ptr->backchain)->forechain = ptr->forechain;
        else web.skel[type].used = ptr->forechain;
        ptr->forechain = web.skel[type].free;
        ptr->backchain = NULLID;
        web.skel[type].free = id;
      }
      id = next_id;
    }
    web.skel[type].discard_count = 0;
  }

}

/**************************************************************************
*
* function: move_to_free_front()
*
* purpose: get particular id element to front of free list so id will 
*             match datafile number.  Called only during datafile.
*/

void move_to_free_front(type,id)
int type; /* element type */
int id;    /* element id    */
{ int ord = id - 1;
  element_id eid;
  struct element *pptr,*bbptr;
  element_id prev;

  if ( !match_id_flag ) return; /* for old way */
  while ( id > web.skel[type].maxcount ) extend(type,EXTEND_BATCH);
  /* linear search through free list */
  eid = web.skel[type].free;
  prev = NULLID;
  while ( valid_id(eid) )
  { if ( ordinal(eid) == ord ) break;
    prev = eid;
    eid = elptr(eid)->forechain;
  }
  if ( !valid_id(eid) )
  {  kb_error(2187,"Internal error: Cannot find element in free list.\n",
        DATAFILE_ERROR);
      return;
  }
  if ( eid == web.skel[type].free ) return; /* already in place */
  /* now cut and paste */
  bbptr = elptr(eid);
  pptr = elptr(prev);
  pptr->forechain = bbptr->forechain; /* cut */
  bbptr->forechain = web.skel[type].free; /* paste */
  web.skel[type].free = eid;
}

/*********************************************************************
* 
* Function: reorder_storage()
*
* Purpose: order storage of element structures according to value
* of element extra attribute order_key (real). Meant to order storage
* in cache and swap friendly way.  Invoked by command reorder_storage.
*
* Elements generators work through forechain pointer, so we must
* re-order storage and re-link forechain pointers.
**********************************************************************/

static int key_offset; /* offset of key in element structure */
static char * keynames[NUMELEMENTS] = {"vertex_order_key",
    "edge_order_key","facet_order_key","body_order_key",
    "facetedge_order_key"};
int esort ARGS((char*,char*));

int esort (a,b)  /* sort routine */
char *a, *b;
{ if ( *(REAL*)(a+key_offset) < *(REAL*)(b+key_offset) ) return -1;
  if ( *(REAL*)(a+key_offset) > *(REAL*)(b+key_offset) ) return  1;
  return 0;
}

void reorder_storage()
{ struct element *newblock[NUMELEMENTS];
  int i,j,n;

  free_discards(DISCARDS_ALL);

  /* allocate single-block space for each type  */
  for ( n = 0 ; n < NUMELEMENTS ; n++ )
    newblock[n] = (struct element *)mycalloc(web.skel[n].maxcount+1,
                          web.sizes[n]);
  
  /* copy element structures. */
  for ( n = 0 ; n < NUMELEMENTS ; n++ )
  { element_id id = web.skel[n].used;
    char *spot = (char*)(newblock[n]);
    while ( valid_id(id) )
    { struct element *ep = elptr(id);
      memcpy(spot,(char*)ep,web.sizes[n]);
      id = ep->forechain;
      spot += web.sizes[n];
    }
    /* free list at end */
    id = web.skel[n].free;
    while ( valid_id(id) )
    { struct element *ep = elptr(id);
      memcpy(spot,(char*)ep,web.sizes[n]);
      id = ep->forechain;
      spot += web.sizes[n];
    }
  }

  /* sort elements in use */
  for ( i = 0 ; i < NUMELEMENTS ; i++ )
  { struct extra *ex;
    int k;
     
    if ( web.skel[i].count == 0 ) continue;
    key_offset = -1; /* sentinel value */
    for ( k = 0, ex = EXTRAS(i) ; k < web.skel[i].extra_count ; k++ , ex++ )
      if ( stricmp(keynames[i], ex->name) == 0 )
      { key_offset = ex->offset;
        break;
      }
    if ( key_offset < 0 )
    { for ( n = 0 ; n < NUMELEMENTS ; n++ ) myfree((char*)newblock[n]);
      kb_error(2188,"reorder_storage: Key_order attribute not defined.\n",
         RECOVERABLE);
    }
    qsort((char*)newblock[i],web.skel[i].count,web.sizes[i],FCAST esort);
  }

  /* reset ibase array of pointers and list links */
  for ( i = 0 ; i < NUMELEMENTS ; i++ )
  { struct element *ep,*nextep;
    int k;

    if  ( web.skel[i].maxcount == 0 ) continue;
    
    if ( web.skel[i].count )
    {
      web.skel[i].used = newblock[i]->self_id;
      newblock[i]->backchain = NULLID;
      for ( k = 0, ep = newblock[i] ; k < web.skel[i].count-1 ; k++ )
      { nextep = (struct element *)((char*)ep + web.sizes[i]);
        ep->forechain = nextep->self_id;
        nextep->backchain = ep->self_id;
        web.skel[i].ibase[ordinal(ep->self_id)] = ep;
        ep = nextep;
      }
      web.skel[i].ibase[ordinal(ep->self_id)] = ep;
      ep->forechain = NULLID;
      web.skel[i].last = ep->self_id;
    }
    /* and free list */
    for (  k = web.skel[i].count ; k < web.skel[i].maxcount ; k++ )
    { ep = (struct element *)((char*)(newblock[i]) + k*web.sizes[i]);
      web.skel[i].ibase[ordinal(ep->self_id)] = ep;
    }
  }

  /* free old storage */
  for ( i = 0 ; i < NUMELEMENTS ; i++ )
  { for ( j = 0 ; j < blockcount[i] ; j++ )
          myfree((char *)blocklist[i][j].blockptr);
    blockmax[i] = blockcount[i] = 0;
  }

  /* establish new */
  for ( i = 0 ; i < NUMELEMENTS ; i++ )
  { if ( web.skel[i].count == 0 ) continue;
    blocklist[i][0].blockptr = newblock[i];
    blocklist[i][0].count = web.skel[i].maxcount;
    blockmax[i] = blockcount[i] = 1;
  }

  global_timestamp++;
  top_timestamp = global_timestamp;
}

/*************************************************************************
*
* Function: renumber_all()
*
* Purpose: Renumber elements according to linked list order.
*
*/

#define copy_sign(id1,id2)     (((id1)&~SIGNMASK) | ((id2) & SIGNMASK))
void renumber_all()
{ int type;
  struct element *ep;
  element_id id,newid;
  int k;
  int dim,off,dima,offa,dimb,offb,dimc,offc;
  struct element **newibase[NUMELEMENTS];

  free_discards(DISCARDS_ALL);

  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
     newibase[type] = 
         (struct element **)mycalloc(web.skel[type].ialloc,
                  sizeof(element_id *)); 

  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
  { element_id count;

    /* reset self-id of used elements  */
    count = 0;
    id = web.skel[type].used;
    while ( valid_id(id) )
    { ep = elptr(id);
      ep->self_id = (ep->self_id & (~OFFSETMASK)) | count;
      count++;
      id = ep->forechain;
    }

  }
 /* reset mutual links within structures */
  FOR_ALL_FACETEDGES(id)
  { struct facetedge *ep = feptr(id);
    ep->fe_edge_id = copy_sign(elptr(ep->fe_edge_id)->self_id,ep->fe_edge_id);
    ep->fe_facet_id = copy_sign(elptr(ep->fe_facet_id)->self_id,ep->fe_facet_id);
    ep->nextedge[0] = copy_sign(elptr(ep->nextedge[0])->self_id,ep->nextedge[0]);
    ep->nextedge[1] = copy_sign(elptr(ep->nextedge[1])->self_id,ep->nextedge[1]);
    ep->nextfacet[0] = copy_sign(elptr(ep->nextfacet[0])->self_id,ep->nextfacet[0]);
    ep->nextfacet[1] = copy_sign(elptr(ep->nextfacet[1])->self_id,ep->nextfacet[1]);
  }

  FOR_ALL_VERTICES(id)
  { struct vertex *ep = vptr(id);
    ep->e_id = copy_sign(elptr(ep->e_id)->self_id,ep->e_id);
  }

  off = EXTRAS(EDGE)[E_VERTICES_ATTR].offset;  /* endpoints */
  dim = EXTRAS(EDGE)[E_VERTICES_ATTR].datacount;  
  FOR_ALL_EDGES(id)    
  { struct edge *ep = eptr(id);
    vertex_id *vp;
    ep->fe_id = copy_sign(elptr(ep->fe_id)->self_id,ep->fe_id);
    ep->next_vedge[0] = copy_sign(elptr(ep->next_vedge[0])->self_id,ep->next_vedge[0]);
    ep->next_vedge[1] = copy_sign(elptr(ep->next_vedge[1])->self_id,ep->next_vedge[1]);
    vp = (vertex_id*)((char*)ep + off);
    for ( k = 0 ; k < dim ; k++ )
      vp[k] = vptr(vp[k])->self_id;
  }

  off = EXTRAS(FACET)[F_VERTICES_ATTR].offset;  /* endpoints */
  dim = EXTRAS(FACET)[F_VERTICES_ATTR].datacount;  
  offa = EXTRAS(FACET)[F_BODY_LIST_ATTR].offset;  /* bodies */
  dima = EXTRAS(FACET)[F_BODY_LIST_ATTR].datacount;  
  offb = EXTRAS(FACET)[F_NEXT_VFACET_ATTR].offset;  /* links */
  dimb = EXTRAS(FACET)[F_NEXT_VFACET_ATTR].datacount;  
  offc = EXTRAS(FACET)[F_NEXT_BFACET_ATTR].offset;  /* links */
  dimc = EXTRAS(FACET)[F_NEXT_BFACET_ATTR].datacount;  
  FOR_ALL_FACETS(id)  
  { struct facet *ep = fptr(id);
    element_id *p;
    ep->fe_id = copy_sign(elptr(ep->fe_id)->self_id,ep->fe_id);
    p = (element_id*)((char*)ep + off);
    for ( k = 0 ; k < dim ; k++ )
      if ( valid_id(p[k]) )
         p[k] = copy_sign(elptr(p[k])->self_id,p[k]);
    p = (element_id*)((char*)ep + offa);
    for ( k = 0 ; k < dima ; k++ )
      if ( valid_id(p[k]) )
         p[k] = copy_sign(elptr(p[k])->self_id,p[k]);
    p = (element_id*)((char*)ep + offb);
    for ( k = 0 ; k < dimb ; k++ )
      if ( valid_id(p[k]) )
         p[k] = copy_sign(elptr(p[k])->self_id,p[k]);
    p = (element_id*)((char*)ep + offc);
    for ( k = 0 ; k < dimc ; k++ )
      if ( valid_id(p[k]) )
         p[k] = copy_sign(elptr(p[k])->self_id,p[k]);
  }

  FOR_ALL_BODIES(id)    
  { struct body * ep = bptr(id);
     ep->fe_id = copy_sign(elptr(ep->fe_id)->self_id,ep->fe_id);
  }

  /* now new pointers in newibase */
  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
  {  int count = 0;
      if ( web.skel[type].count == 0 ) continue; 
      FOR_ALL_ELEMENTS(type,id)
          newibase[type][count++] = elptr(id);
  } 


  /* fix up links */
  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
  { if ( web.skel[type].count == 0 ) continue;
     if ( web.skel[type].count == 1 )
     { newibase[type][0]->forechain = NULLID;
        newibase[type][0]->backchain = NULLID;
        continue;
     }
     newibase[type][0]->forechain = newibase[type][0]->self_id+1;
     newibase[type][0]->backchain = NULLID;
     for ( k = 1 ; k < web.skel[type].count-1; k++ )
     {
        newibase[type][k]->forechain = newibase[type][k]->self_id+1;
        newibase[type][k]->backchain = newibase[type][k]->self_id-1;
     }
     newibase[type][k]->forechain = NULLID;
     newibase[type][k]->backchain = newibase[type][k]->self_id - 1;
     web.skel[type].used = newibase[type][0]->self_id;
     web.skel[type].last = newibase[type][k]->self_id;
  }

  /* fix up freelists */
  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
  { int count;
    count = web.skel[type].count;
    id = web.skel[type].free;
    if ( !valid_id(id) ) continue;
    newid = (id & (~OFFSETMASK)) | count;
    web.skel[type].free = newid;

    while ( valid_id(id) )
    { ep = elptr(id);
      id = ep->forechain;
      ep->forechain = ++newid;
      newibase[type][count] = ep;
      ep->self_id = (ep->self_id & (~OFFSETMASK)) | count;
      count++;
    } 
    ep->forechain = NULLID;
  }

  /* miscellaneous elements */
  if ( valid_id(web.zoom_v) ) web.zoom_v = vptr(web.zoom_v)->self_id;
  if ( pickvnum ) pickvnum = ordinal(vibase[pickvnum-1]->self_id)+1;
  if ( pickenum ) pickenum = ordinal(eibase[pickenum-1]->self_id)+1;
  if ( pickfnum ) pickfnum = ordinal(fibase[pickfnum-1]->self_id)+1;

  /* swap ibase to new */
  for ( type = VERTEX ; type <= FACETEDGE ; type++ )
  { myfree((char*)(web.skel[type].ibase));
    web.skel[type].ibase = newibase[type];
  }
  vibase = web.skel[VERTEX].ibase;
  eibase = web.skel[EDGE].ibase;
  fibase = web.skel[FACET].ibase;
  bibase = web.skel[BODY].ibase;
  feibase = web.skel[FACETEDGE].ibase;

  global_timestamp++;
  top_timestamp = global_timestamp;
  
}

/*************************************************************************
*
* function: interp_edge_attribute()
*
* purpose:  Find interpolated value of a vertex attribute at a Gauss point
*           on an edge.
*/

REAL interp_edge_attribute(eid,ext,inx,ptnum)
edge_id eid;
struct extra *ext;  /* extra attribute involved */
int inx;  /* index within attribute */
int ptnum;  /* which gauss point */
{
  int ctrl = web.skel[EDGE].ctrlpts;
  REAL sum = 0.0;
  int i;
  vertex_id *v = get_edge_vertices(eid);
  struct gauss_lag *gl = &gauss_lagrange[EDGE][web.gauss1D_order];

  if ( web.modeltype == QUADRATIC )
  { sum += gl->gpoly[ptnum][0]*get_extra_attrib_value(v[1],ext,inx);
    sum += gl->gpoly[ptnum][1]*get_extra_attrib_value(v[2],ext,inx);
    sum += gl->gpoly[ptnum][2]*get_extra_attrib_value(v[0],ext,inx);
  } else
  for ( i = 0 ; i < ctrl ; i++ )
    sum += gl->gpoly[ptnum][i]*get_extra_attrib_value(v[i],ext,inx);
  return sum;
}


/*************************************************************************
*
* function: interp_facet_attribute()
*
* purpose:  Find interpolated value of a vertex attribute at a Gauss point
*           on a facet.
*/

REAL interp_facet_attribute(fid,ext,inx,ptnum)
facet_id fid;
struct extra *ext;  /* extra attribute involved */
int inx;  /* index within attribute */
int ptnum;  /* which gauss point */
{
  int ctrl = web.skel[FACET].ctrlpts;
  REAL sum = 0.0;
  int i;
  vertex_id *v;
  vertex_id vv[2*FACET_VERTS];
  struct gauss_lag *gl = &gauss_lagrange[FACET][web.gauss2D_order];

  if ( web.modeltype == LINEAR )
  { facetedge_id fe = get_facet_fe(fid);
    for ( i = 0 ; i < FACET_VERTS ; i++ )
    { vv[i] = get_fe_tailv(fe);
      fe = get_next_edge(fe);
    }
    v = vv;
  } else if ( web.modeltype == QUADRATIC ) 
  { facetedge_id fe = get_facet_fe(fid);
    vv[0] = get_fe_tailv(fe);
    vv[1] = get_fe_midv(fe);
    vv[2] = get_fe_headv(fe);
    fe = get_next_edge(fe);
    vv[4] = get_fe_midv(fe);
    vv[5] = get_fe_headv(fe);
    fe = get_next_edge(fe);
    vv[3] = get_fe_midv(fe);
    v = vv;
  } else /* LAGRANGE */
  v = get_facet_vertices(fid);

  for ( i = 0 ; i < ctrl ; i++ )
    sum += gl->gpoly[ptnum][i]*get_extra_attrib_value(v[i],ext,inx);
  return sum;
}

/*************************************************************************
*
* function: get_extra_attrib_value()
*
* purpose: Return an extra attribute value as a real.
*
*/

REAL get_extra_attrib_value(id,ext,inx)
element_id id;
struct extra *ext;
int inx;
{
  if ( inx >= ext->datacount )
  { sprintf(errmsg,"Attribute %s total index is %d; maximum is %d.\n",
       ext->name,inx+1,ext->datacount);
    kb_error(1151,errmsg,RECOVERABLE);
  }
  if ( inx < 0 )
  { sprintf(errmsg,"Attribute %s index zero or negative: %d.\n",
    ext->name,inx+1);
  kb_error(2523,errmsg,RECOVERABLE);
  }

  switch ( ext->type )
  { case REAL_ATTR: return ((REAL*)get_extra_ptr(id,ext))[inx];
    case INTEGER_ATTR: return (REAL)(((int*)get_extra_ptr(id,ext))[inx]);
    case ULONG_ATTR: 
      return (REAL)(((unsigned long*)get_extra_ptr(id,ext))[inx]);
    case UCHAR_ATTR: 
      return (REAL)(((unsigned char*)get_extra_ptr(id,ext))[inx]);
  }
  return 0.0;  /* shouldn't get here. */
}

