/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/*****************************************************************
*
*  File: yexparse.c
*
*  Purpose: To read and parse user commands and functions in algebraic form
*          for evolver constraints.  Expressions are given
*          in algebraic form and parsed with yacc (see command.yac
*          and y.tab.c).
*          
*/

#include "include.h" 
#include "lex.h"
#include "ytab.h"
  

/*****************************************************************
*
*  Function exparse()
*
*  Purpose: reading, parsing, checking of algebraic function definition.
*          Takes tokens from yylex() until non-expression token.
*          Function value is size in bytes of tree.
*
*/

#define LISTMAX 200

struct treenode *list;    /* tree */
struct treenode *permlist;    /* to use if user doesn't want own copy */
NTYPE listtop;  /* first spot reserved for root */
NTYPE maxp;
int listmax;  /* allocated */

/* for BREAK and CONTINUE */
#define LOOPMAX 30
int loopdepth;
int loopbase[LOOPMAX];

int exparse(maxparam,enode,flag)
int maxparam;  /* maximum number of parameters allowed */
struct expnode *enode;    /* pointer to storage pointer */
int flag;    /* whether to make copy for user */
{
  int retval;

  listmax = LISTMAX;
  if ( permlist == NULL )
     permlist = 
      (struct treenode *)mycalloc(listmax,sizeof(struct treenode));
  maxp = (NTYPE)maxparam;

  if ( flag == USERCOPY )
  { enode->start = list = 
      (struct treenode *)mycalloc(listmax,sizeof(struct treenode));
  }
  else enode->start = list = permlist;

  listtop = 1;
  parse_error_flag = 0;
  brace_depth = parens = in_quote = 0;
  /* unput expression start token for yacc */
  tok = EXPRESSION_START_; unput_tok();

  memset((char*)&query_locals,0,sizeof(struct locallist_t));
  local_nest_depth = 0;
  init_local_scope(0);
  begin_local_scope();
  retval = yyparse();  /* 0 for accept, 1 for error */
  end_local_scope();
  exit_local_scope();
  if ( query_locals.count )
  { enode->locals = 
      (struct locallist_t*)kb_calloc(1,sizeof(struct locallist_t)); 
    *(enode->locals) = query_locals;
 
  }
  else if ( query_locals.list )
    myfree((char*)query_locals.list );
    memset((char*)&query_locals,0,sizeof(struct locallist_t));
    
  if ( (tok != 0) && (tok != ',') && (tok != LEXERROR) )
  {
    /* push back last token */
    unput_tok();
  }

  if ( (retval == 1) || parse_error_flag  || (listtop == 1) )
  { if ( flag == USERCOPY ) myfree((char *)enode->start); 
    enode->start = NULL;
    return -1; 
  }

  /* free excess list */
  if ( flag == USERCOPY )
    enode->start = list = (struct treenode *)kb_realloc((char *)list,
     (listtop+2)*sizeof(struct treenode),listmax*sizeof(struct treenode));
  
  enode->root = list + listtop - 1;
  list[0] = list[listtop-1];  /* root also in first spot */
  list[0].left += listtop - 1;
  list[0].right += listtop - 1;

  /* put DONE marker after root */
  list[listtop++].type = FINISHED;

  /* figure stack usage */
  stack_usage(enode);

  enode->flag = flag;
  return listtop*sizeof(struct treenode);

}

/*********************************************************************
*
*  Function: free_expr()
*
*  Purpose:  Deallocate lists of expression.
*
*/

void free_expr(ex)
struct expnode *ex;
{ struct treenode *node;

  if ( ex == NULL ) return;
  if ( ex->flag == NOUSERCOPY ) return;
  if ( ex->start )
  { for ( node = ex->start ; node != ex->root ; node ++  )
    if ( node->flags & HAS_STRING )
      myfree(node->op1.string);
     myfree((char *)ex->start);
  }
  ex->root = NULL;
  ex->start = NULL;
}


/*********************************************************************
*
*  Function: perm_free_expr()
*
*  Purpose:  Deallocate lists of permanent expression.
*
*/

void perm_free_expr(ex)
struct expnode *ex;
{ struct treenode *node;

  if ( ex == NULL ) return;
  if ( ex->flag == NOUSERCOPY ) return;
  if ( ex->start )
  { for ( node = ex->start ; node != ex->root ; node ++  )
    if ( node->flags & HAS_STRING )
      free(node->op1.string);
     free((char *)ex->start);
  }
  ex->root = NULL;
  ex->start = NULL;
}

/*********************************************************************
*
*  Function:  makenode()
*
*  Purpose: Add nodes to parse tree when called by yyparse.
*          Linear order of node list is such that sequential
*          execution is postorder.  If left > right, then
*          subtrees physically swapped, but they must be adjacent.
*
*  Return:  Index of created node.
*/

void more_makenode ARGS((NTYPE,NTYPE,NTYPE));

int makenode(ntype,left,right)
NTYPE ntype;  /* type of node */
NTYPE left;  /* left son, if any */
NTYPE right; /* right son, if any */
{
  short type = (short)ntype;
  int n;
  struct treenode *nnode;
  struct global *g;
  int etype = -1; /* element type */
  struct locallist_t *localbase = local_scope_bases[local_nest_depth-1];

  if ( listtop > listmax - 10 )
  { list = (struct treenode *)kb_realloc((char*)list,
              (listmax+LISTMAX)*sizeof(struct treenode),
              listmax*sizeof(struct treenode));
    listmax += LISTMAX;
  }
  else
     memset((char*)(list+listtop),0,2*sizeof(struct treenode)); /* clear nodes */
  switch ( type )
  {
      case NULLBLOCK_:
         list[listtop].type = NULLBLOCK_;
         break;

      case NULLCMD_:
         list[listtop].type = NULLCMD_;
         break;

      case NOP_:
         list[listtop].type = NOP_;
         break;

      case BACKQUOTE_START_:
         list[listtop].type = type;
         break;

      case BACKQUOTE_END_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].right = right - listtop;
         list[left].op1.intval = right-left+1; /* so START can jump over */
         break;

      /* command ',' expr */
      case ACOMMANDEXPR_:
         list[listtop].type = ACOMMANDEXPR_;
      list[listtop].left = left - listtop;
      list[listtop].right = right - listtop;
         break;


      /* IF THEN ELSE parse tree

             ELSE_
             /     \
              IF_    command2
             /    \
          IFTEST_  command1
          /
     testexpr

     sequence: testexpr IFTEST_ command1 IF_ command2 ELSE_

     */
      case IFTEST_:
         list[listtop].type = IFTEST_;
         list[listtop].left = left - listtop; /* test expression */
         /* op1 will be offset for jump if condition false */
         list[listtop].stack_delta = -1;
         break;

      case IF_:
         list[listtop].type = IF_;
         list[listtop].left = left - listtop;  /* IFTEST_ */
         list[listtop].right = right - listtop; /* THEN command */
         list[left].op1.intval = listtop - left; /* in IFTEST_ node */
         /* op1 will be offset to skip ELSE part */
         break;

      case ELSE_: /* really the continue node at root of IF tree */
         list[listtop].type = ELSE_;
         list[listtop].left = left - listtop; /* IF_ node */
         if ( right ) list[listtop].right = right - listtop; /* command2 */
         list[left].op1.intval = listtop - left; /* in IF_ node */
         break;
      
      case DECLARE_LOCAL_:
         list[listtop].type = DECLARE_LOCAL_;
         list[listtop].op1.intval = left; /* identifier */
         break;

      case INDEXSET_:
         list[listtop].type = INDEXSET_;
         if ( left )
         { list[listtop].left = left-listtop;
           list[listtop].op1.intval = list[left].op1.intval+1; /* index count */
         } else list[listtop].op1.intval = 1;
         list[listtop].right = right-listtop;
         /* Sanity check on index */
         if ( list[right].type == PUSHCONST )
         { if ( (int)(list[right].op1.real) < 1 ) 
           { sprintf(errmsg,"Index %d must be positive.\n",
                      list[listtop].op1.intval);
             kb_error(2238,errmsg, COMMAND_ERROR);          
           }
         }
         break;

      case DIMENSIONSET_:
         list[listtop].type = INDEXSET_;
         if ( left )
         { list[listtop].left = left-listtop;
           list[listtop].op1.intval = list[left].op1.intval+1; /* index count */
         } else list[listtop].op1.intval = 1;
         list[listtop].right = right-listtop;
         /* Sanity check on index */
         if ( list[right].type == PUSHCONST )
         { if ( (int)(list[right].op1.real) < 0 ) /* can be 0 in define */
           { sprintf(errmsg,"Index %d must be nonnegative.\n",
                      list[listtop].op1.intval);
             kb_error(2522,errmsg, COMMAND_ERROR);          
           }
         }
         break;

      case SINGLE_ASSIGN_:
   /* tree:      SINGLE_ASSIGN
                /          \
        SINGLE_ELEMENT_    SET_ATTRIBUTE_A
             /              /         \
         single            expr          index expr
      */
         list[listtop].type = type;
         list[listtop].op1.intval = assigntype;
         list[listtop].left = left-listtop;  /* single element */
         list[listtop].right = right-listtop; /* attribute and expression */
         list[listtop].stack_delta = -3;
         break;

      case DEFINE_: /* new element attribute */
         list[listtop].type = type;
         if ( left ) list[listtop].left = left-listtop; /* dimension expression */
         list[listtop].op2.intval = right; /* element type */
         break;

      case DEFINE_INDEX_:  /* index added to extra attribute */
       { struct extra *ex;
         list[listtop].type = type;
         list[listtop].left = left-listtop;  /* definition header */
         list[listtop].right = right-listtop;  /* indexset expression */
         list[listtop].op1.intval = list[left].op1.intval; /* extra number */
         list[listtop].op2.intval = list[left].op2.intval; /* extra type */
         ex = EXTRAS(list[left].op2.intval) + list[left].op1.intval;
         ex->adim = list[right].op1.intval;
         ex->flags |= DIMENSIONED_ATTR;
         list[listtop].stack_delta = -ex->adim;
       }
       break;
 
      case ATTR_FUNCTION_: /* attribute function */
         list[listtop].type = type;
         list[listtop].left = left-listtop; /* main define  */
         break;

      case ATTR_FUNCTION_END_: /* attribute function */
         list[listtop].type = type;
         list[listtop].left = left-listtop; /* ATTR_FUNCTION node  */
         list[listtop].right = right-listtop; /* function */
         break;


      case RETURN_: /* end command */
         list[listtop].type = type;
         if ( left ) list[listtop].left = left - listtop; /* return expr */
         break;

      case PAUSE_:  /* wait for user */
         list[listtop].type = type;
         break;

      case BREAK_:
      case CONTINUE_:
         list[listtop].type = type;
         if ( loopdepth < 1 )
           kb_error(1388,"Cannot BREAK or CONTINUE unless inside a loop.\n",
            COMMAND_ERROR);
         if ( loopdepth < left )
           kb_error(1389,"BREAK or CONTINUE out of too many levels.\n",
            COMMAND_ERROR);
         list[listtop].op1.intval = loopbase[loopdepth-left]-listtop;
         list[listtop].op2.intval = left; /* break depth */
         break;

/* WHILE DO tree:     WHILE_END_
                      /        \
                WHILE_TOP_    command 
                 /
            testexpr

execution sequence: testexpr WHILE_TOP_ command WHILE_END_
*/
      case WHILE_TOP_:
         list[listtop].type = WHILE_TOP_;
         list[listtop].left = left - listtop;
         /* op1 will be offset to after WHILE_END_ */
         loopbase[loopdepth++] = listtop;
             /* find start of WHILE expression */
         for ( n = left ; list[n].left ; n += list[n].left ) ; 
         list[listtop].op3.intval[1] = n - listtop - 1; /* for continue */
         list[listtop].stackpos = add_local_var(NULL) & GLOBMASK; 
         list[listtop].stack_delta = -1;
         break;

      case WHILE_END_:
         list[listtop].type = WHILE_END_;
         list[listtop].left = left - listtop; /* test */
         list[listtop].right = right - listtop; /* command */
         list[left].op1.intval = listtop - left; /* jump if test fails */
         list[left].op3.intval[0] = listtop - left; /* for break */
         /* find start of WHILE test expression and set jump back */
         for ( n = left ; list[n].left ; n += list[n].left ) ; 
         list[listtop].op1.intval = n - listtop - 1; /* allow increment */
         loopdepth--;
         if ( loopdepth < 0 )
           kb_error(1390,"Internal error: loopdepth negative.\n",COMMAND_ERROR);

         break;


/*  DO WHILE tree:        DO_END_
                        /        \
                     DO_TOP_    test expr
                    /             / 
                   DO_ENTRY_     command

execution sequence:  DO_ENTRY command DO_TOP_ test expr DO_END_
*/
      case DO_ENTRY_:
         list[listtop].type = type;
         loopbase[loopdepth++] = listtop;
         /* op3 will hold offsets for break and continue */
         list[listtop].stackpos = add_local_var(NULL) & GLOBMASK; 
         break;

      case DO_TOP_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* DO_ENTRY */
         list[listtop].right = right - listtop; /* command */
         list[listtop].stackpos = add_local_var(NULL) & GLOBMASK; 
         break;

      case DO_END_:
      { int entry;
        list[listtop].type = type;
        list[listtop].left = left - listtop; /* DO_TOP_ */
        list[listtop].right = right - listtop; /* command */
        /* find start of DO loop */
        for ( n = left ; list[n].left ; n += list[n].left ) ; 
        list[listtop].op1.intval = n - listtop - 1; /* allow increment */
        /* find start of test loop */
        for ( n = right ; list[n].left ; n += list[n].left ) ; 
        entry = left + list[left].left;  /* DO_ENTRY_ */
        list[entry].op3.intval[0] = listtop-entry; /* break */
        list[entry].op3.intval[1] = n - entry - 1;  /* continue */
        loopdepth--;
        if ( loopdepth < 0 )
          kb_error(1391,"Internal error: loopdepth negative.\n",COMMAND_ERROR);
        list[listtop].stack_delta = -1;

        break;
      }

/* FOR syntax:   FOR ( command1 ; expr ; command2 ) command3 

 FOR tree:
                       FOR_END_
                      /       \
               FOR_TOP_       command3
              /       \
       FOR_HEAD_    command2
       /       \
  FOR_ENTRY    expr
      |
  command1

 FOR execution sequence:

 command1 expr FOR_HEAD_ command2 FOR_TOP_ command3 FOR_END_
 FOR_ENTRY.op3.intval[] holds continue and break jumps
 FOR_HEAD_.op1.intval is jump to command3
 FOR_HEAD_.op2.intval is jump out of loop
 FOR_TOP_.op1.intval is jump back to expr
 FOR_END_.op1.intval is jump back to command2
*/
      case FOR_ENTRY_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* command1 */ 
         loopbase[loopdepth++] = listtop;
         /* op3 will hold offsets for break and continue */
         list[listtop].stackpos = add_local_var(NULL) & GLOBMASK; 
         break;

      case FOR_HEAD_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* FOR_ENTRY */ 
         list[listtop].right = right - listtop; /* expr */
         list[left].op3.intval[1] = listtop-left; /* expr follows immediately */
         list[listtop].stack_delta = -1;
         break;

      case FOR_TOP_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* FOR_HEAD_ */
         list[listtop].right = right - listtop; /* command2 */
         list[listtop].op1.intval = left + list[left].left - listtop;
         list[left].op1.intval = listtop - left; 
         break;

      case FOR_END_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* FOR_TOP_ */
         list[listtop].right = right - listtop; /* command3 */
         list[left+list[left].left].op2.intval = 
                             listtop - left - list[left].left;
         list[listtop].op1.intval = (left + list[left].left) - listtop;
         n = left+list[left].left;  /* n = FOR_HEAD */
         n += list[n].left;   /* n = FOR_ENTRY */
         list[n].op3.intval[0] = listtop-n; /* break */
         loopdepth--;
         if ( loopdepth < 0 )
          kb_error(2515,"Internal error: loopdepth negative.\n",COMMAND_ERROR);
         break;


/* >> redirection tree:     REDIRECT_END_
                             /        \
                        REDIRECT_     command
                        /
                     stringexpr

execution sequence: stringexpr REDIRECT_ command REDIRECT_END_
*/
      case REDIRECT_: /* >> */
      case REDIRECTOVER_: /* >>> */
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* stringexpr */
         break;

      case REDIRECT_END_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         if ( right ) list[listtop].right = right - listtop;
         list[listtop].stack_delta = -1;
         break;

/* piping tree:      PIPE_END_
                    /        \
                PIPE_     command
                /
             stringexpr

execution sequence: stringexpr PIPE_ command PIPE_END_
*/
      case PIPE_:
         list[listtop].type = PIPE_;
         list[listtop].left = left - listtop; /* command */
         break;

      case PIPE_END_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         if ( right ) list[listtop].right = right - listtop;
         list[listtop].stack_delta = -1;
         break;

      case VIEW_TRANSFORMS_:
      case TRANSFORM_DEPTH_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;    /* number of transforms */
         list[listtop].stack_delta = 1;
         break;

      case VIEW_TRANSFORM_PARITY_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;    /* index of transform */
         break;

      case VIEW_TRANSFORM_SWAP_COLORS_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;    /* index of transform */
         break;

      case VIEW_TRANSFORMS_NOP_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; 
         list[listtop].right = right - listtop; 
         break;
      case VIEW_TRANSFORMS_ELEMENT_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; 
         list[listtop].right = right - listtop; 
         list[listtop].stack_delta = -2;
         break;

      case CREATE_VERTEX_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* expression list for coords */
         if ( list[left].op1.intval != SDIM )
         { sprintf(msg,
          "Need exactly %d coordinates in NEW_VERTEX (...).\n",SDIM);
             kb_error(2217,msg,COMMAND_ERROR);
         }
         list[listtop].stack_delta = 1 - SDIM;
         break;

      case CREATE_EDGE_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;    /* id of tail vertex */
         list[listtop].right = right - listtop;    /* id of head vertex */
         list[listtop].stack_delta = -1;
         break;

      case CREATE_FACET_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;    /* expression list for edges */
         if ( (web.representation == SIMPLEX) 
             && (list[left].op1.intval != web.dimension+1) )
         { sprintf(msg,"Need exactly %d vertices in NEW_FACET (...).\n",
              web.dimension+1);
           kb_error(2218,msg,COMMAND_ERROR);
         }
         list[listtop].stack_delta = 1 - list[left].op1.intval;
         break;

      case CREATE_BODY_:
         list[listtop].type = type;
         list[listtop].stack_delta = 1;
         break;

/* Aggregate loops.
(not all nodes need be present)
Tree:            AGGREGATE_END_
                  /          \
          AGGREGATE_INIT_    AGGREGATE_
             or SET_INIT_   /         \ 
                          WHERE_        SET_ATTRIBUTE_L
                          /      \        /            \
                    element_gen  expr    value expr    index expr

sequence: INIT_ element_gen expr WHERE_ value index SET_AT_L AGGR_  AGGR_END_
         prep                                    cleanup

If present, SET_ATTRIBUTE_L will do setting and AGGREGATE_ does looping.
Else AGGREGATE_ does both.
Actually, AGGREGATE_ type is collection of node types.
*/
      case SET_INIT_: 
         list[listtop].type = type;
         loopbase[loopdepth++] = listtop;
         break;

      case AGGREGATE_INIT_:
         list[listtop].type = type;
         list[listtop].op1.intval = aggrtype;
         loopbase[loopdepth++] = listtop;
         switch ( aggrtype )
         { case MAX_: case MIN_: case SUM_: case COUNT_:
             list[listtop].stack_delta = 1;
             break;
           case AVG_:
             list[listtop].stack_delta = 2;
             break;
           case HISTOGRAM_: case LOGHISTOGRAM_:
             list[listtop].stack_delta = 3 + HISTBINS + 1;
             break;
         }
         /* op1 has aggregate type; op2 will have element type; */
         /* op3 will have break and continue jump offsets */
         break;

      case WHERE_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;  /* generator */
         list[listtop].right = right - listtop;  /* condition expr */
         list[listtop].stack_delta = -1;
         list[listtop].op2.localnum = list[left].op2.localnum;
         /* op1 will hold runtime where count */
         break;
         
      case AGGREGATE_:
         list[listtop].type = aggrtype;
         /* op1 holds offset back to start of loop */
         if ( list[left].type == SINGLE_ELEMENT_ )
         { list[listtop].op1.intval = 1; /* don't go back */
           nnode = list+left;
           nnode->op1.intval = listtop - left; /* in case of WHERE_ */
         }
         else if ( list[left].type == WHERE_ )
         { list[listtop].op1.intval = list[left].left + left - listtop;
           nnode = list+left+list[left].left;
           if ( nnode->type == SINGLE_ELEMENT_ )
              nnode->op1.intval = listtop - (left+list[left].left);
         }
         else 
         { list[listtop].op1.intval = left - listtop; /* no WHERE_ */
           nnode = list+left;
         }
         list[listtop].left = left - listtop;  /* generator */
         if ( right ) list[listtop].right = right - listtop;  /* value */
         /* also handy to have element type */
         etype = (nnode+nnode->left)->op1.intval; /* from INIT */
         /* some type error checking and stuff */
         switch ( aggrtype )
         {
           case SUM_: case AVG_: case COUNT_: case HISTOGRAM_:
           case LOGHISTOGRAM_: case MAX_: case MIN_:
             list[listtop].stack_delta = -1; break;

           case FIX_: 
            if ( etype == BODY )
             kb_error(2230,
              "Cannot FIX bodies. To fix volume, do \"set body target expr\".\n",
                    RECOVERABLE);
            break;

           case UNFIX_: 
            if ( etype == BODY )
             kb_error(2881,
               "Cannot UNFIX bodies. To unfix volume, \"unset body target\".\n",
                    RECOVERABLE);
            break;

           case VERTEX_AVERAGE_:
           case RAW_VERTEX_AVERAGE_:
           case RAWEST_VERTEX_AVERAGE_:
             if ( web.representation == SIMPLEX )
               kb_error(2219,"No vertex_average in simplex model yet.\n",
                 RECOVERABLE);
             if ( (etype != VERTEX) )
               kb_error(1238,"Can vertex_average only vertices.\n",COMMAND_ERROR);
             break;
           case DELETE_: 
             if ( (etype != EDGE) && (etype != FACET) )
               kb_error(1392,"Can delete only edges or facets.\n",COMMAND_ERROR);
             break;

           case POP_:
             if ( web.representation == SIMPLEX )
                kb_error(2432,"Can pop only in SOAPFILM or STRING model.\n",
                  COMMAND_ERROR);
             if ( (etype != EDGE) && (etype != VERTEX) )
                kb_error(2433,"Can pop only edges or vertices.\n",
                  COMMAND_ERROR);
             if ( (etype == EDGE) && (web.representation==STRING) )
                kb_error(2434,"Can pop edges only in soapfilm model.\n",
                   COMMAND_ERROR);
             break;

           case POP_TRI_TO_EDGE_:
             if ( web.representation != SOAPFILM )
                kb_error(2803,"Can pop_tri_to_edge only in SOAPFILM model.\n",
                  COMMAND_ERROR);
             if ( etype != FACET )
                kb_error(2804,"Can pop_tri_to_edge only facets.\n",
                  COMMAND_ERROR);
             break;

           case POP_EDGE_TO_TRI_:
             if ( web.representation != SOAPFILM )
                kb_error(2806,"Can pop_edge_to_tro only in SOAPFILM model.\n",
                  COMMAND_ERROR);
             if ( etype != EDGE )
                kb_error(2807,"Can pop_edge_to_tri only edges.\n",
                  COMMAND_ERROR);
             break;

           case POP_QUAD_TO_QUAD_:
             if ( web.representation != SOAPFILM )
                kb_error(2808,"Can pop_quad_to_quad only in SOAPFILM model.\n",
                  COMMAND_ERROR);
             if ( etype != FACET )
                kb_error(2809,"Can pop_quad_to_quad only facets.\n",
                  COMMAND_ERROR);
             break;

           case EDGESWAP_: 
             if ( web.representation != SOAPFILM )
                kb_error(1393,"Can edgeswap only in the SOAPFILM model.\n",
                   RECOVERABLE);
             if ( (etype != EDGE) )
                kb_error(1394,"Can edgeswap only edges.\n",COMMAND_ERROR);
             break;

           case EQUIANGULATE_: 
             if ( web.representation != SOAPFILM )
                kb_error(2500,"Can equiangulate only in the SOAPFILM model.\n",
                   RECOVERABLE);
             if ( (etype != EDGE) )
                kb_error(2501,"Can edgeswap only edges.\n",COMMAND_ERROR);
             break;

           case REFINE_: 
             if ( (etype != EDGE) && (etype != FACET) )
                kb_error(1241,"Can refine only edges or facets.\n",COMMAND_ERROR);

             break;
           case SET_NO_DISPLAY_: 
             if ( etype != FACET )
                kb_error(1265,"No_display only applies to facets.\n",
                   COMMAND_ERROR);
             break;
           case SET_NONCONTENT_: 
             if ( (etype != EDGE) && (etype != FACET) )
                kb_error(2902,"Noncontent only applies to edges or facets.\n",
                   COMMAND_ERROR);
             break;
           case SET_NO_REFINE_: 
             if ( (etype != EDGE) && (etype != FACET) )
                kb_error(1395,"No_refine only applies to edges or facets.\n",
                   COMMAND_ERROR);
             break;
           case SET_CONSTRAINT_:
           case UNSET_CONSTRAINT_:
           case UNSET_BOUNDARY_:
              list[listtop].stack_delta = -1;
              break;
           case SET_EXTRA_ATTR_:
            { struct extra *ex;
              ex = EXTRAS(etype);
              for ( n = 0 ; n < web.skel[etype].extra_count ; n++,ex++ )
              if ( stricmp(ex->name,set_extra_name) == 0 ) break;
              if ( n == web.skel[etype].extra_count )
                kb_error(1396,"Internal error: Invalid extra attribute number.\n",COMMAND_ERROR);
              n += etype << ESHIFT; /* handy to have type here */
              list[listtop].op3.intval[0] = n;
              list[listtop].flags |= EPHEMERAL;
              list[listtop].stack_delta = -ex->adim-1;
            }
            break;
          }
        if ( (aggrtype >= SET_COORD_) && (aggrtype <= SET_COORD_ + MAXCOORD))
        { list[listtop].type = SET_COORD_;
          n = aggrtype - SET_COORD_ - 1; 
          if ( n >= SDIM )
             kb_error(2220,"Coordinate dimension exceeds space dimension.\n",
               RECOVERABLE);
          if ( etype != VERTEX )
             kb_error(1398,"Can set coordinates only for vertices.\n",

                  COMMAND_ERROR);
          list[listtop].op2.intval = n;
        }
        else if ((aggrtype>=SET_PARAM_) && (aggrtype <= SET_PARAM_+MAXPARAM))
        { list[listtop].type = SET_PARAM_;
          list[listtop].op2.intval = aggrtype - SET_PARAM_ - 1; 
          if ( etype != VERTEX )
             kb_error(1399,"Can set parameters only for vertices.\n",

                  COMMAND_ERROR);
        }
        else list[listtop].op2.intval = etype;
        break;

      case AGGREGATE_END_:
        { struct treenode *wnode;
          list[listtop].type = type;
          list[listtop].op1.intval = aggrtype;
          list[listtop].left = left - listtop;  /* aggr init */
          list[listtop].right = right - listtop;  /* aggr op */
          list[listtop].stack_delta =  -list[left].stack_delta - 3;
          if ( (aggrtype == SUM_) || (aggrtype == COUNT_) ||
               (aggrtype == AVG_) || (aggrtype == MAX_ ) ||
               (aggrtype == MIN_ ) ) list[listtop].stack_delta += 1;
          wnode = list+right + list[right].left;
          if ( wnode[wnode->left].type != SINGLE_ELEMENT_ )
          { if ( wnode->type == WHERE_ )
            { nnode = wnode + wnode->left;
              nnode->op1.intval = list + listtop - nnode;
            }  
            else
              wnode->op1.intval = listtop - (right + list[right].left);
          }
        }
        /* element type in aggr_init node */
        list[left].op2.intval = list[right].op2.intval;
        list[left].op3.intval[0] = listtop-left-1; /* break to here */
        list[left].op3.intval[1] = right-left+list[right].op1.intval-1;
                             /* continue */
        list[loopbase[loopdepth-1]].op3.intval[0] = 
         listtop-loopbase[loopdepth-1]-1; /* break to here */
        loopdepth--;
        if ( loopdepth < 0 )
          kb_error(1400,"Internal error: loopdepth negative.\n",COMMAND_ERROR);
        break;
         

/* Element generator.
Tree:              WHERE_
                  /      \
           element_gen    expr


                SINGLE_ELEMENT_
                /         
              INDEXED_SUBTYPE_
             /             \
          single        expr

             INDEXED_ELEMENT_
             /
          expr

         SYMBOL_ELEMENT_

              NEXT_ELEMENT_
              /
         INIT_ELEMENT_


*/
      case INIT_ELEMENT_:
        if ( (list[listtop-1].type != AGGREGATE_INIT_)
           && (list[listtop-1].type != SET_INIT_) )
        { int k;  /* for break and continue */
          int n = add_local_var(NULL) & GLOBMASK;
          list[listtop].stackpos = n; 
          for ( k = listtop-1; k >= 0 ; k-- )
           if ( list[k].type==AGGREGATE_INIT_ ||
             list[k].type==SET_INIT_ )
           { list[k].stackpos = n; break; }
        }
        list[listtop].op1.intval = left; /* save type */
        list[listtop].op2.localnum = use_given_id ? 0 : add_local_var(NULL);
        /* op3.string will have name, if any */
        switch ( left )
        { case VERTEX: etype = INIT_VERTEX_; break;
          case EDGE  : etype = INIT_EDGE_  ; break;
          case FACET : etype = INIT_FACET_ ; break;
          case BODY  : etype = INIT_BODY_  ; break;
          case FACETEDGE  : etype = INIT_FACETEDGE_  ; break;
          default : kb_error(1401,"Internal error: Bad INIT_ELEMENT_ type.\n",
             COMMAND_ERROR);

        }
        list[listtop].type = etype;
        list[listtop].stack_delta = 3;
        break;
 
      case INIT_SUBELEMENT_: /* subelement of named element */
         etype = right;
         list[listtop].op1.intval = etype; /* save type */
         list[listtop].left = left - listtop; /* single element */
         list[listtop].op2.localnum = list[left].op2.localnum;
         { int k;  /* for break and continue */
           int n = add_local_var(NULL) & GLOBMASK;
           list[listtop].stackpos = 0;
           for ( k = listtop-1; k >= 0 ; k-- )
           if ( list[k].type==AGGREGATE_INIT_ ||
            list[k].type==SET_INIT_ )
             { list[k].stackpos = n; break; }
         }
        
         switch ( list[left].op1.intval ) /* parent type */
         { case VERTEX:
             switch ( etype )
             { 
               case EDGE  : etype = INIT_VERTEX_EDGE_  ;
                 if ( web.representation == SIMPLEX )
                   kb_error(1402,"Vertex edge iterator not valid in simplex model.\n",
                  COMMAND_ERROR); 
                 break;
               case FACET : etype = INIT_VERTEX_FACET_ ;
                 if ( web.dimension == STRING )
                   kb_error(2221,"Cannot do vertex facets in STRING model yet.\n",COMMAND_ERROR);
                 break;
               case BODY  : etype = INIT_VERTEX_BODY_  ; break;
               default : kb_error(1403,"Cannot do vertices of vertex.\n",COMMAND_ERROR);
             } 
             break;
          case EDGE:
             switch ( etype )
             { case VERTEX: etype = INIT_EDGE_VERTEX_; break;
               case EDGE : kb_error(1406,"Cannot do edges of edge.\n",
                    COMMAND_ERROR);
               case FACET : etype = INIT_EDGE_FACET_ ; 
                  if ( web.representation == SIMPLEX )
                    kb_error(1404,"Edge facet iterator not valid in simplex model.\n",
                      COMMAND_ERROR);
                  break;
               case BODY  : etype = INIT_EDGE_BODY_  ;
                  if ( web.representation == SIMPLEX )
                      kb_error(1405,"Edge body iterator not valid in simplex model.\n",
                        COMMAND_ERROR);
                  break;
               default : kb_error(2830,"Cannot do facetedges of edge.\n",
                 COMMAND_ERROR);
             } 
             break;
          case FACET:
            switch ( etype )
             { case VERTEX: etype = INIT_FACET_VERTEX_; break;
               case EDGE  : etype = INIT_FACET_EDGE_  ; 
                 if ( web.representation == SIMPLEX )
                   kb_error(1407,"Facet edge iterator not valid in simplex model.\n",
                  COMMAND_ERROR);
                 break;
               case BODY  : etype = INIT_FACET_BODY_  ;
                 if ( web.representation == SIMPLEX )
                    kb_error(1408,"Facet body iterator not valid in simplex model.\n",
                      COMMAND_ERROR);
                 break;
               default : kb_error(1409,"Cannot do facets of facet.\n",COMMAND_ERROR);
             } 
             break;
          case BODY:
            switch ( etype )
             { case VERTEX: etype = INIT_BODY_VERTEX_; break;
               case EDGE  : etype = INIT_BODY_EDGE_  ; break;
               case FACET : etype = INIT_BODY_FACET_ ; break;
               default : kb_error(1410,"Internal error: Bad INIT_ELEMENT_ type.\n",COMMAND_ERROR);
             } 
             break;
          default: kb_error(1411,"Cannot do bodies of body.\n",COMMAND_ERROR);

         }  
         list[listtop].type = etype;
         list[listtop].stack_delta = 3;
         break;
 
      case NEXT_ELEMENT_:
         list[listtop].left = left - listtop;  /* element initializer */
         /* op3.string will have name, if any */
         /* op2.localnum will point to iteration local variable */
         /* op1 reserved for jump to end of loop */
         /* left->op2.localnum will point to parent */
         list[listtop].op2.localnum = use_given_id ? 0 : add_local_var(NULL);
         if ( loopdepth > 0 ) 
           list[loopbase[loopdepth-1]].op3.intval[1] = 
             listtop-loopbase[loopdepth-1]-1;  /* CONTINUE to here */
         switch ( list[left].type ) /* parent type */
        { 
          case INIT_VERTEX_EDGE_  : type = NEXT_VERTEX_EDGE_  ; break;
          case INIT_VERTEX_FACET_ : type = NEXT_VERTEX_FACET_ ; break;
          case INIT_VERTEX_BODY_  : type = NEXT_VERTEX_BODY_  ; break;
          case INIT_EDGE_VERTEX_: type = NEXT_EDGE_VERTEX_; break;
          case INIT_EDGE_FACET_ : type = NEXT_EDGE_FACET_ ; break;
          case INIT_EDGE_BODY_  : type = NEXT_EDGE_BODY_  ; break;
          case INIT_FACET_VERTEX_: type = NEXT_FACET_VERTEX_; break;
          case INIT_FACET_EDGE_  : type = NEXT_FACET_EDGE_  ; break;
          case INIT_FACET_BODY_  : type = NEXT_FACET_BODY_  ; break;
          case INIT_BODY_VERTEX_: type = NEXT_BODY_VERTEX_; break;
          case INIT_BODY_EDGE_  : type = NEXT_BODY_EDGE_  ; break;
          case INIT_BODY_FACET_ : type = NEXT_BODY_FACET_ ; break;
          case INIT_VERTEX_: type = NEXT_VERTEX_; break;
          case INIT_EDGE_  : type = NEXT_EDGE_  ; break;
          case INIT_FACET_: type = NEXT_FACET_ ; break;
          case INIT_BODY_  : type = NEXT_BODY_  ; break;
          case INIT_FACETEDGE_ : type = NEXT_FACETEDGE_; break;
          default : kb_error(1412,"Internal error: Bad NEXT_ELEMENT_ type.\n",COMMAND_ERROR);

        }
         list[listtop].type = type;
         break;
 
      case RITZ_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].right = right - listtop;
         list[listtop].stack_delta = -2;
         break;

      case WRAP_VERTEX_:
         if ( !web.symmetry_flag )
           kb_error(2222,
             "Can wrap vertex only in torus or symmetry group model.\n",
                RECOVERABLE);
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].right = right - listtop;
         list[listtop].stack_delta = -2;
         break;

      case LANCZOS_:
      case EIGENPROBE_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].stack_delta = -1;
         if ( right ) 
         { list[listtop].right = right - listtop;
           list[listtop].stack_delta = -2;
         }
         break;

      case HESSIAN_SADDLE_:
      case HESSIAN_SEEK_:
         list[listtop].type = type;
         if ( left ) list[listtop].left = left - listtop;
         list[listtop].stack_delta = -1;
         break;

      case MOVE_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].stack_delta = -1;
         break;

      case AREAWEED_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].stack_delta = -1;
         break;

      case METIS_:
      case KMETIS_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].stack_delta = -1;
         break;

      case OMETIS_:
         list[listtop].type = type;
         if ( left )
         {  list[listtop].left = left - listtop;
            list[listtop].stack_delta = -1;
         }
         break;

      case EDGEWEED_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].stack_delta = -1;
         break;

      case EDGEDIVIDE_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].stack_delta = -1;
         break;

    case HISTORY_:
         list[listtop].type = type;
         break;
     
    case LAGRANGE_:
         list[listtop].type = type;
         list[listtop].left = left-listtop; /* degree approximation */
         list[listtop].stack_delta = -1;
         break;
     
    case BURCHARD_:
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* number of steps */
         break;
     
    case ZOOM_:
         list[listtop].type = type;
         if ( left )
         {  list[listtop].left = left - listtop; /* vertex number */
            list[listtop].stack_delta = -1;
         }
         if ( right ) 
         {  list[listtop].left = right - listtop;  /* radius expression */
            list[listtop].stack_delta += -1;
         }
         break;
     
    case SET_GRAVITY_: 
         list[listtop].type = type;
         list[listtop].left = left - listtop;  /* value expression */
         list[listtop].op1.intval = assigntype;
         list[listtop].stack_delta = -1;
         break;

    case SET_AMBIENT_PRESSURE_: case SET_GAP_CONSTANT_:
    case NOTCH_: case SET_AUTOCHOP_: case SET_FIXED_AREA_:
    case SET_OPTIMIZE_: case SET_SCALE_:
    case JIGGLE_: case SET_BACKGROUND_:
    case PRINT_: case STRPRINT_:
    case INVOKE_P_MENU_: case SET_MODEL_: case SKINNY_: case TORDUP_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;  /* value expression */
         list[listtop].stack_delta = -1;
         break;

    case SET_INTERNAL_:
         list[listtop].type = type;
         list[listtop].left = right - listtop;  /* value expression */
         list[listtop].op1.intval = left;  /* variable type */
         list[listtop].op2.intval = assigntype;  
         list[listtop].stack_delta = (assigntype == ASSIGN_) ? -1 : 0;

         switch(left)
        { /* break on settable variables */
          case V_AMBIENT_PRESSURE: case V_HESSIAN_SLANT_CUTOFF:
          case GRAV_CONST_: case V_BREAKFLAG_: case V_VISIBILITY_DEBUG_:
          case V_TOLERANCE: case V_HESS_EPSILON:  break;
          case V_SCALE_SCALE: case V_TIME: case V_SCALE: 
          case V_JIG_TEMP: case V_SCALE_LIMIT: case V_GAP_CONSTANT:
          case V_THICKNESS: case V_TARGET_TOLERANCE: case V_SCROLLBUFFERSIZE_:
          case V_PICKVNUM: case V_PICKENUM: case V_PICKFNUM:
          case V_LINEAR_METRIC_MIX: case V_QUADRATIC_METRIC_MIX:
          case V_RANDOM_SEED: break; case V_BRIGHTNESS: break;
          case V_BACKGROUND: break; case V_LAST_ERROR:
          case V_INTEGRAL_ORDER_1D: case V_INTEGRAL_ORDER_2D: case V_DIFFUSION:
          case V_PS_STRINGWIDTH_: case V_PS_FIXEDEDGEWIDTH_:
          case V_PS_TRIPLEEDGEWIDTH_: case V_PS_GRIDEDGEWIDTH_:
          case V_PS_CONEDGEWIDTH_: case V_PS_BAREEDGEWIDTH_:
            break;

          case V_INTEGRAL_ORDER:
            kb_error(1413,
               "Please use integral_order_1d or integral_order_2d.\n",WARNING);
            break;
          default: 
            sprintf(msg,"Cannot set internal variable '%s'.\n",keywordname(left));
            kb_error(1414,msg,EXPRESSION_ERROR);

        }
         list[listtop].stack_delta = -1;
         break;

    case FIX_QUANTITY_: case UNFIX_QUANTITY_:
    case SET_Q_FIXED_: case SET_Q_ENERGY_: case SET_Q_INFO_: 
    case SET_Q_CONSERVED_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;  /* named quantity var number */
         list[listtop].flags |= EPHEMERAL;
         break;

    case FIX_PARAMETER_:
    case UNFIX_PARAMETER_:
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* variable number */
         if ( !(globals(left)->flags & ORDINARY_PARAM ) )
         { sprintf(errmsg,"'%s' cannot be made an optimizing parameter.\n",
              globals(left)->name);
           kb_error(2223,errmsg,RECOVERABLE);
         }
         list[listtop].flags |= EPHEMERAL;
         break;

    case SET_QUANTITY_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;  /* value expression */
         list[listtop].op1.intval = right;  /* quantity number */
         list[listtop].flags |= EPHEMERAL;
         break;

    case TOGGLE_QUANTITY_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;  /* quantity number */
         list[listtop].op2.intval = right;  /* toggle */
         list[listtop].flags |= EPHEMERAL;
         break;

    case PRINT_ARRAYPART_:
         list[listtop].type = type;
         list[listtop].left = left-listtop;  /* arrayhead */
         list[listtop].op1.intval = list[left].op2.intval;  /* # indexes */
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = -list[listtop].op1.intval;
         break;

    case PRINT_ARRAY_:
    case PRINT_PROCEDURE_:
    case EXPRINT_PROCEDURE_:
    case PRINT_LETTER_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;
         list[listtop].flags |= EPHEMERAL;
         break;

    case PRINT_PERM_PROCEDURE_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;
         break;

    case UNREDEFINE_SINGLE_:
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* letter */
         if ( right > 127 )
           kb_error(1415,"Illegal unredefine.\n",COMMAND_ERROR);
         break;

    case REDEFINE_SINGLE_:
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* letter */
         list[listtop].op3.locals = 
             (struct locallist_t*)mycalloc(sizeof(struct locallist_t),1);
         list[listtop].op3.locals[0] = query_locals;
         memset(&query_locals,0,sizeof(query_locals));
         if ( right > 127 ) 
           kb_error(1416,"Illegal redefine.\n",COMMAND_ERROR);
         right = listtop;
         subtree_swap(&left,&right);
         list[right].op2.intval = left - right; /* proc root */
         list[listtop].line_no = line_no;
         list[listtop].file_no = file_no;
         listtop++;
         list[listtop].type = SET_PROC_END_;
         list[listtop].left = right - listtop; /* action node */
         list[listtop].right = left - listtop; /* procedure */
         break;

    case SET_PROCEDURE_:
       { struct global *g = globals(right);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* variable number */
         if ( g->flags & ORDINARY_PARAM )
         { sprintf(errmsg,"Cannot redefine variable '%s' as a command.\n",
             g->name);
           kb_error(1417,errmsg,RECOVERABLE);
         }
         g->flags |= SUBROUTINE;
         list[listtop].op3.locals = g->attr.procstuff.locals; 
         right = listtop;
         subtree_swap(&left,&right);
         list[right].op2.intval = left - right; /* proc root */
         list[listtop].line_no = line_no;
         list[listtop].file_no = file_no;
         listtop++;
         list[listtop].type = SET_PROC_END_;
         list[listtop].left = right - listtop; /* action node */
         list[listtop].right = left - listtop; /* procedure */
         list[listtop].flags |= EPHEMERAL;
         break;
       }

    case SET_PERM_PROCEDURE_:
       { struct global *g = globals(right);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* variable number */
         if ( g->flags & ORDINARY_PARAM )
         { sprintf(errmsg,"Cannot redefine variable '%s' as a command.\n",
             g->name);
           kb_error(2516,errmsg,RECOVERABLE);
         }
         g->flags |= SUBROUTINE;
         right = listtop;
         subtree_swap(&left,&right);
         /* see if any ephemeral stuff, and mark as part of permanent cmd */
         for ( n = right ; n < listtop ; n++ )
         { if ( list[n].flags & EPHEMERAL )
           { sprintf(errmsg,"Non-permanent items in definition of %s.\n",
               g->name);
             kb_error(2517,errmsg,COMMAND_ERROR);
           }
           list[n].flags |= PERMNODE; 
         }
         list[right].op2.intval = left - right; /* proc root */
         list[listtop].line_no = line_no;
         list[listtop].file_no = file_no;
         list[listtop].op3.locals = g->attr.procstuff.locals; 
         listtop++;
         list[listtop].type = SET_PERM_PROC_END_;
         list[listtop].left = right - listtop; /* action node */
         list[listtop].right = left - listtop; /* procedure */
         break;
       }

    case ARGLIST_:
         list[listtop].type = type;
         list[listtop].left = left ? left - listtop : 0; /* prev args */
         if ( right )
         { list[listtop].op1.intval = right; /* name id */
           globals(right)->flags |= ORDINARY_PARAM;
         }
         list[listtop].op2.intval =  /* count of arguments */
              (left ? list[left].op2.intval : 0) + (right?1:0);                   
         list[listtop].op3.intval[1] = int_val; /* argument type */
         break;

    case FUNCTION_DEF_START_:
    case FUNCTION_PROTO_START_:
       { struct global *g = globals(left);
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* name id */
         if ( g->flags & ORDINARY_PARAM )
         { sprintf(errmsg,"Cannot redefine '%s' as a function.\n",g->name);
           kb_error(1417,errmsg,RECOVERABLE);
         }
         g->flags |= FUNCTION_NAME;
         /* op2.intval for skip ahead, set later */
         /* op3.intval[0] for argument count */
         list[listtop].op3.intval[1] = right;  /* type of return value */
         break;
       }
       
    case FUNCTION_HEAD_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* function def start */
         list[listtop].right = right - listtop; /* function arglist */
         list[listtop].op1.intval = list[left].op1.intval; /* name id */
         list[left].op3.intval[0] = list[right].op2.intval; /* arg count */
         globals(list[left].op1.intval)->attr.procstuff.argcount =
              list[right].op2.intval;
         list[listtop].op3.intval[1] = list[left].op3.intval[1]; /* type */
         break;

    case SET_FUNCTION_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* function head */
         list[listtop].right = right - listtop; /* function body */
         list[listtop].op1.intval = list[left].op1.intval; /* name id */
         n = left + list[left].left;
         list[n].op2.intval = listtop - n; /* for FUNCTION_START_ jump */
         list[listtop].flags |= EPHEMERAL;
         list[listtop].op3.intval[1] = list[left].op3.intval[1]; /* type */
         break;

    case FUNCTION_PROTO_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* function head */
         list[listtop].op1.intval = list[left].op1.intval; /* name id */
         n = left + list[left].left;
         list[n].op2.intval = listtop - n; /* for FUNCTION_PROTO_START_ jump */
         list[n].type = FUNCTION_PROTO_START_;
         list[listtop].flags |= EPHEMERAL;
         list[listtop].op3.intval[1] = list[left].op3.intval[1]; /* type */
         break;

    case FUNCTION_CALL_:
       { struct global *g = globals(left);
         list[listtop].type = type;
         if ( right )
           list[listtop].left = right - listtop; /* argument expressions */
         list[listtop].op1.intval = left;  /* function name id */
         if ( list[right].op1.intval != g->attr.procstuff.argcount )
         { sprintf(errmsg,"Function \"%s\" needs %d arguments; call has %d,\n",
               g->name,g->attr.procstuff.argcount,list[right].op1.intval);
           kb_error(2620,errmsg,COMMAND_ERROR);
         }
         list[listtop].op2.intval = g->attr.procstuff.argcount;
         list[listtop].stack_delta = 1 - g->attr.procstuff.argcount;
         break;
      }

    case PROCEDURE_DEF_START_:
    case PROCEDURE_PROTO_START_:
       { struct global *g = globals(left);
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* name id */
         if ( g->flags & ORDINARY_PARAM )
         { sprintf(errmsg,"Cannot redefine '%s' as a function.\n",g->name);
           kb_error(1417,errmsg,RECOVERABLE);
         }
         g->flags |= PROCEDURE_NAME;
         /* op2.intval for skip ahead, set later */
         /* op3.intval[0] for argument count */
         break;
       }
       
    case PROCEDURE_HEAD_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* procedure def start */
         list[listtop].right = right - listtop; /* procedure arglist */
         list[listtop].op1.intval = list[left].op1.intval; /* name id */
         list[left].op3.intval[0] = list[right].op2.intval; /* arg count */
         globals(list[left].op1.intval)->attr.procstuff.argcount =
              list[right].op2.intval;
         break;

    case SET_ARGSPROC_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* function head */
         list[listtop].right = right - listtop; /* function body */
         list[listtop].op1.intval = list[left].op1.intval; /* name id */
         n = left + list[left].left;
         list[n].op2.intval = listtop - n; /* for PROCEDURE_START_ jump */
         list[listtop].flags |= EPHEMERAL;
         break;

    case PROCEDURE_PROTO_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* function head */
         list[listtop].op1.intval = list[left].op1.intval; /* name id */
         n = left + list[left].left;
         list[n].op2.intval = listtop - n; /* for PROCEDURE_PROTO_START_ jump */
         list[n].type = PROCEDURE_PROTO_START_;
         list[listtop].flags |= EPHEMERAL;
         break;

    case PROCEDURE_CALL_:
       { struct global *g = globals(left);
         list[listtop].type = type;
         if ( right )
           list[listtop].left = right - listtop; /* argument expressions */
         list[listtop].op1.intval = left;  /* procedure name id */
         if ( list[right].op1.intval != g->attr.procstuff.argcount )
         { sprintf(errmsg,"Procedure \"%s\" needs %d arguments; call has %d.\n",
               g->name,g->attr.procstuff.argcount,list[right].op1.intval);
           kb_error(2623,errmsg,COMMAND_ERROR);
         }
         list[listtop].op2.intval = g->attr.procstuff.argcount; 
         list[listtop].stack_delta = -g->attr.procstuff.argcount;
         break;
      }

    case DEFINE_IDENT_:
       { struct global *g = globals(left);
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* variable number */
         list[listtop].op2.intval = right; /* type */
         if ( right == STRING_TYPE_ ) g->flags |= STRINGVAL;
         else if ( !(g->flags & ANY_TYPE)  )
         { g->flags |= ORDINARY_PARAM;
           list[listtop].flags |= EPHEMERAL;
           g->attr.varstuff.delta = OPTPARAM_DELTA;
           g->attr.varstuff.pscale = 1.0;
         }
         break;
       }
       
    case DEFINE_ARRAY_:
       { struct global *g = globals(left);
         
         if ( (int_val != REAL_TYPE_) && (int_val != INTEGER_TYPE_) )
           kb_error(2483,"Arrays only support integer and real type.\n",
              COMMAND_ERROR);
         list[listtop].type = type;
         list[listtop].left = right-listtop; /* index expression list */
         list[listtop].op1.intval = int_val; /* type */
         list[listtop].op2.intval = left; /* global variable number */
         g->flags |= ARRAY_PARAM;
         if ( !(g->flags & GLOB_LOCALVAR) && g->value.arrayptr &&
                   g->value.arrayptr->datatype != int_val )
         { sprintf(errmsg,
             "Array %s type declaration conflicts with previous type.\n",
                 g->name);
           kb_error(2224,errmsg,COMMAND_ERROR);
         }
                
         if ( (g->attr.varstuff.dim > 0) &&
                        (g->attr.varstuff.dim != list[right].op1.intval) )
         { sprintf(errmsg,"Cannot change the number of dimensions of array %s.\n",
                 g->name);
           kb_error(2225,errmsg,COMMAND_ERROR);
         }
         else globals(left)->attr.varstuff.dim = list[right].op1.intval;
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = -list[right].op1.intval;
         break;
       }

    case ARRAY_HEAD_:
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* global variable number */
         list[listtop].left = left-listtop;  /* indexset */
         list[listtop].op2.intval = list[left].op1.intval; /* # indices */
/*
         if ( list[left].op1.intval != globals(right)->dim ) 
         { sprintf(errmsg,
              "Array %s has wrong number of dimensions; should be %d\n",
                 globals(int_val)->name,globals(int_val)->dim);
           kb_error(2226,errmsg,COMMAND_ERROR);
         }
*/
         list[listtop].flags |= EPHEMERAL;
         break;

    case ARRAYASSIGN:
         list[listtop].type = type;
         list[listtop].left = left-listtop;  /* arrayhead */
         list[listtop].right = right-listtop;  /* rhs */
         list[listtop].op1.intval = assigntype;
         list[listtop].op2.intval = list[left].op1.intval; /* glob var number */

         g =  globals(list[left].op1.intval);
         if ( list[left].op2.intval != g->attr.varstuff.dim ) 
         { sprintf(errmsg,
              "Array %s has wrong number of dimensions; should be %d\n",
                 g->name,g->attr.varstuff.dim);
           kb_error(2226,errmsg,COMMAND_ERROR);
         } 
         if ( g->flags & READONLY )
         { sprintf(errmsg, "Array %s is read-only.\n",g->name);
           kb_error(2846,errmsg,COMMAND_ERROR);
         }
         list[listtop].stack_delta = -list[left].op2.intval-1;
         if ( !(g->flags & PERMANENT) )
           list[listtop].flags |= EPHEMERAL;

         break;


    case ARRAYEVAL:
         list[listtop].type = type;
         list[listtop].left = left-listtop;
         list[listtop].op2.intval = list[left].op1.intval;
         list[listtop].flags |= EPHEMERAL;
         if ( list[left].op2.intval != 
                 globals(list[left].op1.intval)->attr.varstuff.dim ) 
         { sprintf(errmsg,
              "Array %s has wrong number of dimensions; should be %d\n",
                 globals(list[left].op1.intval)->name,
                 globals(list[left].op1.intval)->attr.varstuff.dim );
           kb_error(2608,errmsg,COMMAND_ERROR);
         }
         list[listtop].stack_delta = -list[left].op2.intval+1;
         break;


    case SET_DELTA_: case SET_PARAM_SCALE:
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* variable number */
         list[listtop].left = right - listtop;  /* value expression */
         list[listtop].op2.intval = assigntype;
         if ( !(globals(left)->flags & ANY_TYPE)  )
            globals(left)->flags |= ORDINARY_PARAM;
         if ( perm_flag ) globals(left)->flags |= PERMANENT;
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = -1;
         break;

    case SET_GLOBAL_: 
    case PLUSASSIGN_: case SUBASSIGN_: case MULTASSIGN_: case DIVASSIGN_:
      {  struct locallist_t *localbase = local_scope_bases[local_nest_depth-1];
         struct global *g = globals(left);

         list[listtop].type = type;
         list[listtop].op1.intval = left; /* variable number */
         list[listtop].left = right - listtop;  /* value expression */
         if ( !(g->flags & ANY_TYPE)  )
            g->flags |= ORDINARY_PARAM;
         if ( perm_flag ) g->flags |= PERMANENT;
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = -1;
         break;
      }

    case SET_PERM_GLOBAL_: 
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* variable number */
         list[listtop].left = right - listtop;  /* value expression */
         perm_globals(left)->flags |= ORDINARY_PARAM;
         perm_globals(left)->attr.varstuff.delta = OPTPARAM_DELTA;
         perm_globals(left)->attr.varstuff.pscale = 1.0;
         list[listtop].stack_delta = -1;
         break;

    case SET_SGLOBAL_:
    case SET_PERM_SGLOBAL_:
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* variable number */
         list[listtop].left = right - listtop;  /* value expression */
         globals(left)->flags |= STRINGVAL;
         if ( type == SET_SGLOBAL_ ) list[listtop].flags |= EPHEMERAL;
         if ( perm_flag ) globals(left)->flags |= PERMANENT;
         list[listtop].stack_delta = -1;
         break;

    case SHOW_:
    case SHOW_EXPR_:
         list[listtop].type = type;
         /* put node in front of expression */
         right = listtop;
         subtree_swap(&left,&right);
         list[right].op1.intval = left - right; 
         list[listtop].line_no = line_no;
         list[listtop].file_no = file_no;
         listtop++;
         list[listtop].type = SHOW_END_;
         list[listtop].left = right - listtop; /* action node */
         list[listtop].right = left - listtop; /* procedure */
         break;

    case SELF_ELEMENT_:
         list[listtop].type = type;
         list[listtop].op1.intval = left; 
         break;

    case SYMBOL_ELEMENT_:
         list[listtop].type = type;
         list[listtop].op1.intval = symtable[left].type; 
         list[listtop].op2.localnum = symtable[left].localnum; 
         list[listtop].op3.string =
           (char*)mycalloc(strlen(symtable[left].name)+1,1);
         strcpy(list[listtop].op3.string,symtable[left].name);
         break;

    case SINGLE_ELEMENT_: /* generator */
         list[listtop].type = SINGLE_ELEMENT_INIT_; /* for dummy loop */
         list[listtop].line_no = line_no;
         list[listtop].file_no = file_no;
         list[listtop].stack_delta = 3;
         listtop++;
         list[listtop].type = type;
         list[listtop].left = left-listtop;  /* index value */
         list[listtop].right = -1; /* to INIT */
         list[listtop].op1.intval = right;    /* element type */
         list[listtop].op2.localnum = add_local_var(NULL);
                    /* local will hold id at runtime */
         /* op1 will hold jump to end of loop */
         break;

    case INDEXED_ELEMENT_: 
         list[listtop].type = type;
         list[listtop].left = right-listtop;  /* index value */
         list[listtop].op1.intval = left;    /* element type */
         list[listtop].stack_delta = -1;
         list[listtop].op2.localnum = add_local_var(NULL);
                    /* local will hold id at runtime */
         /* Sanity check on index */
         if ( list[right].type == PUSHCONST )
         { if ( list[right].op1.real == 0.0 ) 
           kb_error(2228,"Element index must be nonzero.\n",COMMAND_ERROR);
         }
         break;

    case INDEXED_SUBTYPE_: 
         list[listtop].type = type;
         list[listtop].left = left-listtop;  /* single element */
         list[listtop].right = right-listtop;  /* index value */
         list[listtop].op1.intval = subtype;    /* element type */
         list[listtop].stack_delta = -1;
         list[listtop].op2.localnum = add_local_var(NULL);
                    /* local will hold id at runtime */
         /* Sanity check on index */
         if ( list[right].type == PUSHCONST )
         { if ( list[right].op1.real <= 0.0 ) 
           kb_error(2229,"Element index must be positive.\n",COMMAND_ERROR);
         }
         /* some type checking */
         etype = list[left].op1.intval;
         if ( etype == subtype )
           kb_error(1418,"Cannot do same subtype as type of element.\n",
            COMMAND_ERROR);
         switch ( etype )
         { case VERTEX:
             switch ( subtype /* subtype */ )
             { case BODY: case FACETEDGE: 
                  kb_error(1419,"Unimplemented subtype of vertex.\n",COMMAND_ERROR);
             }
             break;
           case EDGE:
             switch ( subtype /* subtype */ )
             { case BODY:
                  kb_error(1420,"Unimplemented subtype of edge.\n",COMMAND_ERROR);
             }
             break;
           case FACET:
             switch ( subtype /* subtype */ )
             { case FACETEDGE: 
                  kb_error(1421,"Unimplemented subtype of facet.\n",COMMAND_ERROR);
             }
             break;
           case BODY:
             switch ( subtype /* subtype */ )
             { case FACETEDGE: 
               case VERTEX:
               case EDGE:
                  kb_error(1422,"Unimplemented subtype of body.\n",COMMAND_ERROR);
             }
             break;
           case FACETEDGE:
             switch ( subtype /* subtype */ )
             { case FACETEDGE: 
               case VERTEX:
               case BODY:
                  kb_error(1423,"Unimplemented subtype of facet.\n",COMMAND_ERROR);
             }
             break;
         }
         break;

    case FIX_: case UNFIX_:
         if ( right == BODY )
            kb_error(2230,
               "Cannot fix/unfix bodies. Try  set/unset body target.\n",
              RECOVERABLE);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case DISSOLVE_:
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case VERTEX_AVERAGE_:
    case RAW_VERTEX_AVERAGE_:
    case RAWEST_VERTEX_AVERAGE_:
         if ( type != VERTEX )
          kb_error(2231,"Can only vertex_average vertices.\n",RECOVERABLE);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case DELETE_:
         if ( (type != EDGE) && (type != FACET) )
            kb_error(2232,"Can only delete edges and facets.\n",RECOVERABLE);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case REFINE_:
         if ( (type != EDGE) && (type != FACET) )
            kb_error(2233,"Can only refine edges and facets.\n",RECOVERABLE);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case POP_:
         if ( web.representation == SIMPLEX )
            kb_error(2435,"Can pop only in SOAPFILM or STRING model.\n",
               COMMAND_ERROR);
         if ( (type != EDGE) && (etype != VERTEX) )
            kb_error(2436,"Can pop only edges or vertices.\n",COMMAND_ERROR);
         if ( (type == EDGE) && (web.representation==STRING) )
            kb_error(2437,"Can pop edges only in soapfilm model.\n",
              COMMAND_ERROR);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case POP_TRI_TO_EDGE_:
         if ( web.representation != SOAPFILM )
            kb_error(2810,"Can pop_tri_to_edge only in SOAPFILM  model.\n",
               COMMAND_ERROR);
         if ( type != FACET )
            kb_error(2811,"Can pop_tri_to_edge only facets.\n",COMMAND_ERROR);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case POP_EDGE_TO_TRI_:
         if ( web.representation != SOAPFILM )
            kb_error(2812,"Can pop_edge_to_tri only in SOAPFILM  model.\n",
               COMMAND_ERROR);
         if ( type != EDGE )
            kb_error(2813,"Can pop_edge_to_tri only edges.\n",COMMAND_ERROR);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case POP_QUAD_TO_QUAD_:
         if ( web.representation != SOAPFILM )
            kb_error(2814,"Can pop_quad_to_quad only in SOAPFILM  model.\n",
               COMMAND_ERROR);
         if ( type != FACET )
            kb_error(2815,"Can pop_quad_to_quad only facets.\n",COMMAND_ERROR);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case EDGESWAP_:
         if ( type != EDGE )
          kb_error(2234,"Can only edgeswap edges.\n",RECOVERABLE);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case EQUIANGULATE_:
         if ( type != EDGE )
          kb_error(2502,"Can only equiangulate edges.\n",RECOVERABLE);
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break;

    case LIST_:
         list[listtop].type = type;
         list[listtop].op1.intval = right; /* element type */
         list[listtop].left = left - listtop;  /* where expression */
         break; 

    case SHOW_VOL_:  case CHECK_: case LONG_JIGGLE_: case RAW_VERAVG_:
    case STABILITY_TEST_: case UTEST_: case GO_: case SHELL_: 
    case ALICE_:      case TOPINFO_: case RECALC_: case COUNTS_:
    case EXTRAPOLATE_: case HESSIAN_: case SOBOLEV_: 
    case SHOWQ_:case RAWEST_VERAVG_: case DIRICHLET_: case BOTTOMINFO_:
    case CLOSE_SHOW_: case REBODY_: case LIST_PROCS_: case HESSIAN_MENU_:
    case DIRICHLET_SEEK_: case SOBOLEV_SEEK_: case CONVERT_TO_QUANTS_:
    case LIST_ATTRIBUTES_: case REORDER_STORAGE_: case RENUMBER_ALL_:
         list[listtop].type = type;
         break;

    /* toggles */ case LINEAR_: case QUADRATIC_: case QUIETGO_:
    case KUSNER_: case ESTIMATE_: case DETURCK_: case HOMOTHETY_:
    case SQGAUSS_: case AUTOPOP_: case AUTOCHOP_: case QUIET_:
    case OLD_AREA_: case APPROX_CURV_: case RUNGE_KUTTA_: 
    case CHECK_INCREASE_:  case DEBUG_: case MEAN_CURV_: case RIBIERE_CG_:
    case DIFFUSION_: case GRAVITY_: case CONJ_GRAD_: case TRANSFORMS_:
    case CONF_EDGE_SQCURV_: case EFFECTIVE_AREA_: case AREA_FIXED_:
    case RAW_CELLS_: case CONNECTED_CELLS_: case CLIPPED_CELLS_:
    case THICKEN_: case SHOW_INNER_: case SHOW_OUTER_: case COLORMAP_: 
    case HESSIAN_DIFF_: case POST_PROJECT_: case MEAN_CURV_INT_:
    case OPTIMIZE_: case NORMAL_CURVATURE_: case DIV_NORMAL_CURVATURE_:
    case SHADING_:  case FACET_COLORS_: case BOUNDARY_CURVATURE_:
    case NORMAL_MOTION_: case PINNING_: case VIEW_4D_: case MEMDEBUG_:
    case METRIC_CONVERSION_: case AUTORECALC_: case GV_BINARY_:
    case SELF_SIMILAR_: case AUTODISPLAY_: case FORCE_POS_DEF_:
    case ASSUME_ORIENTED_: case HESSIAN_QUIET_: case JIGGLE_TOGGLE_:
    case HESSIAN_NORMAL_: case YSMP_: case BUNCH_KAUFMAN_:
    case QUANTITIES_ONLY_: case LINEAR_METRIC_: case GEOMVIEW_TOGGLE_:
    case GEOMPIPE_TOGGLE_: case SQUARED_GRADIENT_: case H_INVERSE_METRIC_:
    case HESSIAN_DOUBLE_NORMAL_: case INTERP_BDRY_PARAM_: case LOGFILE_TOGGLE_:
    case HESSIAN_NORMAL_ONE_: case PSCOLORFLAG_: case GRIDFLAG_:
    case CROSSINGFLAG_: case LABELFLAG_: case SHOW_ALL_QUANTITIES_:
    case HESSIAN_NORMAL_PERP_: case HESSIAN_SPECIAL_NORMAL_: case ITDEBUG_:
    case METIS_FACTOR_: case VOLGRADS_EVERY_: case ZENER_DRAG_: 
    case BACKCULL_: case INTERP_NORMALS_: case TORUS_FILLED_: case VERBOSE_:
    case AMBIENT_PRESSURE_: case DIRICHLET_MODE_: case SOBOLEV_MODE_:
    case KRAYNIKPOPVERTEX_FLAG_: case KEYLOGFILE_TOGGLE_:
    case KRAYNIKPOPEDGE_FLAG_: case VISIBILITY_TEST_: case SPARSE_CONSTRAINTS_:
    case BLAS_FLAG_: case AUGMENTED_HESSIAN_: case BREAK_AFTER_WARNING_:
    case RGB_COLORS_FLAG_:  case CIRCULAR_ARC_DRAW_: case BEZIER_BASIS_:
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* toggle state */
         break;
     
    case SYSTEM_: 
    case READ_:
    case EXEC_:
    case CHDIR_:
    case SHOW_TRANS_:
    case SET_COLORMAP_:
    case TRANSFORM_EXPR_:  case KEYLOGFILE_: case OOGLFILE_:
    case GEOMVIEW_: case GEOMPIPE_: case POSTSCRIPT_: case LOGFILE_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].stack_delta = -1;
         break;

    case PRINTFHEAD_:
    case ERRPRINTFHEAD_:
    case SPRINTFHEAD_:
         list[listtop].type = type;
         /* copy and convert escape sequences */
         if ( left ) 
         { list[listtop].left = left - listtop;
           list[listtop].stack_delta = -1;
         }
         else 
         {
           char *c, *s;
           list[listtop].op1.string = 
              (char*)mycalloc(strlen(yytext)+1,sizeof(char));
           list[listtop].flags |= HAS_STRING;
           s = yytext; /* source */
           c = list[listtop].op1.string; /* destination */
           while ( *s )
             if ( *s == '\\' )
             { switch ( s[1] )
               { case 'n' : *(c++) = '\n'; s += 2; break;
                 case 't' : *(c++) = '\t'; s += 2; break;
                 case 'b' : *(c++) = '\b'; s += 2; break;
                 case 'r' : *(c++) = '\r'; s += 2; break;
                 default:  *(c++) =  s[1]; s += 2; break;
               }
             }
             else *(c++) = *(s++);
           *c = '\0';
         }
         break;

    case SPRINTF_: /* with expressions */
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* has string */
         list[listtop].right = right - listtop; /* expressions */
         if ( list[right].op1.intval > 10 )
             kb_error(1424,"Limit of 10 arguments to PRINTF.\n",RECOVERABLE);
         list[left].type = PRESPRINTF_;
         list[left].stack_delta = 0;
         list[listtop].stack_delta = -list[right].op1.intval;
         break;

    case PRINTF_: /* with expressions */
         list[listtop].type = type;
         list[listtop].left = left - listtop; /* has string */
         list[listtop].right = right - listtop; /* expressions */
         if ( list[right].op1.intval > 10 )
             kb_error(1266,"Limit of 10 arguments to PRINTF.\n",RECOVERABLE);
         list[left].type = PREPRINTF_;
         list[left].stack_delta = 0;
         list[listtop].stack_delta = -list[right].op1.intval-1;
         break;

    case EXPRLIST_:
         list[listtop].type = type;
         list[listtop].left = left - listtop; /*  expr */ 
         if ( right )
         { list[listtop].right = right - listtop; /* exprlist */
           list[listtop].op1.intval = list[right].op1.intval+1;
         }
         else list[listtop].op1.intval = 1; /* arg count */
         break;

    case DATAFILENAME_:
    case WARNING_MESSAGES_:
    case GET_TRANSFORM_EXPR_:
         list[listtop].type = type;
         list[listtop].stack_delta = 1;
         break;

    case HELP_KEYWORD:
         list[listtop].type = type;
         list[listtop].op1.string = 
            (char*)mycalloc(strlen(yytext)+1,sizeof(char));
         list[listtop].flags |= HAS_STRING;
         strcpy(list[listtop].op1.string,yytext);
         break;

    default:
      more_makenode(type,left,right);
      break;
    }

  if ( listtop > listmax-2 )
        kb_error(1336,"Expression too long",RECOVERABLE);

  list[listtop].line_no = line_no;
  list[listtop].file_no = file_no;
  return listtop++;
}


/*********************************************************************
*
*  Function:  subtree_swap()
*
*  Purpose:    Swap two adjacent subtrees at the end of the list
*              so they are in proper order for sequential execution.
*
*  Return:     Adjust index values in arguments.
*/

void subtree_swap(left,right)
int *left;  /* index of left subtree */
int *right; /* index of right subtree */
{ int n;
  struct treenode *temp;

  if ( *left && *right && (*left < *right ) )
  { int leftstart = *left, rightstart = *right;
    int leftsize,rightsize;
    for (;;)
    { if ( list[leftstart].left && list[leftstart].right )
      { if ( list[leftstart].left < list[leftstart].right )
          leftstart += list[leftstart].left;
        else leftstart += list[leftstart].right;
      }
      else if ( list[leftstart].left )
        leftstart += list[leftstart].left;
      else if ( list[leftstart].right )
        leftstart += list[leftstart].right;
      else break;
    }
    for (;;)
    { if ( list[rightstart].left && list[rightstart].right )
      { if ( list[rightstart].left < list[rightstart].right )
          rightstart += list[rightstart].left;
        else rightstart += list[rightstart].right;
      }
      else if ( list[rightstart].left )
        rightstart += list[rightstart].left;
      else if ( list[rightstart].right )
        rightstart += list[rightstart].right;
      else break;
    }
    if ( rightstart != *left + 1 )
     { sprintf(errmsg,
          "Internal error: Left: %d-%d  right: %d-%d subtrees not adjacent.\n",
          leftstart,*left,rightstart,*right);
        kb_error(1387,errmsg,COMMAND_ERROR);
     }      

    /* swap subtrees */
    leftsize = *left - leftstart + 1;
    rightsize = *right - rightstart +1;
    temp = (struct treenode*)temp_calloc(leftsize,sizeof(struct treenode));
    memcpy((char*)(temp),(char*)(list+leftstart),
        leftsize*sizeof(struct treenode));
    kb_memmove((char*)(list+leftstart),(char*)(list+rightstart),
        rightsize*sizeof(struct treenode));
    memcpy((char*)(list+leftstart+rightsize),(char*)(temp),
        leftsize*sizeof(struct treenode));
    temp_free((char*)temp);

    /* adjust subtree pointers */
    *left += rightsize;
    *right -= leftsize;
    /* kludge to adjust pointers to local variables */
    for ( n = leftstart ; n < leftstart+rightsize ; n++ )
    { if ( list[n].flags & LOCAL_VAR_REF )
         { if ( n + list[n].op1.intval + leftsize < leftstart )
          list[n].op1.intval += leftsize;
         }
      else if ( list[n].flags & LOCAL_VAR_REF_2 )
         { if ( n + list[n].op2.intval + leftsize < leftstart )
          list[n].op2.intval += leftsize;
         }
      else if ( list[n].flags & LOCAL_VAR_REF_3 )
         { if ( n + list[n].op3.intval[0] + leftsize < leftstart )
          list[n].op3.intval[0] += leftsize;
         }
      if ( ((list[n].type == BREAK_) || (list[n].type == CONTINUE_)) &&
         (n+leftsize+list[n].op1.intval < leftstart)  ) 
            list[n].op1.intval += leftsize;
    }
    for ( n = leftstart+rightsize ; n < leftstart+rightsize+leftsize ; n++ )
    { if ( list[n].flags & LOCAL_VAR_REF )
         { if ( n + list[n].op1.intval - rightsize < leftstart )
          list[n].op1.intval -= rightsize;
         }
      else if ( list[n].flags & LOCAL_VAR_REF_2 )
         { if ( n + list[n].op2.intval - rightsize < leftstart )
          list[n].op2.intval -= rightsize;
         }
      else if ( list[n].flags & LOCAL_VAR_REF_3 )
         { if ( n + list[n].op3.intval[0] - rightsize < leftstart )
          list[n].op3.intval[0] -= rightsize;
         }
      if ( ((list[n].type == BREAK_) || (list[n].type == CONTINUE_)) &&
         (n-rightsize+list[n].op1.intval < leftstart)  ) 
            list[n].op1.intval -= rightsize;
    } 
  }
}    

/**********************************************************************
*
* Function: more_makenode()
*
* Purpose: Overflow from makenode(), to keep code under 32K for Mac
*/

void more_makenode(type,left,right)
NTYPE type,left,right;
{ int n,etype;
  switch (type)
  {
    case QUOTATION_:
         list[listtop].type = type;
         list[listtop].op1.string = 
            (char*)mycalloc(strlen(yytext)+1,sizeof(char));
         list[listtop].flags |= HAS_STRING;
         strcpy(list[listtop].op1.string,yytext);
         list[listtop].stack_delta = 1;
         break;

    case DUMP_:
    case LOAD_:
    case PERMLOAD_:
         list[listtop].type = type;
         if ( left ) list[listtop].left = left - listtop;
         list[listtop].stack_delta = -1;
         break;

    case SINGLE_LETTER_:
         list[listtop].type = SINGLE_LETTER_;
         list[listtop].op1.intval = left;
         break;

    case SINGLE_REDEFD_:
         list[listtop].type = SINGLE_REDEFD_;
         list[listtop].op1.intval = left;
         break;

    case CMDLIST_:  
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         if ( right ) list[listtop].right = right - listtop;
         break;

    case COMMAND_BLOCK_:  
         list[listtop].type = COMMAND_BLOCK_;
         list[listtop].left = left - listtop;
         break;

    case REPEAT_INIT_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].stack_delta = -1;
         break;

    case REPEAT_:
         list[listtop].type = REPEAT_;
         list[listtop].left = left - listtop;
         list[listtop].right = right - listtop;
         list[left].op3.intval[0] = right+1-left; /* for skipping body */
         break;

    case EPRINT_:
         list[listtop].type = type;
         break;

      case COND_TEST_:
         list[listtop].type = COND_TEST_;
         list[listtop].left = left - listtop;
         /* inx will be target for false jmp */
         list[listtop].stack_delta = -1;
         break;

      case COND_EXPR_:
         list[listtop].type = COND_EXPR_;
         list[listtop].left = left - listtop;
         list[listtop].right = right - listtop;
         list[left].op1.intval = listtop - left;
         /* inx will be target to skip ELSE part */
         list[listtop].stack_delta = -1;  /* since one expression skipped */
         break;

      case COND_ELSE_: /* really the continue node at root of IF */
         list[listtop].type = COND_ELSE_;
         list[listtop].left = left - listtop;
         list[listtop].right = right - listtop;
         list[left].op1.intval = listtop - left;
         break;
      
    case SIZEOF_ATTR_: /* current attribute dimension */
         list[listtop].type = SIZEOF_ATTR_;
         list[listtop].op1.intval = left;  /* attr num */
         list[listtop].op2.intval = right; /* etype */
         list[listtop].stack_delta = 1;
         break;

    case SIZEOF_ARRAY_: /* current array total size */
         list[listtop].type = SIZEOF_ARRAY_;
         list[listtop].op1.intval = left;  /* array id */
         list[listtop].stack_delta = 1;
         break;

    case SIZEOF_STRING_: /* string length */
         list[listtop].type = SIZEOF_STRING_;
         list[listtop].op1.intval = left-listtop;  /* string expr */
         list[listtop].stack_delta = 1;
         break;

    case IS_DEFINED_:
         /* Parse-time evaluation. */
         if ( list[listtop-1].type != QUOTATION_ )
            kb_error(2235,"Is_defined() needs quoted string.",COMMAND_ERROR);
         list[listtop].type = type;
         list[listtop].left = left-listtop;
         list[listtop].op1.intval = 
            (identtype(list[listtop-1].op1.string)!=NEWIDENT_);  
         list[listtop].stack_delta = 0;
         break;

    case PUSHCONST:
         list[listtop].type = PUSHCONST;
         list[listtop].op1.real = real_val;
         list[listtop].stack_delta = 1;
         break;

    case PUSH_NAMED_QUANTITY:
    case PUSH_METHOD_INSTANCE_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = 1;
         break;

    case PUSHPI:
         list[listtop].type = type;
         list[listtop].op1.real = M_PI;
         list[listtop].stack_delta = 1;
         break;

    case PUSHE:
         list[listtop].type = type;
         list[listtop].op1.real = M_E;
         list[listtop].stack_delta = 1;
         break;

    case PUSHG:
         list[listtop].type = type;
         list[listtop].stack_delta = 1;
         break;

    case DATE_AND_TIME_:
         list[listtop].type = type;
         list[listtop].stack_delta = 1;
         break;

    case TOGGLEVALUE:
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* which toggle */
         list[listtop].stack_delta = 1;
         break;

    case GET_TORUS_PERIODS_:
    case GET_INVERSE_PERIODS_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].right = right - listtop;
         list[listtop].stack_delta = -1;
         break;

    case EXP:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = exp(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case LOG:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           if ( list[listtop].op1.real < 0.0 )
             kb_error(1425,"log of negative number.\n", RECOVERABLE );
           list[listtop].op1.real = log(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case ABS:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = fabs(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case SIN:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = sin(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case SINH:
         if ( is_constant(left) )
         { /* fold constants */
           REAL x;
           listtop--;
           x = exp(list[listtop].op1.real);
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = (x-1/x)/2;
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case COSH:
         if ( is_constant(left) )
         { /* fold constants */
           REAL x;
           listtop--;
           x = exp(list[listtop].op1.real);
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = (x+1/x)/2;
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case TANH:
         if ( is_constant(left) )
         { /* fold constants */
           REAL x;
           listtop--;
           x = exp(list[listtop].op1.real);
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = (x-1/x)/(x+1/x);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case ATANH:
         if ( is_constant(left) )
         { /* fold constants */
           REAL x;
           listtop--;
           x = list[listtop].op1.real;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = (log(x+1)-log(1-x))/2;
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case ASINH:
         if ( is_constant(left) )
         { /* fold constants */
           REAL x;
           listtop--;
           x = list[listtop].op1.real;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = log(x + sqrt(x*x+1));
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case ACOSH:
         if ( is_constant(left) )
         { /* fold constants */
           REAL x;
           listtop--;
           x = list[listtop].op1.real;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = 2*log(sqrt(x+1) + sqrt(x-1))-log(2.0);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case ASIN:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           if ( fabs(list[listtop].op1.real) > 1.0 )
             kb_error(1426,"asin argument out of bounds.\n", RECOVERABLE );
           list[listtop].op1.real = asin(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case COS:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = cos(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case ACOS:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           if ( fabs(list[listtop].op1.real) > 1.0 )
             kb_error(1427,"acos argument out of bounds.\n", RECOVERABLE );
           list[listtop].op1.real = acos(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case TAN:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = tan(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case ATAN:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = atan(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case SQR:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = list[listtop].op1.real*list[listtop].op1.real;
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case SQRT:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           if ( list[listtop].op1.real < 0.0 )
             kb_error(1428,"sqrt of negative number.\n", RECOVERABLE );
           list[listtop].op1.real = sqrt(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case ELLIPTICK:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = ellipticK(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case ELLIPTICE:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = ellipticE(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;


    case CEIL_:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = ceil(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case FLOOR_:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = floor(list[listtop].op1.real);
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case CHS:
         if ( is_constant(left) )
         { /* fold constants */
           listtop--;
           list[listtop].type = PUSHCONST;
           list[listtop].op1.real = -list[listtop].op1.real;
           break;
         }
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;


    case MAXIMUM_:
         if ( is_constant(right) && is_constant(left) )
         { /* fold constants */
           listtop -= 2;
           if ( (left != listtop) || (right != listtop+1) )
             kb_error(1429,"Constant folding not working.",RECOVERABLE);
           list[listtop].op1.real = 
             (list[left].op1.real > list[right].op1.real) ?
                list[left].op1.real : list[right].op1.real;
           list[listtop].type = PUSHCONST;
           list[listtop].stack_delta = 1;
           break;
         }
         list[listtop].right = right - listtop;
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         list[listtop].stack_delta = -1;
         break;

    case MINIMUM_:
         if ( is_constant(right) && is_constant(left) )
         { /* fold constants */
           listtop -= 2;
           if ( (left != listtop) || (right != listtop+1) )
              kb_error(1430,"Internal error: Constant folding not working.",
                 RECOVERABLE);
           list[listtop].op1.real = 
             (list[left].op1.real < list[right].op1.real) ?
                list[left].op1.real : list[right].op1.real;
           list[listtop].type = PUSHCONST;
           list[listtop].stack_delta = 1;
           break;
         }
         list[listtop].right = right - listtop;
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         list[listtop].stack_delta = -1;
         break;

    case ATAN2_:
    case INCOMPLETE_ELLIPTICF:
    case INCOMPLETE_ELLIPTICE:
         if ( is_constant(right) && is_constant(left) )
         { /* fold constants */
           listtop -= 2;
           if ( (left != listtop) || (right != listtop+1) )
              kb_error(1431,"Internal error: Constant folding not working.",
                 RECOVERABLE);
           switch(type)
           { case ATAN2_: list[listtop].op1.real = 
                atan2(list[left].op1.real,list[right].op1.real);
              break;
             case INCOMPLETE_ELLIPTICF: list[listtop].op1.real = 
                incompleteEllipticF(list[left].op1.real,list[right].op1.real);
				 break;
             case INCOMPLETE_ELLIPTICE: list[listtop].op1.real = 
                incompleteEllipticE(list[left].op1.real,list[right].op1.real);
              break;
           }
           list[listtop].type = PUSHCONST;
           list[listtop].stack_delta = 1;
           break;
         }
         list[listtop].right = right - listtop;
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         list[listtop].stack_delta = -1;
         break;

    case WRAP_COMPOSE_:
         if ( sym_compose == NULL )
           kb_error(2236,"No symmetry group defined for wrap_compose.\n",RECOVERABLE);
         list[listtop].right = right - listtop;
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         list[listtop].stack_delta = -1;
         break;

    case WRAP_INVERSE_:
         if ( sym_inverse == NULL )
           kb_error(2237,"No symmetry group defined for wrap_inverse.\n",RECOVERABLE);
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

    case POW:
      if ( is_constant(right) )
      { REAL p = list[right].op1.real;
        if ( floor(p) == p )
        { /* integer power */
          n = (int)p;
          if ( n == 0 )
          { kb_error(1432,"Exponent 0.  Did you mean this?\n",WARNING);
             listtop--;
             list[listtop].type = REPLACECONST;
             list[listtop].op1.real = 1.0;
             break;
          }
          else if ( n == 1 )
          { /* leave alone */
             kb_error(1433,"Exponent 1.  Did you mean this?\n",WARNING);
             listtop-=2;
             break;
          }
          else if ( n == 2 )
          { listtop--;
            if ( is_constant(left) )
            { listtop--;
              list[listtop].type = PUSHCONST;
              list[listtop].op1.real *= list[listtop].op1.real;
              list[listtop].stack_delta = 1;
            }
            else
            { list[listtop].type = SQR;
              list[listtop].left = left - listtop;
              list[listtop].stack_delta = 0;
            }
            break;
          }
          else if ( n == -1 )
          { listtop--;
            if ( is_constant(left) )
            { listtop--;
              list[listtop].op1.real = 1./list[listtop].op1.real;
            }
            else
            { list[listtop].type = INV;
              list[listtop].left = left - listtop;
              list[listtop].stack_delta = 0;
            }
            break;
          }

          /* general */
          if ( is_constant(left) )
          { /* fold constants */
             int k;
             REAL x;

             listtop -= 2;
             list[listtop].type = PUSHCONST;
             list[listtop].stack_delta = 1;
             if ( left != listtop )
               kb_error(1434,"Internal error: Constant folding not working.",
                 RECOVERABLE);
             x = list[left].op1.real;
             for ( k = 1 ; k < abs(n) ; k++ )
               list[left].op1.real *= x;
             if ( n < 0 )
               list[left].op1.real = 1/list[left].op1.real;
             break;
          }
          listtop--; /* pop constant power */
          list[listtop].left = left - listtop;
          list[listtop].op1.intval = n;
          list[listtop].type = INTPOW;
          break;
        }
      }
      if ( is_constant(right) && is_constant(left) )
        { /* fold constants */
          listtop -= 2;
          if ( (left != listtop) || (right != listtop+1) )
             kb_error(1435,"Internal error: Constant folding not working.",
                RECOVERABLE);
          list[listtop].op1.real = 
              pow(list[left].op1.real,list[right].op1.real);
          list[listtop].type = PUSHCONST;
          list[listtop].stack_delta = 1;
          break;
        }
      list[listtop].right = right - listtop;
      list[listtop].left = left - listtop;
      list[listtop].type = type;
      list[listtop].stack_delta = -1;
      break;

    case '+':
        if ( is_constant(right) && is_constant(left) )
        { /* fold constants */
          
          listtop -= 2;
          if ( (left != listtop) || (right != listtop+1) )
             kb_error(1436,"Internal error: Constant folding not working.",
                 RECOVERABLE);
          list[listtop].op1.real = list[left].op1.real + list[right].op1.real;
          list[listtop].type = PUSHCONST;
          list[listtop].stack_delta = 1;
        }
        else if ( is_constant(right) && (list[right].op1.real==0.0) )
          listtop -= 2; /* just leave left summand */
        else if ( is_constant(left) && (list[left].op1.real==0.0) )
        { subtree_swap(&left,&right);
          listtop -= 2;
        }
        else
        {
          list[listtop].right = right - listtop;
          list[listtop].left = left - listtop;
          list[listtop].type = PLUS;
          list[listtop].stack_delta = -1;
        }
        break;

    case '-':
        if ( is_constant(right) && is_constant(left) )
        { /* fold constants */
          
          listtop -= 2;
          if ( (left != listtop) || (right != listtop+1) )
             kb_error(1437,"Internal error: Constant folding not working.",
                RECOVERABLE);
          list[listtop].op1.real = list[left].op1.real - list[right].op1.real;
          list[listtop].type = PUSHCONST;
          list[listtop].stack_delta = 1;
        }
        else if ( is_constant(right) && (list[right].op1.real==0.0) )
          listtop -= 2; /* just leave left summand */
        else if ( is_constant(left) && (list[left].op1.real==0.0) )
        { subtree_swap(&left,&right);
          listtop -= 1;
          list[listtop].left = right - listtop;
          list[listtop].type = CHS;
          list[listtop].stack_delta = 0;
        }
        else
        {
          list[listtop].right = right - listtop;
          list[listtop].left = left - listtop;
          list[listtop].type = MINUS;
          list[listtop].stack_delta = -1;
        }
        break;

    case '=':  /* equality as low priority minus */
        if ( is_constant(right) && is_constant(left) )
        { /* fold constants */
          
          listtop -= 2;
          if ( (left != listtop) || (right != listtop+1) )
             kb_error(1438,"Internal error: Constant folding not working.",
                RECOVERABLE);
          list[listtop].op1.real = list[left].op1.real - list[right].op1.real;
          list[listtop].type = PUSHCONST;
          list[listtop].stack_delta = 1;
        }
        else if ( is_constant(right) && (list[right].op1.real==0.0) )
          listtop -= 2; /* just leave left summand */
        else
        {
          list[listtop].right = right - listtop;
          list[listtop].left = left - listtop;
          list[listtop].type = EQUATE;
          list[listtop].stack_delta = -1;
        }
        break;

    case '*':
        if ( is_constant(right) && is_constant(left) )
        { /* fold constants */
          
          listtop -= 2;
          if ( (left != listtop) || (right != listtop+1) )
             kb_error(1439,"Internal error: Constant folding not working.",
               RECOVERABLE);
          list[listtop].op1.real = list[left].op1.real * list[right].op1.real;
          list[listtop].type = PUSHCONST;
          list[listtop].stack_delta = 1;
        }
        else if ( is_constant(left) && (list[left].op1.real==0.0) )
         { 
           listtop = left; /* just leave 0 */
         }
        else if ( ( is_constant(right) && (list[right].op1.real==0.0) ) )
         { subtree_swap(&left,&right);
           listtop = right; /* just leave 0 */
         }
        else if ( is_constant(right) && (list[right].op1.real==1.0) )
           listtop -= 2; /* just leave left multiplicand */
        else if ( is_constant(left) && (list[left].op1.real==1.0) )
        { subtree_swap(&left,&right);
          listtop -= 2;
        }
        else
        {
          list[listtop].right = right - listtop;
          list[listtop].left = left - listtop;
          list[listtop].type = TIMES;
          list[listtop].stack_delta = -1;
        }
        break;

    case '/':
        if ( is_constant(right) && is_constant(left) )
        { /* fold constants */
          
          listtop -= 2;
          if ( (left != listtop) || (right != listtop+1) )
             kb_error(1440,"Internal error: Constant folding not working.",
               RECOVERABLE);
          if ( list[right].op1.real == 0.0 ) 
             kb_error(1441,"Divide by zero.",RECOVERABLE);
          list[listtop].op1.real = list[left].op1.real / list[right].op1.real;
          list[listtop].type = PUSHCONST;
        }
        else if  ( is_constant(left) && (list[left].op1.real==0.0) ) 
        { listtop = left; /* just leave 0 */
          list[listtop].op1.real = 0.0;
          list[listtop].type = PUSHCONST;
          list[listtop].stack_delta = 1;
        }
        else if ( is_constant(right) && (list[right].op1.real==0.0) )
           kb_error(1442,"Divide by zero.",RECOVERABLE);
        else if ( is_constant(right) && (list[right].op1.real==1.0) )
           listtop -= 2; /* just leave numerator */
        else
        {
          list[listtop].right = right - listtop;
          list[listtop].left = left - listtop;
          list[listtop].type = DIVIDE;
          list[listtop].stack_delta = -1;
        }
        break;

    case '%':
        if ( is_constant(right) && is_constant(left) )
        { /* fold constants */
          listtop -= 2;
          if ( (left != listtop) || (right != listtop+1) )
             kb_error(1443,"Internal error: Constant folding not working.",
               RECOVERABLE);
          list[listtop].op1.real = list[left].op1.real - 
              floor(list[left].op1.real / list[right].op1.real)
              *list[right].op1.real;
          list[listtop].type = PUSHCONST;
          list[listtop].stack_delta = 1;
        }
        else
        { list[listtop].right = right - listtop;
          list[listtop].left = left - listtop;
          list[listtop].type = REALMOD;
          list[listtop].stack_delta = -1;
        }
        break;

    case IMOD_:
        if ( is_constant(right) && is_constant(left) )
        { /* fold constants */
          listtop -= 2;
          if ( (left != listtop) || (right != listtop+1) )
             kb_error(1444,"Internal error: Constant folding not working.",
               RECOVERABLE);
          list[listtop].op1.real = floor(list[left].op1.real) - 
              floor(floor(list[left].op1.real)/floor(list[right].op1.real))
              *floor(list[right].op1.real);
          list[listtop].type = PUSHCONST;
          list[listtop].stack_delta = 1;
        }
        else
        { list[listtop].right = right - listtop;
          list[listtop].left = left - listtop;
          list[listtop].type = IMOD_;
          list[listtop].stack_delta = -1;
        }
        break;

    case IDIV_:
        if ( is_constant(right) && is_constant(left) )
        { /* fold constants */
          listtop -= 2;
          if ( (left != listtop) || (right != listtop+1) )
             kb_error(1445,"Internal error: Constant folding not working.",
               RECOVERABLE);
          list[listtop].op1.real = 
              (int)(list[left].op1.real)/(int)(list[right].op1.real);
          list[listtop].type = PUSHCONST;
          list[listtop].stack_delta = 1;
        }
        else
        { list[listtop].right = right - listtop;
          list[listtop].left = left - listtop;
          list[listtop].type = IDIV_;
          list[listtop].stack_delta = -1;
        }
        break;

    case PUSHPARAM:
        if ( maxp == 0 )
          kb_error(1446,"Constant expression required.\n",EXPRESSION_ERROR);
        list[listtop].type = PUSHPARAM;
        list[listtop].op1.intval = n = coord_num-1;
        if ( (n < 0) || (n >= maxp) )
        {
          sprintf(errmsg,"Coordinate number too high: %d\n",n+1);
          kb_error(1447,errmsg,EXPRESSION_ERROR);
        }
        list[listtop].stack_delta = 1;
        break;

    case USERFUNC:
         list[listtop].type = USERFUNC;
         list[listtop].op1.intval = (NTYPE)int_val-1;
         if ( int_val < 1 || (userfunc[int_val-1] == NULL) )
         { sprintf(errmsg,"Invalid user function number: %d\n",int_val);
           kb_error(1448,errmsg,EXPRESSION_ERROR);
         }     
         list[listtop].stack_delta = 1;
         break;

    case VIEW_MATRIX_:
         list[listtop].right = right - listtop;
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         list[listtop].stack_delta = -1;
         break;

    case GET_INTERNAL_:
         list[listtop].type = GET_INTERNAL_;
         list[listtop].op1.intval = left;
         list[listtop].stack_delta = 1;
         break;

    case PUSHQPRESSURE_:
    case PUSHQTARGET_:
    case PUSHQVALUE_:
    case PUSHQMODULUS_:
    case PUSHQTOLERANCE_:
    case PUSHMMODULUS_:
    case PUSHQVOLCONST_:
    case PUSHQPARAMETER_1_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;
         list[listtop].stack_delta = 1;
         break;

    case PUSHMVALUE_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;
         if ( reading_comp_quant_flag ) 
            METH_INSTANCE(left)->flags |= Q_COMPOUND;
         list[listtop].stack_delta = 1;
         break;

    case VIEW_MATRIX_LVALUE_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].right = right - listtop;
         break;
    case SET_VIEW_MATRIX_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].right = right - listtop;
         list[listtop].op2.intval = assigntype;
         list[listtop].stack_delta = -3;
         break;
         
    case SET_QMODULUS_:
    case SET_QTOLERANCE_:
    case SET_MMODULUS_:
    case SET_QVOLCONST_:
    case SET_QTARGET_:
    case SET_QPARAMETER_1_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].op1.intval = right;
         list[listtop].op2.intval = assigntype;
         list[listtop].stack_delta = -1;
         break;

    case DYNAMIC_LOAD_FUNC_:
         list[listtop].type = DYNAMIC_LOAD_FUNC_;
         list[listtop].op1.funcptr = globals(left)->value.funcptr;
         list[listtop].op2.intval = left;
         list[listtop].stack_delta = 1;
         break;

    case PUSHDELTA_:
    case PUSH_PARAM_SCALE:
         list[listtop].type = type;
         list[listtop].op1.intval = left;
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = 1;
         break;

    case PUSH_PARAM_EXTRA_:  
         if ( (right != V_VELOCITY_ATTR) && (right != V_FORCE_ATTR) )
           kb_error(2473,"Illegal attribute of variable.\n",COMMAND_ERROR);
         list[listtop].type = type;
         list[listtop].op1.intval = left; /* which parameter */
         list[listtop].op2.intval = right; /* which attribute */
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = 1;
         break;

    case PUSHGLOBAL_:
       { struct locallist_t *localbase = local_scope_bases[local_nest_depth-1];
         struct global *g = globals(left);
         list[listtop].type = PUSHGLOBAL_;
         list[listtop].op1.intval = left;
         list[listtop].flags |= EPHEMERAL;
         if ( (g->flags & METHOD_NAME) && reading_comp_quant_flag )
         { struct method_instance *mi=METH_INSTANCE(left);
           mi->flags |= Q_COMPOUND;
           quant_flags[basic_gen_methods[mi->gen_method].type] 
            |= GEN_QUANT(cur_quant)->flags &
                     (Q_ENERGY|Q_FIXED|Q_INFO|Q_CONSERVED);
           attach_method_num(cur_quant,left);
         }
         list[listtop].stack_delta = 1;
         break;
       }
     
    case PUSH_PERM_GLOBAL_:
    case PERM_STRINGGLOBAL_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;
         list[listtop].stack_delta = 1;
         break;

    case PERM_PROCEDURE_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;
         break;

    case STRINGGLOBAL_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = 1;
         break;

    case PROCEDURE_:
         list[listtop].type = type;
         list[listtop].op1.intval = left;
         list[listtop].flags |= EPHEMERAL;
         break;

    case ON_CONSTRAINT_:
    case HIT_CONSTRAINT_:
    case ON_BOUNDARY_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = 0;
         break;

    case ON_CONSTRAINT_NAME:
    case HIT_CONSTRAINT_NAME:
         list[listtop].type = type;
         list[listtop].op3.intval[0] = globals(left)->value.cnum; 
                                     /* actual constraint number */
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = 1;
         break;

    case ON_BOUNDARY_NAME:
         list[listtop].type = type;
         list[listtop].op3.intval[0] = globals(left)->value.bnum;  /* actual boundary number */
         list[listtop].flags |= EPHEMERAL;
         list[listtop].stack_delta = 1;
         break;


    case INDEXED_COORD_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;   
         if ( list[left].op1.intval != 1 )
           kb_error(2568,"Coordinate can have only one index.\n",COMMAND_ERROR);
         break;

    case PRINT_VERTEXNORMAL_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;    /* element */
         break;

    case GET_VERTEXNORMAL_:
         list[listtop].type = type;
         list[listtop].left = left - listtop;    /* index set */
         if ( list[left].op1.intval != 1 )
           kb_error(2569,"Vertexnormal can have only one index.\n",
              COMMAND_ERROR);
         break;

     case INDEXED_ATTRIBUTE:  /* get extra attribute value */
            { struct extra *ex;
              etype = (unsigned)right >> TYPESHIFT;
              right = right & ~TYPEMASK;
              ex = EXTRAS(etype) + right;
              list[listtop].type = GET_EXTRA_ATTR_;
              list[listtop].op2.intval = etype;
              list[listtop].op3.intval[0] = right;  /* which extra */
              list[listtop].left = left - listtop;  /* index */
              list[listtop].stack_delta = 1 - ex->adim;
              if ( ex->adim != list[left].op1.intval )
              { sprintf(errmsg,"Attribute %s must have %d indices.\n",
                 ex->name,ex->adim);
                kb_error(2513,errmsg,COMMAND_ERROR);
              }
              list[listtop].flags |= EPHEMERAL;
            }
            break;

     case PRINT_ATTR_ARRAY_:  /* print element attribute array or slice */
            { struct extra *ex;
              int exnum;
              etype = int_val >> TYPESHIFT;
              exnum = int_val & ~TYPEMASK;
              ex = EXTRAS(etype) + exnum;
              list[listtop].type = type;
              list[listtop].op1.intval = right ? list[right].op1.intval : 0; /* indices */
              list[listtop].op2.intval = etype;
              list[listtop].op3.intval[0] = exnum;  /* which extra */
              list[listtop].left = left - listtop;  /* element */
              if ( right ) list[listtop].right = right - listtop;  /* index */
              if ( ex->adim < list[right].op1.intval )
              { sprintf(errmsg,"Attribute %s must have at most %d indices.\n",
                 ex->name,ex->adim);
                kb_error(2644,errmsg,COMMAND_ERROR);
              }
              list[listtop].flags |= EPHEMERAL;
              list[listtop].stack_delta = -list[listtop].op1.intval;
            }
            break;

     case QUALIFIED_ATTRIBUTE:
          list[listtop].type = QUALIFIED_ATTRIBUTE;
          list[listtop].left = left-listtop;
          list[listtop].right = right-listtop;
          etype = list[left].op1.intval; /* element type */
          /* element type error checking */
          check_element_type(list[right].type,etype);
          break;
  
     case ATTRIBUTE:
        list[listtop].type = (NTYPE)left;
        /* some special treatments */
        switch ( left )
        { 
          case COORD_: list[listtop].op2.intval = right-1;
             if ( right > SDIM )
             kb_error(2475,"Coordinate dimension exceeds space dimension.\n",
                 RECOVERABLE);
             break;
          case PARAM_: list[listtop].op2.intval = right-1;
             break;
          case GET_INSTANCE_:
          case GET_QUANTITY_: list[listtop].op2.intval = right;
             break;
          case GET_EXTRA_ATTR_:
            { struct extra *ex;
              etype = (unsigned)right >> TYPESHIFT;
              ex = EXTRAS(etype) + (right & ~TYPEMASK);
              if ( ex->adim > 0 ) 
              { sprintf(errmsg,
                  "\"%s\" is an indexed attribute; index is missing.\n",
                      ex->name);
                kb_error(2634,errmsg,Q_ERROR);
              }
              list[listtop].op2.intval = etype;
              list[listtop].op3.intval[0] = (right & ~TYPEMASK);
              list[listtop].flags |= EPHEMERAL;
              list[listtop].stack_delta = 1-ex->adim;
            }
            break;
         }
         if ( left != GET_EXTRA_ATTR_ )
            list[listtop].stack_delta = 1;
         break;

     case SET_ATTRIBUTE_:
     case SET_ATTRIBUTE_L:
     case SET_ATTRIBUTE_A:
          if ( elsym == NULL )
              kb_error(1477,"Don't have element for attribute to apply to.\n",
                    COMMAND_ERROR);
          list[listtop].type = type;
          if ( left ) list[listtop].left = left-listtop;  /* value */
          if ( right )
          { /* index; check which attributes are legal */
            list[listtop].right = right-listtop;
            switch ( attr_kind )
             { case SET_EXTRA_ATTR_: 
                 list[listtop].flags |= EPHEMERAL;
                 break;
               case SET_COORD_1: case SET_PARAM_1: break;
               default: kb_error(1478,"Attribute is not indexed.\n",COMMAND_ERROR);
             }
          }
          list[listtop].op1.intval = elsym->localnum; 
          list[listtop].op2.intval = attr_kind;
         /* to node */
          /* error checking on attributes and element types */
          etype = elsym->type;
          switch ( attr_kind )
          {
            case SET_EXTRA_ATTR_:
            { struct extra *ex;
              ex = EXTRAS(etype);
              for ( n = 0 ; n < web.skel[etype].extra_count ; n++,ex++ )
                if ( stricmp(ex->name,set_extra_name) == 0 ) break;
              if ( n == web.skel[etype].extra_count )
              { sprintf(errmsg,"Invalid extra attribute name '%s'.\n",
                set_extra_name);
                kb_error(1479,errmsg,COMMAND_ERROR);
              }
              list[listtop].op3.intval[0] = (etype << ESHIFT) + n;
              if ( !(ex->flags & DIMENSIONED_ATTR) && right )
              { sprintf(errmsg,"Cannot use index with attribute '%s'.\n",ex->name);
                kb_error(2499,errmsg,COMMAND_ERROR);
              }
              if ( (ex->flags & DIMENSIONED_ATTR) && !right )
              { sprintf(errmsg,"Must use index with attribute '%s'.\n",ex->name);
                kb_error(1481,errmsg,COMMAND_ERROR);
              }
              if ( right && (list[right].op1.intval != ex->adim) )
              { sprintf(errmsg,"Attribute '%s' has %d indexes.\n",
                     ex->name,ex->adim);
                kb_error(2498,errmsg,COMMAND_ERROR);
              }
            list[listtop].stack_delta = -1-ex->adim;
            }
            break;
          case SET_WRAP_:
            if ( etype != EDGE )
            kb_error(2239,"Wrap only for edges.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_COORD_: case SET_COORD_1: case SET_COORD_2:
          case SET_COORD_3: case SET_COORD_4: case SET_COORD_5:
          case SET_COORD_6: case SET_COORD_7: case SET_COORD_8:
            if ( attr_kind-SET_COORD_1 >= SDIM )
             kb_error(2543,"Coordinate dimension exceeds space dimension.\n",
               RECOVERABLE);
            if ( etype != VERTEX )
              kb_error(1482,"Coordinates only for vertices.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_PARAM_: case SET_PARAM_1: case SET_PARAM_2:
          case SET_PARAM_3: case SET_PARAM_4:
            if ( etype != VERTEX )
              kb_error(1483,"Boundary parameters only for vertices.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_TAG_:
            if ( etype != FACET )
              kb_error(1484,"Tag only for facets now.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_OPACITY_:
            if ( etype != FACET )
              kb_error(1485,"Opacity only for facets.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_FRONTCOLOR_:
          case SET_BACKCOLOR_:
            if ( (etype != FACET) )
              kb_error(1304,"Front or back color only for facets.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_FRONTBODY_:
          case SET_BACKBODY_:
            if ( web.representation == STRING )
            { if ( (etype != FACET) && (etype != EDGE) )
                kb_error(1486,
     "Frontbody or backbody only for facets or edges in string model.\n",
                COMMAND_ERROR);
            }
            else if (etype != FACET) 
               kb_error(1308,"Frontbody or backbody only for facets.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_COLOR_:
            if ( !((etype == FACET) || (etype == EDGE)) )
               kb_error(1487,"Color only for edges or facets.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_VOLCONST_:
            if ( etype != BODY )
             kb_error(1488,"Volconst only for bodies.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_TARGET_:
            if ( etype != BODY )
             kb_error(1489,"Target volume only for bodies.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_VOLUME_:
            if ( etype != BODY )
             kb_error(1490,"Target volume only for bodies.\n",COMMAND_ERROR);
            kb_error(1491,
            "Volume is read-only. Setting TARGET instead.\n",
              WARNING);
            list[listtop].op2.intval = SET_TARGET_;
            list[listtop].stack_delta = -1;
            break;
          case SET_PRESSURE_:
            if ( etype != BODY )
            kb_error(1492,"Pressure only for bodies.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_PHASE_:
          case SET_CONSTRAINT_: 
          case SET_ORIGINAL_: 
            list[listtop].stack_delta = -1;
            break;
          case SET_DENSITY_:
            if ( etype == VERTEX )
            kb_error(1493,"No density for vertices.\n",COMMAND_ERROR);
            list[listtop].stack_delta = -1;
            break;
          case SET_FIXED_:
            if ( etype == FACETEDGE )
            kb_error(2527,"No fixedness for facetedges.\n",COMMAND_ERROR);
            if ( etype == BODY )
            kb_error(2528,"Use 'set body target ...' to fix volume.\n",
               COMMAND_ERROR);
            break;
          case SET_NO_DISPLAY_:
            if ( etype != FACET )
            kb_error(1495,"No_display is only for facets.\n",COMMAND_ERROR);
            break;
          case SET_NO_REFINE_:
            if ( etype == BODY || etype == VERTEX )
            kb_error(1496,"No no_refine for vertices or bodies.\n",
                COMMAND_ERROR);
          case SET_NONCONTENT_:
             if ( ((web.representation == STRING) && (etype != EDGE)) || 
                  ((web.representation != STRING) && (etype != FACET)) )
                kb_error(2903,"Noncontent only applies to edges or facets.\n",
                   COMMAND_ERROR);
            break;
         }
         /* error checking for arithmetic assigns */
         if ( type == SET_ATTRIBUTE_A  )
         { switch ( attr_kind )
           { case SET_FIXED_: case SET_TRIPLE_PT_: case SET_TETRA_PT_: 
             case SET_AXIAL_POINT_:
                kb_error(2241,"Cannot assign value to this attribute with :=.\n",
                 COMMAND_ERROR);
           }
           if ( assigntype != ASSIGN_ ) 
           switch ( attr_kind )
           { case SET_ORIENTATION_: case SET_PHASE_: case SET_OPACITY_:
             case SET_CONSTRAINT_: case SET_ORIGINAL_: case SET_COLOR_:
             case SET_FRONTCOLOR_: case SET_BACKCOLOR_: case SET_WRAP_:
             case SET_TAG_: case SET_FRONTBODY_: case SET_BACKBODY_:
             kb_error(2242,
               "Cannot use arithmetic assign with this attribute.\n",
             COMMAND_ERROR);
           } 
         }
         break;

     case EQ_:
     case NE_:
     case GE_:
     case LE_:
     case GT_:
     case LT_:
         list[listtop].right = right - listtop;
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         list[listtop].stack_delta = -1;
         break;

     case AND_:  /* for short-circuit evaluation, have to move test */
     case OR_:    /* node between operands */
      { int top;
         top = listtop;
         list[listtop].type = type;
         subtree_swap(&right,&top);
         list[top].right = right - top;
         list[top].left = left - top;
         list[top].op1.intval = listtop - top;
         listtop++;
         list[listtop].type = CONJUNCTION_END;
         list[listtop].left = top - listtop;
         list[listtop].stack_delta = -1;
         break;
      }
     case NOT_:
         list[listtop].left = left - listtop;
         list[listtop].type = type;
         break;

     default:
        sprintf(errmsg,"Internal error: Unknown MAKENODE type %s\n",tokname(type));
        kb_error(1329,errmsg,RECOVERABLE);

      }
}

/******************************************************************
*
*  Function: is_constant()
*
*  Purpose:  See if tree node type is a constant value.
*/

int is_constant(node)
int node;
{
  switch(list[node].type)
     {  case PUSHPI:
        case PUSHE:
        case PUSHCONST:
           return 1;
        default:
           return 0;
     }
}

/**********************************************************************
*
* Function: check_element_type()
*
* purpose: see if attribute is legal for element 
*/

void check_element_type(attrib,etype)
int attrib;
int etype;
{
  switch ( attrib )
  {  
    case GET_TAG_:
     if ( etype != FACET )
         kb_error(1455,"Tag only for facets now.\n",COMMAND_ERROR);
     break;

    case GET_MIDV_:
     if ( etype != EDGE )
         kb_error(1456,"Midv only for quadratic edges.\n",COMMAND_ERROR);
            break;

    case GET_SQ_MEAN_CURV_:
     if ( etype != VERTEX )
         kb_error(1457,"Square mean curvature only for vertices.\n",
         COMMAND_ERROR);
     break;

    case GET_FRONTBODY_:
    case GET_BACKBODY_:
     if ( (etype != FACET) && !((etype==EDGE)&&(web.representation==STRING)))
         kb_error(1297,"Frontbody or backbody only for facets, or string model edges.\n",
           COMMAND_ERROR);
      break;

    case GET_FRONTCOLOR_:
    case GET_BACKCOLOR_:
     if ( (etype != FACET) )
         kb_error(1458,"Frontcolor or backcolor only for facets.\n",COMMAND_ERROR);
      break;

    case GET_COLOR_:
     if ( !((etype == FACET) || (etype == EDGE)) )
         kb_error(1459,"Color only for edges or facets.\n",COMMAND_ERROR);
      break;

    case BARE_:
      if ( (etype == BODY) || (etype == FACET) )
          kb_error(1460,"No bareness for facets or bodies.\n",COMMAND_ERROR);
      break;

    case GET_SHOW_:
     if ( (etype != EDGE) && (etype != FACET) )
        kb_error(2243,"\"Show\" only for edges or facets.\n",COMMAND_ERROR);
     break;


    case GET_ORIENTATION_:
     if ( (etype != EDGE) && (etype != FACET) )
        kb_error(1461,"Orientation only for edges or facets.\n",COMMAND_ERROR);
     break;

    case GET_LENGTH_:
     if ( etype != EDGE )
         kb_error(1462,"Length only for edges.\n",COMMAND_ERROR);
     break;

    case GET_VERTEXNORMAL_:
     if ( etype != VERTEX )
         kb_error(2244,"Facetnormal only for edges.\n",COMMAND_ERROR);
      break;

    case GET_FIXEDVOL_:
     if ( etype != BODY )
         kb_error(1463,"Fixedvol only for bodies.\n",COMMAND_ERROR);
     break;

    case GET_WRAP_:
     if ( etype != EDGE )
         kb_error(1464,"Wrap only for edges.\n",COMMAND_ERROR);
     break;

    case GET_DIHEDRAL_:
     if ( web.representation == SOAPFILM )
     { if (etype != EDGE) 
          kb_error(1465,"Dihedral only for edges.\n",COMMAND_ERROR);
     }
     else if ( web.representation == STRING )
     { if (etype != VERTEX) 
         kb_error(1466,"Dihedral only for vertices.\n",COMMAND_ERROR);
     }
     else 
      kb_error(1467,
      "Dihedral defined only for STRING and SOAPFILM models.\n",COMMAND_ERROR);
     break;

    case GET_AREA_:
     if ( etype != FACET )
         kb_error(1468,"Area only for facets.\n",COMMAND_ERROR);
     break;

    case GET_EDGE_:
     if ( etype != FACETEDGE )
         kb_error(1469,"Edge only for facetedges.\n",COMMAND_ERROR);
     break;

    case GET_FACET_:
     if ( etype != FACETEDGE )
         kb_error(1470,"Facet only for facetedges.\n",COMMAND_ERROR);
     break;

    case GET_TARGET_:
     if ( etype != BODY )
         kb_error(1471,"Target only for bodies.\n",COMMAND_ERROR);
     break;

    case GET_VOLCONST_:
     if ( etype != BODY )
         kb_error(1472,"Volconst only for bodies.\n",COMMAND_ERROR);
     break;

    case GET_VOLUME_:
     if ( etype != BODY )
         kb_error(1473,"Volume only for bodies.\n",COMMAND_ERROR);
     break;

    case GET_DENSITY_:
     if ( etype == VERTEX )
         kb_error(1474,"No density for vertices.\n",COMMAND_ERROR);
     break;

    case SET_NO_REFINE_:
     if ( etype == BODY || etype == VERTEX )
         kb_error(1494,"No no_refine for vertices or bodies.\n",
         COMMAND_ERROR);
         break;

    case SET_NO_DISPLAY_:
     if ( etype != FACET )
         kb_error(1320,"No_display is only for facets.\n",
         COMMAND_ERROR);
         break;
      }
}
