/****************************************************************\
*                                                                *
*  Comparison : A module for pairwise sequence comparisons       *
*                                                                *
*  Guy St.C. Slater..   mailto:guy@ebi.ac.uk                     *
*  Copyright (C) 2000-2006.  All Rights Reserved.                *
*                                                                *
*  This source code is distributed under the terms of the        *
*  GNU Lesser General Public License. See the file COPYING       *
*  or http://www.fsf.org/copyleft/lesser.html for details        *
*                                                                *
*  If you use this code, please keep this notice intact.         *
*                                                                *
\****************************************************************/

#include "comparison.h"

/**/

static Comparison_Param *Comparison_Param_create_without_mirror(
                                          Alphabet_Type query_type,
                                          Alphabet_Type target_type,
                                          HSP_Param *dna_hsp_param,
                                          HSP_Param *protein_hsp_param,
                                          HSP_Param *codon_hsp_param){
    register Comparison_Param *param = g_new(Comparison_Param, 1);
    param->ref_count = 1;
    param->query_type = query_type;
    param->target_type = target_type;
    param->dna_hsp_param = dna_hsp_param
                         ? HSP_Param_share(dna_hsp_param)
                         : NULL;
    param->protein_hsp_param = protein_hsp_param
                             ? HSP_Param_share(protein_hsp_param)
                             : NULL;
    param->codon_hsp_param = codon_hsp_param
                           ? HSP_Param_share(codon_hsp_param)
                           : NULL;
    param->mirror = NULL;
    return param;
    }

Comparison_Param *Comparison_Param_create(Alphabet_Type query_type,
                                          Alphabet_Type target_type,
                                          HSP_Param *dna_hsp_param,
                                          HSP_Param *protein_hsp_param,
                                          HSP_Param *codon_hsp_param){
    register Comparison_Param *param
        = Comparison_Param_create_without_mirror(
                            query_type, target_type,
                            dna_hsp_param, protein_hsp_param,
                            codon_hsp_param);
    param->mirror = Comparison_Param_create_without_mirror(
                            target_type, query_type,
                            dna_hsp_param, protein_hsp_param,
                            codon_hsp_param);
    param->mirror->mirror = param;
    return param;
    }

static gint Comparison_Param_ref_count(Comparison_Param *param){
    g_assert(param);
    g_assert(param->mirror);
    return param->ref_count + param->mirror->ref_count;
    }

static void Comparison_Param_destroy_without_mirror(
            Comparison_Param *param){
    g_assert(param);
    param->mirror = NULL;
    if(param->dna_hsp_param)
        HSP_Param_destroy(param->dna_hsp_param);
    if(param->protein_hsp_param)
        HSP_Param_destroy(param->protein_hsp_param);
    if(param->codon_hsp_param)
        HSP_Param_destroy(param->codon_hsp_param);
    g_free(param);
    return;
    }

void Comparison_Param_destroy(Comparison_Param *param){
    g_assert(param);
    param->ref_count--;
    if(Comparison_Param_ref_count(param) > 1) /* Allow for mirror */
        return;
    Comparison_Param_destroy_without_mirror(param->mirror);
    Comparison_Param_destroy_without_mirror(param);
    return;
    }

Comparison_Param *Comparison_Param_share(Comparison_Param *param){
    g_assert(param);
    param->ref_count++;
    return param;
    }

Comparison_Param *Comparison_Param_swap(Comparison_Param *param){
    register Comparison_Param *mirror
           = Comparison_Param_share(param->mirror);
    g_assert(param);
    g_assert(param->mirror);
    Comparison_Param_destroy(param);
    return mirror;
    }

/**/

Comparison *Comparison_create(Comparison_Param *param,
                              Sequence *query, Sequence *target){
    register Comparison *comparison = g_new(Comparison, 1);
    g_assert(param);
    g_assert(query);
    g_assert(target);
    comparison->ref_count = 1;
    comparison->param = Comparison_Param_share(param);
    comparison->query = Sequence_share(query);
    comparison->target = Sequence_share(target);
    comparison->dna_hspset = param->dna_hsp_param
                           ? HSPset_create(query, target,
                                           param->dna_hsp_param)
                           : NULL;
    comparison->protein_hspset = param->protein_hsp_param
                               ? HSPset_create(query, target,
                                               param->protein_hsp_param)
                               : NULL;
    comparison->codon_hspset = param->codon_hsp_param
                             ? HSPset_create(query, target,
                                             param->codon_hsp_param)
                             : NULL;
    return comparison;
    }

Comparison *Comparison_share(Comparison *comparison){
    comparison->ref_count++;
    return comparison;
    }

void Comparison_destroy(Comparison *comparison){
    g_assert(comparison);
    if(--comparison->ref_count)
        return;
    Comparison_Param_destroy(comparison->param);
    Sequence_destroy(comparison->query);
    Sequence_destroy(comparison->target);
    if(comparison->dna_hspset)
        HSPset_destroy(comparison->dna_hspset);
    if(comparison->protein_hspset)
        HSPset_destroy(comparison->protein_hspset);
    if(comparison->codon_hspset)
        HSPset_destroy(comparison->codon_hspset);
    g_free(comparison);
    return;
    }

gboolean Comparison_has_hsps(Comparison *comparison){
    if(comparison->dna_hspset)
        if(!HSPset_is_empty(comparison->dna_hspset))
            return TRUE;
    if(comparison->protein_hspset)
        if(!HSPset_is_empty(comparison->protein_hspset))
            return TRUE;
    if(comparison->codon_hspset)
        if(!HSPset_is_empty(comparison->codon_hspset))
            return TRUE;
    return FALSE;
    }

void Comparison_finalise(Comparison *comparison){
    if(comparison->dna_hspset)
        comparison->dna_hspset = HSPset_finalise(comparison->dna_hspset);
    if(comparison->protein_hspset)
        comparison->protein_hspset = HSPset_finalise(comparison->protein_hspset);
    if(comparison->codon_hspset)
        comparison->codon_hspset = HSPset_finalise(comparison->codon_hspset);
    return;
    }

/**/

