/****************************************************************\
*                                                                *
*  Simple client-server code library                             *
*                                                                *
*  Guy St.C. Slater..   mailto:guy@ebi.ac.uk                     *
*  Copyright (C) 2000-2006.  All Rights Reserved.                *
*                                                                *
*  This source code is distributed under the terms of the        *
*  GNU Lesser General Public License. See the file COPYING       *
*  or http://www.fsf.org/copyleft/lesser.html for details        *
*                                                                *
*  If you use this code, please keep this notice intact.         *
*                                                                *
\****************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>    /* For bcopy() */
#include <unistd.h>    /* For close() */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <fcntl.h>
#include <signal.h>   /* For sigaction() */
#include <sys/wait.h> /* For wait()   */

#include "socket.h"

#define Socket_BUFSIZE BUFSIZ

static SocketConnection *SocketConnection_create(gchar *host,
                                                 gint  port){
    register SocketConnection *connection
     = g_new(SocketConnection, 1);
    if((connection->sock = socket(AF_INET, SOCK_STREAM, 0)) < 0){
        perror("opening stream socket");
        exit(1);
        }
    connection->host = g_strdup(host);
    connection->port = port;
    return connection;
    }

SocketClient *SocketClient_create(gchar *host, gint port){
    register SocketClient *client = g_new(SocketClient, 1);
    struct sockaddr_in server;
    struct hostent *hp = gethostbyname(host);
    if(!hp){
        perror("lookup up hostname");
        exit(1);
        }
    client->connection = SocketConnection_create(host, port);
#if 0
    if(fcntl(client->connection->sock, F_SETFL, O_NDELAY) == -1){ /* Make non-blocking */
        perror("Tcp: Could not set O_NDELAY on socket with fcntl");
        exit(1);
        }
#endif /* 0 */
    server.sin_family = AF_INET;
    server.sin_port = htons(port);
    bcopy(hp->h_addr, &server.sin_addr, hp->h_length);
    if(connect(client->connection->sock, (struct sockaddr*)&server,
               sizeof(server)) < 0){
        perror("connecting client socket");
        exit(1);
        }
    return client;
    }

static void SocketConnection_destroy(SocketConnection *connection){
    if(close(connection->sock) == -1){
        perror("closing socket");
        exit(1);
        }
    g_free(connection->host);
    g_free(connection);
    return;
    }

static gchar *SocketConnection_read(gint sock){
    register gint len;
    register gchar *reply;
    register GString *string = g_string_sized_new(Socket_BUFSIZE);
    gchar buffer[Socket_BUFSIZE+1];
    do {
        if((len = recv(sock, buffer, Socket_BUFSIZE, 0)) < 0)
            perror("reading from socket");
        buffer[len] = '\0';
        g_string_append(string, buffer);
    } while(len == Socket_BUFSIZE);
    if(string->len){
        reply = string->str;
        g_string_free(string, FALSE);
        return reply;
        }
    g_string_free(string, TRUE);
    return NULL;
    }

gchar *SocketClient_send(SocketClient *client, gchar *msg){
    register gint len;
    if((len = send(client->connection->sock, msg, strlen(msg), 0)) < 0){
        perror("writing client message");
        exit(1);
        }
    return SocketConnection_read(client->connection->sock);
    }

void SocketClient_destroy(SocketClient *client){
    SocketConnection_destroy(client->connection);
    g_free(client);
    return;
    }

static void SocketServer_reap_dead_children(int signum){
    signal(SIGCHLD, SIG_IGN);
    while(waitpid(-1, NULL, WNOHANG) > 0)
        g_message("I just reaped a dead child!");
    signal(SIGCHLD, SocketServer_reap_dead_children);
    return;
    }

SocketServer *SocketServer_create(gint port, gint max_connections,
              SocketProcessFunc server_process_func,
              SocketConnectionOpenFunc connection_open_func,
              SocketConnectionCloseFunc connection_close_func,
              gpointer user_data){
    register SocketServer *server = g_new(SocketServer, 1);
    struct sockaddr_in sock_server;
    socklen_t len = sizeof(sock_server);
    server->connection = SocketConnection_create("localhost", port);
    g_assert(server_process_func);
    server->server_process_func = server_process_func;
    server->connection_open_func = connection_open_func;
    server->connection_close_func = connection_close_func;
    server->user_data = user_data;
    sock_server.sin_family = AF_INET;
    sock_server.sin_addr.s_addr = INADDR_ANY;
    sock_server.sin_port = htons(port);
    if(bind(server->connection->sock,
            (struct sockaddr*)&sock_server, len)){
        perror("binding stream socket");
        exit(1);
        }
    if(getsockname(server->connection->sock,
                   (struct sockaddr*)&sock_server, &len)){
        perror("getting socket name");
        exit(1);
        }
    server->connection->port = ntohs(sock_server.sin_port);
    g_message("Server started on port [%d]",
              server->connection->port);
    signal(SIGCHLD, SocketServer_reap_dead_children);
    return server;
    }

gboolean SocketServer_listen(SocketServer *server){
    register int msgsock;
    register gchar *msg;
    register gboolean ok = TRUE;
    register gpointer connection_data;
    gchar *reply;
    g_message("listening ...");
    listen(server->connection->sock, 5);
    msgsock = accept(server->connection->sock, NULL, NULL);
    g_message("opened msgsock [%d]", msgsock);
    if(msgsock == -1){
        perror("server accept");
        close(msgsock);
        exit(1);
        }
    /* FIXME: should count children and apply max children limit ?
     *        need to send an busy error message back to the client
     */
    if(fork() == 0){
        connection_data = server->connection_open_func
                        ? server->connection_open_func(server->user_data)
                        : NULL;
        do {
            msg = SocketConnection_read(msgsock);
            reply = NULL;
            if(!msg)
                break;
            ok = server->server_process_func(msg, &reply,
                                     connection_data, server->user_data);
            g_free(msg);
            if(reply){
                if(send(msgsock, reply, strlen(reply), 0) < 0){
                    perror("writing reply");
                    exit(1);
                    }
                g_free(reply);
                }
        } while(ok);
        g_message("close conn norm");
        if(server->connection_close_func)
            server->connection_close_func(connection_data,
                                          server->user_data);
        exit(0);
        }
    close(msgsock);
    g_message("close conn");
    return TRUE; /* Keep server running */
    }

void SocketServer_destroy(SocketServer *server){
    SocketConnection_destroy(server->connection);
    g_free(server);
    return;
    }

