#ifndef PLATFORM_H
#define PLATFORM_H

#ifdef __cplusplus
extern "C" {
#endif

/* Platform specific definitinos go in a special file */
#include "platform_specific.h"

/* Functions for abstracting the details of registers and memory layout for interpreting ptrace stacks and memory */

/* Called once before any other call to ptlib functions */
void ptlib_init();

/* Continue (or detach) a halted process */
int ptlib_continue( int request, pid_t pid, int signal );

/* Called once per new process created */
void ptlib_prepare( pid_t pid );

/* Wait for next event. Returns some data about the event
 * "status" is the status returned by "wait"
 * "data" is extra data returned by wait (such as rusage)
 * ptlib_parse_wait parses the info
 * Reports whether it was a signal delivered at the process (ret gets the signal number)
 * A process stopped due to signal (ret is the signal number)
 * A process terminated (ret is the return code)
 * A process terminated (ret is the signal that killed it)
 * A SYSCALL took place (ret is the syscall number)
 * A new process being created (only if PTLIB_SUPPORTS_{FORK,VFORK,CLONE} is defined for the platform) - ret is the new PID */
enum PTLIB_WAIT_RET { SIGNAL, EXIT, SIGEXIT, SYSCALL, NEWPROCESS };
int ptlib_wait( pid_t *pid, int *status, ptlib_extra_data *data );
long ptlib_parse_wait( pid_t pid, int status, enum PTLIB_WAIT_RET *type );

/* If we get a trace before we run ptlib_prepare, we might mis-interpret the signals */
int ptlib_reinterpret( enum PTLIB_WAIT_RET prestate, pid_t pid, int status, long *ret );

/* Returns/sets the Program Counter (EIP on Intel) for the traced program */
void *ptlib_get_pc( pid_t pid );
int ptlib_set_pc( pid_t pid, void *location );

/* Syscall analysis functions - call only when stopped process just invoked a syscall */

/* Report the syscall number being invoked */
int ptlib_get_syscall( pid_t pid );
int ptlib_set_syscall( pid_t pid, int sc_num ); /* Change the meaning of a just started system call */
int ptlib_generate_syscall( pid_t pid, int sc_num, void *base_memory ); /* Generate a new system call */

/* Return the nth argument passed */
int_ptr ptlib_get_argument( pid_t pid, int argnum );
int ptlib_set_argument( pid_t pid, int argnum, int_ptr value );

int_ptr ptlib_get_retval( pid_t pid );
int ptlib_success( pid_t pid, int sc_num ); /* Report whether the syscall succeeded */
void ptlib_set_retval( pid_t pid, int_ptr val );
void ptlib_set_error( pid_t pid, int sc_num, int error );
int ptlib_get_error( pid_t pid, int sc_num );

/* Copy memory in and out of the process
 * Return TRUE on success */
int ptlib_get_mem( pid_t pid, void *process_ptr, void *local_ptr, size_t len );
int ptlib_set_mem( pid_t pid, const void *local_ptr, void *process_ptr, size_t len );

/* Copy a NULL terminated string. "get" returns the number of bytes copied, including the NULL */
int ptlib_get_string( pid_t pid, void *process_ptr, char *local_ptr, size_t maxlen );
int ptlib_set_string( pid_t pid, const char *local_ptr, void *process_ptr );

/* Get a process' current directory and open fds */
/* Return value is as for "readlink" */
ssize_t ptlib_get_cwd( pid_t pid, char *buffer, size_t buff_size );
ssize_t ptlib_get_fd( pid_t pid, int fd, char *buffer, size_t buff_size );

/* Save/restore the process state */
void ptlib_save_state( pid_t pid, void *buffer );
void ptlib_restore_state( pid_t pid, const void *buffer );

/* Initialize debugger controled memory inside debuggee address space */
const void *ptlib_prepare_memory( ); /* Returns pointer to static buffer with the desired opcods, of ptlib_prepare_memory_len length */
size_t ptlib_prepare_memory_len(); /* How much memory does the platform need beyond how much the process needs */

/* This is a function that must be provided by the user of the library */
void __dlog_( const char *format, ... );
extern int log_level;
#define dlog if( log_level>0 ) __dlog_

#ifdef __cplusplus
};
#endif

#endif /* PLATFORM_H */
