# This file is part of the Falcon repository manager
# Copyright (C) 2005-2008 Dennis Kaarsemaker
# See the file named COPYING in the root of the source tree for license details
#
# mirror.py - Repository mirrors

import falcon
import falcon.pocket
import shutil, os
from django.db import models
import cPickle as pickle

class Mirror(models.Model):
    """Abstraction of a mirror server"""
    name = models.CharField(maxlength=30,unique=True)
    webbase = models.URLField(maxlength=80)
    rsync = models.CharField(maxlength=100, blank=True, default='')
    rsync_opts = models.PickleField(blank=True, default=[])
    sponsor = models.CharField(maxlength=100, default='')
    realcomponents = models.ManyToManyField(falcon.pocket.Component)
    metacomponents = models.ManyToManyField(falcon.pocket.MetaComponent)
    comment = models.TextField()

    def _get_components(self):
        if not self._components:
            self._components = list(self.realcomponents.all()) + list(self.metacomponents.all())
            self._components.sort(lambda x,y: cmp(x.name,y.name))
        return self._components
    components = property(_get_components)

    def __init__(self, *args, **kwargs):
        super(Mirror, self).__init__(*args, **kwargs)
        self.rootdir = os.path.join('.falcon','base-%s' % self.name)
        self._components = None
        if type(self.rsync_opts) == str:
            self.rsync_opts = pickle.loads(self.rsync_opts)

    def __str__(self):
        rsync = self.rsync
        if self.rsync_opts:
            rsync += ' ' + rsync_opts
        components = ', '.join(get_components())
        return "Mirror %s (web: %s, rsync: %s, sponsor: %s, components: %s)"

    def __repr__(self):
        return '<Falcon mirror %s>' % self.name

    def get_components(self):
        return sorted([str(x) for x in self.components])

    def has_all(self):
        return len(self.components) == len(falcon.pocket.Component.objects.all()) + len(falcon.pocket.MetaComponent.objects.all())

    def update(self):
        """Synchronize local copy of the mirrors metadata with the main
           metadata"""
        if not os.path.exists(self.rootdir):
            os.makedirs(self.rootdir)

        # Remove files that no longer exist or should no longer be mirrored
        existing = listdir(self.rootdir)
        for file in reversed(existing):
            if not os.path.exists(file):
                if os.path.isdir(os.path.join(self.rootdir, file)):
                    shutil.rmtree(os.path.join(self.rootdir, file))
                else:
                    os.unlink(os.path.join(self.rootdir, file))
        
        if os.path.exists(os.path.join(self.rootdir,'dists')):
            to_mirror = [str(x) for x in self.components]
            for p in os.listdir(os.path.join(self.rootdir, 'dists')):
                for c in os.listdir(os.path.join(self.rootdir, 'dists', p)):
                    if c in ('Release','Release.gpg','index.html'):
                        continue
                    if c.startswith('Contents-') and c.endswith('.gz'):
                        continue
                    if c.startswith('app-install-data') and c.endswith('.deb'):
                        continue
                    if os.path.join(p,c) not in to_mirror:
                        shutil.rmtree(os.path.join(self.rootdir,'dists',p,c))
                if len(os.listdir(os.path.join(self.rootdir, 'dists', p))) == 3:
                    shutil.rmtree(os.path.join(self.rootdir,'dists',p))

        # Copy over new(er) files
        filters = ['pool','.falcon']

        if not self.has_all():
            filters += [x.distpath for x in falcon.pocket.Pocket.objects.all()]
            filters += [x.distpath for x in falcon.pocket.Component.objects.all()]
            filters += [x.distpath for x in falcon.pocket.MetaComponent.objects.all()]
            
            for z in set([x.distpath for x in [y.pocket for y in self.components]] + [x.distpath for x in self.components]):
                filters.remove(z)
        to_copy = listdir('.', filters = filters)

        for file in to_copy:
            if os.path.isdir(file):
                if not os.path.exists(os.path.join(self.rootdir, file)):
                    os.makedirs(os.path.join(self.rootdir,file))
            else:
                if not os.path.exists(os.path.join(self.rootdir, file)) or falcon.util.newer(file, os.path.join(self.rootdir, file)):
                    shutil.copy(file, os.path.join(self.rootdir, file))

        from django.template import loader, Context
        pockets = falcon.pocket.Pocket.objects.filter(realcomponents__mirror = self).distinct()
        do_appinstall = False
        try:
            do_appinstall = falcon.plugins.app_install.AppInstallDataPlugin.conf.enabled
        except:
            pass
        if not self.has_all():
            template = loader.get_template('pocket.html')
            for pocket in pockets:
                # Generate proper release files and sign them
                components = [x for x in self.components if x.pocket == pocket]
                context = Context({'p': pocket, 'conf': falcon.conf, 'dots': '../../', 'components': components, 'do_appinstall': do_appinstall})
                falcon.util.writefile(os.path.join(self.rootdir, pocket.distpath, 'index.html'),template.render(context))

        template = loader.get_template('base.html')
        mirrors = falcon.mirror.Mirror.objects.all().order_by('name')
        context = Context({'pockets': pockets, 'conf': falcon.conf, 'mirrors': mirrors, 'mirror': self, 'do_appinstall': do_appinstall})
        falcon.util.writefile(os.path.join(self.rootdir, 'index.html'),template.render(context))

    @falcon.util.timer
    @falcon.plugin.wrap_plugin
    def sync(self):
        """Synchronize files to the mirror"""
        args = ['rsync', '-ruRltO']
        if falcon.conf.no_sync: args.append('-n')
        if falcon.conf.verbose: args.append('--progress')

        # Filter very specifically:
        # - all pockets/components that need to be mirrored
        rfilter = ['--filter', '+ pool/']
        dirs = []
        for p in falcon.pocket.Pocket.objects.filter(realcomponents__mirror = self).distinct():
            rfilter += ['--filter', '+ %s' % p.poolpath]
            dirs.append(p.poolpath)
        for c in self.realcomponents.all():
            rfilter += ['--filter', '+ %s' % c.poolpath]
            dirs.append(c.poolpath)
        # - only package files, exclude everything else 
        rfilter += ['--filter', '+ **deb',
                    '--filter', '+ **dsc',
                    '--filter', '+ **tar.gz',
                    '--filter', '+ **diff.gz',
                    '--filter', '- **']
    
        # Upload data
        try:
            comm = args + rfilter + dirs + [self.rsync]
            falcon.util.run(comm, buffer=False)
        except RuntimeError:
            falcon.util.warning(_("Synchronizing with %s failed") % self.name)
            return
        # Sync metadata
        try:
            comm = args + ['--delete', '--force'] + os.listdir(self.rootdir) + [self.rsync]
            falcon.util.run(comm, wd=self.rootdir,buffer=False)
        except RuntimeError:
            falcon.util.warning(_("Synchronizing with %s failed") % self.name)
            return
        # Sync data, now with --delete to get rid of old packages
        try:
            comm = args + ['--existing', '--delete', '--force', 'pool', self.rsync]
            falcon.util.run(comm, buffer=False)
        except RuntimeError:
            falcon.util.warning(_("Synchronizing with %s failed") % self.name)

def listdir(path, base='', filters=[]):
    """Recursively list a directory, possibly filtered"""
    result = []
    for file in os.listdir(path):
        newpath = os.path.join(path, file)
        newbase = os.path.join(base, file)
        brk = False
        for filter in filters:
            if newbase.startswith(filter):
                brk = True
        if brk:
            continue

        if os.path.isdir(newpath):
            result.append(newbase)
            result += listdir(newpath, newbase, filters)
        else:
            result.append(newbase)
    return result

