/*
 * farsight-codec.h - Headers for FarsightCodec
 *
 * Farsight Voice+Video library,
 * Copyright 2005,2006 Collabora Ltd.,
 * Copyright 2005,2006 Nokia Corp.,
 *   @author: Rob Taylor <rob.taylor@collabora.co.uk>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __FARSIGHT_CODEC_H__
#define __FARSIGHT_CODEC_H__

#include <glib.h>

G_BEGIN_DECLS

typedef struct _FarsightCodec FarsightCodec;
typedef struct _FarsightCodecParameter FarsightCodecParameter;
typedef struct _FarsightCodecPreference FarsightCodecPreference;

/**
 * FarsightMediaType:
 * @FARSIGHT_MEDIA_TYPE_AUDIO: A media type that encodes audio.
 * @FARSIGHT_MEDIA_TYPE_VIDEO: A media type that encodes video.
 *
 * Enum used to signify the media type of a codec or stream.
 */
typedef enum
{
  FARSIGHT_MEDIA_TYPE_AUDIO,
  FARSIGHT_MEDIA_TYPE_VIDEO,
  FARSIGHT_MEDIA_TYPE_LAST = FARSIGHT_MEDIA_TYPE_VIDEO
} FarsightMediaType;

/**
 * FarsightCodec:
 * @id: numeric identifier for encoding, eg. PT for SDP
 * @encoding_name: the name of the codec
 * @media_type: type of media this codec is for
 * @clock_rate: clock rate of this stream
 * @channels: Number of channels codec should decode
 * @optional_params:  key pairs of param name to param data
 */ 
struct _FarsightCodec
{
  gint id;
  char *encoding_name;
  FarsightMediaType media_type;
  guint clock_rate;
  guint channels;
  GList *optional_params;
  /*< private >*/
  gpointer _padding[4];         /* padding for binary-compatible
                                   expansion*/
};

/**
 * FarsightCodecParameter:
 * @name: paramter name.
 * @value: parameter value.
 *
 * Used to store arbitary parameters for a codec
 */
struct _FarsightCodecParameter {
    gchar *name;
    gchar *value;
};

/**
 * FarsightCodecPreference:
 * @encoding_name: name of encoding preferred
 * @clock_rate: rate of codec preffered
 *
 * Used to give a preferece for what type of codec to use.
 */
struct _FarsightCodecPreference {
    gchar *encoding_name;
    gint clock_rate;
};

void farsight_codec_init (FarsightCodec *codec, int id, 
                          const char *encoding_name,
                          FarsightMediaType media_type, guint clock_rate);

void farsight_codec_destroy (FarsightCodec * codec);
FarsightCodec * farsight_codec_copy (FarsightCodec * codec);
void farsight_codec_list_destroy (GList *codec_list);
GList * farsight_codec_list_copy (const GList *codec_list);

GList *farsight_codec_list_from_keyfile (const gchar *filename);
gchar *farsight_codec_to_string (FarsightCodec *codec);

const gchar *farsight_media_type_to_string (FarsightMediaType media_type);

G_END_DECLS
#endif

