/*
 * test.c - Farsight tests
 *
 * Farsight Voice+Video library test suite
 *  Copyright 2005,2006 Collabora Ltd.
 *  Copyright 2005,2006 Nokia Corporation
 *   @author: Rob Taylor <rob.taylor@collabora.co.uk>.
 *   @author: Philippe Khalaf <philippe.khalaf@collabora.co.uk>.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <glib.h>
#include <gst/gst.h>
#include <farsight/farsight-session.h>
#include <farsight/farsight-stream.h>
#include <farsight/farsight-transport.h>

GMainLoop *mainloop = NULL;

FarsightSession *session1, *session2;
FarsightStream *s1stream1, *s2stream1;

static void
stream_error (FarsightStream *stream,
       FarsightStreamError error,
       const gchar *debug)
{
    g_print ("%s: stream error: stream=%p error=%s\n", __FUNCTION__, stream, debug);
}

static void
session_error (FarsightSession *stream,
       FarsightSessionError error,
       const gchar *debug)
{
    g_print ("%s: session error: session=%p error=%s\n", __FUNCTION__, stream, debug);
}


static void
new_active_candidate_pair (FarsightStream *stream, gchar* native_candidate, gchar *remote_candidate)
{
    g_print ("%s: new-native-candidate-pair: stream=%p\n", __FUNCTION__, stream);
    farsight_stream_start(stream);
}

static void
codec_changed (FarsightStream *stream, gint codec_id)
{
    g_print ("%s: codec-changed: codec_id=%d, stream=%p\n", __FUNCTION__, codec_id, stream);
}

static void
new_native_candidate (FarsightStream *stream, const gchar *candidate_id)
{
    GList *native_candidate = NULL, *lp;
    //const GList *native_candidates;
    FarsightTransportInfo *info;
    FarsightStream *peer_stream = (stream == s1stream1)?s2stream1:s1stream1;

    g_print ("%s: new-native-candidate called: stream=%p\n", __FUNCTION__,
            stream);

    native_candidate = farsight_stream_get_native_candidate (stream, candidate_id);

    /*
    FarsightTransportInfo *trans = NULL;
    const GList *lp2;
    FarsightTransportInfo *ccopy = NULL;

    for (lp2 = native_candidates; lp2; lp2 = g_list_next (lp2)) {
        trans = (FarsightTransportInfo *) lp2->data;
        if (g_ascii_strcasecmp(trans->candidate_id, candidate_id) == 0)
        {
            ccopy = farsight_transport_copy(trans);
            g_print("%p\n", ccopy);
            native_candidate = g_list_prepend (native_candidate, ccopy);
        }
    }
    native_candidate = g_list_reverse (native_candidate);
    */

    for (lp = native_candidate; lp; lp = g_list_next (lp)) 
    {
      info = (FarsightTransportInfo*)lp->data;
      g_message ("Local transport candidate: %s %d %s %s %s:%d %s %s, pref %f", 
          info->candidate_id, info->component, 
          (info->proto == FARSIGHT_NETWORK_PROTOCOL_TCP)?"TCP":"UDP",
          info->proto_subtype, info->ip, info->port, info->username, info->password,
          (double) info->preference);
    }

    /* Let's add it to our second stream */
    farsight_stream_add_remote_candidate(peer_stream, native_candidate);

    farsight_transport_list_destroy (native_candidate);
}

static void
native_candidates_prepared (FarsightStream *stream)
{
    const GList *transport_candidates, *lp;
    FarsightTransportInfo *info;

    g_print ("%s: preparation-complete: stream=%p\n", __FUNCTION__, stream);

    transport_candidates = farsight_stream_get_native_candidate_list (stream); 
    for (lp = transport_candidates; lp; lp = g_list_next (lp)) 
    {
      info = (FarsightTransportInfo*)lp->data;
      g_message ("Local transport candidate: %s %d %s %s %s:%d, pref %f", 
          info->candidate_id, info->component, (info->proto == FARSIGHT_NETWORK_PROTOCOL_TCP)?"TCP":"UDP",
          info->proto_subtype, info->ip, info->port, (double) info->preference);
    }
    //g_main_loop_quit(mainloop);
}

static void 
state_changed (FarsightStream *stream, 
               FarsightStreamState state,
               FarsightStreamDirection dir)
{
    switch (state) {
      case FARSIGHT_STREAM_STATE_CONNECTING:
        g_message ("%s: %p connecting\n", __FUNCTION__, stream);
        break;
      case FARSIGHT_STREAM_STATE_CONNECTED:
        g_message ("%s: %p connected\n", __FUNCTION__, stream);
        break;
      case FARSIGHT_STREAM_STATE_DISCONNECTED:
            g_message ("%s: %p disconnected\n", __FUNCTION__, stream);
            break;
    }
}

FarsightSession *setup_rtp_session();
FarsightStream *setup_rtp_stream(FarsightSession *session);

gboolean change_codec2(gpointer data)
{
    farsight_stream_set_active_codec(FARSIGHT_STREAM(data), 3);
    return FALSE;
}

gboolean change_codec(gpointer data)
{
    farsight_stream_set_active_codec(FARSIGHT_STREAM(data), 8);
    g_timeout_add (5000, change_codec2, data);
    return FALSE;
}


int main(int argc, char **argv)
{
    GstElement *alsasrc, *alsasink, *fakesrc, *fakesink;

    g_type_init();
    gst_init (&argc, &argv);

    mainloop = g_main_loop_new (NULL, FALSE);

    session1 = setup_rtp_session();
    session2 = setup_rtp_session();

    s1stream1 = setup_rtp_stream(session1);
    s2stream1 = setup_rtp_stream(session2);

    farsight_stream_prepare_transports (FARSIGHT_STREAM (s1stream1));
    farsight_stream_prepare_transports (FARSIGHT_STREAM (s2stream1));

    farsight_stream_set_remote_codecs(s1stream1,
            farsight_stream_get_local_codecs(s1stream1));
    farsight_stream_set_remote_codecs(s2stream1,
            farsight_stream_get_local_codecs(s1stream1));

    //farsight_stream_set_remote_candidate_list(s1stream1,
    //        farsight_stream_get_native_candidate_list(s1stream1));
    //farsight_stream_set_remote_candidate_list(s2stream1,
    //        farsight_stream_get_native_candidate_list(s1stream1));

//    farsight_stream_set_active_candidate_pair(s1stream1, "L2", "L2");
//    farsight_stream_set_active_candidate_pair(s2stream1, "L2", "L2");

    alsasrc = gst_element_factory_make("audiotestsrc", "audiotestsrc"); 
    fakesink = gst_element_factory_make("fakesink", "fakesink");
    fakesrc = gst_element_factory_make("alsasrc", "alsasrc");
    alsasink = gst_element_factory_make("fakesink", "fakesink");

    g_object_set(G_OBJECT(alsasink), "sync", FALSE, NULL);
    g_object_set(G_OBJECT(alsasink), "latency-time", G_GINT64_CONSTANT (20000), NULL);
    g_object_set(G_OBJECT(alsasink), "buffer-time", G_GINT64_CONSTANT (80000), NULL);
    g_object_set(G_OBJECT(alsasrc), "blocksize", 320, NULL);
    g_object_set(G_OBJECT(alsasrc), "latency-time", G_GINT64_CONSTANT (20000), NULL);
    g_object_set(G_OBJECT(alsasrc), "is-live", TRUE, NULL);

    //g_object_set(G_OBJECT(fakesrc), "device", "hw:0,1", NULL);
    //g_object_set(G_OBJECT(fakesink), "device", "hw:0,1", NULL);

    //gst_debug_set_threshold_for_name ("audiotestsrc", 5);
    //gst_debug_set_threshold_for_name ("rtpdemux", 5);
    //gst_debug_set_threshold_for_name ("fakesink", 5)

    //gst_debug_set_threshold_for_name ("audiosink", 5);
    //gst_debug_set_threshold_for_name ("alsasink", 5);
    //gst_debug_set_threshold_for_name ("rtprecv", 5);
    //gst_debug_set_threshold_for_name ("rtpbin", 5);
    //gst_debug_set_threshold_for_name ("basertpdepayload", 5);
    //gst_debug_set_threshold_for_name ("basertppayload", 5);
    //gst_debug_set_threshold_for_name ("basesrc", 5);

    farsight_stream_set_source(s1stream1, alsasrc);
    farsight_stream_set_sink(s1stream1, fakesink);
    farsight_stream_set_source(s2stream1, fakesrc);
    farsight_stream_set_sink(s2stream1, alsasink);

    //gint func_ref = 0;
    //func_ref = g_timeout_add (3000, change_codec, s1stream1);
    //farsight_stream_set_active_codec(s2stream1, 8);
    //farsight_stream_set_active_codec(s1stream1, 3);

    g_main_loop_run(mainloop); 
    //g_source_remove (func_ref);
    return 0;
}

FarsightSession *setup_rtp_session()
{
    FarsightSession *session;

    session = farsight_session_factory_make ("rtp");

    if (!session) {
      g_error("RTP plugin not found");
      exit(1);
    }
    g_print ("protocol details:\n name: %s\n description: %s\n author: %s\n",
             farsight_plugin_get_name (session->plugin),
             farsight_plugin_get_description (session->plugin),
             farsight_plugin_get_author (session->plugin));
    g_signal_connect (G_OBJECT (session), "error", 
                      G_CALLBACK (session_error), NULL);

    return session;
}

FarsightStream *setup_rtp_stream(FarsightSession *session)
{
    FarsightStream *stream;
    const GList *possible_codecs, *lp;
    FarsightCodec *codec;

    stream = farsight_session_create_stream (session,
            FARSIGHT_MEDIA_TYPE_AUDIO, FARSIGHT_STREAM_DIRECTION_BOTH);
    g_signal_connect (G_OBJECT (stream), "error", 
                      G_CALLBACK (stream_error), NULL);
    g_signal_connect (G_OBJECT (stream), "new-active-candidate-pair", 
                      G_CALLBACK (new_active_candidate_pair), NULL);
    g_signal_connect (G_OBJECT (stream), "codec-changed", 
                      G_CALLBACK (codec_changed), NULL);
    g_signal_connect (G_OBJECT (stream), "native-candidates-prepared", 
                      G_CALLBACK (native_candidates_prepared), NULL);
    g_signal_connect (G_OBJECT (stream), "new-native-candidate", 
                      G_CALLBACK (new_native_candidate), NULL);
    g_signal_connect (G_OBJECT (stream), "state-changed", 
                      G_CALLBACK (state_changed), NULL);

    possible_codecs = farsight_stream_get_local_codecs (stream);

    for (lp = possible_codecs; lp; lp = g_list_next (lp)) 
    {
      codec = (FarsightCodec*) lp->data;
      g_message ("codec: %d: %s/%d found", codec->id, codec->encoding_name, 
                                       codec->clock_rate);
    }

    g_object_set (G_OBJECT (stream), "transmitter", "rawudp", NULL);


    return stream;
}
