/* $Id: PrintCode.hpp 4323 2009-01-27 13:48:12Z potyra $ 
 *
 * Print intermediate code to stream.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __PRINT_CODE_HPP
#define __PRINT_CODE_HPP

#include "intermediate/visitor/Visitor.hpp"
#include <iostream>

namespace intermediate {

//! print out intermediate code
/** This visitor can be used to print out intermediate code.
 *  It's main use is to debug the intermediate code. The format
 *  how the intermediate code looks like, may still change.
 *
 *  Format conventions:
 *  There is no opcode, which can accept more than three operands.
 *  In case there is a destination operand, which will be the result
 *  of one or more source operands, it will always be the rightmost
 *  operand.
 */
class PrintCode : public Visitor {
public:
	//! c'tor
	/** @param destination target stream, to which code will get written
	 *         to
	 */
	PrintCode(std::ostream &destination);

private:
	//! visit a CodeContainer node
	/** @param node node that gets visited.
	 */
	virtual void visit(CodeContainer &node);

	//! visit a ImmediateOperand node
	/** @param node node that gets visited.
	 */
	virtual void visit(ImmediateOperand &node);

	//! visit a BeginTransfer node
	/** @param node node that gets visited.
	 */
	virtual void visit(BeginTransfer &node);

	//! visit an EndTransfer node
	/** @param node node that gets visited.
	 */
	virtual void visit(EndTransfer &node);

	//! visit a SetParam node
	/** @param node node that gets visited.
	 */
	virtual void visit(SetParam &node);

	//! visit a GetParam node
	/** @param node node that gets visited.
	 */
	virtual void visit(GetParam &node);

	//! visit a Connect node
	/** @param node node that gets visited.
	 */
	virtual void visit(Connect &node);

	//! visit a Mov node
	/** @param node node that gets visited.
	 */
	virtual void visit(Mov &node);

	//! visit a Je node
	/** @param node node that gets visited.
	 */
	virtual void visit(Je &node);

	//! visit a Jbe node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jbe &node);

	//! visit a Jne node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jne &node);

	//! visit a Jb node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jb &node);

	//! visit a Jmp node
	/** @param node node that gets visited.
	 */
	virtual void visit(Jmp &node);

	//! visit a Label node
	/** @param node node that gets visited.
	 */
	virtual void visit(Label &node);

	//! visit an Abort node
	/** @param node node that gets visited.
	 */
	virtual void visit(Abort &node);

	//! visit an Add node
	/** @param node node that gets visited.
	 */
	virtual void visit(Add &node);

	//! visit a Sub node
	/** @param node node that gets visited.
	 */
	virtual void visit(Sub &node);

	//! visit a Call node
	/** @param node node that gets visited.
	 */
	virtual void visit(Call &node);

	//! visit a Return node
	/** @param node node that gets visited.
	 */
	virtual void visit(Return &node);

	//! visit a Proc node
	/** @param node node that gets visited.
	 */
	virtual void visit(Proc &node);

	//! visit an Update node
	/** @param node node that gets visited.
	 */
	virtual void visit(Update &node);

	//! visit a GetSig node
	/** @param node node that gets visited.
	 */
	virtual void visit(GetSig &node);

	//! visit an IMul node
	/** @param node node that gets visited.
	 */
	virtual void visit(IMul &node);

	//! visit a Div node
	/** @param node node that gets visited.
	 */
	virtual void visit(Div &node);

	//! visit an ROffset node
	/** @param node node that gets visited.
	 */
	virtual void visit(ROffset &node);

	//! visit an AOffset node
	/** @param node node that gets visited.
	 */
	virtual void visit(AOffset &node);

	//! visit a Suspend node
	/** @param node node that gets visited.
	 */
	virtual void visit(Suspend &node);

	//! visit a WakeOn node
	/** @param node node that gets visited.
	 */
	virtual void visit(WakeOn &node);

	//! visit a WakeAt node
	/** @param node node that gets visited.
	 */
	virtual void visit(WakeAt &node);

	//! visit a Log node
	/** @param node node that gets visited.
	 */
	virtual void visit(Log &node);

	//! visit an IndirectOperand node
	/** @param node node that gets visited.
	 */
	virtual void visit(IndirectOperand &node);

	//! visit a Reference node
	/** @param node node that gets visited.
	 */
	virtual void visit(Reference &node);

	//! visit a Register node
	/** @param node node that gets visited.
	 */
	virtual void visit(Register &node);

	//! visit a Data node
	/** @param node node that gets visited.
	 */
	virtual void visit(Data &node);

	//! visit a TypeElement node
	/** @param node node that gets visited.
	 */
	virtual void visit(TypeElement &node);

	//! visit a Type node
	/** @param node node that gets visited.
	 */
	virtual void visit(Type &node);

	//! put the annotations to stream, if any.
	/** @param node intermediate code node in question.
	 */
	void putAnnotation(const Node &node) const;

	//! stream to put nodes to.
	std::ostream &dst;

	//! small class to support different levels of nesting.
	struct Nesting {
	public:
		Nesting() : level(0) {}

		void operator++(void) {
			this->level++;
		}

		void operator--(void) {
			this->level--;
		}

		void put(std::ostream &stream) const {
			for (unsigned int i = 0; i < this->level; i++) {
				stream << "        ";
			}
		}

	private:
		unsigned int level;
	} nest;

	//! make sure that << can see class Nesting.
	friend std::ostream& 
	operator<<(std::ostream &stream, const Nesting &n);

};

//! overloaded operator for nesting. Must not be used outside of class.
std::ostream &
operator <<(std::ostream &stream, const PrintCode::Nesting &n);

}; /* namespace intermediate */

#endif /* __PRINT_CODE_HPP */
