/* $Id: glue-vhdl.h 4505 2009-04-27 09:55:46Z potyra $
 *
 *  Foreign VHDL interface of the interpreter. This interface must be 
 *  implemented, so that the interpreter can create foreign signals
 *  and components.
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __GLUE_SIGNAL_H_INCLUDED
#define __GLUE_SIGNAL_H_INCLUDED

#include "fauhdli.h"

/** glue_vhdl object structure. */
struct glue_vhdl;

/** Set a procedure call argument of a foreign procedure.
 *  @param s glue_vhdl instance
 *  @param proc name of the foreign procedure.
 *  @param param name of the parameter
 *  @param value parameter value. Content depending on the signature
 *         of the foreign procedure.
 *         For parameter passing mechanisms, see GenCode.cpp.
 */
extern void
glue_vhdl_proc_set(
	struct glue_vhdl *s,
	const char *proc,
	const char *param, 
	union fauhdli_value value
);

/** Call a foreign procedure.
 *  @param s glue_vhdl instance.
 *  @param proc name of the procedure to call.
 */
extern void
glue_vhdl_proc_call(struct glue_vhdl *s, const char *proc);

/** set a FAUmachine signal with signal id sig_id and data data.
 *  @param s glue_vhdl instance.
 *  @param sig_id signal id (cf. system.h)
 *  @param data data fitting to the signal. (FIXME composites)
 *  @param drv pointer to driver instance.
 */
extern void
glue_vhdl_set(
	struct glue_vhdl *s, 
	unsigned int sig_id, 
	union fauhdli_value data,
	void *drv
);

/** Connect a VHDL signal to a FAUmachine signal so that writes are getting
 *  forwarded to FAUmachine.
 *  This function should get called if a VHDL driver is connected to a 
 *  foreign VHDL signal.
 *  @param s glue_vhdl instance.
 *  @param sig_id FAUmachine signal id.
 *  @param init initial driving value.
 *  @param drv corresponding driver pointer.
 */
extern void
glue_vhdl_connect_out(
	struct glue_vhdl *s,
	unsigned int sig_id,
	union fauhdli_value init,
	void *drv
);

/** Connect a FAUmachine signal that is read from VHDL.
 *  This function will call drv_update or in case a FAUmachine signal with 
 *  sig_id obtains a new value.
 *  @param s glue_vhdl instance.
 *  @param sig_id unique FAUmachine signal id.
 *  @param drv_update callback to update VHDL signals from FAUmachine changes.
 *  @param drv driver instance that will be passed as _drv parameter for
 *  	   drv_update
 */
extern void
glue_vhdl_connect_in(
	struct glue_vhdl *s,
	unsigned int sig_id,
	void (*drv_update)(void *_drv, union fauhdli_value value),
	void *drv
);

/*  Create a FAUmachine signal and return its ID.
 *  @param s glue_vhdl instance.
 *  @param type type of the signal.
 *  @param name signal name (for debugging only)
 *  @return unique signal id.
 */
extern unsigned int
glue_vhdl_create_signal(
	struct glue_vhdl *s, 
	const char *type, 
	const char *name
);

/** in case the VHDL interpreter wants to suspend, it can tell glue_vhdl
 *  to wake it up early with cb(_instance) in case a FAUmachine signal 
 *  event occurs.
 *  If glue_vhdl_set is called, it will automatically disable wakeups.
 *  @param s glue_vhdl instance.
 *  @param _instance parameter passed to cb
 *  @param cb callback in case a FAUmachine signal event occurs.
 */
extern void
glue_vhdl_enable_event_wakeups(
	struct glue_vhdl *s, 
	void *_instance, 
	void (*cb)(void *_inst)
);

/** Create a foreign component (entity).
 *  @param s glue_vhdl instance.
 *  @param type type of the component (entity name).
 *  @param name name of the instantiated unit
 *  @param stack_depth stack (hierarchy) depth of component.
 *  @return unique component id.
 */
extern unsigned int
glue_vhdl_comp_create(
	struct glue_vhdl *s,
	const char *type,
	const char *name,
	unsigned int stack_depth
);

/** Initialize a foreign component.
 *  The foreign component must first have been created with
 *  glue_vhdl_comp_create and the ports must have been connected
 *  with glue_vhdl_comp_port_connect.
 *
 *  @param s glue_vhdl instance.
 *  @param comp component id (cf. glue_vhdl_comp_create).
 */
extern void
glue_vhdl_comp_init(struct glue_vhdl *s, unsigned int comp);

/** connect a foreign signal to the port of a foreign component.
 *  @param s glue_vhdl instance
 *  @param comp unique component id (cf. glue_vhdl_comp_create, return value)
 *  @param port name of the desired port.
 *  @param sig unique signal id (cf. glue_vhdl_create_signal, return value)
 */ 
extern void
glue_vhdl_comp_port_connect(
	struct glue_vhdl *s,
	unsigned int comp, 
	const char *port, 
	unsigned int sig
);

/** set a generic of a foreign component (which is not an array).
 *  @param s glue_vhdl instance
 *  @param comp unique component id (cf. glue_vhdl_comp_create, return value)
 *  @param generic name of the desired generic.
 *  @param val VHDL value (direct value for non-composites, pointer for 
 *         record types).
 *  @param type name of the corresponding VHDL type.
 */
extern void
glue_vhdl_comp_generic_nonarray_set(
	struct glue_vhdl *s,
	unsigned int comp,
	const char *generic,
	union fauhdli_value val,
	const char *type
);

/** set a generic of a foreign component (which is an array).
 *  @param s glue_vhdl instance
 *  @param comp unique component id (cf. glue_vhdl_comp_create, return value)
 *  @param generic name of the desired generic.
 *  @param element_type type name of the element type
 *  @param base pointer to base of array.
 *  @param array_size size of array.
 */
extern void
glue_vhdl_comp_generic_array_set(
	struct glue_vhdl *s,
	unsigned int comp,
	const char *generic,
	const char *element_type,
	union fauhdli_value base,
	universal_integer array_size
);

/** create a glue_vhdl object.
 *  @return newly created glue_vhdl object. */
extern struct glue_vhdl *
glue_vhdl_create(void);

/** destroy a glue_vhdl object. 
 *  @param s instance.
 */
extern void
glue_vhdl_destroy(struct glue_vhdl *s);

#endif /* __GLUE_SIGNAL_H_INCLUDED */
