/*
 * Copyright (C) 2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define EEPROM_OPCODEBITS	 3	/* How many bits an eeprom opcode */
					/* has (always 3) */
#define EEPROM_ADDRESSBITS	 6	/* How many bits an eeprom address */
					/* has (6 on 82557*/
#define SZ_E100EEPROM	64

#ifdef STATE

struct {
	uint16_t data[SZ_E100EEPROM]; /* EEPROM size of 82557:
				       * 64 x 16 bit. */
	unsigned char selected; /* whether the EEPROM is "selected"
				 * (if not it will not react) */
	unsigned char di;	/* status of the EEPROM data-in sig */
	unsigned char clock;    /* status of the EEPROM clock signal */
	unsigned char opcode;   /* Which opcode was requested */
	unsigned char opcbits;  /* How many bits of the opcode have
					* been written already */
	unsigned char address;  /* What address shall be read /
					* written / erased */
	unsigned char adbits;   /* How many bits of the address have
					* been written already */
	unsigned char datapos;  /* Which bit of the eeprom-register
					* will be read on next access */
	unsigned char ewenable; /* Erase / write enabled */
	char filename[1024];
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(resetstatus)(struct cpssp *cpssp)
{
	cpssp->NAME.opcode = 0;
	cpssp->NAME.opcbits = 0;
	cpssp->NAME.address = 0;
	cpssp->NAME.adbits = 0;
	cpssp->NAME.datapos = 0;
}

static void
NAME_(handleclock)(struct cpssp *cpssp, unsigned char disig)
{
	if (cpssp->NAME.opcbits < EEPROM_OPCODEBITS) {
		if (cpssp->NAME.opcbits != 0
		 || disig) {
			/* The First Bit of the opcode always has to be a 1 */
			cpssp->NAME.opcode <<= 1;
			cpssp->NAME.opcode += disig;
			cpssp->NAME.opcbits++;
		}
	} else {  /* Opcode already complete */
		if (cpssp->NAME.adbits < EEPROM_ADDRESSBITS) {
			cpssp->NAME.address <<= 1;
			cpssp->NAME.address += disig;
			cpssp->NAME.adbits++;
			if (cpssp->NAME.adbits == EEPROM_ADDRESSBITS) {
				/* Just completed the address */
				cpssp->NAME.datapos=0;
			}
		} else { /* Address already complete too */
			if (16 <= cpssp->NAME.datapos) {
				/* Transfer complete, reset to initial state */
				NAME_(resetstatus)(cpssp);
			} else {
				cpssp->NAME.datapos++;
			}
		}
	}
}

static void
NAME_(cs_set)(struct cpssp *cpssp, unsigned int val)
{
	if (cpssp->eeprom.selected != val) {
		/* There has been a change in the selection - reset
		 * our internal status */
		NAME_(resetstatus)(cpssp);
		cpssp->eeprom.selected = val;
	}
}

static void
NAME_(di_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME.di = val;
}

static void
NAME_(read)(struct cpssp *cpssp)
{
	if (16 < cpssp->NAME.datapos)  {
		NAME_(do_set)(cpssp, 0);
	} else {
		NAME_(do_set)(cpssp,
				(cpssp->NAME.data[cpssp->NAME.address]
				>> (16 - cpssp->NAME.datapos)) & 1);
	}
}

static void
NAME_(write)(struct cpssp *cpssp)
{
	if (16 < cpssp->NAME.datapos
	 || cpssp->NAME.datapos == 0) {
		NAME_(do_set)(cpssp, 0);
		return;
	}
	NAME_(do_set)(cpssp,
			(cpssp->NAME.data[cpssp->NAME.address]
			>> (16 - cpssp->NAME.datapos)) & 1);
	if (! cpssp->NAME.clock) {
		return;
	}
	cpssp->NAME.data[cpssp->NAME.address]
		&= ~(1 << (16 - cpssp->NAME.datapos));
	cpssp->NAME.data[cpssp->NAME.address]
		|= cpssp->NAME.di << (16 - cpssp->NAME.datapos);

	if (cpssp->NAME.datapos == 16) {
		int fd;
		
		fd = open(cpssp->NAME.filename, O_WRONLY | O_CREAT, 0666);
		if (fd < 0) {
			fprintf(stderr, "%s\n","Could not save EEPROM to disc!");
		} else {
			io_write(fd, cpssp->NAME.data, SZ_E100EEPROM * 2);
			close(fd);
		}
	}
}

static void
NAME_(clock_set)(struct cpssp *cpssp, unsigned int val)
{
	if (val != (cpssp->NAME.clock)) {
		/* Clock signal changed */
		cpssp->NAME.clock = val;
		if (val) { /* Clock signal high */
			if (cpssp->NAME.selected) {
				NAME_(handleclock)(cpssp, cpssp->NAME.di);
			}
		}
	}
	if (cpssp->NAME.selected) {
		if (EEPROM_OPCODEBITS <= cpssp->NAME.opcbits
		 && EEPROM_ADDRESSBITS <= cpssp->NAME.adbits) {
			switch (cpssp->NAME.opcode & 0x03) {
			case 0:	/* Special Functions like erase
				 * everything or write everything */
				switch (cpssp->NAME.address >> 4) {
				case 0:
					cpssp->NAME.ewenable = 0;
					break;
				case 3:
					cpssp->NAME.ewenable = 1;
					break;
				default:
					/* Erase everything and write
					 * everything are
					 * UNIMPLEMENTED */
					fprintf(stderr, "Unimplemented EEPROM "
						"command %d\n",
						cpssp->NAME.address >> 6);
					break;
				};
			case 1:	/* Write */
				if (cpssp->NAME.ewenable) {
					eeprom_write(cpssp);
				}
				break;
			case 2:	/* Read */
				eeprom_read(cpssp);
				break;
			case 3:	/* Erase */
				/* There is no data following the
				 * command, so set datapos */
				if (cpssp->NAME.ewenable) {
					cpssp->NAME.data[cpssp->NAME.address] = 0;
				}
				cpssp->NAME.datapos = 17;
				break;
			};
		} else {
			NAME_(do_set)(cpssp, 1);
		}
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
